# -*- coding: utf-8 -*- #
# Copyright 2020 Google LLC. All Rights Reserved.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#    http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
"""Import target HTTP Proxies command."""

from __future__ import absolute_import
from __future__ import division
from __future__ import unicode_literals

from apitools.base.py import exceptions as apitools_exceptions
from googlecloudsdk.api_lib.compute import base_classes
from googlecloudsdk.calliope import base
from googlecloudsdk.calliope import exceptions
from googlecloudsdk.command_lib.compute import flags as compute_flags
from googlecloudsdk.command_lib.compute import scope as compute_scope
from googlecloudsdk.command_lib.compute.target_http_proxies import flags
from googlecloudsdk.command_lib.compute.target_http_proxies import target_http_proxies_utils
from googlecloudsdk.command_lib.export import util as export_util
from googlecloudsdk.core import yaml_validator
from googlecloudsdk.core.console import console_io


def _DetailedHelp():
  return {
      'brief':
          'Import a target HTTP proxy.',
      'DESCRIPTION':
          """\
          Imports a target HTTP proxy's configuration from a file.
          """,
      'EXAMPLES':
          """\
          A target HTTP proxy can be imported by running:

            $ {command} NAME --source=<path-to-file>
          """
  }


def _GetApiVersion(release_track):
  """Returns the API version based on the release track."""
  if release_track == base.ReleaseTrack.ALPHA:
    return 'alpha'
  elif release_track == base.ReleaseTrack.BETA:
    return 'beta'
  return 'v1'


def _GetSchemaPath(release_track, for_help=False):
  """Returns the resource schema path."""
  return export_util.GetSchemaPath(
      'compute',
      _GetApiVersion(release_track),
      'TargetHttpProxy',
      for_help=for_help)


def _SendGetRequest(client, target_http_proxy_ref):
  """Sends Target HTTP Proxy get request."""
  if target_http_proxy_ref.Collection() == 'compute.regionTargetHttpProxies':
    request = client.messages.ComputeRegionTargetHttpProxiesGetRequest(
        **target_http_proxy_ref.AsDict())
    collection = client.apitools_client.regionTargetHttpProxies
    return client.MakeRequests([(collection, 'Get', request)])[0]

  request = client.messages.ComputeTargetHttpProxiesGetRequest(
      **target_http_proxy_ref.AsDict())
  collection = client.apitools_client.targetHttpProxies
  return client.MakeRequests([(collection, 'Get', request)])[0]


def _SendInsertRequest(client, target_http_proxy_ref, target_http_proxy):
  """Sends Target HTTP Proxy insert request."""
  if target_http_proxy_ref.Collection() == 'compute.regionTargetHttpProxies':
    request = client.messages.ComputeRegionTargetHttpProxiesInsertRequest(
        project=target_http_proxy_ref.project,
        targetHttpProxy=target_http_proxy,
        region=target_http_proxy_ref.region)
    collection = client.apitools_client.regionTargetHttpProxies
    return client.MakeRequests([(collection, 'Insert', request)])[0]

  request = client.messages.ComputeTargetHttpProxiesInsertRequest(
      project=target_http_proxy_ref.project, targetHttpProxy=target_http_proxy)
  collection = client.apitools_client.targetHttpProxies
  return client.MakeRequests([(collection, 'Insert', request)])[0]


def _SendPatchRequest(client, target_http_proxy_ref, target_http_proxy):
  """Make target HTTP proxy patch request."""
  # TODO(b/129339772) Fix inconsistent behavior for L7 resource updates.
  if target_http_proxy_ref.Collection() == 'compute.regionTargetHttpProxies':
    console_message = ('Target HTTP Proxy [{0}] cannot be updated'.format(
        target_http_proxy_ref.Name()))
    raise NotImplementedError(console_message)

  request = client.messages.ComputeTargetHttpProxiesPatchRequest(
      project=target_http_proxy_ref.project,
      targetHttpProxy=target_http_proxy_ref.Name(),
      targetHttpProxyResource=target_http_proxy)
  collection = client.apitools_client.targetHttpProxies
  return client.MakeRequests([(collection, 'Patch', request)])[0]


def _Run(args, holder, target_http_proxy_arg, release_track):
  """Issues requests necessary to import target HTTP proxies."""
  client = holder.client

  target_http_proxy_ref = target_http_proxy_arg.ResolveAsResource(
      args,
      holder.resources,
      default_scope=compute_scope.ScopeEnum.GLOBAL,
      scope_lister=compute_flags.GetDefaultScopeLister(client))

  data = console_io.ReadFromFileOrStdin(args.source or '-', binary=False)

  try:
    target_http_proxy = export_util.Import(
        message_type=client.messages.TargetHttpProxy,
        stream=data,
        schema_path=_GetSchemaPath(release_track))
  except yaml_validator.ValidationError as e:
    raise exceptions.ToolException(e.message)

  # Get existing target HTTP proxy.
  try:
    target_http_proxy_old = target_http_proxies_utils.SendGetRequest(
        client, target_http_proxy_ref)
  except apitools_exceptions.HttpError as error:
    if error.status_code != 404:
      raise error
    # Target HTTP proxy does not exist, create a new one.
    return _SendInsertRequest(client, target_http_proxy_ref, target_http_proxy)

  if target_http_proxy_old == target_http_proxy:
    return

  console_io.PromptContinue(
      message=('Target Http Proxy [{0}] will be overwritten.').format(
          target_http_proxy_ref.Name()),
      cancel_on_no=True)

  # Populate id and fingerprint fields. These two fields are manually
  # removed from the schema files.
  target_http_proxy.id = target_http_proxy_old.id
  target_http_proxy.fingerprint = target_http_proxy_old.fingerprint

  return _SendPatchRequest(client, target_http_proxy_ref, target_http_proxy)


@base.ReleaseTracks(base.ReleaseTrack.GA, base.ReleaseTrack.BETA,
                    base.ReleaseTrack.ALPHA)
class Import(base.UpdateCommand):
  """Import a target HTTP Proxy."""

  detailed_help = _DetailedHelp()
  TARGET_HTTP_PROXY_ARG = None

  @classmethod
  def Args(cls, parser):
    cls.TARGET_HTTP_PROXY_ARG = flags.TargetHttpProxyArgument()
    cls.TARGET_HTTP_PROXY_ARG.AddArgument(parser, operation_type='import')
    export_util.AddImportFlags(
        parser, _GetSchemaPath(cls.ReleaseTrack(), for_help=True))

  def Run(self, args):
    holder = base_classes.ComputeApiHolder(self.ReleaseTrack())
    return _Run(args, holder, self.TARGET_HTTP_PROXY_ARG, self.ReleaseTrack())
