package yqlib

import (
	"container/list"
	"fmt"

	yaml "gopkg.in/yaml.v3"
)

func parseStyle(customStyle string) (yaml.Style, error) {
	if customStyle == "tagged" {
		return yaml.TaggedStyle, nil
	} else if customStyle == "double" {
		return yaml.DoubleQuotedStyle, nil
	} else if customStyle == "single" {
		return yaml.SingleQuotedStyle, nil
	} else if customStyle == "literal" {
		return yaml.LiteralStyle, nil
	} else if customStyle == "folded" {
		return yaml.FoldedStyle, nil
	} else if customStyle == "flow" {
		return yaml.FlowStyle, nil
	} else if customStyle != "" {
		return 0, fmt.Errorf("Unknown style %v", customStyle)
	}
	return 0, nil
}

func AssignStyleOperator(d *dataTreeNavigator, matchingNodes *list.List, pathNode *PathTreeNode) (*list.List, error) {

	log.Debugf("AssignStyleOperator: %v")
	var style yaml.Style
	if !pathNode.Operation.UpdateAssign {
		rhs, err := d.GetMatchingNodes(matchingNodes, pathNode.Rhs)
		if err != nil {
			return nil, err
		}

		if rhs.Front() != nil {
			style, err = parseStyle(rhs.Front().Value.(*CandidateNode).Node.Value)
			if err != nil {
				return nil, err
			}
		}
	}

	lhs, err := d.GetMatchingNodes(matchingNodes, pathNode.Lhs)

	if err != nil {
		return nil, err
	}

	for el := lhs.Front(); el != nil; el = el.Next() {
		candidate := el.Value.(*CandidateNode)
		log.Debugf("Setting style of : %v", candidate.GetKey())
		if pathNode.Operation.UpdateAssign {
			rhs, err := d.GetMatchingNodes(nodeToMap(candidate), pathNode.Rhs)
			if err != nil {
				return nil, err
			}

			if rhs.Front() != nil {
				style, err = parseStyle(rhs.Front().Value.(*CandidateNode).Node.Value)
				if err != nil {
					return nil, err
				}
			}
		}

		candidate.Node.Style = style
	}

	return matchingNodes, nil
}

func GetStyleOperator(d *dataTreeNavigator, matchingNodes *list.List, pathNode *PathTreeNode) (*list.List, error) {
	log.Debugf("GetStyleOperator")

	var results = list.New()

	for el := matchingNodes.Front(); el != nil; el = el.Next() {
		candidate := el.Value.(*CandidateNode)
		var style string
		switch candidate.Node.Style {
		case yaml.TaggedStyle:
			style = "tagged"
		case yaml.DoubleQuotedStyle:
			style = "double"
		case yaml.SingleQuotedStyle:
			style = "single"
		case yaml.LiteralStyle:
			style = "literal"
		case yaml.FoldedStyle:
			style = "folded"
		case yaml.FlowStyle:
			style = "flow"
		case 0:
			style = ""
		default:
			style = "<unknown>"
		}
		node := &yaml.Node{Kind: yaml.ScalarNode, Value: style, Tag: "!!str"}
		lengthCand := &CandidateNode{Node: node, Document: candidate.Document, Path: candidate.Path}
		results.PushBack(lengthCand)
	}

	return results, nil
}
