package serverinstall

import (
	"context"

	"github.com/hashicorp/go-hclog"
	"github.com/hashicorp/waypoint-plugin-sdk/terminal"
	"github.com/hashicorp/waypoint/internal/clicontext"
	"github.com/hashicorp/waypoint/internal/pkg/flag"
	pb "github.com/hashicorp/waypoint/internal/server/gen"
	"github.com/hashicorp/waypoint/internal/serverconfig"
)

// Installer is implemented by the server platforms and is responsible for managing
// the installation of the Waypoint server.
type Installer interface {
	// Install expects the Waypoint server to be installed.
	Install(context.Context, *InstallOpts) (*InstallResults, error)

	// InstallFlags is called prior to Install and allows the installer to
	// specify flags for the install CLI. The flags should be prefixed with
	// the platform name to avoid conflicts with other flags.
	InstallFlags(*flag.Set)

	// Upgrade expects the Waypoint server to be upgraded from a previous install
	Upgrade(ctx context.Context, opts *InstallOpts, serverCfg serverconfig.Client) (*InstallResults, error)

	// UpgradeFlags is called prior to Upgrade and allows the upgrader to
	// specify flags for the upgrade CLI. The flags should be prefixed with
	// the platform name to avoid conflicts with other flags.
	UpgradeFlags(*flag.Set)

	// Uninstall expects the Waypoint server to be uninstalled.
	Uninstall(context.Context, *InstallOpts) error

	// UninstallFlags is called prior to Uninstall and allows the Uninstaller to
	// specify flags for the uninstall CLI. The flags should be prefixed with the
	// platform name to avoid conflicts with other flags.
	UninstallFlags(*flag.Set)
}

// InstallOpts are the options sent to Installer.Install.
type InstallOpts struct {
	Log hclog.Logger
	UI  terminal.UI
}

// InstallResults are the results expected for a successful Installer.Install.
type InstallResults struct {
	// Context is the connection context that can be used to connect from
	// the CLI to the server. This will be used to establish an API client.
	Context *clicontext.Config

	// AdvertiseAddr is the configuration for the advertised address
	// that entrypoints (deployed workloads) will use to communicate back
	// to the server. This may be different from the context info because this
	// may be a private address.
	AdvertiseAddr *pb.ServerConfig_AdvertiseAddr

	// HTTPAddr is the address to the HTTP listener on the server. This generally
	// is reachable from the CLI immediately and not a private address.
	HTTPAddr string
}

var Platforms = map[string]Installer{
	"kubernetes": &K8sInstaller{},
	"nomad":      &NomadInstaller{},
	"docker":     &DockerInstaller{},
}

const (
	serverName = "waypoint-server"

	defaultServerImage = "hashicorp/waypoint:latest"
)

// Default server ports to use
var (
	defaultGrpcPort = "9701"
	defaultHttpPort = "9702"
)
