// Copyright 2021 Google LLC.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.
//
///////////////////////////////////////////////////////////////////////////////

#include "tink/jwt/internal/jwt_public_key_verify_impl.h"

#include "absl/strings/escaping.h"
#include "absl/strings/str_split.h"
#include "tink/jwt/internal/json_util.h"
#include "tink/jwt/internal/jwt_format.h"

namespace crypto {
namespace tink {
namespace jwt_internal {

util::StatusOr<VerifiedJwt> JwtPublicKeyVerifyImpl::VerifyAndDecode(
    absl::string_view compact, const JwtValidator& validator) const {
  // TODO(juerg): Refactor this code into a util function.
  std::size_t signature_pos = compact.find_last_of('.');
  if (signature_pos == absl::string_view::npos) {
    return util::Status(util::error::INVALID_ARGUMENT, "invalid token");
  }
  absl::string_view unsigned_token = compact.substr(0, signature_pos);
  std::string signature;
  if (!DecodeSignature(compact.substr(signature_pos + 1), &signature)) {
    return util::Status(util::error::INVALID_ARGUMENT, "invalid JWT signature");
  }
  util::Status verify_result = verify_->Verify(signature, unsigned_token);
  if (!verify_result.ok()) {
    return verify_result;
  }
  std::vector<absl::string_view> parts = absl::StrSplit(unsigned_token, '.');
  if (parts.size() != 2) {
    return util::Status(
        util::error::INVALID_ARGUMENT,
        "only tokens in JWS compact serialization format are supported");
  }
  std::string json_header;
  if (!DecodeHeader(parts[0], &json_header)) {
    return util::Status(util::error::INVALID_ARGUMENT, "invalid header");
  }
  auto header_or = JsonStringToProtoStruct(json_header);
  if (!header_or.ok()) {
    return header_or.status();
  }
  util::Status validate_header_result =
      ValidateHeader(header_or.ValueOrDie(), algorithm_);
  if (!validate_header_result.ok()) {
    return validate_header_result;
  }
  std::string json_payload;
  if (!DecodePayload(parts[1], &json_payload)) {
    return util::Status(util::error::INVALID_ARGUMENT, "invalid JWT payload");
  }
  auto raw_jwt_or =
      RawJwt::FromJson(GetTypeHeader(header_or.ValueOrDie()), json_payload);
  if (!raw_jwt_or.ok()) {
    return raw_jwt_or.status();
  }
  RawJwt raw_jwt = raw_jwt_or.ValueOrDie();
  util::Status validate_result = validator.Validate(raw_jwt);
  if (!validate_result.ok()) {
    return validate_result;
  }
  return VerifiedJwt(raw_jwt);
}

}  // namespace jwt_internal
}  // namespace tink
}  // namespace crypto
