# Module Structure

A Timoni module contains a set of CUE definitions and constraints organised
into a [CUE module](https://cuelang.org/docs/concepts/packages/)
with an opinionated file structure.

## File Structure

A module consists of a collection of CUE files inside a directory
with the following structure:

```sh
myapp
├── README.md
├── cue.mod
│   ├── gen # Kubernetes types
│   └── module.cue # Module metadata
├── templates
│   ├── config.cue # Config schema and default values
│   ├── deployment.cue # Kubernetes Deployment template
│   └── service.cue # Kubernetes Service template
├── timoni.cue # Timoni entry point
└── values.cue # Timoni values placeholder
```

To create a new module in the current directory:

```shell
timoni mod init myapp .
```

## Entry point

The `timoni.cue` file contains the definition of how Timoni should
validate, build and deploy a module instance.

This file is generated by `timoni mod init` with the following content:

```go
// source: myapp/timoni.cue

package main

import (
	templates "timoni.sh/myapp/templates"
)

// Define the schema for the user-supplied values.
// At runtime, Timoni injects the supplied values
// and validates them according to the Config schema.
values: templates.#Config

// Define how Timoni should build, validate and
// apply the Kubernetes resources.
timoni: {
	apiVersion: "v1alpha1"

	// Define the instance that generates the Kubernetes resources.
	// At runtime, Timoni builds the instance and validates
	// the resulting resources according to their Kubernetes schema.
	instance: templates.#Instance & {
		// The user-supplied values are merged with the
		// default values at runtime by Timoni.
		config: values
		// The instance name and namespace tag values
		// are injected at runtime by Timoni.
		config: metadata: {
			name:      string @tag(name)
			namespace: string @tag(namespace)
		}
	}

	// Pass the generated Kubernetes resources
	// to Timoni's multi-step apply.
	apply: all: [ for obj in instance.objects {obj}]
}
```

## Values

The `values.cue` file is a placeholder for the user-supplied values.
This file is generated by `timoni mod init`, no customisations are allowed.

The `values` schema is set in the `timoni.cue` file:

```go
// source: myapp/timoni.cue

values: templates.#Config
```

Note that the `README.md` file should contain the config values schema documentation.

## Templates

The templates directory is where module authors define Kubernetes resources
and their configuration schema.

### Config

The schema and defaults for the user-supplied values are defined in `templates/config.cue`.

Example of a minimal config for an app deployment:

```go
// source: myapp/templates/config.cue

#Config: {
	metadata: metav1.#ObjectMeta
	metadata: name:      *"myapp" | string
	metadata: namespace: *"default" | string
	metadata: annotations?: {[string]: string}

	service: port: *80 | int & >0 & <=65535
	
	image: {
		repository: *"docker.io/myorg/myapp" | string
		tag:        *"1.0.0" | string
		pullPolicy: *"IfNotPresent" | string
	}
	
	metadata: labels: {
		"app.kubernetes.io/name":    metadata.name
		"app.kubernetes.io/version": image.tag
	}
	
	resources?: corev1.#ResourceRequirements
}
```

The user-supplied values can:

- set a different value for the fields with defaults e.g. the service port
- add labels to metadata except for the name and version
- add annotations to metadata
- set resource requirements requests and/or limits

```go
// source: myapp-values/values.cue

values: {
	service: port: 8080
	metadata: annontations: "org/owner": "dev-team"
	resources: limits: memory: "1Gi"
}
```

### Instance

Example of defining an instance containing a Kubernetes Service and Deployment:

```go
// source: myapp/templates/config.cue

#Instance: {
	config: #Config

	objects: {
		svc:    #Service & {_config:        config}
		deploy: #Deployment & {_config:     config}
	}
}
```

### Kubernetes resources

Example of a Kubernetes Service template:

```go
// source: myapp/templates/service.cue

package templates

import (
	corev1 "k8s.io/api/core/v1"
)

#Service: corev1.#Service & {
	_config:    #Config
	apiVersion: "v1"
	kind:       "Service"
	metadata:   _config.metadata
	spec:       corev1.#ServiceSpec & {
		type:     corev1.#ServiceTypeClusterIP
		selector: _config.metadata.labels
		ports: [
			{
				name:       "http"
				targetPort: name
				port:       _config.service.port
			},
		]
	}
}
```

## Kubernetes types

The `cue.mod` directory contains the Kubernetes types and their schema.
These files are automatically generated by CUE from the Kubernetes API Go packages.

```shell
./myapp/cue.mod/
├── gen
│   └── k8s.io
│       ├── api
│       │   ├── apps
│       │   └── core
│       └── apimachinery
└── module.cue
```

To update the schemas to a specific Kubernetes version, run the following commands
from within the `myapp` directory:

```shell
# download the Kubernetes Go APIs
go get k8s.io/api/...@v0.26.2

# generate the CUE types for v1 core and apps APIs
cue get go k8s.io/api/core/v1
cue get go k8s.io/api/apps/v1
```

To add other Kubernetes types e.g. `Ingress` and `NetworkPolicy`:

```shell
cue get go k8s.io/api/networking/v1
```

### Kubernetes CRDs

To add 3rd-party Kubernetes types e.g. Prometheus Operator `ServiceMonitor`:

```shell
# download the Prometheus Operator Go APIs
go get github.com/prometheus-operator/prometheus-operator/pkg/apis/monitoring/v1@v0.63.0

# generate the CUE types for v1 monitoring APIs
cue get go github.com/prometheus-operator/prometheus-operator/pkg/apis/monitoring/v1
```

The Kubernetes CUE types are used in templates to set the validation schema of a resource e.g.:

```go
package templates

import (
	promv1 "github.com/prometheus-operator/prometheus-operator/pkg/apis/monitoring/v1"
)

#ServiceMonitor: promv1.#ServiceMonitor & {
	_config:    #Config
	apiVersion: "monitoring.coreos.com/v1"
	kind:       promv1.#ServiceMonitorsKind
	metadata:   _config.metadata
	spec: {
		endpoints: [{
			path:     "/metrics"
			port:     "http-metrics"
			interval: _config.monitoring.interval
		}]
		namespaceSelector: matchNames: [_config.metadata.namespace]
		selector: matchLabels: _config.selectorLabels
	}
}
```
