/*
    Copyright (C) 2022 Tenable, Inc.

	Licensed under the Apache License, Version 2.0 (the "License");
    you may not use this file except in compliance with the License.
    You may obtain a copy of the License at

		http://www.apache.org/licenses/LICENSE-2.0

	Unless required by applicable law or agreed to in writing, software
    distributed under the License is distributed on an "AS IS" BASIS,
    WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
    See the License for the specific language governing permissions and
    limitations under the License.
*/

package vulnerability

import (
	"strings"

	"github.com/tenable/terrascan/pkg/iac-providers/output"
	"github.com/tenable/terrascan/pkg/results"
	"go.uber.org/zap"
)

var containerRegistry = make(map[string]ContainerRegistry)

const (
	//VulnerabilityCVSSSource hold type of cvss source
	VulnerabilityCVSSSource = "nvd"
)

//RegisterContainerRegistry register the container registry for vulnerability scanning
func RegisterContainerRegistry(registryType string, registry ContainerRegistry) {
	containerRegistry[registryType] = registry
}

// NewVulEngine returns a new vulnerability engine
func NewVulEngine() (*VulEngine, error) {

	// vulnerability engine struct
	engine := &VulEngine{}

	// successful
	return engine, nil
}

//FetchVulnerabilities fetch vulnerabilities for images found in IaC files
func (v *VulEngine) FetchVulnerabilities(resourceConfigs output.AllResourceConfigs, options map[string]interface{}) output.AllResourceConfigs {
	for i, congigs := range resourceConfigs {
		for j, config := range congigs {
			for k, container := range config.ContainerImages {
				resourceConfigs[i][j].ContainerImages[k].Vulnerabilities =
					append(resourceConfigs[i][j].ContainerImages[k].Vulnerabilities,
						getVulnerabilitiesByRepository(container, options)...)
			}
			for l, container := range config.InitContainerImages {
				resourceConfigs[i][j].InitContainerImages[l].Vulnerabilities =
					append(resourceConfigs[i][j].InitContainerImages[l].Vulnerabilities,
						getVulnerabilitiesByRepository(container, options)...)
			}
		}
	}
	return resourceConfigs
}

//getVulnerabilitiesByRepository get vulnerabilities depending on type of registry
func getVulnerabilitiesByRepository(container output.ContainerDetails, options map[string]interface{}) []output.Vulnerability {
	for _, registry := range containerRegistry {
		if registry.checkRegistry(container.Image) {
			return registry.getVulnerabilities(container, options)
		}
	}
	zap.S().Debugf("no registered container registry found for image %s", container.Image)
	return []output.Vulnerability{}
}

// ReportVulnerability Add a vulnrability for a given resource in scan summary
func (v *VulEngine) ReportVulnerability(engineInput EngineInput, options map[string]interface{}) EngineOutput {
	v.results.ViolationStore = results.NewViolationStore()
	for _, resources := range *engineInput.InputData {
		for _, resource := range resources {
			for _, container := range resource.ContainerImages {
				vulnerabilities := prepareVulnerabilityObject(container, resource, options)
				v.results.ViolationStore.Vulnerabilities =
					append(v.results.ViolationStore.Vulnerabilities, vulnerabilities...)

			}
			for _, container := range resource.InitContainerImages {
				vulnerabilities := prepareVulnerabilityObject(container, resource, options)
				v.results.ViolationStore.Vulnerabilities =
					append(v.results.ViolationStore.Vulnerabilities, vulnerabilities...)
			}
		}
	}
	count := len(v.results.ViolationStore.Vulnerabilities)
	v.results.Summary.Vulnerabilities = &count
	return v.results
}

// prepareVulnerabilityObject Add a vulnrability for a given resource
func prepareVulnerabilityObject(container output.ContainerDetails, resource output.ResourceConfig, options map[string]interface{}) []*results.Vulnerability {
	vulnerabilities := []*results.Vulnerability{}
	for _, vul := range container.Vulnerabilities {
		vulnerability := &results.Vulnerability{
			Image:            container.Image,
			Container:        container.Name,
			Package:          vul.PkgName,
			Severity:         strings.ToUpper(vul.Severity),
			VulnerabilityID:  vul.VulnerabilityID,
			Description:      vul.Description,
			InstalledVersion: vul.InstalledVersion,
			File:             resource.Source,
			LineNumber:       resource.Line,
			PrimaryURL:       vul.PrimaryURL,
			ResourceType:     resource.Type,
			ResourceName:     resource.Name,
		}
		if cvss, ok := vul.CVSS[VulnerabilityCVSSSource]; ok {
			vulnerability.CVSSScore = results.CVSS{
				V2Vector: cvss.V2Vector,
				V2Score:  cvss.V2Score,
				V3Vector: cvss.V3Vector,
				V3Score:  cvss.V3Score,
			}
		}
		vulnerabilities = append(vulnerabilities, vulnerability)
	}

	return vulnerabilities

}
