/*
    Copyright (C) 2020 Accurics, Inc.

	Licensed under the Apache License, Version 2.0 (the "License");
    you may not use this file except in compliance with the License.
    You may obtain a copy of the License at

		http://www.apache.org/licenses/LICENSE-2.0

	Unless required by applicable law or agreed to in writing, software
    distributed under the License is distributed on an "AS IS" BASIS,
    WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
    See the License for the specific language governing permissions and
    limitations under the License.
*/

package vulnerability

import (
	"encoding/xml"
	"reflect"
	"strings"
	"testing"

	"github.com/accurics/terrascan/pkg/iac-providers/output"
	"github.com/accurics/terrascan/pkg/results"
)

var resourceConfigs = output.AllResourceConfigs{
	"test": []output.ResourceConfig{
		{
			ContainerImages: []output.ContainerDetails{
				{
					Name:  "test",
					Image: "test",
					Vulnerabilities: []output.Vulnerability{
						{Severity: "low",
							CVSS: output.VendorCVSS{
								"nvd": output.CVSS{
									V2Vector: "test",
									V3Vector: "test",
									V2Score:  8,
									V3Score:  6,
								},
							}},
					},
				},
			},
			InitContainerImages: []output.ContainerDetails{
				{
					Name:  "inittest",
					Image: "test",
					Vulnerabilities: []output.Vulnerability{
						{Severity: "high",
							CVSS: output.VendorCVSS{
								"nvd": output.CVSS{
									V2Vector: "test",
									V3Vector: "test",
									V2Score:  8,
									V3Score:  6,
								},
							}},
					},
				},
			},
		},
	},
}

func TestPrepareVulnerabilityObject(t *testing.T) {
	options := make(map[string]interface{})
	type args struct {
		container output.ContainerDetails
		resource  output.ResourceConfig
	}
	tests := []struct {
		name string
		args args
		want []*results.Vulnerability
	}{
		{name: "output with 1 result",
			args: args{
				container: output.ContainerDetails{
					Name:  "test",
					Image: "test",
					Vulnerabilities: []output.Vulnerability{
						{Severity: "low",
							CVSS: output.VendorCVSS{
								"nvd": output.CVSS{
									V2Vector: "test",
									V3Vector: "test",
									V2Score:  8,
									V3Score:  6,
								},
							}},
					},
				},
				resource: output.ResourceConfig{
					Line:   2,
					Type:   "test",
					Source: "test",
					Name:   "test",
				},
			},
			want: []*results.Vulnerability{
				{
					Image:        "test",
					Container:    "test",
					Severity:     strings.ToUpper("low"),
					File:         "test",
					LineNumber:   2,
					ResourceType: "test",
					ResourceName: "test",
					CVSSScore: results.CVSS{
						V2Vector: "test",
						V3Vector: "test",
						V2Score:  8,
						V3Score:  6,
					},
				},
			},
		},
	}
	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			if got := prepareVulnerabilityObject(tt.args.container, tt.args.resource, options); !reflect.DeepEqual(got, tt.want) {
				t.Errorf("prepareVulnerabilityObject() = %v, want %v", got, tt.want)
			}
		})
	}
}

func TestGetVulnerabilitiesByRepository(t *testing.T) {
	options := make(map[string]interface{})
	type args struct {
		container output.ContainerDetails
	}
	tests := []struct {
		name string
		args args
		want []output.Vulnerability
	}{
		{
			name: "no registered registries",
			args: args{
				container: output.ContainerDetails{
					Image: "test",
				},
			},
			want: []output.Vulnerability{},
		},
	}
	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			if got := getVulnerabilitiesByRepository(tt.args.container, options); !reflect.DeepEqual(got, tt.want) {
				t.Errorf("getVulnerabilitiesByRepository() = %v, want %v", got, tt.want)
			}
		})
	}
}

func TestVulEngine_ReportVulnerability(t *testing.T) {
	count := 2
	type fields struct {
		results EngineOutput
	}
	type args struct {
		engineInput EngineInput
		options     map[string]interface{}
	}
	tests := []struct {
		name   string
		fields fields
		args   args
		want   EngineOutput
	}{
		{
			name: "get vulnerability",
			args: args{
				engineInput: EngineInput{
					InputData: &resourceConfigs,
				},
			},
			fields: fields{},
			want: EngineOutput{
				XMLName: xml.Name{Space: "", Local: ""},
				ViolationStore: &results.ViolationStore{
					Summary: results.ScanSummary{
						Vulnerabilities: &count,
					},
					Vulnerabilities: []*results.Vulnerability{
						{
							Image:     "test",
							Container: "test",
							Severity:  strings.ToUpper("low"),
							CVSSScore: results.CVSS{
								V2Vector: "test",
								V3Vector: "test",
								V2Score:  8,
								V3Score:  6,
							},
						},
						{
							Image:     "test",
							Container: "inittest",
							Severity:  strings.ToUpper("high"),
							CVSSScore: results.CVSS{
								V2Vector: "test",
								V3Vector: "test",
								V2Score:  8,
								V3Score:  6,
							},
						},
					},
				},
			},
		},
	}
	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			v := &VulEngine{
				results: tt.fields.results,
			}
			if got := v.ReportVulnerability(tt.args.engineInput, tt.args.options); !reflect.DeepEqual(got.ViolationStore.Vulnerabilities, tt.want.ViolationStore.Vulnerabilities) {
				t.Errorf("VulEngine.ReportVulnerability() = %v, want %v", got, tt.want)
			}
		})
	}
}

func TestVulEngineFetchVulnerabilities(t *testing.T) {
	type fields struct {
		results EngineOutput
	}
	type args struct {
		resourceConfigs output.AllResourceConfigs
		options         map[string]interface{}
	}
	tests := []struct {
		name   string
		fields fields
		args   args
		want   output.AllResourceConfigs
	}{
		{
			name:   "find vulnerabilities",
			fields: fields{},
			args: args{
				resourceConfigs: resourceConfigs,
			},
			want: resourceConfigs,
		},
	}
	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			v := &VulEngine{
				results: tt.fields.results,
			}
			if got := v.FetchVulnerabilities(tt.args.resourceConfigs, tt.args.options); !reflect.DeepEqual(got, tt.want) {
				t.Errorf("VulEngine.FindVulnerabilities() = %v, want %v", got, tt.want)
			}
		})
	}
}
