---
title: "How To's"
description: "terraform-docs how to's on variety of topics."
menu:
  docs:
    parent: "user-guide"
weight: 140
toc: true
---

## Visibility of Sections

Output generated by `terraform-docs` consists of different [sections] which are
visible by default. The visibility of these can be controlled by one or combination
of:

- `--show-all`
- `--hide-all`
- `--show <name>`
- and `--hide <name>`

```console
terraform-docs --show-all --hide header ...                # show all sections except 'header'
terraform-docs --hide-all --show inputs --show outputs ... # hide all sections except 'inputs' and 'outputs'
```

## Module Header

Module header can be extracted from different sources. Default file to extract
header from is `main.tf`, otherwise you can specify the file with `--header-from FILE`
or corresponding `header-from` in configuration file. Supported file formats to
read header from are:

- `.adoc`
- `.md`
- `.tf`
- `.txt`

The whole file content is being extracted as module header when extracting from
`.adoc`, `.md`, or `.txt`. But to extract header from `.tf` file you need to use
following javascript, c or java like multi-line comment:

```tf
/**
 * # Main title
 *
 * Everything in this comment block will get extracted.
 *
 * You can put simple text or complete Markdown content
 * here. Subsequently if you want to render AsciiDoc format
 * you can put AsciiDoc compatible content in this comment
 * block.
 */

resource "foo" "bar" { ... }
```

**Note:** This comment must start at the immediate first line of the `.tf` file
before any `resource`, `variable`, `module`, etc.

**Note:** we will never alter line-endings of extracted header text and will assume
whatever extracted is intended as is. It's up to you to apply any kind of Markdown
formatting to them (i.e. adding `<SPACE><SPACE>` at the end of lines for break, etc.)

## Module Footer

Extracting module footer works exactly like header with one exception. There is no
default file to attempt extraction from, you need to explicitly specify desired file
to extract content from with `--footer-from FILE` or corresponding `footer-from` in
configuration file.

## Insert Output To File

Since `v0.12.0`, generated output can be insterted directly into the file. There
are two modes of insersion: `inject` (default) or `replace`. Take a look at [output]
configuration for all the details.

```console
terraform-docs markdown table --output-file README.md --output-mode inject /path/to/module
```

## Generate terraform.tfvars

You can generate `terraform.tfvars` in both `hcl` and `json` format by executing
the following, respectively:

```console
terraform-docs tfvars hcl /path/to/module

terraform-docs tfvars json /path/to/module
```

**Note:** Required input variables will be `""` (empty) in HCL and `null` in JSON
format.

## GitHub Action

To use terraform-docs GitHub Action, configure a YAML workflow file (e.g.
`.github/workflows/documentation.yml`) with the following:

```yaml
name: Generate terraform docs
on:
  - pull_request

jobs:
  docs:
    runs-on: ubuntu-latest
    steps:
    - uses: actions/checkout@v2
      with:
        ref: ${{ github.event.pull_request.head.ref }}

    - name: Render terraform docs and push changes back to PR
      uses: terraform-docs/gh-actions@v0.6.0
      with:
        working-dir: .
        output-file: USAGE.md
        output-method: inject
        git-push: "true"
```

Read more about [terraform-docs GitHub Action] and its configuration and
examples.

## Pre-commit Hook

With [`pre-commit`], you can ensure your Terraform module documentation is kept
up-to-date each time you make a commit.

First, simply create or update a `.pre-commit-config.yaml`
in the root of your Git repo with at least the following content:

```yaml
repos:
  - repo: https://github.com/terraform-docs/terraform-docs
    rev: <VERSION TAG OR SHA TO USE> # For example: "v0.12.0"
    hooks:
      - id: terraform-docs-go
        args: [<ARGS TO PASS INCLUDING PATH>]  # For example: ["--output-file", "README.md", "./mymodule/path"]
```

(You can also include more than one entry under `hooks:` to update multiple docs.
Just be sure to adjust the `args:` to pass the path you want terraform-docs to scan.)

Second, install [`pre-commit`] and run `pre-commit` to activate the hooks.

Then, make a Terraform change, `git add` and `git commit`!
Pre-commit will regenerate your Terraform docs, after which you can
rerun `git add` and `git commit` to commit the code and doc changes together.

You can also regenerate the docs manually by running `pre-commit -a terraform-docs`.

### Pre-commit via Docker

The pre-commit hook can also be run via Docker, for those who don't have Go installed.
Just use `id: terraform-docs-docker` in the previous example.

This will build the Docker image from the repo, which can be quite slow.
To download the pre-built image instead, change your `.pre-commit-config.yaml` to:

```yaml
repos:
  - repo: local
    hooks:
      - id: terraform-docs
        name: terraform-docs
        language: docker_image
        entry: quay.io/terraform-docs/terraform-docs:latest  # Or, change latest to pin to a specific version
        args: [<ARGS TO PASS INCLUDING PATH>]  # For example: ["--output-file", "README.md", "./mymodule/path"]
        pass_filenames: false
```

## Git Hook

A simple git hook (`.git/hooks/pre-commit`) added to your local terraform
repository can keep your Terraform module documentation up to date whenever you
make a commit. See also [git hooks] documentation.

```sh
#!/bin/sh

# Keep module docs up to date
for d in modules/*; do
  if terraform-docs md "$d" > "$d/README.md"; then
    git add "./$d/README.md"
  fi
done
```

**Note:** This is very basic and higly simplified version of [pre-commit-terraform].
Please refer to it for complete examples and guides.

[sections]: {{< ref "configuration/#sections" >}}
[output]: {{< ref "configuration/#output" >}}
[terraform-docs GitHub Action]: https://github.com/terraform-docs/gh-actions
[`pre-commit`]: https://pre-commit.com/
[git hooks]: https://git-scm.com/book/en/v2/Customizing-Git-Git-Hooks
[pre-commit-terraform]: https://github.com/antonbabenko/pre-commit-terraform
