# bash completion for stripe                               -*- shell-script -*-

__stripe_debug()
{
    if [[ -n ${BASH_COMP_DEBUG_FILE} ]]; then
        echo "$*" >> "${BASH_COMP_DEBUG_FILE}"
    fi
}

# Homebrew on Macs have version 1.3 of bash-completion which doesn't include
# _init_completion. This is a very minimal version of that function.
__stripe_init_completion()
{
    COMPREPLY=()
    _get_comp_words_by_ref "$@" cur prev words cword
}

__stripe_index_of_word()
{
    local w word=$1
    shift
    index=0
    for w in "$@"; do
        [[ $w = "$word" ]] && return
        index=$((index+1))
    done
    index=-1
}

__stripe_contains_word()
{
    local w word=$1; shift
    for w in "$@"; do
        [[ $w = "$word" ]] && return
    done
    return 1
}

__stripe_handle_go_custom_completion()
{
    __stripe_debug "${FUNCNAME[0]}: cur is ${cur}, words[*] is ${words[*]}, #words[@] is ${#words[@]}"

    local shellCompDirectiveError=1
    local shellCompDirectiveNoSpace=2
    local shellCompDirectiveNoFileComp=4
    local shellCompDirectiveFilterFileExt=8
    local shellCompDirectiveFilterDirs=16

    local out requestComp lastParam lastChar comp directive args

    # Prepare the command to request completions for the program.
    # Calling ${words[0]} instead of directly stripe allows to handle aliases
    args=("${words[@]:1}")
    requestComp="${words[0]} __completeNoDesc ${args[*]}"

    lastParam=${words[$((${#words[@]}-1))]}
    lastChar=${lastParam:$((${#lastParam}-1)):1}
    __stripe_debug "${FUNCNAME[0]}: lastParam ${lastParam}, lastChar ${lastChar}"

    if [ -z "${cur}" ] && [ "${lastChar}" != "=" ]; then
        # If the last parameter is complete (there is a space following it)
        # We add an extra empty parameter so we can indicate this to the go method.
        __stripe_debug "${FUNCNAME[0]}: Adding extra empty parameter"
        requestComp="${requestComp} \"\""
    fi

    __stripe_debug "${FUNCNAME[0]}: calling ${requestComp}"
    # Use eval to handle any environment variables and such
    out=$(eval "${requestComp}" 2>/dev/null)

    # Extract the directive integer at the very end of the output following a colon (:)
    directive=${out##*:}
    # Remove the directive
    out=${out%:*}
    if [ "${directive}" = "${out}" ]; then
        # There is not directive specified
        directive=0
    fi
    __stripe_debug "${FUNCNAME[0]}: the completion directive is: ${directive}"
    __stripe_debug "${FUNCNAME[0]}: the completions are: ${out[*]}"

    if [ $((directive & shellCompDirectiveError)) -ne 0 ]; then
        # Error code.  No completion.
        __stripe_debug "${FUNCNAME[0]}: received error from custom completion go code"
        return
    else
        if [ $((directive & shellCompDirectiveNoSpace)) -ne 0 ]; then
            if [[ $(type -t compopt) = "builtin" ]]; then
                __stripe_debug "${FUNCNAME[0]}: activating no space"
                compopt -o nospace
            fi
        fi
        if [ $((directive & shellCompDirectiveNoFileComp)) -ne 0 ]; then
            if [[ $(type -t compopt) = "builtin" ]]; then
                __stripe_debug "${FUNCNAME[0]}: activating no file completion"
                compopt +o default
            fi
        fi
    fi

    if [ $((directive & shellCompDirectiveFilterFileExt)) -ne 0 ]; then
        # File extension filtering
        local fullFilter filter filteringCmd
        # Do not use quotes around the $out variable or else newline
        # characters will be kept.
        for filter in ${out[*]}; do
            fullFilter+="$filter|"
        done

        filteringCmd="_filedir $fullFilter"
        __stripe_debug "File filtering command: $filteringCmd"
        $filteringCmd
    elif [ $((directive & shellCompDirectiveFilterDirs)) -ne 0 ]; then
        # File completion for directories only
        local subDir
        # Use printf to strip any trailing newline
        subdir=$(printf "%s" "${out[0]}")
        if [ -n "$subdir" ]; then
            __stripe_debug "Listing directories in $subdir"
            __stripe_handle_subdirs_in_dir_flag "$subdir"
        else
            __stripe_debug "Listing directories in ."
            _filedir -d
        fi
    else
        while IFS='' read -r comp; do
            COMPREPLY+=("$comp")
        done < <(compgen -W "${out[*]}" -- "$cur")
    fi
}

__stripe_handle_reply()
{
    __stripe_debug "${FUNCNAME[0]}"
    local comp
    case $cur in
        -*)
            if [[ $(type -t compopt) = "builtin" ]]; then
                compopt -o nospace
            fi
            local allflags
            if [ ${#must_have_one_flag[@]} -ne 0 ]; then
                allflags=("${must_have_one_flag[@]}")
            else
                allflags=("${flags[*]} ${two_word_flags[*]}")
            fi
            while IFS='' read -r comp; do
                COMPREPLY+=("$comp")
            done < <(compgen -W "${allflags[*]}" -- "$cur")
            if [[ $(type -t compopt) = "builtin" ]]; then
                [[ "${COMPREPLY[0]}" == *= ]] || compopt +o nospace
            fi

            # complete after --flag=abc
            if [[ $cur == *=* ]]; then
                if [[ $(type -t compopt) = "builtin" ]]; then
                    compopt +o nospace
                fi

                local index flag
                flag="${cur%=*}"
                __stripe_index_of_word "${flag}" "${flags_with_completion[@]}"
                COMPREPLY=()
                if [[ ${index} -ge 0 ]]; then
                    PREFIX=""
                    cur="${cur#*=}"
                    ${flags_completion[${index}]}
                    if [ -n "${ZSH_VERSION}" ]; then
                        # zsh completion needs --flag= prefix
                        eval "COMPREPLY=( \"\${COMPREPLY[@]/#/${flag}=}\" )"
                    fi
                fi
            fi
            return 0;
            ;;
    esac

    # check if we are handling a flag with special work handling
    local index
    __stripe_index_of_word "${prev}" "${flags_with_completion[@]}"
    if [[ ${index} -ge 0 ]]; then
        ${flags_completion[${index}]}
        return
    fi

    # we are parsing a flag and don't have a special handler, no completion
    if [[ ${cur} != "${words[cword]}" ]]; then
        return
    fi

    local completions
    completions=("${commands[@]}")
    if [[ ${#must_have_one_noun[@]} -ne 0 ]]; then
        completions+=("${must_have_one_noun[@]}")
    elif [[ -n "${has_completion_function}" ]]; then
        # if a go completion function is provided, defer to that function
        __stripe_handle_go_custom_completion
    fi
    if [[ ${#must_have_one_flag[@]} -ne 0 ]]; then
        completions+=("${must_have_one_flag[@]}")
    fi
    while IFS='' read -r comp; do
        COMPREPLY+=("$comp")
    done < <(compgen -W "${completions[*]}" -- "$cur")

    if [[ ${#COMPREPLY[@]} -eq 0 && ${#noun_aliases[@]} -gt 0 && ${#must_have_one_noun[@]} -ne 0 ]]; then
        while IFS='' read -r comp; do
            COMPREPLY+=("$comp")
        done < <(compgen -W "${noun_aliases[*]}" -- "$cur")
    fi

    if [[ ${#COMPREPLY[@]} -eq 0 ]]; then
		if declare -F __stripe_custom_func >/dev/null; then
			# try command name qualified custom func
			__stripe_custom_func
		else
			# otherwise fall back to unqualified for compatibility
			declare -F __custom_func >/dev/null && __custom_func
		fi
    fi

    # available in bash-completion >= 2, not always present on macOS
    if declare -F __ltrim_colon_completions >/dev/null; then
        __ltrim_colon_completions "$cur"
    fi

    # If there is only 1 completion and it is a flag with an = it will be completed
    # but we don't want a space after the =
    if [[ "${#COMPREPLY[@]}" -eq "1" ]] && [[ $(type -t compopt) = "builtin" ]] && [[ "${COMPREPLY[0]}" == --*= ]]; then
       compopt -o nospace
    fi
}

# The arguments should be in the form "ext1|ext2|extn"
__stripe_handle_filename_extension_flag()
{
    local ext="$1"
    _filedir "@(${ext})"
}

__stripe_handle_subdirs_in_dir_flag()
{
    local dir="$1"
    pushd "${dir}" >/dev/null 2>&1 && _filedir -d && popd >/dev/null 2>&1 || return
}

__stripe_handle_flag()
{
    __stripe_debug "${FUNCNAME[0]}: c is $c words[c] is ${words[c]}"

    # if a command required a flag, and we found it, unset must_have_one_flag()
    local flagname=${words[c]}
    local flagvalue
    # if the word contained an =
    if [[ ${words[c]} == *"="* ]]; then
        flagvalue=${flagname#*=} # take in as flagvalue after the =
        flagname=${flagname%=*} # strip everything after the =
        flagname="${flagname}=" # but put the = back
    fi
    __stripe_debug "${FUNCNAME[0]}: looking for ${flagname}"
    if __stripe_contains_word "${flagname}" "${must_have_one_flag[@]}"; then
        must_have_one_flag=()
    fi

    # if you set a flag which only applies to this command, don't show subcommands
    if __stripe_contains_word "${flagname}" "${local_nonpersistent_flags[@]}"; then
      commands=()
    fi

    # keep flag value with flagname as flaghash
    # flaghash variable is an associative array which is only supported in bash > 3.
    if [[ -z "${BASH_VERSION}" || "${BASH_VERSINFO[0]}" -gt 3 ]]; then
        if [ -n "${flagvalue}" ] ; then
            flaghash[${flagname}]=${flagvalue}
        elif [ -n "${words[ $((c+1)) ]}" ] ; then
            flaghash[${flagname}]=${words[ $((c+1)) ]}
        else
            flaghash[${flagname}]="true" # pad "true" for bool flag
        fi
    fi

    # skip the argument to a two word flag
    if [[ ${words[c]} != *"="* ]] && __stripe_contains_word "${words[c]}" "${two_word_flags[@]}"; then
			  __stripe_debug "${FUNCNAME[0]}: found a flag ${words[c]}, skip the next argument"
        c=$((c+1))
        # if we are looking for a flags value, don't show commands
        if [[ $c -eq $cword ]]; then
            commands=()
        fi
    fi

    c=$((c+1))

}

__stripe_handle_noun()
{
    __stripe_debug "${FUNCNAME[0]}: c is $c words[c] is ${words[c]}"

    if __stripe_contains_word "${words[c]}" "${must_have_one_noun[@]}"; then
        must_have_one_noun=()
    elif __stripe_contains_word "${words[c]}" "${noun_aliases[@]}"; then
        must_have_one_noun=()
    fi

    nouns+=("${words[c]}")
    c=$((c+1))
}

__stripe_handle_command()
{
    __stripe_debug "${FUNCNAME[0]}: c is $c words[c] is ${words[c]}"

    local next_command
    if [[ -n ${last_command} ]]; then
        next_command="_${last_command}_${words[c]//:/__}"
    else
        if [[ $c -eq 0 ]]; then
            next_command="_stripe_root_command"
        else
            next_command="_${words[c]//:/__}"
        fi
    fi
    c=$((c+1))
    __stripe_debug "${FUNCNAME[0]}: looking for ${next_command}"
    declare -F "$next_command" >/dev/null && $next_command
}

__stripe_handle_word()
{
    if [[ $c -ge $cword ]]; then
        __stripe_handle_reply
        return
    fi
    __stripe_debug "${FUNCNAME[0]}: c is $c words[c] is ${words[c]}"
    if [[ "${words[c]}" == -* ]]; then
        __stripe_handle_flag
    elif __stripe_contains_word "${words[c]}" "${commands[@]}"; then
        __stripe_handle_command
    elif [[ $c -eq 0 ]]; then
        __stripe_handle_command
    elif __stripe_contains_word "${words[c]}" "${command_aliases[@]}"; then
        # aliashash variable is an associative array which is only supported in bash > 3.
        if [[ -z "${BASH_VERSION}" || "${BASH_VERSINFO[0]}" -gt 3 ]]; then
            words[c]=${aliashash[${words[c]}]}
            __stripe_handle_command
        else
            __stripe_handle_noun
        fi
    else
        __stripe_handle_noun
    fi
    __stripe_handle_word
}

_stripe_3d_secure_create()
{
    last_command="stripe_3d_secure_create"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--amount=")
    two_word_flags+=("--amount")
    local_nonpersistent_flags+=("--amount")
    local_nonpersistent_flags+=("--amount=")
    flags+=("--card=")
    two_word_flags+=("--card")
    local_nonpersistent_flags+=("--card")
    local_nonpersistent_flags+=("--card=")
    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--currency=")
    two_word_flags+=("--currency")
    local_nonpersistent_flags+=("--currency")
    local_nonpersistent_flags+=("--currency=")
    flags+=("--customer=")
    two_word_flags+=("--customer")
    local_nonpersistent_flags+=("--customer")
    local_nonpersistent_flags+=("--customer=")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--return-url=")
    two_word_flags+=("--return-url")
    local_nonpersistent_flags+=("--return-url")
    local_nonpersistent_flags+=("--return-url=")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_3d_secure_retrieve()
{
    last_command="stripe_3d_secure_retrieve"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--ending-before=")
    two_word_flags+=("--ending-before")
    two_word_flags+=("-b")
    local_nonpersistent_flags+=("--ending-before")
    local_nonpersistent_flags+=("--ending-before=")
    local_nonpersistent_flags+=("-b")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--limit=")
    two_word_flags+=("--limit")
    two_word_flags+=("-l")
    local_nonpersistent_flags+=("--limit")
    local_nonpersistent_flags+=("--limit=")
    local_nonpersistent_flags+=("-l")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--starting-after=")
    two_word_flags+=("--starting-after")
    two_word_flags+=("-a")
    local_nonpersistent_flags+=("--starting-after")
    local_nonpersistent_flags+=("--starting-after=")
    local_nonpersistent_flags+=("-a")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_3d_secure()
{
    last_command="stripe_3d_secure"

    command_aliases=()

    commands=()
    commands+=("create")
    commands+=("retrieve")

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_account_links_create()
{
    last_command="stripe_account_links_create"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--account=")
    two_word_flags+=("--account")
    local_nonpersistent_flags+=("--account")
    local_nonpersistent_flags+=("--account=")
    flags+=("--collect=")
    two_word_flags+=("--collect")
    local_nonpersistent_flags+=("--collect")
    local_nonpersistent_flags+=("--collect=")
    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--refresh-url=")
    two_word_flags+=("--refresh-url")
    local_nonpersistent_flags+=("--refresh-url")
    local_nonpersistent_flags+=("--refresh-url=")
    flags+=("--return-url=")
    two_word_flags+=("--return-url")
    local_nonpersistent_flags+=("--return-url")
    local_nonpersistent_flags+=("--return-url=")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--type=")
    two_word_flags+=("--type")
    local_nonpersistent_flags+=("--type")
    local_nonpersistent_flags+=("--type=")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_account_links()
{
    last_command="stripe_account_links"

    command_aliases=()

    commands=()
    commands+=("create")

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_accounts_capabilities()
{
    last_command="stripe_accounts_capabilities"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--ending-before=")
    two_word_flags+=("--ending-before")
    two_word_flags+=("-b")
    local_nonpersistent_flags+=("--ending-before")
    local_nonpersistent_flags+=("--ending-before=")
    local_nonpersistent_flags+=("-b")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--limit=")
    two_word_flags+=("--limit")
    two_word_flags+=("-l")
    local_nonpersistent_flags+=("--limit")
    local_nonpersistent_flags+=("--limit=")
    local_nonpersistent_flags+=("-l")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--starting-after=")
    two_word_flags+=("--starting-after")
    two_word_flags+=("-a")
    local_nonpersistent_flags+=("--starting-after")
    local_nonpersistent_flags+=("--starting-after=")
    local_nonpersistent_flags+=("-a")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_accounts_create()
{
    last_command="stripe_accounts_create"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--account-token=")
    two_word_flags+=("--account-token")
    local_nonpersistent_flags+=("--account-token")
    local_nonpersistent_flags+=("--account-token=")
    flags+=("--business-type=")
    two_word_flags+=("--business-type")
    local_nonpersistent_flags+=("--business-type")
    local_nonpersistent_flags+=("--business-type=")
    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--country=")
    two_word_flags+=("--country")
    local_nonpersistent_flags+=("--country")
    local_nonpersistent_flags+=("--country=")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--default-currency=")
    two_word_flags+=("--default-currency")
    local_nonpersistent_flags+=("--default-currency")
    local_nonpersistent_flags+=("--default-currency=")
    flags+=("--email=")
    two_word_flags+=("--email")
    local_nonpersistent_flags+=("--email")
    local_nonpersistent_flags+=("--email=")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--external-account=")
    two_word_flags+=("--external-account")
    local_nonpersistent_flags+=("--external-account")
    local_nonpersistent_flags+=("--external-account=")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--type=")
    two_word_flags+=("--type")
    local_nonpersistent_flags+=("--type")
    local_nonpersistent_flags+=("--type=")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_accounts_delete()
{
    last_command="stripe_accounts_delete"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_accounts_list()
{
    last_command="stripe_accounts_list"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--created=")
    two_word_flags+=("--created")
    local_nonpersistent_flags+=("--created")
    local_nonpersistent_flags+=("--created=")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--ending-before=")
    two_word_flags+=("--ending-before")
    local_nonpersistent_flags+=("--ending-before")
    local_nonpersistent_flags+=("--ending-before=")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--limit=")
    two_word_flags+=("--limit")
    local_nonpersistent_flags+=("--limit")
    local_nonpersistent_flags+=("--limit=")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--starting-after=")
    two_word_flags+=("--starting-after")
    local_nonpersistent_flags+=("--starting-after")
    local_nonpersistent_flags+=("--starting-after=")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_accounts_reject()
{
    last_command="stripe_accounts_reject"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--reason=")
    two_word_flags+=("--reason")
    local_nonpersistent_flags+=("--reason")
    local_nonpersistent_flags+=("--reason=")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_accounts_retrieve()
{
    last_command="stripe_accounts_retrieve"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--ending-before=")
    two_word_flags+=("--ending-before")
    two_word_flags+=("-b")
    local_nonpersistent_flags+=("--ending-before")
    local_nonpersistent_flags+=("--ending-before=")
    local_nonpersistent_flags+=("-b")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--limit=")
    two_word_flags+=("--limit")
    two_word_flags+=("-l")
    local_nonpersistent_flags+=("--limit")
    local_nonpersistent_flags+=("--limit=")
    local_nonpersistent_flags+=("-l")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--starting-after=")
    two_word_flags+=("--starting-after")
    two_word_flags+=("-a")
    local_nonpersistent_flags+=("--starting-after")
    local_nonpersistent_flags+=("--starting-after=")
    local_nonpersistent_flags+=("-a")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_accounts_update()
{
    last_command="stripe_accounts_update"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--account-token=")
    two_word_flags+=("--account-token")
    local_nonpersistent_flags+=("--account-token")
    local_nonpersistent_flags+=("--account-token=")
    flags+=("--business-type=")
    two_word_flags+=("--business-type")
    local_nonpersistent_flags+=("--business-type")
    local_nonpersistent_flags+=("--business-type=")
    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--default-currency=")
    two_word_flags+=("--default-currency")
    local_nonpersistent_flags+=("--default-currency")
    local_nonpersistent_flags+=("--default-currency=")
    flags+=("--email=")
    two_word_flags+=("--email")
    local_nonpersistent_flags+=("--email")
    local_nonpersistent_flags+=("--email=")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--external-account=")
    two_word_flags+=("--external-account")
    local_nonpersistent_flags+=("--external-account")
    local_nonpersistent_flags+=("--external-account=")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_accounts()
{
    last_command="stripe_accounts"

    command_aliases=()

    commands=()
    commands+=("capabilities")
    commands+=("create")
    commands+=("delete")
    commands+=("list")
    commands+=("reject")
    commands+=("retrieve")
    commands+=("update")

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_apple_pay_domains_create()
{
    last_command="stripe_apple_pay_domains_create"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--domain-name=")
    two_word_flags+=("--domain-name")
    local_nonpersistent_flags+=("--domain-name")
    local_nonpersistent_flags+=("--domain-name=")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_apple_pay_domains_delete()
{
    last_command="stripe_apple_pay_domains_delete"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_apple_pay_domains_list()
{
    last_command="stripe_apple_pay_domains_list"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--domain-name=")
    two_word_flags+=("--domain-name")
    local_nonpersistent_flags+=("--domain-name")
    local_nonpersistent_flags+=("--domain-name=")
    flags+=("--ending-before=")
    two_word_flags+=("--ending-before")
    local_nonpersistent_flags+=("--ending-before")
    local_nonpersistent_flags+=("--ending-before=")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--limit=")
    two_word_flags+=("--limit")
    local_nonpersistent_flags+=("--limit")
    local_nonpersistent_flags+=("--limit=")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--starting-after=")
    two_word_flags+=("--starting-after")
    local_nonpersistent_flags+=("--starting-after")
    local_nonpersistent_flags+=("--starting-after=")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_apple_pay_domains_retrieve()
{
    last_command="stripe_apple_pay_domains_retrieve"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--ending-before=")
    two_word_flags+=("--ending-before")
    two_word_flags+=("-b")
    local_nonpersistent_flags+=("--ending-before")
    local_nonpersistent_flags+=("--ending-before=")
    local_nonpersistent_flags+=("-b")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--limit=")
    two_word_flags+=("--limit")
    two_word_flags+=("-l")
    local_nonpersistent_flags+=("--limit")
    local_nonpersistent_flags+=("--limit=")
    local_nonpersistent_flags+=("-l")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--starting-after=")
    two_word_flags+=("--starting-after")
    two_word_flags+=("-a")
    local_nonpersistent_flags+=("--starting-after")
    local_nonpersistent_flags+=("--starting-after=")
    local_nonpersistent_flags+=("-a")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_apple_pay_domains()
{
    last_command="stripe_apple_pay_domains"

    command_aliases=()

    commands=()
    commands+=("create")
    commands+=("delete")
    commands+=("list")
    commands+=("retrieve")

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_application_fees_list()
{
    last_command="stripe_application_fees_list"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--charge=")
    two_word_flags+=("--charge")
    local_nonpersistent_flags+=("--charge")
    local_nonpersistent_flags+=("--charge=")
    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--created=")
    two_word_flags+=("--created")
    local_nonpersistent_flags+=("--created")
    local_nonpersistent_flags+=("--created=")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--ending-before=")
    two_word_flags+=("--ending-before")
    local_nonpersistent_flags+=("--ending-before")
    local_nonpersistent_flags+=("--ending-before=")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--limit=")
    two_word_flags+=("--limit")
    local_nonpersistent_flags+=("--limit")
    local_nonpersistent_flags+=("--limit=")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--starting-after=")
    two_word_flags+=("--starting-after")
    local_nonpersistent_flags+=("--starting-after")
    local_nonpersistent_flags+=("--starting-after=")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_application_fees_retrieve()
{
    last_command="stripe_application_fees_retrieve"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--ending-before=")
    two_word_flags+=("--ending-before")
    two_word_flags+=("-b")
    local_nonpersistent_flags+=("--ending-before")
    local_nonpersistent_flags+=("--ending-before=")
    local_nonpersistent_flags+=("-b")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--limit=")
    two_word_flags+=("--limit")
    two_word_flags+=("-l")
    local_nonpersistent_flags+=("--limit")
    local_nonpersistent_flags+=("--limit=")
    local_nonpersistent_flags+=("-l")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--starting-after=")
    two_word_flags+=("--starting-after")
    two_word_flags+=("-a")
    local_nonpersistent_flags+=("--starting-after")
    local_nonpersistent_flags+=("--starting-after=")
    local_nonpersistent_flags+=("-a")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_application_fees()
{
    last_command="stripe_application_fees"

    command_aliases=()

    commands=()
    commands+=("list")
    commands+=("retrieve")

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_balance_retrieve()
{
    last_command="stripe_balance_retrieve"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--ending-before=")
    two_word_flags+=("--ending-before")
    two_word_flags+=("-b")
    local_nonpersistent_flags+=("--ending-before")
    local_nonpersistent_flags+=("--ending-before=")
    local_nonpersistent_flags+=("-b")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--limit=")
    two_word_flags+=("--limit")
    two_word_flags+=("-l")
    local_nonpersistent_flags+=("--limit")
    local_nonpersistent_flags+=("--limit=")
    local_nonpersistent_flags+=("-l")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--starting-after=")
    two_word_flags+=("--starting-after")
    two_word_flags+=("-a")
    local_nonpersistent_flags+=("--starting-after")
    local_nonpersistent_flags+=("--starting-after=")
    local_nonpersistent_flags+=("-a")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_balance()
{
    last_command="stripe_balance"

    command_aliases=()

    commands=()
    commands+=("retrieve")

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_balance_transactions_list()
{
    last_command="stripe_balance_transactions_list"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--available-on=")
    two_word_flags+=("--available-on")
    local_nonpersistent_flags+=("--available-on")
    local_nonpersistent_flags+=("--available-on=")
    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--created=")
    two_word_flags+=("--created")
    local_nonpersistent_flags+=("--created")
    local_nonpersistent_flags+=("--created=")
    flags+=("--currency=")
    two_word_flags+=("--currency")
    local_nonpersistent_flags+=("--currency")
    local_nonpersistent_flags+=("--currency=")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--ending-before=")
    two_word_flags+=("--ending-before")
    local_nonpersistent_flags+=("--ending-before")
    local_nonpersistent_flags+=("--ending-before=")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--limit=")
    two_word_flags+=("--limit")
    local_nonpersistent_flags+=("--limit")
    local_nonpersistent_flags+=("--limit=")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--payout=")
    two_word_flags+=("--payout")
    local_nonpersistent_flags+=("--payout")
    local_nonpersistent_flags+=("--payout=")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--source=")
    two_word_flags+=("--source")
    local_nonpersistent_flags+=("--source")
    local_nonpersistent_flags+=("--source=")
    flags+=("--starting-after=")
    two_word_flags+=("--starting-after")
    local_nonpersistent_flags+=("--starting-after")
    local_nonpersistent_flags+=("--starting-after=")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--type=")
    two_word_flags+=("--type")
    local_nonpersistent_flags+=("--type")
    local_nonpersistent_flags+=("--type=")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_balance_transactions_retrieve()
{
    last_command="stripe_balance_transactions_retrieve"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--ending-before=")
    two_word_flags+=("--ending-before")
    two_word_flags+=("-b")
    local_nonpersistent_flags+=("--ending-before")
    local_nonpersistent_flags+=("--ending-before=")
    local_nonpersistent_flags+=("-b")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--limit=")
    two_word_flags+=("--limit")
    two_word_flags+=("-l")
    local_nonpersistent_flags+=("--limit")
    local_nonpersistent_flags+=("--limit=")
    local_nonpersistent_flags+=("-l")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--starting-after=")
    two_word_flags+=("--starting-after")
    two_word_flags+=("-a")
    local_nonpersistent_flags+=("--starting-after")
    local_nonpersistent_flags+=("--starting-after=")
    local_nonpersistent_flags+=("-a")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_balance_transactions()
{
    last_command="stripe_balance_transactions"

    command_aliases=()

    commands=()
    commands+=("list")
    commands+=("retrieve")

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_bank_accounts_delete()
{
    last_command="stripe_bank_accounts_delete"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_bank_accounts_update()
{
    last_command="stripe_bank_accounts_update"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--account-holder-name=")
    two_word_flags+=("--account-holder-name")
    local_nonpersistent_flags+=("--account-holder-name")
    local_nonpersistent_flags+=("--account-holder-name=")
    flags+=("--account-holder-type=")
    two_word_flags+=("--account-holder-type")
    local_nonpersistent_flags+=("--account-holder-type")
    local_nonpersistent_flags+=("--account-holder-type=")
    flags+=("--address-city=")
    two_word_flags+=("--address-city")
    local_nonpersistent_flags+=("--address-city")
    local_nonpersistent_flags+=("--address-city=")
    flags+=("--address-country=")
    two_word_flags+=("--address-country")
    local_nonpersistent_flags+=("--address-country")
    local_nonpersistent_flags+=("--address-country=")
    flags+=("--address-line1=")
    two_word_flags+=("--address-line1")
    local_nonpersistent_flags+=("--address-line1")
    local_nonpersistent_flags+=("--address-line1=")
    flags+=("--address-line2=")
    two_word_flags+=("--address-line2")
    local_nonpersistent_flags+=("--address-line2")
    local_nonpersistent_flags+=("--address-line2=")
    flags+=("--address-state=")
    two_word_flags+=("--address-state")
    local_nonpersistent_flags+=("--address-state")
    local_nonpersistent_flags+=("--address-state=")
    flags+=("--address-zip=")
    two_word_flags+=("--address-zip")
    local_nonpersistent_flags+=("--address-zip")
    local_nonpersistent_flags+=("--address-zip=")
    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--exp-month=")
    two_word_flags+=("--exp-month")
    local_nonpersistent_flags+=("--exp-month")
    local_nonpersistent_flags+=("--exp-month=")
    flags+=("--exp-year=")
    two_word_flags+=("--exp-year")
    local_nonpersistent_flags+=("--exp-year")
    local_nonpersistent_flags+=("--exp-year=")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--name=")
    two_word_flags+=("--name")
    local_nonpersistent_flags+=("--name")
    local_nonpersistent_flags+=("--name=")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_bank_accounts_verify()
{
    last_command="stripe_bank_accounts_verify"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_bank_accounts()
{
    last_command="stripe_bank_accounts"

    command_aliases=()

    commands=()
    commands+=("delete")
    commands+=("update")
    commands+=("verify")

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_billing_portal_configurations_create()
{
    last_command="stripe_billing_portal_configurations_create"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--default-return-url=")
    two_word_flags+=("--default-return-url")
    local_nonpersistent_flags+=("--default-return-url")
    local_nonpersistent_flags+=("--default-return-url=")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_billing_portal_configurations_list()
{
    last_command="stripe_billing_portal_configurations_list"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--active=")
    two_word_flags+=("--active")
    local_nonpersistent_flags+=("--active")
    local_nonpersistent_flags+=("--active=")
    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--ending-before=")
    two_word_flags+=("--ending-before")
    local_nonpersistent_flags+=("--ending-before")
    local_nonpersistent_flags+=("--ending-before=")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--is-default=")
    two_word_flags+=("--is-default")
    local_nonpersistent_flags+=("--is-default")
    local_nonpersistent_flags+=("--is-default=")
    flags+=("--limit=")
    two_word_flags+=("--limit")
    local_nonpersistent_flags+=("--limit")
    local_nonpersistent_flags+=("--limit=")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--starting-after=")
    two_word_flags+=("--starting-after")
    local_nonpersistent_flags+=("--starting-after")
    local_nonpersistent_flags+=("--starting-after=")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_billing_portal_configurations_retrieve()
{
    last_command="stripe_billing_portal_configurations_retrieve"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--ending-before=")
    two_word_flags+=("--ending-before")
    two_word_flags+=("-b")
    local_nonpersistent_flags+=("--ending-before")
    local_nonpersistent_flags+=("--ending-before=")
    local_nonpersistent_flags+=("-b")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--limit=")
    two_word_flags+=("--limit")
    two_word_flags+=("-l")
    local_nonpersistent_flags+=("--limit")
    local_nonpersistent_flags+=("--limit=")
    local_nonpersistent_flags+=("-l")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--starting-after=")
    two_word_flags+=("--starting-after")
    two_word_flags+=("-a")
    local_nonpersistent_flags+=("--starting-after")
    local_nonpersistent_flags+=("--starting-after=")
    local_nonpersistent_flags+=("-a")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_billing_portal_configurations_update()
{
    last_command="stripe_billing_portal_configurations_update"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--active=")
    two_word_flags+=("--active")
    local_nonpersistent_flags+=("--active")
    local_nonpersistent_flags+=("--active=")
    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--default-return-url=")
    two_word_flags+=("--default-return-url")
    local_nonpersistent_flags+=("--default-return-url")
    local_nonpersistent_flags+=("--default-return-url=")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_billing_portal_configurations()
{
    last_command="stripe_billing_portal_configurations"

    command_aliases=()

    commands=()
    commands+=("create")
    commands+=("list")
    commands+=("retrieve")
    commands+=("update")

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_billing_portal_sessions_create()
{
    last_command="stripe_billing_portal_sessions_create"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--configuration=")
    two_word_flags+=("--configuration")
    local_nonpersistent_flags+=("--configuration")
    local_nonpersistent_flags+=("--configuration=")
    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--customer=")
    two_word_flags+=("--customer")
    local_nonpersistent_flags+=("--customer")
    local_nonpersistent_flags+=("--customer=")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--locale=")
    two_word_flags+=("--locale")
    local_nonpersistent_flags+=("--locale")
    local_nonpersistent_flags+=("--locale=")
    flags+=("--on-behalf-of=")
    two_word_flags+=("--on-behalf-of")
    local_nonpersistent_flags+=("--on-behalf-of")
    local_nonpersistent_flags+=("--on-behalf-of=")
    flags+=("--return-url=")
    two_word_flags+=("--return-url")
    local_nonpersistent_flags+=("--return-url")
    local_nonpersistent_flags+=("--return-url=")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_billing_portal_sessions()
{
    last_command="stripe_billing_portal_sessions"

    command_aliases=()

    commands=()
    commands+=("create")

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_billing_portal()
{
    last_command="stripe_billing_portal"

    command_aliases=()

    commands=()
    commands+=("configurations")
    commands+=("sessions")

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_capabilities_list()
{
    last_command="stripe_capabilities_list"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--ending-before=")
    two_word_flags+=("--ending-before")
    two_word_flags+=("-b")
    local_nonpersistent_flags+=("--ending-before")
    local_nonpersistent_flags+=("--ending-before=")
    local_nonpersistent_flags+=("-b")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--limit=")
    two_word_flags+=("--limit")
    two_word_flags+=("-l")
    local_nonpersistent_flags+=("--limit")
    local_nonpersistent_flags+=("--limit=")
    local_nonpersistent_flags+=("-l")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--starting-after=")
    two_word_flags+=("--starting-after")
    two_word_flags+=("-a")
    local_nonpersistent_flags+=("--starting-after")
    local_nonpersistent_flags+=("--starting-after=")
    local_nonpersistent_flags+=("-a")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_capabilities_retrieve()
{
    last_command="stripe_capabilities_retrieve"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--ending-before=")
    two_word_flags+=("--ending-before")
    two_word_flags+=("-b")
    local_nonpersistent_flags+=("--ending-before")
    local_nonpersistent_flags+=("--ending-before=")
    local_nonpersistent_flags+=("-b")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--limit=")
    two_word_flags+=("--limit")
    two_word_flags+=("-l")
    local_nonpersistent_flags+=("--limit")
    local_nonpersistent_flags+=("--limit=")
    local_nonpersistent_flags+=("-l")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--starting-after=")
    two_word_flags+=("--starting-after")
    two_word_flags+=("-a")
    local_nonpersistent_flags+=("--starting-after")
    local_nonpersistent_flags+=("--starting-after=")
    local_nonpersistent_flags+=("-a")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_capabilities_update()
{
    last_command="stripe_capabilities_update"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--requested=")
    two_word_flags+=("--requested")
    local_nonpersistent_flags+=("--requested")
    local_nonpersistent_flags+=("--requested=")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_capabilities()
{
    last_command="stripe_capabilities"

    command_aliases=()

    commands=()
    commands+=("list")
    commands+=("retrieve")
    commands+=("update")

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_cards_delete()
{
    last_command="stripe_cards_delete"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_cards_update()
{
    last_command="stripe_cards_update"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--account-holder-name=")
    two_word_flags+=("--account-holder-name")
    local_nonpersistent_flags+=("--account-holder-name")
    local_nonpersistent_flags+=("--account-holder-name=")
    flags+=("--account-holder-type=")
    two_word_flags+=("--account-holder-type")
    local_nonpersistent_flags+=("--account-holder-type")
    local_nonpersistent_flags+=("--account-holder-type=")
    flags+=("--address-city=")
    two_word_flags+=("--address-city")
    local_nonpersistent_flags+=("--address-city")
    local_nonpersistent_flags+=("--address-city=")
    flags+=("--address-country=")
    two_word_flags+=("--address-country")
    local_nonpersistent_flags+=("--address-country")
    local_nonpersistent_flags+=("--address-country=")
    flags+=("--address-line1=")
    two_word_flags+=("--address-line1")
    local_nonpersistent_flags+=("--address-line1")
    local_nonpersistent_flags+=("--address-line1=")
    flags+=("--address-line2=")
    two_word_flags+=("--address-line2")
    local_nonpersistent_flags+=("--address-line2")
    local_nonpersistent_flags+=("--address-line2=")
    flags+=("--address-state=")
    two_word_flags+=("--address-state")
    local_nonpersistent_flags+=("--address-state")
    local_nonpersistent_flags+=("--address-state=")
    flags+=("--address-zip=")
    two_word_flags+=("--address-zip")
    local_nonpersistent_flags+=("--address-zip")
    local_nonpersistent_flags+=("--address-zip=")
    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--exp-month=")
    two_word_flags+=("--exp-month")
    local_nonpersistent_flags+=("--exp-month")
    local_nonpersistent_flags+=("--exp-month=")
    flags+=("--exp-year=")
    two_word_flags+=("--exp-year")
    local_nonpersistent_flags+=("--exp-year")
    local_nonpersistent_flags+=("--exp-year=")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--name=")
    two_word_flags+=("--name")
    local_nonpersistent_flags+=("--name")
    local_nonpersistent_flags+=("--name=")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_cards()
{
    last_command="stripe_cards"

    command_aliases=()

    commands=()
    commands+=("delete")
    commands+=("update")

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_charges_capture()
{
    last_command="stripe_charges_capture"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--amount=")
    two_word_flags+=("--amount")
    local_nonpersistent_flags+=("--amount")
    local_nonpersistent_flags+=("--amount=")
    flags+=("--application-fee=")
    two_word_flags+=("--application-fee")
    local_nonpersistent_flags+=("--application-fee")
    local_nonpersistent_flags+=("--application-fee=")
    flags+=("--application-fee-amount=")
    two_word_flags+=("--application-fee-amount")
    local_nonpersistent_flags+=("--application-fee-amount")
    local_nonpersistent_flags+=("--application-fee-amount=")
    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--receipt-email=")
    two_word_flags+=("--receipt-email")
    local_nonpersistent_flags+=("--receipt-email")
    local_nonpersistent_flags+=("--receipt-email=")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--statement-descriptor=")
    two_word_flags+=("--statement-descriptor")
    local_nonpersistent_flags+=("--statement-descriptor")
    local_nonpersistent_flags+=("--statement-descriptor=")
    flags+=("--statement-descriptor-suffix=")
    two_word_flags+=("--statement-descriptor-suffix")
    local_nonpersistent_flags+=("--statement-descriptor-suffix")
    local_nonpersistent_flags+=("--statement-descriptor-suffix=")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--transfer-group=")
    two_word_flags+=("--transfer-group")
    local_nonpersistent_flags+=("--transfer-group")
    local_nonpersistent_flags+=("--transfer-group=")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_charges_create()
{
    last_command="stripe_charges_create"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--amount=")
    two_word_flags+=("--amount")
    local_nonpersistent_flags+=("--amount")
    local_nonpersistent_flags+=("--amount=")
    flags+=("--application-fee=")
    two_word_flags+=("--application-fee")
    local_nonpersistent_flags+=("--application-fee")
    local_nonpersistent_flags+=("--application-fee=")
    flags+=("--application-fee-amount=")
    two_word_flags+=("--application-fee-amount")
    local_nonpersistent_flags+=("--application-fee-amount")
    local_nonpersistent_flags+=("--application-fee-amount=")
    flags+=("--capture=")
    two_word_flags+=("--capture")
    local_nonpersistent_flags+=("--capture")
    local_nonpersistent_flags+=("--capture=")
    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--currency=")
    two_word_flags+=("--currency")
    local_nonpersistent_flags+=("--currency")
    local_nonpersistent_flags+=("--currency=")
    flags+=("--customer=")
    two_word_flags+=("--customer")
    local_nonpersistent_flags+=("--customer")
    local_nonpersistent_flags+=("--customer=")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--description=")
    two_word_flags+=("--description")
    local_nonpersistent_flags+=("--description")
    local_nonpersistent_flags+=("--description=")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--on-behalf-of=")
    two_word_flags+=("--on-behalf-of")
    local_nonpersistent_flags+=("--on-behalf-of")
    local_nonpersistent_flags+=("--on-behalf-of=")
    flags+=("--receipt-email=")
    two_word_flags+=("--receipt-email")
    local_nonpersistent_flags+=("--receipt-email")
    local_nonpersistent_flags+=("--receipt-email=")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--source=")
    two_word_flags+=("--source")
    local_nonpersistent_flags+=("--source")
    local_nonpersistent_flags+=("--source=")
    flags+=("--statement-descriptor=")
    two_word_flags+=("--statement-descriptor")
    local_nonpersistent_flags+=("--statement-descriptor")
    local_nonpersistent_flags+=("--statement-descriptor=")
    flags+=("--statement-descriptor-suffix=")
    two_word_flags+=("--statement-descriptor-suffix")
    local_nonpersistent_flags+=("--statement-descriptor-suffix")
    local_nonpersistent_flags+=("--statement-descriptor-suffix=")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--transfer-group=")
    two_word_flags+=("--transfer-group")
    local_nonpersistent_flags+=("--transfer-group")
    local_nonpersistent_flags+=("--transfer-group=")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_charges_list()
{
    last_command="stripe_charges_list"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--created=")
    two_word_flags+=("--created")
    local_nonpersistent_flags+=("--created")
    local_nonpersistent_flags+=("--created=")
    flags+=("--customer=")
    two_word_flags+=("--customer")
    local_nonpersistent_flags+=("--customer")
    local_nonpersistent_flags+=("--customer=")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--ending-before=")
    two_word_flags+=("--ending-before")
    local_nonpersistent_flags+=("--ending-before")
    local_nonpersistent_flags+=("--ending-before=")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--limit=")
    two_word_flags+=("--limit")
    local_nonpersistent_flags+=("--limit")
    local_nonpersistent_flags+=("--limit=")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--payment-intent=")
    two_word_flags+=("--payment-intent")
    local_nonpersistent_flags+=("--payment-intent")
    local_nonpersistent_flags+=("--payment-intent=")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--starting-after=")
    two_word_flags+=("--starting-after")
    local_nonpersistent_flags+=("--starting-after")
    local_nonpersistent_flags+=("--starting-after=")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--transfer-group=")
    two_word_flags+=("--transfer-group")
    local_nonpersistent_flags+=("--transfer-group")
    local_nonpersistent_flags+=("--transfer-group=")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_charges_retrieve()
{
    last_command="stripe_charges_retrieve"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--ending-before=")
    two_word_flags+=("--ending-before")
    two_word_flags+=("-b")
    local_nonpersistent_flags+=("--ending-before")
    local_nonpersistent_flags+=("--ending-before=")
    local_nonpersistent_flags+=("-b")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--limit=")
    two_word_flags+=("--limit")
    two_word_flags+=("-l")
    local_nonpersistent_flags+=("--limit")
    local_nonpersistent_flags+=("--limit=")
    local_nonpersistent_flags+=("-l")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--starting-after=")
    two_word_flags+=("--starting-after")
    two_word_flags+=("-a")
    local_nonpersistent_flags+=("--starting-after")
    local_nonpersistent_flags+=("--starting-after=")
    local_nonpersistent_flags+=("-a")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_charges_update()
{
    last_command="stripe_charges_update"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--customer=")
    two_word_flags+=("--customer")
    local_nonpersistent_flags+=("--customer")
    local_nonpersistent_flags+=("--customer=")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--description=")
    two_word_flags+=("--description")
    local_nonpersistent_flags+=("--description")
    local_nonpersistent_flags+=("--description=")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--receipt-email=")
    two_word_flags+=("--receipt-email")
    local_nonpersistent_flags+=("--receipt-email")
    local_nonpersistent_flags+=("--receipt-email=")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--transfer-group=")
    two_word_flags+=("--transfer-group")
    local_nonpersistent_flags+=("--transfer-group")
    local_nonpersistent_flags+=("--transfer-group=")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_charges()
{
    last_command="stripe_charges"

    command_aliases=()

    commands=()
    commands+=("capture")
    commands+=("create")
    commands+=("list")
    commands+=("retrieve")
    commands+=("update")

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_checkout_sessions_create()
{
    last_command="stripe_checkout_sessions_create"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--allow-promotion-codes=")
    two_word_flags+=("--allow-promotion-codes")
    local_nonpersistent_flags+=("--allow-promotion-codes")
    local_nonpersistent_flags+=("--allow-promotion-codes=")
    flags+=("--billing-address-collection=")
    two_word_flags+=("--billing-address-collection")
    local_nonpersistent_flags+=("--billing-address-collection")
    local_nonpersistent_flags+=("--billing-address-collection=")
    flags+=("--cancel-url=")
    two_word_flags+=("--cancel-url")
    local_nonpersistent_flags+=("--cancel-url")
    local_nonpersistent_flags+=("--cancel-url=")
    flags+=("--client-reference-id=")
    two_word_flags+=("--client-reference-id")
    local_nonpersistent_flags+=("--client-reference-id")
    local_nonpersistent_flags+=("--client-reference-id=")
    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--customer=")
    two_word_flags+=("--customer")
    local_nonpersistent_flags+=("--customer")
    local_nonpersistent_flags+=("--customer=")
    flags+=("--customer-email=")
    two_word_flags+=("--customer-email")
    local_nonpersistent_flags+=("--customer-email")
    local_nonpersistent_flags+=("--customer-email=")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--expires-at=")
    two_word_flags+=("--expires-at")
    local_nonpersistent_flags+=("--expires-at")
    local_nonpersistent_flags+=("--expires-at=")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--locale=")
    two_word_flags+=("--locale")
    local_nonpersistent_flags+=("--locale")
    local_nonpersistent_flags+=("--locale=")
    flags+=("--mode=")
    two_word_flags+=("--mode")
    local_nonpersistent_flags+=("--mode")
    local_nonpersistent_flags+=("--mode=")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--submit-type=")
    two_word_flags+=("--submit-type")
    local_nonpersistent_flags+=("--submit-type")
    local_nonpersistent_flags+=("--submit-type=")
    flags+=("--success-url=")
    two_word_flags+=("--success-url")
    local_nonpersistent_flags+=("--success-url")
    local_nonpersistent_flags+=("--success-url=")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_checkout_sessions_list()
{
    last_command="stripe_checkout_sessions_list"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--ending-before=")
    two_word_flags+=("--ending-before")
    local_nonpersistent_flags+=("--ending-before")
    local_nonpersistent_flags+=("--ending-before=")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--limit=")
    two_word_flags+=("--limit")
    local_nonpersistent_flags+=("--limit")
    local_nonpersistent_flags+=("--limit=")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--payment-intent=")
    two_word_flags+=("--payment-intent")
    local_nonpersistent_flags+=("--payment-intent")
    local_nonpersistent_flags+=("--payment-intent=")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--starting-after=")
    two_word_flags+=("--starting-after")
    local_nonpersistent_flags+=("--starting-after")
    local_nonpersistent_flags+=("--starting-after=")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--subscription=")
    two_word_flags+=("--subscription")
    local_nonpersistent_flags+=("--subscription")
    local_nonpersistent_flags+=("--subscription=")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_checkout_sessions_retrieve()
{
    last_command="stripe_checkout_sessions_retrieve"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--ending-before=")
    two_word_flags+=("--ending-before")
    two_word_flags+=("-b")
    local_nonpersistent_flags+=("--ending-before")
    local_nonpersistent_flags+=("--ending-before=")
    local_nonpersistent_flags+=("-b")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--limit=")
    two_word_flags+=("--limit")
    two_word_flags+=("-l")
    local_nonpersistent_flags+=("--limit")
    local_nonpersistent_flags+=("--limit=")
    local_nonpersistent_flags+=("-l")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--starting-after=")
    two_word_flags+=("--starting-after")
    two_word_flags+=("-a")
    local_nonpersistent_flags+=("--starting-after")
    local_nonpersistent_flags+=("--starting-after=")
    local_nonpersistent_flags+=("-a")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_checkout_sessions()
{
    last_command="stripe_checkout_sessions"

    command_aliases=()

    commands=()
    commands+=("create")
    commands+=("list")
    commands+=("retrieve")

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_checkout()
{
    last_command="stripe_checkout"

    command_aliases=()

    commands=()
    commands+=("sessions")

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_community()
{
    last_command="stripe_community"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_completion()
{
    last_command="stripe_completion"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--help")
    flags+=("-h")
    local_nonpersistent_flags+=("--help")
    local_nonpersistent_flags+=("-h")
    flags+=("--shell=")
    two_word_flags+=("--shell")
    local_nonpersistent_flags+=("--shell")
    local_nonpersistent_flags+=("--shell=")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_config()
{
    last_command="stripe_config"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--edit")
    flags+=("-e")
    local_nonpersistent_flags+=("--edit")
    local_nonpersistent_flags+=("-e")
    flags+=("--list")
    local_nonpersistent_flags+=("--list")
    flags+=("--set")
    local_nonpersistent_flags+=("--set")
    flags+=("--unset=")
    two_word_flags+=("--unset")
    local_nonpersistent_flags+=("--unset")
    local_nonpersistent_flags+=("--unset=")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_country_specs_list()
{
    last_command="stripe_country_specs_list"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--ending-before=")
    two_word_flags+=("--ending-before")
    local_nonpersistent_flags+=("--ending-before")
    local_nonpersistent_flags+=("--ending-before=")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--limit=")
    two_word_flags+=("--limit")
    local_nonpersistent_flags+=("--limit")
    local_nonpersistent_flags+=("--limit=")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--starting-after=")
    two_word_flags+=("--starting-after")
    local_nonpersistent_flags+=("--starting-after")
    local_nonpersistent_flags+=("--starting-after=")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_country_specs_retrieve()
{
    last_command="stripe_country_specs_retrieve"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--ending-before=")
    two_word_flags+=("--ending-before")
    two_word_flags+=("-b")
    local_nonpersistent_flags+=("--ending-before")
    local_nonpersistent_flags+=("--ending-before=")
    local_nonpersistent_flags+=("-b")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--limit=")
    two_word_flags+=("--limit")
    two_word_flags+=("-l")
    local_nonpersistent_flags+=("--limit")
    local_nonpersistent_flags+=("--limit=")
    local_nonpersistent_flags+=("-l")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--starting-after=")
    two_word_flags+=("--starting-after")
    two_word_flags+=("-a")
    local_nonpersistent_flags+=("--starting-after")
    local_nonpersistent_flags+=("--starting-after=")
    local_nonpersistent_flags+=("-a")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_country_specs()
{
    last_command="stripe_country_specs"

    command_aliases=()

    commands=()
    commands+=("list")
    commands+=("retrieve")

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_coupons_create()
{
    last_command="stripe_coupons_create"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--amount-off=")
    two_word_flags+=("--amount-off")
    local_nonpersistent_flags+=("--amount-off")
    local_nonpersistent_flags+=("--amount-off=")
    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--currency=")
    two_word_flags+=("--currency")
    local_nonpersistent_flags+=("--currency")
    local_nonpersistent_flags+=("--currency=")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--duration=")
    two_word_flags+=("--duration")
    local_nonpersistent_flags+=("--duration")
    local_nonpersistent_flags+=("--duration=")
    flags+=("--duration-in-months=")
    two_word_flags+=("--duration-in-months")
    local_nonpersistent_flags+=("--duration-in-months")
    local_nonpersistent_flags+=("--duration-in-months=")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--id=")
    two_word_flags+=("--id")
    local_nonpersistent_flags+=("--id")
    local_nonpersistent_flags+=("--id=")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--max-redemptions=")
    two_word_flags+=("--max-redemptions")
    local_nonpersistent_flags+=("--max-redemptions")
    local_nonpersistent_flags+=("--max-redemptions=")
    flags+=("--name=")
    two_word_flags+=("--name")
    local_nonpersistent_flags+=("--name")
    local_nonpersistent_flags+=("--name=")
    flags+=("--percent-off=")
    two_word_flags+=("--percent-off")
    local_nonpersistent_flags+=("--percent-off")
    local_nonpersistent_flags+=("--percent-off=")
    flags+=("--redeem-by=")
    two_word_flags+=("--redeem-by")
    local_nonpersistent_flags+=("--redeem-by")
    local_nonpersistent_flags+=("--redeem-by=")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_coupons_delete()
{
    last_command="stripe_coupons_delete"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_coupons_list()
{
    last_command="stripe_coupons_list"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--created=")
    two_word_flags+=("--created")
    local_nonpersistent_flags+=("--created")
    local_nonpersistent_flags+=("--created=")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--ending-before=")
    two_word_flags+=("--ending-before")
    local_nonpersistent_flags+=("--ending-before")
    local_nonpersistent_flags+=("--ending-before=")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--limit=")
    two_word_flags+=("--limit")
    local_nonpersistent_flags+=("--limit")
    local_nonpersistent_flags+=("--limit=")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--starting-after=")
    two_word_flags+=("--starting-after")
    local_nonpersistent_flags+=("--starting-after")
    local_nonpersistent_flags+=("--starting-after=")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_coupons_retrieve()
{
    last_command="stripe_coupons_retrieve"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--ending-before=")
    two_word_flags+=("--ending-before")
    two_word_flags+=("-b")
    local_nonpersistent_flags+=("--ending-before")
    local_nonpersistent_flags+=("--ending-before=")
    local_nonpersistent_flags+=("-b")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--limit=")
    two_word_flags+=("--limit")
    two_word_flags+=("-l")
    local_nonpersistent_flags+=("--limit")
    local_nonpersistent_flags+=("--limit=")
    local_nonpersistent_flags+=("-l")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--starting-after=")
    two_word_flags+=("--starting-after")
    two_word_flags+=("-a")
    local_nonpersistent_flags+=("--starting-after")
    local_nonpersistent_flags+=("--starting-after=")
    local_nonpersistent_flags+=("-a")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_coupons_update()
{
    last_command="stripe_coupons_update"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--name=")
    two_word_flags+=("--name")
    local_nonpersistent_flags+=("--name")
    local_nonpersistent_flags+=("--name=")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_coupons()
{
    last_command="stripe_coupons"

    command_aliases=()

    commands=()
    commands+=("create")
    commands+=("delete")
    commands+=("list")
    commands+=("retrieve")
    commands+=("update")

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_credit_note_line_items_list()
{
    last_command="stripe_credit_note_line_items_list"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--ending-before=")
    two_word_flags+=("--ending-before")
    local_nonpersistent_flags+=("--ending-before")
    local_nonpersistent_flags+=("--ending-before=")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--limit=")
    two_word_flags+=("--limit")
    local_nonpersistent_flags+=("--limit")
    local_nonpersistent_flags+=("--limit=")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--starting-after=")
    two_word_flags+=("--starting-after")
    local_nonpersistent_flags+=("--starting-after")
    local_nonpersistent_flags+=("--starting-after=")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_credit_note_line_items()
{
    last_command="stripe_credit_note_line_items"

    command_aliases=()

    commands=()
    commands+=("list")

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_credit_notes_create()
{
    last_command="stripe_credit_notes_create"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--amount=")
    two_word_flags+=("--amount")
    local_nonpersistent_flags+=("--amount")
    local_nonpersistent_flags+=("--amount=")
    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--credit-amount=")
    two_word_flags+=("--credit-amount")
    local_nonpersistent_flags+=("--credit-amount")
    local_nonpersistent_flags+=("--credit-amount=")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--invoice=")
    two_word_flags+=("--invoice")
    local_nonpersistent_flags+=("--invoice")
    local_nonpersistent_flags+=("--invoice=")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--memo=")
    two_word_flags+=("--memo")
    local_nonpersistent_flags+=("--memo")
    local_nonpersistent_flags+=("--memo=")
    flags+=("--out-of-band-amount=")
    two_word_flags+=("--out-of-band-amount")
    local_nonpersistent_flags+=("--out-of-band-amount")
    local_nonpersistent_flags+=("--out-of-band-amount=")
    flags+=("--reason=")
    two_word_flags+=("--reason")
    local_nonpersistent_flags+=("--reason")
    local_nonpersistent_flags+=("--reason=")
    flags+=("--refund=")
    two_word_flags+=("--refund")
    local_nonpersistent_flags+=("--refund")
    local_nonpersistent_flags+=("--refund=")
    flags+=("--refund-amount=")
    two_word_flags+=("--refund-amount")
    local_nonpersistent_flags+=("--refund-amount")
    local_nonpersistent_flags+=("--refund-amount=")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_credit_notes_list()
{
    last_command="stripe_credit_notes_list"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--customer=")
    two_word_flags+=("--customer")
    local_nonpersistent_flags+=("--customer")
    local_nonpersistent_flags+=("--customer=")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--ending-before=")
    two_word_flags+=("--ending-before")
    local_nonpersistent_flags+=("--ending-before")
    local_nonpersistent_flags+=("--ending-before=")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--invoice=")
    two_word_flags+=("--invoice")
    local_nonpersistent_flags+=("--invoice")
    local_nonpersistent_flags+=("--invoice=")
    flags+=("--limit=")
    two_word_flags+=("--limit")
    local_nonpersistent_flags+=("--limit")
    local_nonpersistent_flags+=("--limit=")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--starting-after=")
    two_word_flags+=("--starting-after")
    local_nonpersistent_flags+=("--starting-after")
    local_nonpersistent_flags+=("--starting-after=")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_credit_notes_preview()
{
    last_command="stripe_credit_notes_preview"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--amount=")
    two_word_flags+=("--amount")
    local_nonpersistent_flags+=("--amount")
    local_nonpersistent_flags+=("--amount=")
    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--credit-amount=")
    two_word_flags+=("--credit-amount")
    local_nonpersistent_flags+=("--credit-amount")
    local_nonpersistent_flags+=("--credit-amount=")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--ending-before=")
    two_word_flags+=("--ending-before")
    two_word_flags+=("-b")
    local_nonpersistent_flags+=("--ending-before")
    local_nonpersistent_flags+=("--ending-before=")
    local_nonpersistent_flags+=("-b")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--invoice=")
    two_word_flags+=("--invoice")
    local_nonpersistent_flags+=("--invoice")
    local_nonpersistent_flags+=("--invoice=")
    flags+=("--limit=")
    two_word_flags+=("--limit")
    two_word_flags+=("-l")
    local_nonpersistent_flags+=("--limit")
    local_nonpersistent_flags+=("--limit=")
    local_nonpersistent_flags+=("-l")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--memo=")
    two_word_flags+=("--memo")
    local_nonpersistent_flags+=("--memo")
    local_nonpersistent_flags+=("--memo=")
    flags+=("--out-of-band-amount=")
    two_word_flags+=("--out-of-band-amount")
    local_nonpersistent_flags+=("--out-of-band-amount")
    local_nonpersistent_flags+=("--out-of-band-amount=")
    flags+=("--reason=")
    two_word_flags+=("--reason")
    local_nonpersistent_flags+=("--reason")
    local_nonpersistent_flags+=("--reason=")
    flags+=("--refund=")
    two_word_flags+=("--refund")
    local_nonpersistent_flags+=("--refund")
    local_nonpersistent_flags+=("--refund=")
    flags+=("--refund-amount=")
    two_word_flags+=("--refund-amount")
    local_nonpersistent_flags+=("--refund-amount")
    local_nonpersistent_flags+=("--refund-amount=")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--starting-after=")
    two_word_flags+=("--starting-after")
    two_word_flags+=("-a")
    local_nonpersistent_flags+=("--starting-after")
    local_nonpersistent_flags+=("--starting-after=")
    local_nonpersistent_flags+=("-a")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_credit_notes_preview_lines()
{
    last_command="stripe_credit_notes_preview_lines"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--amount=")
    two_word_flags+=("--amount")
    local_nonpersistent_flags+=("--amount")
    local_nonpersistent_flags+=("--amount=")
    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--credit-amount=")
    two_word_flags+=("--credit-amount")
    local_nonpersistent_flags+=("--credit-amount")
    local_nonpersistent_flags+=("--credit-amount=")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--ending-before=")
    two_word_flags+=("--ending-before")
    local_nonpersistent_flags+=("--ending-before")
    local_nonpersistent_flags+=("--ending-before=")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--invoice=")
    two_word_flags+=("--invoice")
    local_nonpersistent_flags+=("--invoice")
    local_nonpersistent_flags+=("--invoice=")
    flags+=("--limit=")
    two_word_flags+=("--limit")
    local_nonpersistent_flags+=("--limit")
    local_nonpersistent_flags+=("--limit=")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--memo=")
    two_word_flags+=("--memo")
    local_nonpersistent_flags+=("--memo")
    local_nonpersistent_flags+=("--memo=")
    flags+=("--out-of-band-amount=")
    two_word_flags+=("--out-of-band-amount")
    local_nonpersistent_flags+=("--out-of-band-amount")
    local_nonpersistent_flags+=("--out-of-band-amount=")
    flags+=("--reason=")
    two_word_flags+=("--reason")
    local_nonpersistent_flags+=("--reason")
    local_nonpersistent_flags+=("--reason=")
    flags+=("--refund=")
    two_word_flags+=("--refund")
    local_nonpersistent_flags+=("--refund")
    local_nonpersistent_flags+=("--refund=")
    flags+=("--refund-amount=")
    two_word_flags+=("--refund-amount")
    local_nonpersistent_flags+=("--refund-amount")
    local_nonpersistent_flags+=("--refund-amount=")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--starting-after=")
    two_word_flags+=("--starting-after")
    local_nonpersistent_flags+=("--starting-after")
    local_nonpersistent_flags+=("--starting-after=")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_credit_notes_retrieve()
{
    last_command="stripe_credit_notes_retrieve"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--ending-before=")
    two_word_flags+=("--ending-before")
    two_word_flags+=("-b")
    local_nonpersistent_flags+=("--ending-before")
    local_nonpersistent_flags+=("--ending-before=")
    local_nonpersistent_flags+=("-b")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--limit=")
    two_word_flags+=("--limit")
    two_word_flags+=("-l")
    local_nonpersistent_flags+=("--limit")
    local_nonpersistent_flags+=("--limit=")
    local_nonpersistent_flags+=("-l")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--starting-after=")
    two_word_flags+=("--starting-after")
    two_word_flags+=("-a")
    local_nonpersistent_flags+=("--starting-after")
    local_nonpersistent_flags+=("--starting-after=")
    local_nonpersistent_flags+=("-a")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_credit_notes_update()
{
    last_command="stripe_credit_notes_update"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--memo=")
    two_word_flags+=("--memo")
    local_nonpersistent_flags+=("--memo")
    local_nonpersistent_flags+=("--memo=")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_credit_notes_void_credit_note()
{
    last_command="stripe_credit_notes_void_credit_note"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_credit_notes()
{
    last_command="stripe_credit_notes"

    command_aliases=()

    commands=()
    commands+=("create")
    commands+=("list")
    commands+=("preview")
    commands+=("preview_lines")
    commands+=("retrieve")
    commands+=("update")
    commands+=("void_credit_note")

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_customer_balance_transactions_create()
{
    last_command="stripe_customer_balance_transactions_create"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--amount=")
    two_word_flags+=("--amount")
    local_nonpersistent_flags+=("--amount")
    local_nonpersistent_flags+=("--amount=")
    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--currency=")
    two_word_flags+=("--currency")
    local_nonpersistent_flags+=("--currency")
    local_nonpersistent_flags+=("--currency=")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--description=")
    two_word_flags+=("--description")
    local_nonpersistent_flags+=("--description")
    local_nonpersistent_flags+=("--description=")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_customer_balance_transactions_list()
{
    last_command="stripe_customer_balance_transactions_list"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--ending-before=")
    two_word_flags+=("--ending-before")
    local_nonpersistent_flags+=("--ending-before")
    local_nonpersistent_flags+=("--ending-before=")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--limit=")
    two_word_flags+=("--limit")
    local_nonpersistent_flags+=("--limit")
    local_nonpersistent_flags+=("--limit=")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--starting-after=")
    two_word_flags+=("--starting-after")
    local_nonpersistent_flags+=("--starting-after")
    local_nonpersistent_flags+=("--starting-after=")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_customer_balance_transactions_retrieve()
{
    last_command="stripe_customer_balance_transactions_retrieve"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--ending-before=")
    two_word_flags+=("--ending-before")
    two_word_flags+=("-b")
    local_nonpersistent_flags+=("--ending-before")
    local_nonpersistent_flags+=("--ending-before=")
    local_nonpersistent_flags+=("-b")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--limit=")
    two_word_flags+=("--limit")
    two_word_flags+=("-l")
    local_nonpersistent_flags+=("--limit")
    local_nonpersistent_flags+=("--limit=")
    local_nonpersistent_flags+=("-l")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--starting-after=")
    two_word_flags+=("--starting-after")
    two_word_flags+=("-a")
    local_nonpersistent_flags+=("--starting-after")
    local_nonpersistent_flags+=("--starting-after=")
    local_nonpersistent_flags+=("-a")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_customer_balance_transactions_update()
{
    last_command="stripe_customer_balance_transactions_update"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--description=")
    two_word_flags+=("--description")
    local_nonpersistent_flags+=("--description")
    local_nonpersistent_flags+=("--description=")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_customer_balance_transactions()
{
    last_command="stripe_customer_balance_transactions"

    command_aliases=()

    commands=()
    commands+=("create")
    commands+=("list")
    commands+=("retrieve")
    commands+=("update")

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_customers_create()
{
    last_command="stripe_customers_create"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--balance=")
    two_word_flags+=("--balance")
    local_nonpersistent_flags+=("--balance")
    local_nonpersistent_flags+=("--balance=")
    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--coupon=")
    two_word_flags+=("--coupon")
    local_nonpersistent_flags+=("--coupon")
    local_nonpersistent_flags+=("--coupon=")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--description=")
    two_word_flags+=("--description")
    local_nonpersistent_flags+=("--description")
    local_nonpersistent_flags+=("--description=")
    flags+=("--email=")
    two_word_flags+=("--email")
    local_nonpersistent_flags+=("--email")
    local_nonpersistent_flags+=("--email=")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--invoice-prefix=")
    two_word_flags+=("--invoice-prefix")
    local_nonpersistent_flags+=("--invoice-prefix")
    local_nonpersistent_flags+=("--invoice-prefix=")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--name=")
    two_word_flags+=("--name")
    local_nonpersistent_flags+=("--name")
    local_nonpersistent_flags+=("--name=")
    flags+=("--next-invoice-sequence=")
    two_word_flags+=("--next-invoice-sequence")
    local_nonpersistent_flags+=("--next-invoice-sequence")
    local_nonpersistent_flags+=("--next-invoice-sequence=")
    flags+=("--payment-method=")
    two_word_flags+=("--payment-method")
    local_nonpersistent_flags+=("--payment-method")
    local_nonpersistent_flags+=("--payment-method=")
    flags+=("--phone=")
    two_word_flags+=("--phone")
    local_nonpersistent_flags+=("--phone")
    local_nonpersistent_flags+=("--phone=")
    flags+=("--promotion-code=")
    two_word_flags+=("--promotion-code")
    local_nonpersistent_flags+=("--promotion-code")
    local_nonpersistent_flags+=("--promotion-code=")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--source=")
    two_word_flags+=("--source")
    local_nonpersistent_flags+=("--source")
    local_nonpersistent_flags+=("--source=")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--tax-exempt=")
    two_word_flags+=("--tax-exempt")
    local_nonpersistent_flags+=("--tax-exempt")
    local_nonpersistent_flags+=("--tax-exempt=")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_customers_delete()
{
    last_command="stripe_customers_delete"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_customers_delete_discount()
{
    last_command="stripe_customers_delete_discount"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_customers_list()
{
    last_command="stripe_customers_list"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--created=")
    two_word_flags+=("--created")
    local_nonpersistent_flags+=("--created")
    local_nonpersistent_flags+=("--created=")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--email=")
    two_word_flags+=("--email")
    local_nonpersistent_flags+=("--email")
    local_nonpersistent_flags+=("--email=")
    flags+=("--ending-before=")
    two_word_flags+=("--ending-before")
    local_nonpersistent_flags+=("--ending-before")
    local_nonpersistent_flags+=("--ending-before=")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--limit=")
    two_word_flags+=("--limit")
    local_nonpersistent_flags+=("--limit")
    local_nonpersistent_flags+=("--limit=")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--starting-after=")
    two_word_flags+=("--starting-after")
    local_nonpersistent_flags+=("--starting-after")
    local_nonpersistent_flags+=("--starting-after=")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_customers_list_payment_methods()
{
    last_command="stripe_customers_list_payment_methods"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--ending-before=")
    two_word_flags+=("--ending-before")
    local_nonpersistent_flags+=("--ending-before")
    local_nonpersistent_flags+=("--ending-before=")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--limit=")
    two_word_flags+=("--limit")
    local_nonpersistent_flags+=("--limit")
    local_nonpersistent_flags+=("--limit=")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--starting-after=")
    two_word_flags+=("--starting-after")
    local_nonpersistent_flags+=("--starting-after")
    local_nonpersistent_flags+=("--starting-after=")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--type=")
    two_word_flags+=("--type")
    local_nonpersistent_flags+=("--type")
    local_nonpersistent_flags+=("--type=")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_customers_retrieve()
{
    last_command="stripe_customers_retrieve"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--ending-before=")
    two_word_flags+=("--ending-before")
    two_word_flags+=("-b")
    local_nonpersistent_flags+=("--ending-before")
    local_nonpersistent_flags+=("--ending-before=")
    local_nonpersistent_flags+=("-b")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--limit=")
    two_word_flags+=("--limit")
    two_word_flags+=("-l")
    local_nonpersistent_flags+=("--limit")
    local_nonpersistent_flags+=("--limit=")
    local_nonpersistent_flags+=("-l")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--starting-after=")
    two_word_flags+=("--starting-after")
    two_word_flags+=("-a")
    local_nonpersistent_flags+=("--starting-after")
    local_nonpersistent_flags+=("--starting-after=")
    local_nonpersistent_flags+=("-a")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_customers_update()
{
    last_command="stripe_customers_update"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--balance=")
    two_word_flags+=("--balance")
    local_nonpersistent_flags+=("--balance")
    local_nonpersistent_flags+=("--balance=")
    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--coupon=")
    two_word_flags+=("--coupon")
    local_nonpersistent_flags+=("--coupon")
    local_nonpersistent_flags+=("--coupon=")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--default-source=")
    two_word_flags+=("--default-source")
    local_nonpersistent_flags+=("--default-source")
    local_nonpersistent_flags+=("--default-source=")
    flags+=("--description=")
    two_word_flags+=("--description")
    local_nonpersistent_flags+=("--description")
    local_nonpersistent_flags+=("--description=")
    flags+=("--email=")
    two_word_flags+=("--email")
    local_nonpersistent_flags+=("--email")
    local_nonpersistent_flags+=("--email=")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--invoice-prefix=")
    two_word_flags+=("--invoice-prefix")
    local_nonpersistent_flags+=("--invoice-prefix")
    local_nonpersistent_flags+=("--invoice-prefix=")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--name=")
    two_word_flags+=("--name")
    local_nonpersistent_flags+=("--name")
    local_nonpersistent_flags+=("--name=")
    flags+=("--next-invoice-sequence=")
    two_word_flags+=("--next-invoice-sequence")
    local_nonpersistent_flags+=("--next-invoice-sequence")
    local_nonpersistent_flags+=("--next-invoice-sequence=")
    flags+=("--phone=")
    two_word_flags+=("--phone")
    local_nonpersistent_flags+=("--phone")
    local_nonpersistent_flags+=("--phone=")
    flags+=("--promotion-code=")
    two_word_flags+=("--promotion-code")
    local_nonpersistent_flags+=("--promotion-code")
    local_nonpersistent_flags+=("--promotion-code=")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--source=")
    two_word_flags+=("--source")
    local_nonpersistent_flags+=("--source")
    local_nonpersistent_flags+=("--source=")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--tax-exempt=")
    two_word_flags+=("--tax-exempt")
    local_nonpersistent_flags+=("--tax-exempt")
    local_nonpersistent_flags+=("--tax-exempt=")
    flags+=("--trial-end=")
    two_word_flags+=("--trial-end")
    local_nonpersistent_flags+=("--trial-end")
    local_nonpersistent_flags+=("--trial-end=")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_customers()
{
    last_command="stripe_customers"

    command_aliases=()

    commands=()
    commands+=("create")
    commands+=("delete")
    commands+=("delete_discount")
    commands+=("list")
    commands+=("list_payment_methods")
    commands+=("retrieve")
    commands+=("update")

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_delete()
{
    last_command="stripe_delete"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_disputes_close()
{
    last_command="stripe_disputes_close"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_disputes_list()
{
    last_command="stripe_disputes_list"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--charge=")
    two_word_flags+=("--charge")
    local_nonpersistent_flags+=("--charge")
    local_nonpersistent_flags+=("--charge=")
    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--created=")
    two_word_flags+=("--created")
    local_nonpersistent_flags+=("--created")
    local_nonpersistent_flags+=("--created=")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--ending-before=")
    two_word_flags+=("--ending-before")
    local_nonpersistent_flags+=("--ending-before")
    local_nonpersistent_flags+=("--ending-before=")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--limit=")
    two_word_flags+=("--limit")
    local_nonpersistent_flags+=("--limit")
    local_nonpersistent_flags+=("--limit=")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--payment-intent=")
    two_word_flags+=("--payment-intent")
    local_nonpersistent_flags+=("--payment-intent")
    local_nonpersistent_flags+=("--payment-intent=")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--starting-after=")
    two_word_flags+=("--starting-after")
    local_nonpersistent_flags+=("--starting-after")
    local_nonpersistent_flags+=("--starting-after=")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_disputes_retrieve()
{
    last_command="stripe_disputes_retrieve"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--ending-before=")
    two_word_flags+=("--ending-before")
    two_word_flags+=("-b")
    local_nonpersistent_flags+=("--ending-before")
    local_nonpersistent_flags+=("--ending-before=")
    local_nonpersistent_flags+=("-b")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--limit=")
    two_word_flags+=("--limit")
    two_word_flags+=("-l")
    local_nonpersistent_flags+=("--limit")
    local_nonpersistent_flags+=("--limit=")
    local_nonpersistent_flags+=("-l")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--starting-after=")
    two_word_flags+=("--starting-after")
    two_word_flags+=("-a")
    local_nonpersistent_flags+=("--starting-after")
    local_nonpersistent_flags+=("--starting-after=")
    local_nonpersistent_flags+=("-a")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_disputes_update()
{
    last_command="stripe_disputes_update"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--submit=")
    two_word_flags+=("--submit")
    local_nonpersistent_flags+=("--submit")
    local_nonpersistent_flags+=("--submit=")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_disputes()
{
    last_command="stripe_disputes"

    command_aliases=()

    commands=()
    commands+=("close")
    commands+=("list")
    commands+=("retrieve")
    commands+=("update")

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_ephemeral_keys_create()
{
    last_command="stripe_ephemeral_keys_create"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--customer=")
    two_word_flags+=("--customer")
    local_nonpersistent_flags+=("--customer")
    local_nonpersistent_flags+=("--customer=")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--issuing-card=")
    two_word_flags+=("--issuing-card")
    local_nonpersistent_flags+=("--issuing-card")
    local_nonpersistent_flags+=("--issuing-card=")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_ephemeral_keys_delete()
{
    last_command="stripe_ephemeral_keys_delete"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_ephemeral_keys()
{
    last_command="stripe_ephemeral_keys"

    command_aliases=()

    commands=()
    commands+=("create")
    commands+=("delete")

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_events_list()
{
    last_command="stripe_events_list"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--created=")
    two_word_flags+=("--created")
    local_nonpersistent_flags+=("--created")
    local_nonpersistent_flags+=("--created=")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--delivery-success=")
    two_word_flags+=("--delivery-success")
    local_nonpersistent_flags+=("--delivery-success")
    local_nonpersistent_flags+=("--delivery-success=")
    flags+=("--ending-before=")
    two_word_flags+=("--ending-before")
    local_nonpersistent_flags+=("--ending-before")
    local_nonpersistent_flags+=("--ending-before=")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--limit=")
    two_word_flags+=("--limit")
    local_nonpersistent_flags+=("--limit")
    local_nonpersistent_flags+=("--limit=")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--starting-after=")
    two_word_flags+=("--starting-after")
    local_nonpersistent_flags+=("--starting-after")
    local_nonpersistent_flags+=("--starting-after=")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--type=")
    two_word_flags+=("--type")
    local_nonpersistent_flags+=("--type")
    local_nonpersistent_flags+=("--type=")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_events_resend()
{
    last_command="stripe_events_resend"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--account=")
    two_word_flags+=("--account")
    local_nonpersistent_flags+=("--account")
    local_nonpersistent_flags+=("--account=")
    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--webhook-endpoint=")
    two_word_flags+=("--webhook-endpoint")
    local_nonpersistent_flags+=("--webhook-endpoint")
    local_nonpersistent_flags+=("--webhook-endpoint=")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_events_retrieve()
{
    last_command="stripe_events_retrieve"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--ending-before=")
    two_word_flags+=("--ending-before")
    two_word_flags+=("-b")
    local_nonpersistent_flags+=("--ending-before")
    local_nonpersistent_flags+=("--ending-before=")
    local_nonpersistent_flags+=("-b")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--limit=")
    two_word_flags+=("--limit")
    two_word_flags+=("-l")
    local_nonpersistent_flags+=("--limit")
    local_nonpersistent_flags+=("--limit=")
    local_nonpersistent_flags+=("-l")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--starting-after=")
    two_word_flags+=("--starting-after")
    two_word_flags+=("-a")
    local_nonpersistent_flags+=("--starting-after")
    local_nonpersistent_flags+=("--starting-after=")
    local_nonpersistent_flags+=("-a")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_events()
{
    last_command="stripe_events"

    command_aliases=()

    commands=()
    commands+=("list")
    commands+=("resend")
    commands+=("retrieve")

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_exchange_rates_list()
{
    last_command="stripe_exchange_rates_list"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--ending-before=")
    two_word_flags+=("--ending-before")
    local_nonpersistent_flags+=("--ending-before")
    local_nonpersistent_flags+=("--ending-before=")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--limit=")
    two_word_flags+=("--limit")
    local_nonpersistent_flags+=("--limit")
    local_nonpersistent_flags+=("--limit=")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--starting-after=")
    two_word_flags+=("--starting-after")
    local_nonpersistent_flags+=("--starting-after")
    local_nonpersistent_flags+=("--starting-after=")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_exchange_rates_retrieve()
{
    last_command="stripe_exchange_rates_retrieve"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--ending-before=")
    two_word_flags+=("--ending-before")
    two_word_flags+=("-b")
    local_nonpersistent_flags+=("--ending-before")
    local_nonpersistent_flags+=("--ending-before=")
    local_nonpersistent_flags+=("-b")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--limit=")
    two_word_flags+=("--limit")
    two_word_flags+=("-l")
    local_nonpersistent_flags+=("--limit")
    local_nonpersistent_flags+=("--limit=")
    local_nonpersistent_flags+=("-l")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--starting-after=")
    two_word_flags+=("--starting-after")
    two_word_flags+=("-a")
    local_nonpersistent_flags+=("--starting-after")
    local_nonpersistent_flags+=("--starting-after=")
    local_nonpersistent_flags+=("-a")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_exchange_rates()
{
    last_command="stripe_exchange_rates"

    command_aliases=()

    commands=()
    commands+=("list")
    commands+=("retrieve")

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_external_accounts_create()
{
    last_command="stripe_external_accounts_create"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--default-for-currency=")
    two_word_flags+=("--default-for-currency")
    local_nonpersistent_flags+=("--default-for-currency")
    local_nonpersistent_flags+=("--default-for-currency=")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--external-account=")
    two_word_flags+=("--external-account")
    local_nonpersistent_flags+=("--external-account")
    local_nonpersistent_flags+=("--external-account=")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_external_accounts_delete()
{
    last_command="stripe_external_accounts_delete"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_external_accounts_list()
{
    last_command="stripe_external_accounts_list"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--ending-before=")
    two_word_flags+=("--ending-before")
    local_nonpersistent_flags+=("--ending-before")
    local_nonpersistent_flags+=("--ending-before=")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--limit=")
    two_word_flags+=("--limit")
    local_nonpersistent_flags+=("--limit")
    local_nonpersistent_flags+=("--limit=")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--starting-after=")
    two_word_flags+=("--starting-after")
    local_nonpersistent_flags+=("--starting-after")
    local_nonpersistent_flags+=("--starting-after=")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_external_accounts_retrieve()
{
    last_command="stripe_external_accounts_retrieve"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--ending-before=")
    two_word_flags+=("--ending-before")
    two_word_flags+=("-b")
    local_nonpersistent_flags+=("--ending-before")
    local_nonpersistent_flags+=("--ending-before=")
    local_nonpersistent_flags+=("-b")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--limit=")
    two_word_flags+=("--limit")
    two_word_flags+=("-l")
    local_nonpersistent_flags+=("--limit")
    local_nonpersistent_flags+=("--limit=")
    local_nonpersistent_flags+=("-l")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--starting-after=")
    two_word_flags+=("--starting-after")
    two_word_flags+=("-a")
    local_nonpersistent_flags+=("--starting-after")
    local_nonpersistent_flags+=("--starting-after=")
    local_nonpersistent_flags+=("-a")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_external_accounts_update()
{
    last_command="stripe_external_accounts_update"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--account-holder-name=")
    two_word_flags+=("--account-holder-name")
    local_nonpersistent_flags+=("--account-holder-name")
    local_nonpersistent_flags+=("--account-holder-name=")
    flags+=("--account-holder-type=")
    two_word_flags+=("--account-holder-type")
    local_nonpersistent_flags+=("--account-holder-type")
    local_nonpersistent_flags+=("--account-holder-type=")
    flags+=("--account-type=")
    two_word_flags+=("--account-type")
    local_nonpersistent_flags+=("--account-type")
    local_nonpersistent_flags+=("--account-type=")
    flags+=("--address-city=")
    two_word_flags+=("--address-city")
    local_nonpersistent_flags+=("--address-city")
    local_nonpersistent_flags+=("--address-city=")
    flags+=("--address-country=")
    two_word_flags+=("--address-country")
    local_nonpersistent_flags+=("--address-country")
    local_nonpersistent_flags+=("--address-country=")
    flags+=("--address-line1=")
    two_word_flags+=("--address-line1")
    local_nonpersistent_flags+=("--address-line1")
    local_nonpersistent_flags+=("--address-line1=")
    flags+=("--address-line2=")
    two_word_flags+=("--address-line2")
    local_nonpersistent_flags+=("--address-line2")
    local_nonpersistent_flags+=("--address-line2=")
    flags+=("--address-state=")
    two_word_flags+=("--address-state")
    local_nonpersistent_flags+=("--address-state")
    local_nonpersistent_flags+=("--address-state=")
    flags+=("--address-zip=")
    two_word_flags+=("--address-zip")
    local_nonpersistent_flags+=("--address-zip")
    local_nonpersistent_flags+=("--address-zip=")
    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--default-for-currency=")
    two_word_flags+=("--default-for-currency")
    local_nonpersistent_flags+=("--default-for-currency")
    local_nonpersistent_flags+=("--default-for-currency=")
    flags+=("--exp-month=")
    two_word_flags+=("--exp-month")
    local_nonpersistent_flags+=("--exp-month")
    local_nonpersistent_flags+=("--exp-month=")
    flags+=("--exp-year=")
    two_word_flags+=("--exp-year")
    local_nonpersistent_flags+=("--exp-year")
    local_nonpersistent_flags+=("--exp-year=")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--name=")
    two_word_flags+=("--name")
    local_nonpersistent_flags+=("--name")
    local_nonpersistent_flags+=("--name=")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_external_accounts()
{
    last_command="stripe_external_accounts"

    command_aliases=()

    commands=()
    commands+=("create")
    commands+=("delete")
    commands+=("list")
    commands+=("retrieve")
    commands+=("update")

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_fee_refunds_create()
{
    last_command="stripe_fee_refunds_create"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--amount=")
    two_word_flags+=("--amount")
    local_nonpersistent_flags+=("--amount")
    local_nonpersistent_flags+=("--amount=")
    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_fee_refunds_list()
{
    last_command="stripe_fee_refunds_list"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--ending-before=")
    two_word_flags+=("--ending-before")
    local_nonpersistent_flags+=("--ending-before")
    local_nonpersistent_flags+=("--ending-before=")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--limit=")
    two_word_flags+=("--limit")
    local_nonpersistent_flags+=("--limit")
    local_nonpersistent_flags+=("--limit=")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--starting-after=")
    two_word_flags+=("--starting-after")
    local_nonpersistent_flags+=("--starting-after")
    local_nonpersistent_flags+=("--starting-after=")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_fee_refunds_retrieve()
{
    last_command="stripe_fee_refunds_retrieve"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--ending-before=")
    two_word_flags+=("--ending-before")
    two_word_flags+=("-b")
    local_nonpersistent_flags+=("--ending-before")
    local_nonpersistent_flags+=("--ending-before=")
    local_nonpersistent_flags+=("-b")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--limit=")
    two_word_flags+=("--limit")
    two_word_flags+=("-l")
    local_nonpersistent_flags+=("--limit")
    local_nonpersistent_flags+=("--limit=")
    local_nonpersistent_flags+=("-l")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--starting-after=")
    two_word_flags+=("--starting-after")
    two_word_flags+=("-a")
    local_nonpersistent_flags+=("--starting-after")
    local_nonpersistent_flags+=("--starting-after=")
    local_nonpersistent_flags+=("-a")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_fee_refunds_update()
{
    last_command="stripe_fee_refunds_update"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_fee_refunds()
{
    last_command="stripe_fee_refunds"

    command_aliases=()

    commands=()
    commands+=("create")
    commands+=("list")
    commands+=("retrieve")
    commands+=("update")

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_feedback()
{
    last_command="stripe_feedback"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_file_links_create()
{
    last_command="stripe_file_links_create"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--expires-at=")
    two_word_flags+=("--expires-at")
    local_nonpersistent_flags+=("--expires-at")
    local_nonpersistent_flags+=("--expires-at=")
    flags+=("--file=")
    two_word_flags+=("--file")
    local_nonpersistent_flags+=("--file")
    local_nonpersistent_flags+=("--file=")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_file_links_list()
{
    last_command="stripe_file_links_list"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--created=")
    two_word_flags+=("--created")
    local_nonpersistent_flags+=("--created")
    local_nonpersistent_flags+=("--created=")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--ending-before=")
    two_word_flags+=("--ending-before")
    local_nonpersistent_flags+=("--ending-before")
    local_nonpersistent_flags+=("--ending-before=")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--expired=")
    two_word_flags+=("--expired")
    local_nonpersistent_flags+=("--expired")
    local_nonpersistent_flags+=("--expired=")
    flags+=("--file=")
    two_word_flags+=("--file")
    local_nonpersistent_flags+=("--file")
    local_nonpersistent_flags+=("--file=")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--limit=")
    two_word_flags+=("--limit")
    local_nonpersistent_flags+=("--limit")
    local_nonpersistent_flags+=("--limit=")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--starting-after=")
    two_word_flags+=("--starting-after")
    local_nonpersistent_flags+=("--starting-after")
    local_nonpersistent_flags+=("--starting-after=")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_file_links_retrieve()
{
    last_command="stripe_file_links_retrieve"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--ending-before=")
    two_word_flags+=("--ending-before")
    two_word_flags+=("-b")
    local_nonpersistent_flags+=("--ending-before")
    local_nonpersistent_flags+=("--ending-before=")
    local_nonpersistent_flags+=("-b")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--limit=")
    two_word_flags+=("--limit")
    two_word_flags+=("-l")
    local_nonpersistent_flags+=("--limit")
    local_nonpersistent_flags+=("--limit=")
    local_nonpersistent_flags+=("-l")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--starting-after=")
    two_word_flags+=("--starting-after")
    two_word_flags+=("-a")
    local_nonpersistent_flags+=("--starting-after")
    local_nonpersistent_flags+=("--starting-after=")
    local_nonpersistent_flags+=("-a")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_file_links_update()
{
    last_command="stripe_file_links_update"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--expires-at=")
    two_word_flags+=("--expires-at")
    local_nonpersistent_flags+=("--expires-at")
    local_nonpersistent_flags+=("--expires-at=")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_file_links()
{
    last_command="stripe_file_links"

    command_aliases=()

    commands=()
    commands+=("create")
    commands+=("list")
    commands+=("retrieve")
    commands+=("update")

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_files_create()
{
    last_command="stripe_files_create"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_files_list()
{
    last_command="stripe_files_list"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--created=")
    two_word_flags+=("--created")
    local_nonpersistent_flags+=("--created")
    local_nonpersistent_flags+=("--created=")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--ending-before=")
    two_word_flags+=("--ending-before")
    local_nonpersistent_flags+=("--ending-before")
    local_nonpersistent_flags+=("--ending-before=")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--limit=")
    two_word_flags+=("--limit")
    local_nonpersistent_flags+=("--limit")
    local_nonpersistent_flags+=("--limit=")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--purpose=")
    two_word_flags+=("--purpose")
    local_nonpersistent_flags+=("--purpose")
    local_nonpersistent_flags+=("--purpose=")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--starting-after=")
    two_word_flags+=("--starting-after")
    local_nonpersistent_flags+=("--starting-after")
    local_nonpersistent_flags+=("--starting-after=")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_files_retrieve()
{
    last_command="stripe_files_retrieve"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--ending-before=")
    two_word_flags+=("--ending-before")
    two_word_flags+=("-b")
    local_nonpersistent_flags+=("--ending-before")
    local_nonpersistent_flags+=("--ending-before=")
    local_nonpersistent_flags+=("-b")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--limit=")
    two_word_flags+=("--limit")
    two_word_flags+=("-l")
    local_nonpersistent_flags+=("--limit")
    local_nonpersistent_flags+=("--limit=")
    local_nonpersistent_flags+=("-l")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--starting-after=")
    two_word_flags+=("--starting-after")
    two_word_flags+=("-a")
    local_nonpersistent_flags+=("--starting-after")
    local_nonpersistent_flags+=("--starting-after=")
    local_nonpersistent_flags+=("-a")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_files()
{
    last_command="stripe_files"

    command_aliases=()

    commands=()
    commands+=("create")
    commands+=("list")
    commands+=("retrieve")

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_fixtures()
{
    last_command="stripe_fixtures"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--add=")
    two_word_flags+=("--add")
    local_nonpersistent_flags+=("--add")
    local_nonpersistent_flags+=("--add=")
    flags+=("--override=")
    two_word_flags+=("--override")
    local_nonpersistent_flags+=("--override")
    local_nonpersistent_flags+=("--override=")
    flags+=("--remove=")
    two_word_flags+=("--remove")
    local_nonpersistent_flags+=("--remove")
    local_nonpersistent_flags+=("--remove=")
    flags+=("--skip=")
    two_word_flags+=("--skip")
    local_nonpersistent_flags+=("--skip")
    local_nonpersistent_flags+=("--skip=")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_get()
{
    last_command="stripe_get"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--ending-before=")
    two_word_flags+=("--ending-before")
    two_word_flags+=("-b")
    local_nonpersistent_flags+=("--ending-before")
    local_nonpersistent_flags+=("--ending-before=")
    local_nonpersistent_flags+=("-b")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--limit=")
    two_word_flags+=("--limit")
    two_word_flags+=("-l")
    local_nonpersistent_flags+=("--limit")
    local_nonpersistent_flags+=("--limit=")
    local_nonpersistent_flags+=("-l")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--starting-after=")
    two_word_flags+=("--starting-after")
    two_word_flags+=("-a")
    local_nonpersistent_flags+=("--starting-after")
    local_nonpersistent_flags+=("--starting-after=")
    local_nonpersistent_flags+=("-a")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_help()
{
    last_command="stripe_help"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    has_completion_function=1
    noun_aliases=()
}

_stripe_identity_verification_reports_list()
{
    last_command="stripe_identity_verification_reports_list"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--created=")
    two_word_flags+=("--created")
    local_nonpersistent_flags+=("--created")
    local_nonpersistent_flags+=("--created=")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--ending-before=")
    two_word_flags+=("--ending-before")
    local_nonpersistent_flags+=("--ending-before")
    local_nonpersistent_flags+=("--ending-before=")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--limit=")
    two_word_flags+=("--limit")
    local_nonpersistent_flags+=("--limit")
    local_nonpersistent_flags+=("--limit=")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--starting-after=")
    two_word_flags+=("--starting-after")
    local_nonpersistent_flags+=("--starting-after")
    local_nonpersistent_flags+=("--starting-after=")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--type=")
    two_word_flags+=("--type")
    local_nonpersistent_flags+=("--type")
    local_nonpersistent_flags+=("--type=")
    flags+=("--verification-session=")
    two_word_flags+=("--verification-session")
    local_nonpersistent_flags+=("--verification-session")
    local_nonpersistent_flags+=("--verification-session=")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_identity_verification_reports_retrieve()
{
    last_command="stripe_identity_verification_reports_retrieve"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--ending-before=")
    two_word_flags+=("--ending-before")
    two_word_flags+=("-b")
    local_nonpersistent_flags+=("--ending-before")
    local_nonpersistent_flags+=("--ending-before=")
    local_nonpersistent_flags+=("-b")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--limit=")
    two_word_flags+=("--limit")
    two_word_flags+=("-l")
    local_nonpersistent_flags+=("--limit")
    local_nonpersistent_flags+=("--limit=")
    local_nonpersistent_flags+=("-l")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--starting-after=")
    two_word_flags+=("--starting-after")
    two_word_flags+=("-a")
    local_nonpersistent_flags+=("--starting-after")
    local_nonpersistent_flags+=("--starting-after=")
    local_nonpersistent_flags+=("-a")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_identity_verification_reports()
{
    last_command="stripe_identity_verification_reports"

    command_aliases=()

    commands=()
    commands+=("list")
    commands+=("retrieve")

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_identity_verification_sessions_cancel()
{
    last_command="stripe_identity_verification_sessions_cancel"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_identity_verification_sessions_create()
{
    last_command="stripe_identity_verification_sessions_create"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--return-url=")
    two_word_flags+=("--return-url")
    local_nonpersistent_flags+=("--return-url")
    local_nonpersistent_flags+=("--return-url=")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--type=")
    two_word_flags+=("--type")
    local_nonpersistent_flags+=("--type")
    local_nonpersistent_flags+=("--type=")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_identity_verification_sessions_list()
{
    last_command="stripe_identity_verification_sessions_list"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--created=")
    two_word_flags+=("--created")
    local_nonpersistent_flags+=("--created")
    local_nonpersistent_flags+=("--created=")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--ending-before=")
    two_word_flags+=("--ending-before")
    local_nonpersistent_flags+=("--ending-before")
    local_nonpersistent_flags+=("--ending-before=")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--limit=")
    two_word_flags+=("--limit")
    local_nonpersistent_flags+=("--limit")
    local_nonpersistent_flags+=("--limit=")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--starting-after=")
    two_word_flags+=("--starting-after")
    local_nonpersistent_flags+=("--starting-after")
    local_nonpersistent_flags+=("--starting-after=")
    flags+=("--status=")
    two_word_flags+=("--status")
    local_nonpersistent_flags+=("--status")
    local_nonpersistent_flags+=("--status=")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_identity_verification_sessions_redact()
{
    last_command="stripe_identity_verification_sessions_redact"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_identity_verification_sessions_retrieve()
{
    last_command="stripe_identity_verification_sessions_retrieve"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--ending-before=")
    two_word_flags+=("--ending-before")
    two_word_flags+=("-b")
    local_nonpersistent_flags+=("--ending-before")
    local_nonpersistent_flags+=("--ending-before=")
    local_nonpersistent_flags+=("-b")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--limit=")
    two_word_flags+=("--limit")
    two_word_flags+=("-l")
    local_nonpersistent_flags+=("--limit")
    local_nonpersistent_flags+=("--limit=")
    local_nonpersistent_flags+=("-l")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--starting-after=")
    two_word_flags+=("--starting-after")
    two_word_flags+=("-a")
    local_nonpersistent_flags+=("--starting-after")
    local_nonpersistent_flags+=("--starting-after=")
    local_nonpersistent_flags+=("-a")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_identity_verification_sessions_update()
{
    last_command="stripe_identity_verification_sessions_update"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--type=")
    two_word_flags+=("--type")
    local_nonpersistent_flags+=("--type")
    local_nonpersistent_flags+=("--type=")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_identity_verification_sessions()
{
    last_command="stripe_identity_verification_sessions"

    command_aliases=()

    commands=()
    commands+=("cancel")
    commands+=("create")
    commands+=("list")
    commands+=("redact")
    commands+=("retrieve")
    commands+=("update")

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_identity()
{
    last_command="stripe_identity"

    command_aliases=()

    commands=()
    commands+=("verification_reports")
    commands+=("verification_sessions")

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_invoiceitems_create()
{
    last_command="stripe_invoiceitems_create"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--amount=")
    two_word_flags+=("--amount")
    local_nonpersistent_flags+=("--amount")
    local_nonpersistent_flags+=("--amount=")
    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--currency=")
    two_word_flags+=("--currency")
    local_nonpersistent_flags+=("--currency")
    local_nonpersistent_flags+=("--currency=")
    flags+=("--customer=")
    two_word_flags+=("--customer")
    local_nonpersistent_flags+=("--customer")
    local_nonpersistent_flags+=("--customer=")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--description=")
    two_word_flags+=("--description")
    local_nonpersistent_flags+=("--description")
    local_nonpersistent_flags+=("--description=")
    flags+=("--discountable=")
    two_word_flags+=("--discountable")
    local_nonpersistent_flags+=("--discountable")
    local_nonpersistent_flags+=("--discountable=")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--invoice=")
    two_word_flags+=("--invoice")
    local_nonpersistent_flags+=("--invoice")
    local_nonpersistent_flags+=("--invoice=")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--price=")
    two_word_flags+=("--price")
    local_nonpersistent_flags+=("--price")
    local_nonpersistent_flags+=("--price=")
    flags+=("--quantity=")
    two_word_flags+=("--quantity")
    local_nonpersistent_flags+=("--quantity")
    local_nonpersistent_flags+=("--quantity=")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--subscription=")
    two_word_flags+=("--subscription")
    local_nonpersistent_flags+=("--subscription")
    local_nonpersistent_flags+=("--subscription=")
    flags+=("--unit-amount=")
    two_word_flags+=("--unit-amount")
    local_nonpersistent_flags+=("--unit-amount")
    local_nonpersistent_flags+=("--unit-amount=")
    flags+=("--unit-amount-decimal=")
    two_word_flags+=("--unit-amount-decimal")
    local_nonpersistent_flags+=("--unit-amount-decimal")
    local_nonpersistent_flags+=("--unit-amount-decimal=")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_invoiceitems_delete()
{
    last_command="stripe_invoiceitems_delete"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_invoiceitems_list()
{
    last_command="stripe_invoiceitems_list"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--created=")
    two_word_flags+=("--created")
    local_nonpersistent_flags+=("--created")
    local_nonpersistent_flags+=("--created=")
    flags+=("--customer=")
    two_word_flags+=("--customer")
    local_nonpersistent_flags+=("--customer")
    local_nonpersistent_flags+=("--customer=")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--ending-before=")
    two_word_flags+=("--ending-before")
    local_nonpersistent_flags+=("--ending-before")
    local_nonpersistent_flags+=("--ending-before=")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--invoice=")
    two_word_flags+=("--invoice")
    local_nonpersistent_flags+=("--invoice")
    local_nonpersistent_flags+=("--invoice=")
    flags+=("--limit=")
    two_word_flags+=("--limit")
    local_nonpersistent_flags+=("--limit")
    local_nonpersistent_flags+=("--limit=")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--pending=")
    two_word_flags+=("--pending")
    local_nonpersistent_flags+=("--pending")
    local_nonpersistent_flags+=("--pending=")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--starting-after=")
    two_word_flags+=("--starting-after")
    local_nonpersistent_flags+=("--starting-after")
    local_nonpersistent_flags+=("--starting-after=")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_invoiceitems_retrieve()
{
    last_command="stripe_invoiceitems_retrieve"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--ending-before=")
    two_word_flags+=("--ending-before")
    two_word_flags+=("-b")
    local_nonpersistent_flags+=("--ending-before")
    local_nonpersistent_flags+=("--ending-before=")
    local_nonpersistent_flags+=("-b")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--limit=")
    two_word_flags+=("--limit")
    two_word_flags+=("-l")
    local_nonpersistent_flags+=("--limit")
    local_nonpersistent_flags+=("--limit=")
    local_nonpersistent_flags+=("-l")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--starting-after=")
    two_word_flags+=("--starting-after")
    two_word_flags+=("-a")
    local_nonpersistent_flags+=("--starting-after")
    local_nonpersistent_flags+=("--starting-after=")
    local_nonpersistent_flags+=("-a")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_invoiceitems_update()
{
    last_command="stripe_invoiceitems_update"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--amount=")
    two_word_flags+=("--amount")
    local_nonpersistent_flags+=("--amount")
    local_nonpersistent_flags+=("--amount=")
    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--description=")
    two_word_flags+=("--description")
    local_nonpersistent_flags+=("--description")
    local_nonpersistent_flags+=("--description=")
    flags+=("--discountable=")
    two_word_flags+=("--discountable")
    local_nonpersistent_flags+=("--discountable")
    local_nonpersistent_flags+=("--discountable=")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--price=")
    two_word_flags+=("--price")
    local_nonpersistent_flags+=("--price")
    local_nonpersistent_flags+=("--price=")
    flags+=("--quantity=")
    two_word_flags+=("--quantity")
    local_nonpersistent_flags+=("--quantity")
    local_nonpersistent_flags+=("--quantity=")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--unit-amount=")
    two_word_flags+=("--unit-amount")
    local_nonpersistent_flags+=("--unit-amount")
    local_nonpersistent_flags+=("--unit-amount=")
    flags+=("--unit-amount-decimal=")
    two_word_flags+=("--unit-amount-decimal")
    local_nonpersistent_flags+=("--unit-amount-decimal")
    local_nonpersistent_flags+=("--unit-amount-decimal=")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_invoiceitems()
{
    last_command="stripe_invoiceitems"

    command_aliases=()

    commands=()
    commands+=("create")
    commands+=("delete")
    commands+=("list")
    commands+=("retrieve")
    commands+=("update")

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_invoices_create()
{
    last_command="stripe_invoices_create"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--application-fee-amount=")
    two_word_flags+=("--application-fee-amount")
    local_nonpersistent_flags+=("--application-fee-amount")
    local_nonpersistent_flags+=("--application-fee-amount=")
    flags+=("--auto-advance=")
    two_word_flags+=("--auto-advance")
    local_nonpersistent_flags+=("--auto-advance")
    local_nonpersistent_flags+=("--auto-advance=")
    flags+=("--collection-method=")
    two_word_flags+=("--collection-method")
    local_nonpersistent_flags+=("--collection-method")
    local_nonpersistent_flags+=("--collection-method=")
    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--customer=")
    two_word_flags+=("--customer")
    local_nonpersistent_flags+=("--customer")
    local_nonpersistent_flags+=("--customer=")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--days-until-due=")
    two_word_flags+=("--days-until-due")
    local_nonpersistent_flags+=("--days-until-due")
    local_nonpersistent_flags+=("--days-until-due=")
    flags+=("--default-payment-method=")
    two_word_flags+=("--default-payment-method")
    local_nonpersistent_flags+=("--default-payment-method")
    local_nonpersistent_flags+=("--default-payment-method=")
    flags+=("--default-source=")
    two_word_flags+=("--default-source")
    local_nonpersistent_flags+=("--default-source")
    local_nonpersistent_flags+=("--default-source=")
    flags+=("--description=")
    two_word_flags+=("--description")
    local_nonpersistent_flags+=("--description")
    local_nonpersistent_flags+=("--description=")
    flags+=("--due-date=")
    two_word_flags+=("--due-date")
    local_nonpersistent_flags+=("--due-date")
    local_nonpersistent_flags+=("--due-date=")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--footer=")
    two_word_flags+=("--footer")
    local_nonpersistent_flags+=("--footer")
    local_nonpersistent_flags+=("--footer=")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--on-behalf-of=")
    two_word_flags+=("--on-behalf-of")
    local_nonpersistent_flags+=("--on-behalf-of")
    local_nonpersistent_flags+=("--on-behalf-of=")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--statement-descriptor=")
    two_word_flags+=("--statement-descriptor")
    local_nonpersistent_flags+=("--statement-descriptor")
    local_nonpersistent_flags+=("--statement-descriptor=")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--subscription=")
    two_word_flags+=("--subscription")
    local_nonpersistent_flags+=("--subscription")
    local_nonpersistent_flags+=("--subscription=")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_invoices_delete()
{
    last_command="stripe_invoices_delete"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_invoices_finalize_invoice()
{
    last_command="stripe_invoices_finalize_invoice"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--auto-advance=")
    two_word_flags+=("--auto-advance")
    local_nonpersistent_flags+=("--auto-advance")
    local_nonpersistent_flags+=("--auto-advance=")
    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_invoices_list()
{
    last_command="stripe_invoices_list"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--collection-method=")
    two_word_flags+=("--collection-method")
    local_nonpersistent_flags+=("--collection-method")
    local_nonpersistent_flags+=("--collection-method=")
    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--created=")
    two_word_flags+=("--created")
    local_nonpersistent_flags+=("--created")
    local_nonpersistent_flags+=("--created=")
    flags+=("--customer=")
    two_word_flags+=("--customer")
    local_nonpersistent_flags+=("--customer")
    local_nonpersistent_flags+=("--customer=")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--due-date=")
    two_word_flags+=("--due-date")
    local_nonpersistent_flags+=("--due-date")
    local_nonpersistent_flags+=("--due-date=")
    flags+=("--ending-before=")
    two_word_flags+=("--ending-before")
    local_nonpersistent_flags+=("--ending-before")
    local_nonpersistent_flags+=("--ending-before=")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--limit=")
    two_word_flags+=("--limit")
    local_nonpersistent_flags+=("--limit")
    local_nonpersistent_flags+=("--limit=")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--starting-after=")
    two_word_flags+=("--starting-after")
    local_nonpersistent_flags+=("--starting-after")
    local_nonpersistent_flags+=("--starting-after=")
    flags+=("--status=")
    two_word_flags+=("--status")
    local_nonpersistent_flags+=("--status")
    local_nonpersistent_flags+=("--status=")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--subscription=")
    two_word_flags+=("--subscription")
    local_nonpersistent_flags+=("--subscription")
    local_nonpersistent_flags+=("--subscription=")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_invoices_mark_uncollectible()
{
    last_command="stripe_invoices_mark_uncollectible"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_invoices_pay()
{
    last_command="stripe_invoices_pay"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--forgive=")
    two_word_flags+=("--forgive")
    local_nonpersistent_flags+=("--forgive")
    local_nonpersistent_flags+=("--forgive=")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--off-session=")
    two_word_flags+=("--off-session")
    local_nonpersistent_flags+=("--off-session")
    local_nonpersistent_flags+=("--off-session=")
    flags+=("--paid-out-of-band=")
    two_word_flags+=("--paid-out-of-band")
    local_nonpersistent_flags+=("--paid-out-of-band")
    local_nonpersistent_flags+=("--paid-out-of-band=")
    flags+=("--payment-method=")
    two_word_flags+=("--payment-method")
    local_nonpersistent_flags+=("--payment-method")
    local_nonpersistent_flags+=("--payment-method=")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--source=")
    two_word_flags+=("--source")
    local_nonpersistent_flags+=("--source")
    local_nonpersistent_flags+=("--source=")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_invoices_retrieve()
{
    last_command="stripe_invoices_retrieve"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--ending-before=")
    two_word_flags+=("--ending-before")
    two_word_flags+=("-b")
    local_nonpersistent_flags+=("--ending-before")
    local_nonpersistent_flags+=("--ending-before=")
    local_nonpersistent_flags+=("-b")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--limit=")
    two_word_flags+=("--limit")
    two_word_flags+=("-l")
    local_nonpersistent_flags+=("--limit")
    local_nonpersistent_flags+=("--limit=")
    local_nonpersistent_flags+=("-l")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--starting-after=")
    two_word_flags+=("--starting-after")
    two_word_flags+=("-a")
    local_nonpersistent_flags+=("--starting-after")
    local_nonpersistent_flags+=("--starting-after=")
    local_nonpersistent_flags+=("-a")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_invoices_send_invoice()
{
    last_command="stripe_invoices_send_invoice"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_invoices_upcoming()
{
    last_command="stripe_invoices_upcoming"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--coupon=")
    two_word_flags+=("--coupon")
    local_nonpersistent_flags+=("--coupon")
    local_nonpersistent_flags+=("--coupon=")
    flags+=("--customer=")
    two_word_flags+=("--customer")
    local_nonpersistent_flags+=("--customer")
    local_nonpersistent_flags+=("--customer=")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--ending-before=")
    two_word_flags+=("--ending-before")
    two_word_flags+=("-b")
    local_nonpersistent_flags+=("--ending-before")
    local_nonpersistent_flags+=("--ending-before=")
    local_nonpersistent_flags+=("-b")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--limit=")
    two_word_flags+=("--limit")
    two_word_flags+=("-l")
    local_nonpersistent_flags+=("--limit")
    local_nonpersistent_flags+=("--limit=")
    local_nonpersistent_flags+=("-l")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--schedule=")
    two_word_flags+=("--schedule")
    local_nonpersistent_flags+=("--schedule")
    local_nonpersistent_flags+=("--schedule=")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--starting-after=")
    two_word_flags+=("--starting-after")
    two_word_flags+=("-a")
    local_nonpersistent_flags+=("--starting-after")
    local_nonpersistent_flags+=("--starting-after=")
    local_nonpersistent_flags+=("-a")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--subscription=")
    two_word_flags+=("--subscription")
    local_nonpersistent_flags+=("--subscription")
    local_nonpersistent_flags+=("--subscription=")
    flags+=("--subscription-billing-cycle-anchor=")
    two_word_flags+=("--subscription-billing-cycle-anchor")
    local_nonpersistent_flags+=("--subscription-billing-cycle-anchor")
    local_nonpersistent_flags+=("--subscription-billing-cycle-anchor=")
    flags+=("--subscription-cancel-at=")
    two_word_flags+=("--subscription-cancel-at")
    local_nonpersistent_flags+=("--subscription-cancel-at")
    local_nonpersistent_flags+=("--subscription-cancel-at=")
    flags+=("--subscription-cancel-at-period-end=")
    two_word_flags+=("--subscription-cancel-at-period-end")
    local_nonpersistent_flags+=("--subscription-cancel-at-period-end")
    local_nonpersistent_flags+=("--subscription-cancel-at-period-end=")
    flags+=("--subscription-cancel-now=")
    two_word_flags+=("--subscription-cancel-now")
    local_nonpersistent_flags+=("--subscription-cancel-now")
    local_nonpersistent_flags+=("--subscription-cancel-now=")
    flags+=("--subscription-proration-behavior=")
    two_word_flags+=("--subscription-proration-behavior")
    local_nonpersistent_flags+=("--subscription-proration-behavior")
    local_nonpersistent_flags+=("--subscription-proration-behavior=")
    flags+=("--subscription-proration-date=")
    two_word_flags+=("--subscription-proration-date")
    local_nonpersistent_flags+=("--subscription-proration-date")
    local_nonpersistent_flags+=("--subscription-proration-date=")
    flags+=("--subscription-start-date=")
    two_word_flags+=("--subscription-start-date")
    local_nonpersistent_flags+=("--subscription-start-date")
    local_nonpersistent_flags+=("--subscription-start-date=")
    flags+=("--subscription-trial-end=")
    two_word_flags+=("--subscription-trial-end")
    local_nonpersistent_flags+=("--subscription-trial-end")
    local_nonpersistent_flags+=("--subscription-trial-end=")
    flags+=("--subscription-trial-from-plan=")
    two_word_flags+=("--subscription-trial-from-plan")
    local_nonpersistent_flags+=("--subscription-trial-from-plan")
    local_nonpersistent_flags+=("--subscription-trial-from-plan=")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_invoices_upcomingLines()
{
    last_command="stripe_invoices_upcomingLines"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--coupon=")
    two_word_flags+=("--coupon")
    local_nonpersistent_flags+=("--coupon")
    local_nonpersistent_flags+=("--coupon=")
    flags+=("--customer=")
    two_word_flags+=("--customer")
    local_nonpersistent_flags+=("--customer")
    local_nonpersistent_flags+=("--customer=")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--ending-before=")
    two_word_flags+=("--ending-before")
    local_nonpersistent_flags+=("--ending-before")
    local_nonpersistent_flags+=("--ending-before=")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--limit=")
    two_word_flags+=("--limit")
    local_nonpersistent_flags+=("--limit")
    local_nonpersistent_flags+=("--limit=")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--schedule=")
    two_word_flags+=("--schedule")
    local_nonpersistent_flags+=("--schedule")
    local_nonpersistent_flags+=("--schedule=")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--starting-after=")
    two_word_flags+=("--starting-after")
    local_nonpersistent_flags+=("--starting-after")
    local_nonpersistent_flags+=("--starting-after=")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--subscription=")
    two_word_flags+=("--subscription")
    local_nonpersistent_flags+=("--subscription")
    local_nonpersistent_flags+=("--subscription=")
    flags+=("--subscription-billing-cycle-anchor=")
    two_word_flags+=("--subscription-billing-cycle-anchor")
    local_nonpersistent_flags+=("--subscription-billing-cycle-anchor")
    local_nonpersistent_flags+=("--subscription-billing-cycle-anchor=")
    flags+=("--subscription-cancel-at=")
    two_word_flags+=("--subscription-cancel-at")
    local_nonpersistent_flags+=("--subscription-cancel-at")
    local_nonpersistent_flags+=("--subscription-cancel-at=")
    flags+=("--subscription-cancel-at-period-end=")
    two_word_flags+=("--subscription-cancel-at-period-end")
    local_nonpersistent_flags+=("--subscription-cancel-at-period-end")
    local_nonpersistent_flags+=("--subscription-cancel-at-period-end=")
    flags+=("--subscription-cancel-now=")
    two_word_flags+=("--subscription-cancel-now")
    local_nonpersistent_flags+=("--subscription-cancel-now")
    local_nonpersistent_flags+=("--subscription-cancel-now=")
    flags+=("--subscription-proration-behavior=")
    two_word_flags+=("--subscription-proration-behavior")
    local_nonpersistent_flags+=("--subscription-proration-behavior")
    local_nonpersistent_flags+=("--subscription-proration-behavior=")
    flags+=("--subscription-proration-date=")
    two_word_flags+=("--subscription-proration-date")
    local_nonpersistent_flags+=("--subscription-proration-date")
    local_nonpersistent_flags+=("--subscription-proration-date=")
    flags+=("--subscription-start-date=")
    two_word_flags+=("--subscription-start-date")
    local_nonpersistent_flags+=("--subscription-start-date")
    local_nonpersistent_flags+=("--subscription-start-date=")
    flags+=("--subscription-trial-end=")
    two_word_flags+=("--subscription-trial-end")
    local_nonpersistent_flags+=("--subscription-trial-end")
    local_nonpersistent_flags+=("--subscription-trial-end=")
    flags+=("--subscription-trial-from-plan=")
    two_word_flags+=("--subscription-trial-from-plan")
    local_nonpersistent_flags+=("--subscription-trial-from-plan")
    local_nonpersistent_flags+=("--subscription-trial-from-plan=")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_invoices_update()
{
    last_command="stripe_invoices_update"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--application-fee-amount=")
    two_word_flags+=("--application-fee-amount")
    local_nonpersistent_flags+=("--application-fee-amount")
    local_nonpersistent_flags+=("--application-fee-amount=")
    flags+=("--auto-advance=")
    two_word_flags+=("--auto-advance")
    local_nonpersistent_flags+=("--auto-advance")
    local_nonpersistent_flags+=("--auto-advance=")
    flags+=("--collection-method=")
    two_word_flags+=("--collection-method")
    local_nonpersistent_flags+=("--collection-method")
    local_nonpersistent_flags+=("--collection-method=")
    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--days-until-due=")
    two_word_flags+=("--days-until-due")
    local_nonpersistent_flags+=("--days-until-due")
    local_nonpersistent_flags+=("--days-until-due=")
    flags+=("--default-payment-method=")
    two_word_flags+=("--default-payment-method")
    local_nonpersistent_flags+=("--default-payment-method")
    local_nonpersistent_flags+=("--default-payment-method=")
    flags+=("--default-source=")
    two_word_flags+=("--default-source")
    local_nonpersistent_flags+=("--default-source")
    local_nonpersistent_flags+=("--default-source=")
    flags+=("--description=")
    two_word_flags+=("--description")
    local_nonpersistent_flags+=("--description")
    local_nonpersistent_flags+=("--description=")
    flags+=("--due-date=")
    two_word_flags+=("--due-date")
    local_nonpersistent_flags+=("--due-date")
    local_nonpersistent_flags+=("--due-date=")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--footer=")
    two_word_flags+=("--footer")
    local_nonpersistent_flags+=("--footer")
    local_nonpersistent_flags+=("--footer=")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--on-behalf-of=")
    two_word_flags+=("--on-behalf-of")
    local_nonpersistent_flags+=("--on-behalf-of")
    local_nonpersistent_flags+=("--on-behalf-of=")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--statement-descriptor=")
    two_word_flags+=("--statement-descriptor")
    local_nonpersistent_flags+=("--statement-descriptor")
    local_nonpersistent_flags+=("--statement-descriptor=")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_invoices_void_invoice()
{
    last_command="stripe_invoices_void_invoice"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_invoices()
{
    last_command="stripe_invoices"

    command_aliases=()

    commands=()
    commands+=("create")
    commands+=("delete")
    commands+=("finalize_invoice")
    commands+=("list")
    commands+=("mark_uncollectible")
    commands+=("pay")
    commands+=("retrieve")
    commands+=("send_invoice")
    commands+=("upcoming")
    commands+=("upcomingLines")
    commands+=("update")
    commands+=("void_invoice")

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_issuing_authorizations_approve()
{
    last_command="stripe_issuing_authorizations_approve"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--amount=")
    two_word_flags+=("--amount")
    local_nonpersistent_flags+=("--amount")
    local_nonpersistent_flags+=("--amount=")
    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_issuing_authorizations_decline()
{
    last_command="stripe_issuing_authorizations_decline"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_issuing_authorizations_list()
{
    last_command="stripe_issuing_authorizations_list"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--card=")
    two_word_flags+=("--card")
    local_nonpersistent_flags+=("--card")
    local_nonpersistent_flags+=("--card=")
    flags+=("--cardholder=")
    two_word_flags+=("--cardholder")
    local_nonpersistent_flags+=("--cardholder")
    local_nonpersistent_flags+=("--cardholder=")
    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--created=")
    two_word_flags+=("--created")
    local_nonpersistent_flags+=("--created")
    local_nonpersistent_flags+=("--created=")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--ending-before=")
    two_word_flags+=("--ending-before")
    local_nonpersistent_flags+=("--ending-before")
    local_nonpersistent_flags+=("--ending-before=")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--limit=")
    two_word_flags+=("--limit")
    local_nonpersistent_flags+=("--limit")
    local_nonpersistent_flags+=("--limit=")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--starting-after=")
    two_word_flags+=("--starting-after")
    local_nonpersistent_flags+=("--starting-after")
    local_nonpersistent_flags+=("--starting-after=")
    flags+=("--status=")
    two_word_flags+=("--status")
    local_nonpersistent_flags+=("--status")
    local_nonpersistent_flags+=("--status=")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_issuing_authorizations_retrieve()
{
    last_command="stripe_issuing_authorizations_retrieve"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--ending-before=")
    two_word_flags+=("--ending-before")
    two_word_flags+=("-b")
    local_nonpersistent_flags+=("--ending-before")
    local_nonpersistent_flags+=("--ending-before=")
    local_nonpersistent_flags+=("-b")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--limit=")
    two_word_flags+=("--limit")
    two_word_flags+=("-l")
    local_nonpersistent_flags+=("--limit")
    local_nonpersistent_flags+=("--limit=")
    local_nonpersistent_flags+=("-l")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--starting-after=")
    two_word_flags+=("--starting-after")
    two_word_flags+=("-a")
    local_nonpersistent_flags+=("--starting-after")
    local_nonpersistent_flags+=("--starting-after=")
    local_nonpersistent_flags+=("-a")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_issuing_authorizations_update()
{
    last_command="stripe_issuing_authorizations_update"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_issuing_authorizations()
{
    last_command="stripe_issuing_authorizations"

    command_aliases=()

    commands=()
    commands+=("approve")
    commands+=("decline")
    commands+=("list")
    commands+=("retrieve")
    commands+=("update")

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_issuing_cardholders_create()
{
    last_command="stripe_issuing_cardholders_create"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--email=")
    two_word_flags+=("--email")
    local_nonpersistent_flags+=("--email")
    local_nonpersistent_flags+=("--email=")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--name=")
    two_word_flags+=("--name")
    local_nonpersistent_flags+=("--name")
    local_nonpersistent_flags+=("--name=")
    flags+=("--phone-number=")
    two_word_flags+=("--phone-number")
    local_nonpersistent_flags+=("--phone-number")
    local_nonpersistent_flags+=("--phone-number=")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--status=")
    two_word_flags+=("--status")
    local_nonpersistent_flags+=("--status")
    local_nonpersistent_flags+=("--status=")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--type=")
    two_word_flags+=("--type")
    local_nonpersistent_flags+=("--type")
    local_nonpersistent_flags+=("--type=")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_issuing_cardholders_list()
{
    last_command="stripe_issuing_cardholders_list"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--created=")
    two_word_flags+=("--created")
    local_nonpersistent_flags+=("--created")
    local_nonpersistent_flags+=("--created=")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--email=")
    two_word_flags+=("--email")
    local_nonpersistent_flags+=("--email")
    local_nonpersistent_flags+=("--email=")
    flags+=("--ending-before=")
    two_word_flags+=("--ending-before")
    local_nonpersistent_flags+=("--ending-before")
    local_nonpersistent_flags+=("--ending-before=")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--limit=")
    two_word_flags+=("--limit")
    local_nonpersistent_flags+=("--limit")
    local_nonpersistent_flags+=("--limit=")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--phone-number=")
    two_word_flags+=("--phone-number")
    local_nonpersistent_flags+=("--phone-number")
    local_nonpersistent_flags+=("--phone-number=")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--starting-after=")
    two_word_flags+=("--starting-after")
    local_nonpersistent_flags+=("--starting-after")
    local_nonpersistent_flags+=("--starting-after=")
    flags+=("--status=")
    two_word_flags+=("--status")
    local_nonpersistent_flags+=("--status")
    local_nonpersistent_flags+=("--status=")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--type=")
    two_word_flags+=("--type")
    local_nonpersistent_flags+=("--type")
    local_nonpersistent_flags+=("--type=")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_issuing_cardholders_retrieve()
{
    last_command="stripe_issuing_cardholders_retrieve"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--ending-before=")
    two_word_flags+=("--ending-before")
    two_word_flags+=("-b")
    local_nonpersistent_flags+=("--ending-before")
    local_nonpersistent_flags+=("--ending-before=")
    local_nonpersistent_flags+=("-b")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--limit=")
    two_word_flags+=("--limit")
    two_word_flags+=("-l")
    local_nonpersistent_flags+=("--limit")
    local_nonpersistent_flags+=("--limit=")
    local_nonpersistent_flags+=("-l")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--starting-after=")
    two_word_flags+=("--starting-after")
    two_word_flags+=("-a")
    local_nonpersistent_flags+=("--starting-after")
    local_nonpersistent_flags+=("--starting-after=")
    local_nonpersistent_flags+=("-a")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_issuing_cardholders_update()
{
    last_command="stripe_issuing_cardholders_update"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--email=")
    two_word_flags+=("--email")
    local_nonpersistent_flags+=("--email")
    local_nonpersistent_flags+=("--email=")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--phone-number=")
    two_word_flags+=("--phone-number")
    local_nonpersistent_flags+=("--phone-number")
    local_nonpersistent_flags+=("--phone-number=")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--status=")
    two_word_flags+=("--status")
    local_nonpersistent_flags+=("--status")
    local_nonpersistent_flags+=("--status=")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_issuing_cardholders()
{
    last_command="stripe_issuing_cardholders"

    command_aliases=()

    commands=()
    commands+=("create")
    commands+=("list")
    commands+=("retrieve")
    commands+=("update")

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_issuing_cards_create()
{
    last_command="stripe_issuing_cards_create"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--cardholder=")
    two_word_flags+=("--cardholder")
    local_nonpersistent_flags+=("--cardholder")
    local_nonpersistent_flags+=("--cardholder=")
    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--currency=")
    two_word_flags+=("--currency")
    local_nonpersistent_flags+=("--currency")
    local_nonpersistent_flags+=("--currency=")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--replacement-for=")
    two_word_flags+=("--replacement-for")
    local_nonpersistent_flags+=("--replacement-for")
    local_nonpersistent_flags+=("--replacement-for=")
    flags+=("--replacement-reason=")
    two_word_flags+=("--replacement-reason")
    local_nonpersistent_flags+=("--replacement-reason")
    local_nonpersistent_flags+=("--replacement-reason=")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--status=")
    two_word_flags+=("--status")
    local_nonpersistent_flags+=("--status")
    local_nonpersistent_flags+=("--status=")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--type=")
    two_word_flags+=("--type")
    local_nonpersistent_flags+=("--type")
    local_nonpersistent_flags+=("--type=")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_issuing_cards_list()
{
    last_command="stripe_issuing_cards_list"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--cardholder=")
    two_word_flags+=("--cardholder")
    local_nonpersistent_flags+=("--cardholder")
    local_nonpersistent_flags+=("--cardholder=")
    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--created=")
    two_word_flags+=("--created")
    local_nonpersistent_flags+=("--created")
    local_nonpersistent_flags+=("--created=")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--ending-before=")
    two_word_flags+=("--ending-before")
    local_nonpersistent_flags+=("--ending-before")
    local_nonpersistent_flags+=("--ending-before=")
    flags+=("--exp-month=")
    two_word_flags+=("--exp-month")
    local_nonpersistent_flags+=("--exp-month")
    local_nonpersistent_flags+=("--exp-month=")
    flags+=("--exp-year=")
    two_word_flags+=("--exp-year")
    local_nonpersistent_flags+=("--exp-year")
    local_nonpersistent_flags+=("--exp-year=")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--last4=")
    two_word_flags+=("--last4")
    local_nonpersistent_flags+=("--last4")
    local_nonpersistent_flags+=("--last4=")
    flags+=("--limit=")
    two_word_flags+=("--limit")
    local_nonpersistent_flags+=("--limit")
    local_nonpersistent_flags+=("--limit=")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--starting-after=")
    two_word_flags+=("--starting-after")
    local_nonpersistent_flags+=("--starting-after")
    local_nonpersistent_flags+=("--starting-after=")
    flags+=("--status=")
    two_word_flags+=("--status")
    local_nonpersistent_flags+=("--status")
    local_nonpersistent_flags+=("--status=")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--type=")
    two_word_flags+=("--type")
    local_nonpersistent_flags+=("--type")
    local_nonpersistent_flags+=("--type=")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_issuing_cards_retrieve()
{
    last_command="stripe_issuing_cards_retrieve"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--ending-before=")
    two_word_flags+=("--ending-before")
    two_word_flags+=("-b")
    local_nonpersistent_flags+=("--ending-before")
    local_nonpersistent_flags+=("--ending-before=")
    local_nonpersistent_flags+=("-b")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--limit=")
    two_word_flags+=("--limit")
    two_word_flags+=("-l")
    local_nonpersistent_flags+=("--limit")
    local_nonpersistent_flags+=("--limit=")
    local_nonpersistent_flags+=("-l")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--starting-after=")
    two_word_flags+=("--starting-after")
    two_word_flags+=("-a")
    local_nonpersistent_flags+=("--starting-after")
    local_nonpersistent_flags+=("--starting-after=")
    local_nonpersistent_flags+=("-a")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_issuing_cards_update()
{
    last_command="stripe_issuing_cards_update"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--cancellation-reason=")
    two_word_flags+=("--cancellation-reason")
    local_nonpersistent_flags+=("--cancellation-reason")
    local_nonpersistent_flags+=("--cancellation-reason=")
    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--status=")
    two_word_flags+=("--status")
    local_nonpersistent_flags+=("--status")
    local_nonpersistent_flags+=("--status=")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_issuing_cards()
{
    last_command="stripe_issuing_cards"

    command_aliases=()

    commands=()
    commands+=("create")
    commands+=("list")
    commands+=("retrieve")
    commands+=("update")

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_issuing_disputes_create()
{
    last_command="stripe_issuing_disputes_create"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--transaction=")
    two_word_flags+=("--transaction")
    local_nonpersistent_flags+=("--transaction")
    local_nonpersistent_flags+=("--transaction=")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_issuing_disputes_list()
{
    last_command="stripe_issuing_disputes_list"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--created=")
    two_word_flags+=("--created")
    local_nonpersistent_flags+=("--created")
    local_nonpersistent_flags+=("--created=")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--ending-before=")
    two_word_flags+=("--ending-before")
    local_nonpersistent_flags+=("--ending-before")
    local_nonpersistent_flags+=("--ending-before=")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--limit=")
    two_word_flags+=("--limit")
    local_nonpersistent_flags+=("--limit")
    local_nonpersistent_flags+=("--limit=")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--starting-after=")
    two_word_flags+=("--starting-after")
    local_nonpersistent_flags+=("--starting-after")
    local_nonpersistent_flags+=("--starting-after=")
    flags+=("--status=")
    two_word_flags+=("--status")
    local_nonpersistent_flags+=("--status")
    local_nonpersistent_flags+=("--status=")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--transaction=")
    two_word_flags+=("--transaction")
    local_nonpersistent_flags+=("--transaction")
    local_nonpersistent_flags+=("--transaction=")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_issuing_disputes_retrieve()
{
    last_command="stripe_issuing_disputes_retrieve"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--ending-before=")
    two_word_flags+=("--ending-before")
    two_word_flags+=("-b")
    local_nonpersistent_flags+=("--ending-before")
    local_nonpersistent_flags+=("--ending-before=")
    local_nonpersistent_flags+=("-b")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--limit=")
    two_word_flags+=("--limit")
    two_word_flags+=("-l")
    local_nonpersistent_flags+=("--limit")
    local_nonpersistent_flags+=("--limit=")
    local_nonpersistent_flags+=("-l")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--starting-after=")
    two_word_flags+=("--starting-after")
    two_word_flags+=("-a")
    local_nonpersistent_flags+=("--starting-after")
    local_nonpersistent_flags+=("--starting-after=")
    local_nonpersistent_flags+=("-a")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_issuing_disputes_submit()
{
    last_command="stripe_issuing_disputes_submit"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_issuing_disputes_update()
{
    last_command="stripe_issuing_disputes_update"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_issuing_disputes()
{
    last_command="stripe_issuing_disputes"

    command_aliases=()

    commands=()
    commands+=("create")
    commands+=("list")
    commands+=("retrieve")
    commands+=("submit")
    commands+=("update")

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_issuing_transactions_list()
{
    last_command="stripe_issuing_transactions_list"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--card=")
    two_word_flags+=("--card")
    local_nonpersistent_flags+=("--card")
    local_nonpersistent_flags+=("--card=")
    flags+=("--cardholder=")
    two_word_flags+=("--cardholder")
    local_nonpersistent_flags+=("--cardholder")
    local_nonpersistent_flags+=("--cardholder=")
    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--created=")
    two_word_flags+=("--created")
    local_nonpersistent_flags+=("--created")
    local_nonpersistent_flags+=("--created=")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--ending-before=")
    two_word_flags+=("--ending-before")
    local_nonpersistent_flags+=("--ending-before")
    local_nonpersistent_flags+=("--ending-before=")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--limit=")
    two_word_flags+=("--limit")
    local_nonpersistent_flags+=("--limit")
    local_nonpersistent_flags+=("--limit=")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--starting-after=")
    two_word_flags+=("--starting-after")
    local_nonpersistent_flags+=("--starting-after")
    local_nonpersistent_flags+=("--starting-after=")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--type=")
    two_word_flags+=("--type")
    local_nonpersistent_flags+=("--type")
    local_nonpersistent_flags+=("--type=")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_issuing_transactions_retrieve()
{
    last_command="stripe_issuing_transactions_retrieve"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--ending-before=")
    two_word_flags+=("--ending-before")
    two_word_flags+=("-b")
    local_nonpersistent_flags+=("--ending-before")
    local_nonpersistent_flags+=("--ending-before=")
    local_nonpersistent_flags+=("-b")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--limit=")
    two_word_flags+=("--limit")
    two_word_flags+=("-l")
    local_nonpersistent_flags+=("--limit")
    local_nonpersistent_flags+=("--limit=")
    local_nonpersistent_flags+=("-l")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--starting-after=")
    two_word_flags+=("--starting-after")
    two_word_flags+=("-a")
    local_nonpersistent_flags+=("--starting-after")
    local_nonpersistent_flags+=("--starting-after=")
    local_nonpersistent_flags+=("-a")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_issuing_transactions_update()
{
    last_command="stripe_issuing_transactions_update"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_issuing_transactions()
{
    last_command="stripe_issuing_transactions"

    command_aliases=()

    commands=()
    commands+=("list")
    commands+=("retrieve")
    commands+=("update")

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_issuing()
{
    last_command="stripe_issuing"

    command_aliases=()

    commands=()
    commands+=("authorizations")
    commands+=("cardholders")
    commands+=("cards")
    commands+=("disputes")
    commands+=("transactions")

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_items_list()
{
    last_command="stripe_items_list"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--ending-before=")
    two_word_flags+=("--ending-before")
    local_nonpersistent_flags+=("--ending-before")
    local_nonpersistent_flags+=("--ending-before=")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--limit=")
    two_word_flags+=("--limit")
    local_nonpersistent_flags+=("--limit")
    local_nonpersistent_flags+=("--limit=")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--starting-after=")
    two_word_flags+=("--starting-after")
    local_nonpersistent_flags+=("--starting-after")
    local_nonpersistent_flags+=("--starting-after=")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_items()
{
    last_command="stripe_items"

    command_aliases=()

    commands=()
    commands+=("list")

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_line_items_list()
{
    last_command="stripe_line_items_list"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--ending-before=")
    two_word_flags+=("--ending-before")
    local_nonpersistent_flags+=("--ending-before")
    local_nonpersistent_flags+=("--ending-before=")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--limit=")
    two_word_flags+=("--limit")
    local_nonpersistent_flags+=("--limit")
    local_nonpersistent_flags+=("--limit=")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--starting-after=")
    two_word_flags+=("--starting-after")
    local_nonpersistent_flags+=("--starting-after")
    local_nonpersistent_flags+=("--starting-after=")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_line_items()
{
    last_command="stripe_line_items"

    command_aliases=()

    commands=()
    commands+=("list")

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_listen()
{
    last_command="stripe_listen"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--connect-headers=")
    two_word_flags+=("--connect-headers")
    local_nonpersistent_flags+=("--connect-headers")
    local_nonpersistent_flags+=("--connect-headers=")
    flags+=("--events=")
    two_word_flags+=("--events")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--events")
    local_nonpersistent_flags+=("--events=")
    local_nonpersistent_flags+=("-e")
    flags+=("--format=")
    two_word_flags+=("--format")
    local_nonpersistent_flags+=("--format")
    local_nonpersistent_flags+=("--format=")
    flags+=("--forward-connect-to=")
    two_word_flags+=("--forward-connect-to")
    two_word_flags+=("-c")
    local_nonpersistent_flags+=("--forward-connect-to")
    local_nonpersistent_flags+=("--forward-connect-to=")
    local_nonpersistent_flags+=("-c")
    flags+=("--forward-to=")
    two_word_flags+=("--forward-to")
    two_word_flags+=("-f")
    local_nonpersistent_flags+=("--forward-to")
    local_nonpersistent_flags+=("--forward-to=")
    local_nonpersistent_flags+=("-f")
    flags+=("--headers=")
    two_word_flags+=("--headers")
    two_word_flags+=("-H")
    local_nonpersistent_flags+=("--headers")
    local_nonpersistent_flags+=("--headers=")
    local_nonpersistent_flags+=("-H")
    flags+=("--latest")
    flags+=("-l")
    local_nonpersistent_flags+=("--latest")
    local_nonpersistent_flags+=("-l")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--print-secret")
    local_nonpersistent_flags+=("--print-secret")
    flags+=("--skip-update")
    flags+=("-s")
    local_nonpersistent_flags+=("--skip-update")
    local_nonpersistent_flags+=("-s")
    flags+=("--skip-verify")
    local_nonpersistent_flags+=("--skip-verify")
    flags+=("--use-configured-webhooks")
    flags+=("-a")
    local_nonpersistent_flags+=("--use-configured-webhooks")
    local_nonpersistent_flags+=("-a")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_login()
{
    last_command="stripe_login"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--interactive")
    flags+=("-i")
    local_nonpersistent_flags+=("--interactive")
    local_nonpersistent_flags+=("-i")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_login_links_create()
{
    last_command="stripe_login_links_create"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--redirect-url=")
    two_word_flags+=("--redirect-url")
    local_nonpersistent_flags+=("--redirect-url")
    local_nonpersistent_flags+=("--redirect-url=")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_login_links()
{
    last_command="stripe_login_links"

    command_aliases=()

    commands=()
    commands+=("create")

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_logout()
{
    last_command="stripe_logout"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--all")
    flags+=("-a")
    local_nonpersistent_flags+=("--all")
    local_nonpersistent_flags+=("-a")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_logs_tail()
{
    last_command="stripe_logs_tail"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--filter-account=")
    two_word_flags+=("--filter-account")
    local_nonpersistent_flags+=("--filter-account")
    local_nonpersistent_flags+=("--filter-account=")
    flags+=("--filter-http-method=")
    two_word_flags+=("--filter-http-method")
    local_nonpersistent_flags+=("--filter-http-method")
    local_nonpersistent_flags+=("--filter-http-method=")
    flags+=("--filter-ip-address=")
    two_word_flags+=("--filter-ip-address")
    local_nonpersistent_flags+=("--filter-ip-address")
    local_nonpersistent_flags+=("--filter-ip-address=")
    flags+=("--filter-request-path=")
    two_word_flags+=("--filter-request-path")
    local_nonpersistent_flags+=("--filter-request-path")
    local_nonpersistent_flags+=("--filter-request-path=")
    flags+=("--filter-request-status=")
    two_word_flags+=("--filter-request-status")
    local_nonpersistent_flags+=("--filter-request-status")
    local_nonpersistent_flags+=("--filter-request-status=")
    flags+=("--filter-source=")
    two_word_flags+=("--filter-source")
    local_nonpersistent_flags+=("--filter-source")
    local_nonpersistent_flags+=("--filter-source=")
    flags+=("--filter-status-code=")
    two_word_flags+=("--filter-status-code")
    local_nonpersistent_flags+=("--filter-status-code")
    local_nonpersistent_flags+=("--filter-status-code=")
    flags+=("--filter-status-code-type=")
    two_word_flags+=("--filter-status-code-type")
    local_nonpersistent_flags+=("--filter-status-code-type")
    local_nonpersistent_flags+=("--filter-status-code-type=")
    flags+=("--format=")
    two_word_flags+=("--format")
    local_nonpersistent_flags+=("--format")
    local_nonpersistent_flags+=("--format=")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_logs()
{
    last_command="stripe_logs"

    command_aliases=()

    commands=()
    commands+=("tail")

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_mandates_retrieve()
{
    last_command="stripe_mandates_retrieve"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--ending-before=")
    two_word_flags+=("--ending-before")
    two_word_flags+=("-b")
    local_nonpersistent_flags+=("--ending-before")
    local_nonpersistent_flags+=("--ending-before=")
    local_nonpersistent_flags+=("-b")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--limit=")
    two_word_flags+=("--limit")
    two_word_flags+=("-l")
    local_nonpersistent_flags+=("--limit")
    local_nonpersistent_flags+=("--limit=")
    local_nonpersistent_flags+=("-l")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--starting-after=")
    two_word_flags+=("--starting-after")
    two_word_flags+=("-a")
    local_nonpersistent_flags+=("--starting-after")
    local_nonpersistent_flags+=("--starting-after=")
    local_nonpersistent_flags+=("-a")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_mandates()
{
    last_command="stripe_mandates"

    command_aliases=()

    commands=()
    commands+=("retrieve")

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_open()
{
    last_command="stripe_open"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--list")
    local_nonpersistent_flags+=("--list")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    must_have_one_noun+=("api")
    must_have_one_noun+=("apiref")
    must_have_one_noun+=("cliref")
    must_have_one_noun+=("dashboard")
    must_have_one_noun+=("dashboard/apikeys")
    must_have_one_noun+=("dashboard/atlas")
    must_have_one_noun+=("dashboard/balance")
    must_have_one_noun+=("dashboard/billing")
    must_have_one_noun+=("dashboard/connect")
    must_have_one_noun+=("dashboard/connect/accounts")
    must_have_one_noun+=("dashboard/connect/collected-fees")
    must_have_one_noun+=("dashboard/connect/transfers")
    must_have_one_noun+=("dashboard/coupons")
    must_have_one_noun+=("dashboard/customers")
    must_have_one_noun+=("dashboard/developers")
    must_have_one_noun+=("dashboard/disputes")
    must_have_one_noun+=("dashboard/events")
    must_have_one_noun+=("dashboard/invoices")
    must_have_one_noun+=("dashboard/logs")
    must_have_one_noun+=("dashboard/orders")
    must_have_one_noun+=("dashboard/orders/products")
    must_have_one_noun+=("dashboard/payments")
    must_have_one_noun+=("dashboard/payouts")
    must_have_one_noun+=("dashboard/radar")
    must_have_one_noun+=("dashboard/radar/list")
    must_have_one_noun+=("dashboard/radar/reviews")
    must_have_one_noun+=("dashboard/radar/rules")
    must_have_one_noun+=("dashboard/settings")
    must_have_one_noun+=("dashboard/subscriptions")
    must_have_one_noun+=("dashboard/subscriptions/products")
    must_have_one_noun+=("dashboard/tax-rates")
    must_have_one_noun+=("dashboard/terminal")
    must_have_one_noun+=("dashboard/terminal/hardware_orders")
    must_have_one_noun+=("dashboard/terminal/locations")
    must_have_one_noun+=("dashboard/topups")
    must_have_one_noun+=("dashboard/transactions")
    must_have_one_noun+=("dashboard/webhooks")
    must_have_one_noun+=("docs")
    noun_aliases=()
}

_stripe_order_returns_list()
{
    last_command="stripe_order_returns_list"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--created=")
    two_word_flags+=("--created")
    local_nonpersistent_flags+=("--created")
    local_nonpersistent_flags+=("--created=")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--ending-before=")
    two_word_flags+=("--ending-before")
    local_nonpersistent_flags+=("--ending-before")
    local_nonpersistent_flags+=("--ending-before=")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--limit=")
    two_word_flags+=("--limit")
    local_nonpersistent_flags+=("--limit")
    local_nonpersistent_flags+=("--limit=")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--order=")
    two_word_flags+=("--order")
    local_nonpersistent_flags+=("--order")
    local_nonpersistent_flags+=("--order=")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--starting-after=")
    two_word_flags+=("--starting-after")
    local_nonpersistent_flags+=("--starting-after")
    local_nonpersistent_flags+=("--starting-after=")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_order_returns_retrieve()
{
    last_command="stripe_order_returns_retrieve"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--ending-before=")
    two_word_flags+=("--ending-before")
    two_word_flags+=("-b")
    local_nonpersistent_flags+=("--ending-before")
    local_nonpersistent_flags+=("--ending-before=")
    local_nonpersistent_flags+=("-b")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--limit=")
    two_word_flags+=("--limit")
    two_word_flags+=("-l")
    local_nonpersistent_flags+=("--limit")
    local_nonpersistent_flags+=("--limit=")
    local_nonpersistent_flags+=("-l")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--starting-after=")
    two_word_flags+=("--starting-after")
    two_word_flags+=("-a")
    local_nonpersistent_flags+=("--starting-after")
    local_nonpersistent_flags+=("--starting-after=")
    local_nonpersistent_flags+=("-a")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_order_returns()
{
    last_command="stripe_order_returns"

    command_aliases=()

    commands=()
    commands+=("list")
    commands+=("retrieve")

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_orders_create()
{
    last_command="stripe_orders_create"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--coupon=")
    two_word_flags+=("--coupon")
    local_nonpersistent_flags+=("--coupon")
    local_nonpersistent_flags+=("--coupon=")
    flags+=("--currency=")
    two_word_flags+=("--currency")
    local_nonpersistent_flags+=("--currency")
    local_nonpersistent_flags+=("--currency=")
    flags+=("--customer=")
    two_word_flags+=("--customer")
    local_nonpersistent_flags+=("--customer")
    local_nonpersistent_flags+=("--customer=")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--email=")
    two_word_flags+=("--email")
    local_nonpersistent_flags+=("--email")
    local_nonpersistent_flags+=("--email=")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_orders_list()
{
    last_command="stripe_orders_list"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--created=")
    two_word_flags+=("--created")
    local_nonpersistent_flags+=("--created")
    local_nonpersistent_flags+=("--created=")
    flags+=("--customer=")
    two_word_flags+=("--customer")
    local_nonpersistent_flags+=("--customer")
    local_nonpersistent_flags+=("--customer=")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--ending-before=")
    two_word_flags+=("--ending-before")
    local_nonpersistent_flags+=("--ending-before")
    local_nonpersistent_flags+=("--ending-before=")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--limit=")
    two_word_flags+=("--limit")
    local_nonpersistent_flags+=("--limit")
    local_nonpersistent_flags+=("--limit=")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--starting-after=")
    two_word_flags+=("--starting-after")
    local_nonpersistent_flags+=("--starting-after")
    local_nonpersistent_flags+=("--starting-after=")
    flags+=("--status=")
    two_word_flags+=("--status")
    local_nonpersistent_flags+=("--status")
    local_nonpersistent_flags+=("--status=")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_orders_pay()
{
    last_command="stripe_orders_pay"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--application-fee=")
    two_word_flags+=("--application-fee")
    local_nonpersistent_flags+=("--application-fee")
    local_nonpersistent_flags+=("--application-fee=")
    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--customer=")
    two_word_flags+=("--customer")
    local_nonpersistent_flags+=("--customer")
    local_nonpersistent_flags+=("--customer=")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--email=")
    two_word_flags+=("--email")
    local_nonpersistent_flags+=("--email")
    local_nonpersistent_flags+=("--email=")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--source=")
    two_word_flags+=("--source")
    local_nonpersistent_flags+=("--source")
    local_nonpersistent_flags+=("--source=")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_orders_retrieve()
{
    last_command="stripe_orders_retrieve"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--ending-before=")
    two_word_flags+=("--ending-before")
    two_word_flags+=("-b")
    local_nonpersistent_flags+=("--ending-before")
    local_nonpersistent_flags+=("--ending-before=")
    local_nonpersistent_flags+=("-b")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--limit=")
    two_word_flags+=("--limit")
    two_word_flags+=("-l")
    local_nonpersistent_flags+=("--limit")
    local_nonpersistent_flags+=("--limit=")
    local_nonpersistent_flags+=("-l")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--starting-after=")
    two_word_flags+=("--starting-after")
    two_word_flags+=("-a")
    local_nonpersistent_flags+=("--starting-after")
    local_nonpersistent_flags+=("--starting-after=")
    local_nonpersistent_flags+=("-a")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_orders_return_order()
{
    last_command="stripe_orders_return_order"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_orders_update()
{
    last_command="stripe_orders_update"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--coupon=")
    two_word_flags+=("--coupon")
    local_nonpersistent_flags+=("--coupon")
    local_nonpersistent_flags+=("--coupon=")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--selected-shipping-method=")
    two_word_flags+=("--selected-shipping-method")
    local_nonpersistent_flags+=("--selected-shipping-method")
    local_nonpersistent_flags+=("--selected-shipping-method=")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--status=")
    two_word_flags+=("--status")
    local_nonpersistent_flags+=("--status")
    local_nonpersistent_flags+=("--status=")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_orders()
{
    last_command="stripe_orders"

    command_aliases=()

    commands=()
    commands+=("create")
    commands+=("list")
    commands+=("pay")
    commands+=("retrieve")
    commands+=("return_order")
    commands+=("update")

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_payment_intents_cancel()
{
    last_command="stripe_payment_intents_cancel"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--cancellation-reason=")
    two_word_flags+=("--cancellation-reason")
    local_nonpersistent_flags+=("--cancellation-reason")
    local_nonpersistent_flags+=("--cancellation-reason=")
    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_payment_intents_capture()
{
    last_command="stripe_payment_intents_capture"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--amount-to-capture=")
    two_word_flags+=("--amount-to-capture")
    local_nonpersistent_flags+=("--amount-to-capture")
    local_nonpersistent_flags+=("--amount-to-capture=")
    flags+=("--application-fee-amount=")
    two_word_flags+=("--application-fee-amount")
    local_nonpersistent_flags+=("--application-fee-amount")
    local_nonpersistent_flags+=("--application-fee-amount=")
    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--statement-descriptor=")
    two_word_flags+=("--statement-descriptor")
    local_nonpersistent_flags+=("--statement-descriptor")
    local_nonpersistent_flags+=("--statement-descriptor=")
    flags+=("--statement-descriptor-suffix=")
    two_word_flags+=("--statement-descriptor-suffix")
    local_nonpersistent_flags+=("--statement-descriptor-suffix")
    local_nonpersistent_flags+=("--statement-descriptor-suffix=")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_payment_intents_confirm()
{
    last_command="stripe_payment_intents_confirm"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--error-on-requires-action=")
    two_word_flags+=("--error-on-requires-action")
    local_nonpersistent_flags+=("--error-on-requires-action")
    local_nonpersistent_flags+=("--error-on-requires-action=")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--mandate=")
    two_word_flags+=("--mandate")
    local_nonpersistent_flags+=("--mandate")
    local_nonpersistent_flags+=("--mandate=")
    flags+=("--off-session=")
    two_word_flags+=("--off-session")
    local_nonpersistent_flags+=("--off-session")
    local_nonpersistent_flags+=("--off-session=")
    flags+=("--payment-method=")
    two_word_flags+=("--payment-method")
    local_nonpersistent_flags+=("--payment-method")
    local_nonpersistent_flags+=("--payment-method=")
    flags+=("--receipt-email=")
    two_word_flags+=("--receipt-email")
    local_nonpersistent_flags+=("--receipt-email")
    local_nonpersistent_flags+=("--receipt-email=")
    flags+=("--return-url=")
    two_word_flags+=("--return-url")
    local_nonpersistent_flags+=("--return-url")
    local_nonpersistent_flags+=("--return-url=")
    flags+=("--setup-future-usage=")
    two_word_flags+=("--setup-future-usage")
    local_nonpersistent_flags+=("--setup-future-usage")
    local_nonpersistent_flags+=("--setup-future-usage=")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--use-stripe-sdk=")
    two_word_flags+=("--use-stripe-sdk")
    local_nonpersistent_flags+=("--use-stripe-sdk")
    local_nonpersistent_flags+=("--use-stripe-sdk=")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_payment_intents_create()
{
    last_command="stripe_payment_intents_create"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--amount=")
    two_word_flags+=("--amount")
    local_nonpersistent_flags+=("--amount")
    local_nonpersistent_flags+=("--amount=")
    flags+=("--application-fee-amount=")
    two_word_flags+=("--application-fee-amount")
    local_nonpersistent_flags+=("--application-fee-amount")
    local_nonpersistent_flags+=("--application-fee-amount=")
    flags+=("--capture-method=")
    two_word_flags+=("--capture-method")
    local_nonpersistent_flags+=("--capture-method")
    local_nonpersistent_flags+=("--capture-method=")
    flags+=("--confirm=")
    two_word_flags+=("--confirm")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("--confirm=")
    flags+=("--confirmation-method=")
    two_word_flags+=("--confirmation-method")
    local_nonpersistent_flags+=("--confirmation-method")
    local_nonpersistent_flags+=("--confirmation-method=")
    flags+=("--currency=")
    two_word_flags+=("--currency")
    local_nonpersistent_flags+=("--currency")
    local_nonpersistent_flags+=("--currency=")
    flags+=("--customer=")
    two_word_flags+=("--customer")
    local_nonpersistent_flags+=("--customer")
    local_nonpersistent_flags+=("--customer=")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--description=")
    two_word_flags+=("--description")
    local_nonpersistent_flags+=("--description")
    local_nonpersistent_flags+=("--description=")
    flags+=("--error-on-requires-action=")
    two_word_flags+=("--error-on-requires-action")
    local_nonpersistent_flags+=("--error-on-requires-action")
    local_nonpersistent_flags+=("--error-on-requires-action=")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--mandate=")
    two_word_flags+=("--mandate")
    local_nonpersistent_flags+=("--mandate")
    local_nonpersistent_flags+=("--mandate=")
    flags+=("--off-session=")
    two_word_flags+=("--off-session")
    local_nonpersistent_flags+=("--off-session")
    local_nonpersistent_flags+=("--off-session=")
    flags+=("--on-behalf-of=")
    two_word_flags+=("--on-behalf-of")
    local_nonpersistent_flags+=("--on-behalf-of")
    local_nonpersistent_flags+=("--on-behalf-of=")
    flags+=("--payment-method=")
    two_word_flags+=("--payment-method")
    local_nonpersistent_flags+=("--payment-method")
    local_nonpersistent_flags+=("--payment-method=")
    flags+=("--receipt-email=")
    two_word_flags+=("--receipt-email")
    local_nonpersistent_flags+=("--receipt-email")
    local_nonpersistent_flags+=("--receipt-email=")
    flags+=("--return-url=")
    two_word_flags+=("--return-url")
    local_nonpersistent_flags+=("--return-url")
    local_nonpersistent_flags+=("--return-url=")
    flags+=("--setup-future-usage=")
    two_word_flags+=("--setup-future-usage")
    local_nonpersistent_flags+=("--setup-future-usage")
    local_nonpersistent_flags+=("--setup-future-usage=")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--statement-descriptor=")
    two_word_flags+=("--statement-descriptor")
    local_nonpersistent_flags+=("--statement-descriptor")
    local_nonpersistent_flags+=("--statement-descriptor=")
    flags+=("--statement-descriptor-suffix=")
    two_word_flags+=("--statement-descriptor-suffix")
    local_nonpersistent_flags+=("--statement-descriptor-suffix")
    local_nonpersistent_flags+=("--statement-descriptor-suffix=")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--transfer-group=")
    two_word_flags+=("--transfer-group")
    local_nonpersistent_flags+=("--transfer-group")
    local_nonpersistent_flags+=("--transfer-group=")
    flags+=("--use-stripe-sdk=")
    two_word_flags+=("--use-stripe-sdk")
    local_nonpersistent_flags+=("--use-stripe-sdk")
    local_nonpersistent_flags+=("--use-stripe-sdk=")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_payment_intents_list()
{
    last_command="stripe_payment_intents_list"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--created=")
    two_word_flags+=("--created")
    local_nonpersistent_flags+=("--created")
    local_nonpersistent_flags+=("--created=")
    flags+=("--customer=")
    two_word_flags+=("--customer")
    local_nonpersistent_flags+=("--customer")
    local_nonpersistent_flags+=("--customer=")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--ending-before=")
    two_word_flags+=("--ending-before")
    local_nonpersistent_flags+=("--ending-before")
    local_nonpersistent_flags+=("--ending-before=")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--limit=")
    two_word_flags+=("--limit")
    local_nonpersistent_flags+=("--limit")
    local_nonpersistent_flags+=("--limit=")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--starting-after=")
    two_word_flags+=("--starting-after")
    local_nonpersistent_flags+=("--starting-after")
    local_nonpersistent_flags+=("--starting-after=")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_payment_intents_retrieve()
{
    last_command="stripe_payment_intents_retrieve"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--client-secret=")
    two_word_flags+=("--client-secret")
    local_nonpersistent_flags+=("--client-secret")
    local_nonpersistent_flags+=("--client-secret=")
    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--ending-before=")
    two_word_flags+=("--ending-before")
    two_word_flags+=("-b")
    local_nonpersistent_flags+=("--ending-before")
    local_nonpersistent_flags+=("--ending-before=")
    local_nonpersistent_flags+=("-b")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--limit=")
    two_word_flags+=("--limit")
    two_word_flags+=("-l")
    local_nonpersistent_flags+=("--limit")
    local_nonpersistent_flags+=("--limit=")
    local_nonpersistent_flags+=("-l")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--starting-after=")
    two_word_flags+=("--starting-after")
    two_word_flags+=("-a")
    local_nonpersistent_flags+=("--starting-after")
    local_nonpersistent_flags+=("--starting-after=")
    local_nonpersistent_flags+=("-a")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_payment_intents_update()
{
    last_command="stripe_payment_intents_update"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--amount=")
    two_word_flags+=("--amount")
    local_nonpersistent_flags+=("--amount")
    local_nonpersistent_flags+=("--amount=")
    flags+=("--application-fee-amount=")
    two_word_flags+=("--application-fee-amount")
    local_nonpersistent_flags+=("--application-fee-amount")
    local_nonpersistent_flags+=("--application-fee-amount=")
    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--currency=")
    two_word_flags+=("--currency")
    local_nonpersistent_flags+=("--currency")
    local_nonpersistent_flags+=("--currency=")
    flags+=("--customer=")
    two_word_flags+=("--customer")
    local_nonpersistent_flags+=("--customer")
    local_nonpersistent_flags+=("--customer=")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--description=")
    two_word_flags+=("--description")
    local_nonpersistent_flags+=("--description")
    local_nonpersistent_flags+=("--description=")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--payment-method=")
    two_word_flags+=("--payment-method")
    local_nonpersistent_flags+=("--payment-method")
    local_nonpersistent_flags+=("--payment-method=")
    flags+=("--receipt-email=")
    two_word_flags+=("--receipt-email")
    local_nonpersistent_flags+=("--receipt-email")
    local_nonpersistent_flags+=("--receipt-email=")
    flags+=("--setup-future-usage=")
    two_word_flags+=("--setup-future-usage")
    local_nonpersistent_flags+=("--setup-future-usage")
    local_nonpersistent_flags+=("--setup-future-usage=")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--statement-descriptor=")
    two_word_flags+=("--statement-descriptor")
    local_nonpersistent_flags+=("--statement-descriptor")
    local_nonpersistent_flags+=("--statement-descriptor=")
    flags+=("--statement-descriptor-suffix=")
    two_word_flags+=("--statement-descriptor-suffix")
    local_nonpersistent_flags+=("--statement-descriptor-suffix")
    local_nonpersistent_flags+=("--statement-descriptor-suffix=")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--transfer-group=")
    two_word_flags+=("--transfer-group")
    local_nonpersistent_flags+=("--transfer-group")
    local_nonpersistent_flags+=("--transfer-group=")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_payment_intents()
{
    last_command="stripe_payment_intents"

    command_aliases=()

    commands=()
    commands+=("cancel")
    commands+=("capture")
    commands+=("confirm")
    commands+=("create")
    commands+=("list")
    commands+=("retrieve")
    commands+=("update")

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_payment_methods_attach()
{
    last_command="stripe_payment_methods_attach"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--customer=")
    two_word_flags+=("--customer")
    local_nonpersistent_flags+=("--customer")
    local_nonpersistent_flags+=("--customer=")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_payment_methods_create()
{
    last_command="stripe_payment_methods_create"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--customer=")
    two_word_flags+=("--customer")
    local_nonpersistent_flags+=("--customer")
    local_nonpersistent_flags+=("--customer=")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--payment-method=")
    two_word_flags+=("--payment-method")
    local_nonpersistent_flags+=("--payment-method")
    local_nonpersistent_flags+=("--payment-method=")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--type=")
    two_word_flags+=("--type")
    local_nonpersistent_flags+=("--type")
    local_nonpersistent_flags+=("--type=")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_payment_methods_detach()
{
    last_command="stripe_payment_methods_detach"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_payment_methods_list()
{
    last_command="stripe_payment_methods_list"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--customer=")
    two_word_flags+=("--customer")
    local_nonpersistent_flags+=("--customer")
    local_nonpersistent_flags+=("--customer=")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--ending-before=")
    two_word_flags+=("--ending-before")
    local_nonpersistent_flags+=("--ending-before")
    local_nonpersistent_flags+=("--ending-before=")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--limit=")
    two_word_flags+=("--limit")
    local_nonpersistent_flags+=("--limit")
    local_nonpersistent_flags+=("--limit=")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--starting-after=")
    two_word_flags+=("--starting-after")
    local_nonpersistent_flags+=("--starting-after")
    local_nonpersistent_flags+=("--starting-after=")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--type=")
    two_word_flags+=("--type")
    local_nonpersistent_flags+=("--type")
    local_nonpersistent_flags+=("--type=")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_payment_methods_retrieve()
{
    last_command="stripe_payment_methods_retrieve"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--ending-before=")
    two_word_flags+=("--ending-before")
    two_word_flags+=("-b")
    local_nonpersistent_flags+=("--ending-before")
    local_nonpersistent_flags+=("--ending-before=")
    local_nonpersistent_flags+=("-b")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--limit=")
    two_word_flags+=("--limit")
    two_word_flags+=("-l")
    local_nonpersistent_flags+=("--limit")
    local_nonpersistent_flags+=("--limit=")
    local_nonpersistent_flags+=("-l")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--starting-after=")
    two_word_flags+=("--starting-after")
    two_word_flags+=("-a")
    local_nonpersistent_flags+=("--starting-after")
    local_nonpersistent_flags+=("--starting-after=")
    local_nonpersistent_flags+=("-a")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_payment_methods_update()
{
    last_command="stripe_payment_methods_update"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_payment_methods()
{
    last_command="stripe_payment_methods"

    command_aliases=()

    commands=()
    commands+=("attach")
    commands+=("create")
    commands+=("detach")
    commands+=("list")
    commands+=("retrieve")
    commands+=("update")

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_payment_sources_create()
{
    last_command="stripe_payment_sources_create"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--source=")
    two_word_flags+=("--source")
    local_nonpersistent_flags+=("--source")
    local_nonpersistent_flags+=("--source=")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_payment_sources_list()
{
    last_command="stripe_payment_sources_list"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--ending-before=")
    two_word_flags+=("--ending-before")
    local_nonpersistent_flags+=("--ending-before")
    local_nonpersistent_flags+=("--ending-before=")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--limit=")
    two_word_flags+=("--limit")
    local_nonpersistent_flags+=("--limit")
    local_nonpersistent_flags+=("--limit=")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--object=")
    two_word_flags+=("--object")
    local_nonpersistent_flags+=("--object")
    local_nonpersistent_flags+=("--object=")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--starting-after=")
    two_word_flags+=("--starting-after")
    local_nonpersistent_flags+=("--starting-after")
    local_nonpersistent_flags+=("--starting-after=")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_payment_sources_retrieve()
{
    last_command="stripe_payment_sources_retrieve"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--ending-before=")
    two_word_flags+=("--ending-before")
    two_word_flags+=("-b")
    local_nonpersistent_flags+=("--ending-before")
    local_nonpersistent_flags+=("--ending-before=")
    local_nonpersistent_flags+=("-b")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--limit=")
    two_word_flags+=("--limit")
    two_word_flags+=("-l")
    local_nonpersistent_flags+=("--limit")
    local_nonpersistent_flags+=("--limit=")
    local_nonpersistent_flags+=("-l")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--starting-after=")
    two_word_flags+=("--starting-after")
    two_word_flags+=("-a")
    local_nonpersistent_flags+=("--starting-after")
    local_nonpersistent_flags+=("--starting-after=")
    local_nonpersistent_flags+=("-a")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_payment_sources()
{
    last_command="stripe_payment_sources"

    command_aliases=()

    commands=()
    commands+=("create")
    commands+=("list")
    commands+=("retrieve")

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_payouts_cancel()
{
    last_command="stripe_payouts_cancel"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_payouts_create()
{
    last_command="stripe_payouts_create"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--amount=")
    two_word_flags+=("--amount")
    local_nonpersistent_flags+=("--amount")
    local_nonpersistent_flags+=("--amount=")
    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--currency=")
    two_word_flags+=("--currency")
    local_nonpersistent_flags+=("--currency")
    local_nonpersistent_flags+=("--currency=")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--description=")
    two_word_flags+=("--description")
    local_nonpersistent_flags+=("--description")
    local_nonpersistent_flags+=("--description=")
    flags+=("--destination=")
    two_word_flags+=("--destination")
    local_nonpersistent_flags+=("--destination")
    local_nonpersistent_flags+=("--destination=")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--method=")
    two_word_flags+=("--method")
    local_nonpersistent_flags+=("--method")
    local_nonpersistent_flags+=("--method=")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--source-type=")
    two_word_flags+=("--source-type")
    local_nonpersistent_flags+=("--source-type")
    local_nonpersistent_flags+=("--source-type=")
    flags+=("--statement-descriptor=")
    two_word_flags+=("--statement-descriptor")
    local_nonpersistent_flags+=("--statement-descriptor")
    local_nonpersistent_flags+=("--statement-descriptor=")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_payouts_list()
{
    last_command="stripe_payouts_list"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--arrival-date=")
    two_word_flags+=("--arrival-date")
    local_nonpersistent_flags+=("--arrival-date")
    local_nonpersistent_flags+=("--arrival-date=")
    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--created=")
    two_word_flags+=("--created")
    local_nonpersistent_flags+=("--created")
    local_nonpersistent_flags+=("--created=")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--destination=")
    two_word_flags+=("--destination")
    local_nonpersistent_flags+=("--destination")
    local_nonpersistent_flags+=("--destination=")
    flags+=("--ending-before=")
    two_word_flags+=("--ending-before")
    local_nonpersistent_flags+=("--ending-before")
    local_nonpersistent_flags+=("--ending-before=")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--limit=")
    two_word_flags+=("--limit")
    local_nonpersistent_flags+=("--limit")
    local_nonpersistent_flags+=("--limit=")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--starting-after=")
    two_word_flags+=("--starting-after")
    local_nonpersistent_flags+=("--starting-after")
    local_nonpersistent_flags+=("--starting-after=")
    flags+=("--status=")
    two_word_flags+=("--status")
    local_nonpersistent_flags+=("--status")
    local_nonpersistent_flags+=("--status=")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_payouts_retrieve()
{
    last_command="stripe_payouts_retrieve"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--ending-before=")
    two_word_flags+=("--ending-before")
    two_word_flags+=("-b")
    local_nonpersistent_flags+=("--ending-before")
    local_nonpersistent_flags+=("--ending-before=")
    local_nonpersistent_flags+=("-b")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--limit=")
    two_word_flags+=("--limit")
    two_word_flags+=("-l")
    local_nonpersistent_flags+=("--limit")
    local_nonpersistent_flags+=("--limit=")
    local_nonpersistent_flags+=("-l")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--starting-after=")
    two_word_flags+=("--starting-after")
    two_word_flags+=("-a")
    local_nonpersistent_flags+=("--starting-after")
    local_nonpersistent_flags+=("--starting-after=")
    local_nonpersistent_flags+=("-a")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_payouts_reverse()
{
    last_command="stripe_payouts_reverse"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_payouts_update()
{
    last_command="stripe_payouts_update"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_payouts()
{
    last_command="stripe_payouts"

    command_aliases=()

    commands=()
    commands+=("cancel")
    commands+=("create")
    commands+=("list")
    commands+=("retrieve")
    commands+=("reverse")
    commands+=("update")

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_persons_create()
{
    last_command="stripe_persons_create"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--email=")
    two_word_flags+=("--email")
    local_nonpersistent_flags+=("--email")
    local_nonpersistent_flags+=("--email=")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--first-name=")
    two_word_flags+=("--first-name")
    local_nonpersistent_flags+=("--first-name")
    local_nonpersistent_flags+=("--first-name=")
    flags+=("--first-name-kana=")
    two_word_flags+=("--first-name-kana")
    local_nonpersistent_flags+=("--first-name-kana")
    local_nonpersistent_flags+=("--first-name-kana=")
    flags+=("--first-name-kanji=")
    two_word_flags+=("--first-name-kanji")
    local_nonpersistent_flags+=("--first-name-kanji")
    local_nonpersistent_flags+=("--first-name-kanji=")
    flags+=("--gender=")
    two_word_flags+=("--gender")
    local_nonpersistent_flags+=("--gender")
    local_nonpersistent_flags+=("--gender=")
    flags+=("--id-number=")
    two_word_flags+=("--id-number")
    local_nonpersistent_flags+=("--id-number")
    local_nonpersistent_flags+=("--id-number=")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--last-name=")
    two_word_flags+=("--last-name")
    local_nonpersistent_flags+=("--last-name")
    local_nonpersistent_flags+=("--last-name=")
    flags+=("--last-name-kana=")
    two_word_flags+=("--last-name-kana")
    local_nonpersistent_flags+=("--last-name-kana")
    local_nonpersistent_flags+=("--last-name-kana=")
    flags+=("--last-name-kanji=")
    two_word_flags+=("--last-name-kanji")
    local_nonpersistent_flags+=("--last-name-kanji")
    local_nonpersistent_flags+=("--last-name-kanji=")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--maiden-name=")
    two_word_flags+=("--maiden-name")
    local_nonpersistent_flags+=("--maiden-name")
    local_nonpersistent_flags+=("--maiden-name=")
    flags+=("--nationality=")
    two_word_flags+=("--nationality")
    local_nonpersistent_flags+=("--nationality")
    local_nonpersistent_flags+=("--nationality=")
    flags+=("--person-token=")
    two_word_flags+=("--person-token")
    local_nonpersistent_flags+=("--person-token")
    local_nonpersistent_flags+=("--person-token=")
    flags+=("--phone=")
    two_word_flags+=("--phone")
    local_nonpersistent_flags+=("--phone")
    local_nonpersistent_flags+=("--phone=")
    flags+=("--political-exposure=")
    two_word_flags+=("--political-exposure")
    local_nonpersistent_flags+=("--political-exposure")
    local_nonpersistent_flags+=("--political-exposure=")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--ssn-last-4=")
    two_word_flags+=("--ssn-last-4")
    local_nonpersistent_flags+=("--ssn-last-4")
    local_nonpersistent_flags+=("--ssn-last-4=")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_persons_delete()
{
    last_command="stripe_persons_delete"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_persons_list()
{
    last_command="stripe_persons_list"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--ending-before=")
    two_word_flags+=("--ending-before")
    local_nonpersistent_flags+=("--ending-before")
    local_nonpersistent_flags+=("--ending-before=")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--limit=")
    two_word_flags+=("--limit")
    local_nonpersistent_flags+=("--limit")
    local_nonpersistent_flags+=("--limit=")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--starting-after=")
    two_word_flags+=("--starting-after")
    local_nonpersistent_flags+=("--starting-after")
    local_nonpersistent_flags+=("--starting-after=")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_persons_retrieve()
{
    last_command="stripe_persons_retrieve"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--ending-before=")
    two_word_flags+=("--ending-before")
    two_word_flags+=("-b")
    local_nonpersistent_flags+=("--ending-before")
    local_nonpersistent_flags+=("--ending-before=")
    local_nonpersistent_flags+=("-b")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--limit=")
    two_word_flags+=("--limit")
    two_word_flags+=("-l")
    local_nonpersistent_flags+=("--limit")
    local_nonpersistent_flags+=("--limit=")
    local_nonpersistent_flags+=("-l")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--starting-after=")
    two_word_flags+=("--starting-after")
    two_word_flags+=("-a")
    local_nonpersistent_flags+=("--starting-after")
    local_nonpersistent_flags+=("--starting-after=")
    local_nonpersistent_flags+=("-a")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_persons_update()
{
    last_command="stripe_persons_update"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--email=")
    two_word_flags+=("--email")
    local_nonpersistent_flags+=("--email")
    local_nonpersistent_flags+=("--email=")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--first-name=")
    two_word_flags+=("--first-name")
    local_nonpersistent_flags+=("--first-name")
    local_nonpersistent_flags+=("--first-name=")
    flags+=("--first-name-kana=")
    two_word_flags+=("--first-name-kana")
    local_nonpersistent_flags+=("--first-name-kana")
    local_nonpersistent_flags+=("--first-name-kana=")
    flags+=("--first-name-kanji=")
    two_word_flags+=("--first-name-kanji")
    local_nonpersistent_flags+=("--first-name-kanji")
    local_nonpersistent_flags+=("--first-name-kanji=")
    flags+=("--gender=")
    two_word_flags+=("--gender")
    local_nonpersistent_flags+=("--gender")
    local_nonpersistent_flags+=("--gender=")
    flags+=("--id-number=")
    two_word_flags+=("--id-number")
    local_nonpersistent_flags+=("--id-number")
    local_nonpersistent_flags+=("--id-number=")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--last-name=")
    two_word_flags+=("--last-name")
    local_nonpersistent_flags+=("--last-name")
    local_nonpersistent_flags+=("--last-name=")
    flags+=("--last-name-kana=")
    two_word_flags+=("--last-name-kana")
    local_nonpersistent_flags+=("--last-name-kana")
    local_nonpersistent_flags+=("--last-name-kana=")
    flags+=("--last-name-kanji=")
    two_word_flags+=("--last-name-kanji")
    local_nonpersistent_flags+=("--last-name-kanji")
    local_nonpersistent_flags+=("--last-name-kanji=")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--maiden-name=")
    two_word_flags+=("--maiden-name")
    local_nonpersistent_flags+=("--maiden-name")
    local_nonpersistent_flags+=("--maiden-name=")
    flags+=("--nationality=")
    two_word_flags+=("--nationality")
    local_nonpersistent_flags+=("--nationality")
    local_nonpersistent_flags+=("--nationality=")
    flags+=("--person-token=")
    two_word_flags+=("--person-token")
    local_nonpersistent_flags+=("--person-token")
    local_nonpersistent_flags+=("--person-token=")
    flags+=("--phone=")
    two_word_flags+=("--phone")
    local_nonpersistent_flags+=("--phone")
    local_nonpersistent_flags+=("--phone=")
    flags+=("--political-exposure=")
    two_word_flags+=("--political-exposure")
    local_nonpersistent_flags+=("--political-exposure")
    local_nonpersistent_flags+=("--political-exposure=")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--ssn-last-4=")
    two_word_flags+=("--ssn-last-4")
    local_nonpersistent_flags+=("--ssn-last-4")
    local_nonpersistent_flags+=("--ssn-last-4=")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_persons()
{
    last_command="stripe_persons"

    command_aliases=()

    commands=()
    commands+=("create")
    commands+=("delete")
    commands+=("list")
    commands+=("retrieve")
    commands+=("update")

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_plans_create()
{
    last_command="stripe_plans_create"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--active=")
    two_word_flags+=("--active")
    local_nonpersistent_flags+=("--active")
    local_nonpersistent_flags+=("--active=")
    flags+=("--aggregate-usage=")
    two_word_flags+=("--aggregate-usage")
    local_nonpersistent_flags+=("--aggregate-usage")
    local_nonpersistent_flags+=("--aggregate-usage=")
    flags+=("--amount=")
    two_word_flags+=("--amount")
    local_nonpersistent_flags+=("--amount")
    local_nonpersistent_flags+=("--amount=")
    flags+=("--amount-decimal=")
    two_word_flags+=("--amount-decimal")
    local_nonpersistent_flags+=("--amount-decimal")
    local_nonpersistent_flags+=("--amount-decimal=")
    flags+=("--billing-scheme=")
    two_word_flags+=("--billing-scheme")
    local_nonpersistent_flags+=("--billing-scheme")
    local_nonpersistent_flags+=("--billing-scheme=")
    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--currency=")
    two_word_flags+=("--currency")
    local_nonpersistent_flags+=("--currency")
    local_nonpersistent_flags+=("--currency=")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--id=")
    two_word_flags+=("--id")
    local_nonpersistent_flags+=("--id")
    local_nonpersistent_flags+=("--id=")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--interval=")
    two_word_flags+=("--interval")
    local_nonpersistent_flags+=("--interval")
    local_nonpersistent_flags+=("--interval=")
    flags+=("--interval-count=")
    two_word_flags+=("--interval-count")
    local_nonpersistent_flags+=("--interval-count")
    local_nonpersistent_flags+=("--interval-count=")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--nickname=")
    two_word_flags+=("--nickname")
    local_nonpersistent_flags+=("--nickname")
    local_nonpersistent_flags+=("--nickname=")
    flags+=("--product=")
    two_word_flags+=("--product")
    local_nonpersistent_flags+=("--product")
    local_nonpersistent_flags+=("--product=")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--tiers-mode=")
    two_word_flags+=("--tiers-mode")
    local_nonpersistent_flags+=("--tiers-mode")
    local_nonpersistent_flags+=("--tiers-mode=")
    flags+=("--trial-period-days=")
    two_word_flags+=("--trial-period-days")
    local_nonpersistent_flags+=("--trial-period-days")
    local_nonpersistent_flags+=("--trial-period-days=")
    flags+=("--usage-type=")
    two_word_flags+=("--usage-type")
    local_nonpersistent_flags+=("--usage-type")
    local_nonpersistent_flags+=("--usage-type=")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_plans_delete()
{
    last_command="stripe_plans_delete"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_plans_list()
{
    last_command="stripe_plans_list"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--active=")
    two_word_flags+=("--active")
    local_nonpersistent_flags+=("--active")
    local_nonpersistent_flags+=("--active=")
    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--created=")
    two_word_flags+=("--created")
    local_nonpersistent_flags+=("--created")
    local_nonpersistent_flags+=("--created=")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--ending-before=")
    two_word_flags+=("--ending-before")
    local_nonpersistent_flags+=("--ending-before")
    local_nonpersistent_flags+=("--ending-before=")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--limit=")
    two_word_flags+=("--limit")
    local_nonpersistent_flags+=("--limit")
    local_nonpersistent_flags+=("--limit=")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--product=")
    two_word_flags+=("--product")
    local_nonpersistent_flags+=("--product")
    local_nonpersistent_flags+=("--product=")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--starting-after=")
    two_word_flags+=("--starting-after")
    local_nonpersistent_flags+=("--starting-after")
    local_nonpersistent_flags+=("--starting-after=")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_plans_retrieve()
{
    last_command="stripe_plans_retrieve"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--ending-before=")
    two_word_flags+=("--ending-before")
    two_word_flags+=("-b")
    local_nonpersistent_flags+=("--ending-before")
    local_nonpersistent_flags+=("--ending-before=")
    local_nonpersistent_flags+=("-b")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--limit=")
    two_word_flags+=("--limit")
    two_word_flags+=("-l")
    local_nonpersistent_flags+=("--limit")
    local_nonpersistent_flags+=("--limit=")
    local_nonpersistent_flags+=("-l")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--starting-after=")
    two_word_flags+=("--starting-after")
    two_word_flags+=("-a")
    local_nonpersistent_flags+=("--starting-after")
    local_nonpersistent_flags+=("--starting-after=")
    local_nonpersistent_flags+=("-a")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_plans_update()
{
    last_command="stripe_plans_update"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--active=")
    two_word_flags+=("--active")
    local_nonpersistent_flags+=("--active")
    local_nonpersistent_flags+=("--active=")
    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--nickname=")
    two_word_flags+=("--nickname")
    local_nonpersistent_flags+=("--nickname")
    local_nonpersistent_flags+=("--nickname=")
    flags+=("--product=")
    two_word_flags+=("--product")
    local_nonpersistent_flags+=("--product")
    local_nonpersistent_flags+=("--product=")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--trial-period-days=")
    two_word_flags+=("--trial-period-days")
    local_nonpersistent_flags+=("--trial-period-days")
    local_nonpersistent_flags+=("--trial-period-days=")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_plans()
{
    last_command="stripe_plans"

    command_aliases=()

    commands=()
    commands+=("create")
    commands+=("delete")
    commands+=("list")
    commands+=("retrieve")
    commands+=("update")

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_post()
{
    last_command="stripe_post"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_prices_create()
{
    last_command="stripe_prices_create"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--active=")
    two_word_flags+=("--active")
    local_nonpersistent_flags+=("--active")
    local_nonpersistent_flags+=("--active=")
    flags+=("--billing-scheme=")
    two_word_flags+=("--billing-scheme")
    local_nonpersistent_flags+=("--billing-scheme")
    local_nonpersistent_flags+=("--billing-scheme=")
    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--currency=")
    two_word_flags+=("--currency")
    local_nonpersistent_flags+=("--currency")
    local_nonpersistent_flags+=("--currency=")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--lookup-key=")
    two_word_flags+=("--lookup-key")
    local_nonpersistent_flags+=("--lookup-key")
    local_nonpersistent_flags+=("--lookup-key=")
    flags+=("--nickname=")
    two_word_flags+=("--nickname")
    local_nonpersistent_flags+=("--nickname")
    local_nonpersistent_flags+=("--nickname=")
    flags+=("--product=")
    two_word_flags+=("--product")
    local_nonpersistent_flags+=("--product")
    local_nonpersistent_flags+=("--product=")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--tax-behavior=")
    two_word_flags+=("--tax-behavior")
    local_nonpersistent_flags+=("--tax-behavior")
    local_nonpersistent_flags+=("--tax-behavior=")
    flags+=("--tiers-mode=")
    two_word_flags+=("--tiers-mode")
    local_nonpersistent_flags+=("--tiers-mode")
    local_nonpersistent_flags+=("--tiers-mode=")
    flags+=("--transfer-lookup-key=")
    two_word_flags+=("--transfer-lookup-key")
    local_nonpersistent_flags+=("--transfer-lookup-key")
    local_nonpersistent_flags+=("--transfer-lookup-key=")
    flags+=("--unit-amount=")
    two_word_flags+=("--unit-amount")
    local_nonpersistent_flags+=("--unit-amount")
    local_nonpersistent_flags+=("--unit-amount=")
    flags+=("--unit-amount-decimal=")
    two_word_flags+=("--unit-amount-decimal")
    local_nonpersistent_flags+=("--unit-amount-decimal")
    local_nonpersistent_flags+=("--unit-amount-decimal=")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_prices_list()
{
    last_command="stripe_prices_list"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--active=")
    two_word_flags+=("--active")
    local_nonpersistent_flags+=("--active")
    local_nonpersistent_flags+=("--active=")
    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--created=")
    two_word_flags+=("--created")
    local_nonpersistent_flags+=("--created")
    local_nonpersistent_flags+=("--created=")
    flags+=("--currency=")
    two_word_flags+=("--currency")
    local_nonpersistent_flags+=("--currency")
    local_nonpersistent_flags+=("--currency=")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--ending-before=")
    two_word_flags+=("--ending-before")
    local_nonpersistent_flags+=("--ending-before")
    local_nonpersistent_flags+=("--ending-before=")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--limit=")
    two_word_flags+=("--limit")
    local_nonpersistent_flags+=("--limit")
    local_nonpersistent_flags+=("--limit=")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--product=")
    two_word_flags+=("--product")
    local_nonpersistent_flags+=("--product")
    local_nonpersistent_flags+=("--product=")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--starting-after=")
    two_word_flags+=("--starting-after")
    local_nonpersistent_flags+=("--starting-after")
    local_nonpersistent_flags+=("--starting-after=")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--type=")
    two_word_flags+=("--type")
    local_nonpersistent_flags+=("--type")
    local_nonpersistent_flags+=("--type=")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_prices_retrieve()
{
    last_command="stripe_prices_retrieve"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--ending-before=")
    two_word_flags+=("--ending-before")
    two_word_flags+=("-b")
    local_nonpersistent_flags+=("--ending-before")
    local_nonpersistent_flags+=("--ending-before=")
    local_nonpersistent_flags+=("-b")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--limit=")
    two_word_flags+=("--limit")
    two_word_flags+=("-l")
    local_nonpersistent_flags+=("--limit")
    local_nonpersistent_flags+=("--limit=")
    local_nonpersistent_flags+=("-l")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--starting-after=")
    two_word_flags+=("--starting-after")
    two_word_flags+=("-a")
    local_nonpersistent_flags+=("--starting-after")
    local_nonpersistent_flags+=("--starting-after=")
    local_nonpersistent_flags+=("-a")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_prices_update()
{
    last_command="stripe_prices_update"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--active=")
    two_word_flags+=("--active")
    local_nonpersistent_flags+=("--active")
    local_nonpersistent_flags+=("--active=")
    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--lookup-key=")
    two_word_flags+=("--lookup-key")
    local_nonpersistent_flags+=("--lookup-key")
    local_nonpersistent_flags+=("--lookup-key=")
    flags+=("--nickname=")
    two_word_flags+=("--nickname")
    local_nonpersistent_flags+=("--nickname")
    local_nonpersistent_flags+=("--nickname=")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--tax-behavior=")
    two_word_flags+=("--tax-behavior")
    local_nonpersistent_flags+=("--tax-behavior")
    local_nonpersistent_flags+=("--tax-behavior=")
    flags+=("--transfer-lookup-key=")
    two_word_flags+=("--transfer-lookup-key")
    local_nonpersistent_flags+=("--transfer-lookup-key")
    local_nonpersistent_flags+=("--transfer-lookup-key=")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_prices()
{
    last_command="stripe_prices"

    command_aliases=()

    commands=()
    commands+=("create")
    commands+=("list")
    commands+=("retrieve")
    commands+=("update")

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_products_create()
{
    last_command="stripe_products_create"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--active=")
    two_word_flags+=("--active")
    local_nonpersistent_flags+=("--active")
    local_nonpersistent_flags+=("--active=")
    flags+=("--caption=")
    two_word_flags+=("--caption")
    local_nonpersistent_flags+=("--caption")
    local_nonpersistent_flags+=("--caption=")
    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--description=")
    two_word_flags+=("--description")
    local_nonpersistent_flags+=("--description")
    local_nonpersistent_flags+=("--description=")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--id=")
    two_word_flags+=("--id")
    local_nonpersistent_flags+=("--id")
    local_nonpersistent_flags+=("--id=")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--name=")
    two_word_flags+=("--name")
    local_nonpersistent_flags+=("--name")
    local_nonpersistent_flags+=("--name=")
    flags+=("--shippable=")
    two_word_flags+=("--shippable")
    local_nonpersistent_flags+=("--shippable")
    local_nonpersistent_flags+=("--shippable=")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--statement-descriptor=")
    two_word_flags+=("--statement-descriptor")
    local_nonpersistent_flags+=("--statement-descriptor")
    local_nonpersistent_flags+=("--statement-descriptor=")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--tax-code=")
    two_word_flags+=("--tax-code")
    local_nonpersistent_flags+=("--tax-code")
    local_nonpersistent_flags+=("--tax-code=")
    flags+=("--type=")
    two_word_flags+=("--type")
    local_nonpersistent_flags+=("--type")
    local_nonpersistent_flags+=("--type=")
    flags+=("--unit-label=")
    two_word_flags+=("--unit-label")
    local_nonpersistent_flags+=("--unit-label")
    local_nonpersistent_flags+=("--unit-label=")
    flags+=("--url=")
    two_word_flags+=("--url")
    local_nonpersistent_flags+=("--url")
    local_nonpersistent_flags+=("--url=")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_products_delete()
{
    last_command="stripe_products_delete"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_products_list()
{
    last_command="stripe_products_list"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--active=")
    two_word_flags+=("--active")
    local_nonpersistent_flags+=("--active")
    local_nonpersistent_flags+=("--active=")
    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--created=")
    two_word_flags+=("--created")
    local_nonpersistent_flags+=("--created")
    local_nonpersistent_flags+=("--created=")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--ending-before=")
    two_word_flags+=("--ending-before")
    local_nonpersistent_flags+=("--ending-before")
    local_nonpersistent_flags+=("--ending-before=")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--limit=")
    two_word_flags+=("--limit")
    local_nonpersistent_flags+=("--limit")
    local_nonpersistent_flags+=("--limit=")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--shippable=")
    two_word_flags+=("--shippable")
    local_nonpersistent_flags+=("--shippable")
    local_nonpersistent_flags+=("--shippable=")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--starting-after=")
    two_word_flags+=("--starting-after")
    local_nonpersistent_flags+=("--starting-after")
    local_nonpersistent_flags+=("--starting-after=")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--type=")
    two_word_flags+=("--type")
    local_nonpersistent_flags+=("--type")
    local_nonpersistent_flags+=("--type=")
    flags+=("--url=")
    two_word_flags+=("--url")
    local_nonpersistent_flags+=("--url")
    local_nonpersistent_flags+=("--url=")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_products_retrieve()
{
    last_command="stripe_products_retrieve"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--ending-before=")
    two_word_flags+=("--ending-before")
    two_word_flags+=("-b")
    local_nonpersistent_flags+=("--ending-before")
    local_nonpersistent_flags+=("--ending-before=")
    local_nonpersistent_flags+=("-b")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--limit=")
    two_word_flags+=("--limit")
    two_word_flags+=("-l")
    local_nonpersistent_flags+=("--limit")
    local_nonpersistent_flags+=("--limit=")
    local_nonpersistent_flags+=("-l")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--starting-after=")
    two_word_flags+=("--starting-after")
    two_word_flags+=("-a")
    local_nonpersistent_flags+=("--starting-after")
    local_nonpersistent_flags+=("--starting-after=")
    local_nonpersistent_flags+=("-a")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_products_update()
{
    last_command="stripe_products_update"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--active=")
    two_word_flags+=("--active")
    local_nonpersistent_flags+=("--active")
    local_nonpersistent_flags+=("--active=")
    flags+=("--caption=")
    two_word_flags+=("--caption")
    local_nonpersistent_flags+=("--caption")
    local_nonpersistent_flags+=("--caption=")
    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--description=")
    two_word_flags+=("--description")
    local_nonpersistent_flags+=("--description")
    local_nonpersistent_flags+=("--description=")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--name=")
    two_word_flags+=("--name")
    local_nonpersistent_flags+=("--name")
    local_nonpersistent_flags+=("--name=")
    flags+=("--shippable=")
    two_word_flags+=("--shippable")
    local_nonpersistent_flags+=("--shippable")
    local_nonpersistent_flags+=("--shippable=")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--statement-descriptor=")
    two_word_flags+=("--statement-descriptor")
    local_nonpersistent_flags+=("--statement-descriptor")
    local_nonpersistent_flags+=("--statement-descriptor=")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--tax-code=")
    two_word_flags+=("--tax-code")
    local_nonpersistent_flags+=("--tax-code")
    local_nonpersistent_flags+=("--tax-code=")
    flags+=("--unit-label=")
    two_word_flags+=("--unit-label")
    local_nonpersistent_flags+=("--unit-label")
    local_nonpersistent_flags+=("--unit-label=")
    flags+=("--url=")
    two_word_flags+=("--url")
    local_nonpersistent_flags+=("--url")
    local_nonpersistent_flags+=("--url=")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_products()
{
    last_command="stripe_products"

    command_aliases=()

    commands=()
    commands+=("create")
    commands+=("delete")
    commands+=("list")
    commands+=("retrieve")
    commands+=("update")

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_promotion_codes_create()
{
    last_command="stripe_promotion_codes_create"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--active=")
    two_word_flags+=("--active")
    local_nonpersistent_flags+=("--active")
    local_nonpersistent_flags+=("--active=")
    flags+=("--code=")
    two_word_flags+=("--code")
    local_nonpersistent_flags+=("--code")
    local_nonpersistent_flags+=("--code=")
    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--coupon=")
    two_word_flags+=("--coupon")
    local_nonpersistent_flags+=("--coupon")
    local_nonpersistent_flags+=("--coupon=")
    flags+=("--customer=")
    two_word_flags+=("--customer")
    local_nonpersistent_flags+=("--customer")
    local_nonpersistent_flags+=("--customer=")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--expires-at=")
    two_word_flags+=("--expires-at")
    local_nonpersistent_flags+=("--expires-at")
    local_nonpersistent_flags+=("--expires-at=")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--max-redemptions=")
    two_word_flags+=("--max-redemptions")
    local_nonpersistent_flags+=("--max-redemptions")
    local_nonpersistent_flags+=("--max-redemptions=")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_promotion_codes_list()
{
    last_command="stripe_promotion_codes_list"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--active=")
    two_word_flags+=("--active")
    local_nonpersistent_flags+=("--active")
    local_nonpersistent_flags+=("--active=")
    flags+=("--code=")
    two_word_flags+=("--code")
    local_nonpersistent_flags+=("--code")
    local_nonpersistent_flags+=("--code=")
    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--coupon=")
    two_word_flags+=("--coupon")
    local_nonpersistent_flags+=("--coupon")
    local_nonpersistent_flags+=("--coupon=")
    flags+=("--created=")
    two_word_flags+=("--created")
    local_nonpersistent_flags+=("--created")
    local_nonpersistent_flags+=("--created=")
    flags+=("--customer=")
    two_word_flags+=("--customer")
    local_nonpersistent_flags+=("--customer")
    local_nonpersistent_flags+=("--customer=")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--ending-before=")
    two_word_flags+=("--ending-before")
    local_nonpersistent_flags+=("--ending-before")
    local_nonpersistent_flags+=("--ending-before=")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--limit=")
    two_word_flags+=("--limit")
    local_nonpersistent_flags+=("--limit")
    local_nonpersistent_flags+=("--limit=")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--starting-after=")
    two_word_flags+=("--starting-after")
    local_nonpersistent_flags+=("--starting-after")
    local_nonpersistent_flags+=("--starting-after=")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_promotion_codes_retrieve()
{
    last_command="stripe_promotion_codes_retrieve"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--ending-before=")
    two_word_flags+=("--ending-before")
    two_word_flags+=("-b")
    local_nonpersistent_flags+=("--ending-before")
    local_nonpersistent_flags+=("--ending-before=")
    local_nonpersistent_flags+=("-b")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--limit=")
    two_word_flags+=("--limit")
    two_word_flags+=("-l")
    local_nonpersistent_flags+=("--limit")
    local_nonpersistent_flags+=("--limit=")
    local_nonpersistent_flags+=("-l")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--starting-after=")
    two_word_flags+=("--starting-after")
    two_word_flags+=("-a")
    local_nonpersistent_flags+=("--starting-after")
    local_nonpersistent_flags+=("--starting-after=")
    local_nonpersistent_flags+=("-a")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_promotion_codes_update()
{
    last_command="stripe_promotion_codes_update"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--active=")
    two_word_flags+=("--active")
    local_nonpersistent_flags+=("--active")
    local_nonpersistent_flags+=("--active=")
    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_promotion_codes()
{
    last_command="stripe_promotion_codes"

    command_aliases=()

    commands=()
    commands+=("create")
    commands+=("list")
    commands+=("retrieve")
    commands+=("update")

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_quotes_accept()
{
    last_command="stripe_quotes_accept"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_quotes_cancel()
{
    last_command="stripe_quotes_cancel"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_quotes_create()
{
    last_command="stripe_quotes_create"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--application-fee-amount=")
    two_word_flags+=("--application-fee-amount")
    local_nonpersistent_flags+=("--application-fee-amount")
    local_nonpersistent_flags+=("--application-fee-amount=")
    flags+=("--application-fee-percent=")
    two_word_flags+=("--application-fee-percent")
    local_nonpersistent_flags+=("--application-fee-percent")
    local_nonpersistent_flags+=("--application-fee-percent=")
    flags+=("--collection-method=")
    two_word_flags+=("--collection-method")
    local_nonpersistent_flags+=("--collection-method")
    local_nonpersistent_flags+=("--collection-method=")
    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--customer=")
    two_word_flags+=("--customer")
    local_nonpersistent_flags+=("--customer")
    local_nonpersistent_flags+=("--customer=")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--description=")
    two_word_flags+=("--description")
    local_nonpersistent_flags+=("--description")
    local_nonpersistent_flags+=("--description=")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--expires-at=")
    two_word_flags+=("--expires-at")
    local_nonpersistent_flags+=("--expires-at")
    local_nonpersistent_flags+=("--expires-at=")
    flags+=("--footer=")
    two_word_flags+=("--footer")
    local_nonpersistent_flags+=("--footer")
    local_nonpersistent_flags+=("--footer=")
    flags+=("--header=")
    two_word_flags+=("--header")
    local_nonpersistent_flags+=("--header")
    local_nonpersistent_flags+=("--header=")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--on-behalf-of=")
    two_word_flags+=("--on-behalf-of")
    local_nonpersistent_flags+=("--on-behalf-of")
    local_nonpersistent_flags+=("--on-behalf-of=")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_quotes_finalize_quote()
{
    last_command="stripe_quotes_finalize_quote"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--expires-at=")
    two_word_flags+=("--expires-at")
    local_nonpersistent_flags+=("--expires-at")
    local_nonpersistent_flags+=("--expires-at=")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_quotes_list()
{
    last_command="stripe_quotes_list"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--customer=")
    two_word_flags+=("--customer")
    local_nonpersistent_flags+=("--customer")
    local_nonpersistent_flags+=("--customer=")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--ending-before=")
    two_word_flags+=("--ending-before")
    local_nonpersistent_flags+=("--ending-before")
    local_nonpersistent_flags+=("--ending-before=")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--limit=")
    two_word_flags+=("--limit")
    local_nonpersistent_flags+=("--limit")
    local_nonpersistent_flags+=("--limit=")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--starting-after=")
    two_word_flags+=("--starting-after")
    local_nonpersistent_flags+=("--starting-after")
    local_nonpersistent_flags+=("--starting-after=")
    flags+=("--status=")
    two_word_flags+=("--status")
    local_nonpersistent_flags+=("--status")
    local_nonpersistent_flags+=("--status=")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_quotes_list_computed_upfront_line_items()
{
    last_command="stripe_quotes_list_computed_upfront_line_items"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--ending-before=")
    two_word_flags+=("--ending-before")
    local_nonpersistent_flags+=("--ending-before")
    local_nonpersistent_flags+=("--ending-before=")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--limit=")
    two_word_flags+=("--limit")
    local_nonpersistent_flags+=("--limit")
    local_nonpersistent_flags+=("--limit=")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--starting-after=")
    two_word_flags+=("--starting-after")
    local_nonpersistent_flags+=("--starting-after")
    local_nonpersistent_flags+=("--starting-after=")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_quotes_list_line_items()
{
    last_command="stripe_quotes_list_line_items"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--ending-before=")
    two_word_flags+=("--ending-before")
    local_nonpersistent_flags+=("--ending-before")
    local_nonpersistent_flags+=("--ending-before=")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--limit=")
    two_word_flags+=("--limit")
    local_nonpersistent_flags+=("--limit")
    local_nonpersistent_flags+=("--limit=")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--starting-after=")
    two_word_flags+=("--starting-after")
    local_nonpersistent_flags+=("--starting-after")
    local_nonpersistent_flags+=("--starting-after=")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_quotes_pdf()
{
    last_command="stripe_quotes_pdf"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--ending-before=")
    two_word_flags+=("--ending-before")
    two_word_flags+=("-b")
    local_nonpersistent_flags+=("--ending-before")
    local_nonpersistent_flags+=("--ending-before=")
    local_nonpersistent_flags+=("-b")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--limit=")
    two_word_flags+=("--limit")
    two_word_flags+=("-l")
    local_nonpersistent_flags+=("--limit")
    local_nonpersistent_flags+=("--limit=")
    local_nonpersistent_flags+=("-l")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--starting-after=")
    two_word_flags+=("--starting-after")
    two_word_flags+=("-a")
    local_nonpersistent_flags+=("--starting-after")
    local_nonpersistent_flags+=("--starting-after=")
    local_nonpersistent_flags+=("-a")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_quotes_retrieve()
{
    last_command="stripe_quotes_retrieve"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--ending-before=")
    two_word_flags+=("--ending-before")
    two_word_flags+=("-b")
    local_nonpersistent_flags+=("--ending-before")
    local_nonpersistent_flags+=("--ending-before=")
    local_nonpersistent_flags+=("-b")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--limit=")
    two_word_flags+=("--limit")
    two_word_flags+=("-l")
    local_nonpersistent_flags+=("--limit")
    local_nonpersistent_flags+=("--limit=")
    local_nonpersistent_flags+=("-l")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--starting-after=")
    two_word_flags+=("--starting-after")
    two_word_flags+=("-a")
    local_nonpersistent_flags+=("--starting-after")
    local_nonpersistent_flags+=("--starting-after=")
    local_nonpersistent_flags+=("-a")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_quotes_update()
{
    last_command="stripe_quotes_update"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--application-fee-amount=")
    two_word_flags+=("--application-fee-amount")
    local_nonpersistent_flags+=("--application-fee-amount")
    local_nonpersistent_flags+=("--application-fee-amount=")
    flags+=("--application-fee-percent=")
    two_word_flags+=("--application-fee-percent")
    local_nonpersistent_flags+=("--application-fee-percent")
    local_nonpersistent_flags+=("--application-fee-percent=")
    flags+=("--collection-method=")
    two_word_flags+=("--collection-method")
    local_nonpersistent_flags+=("--collection-method")
    local_nonpersistent_flags+=("--collection-method=")
    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--customer=")
    two_word_flags+=("--customer")
    local_nonpersistent_flags+=("--customer")
    local_nonpersistent_flags+=("--customer=")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--description=")
    two_word_flags+=("--description")
    local_nonpersistent_flags+=("--description")
    local_nonpersistent_flags+=("--description=")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--expires-at=")
    two_word_flags+=("--expires-at")
    local_nonpersistent_flags+=("--expires-at")
    local_nonpersistent_flags+=("--expires-at=")
    flags+=("--footer=")
    two_word_flags+=("--footer")
    local_nonpersistent_flags+=("--footer")
    local_nonpersistent_flags+=("--footer=")
    flags+=("--header=")
    two_word_flags+=("--header")
    local_nonpersistent_flags+=("--header")
    local_nonpersistent_flags+=("--header=")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--on-behalf-of=")
    two_word_flags+=("--on-behalf-of")
    local_nonpersistent_flags+=("--on-behalf-of")
    local_nonpersistent_flags+=("--on-behalf-of=")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_quotes()
{
    last_command="stripe_quotes"

    command_aliases=()

    commands=()
    commands+=("accept")
    commands+=("cancel")
    commands+=("create")
    commands+=("finalize_quote")
    commands+=("list")
    commands+=("list_computed_upfront_line_items")
    commands+=("list_line_items")
    commands+=("pdf")
    commands+=("retrieve")
    commands+=("update")

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_radar_early_fraud_warnings_list()
{
    last_command="stripe_radar_early_fraud_warnings_list"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--charge=")
    two_word_flags+=("--charge")
    local_nonpersistent_flags+=("--charge")
    local_nonpersistent_flags+=("--charge=")
    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--ending-before=")
    two_word_flags+=("--ending-before")
    local_nonpersistent_flags+=("--ending-before")
    local_nonpersistent_flags+=("--ending-before=")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--limit=")
    two_word_flags+=("--limit")
    local_nonpersistent_flags+=("--limit")
    local_nonpersistent_flags+=("--limit=")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--payment-intent=")
    two_word_flags+=("--payment-intent")
    local_nonpersistent_flags+=("--payment-intent")
    local_nonpersistent_flags+=("--payment-intent=")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--starting-after=")
    two_word_flags+=("--starting-after")
    local_nonpersistent_flags+=("--starting-after")
    local_nonpersistent_flags+=("--starting-after=")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_radar_early_fraud_warnings_retrieve()
{
    last_command="stripe_radar_early_fraud_warnings_retrieve"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--ending-before=")
    two_word_flags+=("--ending-before")
    two_word_flags+=("-b")
    local_nonpersistent_flags+=("--ending-before")
    local_nonpersistent_flags+=("--ending-before=")
    local_nonpersistent_flags+=("-b")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--limit=")
    two_word_flags+=("--limit")
    two_word_flags+=("-l")
    local_nonpersistent_flags+=("--limit")
    local_nonpersistent_flags+=("--limit=")
    local_nonpersistent_flags+=("-l")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--starting-after=")
    two_word_flags+=("--starting-after")
    two_word_flags+=("-a")
    local_nonpersistent_flags+=("--starting-after")
    local_nonpersistent_flags+=("--starting-after=")
    local_nonpersistent_flags+=("-a")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_radar_early_fraud_warnings()
{
    last_command="stripe_radar_early_fraud_warnings"

    command_aliases=()

    commands=()
    commands+=("list")
    commands+=("retrieve")

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_radar_value_list_items_create()
{
    last_command="stripe_radar_value_list_items_create"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--value=")
    two_word_flags+=("--value")
    local_nonpersistent_flags+=("--value")
    local_nonpersistent_flags+=("--value=")
    flags+=("--value-list=")
    two_word_flags+=("--value-list")
    local_nonpersistent_flags+=("--value-list")
    local_nonpersistent_flags+=("--value-list=")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_radar_value_list_items_delete()
{
    last_command="stripe_radar_value_list_items_delete"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_radar_value_list_items_list()
{
    last_command="stripe_radar_value_list_items_list"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--created=")
    two_word_flags+=("--created")
    local_nonpersistent_flags+=("--created")
    local_nonpersistent_flags+=("--created=")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--ending-before=")
    two_word_flags+=("--ending-before")
    local_nonpersistent_flags+=("--ending-before")
    local_nonpersistent_flags+=("--ending-before=")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--limit=")
    two_word_flags+=("--limit")
    local_nonpersistent_flags+=("--limit")
    local_nonpersistent_flags+=("--limit=")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--starting-after=")
    two_word_flags+=("--starting-after")
    local_nonpersistent_flags+=("--starting-after")
    local_nonpersistent_flags+=("--starting-after=")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--value=")
    two_word_flags+=("--value")
    local_nonpersistent_flags+=("--value")
    local_nonpersistent_flags+=("--value=")
    flags+=("--value-list=")
    two_word_flags+=("--value-list")
    local_nonpersistent_flags+=("--value-list")
    local_nonpersistent_flags+=("--value-list=")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_radar_value_list_items_retrieve()
{
    last_command="stripe_radar_value_list_items_retrieve"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--ending-before=")
    two_word_flags+=("--ending-before")
    two_word_flags+=("-b")
    local_nonpersistent_flags+=("--ending-before")
    local_nonpersistent_flags+=("--ending-before=")
    local_nonpersistent_flags+=("-b")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--limit=")
    two_word_flags+=("--limit")
    two_word_flags+=("-l")
    local_nonpersistent_flags+=("--limit")
    local_nonpersistent_flags+=("--limit=")
    local_nonpersistent_flags+=("-l")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--starting-after=")
    two_word_flags+=("--starting-after")
    two_word_flags+=("-a")
    local_nonpersistent_flags+=("--starting-after")
    local_nonpersistent_flags+=("--starting-after=")
    local_nonpersistent_flags+=("-a")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_radar_value_list_items()
{
    last_command="stripe_radar_value_list_items"

    command_aliases=()

    commands=()
    commands+=("create")
    commands+=("delete")
    commands+=("list")
    commands+=("retrieve")

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_radar_value_lists_create()
{
    last_command="stripe_radar_value_lists_create"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--alias=")
    two_word_flags+=("--alias")
    local_nonpersistent_flags+=("--alias")
    local_nonpersistent_flags+=("--alias=")
    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--item-type=")
    two_word_flags+=("--item-type")
    local_nonpersistent_flags+=("--item-type")
    local_nonpersistent_flags+=("--item-type=")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--name=")
    two_word_flags+=("--name")
    local_nonpersistent_flags+=("--name")
    local_nonpersistent_flags+=("--name=")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_radar_value_lists_delete()
{
    last_command="stripe_radar_value_lists_delete"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_radar_value_lists_list()
{
    last_command="stripe_radar_value_lists_list"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--alias=")
    two_word_flags+=("--alias")
    local_nonpersistent_flags+=("--alias")
    local_nonpersistent_flags+=("--alias=")
    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--contains=")
    two_word_flags+=("--contains")
    local_nonpersistent_flags+=("--contains")
    local_nonpersistent_flags+=("--contains=")
    flags+=("--created=")
    two_word_flags+=("--created")
    local_nonpersistent_flags+=("--created")
    local_nonpersistent_flags+=("--created=")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--ending-before=")
    two_word_flags+=("--ending-before")
    local_nonpersistent_flags+=("--ending-before")
    local_nonpersistent_flags+=("--ending-before=")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--limit=")
    two_word_flags+=("--limit")
    local_nonpersistent_flags+=("--limit")
    local_nonpersistent_flags+=("--limit=")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--starting-after=")
    two_word_flags+=("--starting-after")
    local_nonpersistent_flags+=("--starting-after")
    local_nonpersistent_flags+=("--starting-after=")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_radar_value_lists_retrieve()
{
    last_command="stripe_radar_value_lists_retrieve"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--ending-before=")
    two_word_flags+=("--ending-before")
    two_word_flags+=("-b")
    local_nonpersistent_flags+=("--ending-before")
    local_nonpersistent_flags+=("--ending-before=")
    local_nonpersistent_flags+=("-b")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--limit=")
    two_word_flags+=("--limit")
    two_word_flags+=("-l")
    local_nonpersistent_flags+=("--limit")
    local_nonpersistent_flags+=("--limit=")
    local_nonpersistent_flags+=("-l")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--starting-after=")
    two_word_flags+=("--starting-after")
    two_word_flags+=("-a")
    local_nonpersistent_flags+=("--starting-after")
    local_nonpersistent_flags+=("--starting-after=")
    local_nonpersistent_flags+=("-a")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_radar_value_lists_update()
{
    last_command="stripe_radar_value_lists_update"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--alias=")
    two_word_flags+=("--alias")
    local_nonpersistent_flags+=("--alias")
    local_nonpersistent_flags+=("--alias=")
    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--name=")
    two_word_flags+=("--name")
    local_nonpersistent_flags+=("--name")
    local_nonpersistent_flags+=("--name=")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_radar_value_lists()
{
    last_command="stripe_radar_value_lists"

    command_aliases=()

    commands=()
    commands+=("create")
    commands+=("delete")
    commands+=("list")
    commands+=("retrieve")
    commands+=("update")

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_radar()
{
    last_command="stripe_radar"

    command_aliases=()

    commands=()
    commands+=("early_fraud_warnings")
    commands+=("value_list_items")
    commands+=("value_lists")

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_refunds_create()
{
    last_command="stripe_refunds_create"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--amount=")
    two_word_flags+=("--amount")
    local_nonpersistent_flags+=("--amount")
    local_nonpersistent_flags+=("--amount=")
    flags+=("--charge=")
    two_word_flags+=("--charge")
    local_nonpersistent_flags+=("--charge")
    local_nonpersistent_flags+=("--charge=")
    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--payment-intent=")
    two_word_flags+=("--payment-intent")
    local_nonpersistent_flags+=("--payment-intent")
    local_nonpersistent_flags+=("--payment-intent=")
    flags+=("--reason=")
    two_word_flags+=("--reason")
    local_nonpersistent_flags+=("--reason")
    local_nonpersistent_flags+=("--reason=")
    flags+=("--refund-application-fee=")
    two_word_flags+=("--refund-application-fee")
    local_nonpersistent_flags+=("--refund-application-fee")
    local_nonpersistent_flags+=("--refund-application-fee=")
    flags+=("--reverse-transfer=")
    two_word_flags+=("--reverse-transfer")
    local_nonpersistent_flags+=("--reverse-transfer")
    local_nonpersistent_flags+=("--reverse-transfer=")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_refunds_list()
{
    last_command="stripe_refunds_list"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--charge=")
    two_word_flags+=("--charge")
    local_nonpersistent_flags+=("--charge")
    local_nonpersistent_flags+=("--charge=")
    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--created=")
    two_word_flags+=("--created")
    local_nonpersistent_flags+=("--created")
    local_nonpersistent_flags+=("--created=")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--ending-before=")
    two_word_flags+=("--ending-before")
    local_nonpersistent_flags+=("--ending-before")
    local_nonpersistent_flags+=("--ending-before=")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--limit=")
    two_word_flags+=("--limit")
    local_nonpersistent_flags+=("--limit")
    local_nonpersistent_flags+=("--limit=")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--payment-intent=")
    two_word_flags+=("--payment-intent")
    local_nonpersistent_flags+=("--payment-intent")
    local_nonpersistent_flags+=("--payment-intent=")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--starting-after=")
    two_word_flags+=("--starting-after")
    local_nonpersistent_flags+=("--starting-after")
    local_nonpersistent_flags+=("--starting-after=")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_refunds_retrieve()
{
    last_command="stripe_refunds_retrieve"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--ending-before=")
    two_word_flags+=("--ending-before")
    two_word_flags+=("-b")
    local_nonpersistent_flags+=("--ending-before")
    local_nonpersistent_flags+=("--ending-before=")
    local_nonpersistent_flags+=("-b")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--limit=")
    two_word_flags+=("--limit")
    two_word_flags+=("-l")
    local_nonpersistent_flags+=("--limit")
    local_nonpersistent_flags+=("--limit=")
    local_nonpersistent_flags+=("-l")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--starting-after=")
    two_word_flags+=("--starting-after")
    two_word_flags+=("-a")
    local_nonpersistent_flags+=("--starting-after")
    local_nonpersistent_flags+=("--starting-after=")
    local_nonpersistent_flags+=("-a")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_refunds_update()
{
    last_command="stripe_refunds_update"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_refunds()
{
    last_command="stripe_refunds"

    command_aliases=()

    commands=()
    commands+=("create")
    commands+=("list")
    commands+=("retrieve")
    commands+=("update")

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_reporting_report_runs_create()
{
    last_command="stripe_reporting_report_runs_create"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--report-type=")
    two_word_flags+=("--report-type")
    local_nonpersistent_flags+=("--report-type")
    local_nonpersistent_flags+=("--report-type=")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_reporting_report_runs_list()
{
    last_command="stripe_reporting_report_runs_list"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--created=")
    two_word_flags+=("--created")
    local_nonpersistent_flags+=("--created")
    local_nonpersistent_flags+=("--created=")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--ending-before=")
    two_word_flags+=("--ending-before")
    local_nonpersistent_flags+=("--ending-before")
    local_nonpersistent_flags+=("--ending-before=")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--limit=")
    two_word_flags+=("--limit")
    local_nonpersistent_flags+=("--limit")
    local_nonpersistent_flags+=("--limit=")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--starting-after=")
    two_word_flags+=("--starting-after")
    local_nonpersistent_flags+=("--starting-after")
    local_nonpersistent_flags+=("--starting-after=")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_reporting_report_runs_retrieve()
{
    last_command="stripe_reporting_report_runs_retrieve"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--ending-before=")
    two_word_flags+=("--ending-before")
    two_word_flags+=("-b")
    local_nonpersistent_flags+=("--ending-before")
    local_nonpersistent_flags+=("--ending-before=")
    local_nonpersistent_flags+=("-b")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--limit=")
    two_word_flags+=("--limit")
    two_word_flags+=("-l")
    local_nonpersistent_flags+=("--limit")
    local_nonpersistent_flags+=("--limit=")
    local_nonpersistent_flags+=("-l")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--starting-after=")
    two_word_flags+=("--starting-after")
    two_word_flags+=("-a")
    local_nonpersistent_flags+=("--starting-after")
    local_nonpersistent_flags+=("--starting-after=")
    local_nonpersistent_flags+=("-a")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_reporting_report_runs()
{
    last_command="stripe_reporting_report_runs"

    command_aliases=()

    commands=()
    commands+=("create")
    commands+=("list")
    commands+=("retrieve")

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_reporting_report_types_list()
{
    last_command="stripe_reporting_report_types_list"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--ending-before=")
    two_word_flags+=("--ending-before")
    two_word_flags+=("-b")
    local_nonpersistent_flags+=("--ending-before")
    local_nonpersistent_flags+=("--ending-before=")
    local_nonpersistent_flags+=("-b")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--limit=")
    two_word_flags+=("--limit")
    two_word_flags+=("-l")
    local_nonpersistent_flags+=("--limit")
    local_nonpersistent_flags+=("--limit=")
    local_nonpersistent_flags+=("-l")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--starting-after=")
    two_word_flags+=("--starting-after")
    two_word_flags+=("-a")
    local_nonpersistent_flags+=("--starting-after")
    local_nonpersistent_flags+=("--starting-after=")
    local_nonpersistent_flags+=("-a")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_reporting_report_types_retrieve()
{
    last_command="stripe_reporting_report_types_retrieve"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--ending-before=")
    two_word_flags+=("--ending-before")
    two_word_flags+=("-b")
    local_nonpersistent_flags+=("--ending-before")
    local_nonpersistent_flags+=("--ending-before=")
    local_nonpersistent_flags+=("-b")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--limit=")
    two_word_flags+=("--limit")
    two_word_flags+=("-l")
    local_nonpersistent_flags+=("--limit")
    local_nonpersistent_flags+=("--limit=")
    local_nonpersistent_flags+=("-l")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--starting-after=")
    two_word_flags+=("--starting-after")
    two_word_flags+=("-a")
    local_nonpersistent_flags+=("--starting-after")
    local_nonpersistent_flags+=("--starting-after=")
    local_nonpersistent_flags+=("-a")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_reporting_report_types()
{
    last_command="stripe_reporting_report_types"

    command_aliases=()

    commands=()
    commands+=("list")
    commands+=("retrieve")

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_reporting()
{
    last_command="stripe_reporting"

    command_aliases=()

    commands=()
    commands+=("report_runs")
    commands+=("report_types")

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_reviews_approve()
{
    last_command="stripe_reviews_approve"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_reviews_list()
{
    last_command="stripe_reviews_list"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--created=")
    two_word_flags+=("--created")
    local_nonpersistent_flags+=("--created")
    local_nonpersistent_flags+=("--created=")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--ending-before=")
    two_word_flags+=("--ending-before")
    local_nonpersistent_flags+=("--ending-before")
    local_nonpersistent_flags+=("--ending-before=")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--limit=")
    two_word_flags+=("--limit")
    local_nonpersistent_flags+=("--limit")
    local_nonpersistent_flags+=("--limit=")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--starting-after=")
    two_word_flags+=("--starting-after")
    local_nonpersistent_flags+=("--starting-after")
    local_nonpersistent_flags+=("--starting-after=")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_reviews_retrieve()
{
    last_command="stripe_reviews_retrieve"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--ending-before=")
    two_word_flags+=("--ending-before")
    two_word_flags+=("-b")
    local_nonpersistent_flags+=("--ending-before")
    local_nonpersistent_flags+=("--ending-before=")
    local_nonpersistent_flags+=("-b")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--limit=")
    two_word_flags+=("--limit")
    two_word_flags+=("-l")
    local_nonpersistent_flags+=("--limit")
    local_nonpersistent_flags+=("--limit=")
    local_nonpersistent_flags+=("-l")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--starting-after=")
    two_word_flags+=("--starting-after")
    two_word_flags+=("-a")
    local_nonpersistent_flags+=("--starting-after")
    local_nonpersistent_flags+=("--starting-after=")
    local_nonpersistent_flags+=("-a")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_reviews()
{
    last_command="stripe_reviews"

    command_aliases=()

    commands=()
    commands+=("approve")
    commands+=("list")
    commands+=("retrieve")

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_samples_create()
{
    last_command="stripe_samples_create"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--force-refresh")
    local_nonpersistent_flags+=("--force-refresh")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_samples_list()
{
    last_command="stripe_samples_list"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_samples()
{
    last_command="stripe_samples"

    command_aliases=()

    commands=()
    commands+=("create")
    commands+=("list")

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_scheduled_query_runs_list()
{
    last_command="stripe_scheduled_query_runs_list"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--ending-before=")
    two_word_flags+=("--ending-before")
    local_nonpersistent_flags+=("--ending-before")
    local_nonpersistent_flags+=("--ending-before=")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--limit=")
    two_word_flags+=("--limit")
    local_nonpersistent_flags+=("--limit")
    local_nonpersistent_flags+=("--limit=")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--starting-after=")
    two_word_flags+=("--starting-after")
    local_nonpersistent_flags+=("--starting-after")
    local_nonpersistent_flags+=("--starting-after=")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_scheduled_query_runs_retrieve()
{
    last_command="stripe_scheduled_query_runs_retrieve"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--ending-before=")
    two_word_flags+=("--ending-before")
    two_word_flags+=("-b")
    local_nonpersistent_flags+=("--ending-before")
    local_nonpersistent_flags+=("--ending-before=")
    local_nonpersistent_flags+=("-b")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--limit=")
    two_word_flags+=("--limit")
    two_word_flags+=("-l")
    local_nonpersistent_flags+=("--limit")
    local_nonpersistent_flags+=("--limit=")
    local_nonpersistent_flags+=("-l")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--starting-after=")
    two_word_flags+=("--starting-after")
    two_word_flags+=("-a")
    local_nonpersistent_flags+=("--starting-after")
    local_nonpersistent_flags+=("--starting-after=")
    local_nonpersistent_flags+=("-a")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_scheduled_query_runs()
{
    last_command="stripe_scheduled_query_runs"

    command_aliases=()

    commands=()
    commands+=("list")
    commands+=("retrieve")

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_serve()
{
    last_command="stripe_serve"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--port=")
    two_word_flags+=("--port")
    local_nonpersistent_flags+=("--port")
    local_nonpersistent_flags+=("--port=")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_setup_attempts_list()
{
    last_command="stripe_setup_attempts_list"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--created=")
    two_word_flags+=("--created")
    local_nonpersistent_flags+=("--created")
    local_nonpersistent_flags+=("--created=")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--ending-before=")
    two_word_flags+=("--ending-before")
    local_nonpersistent_flags+=("--ending-before")
    local_nonpersistent_flags+=("--ending-before=")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--limit=")
    two_word_flags+=("--limit")
    local_nonpersistent_flags+=("--limit")
    local_nonpersistent_flags+=("--limit=")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--setup-intent=")
    two_word_flags+=("--setup-intent")
    local_nonpersistent_flags+=("--setup-intent")
    local_nonpersistent_flags+=("--setup-intent=")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--starting-after=")
    two_word_flags+=("--starting-after")
    local_nonpersistent_flags+=("--starting-after")
    local_nonpersistent_flags+=("--starting-after=")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_setup_attempts()
{
    last_command="stripe_setup_attempts"

    command_aliases=()

    commands=()
    commands+=("list")

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_setup_intents_cancel()
{
    last_command="stripe_setup_intents_cancel"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--cancellation-reason=")
    two_word_flags+=("--cancellation-reason")
    local_nonpersistent_flags+=("--cancellation-reason")
    local_nonpersistent_flags+=("--cancellation-reason=")
    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_setup_intents_confirm()
{
    last_command="stripe_setup_intents_confirm"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--payment-method=")
    two_word_flags+=("--payment-method")
    local_nonpersistent_flags+=("--payment-method")
    local_nonpersistent_flags+=("--payment-method=")
    flags+=("--return-url=")
    two_word_flags+=("--return-url")
    local_nonpersistent_flags+=("--return-url")
    local_nonpersistent_flags+=("--return-url=")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_setup_intents_create()
{
    last_command="stripe_setup_intents_create"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--confirm=")
    two_word_flags+=("--confirm")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("--confirm=")
    flags+=("--customer=")
    two_word_flags+=("--customer")
    local_nonpersistent_flags+=("--customer")
    local_nonpersistent_flags+=("--customer=")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--description=")
    two_word_flags+=("--description")
    local_nonpersistent_flags+=("--description")
    local_nonpersistent_flags+=("--description=")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--on-behalf-of=")
    two_word_flags+=("--on-behalf-of")
    local_nonpersistent_flags+=("--on-behalf-of")
    local_nonpersistent_flags+=("--on-behalf-of=")
    flags+=("--payment-method=")
    two_word_flags+=("--payment-method")
    local_nonpersistent_flags+=("--payment-method")
    local_nonpersistent_flags+=("--payment-method=")
    flags+=("--return-url=")
    two_word_flags+=("--return-url")
    local_nonpersistent_flags+=("--return-url")
    local_nonpersistent_flags+=("--return-url=")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--usage=")
    two_word_flags+=("--usage")
    local_nonpersistent_flags+=("--usage")
    local_nonpersistent_flags+=("--usage=")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_setup_intents_list()
{
    last_command="stripe_setup_intents_list"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--created=")
    two_word_flags+=("--created")
    local_nonpersistent_flags+=("--created")
    local_nonpersistent_flags+=("--created=")
    flags+=("--customer=")
    two_word_flags+=("--customer")
    local_nonpersistent_flags+=("--customer")
    local_nonpersistent_flags+=("--customer=")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--ending-before=")
    two_word_flags+=("--ending-before")
    local_nonpersistent_flags+=("--ending-before")
    local_nonpersistent_flags+=("--ending-before=")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--limit=")
    two_word_flags+=("--limit")
    local_nonpersistent_flags+=("--limit")
    local_nonpersistent_flags+=("--limit=")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--payment-method=")
    two_word_flags+=("--payment-method")
    local_nonpersistent_flags+=("--payment-method")
    local_nonpersistent_flags+=("--payment-method=")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--starting-after=")
    two_word_flags+=("--starting-after")
    local_nonpersistent_flags+=("--starting-after")
    local_nonpersistent_flags+=("--starting-after=")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_setup_intents_retrieve()
{
    last_command="stripe_setup_intents_retrieve"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--client-secret=")
    two_word_flags+=("--client-secret")
    local_nonpersistent_flags+=("--client-secret")
    local_nonpersistent_flags+=("--client-secret=")
    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--ending-before=")
    two_word_flags+=("--ending-before")
    two_word_flags+=("-b")
    local_nonpersistent_flags+=("--ending-before")
    local_nonpersistent_flags+=("--ending-before=")
    local_nonpersistent_flags+=("-b")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--limit=")
    two_word_flags+=("--limit")
    two_word_flags+=("-l")
    local_nonpersistent_flags+=("--limit")
    local_nonpersistent_flags+=("--limit=")
    local_nonpersistent_flags+=("-l")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--starting-after=")
    two_word_flags+=("--starting-after")
    two_word_flags+=("-a")
    local_nonpersistent_flags+=("--starting-after")
    local_nonpersistent_flags+=("--starting-after=")
    local_nonpersistent_flags+=("-a")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_setup_intents_update()
{
    last_command="stripe_setup_intents_update"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--customer=")
    two_word_flags+=("--customer")
    local_nonpersistent_flags+=("--customer")
    local_nonpersistent_flags+=("--customer=")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--description=")
    two_word_flags+=("--description")
    local_nonpersistent_flags+=("--description")
    local_nonpersistent_flags+=("--description=")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--payment-method=")
    two_word_flags+=("--payment-method")
    local_nonpersistent_flags+=("--payment-method")
    local_nonpersistent_flags+=("--payment-method=")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_setup_intents()
{
    last_command="stripe_setup_intents"

    command_aliases=()

    commands=()
    commands+=("cancel")
    commands+=("confirm")
    commands+=("create")
    commands+=("list")
    commands+=("retrieve")
    commands+=("update")

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_skus_create()
{
    last_command="stripe_skus_create"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--active=")
    two_word_flags+=("--active")
    local_nonpersistent_flags+=("--active")
    local_nonpersistent_flags+=("--active=")
    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--currency=")
    two_word_flags+=("--currency")
    local_nonpersistent_flags+=("--currency")
    local_nonpersistent_flags+=("--currency=")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--id=")
    two_word_flags+=("--id")
    local_nonpersistent_flags+=("--id")
    local_nonpersistent_flags+=("--id=")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--image=")
    two_word_flags+=("--image")
    local_nonpersistent_flags+=("--image")
    local_nonpersistent_flags+=("--image=")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--price=")
    two_word_flags+=("--price")
    local_nonpersistent_flags+=("--price")
    local_nonpersistent_flags+=("--price=")
    flags+=("--product=")
    two_word_flags+=("--product")
    local_nonpersistent_flags+=("--product")
    local_nonpersistent_flags+=("--product=")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_skus_delete()
{
    last_command="stripe_skus_delete"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_skus_list()
{
    last_command="stripe_skus_list"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--active=")
    two_word_flags+=("--active")
    local_nonpersistent_flags+=("--active")
    local_nonpersistent_flags+=("--active=")
    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--ending-before=")
    two_word_flags+=("--ending-before")
    local_nonpersistent_flags+=("--ending-before")
    local_nonpersistent_flags+=("--ending-before=")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--in-stock=")
    two_word_flags+=("--in-stock")
    local_nonpersistent_flags+=("--in-stock")
    local_nonpersistent_flags+=("--in-stock=")
    flags+=("--limit=")
    two_word_flags+=("--limit")
    local_nonpersistent_flags+=("--limit")
    local_nonpersistent_flags+=("--limit=")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--product=")
    two_word_flags+=("--product")
    local_nonpersistent_flags+=("--product")
    local_nonpersistent_flags+=("--product=")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--starting-after=")
    two_word_flags+=("--starting-after")
    local_nonpersistent_flags+=("--starting-after")
    local_nonpersistent_flags+=("--starting-after=")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_skus_retrieve()
{
    last_command="stripe_skus_retrieve"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--ending-before=")
    two_word_flags+=("--ending-before")
    two_word_flags+=("-b")
    local_nonpersistent_flags+=("--ending-before")
    local_nonpersistent_flags+=("--ending-before=")
    local_nonpersistent_flags+=("-b")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--limit=")
    two_word_flags+=("--limit")
    two_word_flags+=("-l")
    local_nonpersistent_flags+=("--limit")
    local_nonpersistent_flags+=("--limit=")
    local_nonpersistent_flags+=("-l")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--starting-after=")
    two_word_flags+=("--starting-after")
    two_word_flags+=("-a")
    local_nonpersistent_flags+=("--starting-after")
    local_nonpersistent_flags+=("--starting-after=")
    local_nonpersistent_flags+=("-a")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_skus_update()
{
    last_command="stripe_skus_update"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--active=")
    two_word_flags+=("--active")
    local_nonpersistent_flags+=("--active")
    local_nonpersistent_flags+=("--active=")
    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--currency=")
    two_word_flags+=("--currency")
    local_nonpersistent_flags+=("--currency")
    local_nonpersistent_flags+=("--currency=")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--image=")
    two_word_flags+=("--image")
    local_nonpersistent_flags+=("--image")
    local_nonpersistent_flags+=("--image=")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--price=")
    two_word_flags+=("--price")
    local_nonpersistent_flags+=("--price")
    local_nonpersistent_flags+=("--price=")
    flags+=("--product=")
    two_word_flags+=("--product")
    local_nonpersistent_flags+=("--product")
    local_nonpersistent_flags+=("--product=")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_skus()
{
    last_command="stripe_skus"

    command_aliases=()

    commands=()
    commands+=("create")
    commands+=("delete")
    commands+=("list")
    commands+=("retrieve")
    commands+=("update")

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_sources_create()
{
    last_command="stripe_sources_create"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--amount=")
    two_word_flags+=("--amount")
    local_nonpersistent_flags+=("--amount")
    local_nonpersistent_flags+=("--amount=")
    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--currency=")
    two_word_flags+=("--currency")
    local_nonpersistent_flags+=("--currency")
    local_nonpersistent_flags+=("--currency=")
    flags+=("--customer=")
    two_word_flags+=("--customer")
    local_nonpersistent_flags+=("--customer")
    local_nonpersistent_flags+=("--customer=")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--flow=")
    two_word_flags+=("--flow")
    local_nonpersistent_flags+=("--flow")
    local_nonpersistent_flags+=("--flow=")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--original-source=")
    two_word_flags+=("--original-source")
    local_nonpersistent_flags+=("--original-source")
    local_nonpersistent_flags+=("--original-source=")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--statement-descriptor=")
    two_word_flags+=("--statement-descriptor")
    local_nonpersistent_flags+=("--statement-descriptor")
    local_nonpersistent_flags+=("--statement-descriptor=")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--token=")
    two_word_flags+=("--token")
    local_nonpersistent_flags+=("--token")
    local_nonpersistent_flags+=("--token=")
    flags+=("--type=")
    two_word_flags+=("--type")
    local_nonpersistent_flags+=("--type")
    local_nonpersistent_flags+=("--type=")
    flags+=("--usage=")
    two_word_flags+=("--usage")
    local_nonpersistent_flags+=("--usage")
    local_nonpersistent_flags+=("--usage=")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_sources_detach()
{
    last_command="stripe_sources_detach"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_sources_retrieve()
{
    last_command="stripe_sources_retrieve"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--client-secret=")
    two_word_flags+=("--client-secret")
    local_nonpersistent_flags+=("--client-secret")
    local_nonpersistent_flags+=("--client-secret=")
    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--ending-before=")
    two_word_flags+=("--ending-before")
    two_word_flags+=("-b")
    local_nonpersistent_flags+=("--ending-before")
    local_nonpersistent_flags+=("--ending-before=")
    local_nonpersistent_flags+=("-b")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--limit=")
    two_word_flags+=("--limit")
    two_word_flags+=("-l")
    local_nonpersistent_flags+=("--limit")
    local_nonpersistent_flags+=("--limit=")
    local_nonpersistent_flags+=("-l")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--starting-after=")
    two_word_flags+=("--starting-after")
    two_word_flags+=("-a")
    local_nonpersistent_flags+=("--starting-after")
    local_nonpersistent_flags+=("--starting-after=")
    local_nonpersistent_flags+=("-a")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_sources_source_transactions()
{
    last_command="stripe_sources_source_transactions"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--ending-before=")
    two_word_flags+=("--ending-before")
    local_nonpersistent_flags+=("--ending-before")
    local_nonpersistent_flags+=("--ending-before=")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--limit=")
    two_word_flags+=("--limit")
    local_nonpersistent_flags+=("--limit")
    local_nonpersistent_flags+=("--limit=")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--starting-after=")
    two_word_flags+=("--starting-after")
    local_nonpersistent_flags+=("--starting-after")
    local_nonpersistent_flags+=("--starting-after=")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_sources_update()
{
    last_command="stripe_sources_update"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--amount=")
    two_word_flags+=("--amount")
    local_nonpersistent_flags+=("--amount")
    local_nonpersistent_flags+=("--amount=")
    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_sources_verify()
{
    last_command="stripe_sources_verify"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_sources()
{
    last_command="stripe_sources"

    command_aliases=()

    commands=()
    commands+=("create")
    commands+=("detach")
    commands+=("retrieve")
    commands+=("source_transactions")
    commands+=("update")
    commands+=("verify")

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_status()
{
    last_command="stripe_status"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--format=")
    two_word_flags+=("--format")
    local_nonpersistent_flags+=("--format")
    local_nonpersistent_flags+=("--format=")
    flags+=("--hide-spinner")
    local_nonpersistent_flags+=("--hide-spinner")
    flags+=("--poll")
    local_nonpersistent_flags+=("--poll")
    flags+=("--poll-rate=")
    two_word_flags+=("--poll-rate")
    local_nonpersistent_flags+=("--poll-rate")
    local_nonpersistent_flags+=("--poll-rate=")
    flags+=("--verbose")
    local_nonpersistent_flags+=("--verbose")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_subscription_items_create()
{
    last_command="stripe_subscription_items_create"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--payment-behavior=")
    two_word_flags+=("--payment-behavior")
    local_nonpersistent_flags+=("--payment-behavior")
    local_nonpersistent_flags+=("--payment-behavior=")
    flags+=("--plan=")
    two_word_flags+=("--plan")
    local_nonpersistent_flags+=("--plan")
    local_nonpersistent_flags+=("--plan=")
    flags+=("--price=")
    two_word_flags+=("--price")
    local_nonpersistent_flags+=("--price")
    local_nonpersistent_flags+=("--price=")
    flags+=("--proration-behavior=")
    two_word_flags+=("--proration-behavior")
    local_nonpersistent_flags+=("--proration-behavior")
    local_nonpersistent_flags+=("--proration-behavior=")
    flags+=("--proration-date=")
    two_word_flags+=("--proration-date")
    local_nonpersistent_flags+=("--proration-date")
    local_nonpersistent_flags+=("--proration-date=")
    flags+=("--quantity=")
    two_word_flags+=("--quantity")
    local_nonpersistent_flags+=("--quantity")
    local_nonpersistent_flags+=("--quantity=")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--subscription=")
    two_word_flags+=("--subscription")
    local_nonpersistent_flags+=("--subscription")
    local_nonpersistent_flags+=("--subscription=")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_subscription_items_delete()
{
    last_command="stripe_subscription_items_delete"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_subscription_items_list()
{
    last_command="stripe_subscription_items_list"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--ending-before=")
    two_word_flags+=("--ending-before")
    local_nonpersistent_flags+=("--ending-before")
    local_nonpersistent_flags+=("--ending-before=")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--limit=")
    two_word_flags+=("--limit")
    local_nonpersistent_flags+=("--limit")
    local_nonpersistent_flags+=("--limit=")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--starting-after=")
    two_word_flags+=("--starting-after")
    local_nonpersistent_flags+=("--starting-after")
    local_nonpersistent_flags+=("--starting-after=")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--subscription=")
    two_word_flags+=("--subscription")
    local_nonpersistent_flags+=("--subscription")
    local_nonpersistent_flags+=("--subscription=")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_subscription_items_retrieve()
{
    last_command="stripe_subscription_items_retrieve"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--ending-before=")
    two_word_flags+=("--ending-before")
    two_word_flags+=("-b")
    local_nonpersistent_flags+=("--ending-before")
    local_nonpersistent_flags+=("--ending-before=")
    local_nonpersistent_flags+=("-b")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--limit=")
    two_word_flags+=("--limit")
    two_word_flags+=("-l")
    local_nonpersistent_flags+=("--limit")
    local_nonpersistent_flags+=("--limit=")
    local_nonpersistent_flags+=("-l")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--starting-after=")
    two_word_flags+=("--starting-after")
    two_word_flags+=("-a")
    local_nonpersistent_flags+=("--starting-after")
    local_nonpersistent_flags+=("--starting-after=")
    local_nonpersistent_flags+=("-a")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_subscription_items_update()
{
    last_command="stripe_subscription_items_update"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--off-session=")
    two_word_flags+=("--off-session")
    local_nonpersistent_flags+=("--off-session")
    local_nonpersistent_flags+=("--off-session=")
    flags+=("--payment-behavior=")
    two_word_flags+=("--payment-behavior")
    local_nonpersistent_flags+=("--payment-behavior")
    local_nonpersistent_flags+=("--payment-behavior=")
    flags+=("--plan=")
    two_word_flags+=("--plan")
    local_nonpersistent_flags+=("--plan")
    local_nonpersistent_flags+=("--plan=")
    flags+=("--price=")
    two_word_flags+=("--price")
    local_nonpersistent_flags+=("--price")
    local_nonpersistent_flags+=("--price=")
    flags+=("--proration-behavior=")
    two_word_flags+=("--proration-behavior")
    local_nonpersistent_flags+=("--proration-behavior")
    local_nonpersistent_flags+=("--proration-behavior=")
    flags+=("--proration-date=")
    two_word_flags+=("--proration-date")
    local_nonpersistent_flags+=("--proration-date")
    local_nonpersistent_flags+=("--proration-date=")
    flags+=("--quantity=")
    two_word_flags+=("--quantity")
    local_nonpersistent_flags+=("--quantity")
    local_nonpersistent_flags+=("--quantity=")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_subscription_items_usage_record_summaries()
{
    last_command="stripe_subscription_items_usage_record_summaries"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--ending-before=")
    two_word_flags+=("--ending-before")
    local_nonpersistent_flags+=("--ending-before")
    local_nonpersistent_flags+=("--ending-before=")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--limit=")
    two_word_flags+=("--limit")
    local_nonpersistent_flags+=("--limit")
    local_nonpersistent_flags+=("--limit=")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--starting-after=")
    two_word_flags+=("--starting-after")
    local_nonpersistent_flags+=("--starting-after")
    local_nonpersistent_flags+=("--starting-after=")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_subscription_items()
{
    last_command="stripe_subscription_items"

    command_aliases=()

    commands=()
    commands+=("create")
    commands+=("delete")
    commands+=("list")
    commands+=("retrieve")
    commands+=("update")
    commands+=("usage_record_summaries")

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_subscription_schedules_cancel()
{
    last_command="stripe_subscription_schedules_cancel"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--invoice-now=")
    two_word_flags+=("--invoice-now")
    local_nonpersistent_flags+=("--invoice-now")
    local_nonpersistent_flags+=("--invoice-now=")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--prorate=")
    two_word_flags+=("--prorate")
    local_nonpersistent_flags+=("--prorate")
    local_nonpersistent_flags+=("--prorate=")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_subscription_schedules_create()
{
    last_command="stripe_subscription_schedules_create"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--customer=")
    two_word_flags+=("--customer")
    local_nonpersistent_flags+=("--customer")
    local_nonpersistent_flags+=("--customer=")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--end-behavior=")
    two_word_flags+=("--end-behavior")
    local_nonpersistent_flags+=("--end-behavior")
    local_nonpersistent_flags+=("--end-behavior=")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--from-subscription=")
    two_word_flags+=("--from-subscription")
    local_nonpersistent_flags+=("--from-subscription")
    local_nonpersistent_flags+=("--from-subscription=")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--start-date=")
    two_word_flags+=("--start-date")
    local_nonpersistent_flags+=("--start-date")
    local_nonpersistent_flags+=("--start-date=")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_subscription_schedules_list()
{
    last_command="stripe_subscription_schedules_list"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--canceled-at=")
    two_word_flags+=("--canceled-at")
    local_nonpersistent_flags+=("--canceled-at")
    local_nonpersistent_flags+=("--canceled-at=")
    flags+=("--completed-at=")
    two_word_flags+=("--completed-at")
    local_nonpersistent_flags+=("--completed-at")
    local_nonpersistent_flags+=("--completed-at=")
    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--created=")
    two_word_flags+=("--created")
    local_nonpersistent_flags+=("--created")
    local_nonpersistent_flags+=("--created=")
    flags+=("--customer=")
    two_word_flags+=("--customer")
    local_nonpersistent_flags+=("--customer")
    local_nonpersistent_flags+=("--customer=")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--ending-before=")
    two_word_flags+=("--ending-before")
    local_nonpersistent_flags+=("--ending-before")
    local_nonpersistent_flags+=("--ending-before=")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--limit=")
    two_word_flags+=("--limit")
    local_nonpersistent_flags+=("--limit")
    local_nonpersistent_flags+=("--limit=")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--released-at=")
    two_word_flags+=("--released-at")
    local_nonpersistent_flags+=("--released-at")
    local_nonpersistent_flags+=("--released-at=")
    flags+=("--scheduled=")
    two_word_flags+=("--scheduled")
    local_nonpersistent_flags+=("--scheduled")
    local_nonpersistent_flags+=("--scheduled=")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--starting-after=")
    two_word_flags+=("--starting-after")
    local_nonpersistent_flags+=("--starting-after")
    local_nonpersistent_flags+=("--starting-after=")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_subscription_schedules_release()
{
    last_command="stripe_subscription_schedules_release"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--preserve-cancel-date=")
    two_word_flags+=("--preserve-cancel-date")
    local_nonpersistent_flags+=("--preserve-cancel-date")
    local_nonpersistent_flags+=("--preserve-cancel-date=")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_subscription_schedules_retrieve()
{
    last_command="stripe_subscription_schedules_retrieve"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--ending-before=")
    two_word_flags+=("--ending-before")
    two_word_flags+=("-b")
    local_nonpersistent_flags+=("--ending-before")
    local_nonpersistent_flags+=("--ending-before=")
    local_nonpersistent_flags+=("-b")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--limit=")
    two_word_flags+=("--limit")
    two_word_flags+=("-l")
    local_nonpersistent_flags+=("--limit")
    local_nonpersistent_flags+=("--limit=")
    local_nonpersistent_flags+=("-l")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--starting-after=")
    two_word_flags+=("--starting-after")
    two_word_flags+=("-a")
    local_nonpersistent_flags+=("--starting-after")
    local_nonpersistent_flags+=("--starting-after=")
    local_nonpersistent_flags+=("-a")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_subscription_schedules_update()
{
    last_command="stripe_subscription_schedules_update"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--end-behavior=")
    two_word_flags+=("--end-behavior")
    local_nonpersistent_flags+=("--end-behavior")
    local_nonpersistent_flags+=("--end-behavior=")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--proration-behavior=")
    two_word_flags+=("--proration-behavior")
    local_nonpersistent_flags+=("--proration-behavior")
    local_nonpersistent_flags+=("--proration-behavior=")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_subscription_schedules()
{
    last_command="stripe_subscription_schedules"

    command_aliases=()

    commands=()
    commands+=("cancel")
    commands+=("create")
    commands+=("list")
    commands+=("release")
    commands+=("retrieve")
    commands+=("update")

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_subscriptions_cancel()
{
    last_command="stripe_subscriptions_cancel"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_subscriptions_create()
{
    last_command="stripe_subscriptions_create"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--application-fee-percent=")
    two_word_flags+=("--application-fee-percent")
    local_nonpersistent_flags+=("--application-fee-percent")
    local_nonpersistent_flags+=("--application-fee-percent=")
    flags+=("--backdate-start-date=")
    two_word_flags+=("--backdate-start-date")
    local_nonpersistent_flags+=("--backdate-start-date")
    local_nonpersistent_flags+=("--backdate-start-date=")
    flags+=("--billing-cycle-anchor=")
    two_word_flags+=("--billing-cycle-anchor")
    local_nonpersistent_flags+=("--billing-cycle-anchor")
    local_nonpersistent_flags+=("--billing-cycle-anchor=")
    flags+=("--cancel-at=")
    two_word_flags+=("--cancel-at")
    local_nonpersistent_flags+=("--cancel-at")
    local_nonpersistent_flags+=("--cancel-at=")
    flags+=("--cancel-at-period-end=")
    two_word_flags+=("--cancel-at-period-end")
    local_nonpersistent_flags+=("--cancel-at-period-end")
    local_nonpersistent_flags+=("--cancel-at-period-end=")
    flags+=("--collection-method=")
    two_word_flags+=("--collection-method")
    local_nonpersistent_flags+=("--collection-method")
    local_nonpersistent_flags+=("--collection-method=")
    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--coupon=")
    two_word_flags+=("--coupon")
    local_nonpersistent_flags+=("--coupon")
    local_nonpersistent_flags+=("--coupon=")
    flags+=("--customer=")
    two_word_flags+=("--customer")
    local_nonpersistent_flags+=("--customer")
    local_nonpersistent_flags+=("--customer=")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--days-until-due=")
    two_word_flags+=("--days-until-due")
    local_nonpersistent_flags+=("--days-until-due")
    local_nonpersistent_flags+=("--days-until-due=")
    flags+=("--default-payment-method=")
    two_word_flags+=("--default-payment-method")
    local_nonpersistent_flags+=("--default-payment-method")
    local_nonpersistent_flags+=("--default-payment-method=")
    flags+=("--default-source=")
    two_word_flags+=("--default-source")
    local_nonpersistent_flags+=("--default-source")
    local_nonpersistent_flags+=("--default-source=")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--off-session=")
    two_word_flags+=("--off-session")
    local_nonpersistent_flags+=("--off-session")
    local_nonpersistent_flags+=("--off-session=")
    flags+=("--payment-behavior=")
    two_word_flags+=("--payment-behavior")
    local_nonpersistent_flags+=("--payment-behavior")
    local_nonpersistent_flags+=("--payment-behavior=")
    flags+=("--promotion-code=")
    two_word_flags+=("--promotion-code")
    local_nonpersistent_flags+=("--promotion-code")
    local_nonpersistent_flags+=("--promotion-code=")
    flags+=("--proration-behavior=")
    two_word_flags+=("--proration-behavior")
    local_nonpersistent_flags+=("--proration-behavior")
    local_nonpersistent_flags+=("--proration-behavior=")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--trial-end=")
    two_word_flags+=("--trial-end")
    local_nonpersistent_flags+=("--trial-end")
    local_nonpersistent_flags+=("--trial-end=")
    flags+=("--trial-from-plan=")
    two_word_flags+=("--trial-from-plan")
    local_nonpersistent_flags+=("--trial-from-plan")
    local_nonpersistent_flags+=("--trial-from-plan=")
    flags+=("--trial-period-days=")
    two_word_flags+=("--trial-period-days")
    local_nonpersistent_flags+=("--trial-period-days")
    local_nonpersistent_flags+=("--trial-period-days=")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_subscriptions_delete_discount()
{
    last_command="stripe_subscriptions_delete_discount"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_subscriptions_list()
{
    last_command="stripe_subscriptions_list"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--collection-method=")
    two_word_flags+=("--collection-method")
    local_nonpersistent_flags+=("--collection-method")
    local_nonpersistent_flags+=("--collection-method=")
    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--created=")
    two_word_flags+=("--created")
    local_nonpersistent_flags+=("--created")
    local_nonpersistent_flags+=("--created=")
    flags+=("--current-period-end=")
    two_word_flags+=("--current-period-end")
    local_nonpersistent_flags+=("--current-period-end")
    local_nonpersistent_flags+=("--current-period-end=")
    flags+=("--current-period-start=")
    two_word_flags+=("--current-period-start")
    local_nonpersistent_flags+=("--current-period-start")
    local_nonpersistent_flags+=("--current-period-start=")
    flags+=("--customer=")
    two_word_flags+=("--customer")
    local_nonpersistent_flags+=("--customer")
    local_nonpersistent_flags+=("--customer=")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--ending-before=")
    two_word_flags+=("--ending-before")
    local_nonpersistent_flags+=("--ending-before")
    local_nonpersistent_flags+=("--ending-before=")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--limit=")
    two_word_flags+=("--limit")
    local_nonpersistent_flags+=("--limit")
    local_nonpersistent_flags+=("--limit=")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--plan=")
    two_word_flags+=("--plan")
    local_nonpersistent_flags+=("--plan")
    local_nonpersistent_flags+=("--plan=")
    flags+=("--price=")
    two_word_flags+=("--price")
    local_nonpersistent_flags+=("--price")
    local_nonpersistent_flags+=("--price=")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--starting-after=")
    two_word_flags+=("--starting-after")
    local_nonpersistent_flags+=("--starting-after")
    local_nonpersistent_flags+=("--starting-after=")
    flags+=("--status=")
    two_word_flags+=("--status")
    local_nonpersistent_flags+=("--status")
    local_nonpersistent_flags+=("--status=")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_subscriptions_retrieve()
{
    last_command="stripe_subscriptions_retrieve"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--ending-before=")
    two_word_flags+=("--ending-before")
    two_word_flags+=("-b")
    local_nonpersistent_flags+=("--ending-before")
    local_nonpersistent_flags+=("--ending-before=")
    local_nonpersistent_flags+=("-b")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--limit=")
    two_word_flags+=("--limit")
    two_word_flags+=("-l")
    local_nonpersistent_flags+=("--limit")
    local_nonpersistent_flags+=("--limit=")
    local_nonpersistent_flags+=("-l")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--starting-after=")
    two_word_flags+=("--starting-after")
    two_word_flags+=("-a")
    local_nonpersistent_flags+=("--starting-after")
    local_nonpersistent_flags+=("--starting-after=")
    local_nonpersistent_flags+=("-a")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_subscriptions_update()
{
    last_command="stripe_subscriptions_update"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--application-fee-percent=")
    two_word_flags+=("--application-fee-percent")
    local_nonpersistent_flags+=("--application-fee-percent")
    local_nonpersistent_flags+=("--application-fee-percent=")
    flags+=("--billing-cycle-anchor=")
    two_word_flags+=("--billing-cycle-anchor")
    local_nonpersistent_flags+=("--billing-cycle-anchor")
    local_nonpersistent_flags+=("--billing-cycle-anchor=")
    flags+=("--cancel-at=")
    two_word_flags+=("--cancel-at")
    local_nonpersistent_flags+=("--cancel-at")
    local_nonpersistent_flags+=("--cancel-at=")
    flags+=("--cancel-at-period-end=")
    two_word_flags+=("--cancel-at-period-end")
    local_nonpersistent_flags+=("--cancel-at-period-end")
    local_nonpersistent_flags+=("--cancel-at-period-end=")
    flags+=("--collection-method=")
    two_word_flags+=("--collection-method")
    local_nonpersistent_flags+=("--collection-method")
    local_nonpersistent_flags+=("--collection-method=")
    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--coupon=")
    two_word_flags+=("--coupon")
    local_nonpersistent_flags+=("--coupon")
    local_nonpersistent_flags+=("--coupon=")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--days-until-due=")
    two_word_flags+=("--days-until-due")
    local_nonpersistent_flags+=("--days-until-due")
    local_nonpersistent_flags+=("--days-until-due=")
    flags+=("--default-payment-method=")
    two_word_flags+=("--default-payment-method")
    local_nonpersistent_flags+=("--default-payment-method")
    local_nonpersistent_flags+=("--default-payment-method=")
    flags+=("--default-source=")
    two_word_flags+=("--default-source")
    local_nonpersistent_flags+=("--default-source")
    local_nonpersistent_flags+=("--default-source=")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--off-session=")
    two_word_flags+=("--off-session")
    local_nonpersistent_flags+=("--off-session")
    local_nonpersistent_flags+=("--off-session=")
    flags+=("--payment-behavior=")
    two_word_flags+=("--payment-behavior")
    local_nonpersistent_flags+=("--payment-behavior")
    local_nonpersistent_flags+=("--payment-behavior=")
    flags+=("--promotion-code=")
    two_word_flags+=("--promotion-code")
    local_nonpersistent_flags+=("--promotion-code")
    local_nonpersistent_flags+=("--promotion-code=")
    flags+=("--proration-behavior=")
    two_word_flags+=("--proration-behavior")
    local_nonpersistent_flags+=("--proration-behavior")
    local_nonpersistent_flags+=("--proration-behavior=")
    flags+=("--proration-date=")
    two_word_flags+=("--proration-date")
    local_nonpersistent_flags+=("--proration-date")
    local_nonpersistent_flags+=("--proration-date=")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--trial-end=")
    two_word_flags+=("--trial-end")
    local_nonpersistent_flags+=("--trial-end")
    local_nonpersistent_flags+=("--trial-end=")
    flags+=("--trial-from-plan=")
    two_word_flags+=("--trial-from-plan")
    local_nonpersistent_flags+=("--trial-from-plan")
    local_nonpersistent_flags+=("--trial-from-plan=")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_subscriptions()
{
    last_command="stripe_subscriptions"

    command_aliases=()

    commands=()
    commands+=("cancel")
    commands+=("create")
    commands+=("delete_discount")
    commands+=("list")
    commands+=("retrieve")
    commands+=("update")

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_tax_codes_list()
{
    last_command="stripe_tax_codes_list"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--ending-before=")
    two_word_flags+=("--ending-before")
    local_nonpersistent_flags+=("--ending-before")
    local_nonpersistent_flags+=("--ending-before=")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--limit=")
    two_word_flags+=("--limit")
    local_nonpersistent_flags+=("--limit")
    local_nonpersistent_flags+=("--limit=")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--starting-after=")
    two_word_flags+=("--starting-after")
    local_nonpersistent_flags+=("--starting-after")
    local_nonpersistent_flags+=("--starting-after=")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_tax_codes_retrieve()
{
    last_command="stripe_tax_codes_retrieve"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--ending-before=")
    two_word_flags+=("--ending-before")
    two_word_flags+=("-b")
    local_nonpersistent_flags+=("--ending-before")
    local_nonpersistent_flags+=("--ending-before=")
    local_nonpersistent_flags+=("-b")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--limit=")
    two_word_flags+=("--limit")
    two_word_flags+=("-l")
    local_nonpersistent_flags+=("--limit")
    local_nonpersistent_flags+=("--limit=")
    local_nonpersistent_flags+=("-l")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--starting-after=")
    two_word_flags+=("--starting-after")
    two_word_flags+=("-a")
    local_nonpersistent_flags+=("--starting-after")
    local_nonpersistent_flags+=("--starting-after=")
    local_nonpersistent_flags+=("-a")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_tax_codes()
{
    last_command="stripe_tax_codes"

    command_aliases=()

    commands=()
    commands+=("list")
    commands+=("retrieve")

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_tax_ids_create()
{
    last_command="stripe_tax_ids_create"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--type=")
    two_word_flags+=("--type")
    local_nonpersistent_flags+=("--type")
    local_nonpersistent_flags+=("--type=")
    flags+=("--value=")
    two_word_flags+=("--value")
    local_nonpersistent_flags+=("--value")
    local_nonpersistent_flags+=("--value=")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_tax_ids_delete()
{
    last_command="stripe_tax_ids_delete"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_tax_ids_list()
{
    last_command="stripe_tax_ids_list"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--ending-before=")
    two_word_flags+=("--ending-before")
    local_nonpersistent_flags+=("--ending-before")
    local_nonpersistent_flags+=("--ending-before=")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--limit=")
    two_word_flags+=("--limit")
    local_nonpersistent_flags+=("--limit")
    local_nonpersistent_flags+=("--limit=")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--starting-after=")
    two_word_flags+=("--starting-after")
    local_nonpersistent_flags+=("--starting-after")
    local_nonpersistent_flags+=("--starting-after=")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_tax_ids_retrieve()
{
    last_command="stripe_tax_ids_retrieve"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--ending-before=")
    two_word_flags+=("--ending-before")
    two_word_flags+=("-b")
    local_nonpersistent_flags+=("--ending-before")
    local_nonpersistent_flags+=("--ending-before=")
    local_nonpersistent_flags+=("-b")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--limit=")
    two_word_flags+=("--limit")
    two_word_flags+=("-l")
    local_nonpersistent_flags+=("--limit")
    local_nonpersistent_flags+=("--limit=")
    local_nonpersistent_flags+=("-l")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--starting-after=")
    two_word_flags+=("--starting-after")
    two_word_flags+=("-a")
    local_nonpersistent_flags+=("--starting-after")
    local_nonpersistent_flags+=("--starting-after=")
    local_nonpersistent_flags+=("-a")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_tax_ids()
{
    last_command="stripe_tax_ids"

    command_aliases=()

    commands=()
    commands+=("create")
    commands+=("delete")
    commands+=("list")
    commands+=("retrieve")

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_tax_rates_create()
{
    last_command="stripe_tax_rates_create"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--active=")
    two_word_flags+=("--active")
    local_nonpersistent_flags+=("--active")
    local_nonpersistent_flags+=("--active=")
    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--country=")
    two_word_flags+=("--country")
    local_nonpersistent_flags+=("--country")
    local_nonpersistent_flags+=("--country=")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--description=")
    two_word_flags+=("--description")
    local_nonpersistent_flags+=("--description")
    local_nonpersistent_flags+=("--description=")
    flags+=("--display-name=")
    two_word_flags+=("--display-name")
    local_nonpersistent_flags+=("--display-name")
    local_nonpersistent_flags+=("--display-name=")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--inclusive=")
    two_word_flags+=("--inclusive")
    local_nonpersistent_flags+=("--inclusive")
    local_nonpersistent_flags+=("--inclusive=")
    flags+=("--jurisdiction=")
    two_word_flags+=("--jurisdiction")
    local_nonpersistent_flags+=("--jurisdiction")
    local_nonpersistent_flags+=("--jurisdiction=")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--percentage=")
    two_word_flags+=("--percentage")
    local_nonpersistent_flags+=("--percentage")
    local_nonpersistent_flags+=("--percentage=")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--state=")
    two_word_flags+=("--state")
    local_nonpersistent_flags+=("--state")
    local_nonpersistent_flags+=("--state=")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--tax-type=")
    two_word_flags+=("--tax-type")
    local_nonpersistent_flags+=("--tax-type")
    local_nonpersistent_flags+=("--tax-type=")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_tax_rates_list()
{
    last_command="stripe_tax_rates_list"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--active=")
    two_word_flags+=("--active")
    local_nonpersistent_flags+=("--active")
    local_nonpersistent_flags+=("--active=")
    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--created=")
    two_word_flags+=("--created")
    local_nonpersistent_flags+=("--created")
    local_nonpersistent_flags+=("--created=")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--ending-before=")
    two_word_flags+=("--ending-before")
    local_nonpersistent_flags+=("--ending-before")
    local_nonpersistent_flags+=("--ending-before=")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--inclusive=")
    two_word_flags+=("--inclusive")
    local_nonpersistent_flags+=("--inclusive")
    local_nonpersistent_flags+=("--inclusive=")
    flags+=("--limit=")
    two_word_flags+=("--limit")
    local_nonpersistent_flags+=("--limit")
    local_nonpersistent_flags+=("--limit=")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--starting-after=")
    two_word_flags+=("--starting-after")
    local_nonpersistent_flags+=("--starting-after")
    local_nonpersistent_flags+=("--starting-after=")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_tax_rates_retrieve()
{
    last_command="stripe_tax_rates_retrieve"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--ending-before=")
    two_word_flags+=("--ending-before")
    two_word_flags+=("-b")
    local_nonpersistent_flags+=("--ending-before")
    local_nonpersistent_flags+=("--ending-before=")
    local_nonpersistent_flags+=("-b")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--limit=")
    two_word_flags+=("--limit")
    two_word_flags+=("-l")
    local_nonpersistent_flags+=("--limit")
    local_nonpersistent_flags+=("--limit=")
    local_nonpersistent_flags+=("-l")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--starting-after=")
    two_word_flags+=("--starting-after")
    two_word_flags+=("-a")
    local_nonpersistent_flags+=("--starting-after")
    local_nonpersistent_flags+=("--starting-after=")
    local_nonpersistent_flags+=("-a")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_tax_rates_update()
{
    last_command="stripe_tax_rates_update"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--active=")
    two_word_flags+=("--active")
    local_nonpersistent_flags+=("--active")
    local_nonpersistent_flags+=("--active=")
    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--country=")
    two_word_flags+=("--country")
    local_nonpersistent_flags+=("--country")
    local_nonpersistent_flags+=("--country=")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--description=")
    two_word_flags+=("--description")
    local_nonpersistent_flags+=("--description")
    local_nonpersistent_flags+=("--description=")
    flags+=("--display-name=")
    two_word_flags+=("--display-name")
    local_nonpersistent_flags+=("--display-name")
    local_nonpersistent_flags+=("--display-name=")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--jurisdiction=")
    two_word_flags+=("--jurisdiction")
    local_nonpersistent_flags+=("--jurisdiction")
    local_nonpersistent_flags+=("--jurisdiction=")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--state=")
    two_word_flags+=("--state")
    local_nonpersistent_flags+=("--state")
    local_nonpersistent_flags+=("--state=")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--tax-type=")
    two_word_flags+=("--tax-type")
    local_nonpersistent_flags+=("--tax-type")
    local_nonpersistent_flags+=("--tax-type=")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_tax_rates()
{
    last_command="stripe_tax_rates"

    command_aliases=()

    commands=()
    commands+=("create")
    commands+=("list")
    commands+=("retrieve")
    commands+=("update")

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_terminal_connection_tokens_create()
{
    last_command="stripe_terminal_connection_tokens_create"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--location=")
    two_word_flags+=("--location")
    local_nonpersistent_flags+=("--location")
    local_nonpersistent_flags+=("--location=")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_terminal_connection_tokens()
{
    last_command="stripe_terminal_connection_tokens"

    command_aliases=()

    commands=()
    commands+=("create")

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_terminal_locations_create()
{
    last_command="stripe_terminal_locations_create"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--display-name=")
    two_word_flags+=("--display-name")
    local_nonpersistent_flags+=("--display-name")
    local_nonpersistent_flags+=("--display-name=")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_terminal_locations_delete()
{
    last_command="stripe_terminal_locations_delete"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_terminal_locations_list()
{
    last_command="stripe_terminal_locations_list"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--ending-before=")
    two_word_flags+=("--ending-before")
    local_nonpersistent_flags+=("--ending-before")
    local_nonpersistent_flags+=("--ending-before=")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--limit=")
    two_word_flags+=("--limit")
    local_nonpersistent_flags+=("--limit")
    local_nonpersistent_flags+=("--limit=")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--starting-after=")
    two_word_flags+=("--starting-after")
    local_nonpersistent_flags+=("--starting-after")
    local_nonpersistent_flags+=("--starting-after=")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_terminal_locations_retrieve()
{
    last_command="stripe_terminal_locations_retrieve"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--ending-before=")
    two_word_flags+=("--ending-before")
    two_word_flags+=("-b")
    local_nonpersistent_flags+=("--ending-before")
    local_nonpersistent_flags+=("--ending-before=")
    local_nonpersistent_flags+=("-b")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--limit=")
    two_word_flags+=("--limit")
    two_word_flags+=("-l")
    local_nonpersistent_flags+=("--limit")
    local_nonpersistent_flags+=("--limit=")
    local_nonpersistent_flags+=("-l")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--starting-after=")
    two_word_flags+=("--starting-after")
    two_word_flags+=("-a")
    local_nonpersistent_flags+=("--starting-after")
    local_nonpersistent_flags+=("--starting-after=")
    local_nonpersistent_flags+=("-a")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_terminal_locations_update()
{
    last_command="stripe_terminal_locations_update"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--display-name=")
    two_word_flags+=("--display-name")
    local_nonpersistent_flags+=("--display-name")
    local_nonpersistent_flags+=("--display-name=")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_terminal_locations()
{
    last_command="stripe_terminal_locations"

    command_aliases=()

    commands=()
    commands+=("create")
    commands+=("delete")
    commands+=("list")
    commands+=("retrieve")
    commands+=("update")

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_terminal_quickstart()
{
    last_command="stripe_terminal_quickstart"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_terminal_readers_create()
{
    last_command="stripe_terminal_readers_create"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--label=")
    two_word_flags+=("--label")
    local_nonpersistent_flags+=("--label")
    local_nonpersistent_flags+=("--label=")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--location=")
    two_word_flags+=("--location")
    local_nonpersistent_flags+=("--location")
    local_nonpersistent_flags+=("--location=")
    flags+=("--registration-code=")
    two_word_flags+=("--registration-code")
    local_nonpersistent_flags+=("--registration-code")
    local_nonpersistent_flags+=("--registration-code=")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_terminal_readers_delete()
{
    last_command="stripe_terminal_readers_delete"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_terminal_readers_list()
{
    last_command="stripe_terminal_readers_list"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--device-type=")
    two_word_flags+=("--device-type")
    local_nonpersistent_flags+=("--device-type")
    local_nonpersistent_flags+=("--device-type=")
    flags+=("--ending-before=")
    two_word_flags+=("--ending-before")
    local_nonpersistent_flags+=("--ending-before")
    local_nonpersistent_flags+=("--ending-before=")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--limit=")
    two_word_flags+=("--limit")
    local_nonpersistent_flags+=("--limit")
    local_nonpersistent_flags+=("--limit=")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--location=")
    two_word_flags+=("--location")
    local_nonpersistent_flags+=("--location")
    local_nonpersistent_flags+=("--location=")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--starting-after=")
    two_word_flags+=("--starting-after")
    local_nonpersistent_flags+=("--starting-after")
    local_nonpersistent_flags+=("--starting-after=")
    flags+=("--status=")
    two_word_flags+=("--status")
    local_nonpersistent_flags+=("--status")
    local_nonpersistent_flags+=("--status=")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_terminal_readers_retrieve()
{
    last_command="stripe_terminal_readers_retrieve"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--ending-before=")
    two_word_flags+=("--ending-before")
    two_word_flags+=("-b")
    local_nonpersistent_flags+=("--ending-before")
    local_nonpersistent_flags+=("--ending-before=")
    local_nonpersistent_flags+=("-b")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--limit=")
    two_word_flags+=("--limit")
    two_word_flags+=("-l")
    local_nonpersistent_flags+=("--limit")
    local_nonpersistent_flags+=("--limit=")
    local_nonpersistent_flags+=("-l")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--starting-after=")
    two_word_flags+=("--starting-after")
    two_word_flags+=("-a")
    local_nonpersistent_flags+=("--starting-after")
    local_nonpersistent_flags+=("--starting-after=")
    local_nonpersistent_flags+=("-a")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_terminal_readers_update()
{
    last_command="stripe_terminal_readers_update"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--label=")
    two_word_flags+=("--label")
    local_nonpersistent_flags+=("--label")
    local_nonpersistent_flags+=("--label=")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_terminal_readers()
{
    last_command="stripe_terminal_readers"

    command_aliases=()

    commands=()
    commands+=("create")
    commands+=("delete")
    commands+=("list")
    commands+=("retrieve")
    commands+=("update")

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_terminal()
{
    last_command="stripe_terminal"

    command_aliases=()

    commands=()
    commands+=("connection_tokens")
    commands+=("locations")
    commands+=("quickstart")
    commands+=("readers")

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_tokens_create()
{
    last_command="stripe_tokens_create"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--card=")
    two_word_flags+=("--card")
    local_nonpersistent_flags+=("--card")
    local_nonpersistent_flags+=("--card=")
    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--customer=")
    two_word_flags+=("--customer")
    local_nonpersistent_flags+=("--customer")
    local_nonpersistent_flags+=("--customer=")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_tokens_retrieve()
{
    last_command="stripe_tokens_retrieve"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--ending-before=")
    two_word_flags+=("--ending-before")
    two_word_flags+=("-b")
    local_nonpersistent_flags+=("--ending-before")
    local_nonpersistent_flags+=("--ending-before=")
    local_nonpersistent_flags+=("-b")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--limit=")
    two_word_flags+=("--limit")
    two_word_flags+=("-l")
    local_nonpersistent_flags+=("--limit")
    local_nonpersistent_flags+=("--limit=")
    local_nonpersistent_flags+=("-l")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--starting-after=")
    two_word_flags+=("--starting-after")
    two_word_flags+=("-a")
    local_nonpersistent_flags+=("--starting-after")
    local_nonpersistent_flags+=("--starting-after=")
    local_nonpersistent_flags+=("-a")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_tokens()
{
    last_command="stripe_tokens"

    command_aliases=()

    commands=()
    commands+=("create")
    commands+=("retrieve")

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_topups_cancel()
{
    last_command="stripe_topups_cancel"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_topups_create()
{
    last_command="stripe_topups_create"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--amount=")
    two_word_flags+=("--amount")
    local_nonpersistent_flags+=("--amount")
    local_nonpersistent_flags+=("--amount=")
    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--currency=")
    two_word_flags+=("--currency")
    local_nonpersistent_flags+=("--currency")
    local_nonpersistent_flags+=("--currency=")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--description=")
    two_word_flags+=("--description")
    local_nonpersistent_flags+=("--description")
    local_nonpersistent_flags+=("--description=")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--source=")
    two_word_flags+=("--source")
    local_nonpersistent_flags+=("--source")
    local_nonpersistent_flags+=("--source=")
    flags+=("--statement-descriptor=")
    two_word_flags+=("--statement-descriptor")
    local_nonpersistent_flags+=("--statement-descriptor")
    local_nonpersistent_flags+=("--statement-descriptor=")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--transfer-group=")
    two_word_flags+=("--transfer-group")
    local_nonpersistent_flags+=("--transfer-group")
    local_nonpersistent_flags+=("--transfer-group=")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_topups_list()
{
    last_command="stripe_topups_list"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--amount=")
    two_word_flags+=("--amount")
    local_nonpersistent_flags+=("--amount")
    local_nonpersistent_flags+=("--amount=")
    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--created=")
    two_word_flags+=("--created")
    local_nonpersistent_flags+=("--created")
    local_nonpersistent_flags+=("--created=")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--ending-before=")
    two_word_flags+=("--ending-before")
    local_nonpersistent_flags+=("--ending-before")
    local_nonpersistent_flags+=("--ending-before=")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--limit=")
    two_word_flags+=("--limit")
    local_nonpersistent_flags+=("--limit")
    local_nonpersistent_flags+=("--limit=")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--starting-after=")
    two_word_flags+=("--starting-after")
    local_nonpersistent_flags+=("--starting-after")
    local_nonpersistent_flags+=("--starting-after=")
    flags+=("--status=")
    two_word_flags+=("--status")
    local_nonpersistent_flags+=("--status")
    local_nonpersistent_flags+=("--status=")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_topups_retrieve()
{
    last_command="stripe_topups_retrieve"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--ending-before=")
    two_word_flags+=("--ending-before")
    two_word_flags+=("-b")
    local_nonpersistent_flags+=("--ending-before")
    local_nonpersistent_flags+=("--ending-before=")
    local_nonpersistent_flags+=("-b")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--limit=")
    two_word_flags+=("--limit")
    two_word_flags+=("-l")
    local_nonpersistent_flags+=("--limit")
    local_nonpersistent_flags+=("--limit=")
    local_nonpersistent_flags+=("-l")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--starting-after=")
    two_word_flags+=("--starting-after")
    two_word_flags+=("-a")
    local_nonpersistent_flags+=("--starting-after")
    local_nonpersistent_flags+=("--starting-after=")
    local_nonpersistent_flags+=("-a")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_topups_update()
{
    last_command="stripe_topups_update"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--description=")
    two_word_flags+=("--description")
    local_nonpersistent_flags+=("--description")
    local_nonpersistent_flags+=("--description=")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_topups()
{
    last_command="stripe_topups"

    command_aliases=()

    commands=()
    commands+=("cancel")
    commands+=("create")
    commands+=("list")
    commands+=("retrieve")
    commands+=("update")

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_transfer_reversals_create()
{
    last_command="stripe_transfer_reversals_create"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--amount=")
    two_word_flags+=("--amount")
    local_nonpersistent_flags+=("--amount")
    local_nonpersistent_flags+=("--amount=")
    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--description=")
    two_word_flags+=("--description")
    local_nonpersistent_flags+=("--description")
    local_nonpersistent_flags+=("--description=")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--refund-application-fee=")
    two_word_flags+=("--refund-application-fee")
    local_nonpersistent_flags+=("--refund-application-fee")
    local_nonpersistent_flags+=("--refund-application-fee=")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_transfer_reversals_list()
{
    last_command="stripe_transfer_reversals_list"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--ending-before=")
    two_word_flags+=("--ending-before")
    local_nonpersistent_flags+=("--ending-before")
    local_nonpersistent_flags+=("--ending-before=")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--limit=")
    two_word_flags+=("--limit")
    local_nonpersistent_flags+=("--limit")
    local_nonpersistent_flags+=("--limit=")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--starting-after=")
    two_word_flags+=("--starting-after")
    local_nonpersistent_flags+=("--starting-after")
    local_nonpersistent_flags+=("--starting-after=")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_transfer_reversals_retrieve()
{
    last_command="stripe_transfer_reversals_retrieve"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--ending-before=")
    two_word_flags+=("--ending-before")
    two_word_flags+=("-b")
    local_nonpersistent_flags+=("--ending-before")
    local_nonpersistent_flags+=("--ending-before=")
    local_nonpersistent_flags+=("-b")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--limit=")
    two_word_flags+=("--limit")
    two_word_flags+=("-l")
    local_nonpersistent_flags+=("--limit")
    local_nonpersistent_flags+=("--limit=")
    local_nonpersistent_flags+=("-l")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--starting-after=")
    two_word_flags+=("--starting-after")
    two_word_flags+=("-a")
    local_nonpersistent_flags+=("--starting-after")
    local_nonpersistent_flags+=("--starting-after=")
    local_nonpersistent_flags+=("-a")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_transfer_reversals_update()
{
    last_command="stripe_transfer_reversals_update"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_transfer_reversals()
{
    last_command="stripe_transfer_reversals"

    command_aliases=()

    commands=()
    commands+=("create")
    commands+=("list")
    commands+=("retrieve")
    commands+=("update")

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_transfers_create()
{
    last_command="stripe_transfers_create"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--amount=")
    two_word_flags+=("--amount")
    local_nonpersistent_flags+=("--amount")
    local_nonpersistent_flags+=("--amount=")
    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--currency=")
    two_word_flags+=("--currency")
    local_nonpersistent_flags+=("--currency")
    local_nonpersistent_flags+=("--currency=")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--description=")
    two_word_flags+=("--description")
    local_nonpersistent_flags+=("--description")
    local_nonpersistent_flags+=("--description=")
    flags+=("--destination=")
    two_word_flags+=("--destination")
    local_nonpersistent_flags+=("--destination")
    local_nonpersistent_flags+=("--destination=")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--source-transaction=")
    two_word_flags+=("--source-transaction")
    local_nonpersistent_flags+=("--source-transaction")
    local_nonpersistent_flags+=("--source-transaction=")
    flags+=("--source-type=")
    two_word_flags+=("--source-type")
    local_nonpersistent_flags+=("--source-type")
    local_nonpersistent_flags+=("--source-type=")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--transfer-group=")
    two_word_flags+=("--transfer-group")
    local_nonpersistent_flags+=("--transfer-group")
    local_nonpersistent_flags+=("--transfer-group=")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_transfers_list()
{
    last_command="stripe_transfers_list"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--created=")
    two_word_flags+=("--created")
    local_nonpersistent_flags+=("--created")
    local_nonpersistent_flags+=("--created=")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--destination=")
    two_word_flags+=("--destination")
    local_nonpersistent_flags+=("--destination")
    local_nonpersistent_flags+=("--destination=")
    flags+=("--ending-before=")
    two_word_flags+=("--ending-before")
    local_nonpersistent_flags+=("--ending-before")
    local_nonpersistent_flags+=("--ending-before=")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--limit=")
    two_word_flags+=("--limit")
    local_nonpersistent_flags+=("--limit")
    local_nonpersistent_flags+=("--limit=")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--starting-after=")
    two_word_flags+=("--starting-after")
    local_nonpersistent_flags+=("--starting-after")
    local_nonpersistent_flags+=("--starting-after=")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--transfer-group=")
    two_word_flags+=("--transfer-group")
    local_nonpersistent_flags+=("--transfer-group")
    local_nonpersistent_flags+=("--transfer-group=")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_transfers_retrieve()
{
    last_command="stripe_transfers_retrieve"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--ending-before=")
    two_word_flags+=("--ending-before")
    two_word_flags+=("-b")
    local_nonpersistent_flags+=("--ending-before")
    local_nonpersistent_flags+=("--ending-before=")
    local_nonpersistent_flags+=("-b")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--limit=")
    two_word_flags+=("--limit")
    two_word_flags+=("-l")
    local_nonpersistent_flags+=("--limit")
    local_nonpersistent_flags+=("--limit=")
    local_nonpersistent_flags+=("-l")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--starting-after=")
    two_word_flags+=("--starting-after")
    two_word_flags+=("-a")
    local_nonpersistent_flags+=("--starting-after")
    local_nonpersistent_flags+=("--starting-after=")
    local_nonpersistent_flags+=("-a")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_transfers_update()
{
    last_command="stripe_transfers_update"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--description=")
    two_word_flags+=("--description")
    local_nonpersistent_flags+=("--description")
    local_nonpersistent_flags+=("--description=")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_transfers()
{
    last_command="stripe_transfers"

    command_aliases=()

    commands=()
    commands+=("create")
    commands+=("list")
    commands+=("retrieve")
    commands+=("update")

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_trigger()
{
    last_command="stripe_trigger"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--add=")
    two_word_flags+=("--add")
    local_nonpersistent_flags+=("--add")
    local_nonpersistent_flags+=("--add=")
    flags+=("--override=")
    two_word_flags+=("--override")
    local_nonpersistent_flags+=("--override")
    local_nonpersistent_flags+=("--override=")
    flags+=("--raw=")
    two_word_flags+=("--raw")
    local_nonpersistent_flags+=("--raw")
    local_nonpersistent_flags+=("--raw=")
    flags+=("--remove=")
    two_word_flags+=("--remove")
    local_nonpersistent_flags+=("--remove")
    local_nonpersistent_flags+=("--remove=")
    flags+=("--skip=")
    two_word_flags+=("--skip")
    local_nonpersistent_flags+=("--skip")
    local_nonpersistent_flags+=("--skip=")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    must_have_one_noun+=("account.application.deauthorized")
    must_have_one_noun+=("account.updated")
    must_have_one_noun+=("balance.available")
    must_have_one_noun+=("charge.captured")
    must_have_one_noun+=("charge.dispute.created")
    must_have_one_noun+=("charge.failed")
    must_have_one_noun+=("charge.refunded")
    must_have_one_noun+=("charge.succeeded")
    must_have_one_noun+=("checkout.session.async_payment_failed")
    must_have_one_noun+=("checkout.session.async_payment_succeeded")
    must_have_one_noun+=("checkout.session.completed")
    must_have_one_noun+=("customer.created")
    must_have_one_noun+=("customer.deleted")
    must_have_one_noun+=("customer.source.created")
    must_have_one_noun+=("customer.source.updated")
    must_have_one_noun+=("customer.subscription.created")
    must_have_one_noun+=("customer.subscription.deleted")
    must_have_one_noun+=("customer.subscription.updated")
    must_have_one_noun+=("customer.updated")
    must_have_one_noun+=("invoice.created")
    must_have_one_noun+=("invoice.finalized")
    must_have_one_noun+=("invoice.paid")
    must_have_one_noun+=("invoice.payment_action_required")
    must_have_one_noun+=("invoice.payment_failed")
    must_have_one_noun+=("invoice.payment_succeeded")
    must_have_one_noun+=("invoice.updated")
    must_have_one_noun+=("issuing_authorization.request")
    must_have_one_noun+=("issuing_card.created")
    must_have_one_noun+=("issuing_cardholder.created")
    must_have_one_noun+=("payment_intent.amount_capturable_updated")
    must_have_one_noun+=("payment_intent.canceled")
    must_have_one_noun+=("payment_intent.created")
    must_have_one_noun+=("payment_intent.payment_failed")
    must_have_one_noun+=("payment_intent.succeeded")
    must_have_one_noun+=("payment_method.attached")
    must_have_one_noun+=("payment_method.detached")
    must_have_one_noun+=("payout.created")
    must_have_one_noun+=("payout.updated")
    must_have_one_noun+=("plan.created")
    must_have_one_noun+=("plan.deleted")
    must_have_one_noun+=("plan.updated")
    must_have_one_noun+=("price.created")
    must_have_one_noun+=("price.updated")
    must_have_one_noun+=("product.created")
    must_have_one_noun+=("product.deleted")
    must_have_one_noun+=("product.updated")
    must_have_one_noun+=("quote.accepted")
    must_have_one_noun+=("quote.canceled")
    must_have_one_noun+=("quote.created")
    must_have_one_noun+=("quote.finalized")
    must_have_one_noun+=("reporting.report_run.succeeded")
    must_have_one_noun+=("setup_intent.canceled")
    must_have_one_noun+=("setup_intent.created")
    must_have_one_noun+=("setup_intent.setup_failed")
    must_have_one_noun+=("setup_intent.succeeded")
    must_have_one_noun+=("subscription_schedule.canceled")
    must_have_one_noun+=("subscription_schedule.created")
    must_have_one_noun+=("subscription_schedule.released")
    must_have_one_noun+=("subscription_schedule.updated")
    noun_aliases=()
}

_stripe_usage_record_summaries_list()
{
    last_command="stripe_usage_record_summaries_list"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--ending-before=")
    two_word_flags+=("--ending-before")
    local_nonpersistent_flags+=("--ending-before")
    local_nonpersistent_flags+=("--ending-before=")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--limit=")
    two_word_flags+=("--limit")
    local_nonpersistent_flags+=("--limit")
    local_nonpersistent_flags+=("--limit=")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--starting-after=")
    two_word_flags+=("--starting-after")
    local_nonpersistent_flags+=("--starting-after")
    local_nonpersistent_flags+=("--starting-after=")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_usage_record_summaries()
{
    last_command="stripe_usage_record_summaries"

    command_aliases=()

    commands=()
    commands+=("list")

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_usage_records_create()
{
    last_command="stripe_usage_records_create"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--action=")
    two_word_flags+=("--action")
    local_nonpersistent_flags+=("--action")
    local_nonpersistent_flags+=("--action=")
    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--quantity=")
    two_word_flags+=("--quantity")
    local_nonpersistent_flags+=("--quantity")
    local_nonpersistent_flags+=("--quantity=")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--timestamp=")
    two_word_flags+=("--timestamp")
    local_nonpersistent_flags+=("--timestamp")
    local_nonpersistent_flags+=("--timestamp=")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_usage_records()
{
    last_command="stripe_usage_records"

    command_aliases=()

    commands=()
    commands+=("create")

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_version()
{
    last_command="stripe_version"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags+=("--device-name")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--project-name=")
    two_word_flags+=("--project-name")
    two_word_flags+=("-p")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_stripe_webhook_endpoints_create()
{
    last_command="stripe_webhook_endpoints_create"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--api-version=")
    two_word_flags+=("--api-version")
    local_nonpersistent_flags+=("--api-version")
    local_nonpersistent_flags+=("--api-version=")
    flags+=("--confirm")
    flags+=("-c")
    local_nonpersistent_flags+=("--confirm")
    local_nonpersistent_flags+=("-c")
    flags+=("--connect=")
    two_word_flags+=("--connect")
    local_nonpersistent_flags+=("--connect")
    local_nonpersistent_flags+=("--connect=")
    flags+=("--dark-style")
    local_nonpersistent_flags+=("--dark-style")
    flags+=("--data=")
    two_word_flags+=("--data")
    two_word_flags+=("-d")
    local_nonpersistent_flags+=("--data")
    local_nonpersistent_flags+=("--data=")
    local_nonpersistent_flags+=("-d")
    flags+=("--description=")
    two_word_flags+=("--description")
    local_nonpersistent_flags+=("--description")
    local_nonpersistent_flags+=("--description=")
    flags+=("--expand=")
    two_word_flags+=("--expand")
    two_word_flags+=("-e")
    local_nonpersistent_flags+=("--expand")
    local_nonpersistent_flags+=("--expand=")
    local_nonpersistent_flags+=("-e")
    flags+=("--idempotency=")
    two_word_flags+=("--idempotency")
    two_word_flags+=("-i")
    local_nonpersistent_flags+=("--idempotency")
    local_nonpersistent_flags+=("--idempotency=")
    local_nonpersistent_flags+=("-i")
    flags+=("--live")
    local_nonpersistent_flags+=("--live")
    flags+=("--show-headers")
    flags+=("-s")
    local_nonpersistent_flags+=("--show-headers")
    local_nonpersistent_flags+=("-s")
    flags+=("--stripe-account=")
    two_word_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account")
    local_nonpersistent_flags+=("--stripe-account=")
    flags+=("--stripe-version=")
    two_word_flags+=("--stripe-version")
    two_word_flags+=("-v")
    local_nonpersistent_flags+=("--stripe-version")
    local_nonpersistent_flags+=("--stripe-version=")
    local_nonpersistent_flags+=("-v")
    flags+=("--url=")
    two_word_flags+=("--url")
    local_nonpersistent_flags+=("--url")
    local_nonpersistent_flags+=("--url=")
    flags+=("--api-key=")
    two_word_flags+=("--api-key")
    flags+=("--color=")
    two_word_flags+=("--color")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--device-name=")
    two_word_flags