// Copyright 2016-2019, Pulumi Corporation.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

package main

import (
	"github.com/pulumi/pulumi/pkg/v3/backend/httpstate"
	"github.com/pulumi/pulumi/pkg/v3/secrets"
	"github.com/pulumi/pulumi/pkg/v3/secrets/service"
	"github.com/pulumi/pulumi/sdk/v3/go/common/tokens"
	"github.com/pulumi/pulumi/sdk/v3/go/common/util/contract"
	"github.com/pulumi/pulumi/sdk/v3/go/common/workspace"
)

func newServiceSecretsManager(s httpstate.Stack, stackName tokens.Name, configFile string) (secrets.Manager, error) {
	contract.Assertf(stackName != "", "stackName %s", "!= \"\"")

	if configFile == "" {
		f, err := workspace.DetectProjectStackPath(stackName.Q())
		if err != nil {
			return nil, err
		}
		configFile = f
	}

	info, err := workspace.LoadProjectStack(configFile)
	if err != nil {
		return nil, err
	}

	client := s.Backend().(httpstate.Backend).Client()
	id := s.StackIdentifier()

	// We should only save the ProjectStack at this point IF we have changed the
	// secrets provider. To change the secrets provider to a serviceSecretsManager
	// we would need to ensure that there are no remnants of the old secret manager
	// To remove those remnants, we would set those values to be empty in the project
	// stack, as per changeProjectStackSecretDetails func.
	// If we do not check to see if the secrets provider has changed, then we will actually
	// reload the configuration file to be sorted or an empty {} when creating a stack
	// this is not the desired behaviour.
	if changeProjectStackSecretDetails(info) {
		if err := workspace.SaveProjectStack(stackName.Q(), info); err != nil {
			return nil, err
		}
	}

	return service.NewServiceSecretsManager(client, id)
}

// A passphrase secrets provider has an encryption salt, therefore, changing
// from passphrase to serviceSecretsManager requires the encryption salt
// to be removed.
// A cloud secrets manager has an encryption key and a secrets provider,
// therefore, changing from cloud to serviceSecretsManager requires the
// encryption key and secrets provider to be removed.
// Regardless of what the current secrets provider is, all of these values
// need to be empty otherwise `getStackSecretsManager` in crypto.go can
// potentially return the incorrect secret type for the stack.
func changeProjectStackSecretDetails(info *workspace.ProjectStack) bool {
	var requiresSave bool
	if info.SecretsProvider != "" {
		info.SecretsProvider = ""
		requiresSave = true
	}
	if info.EncryptedKey != "" {
		info.EncryptedKey = ""
		requiresSave = true
	}
	if info.EncryptionSalt != "" {
		info.EncryptionSalt = ""
		requiresSave = true
	}
	return requiresSave
}
