package validation

import (
	"fmt"
	"strings"
	"testing"

	"github.com/pulumi/pulumi/sdk/v3/go/common/apitype"
	"github.com/stretchr/testify/assert"
)

func TestValidateStackTag(t *testing.T) {
	t.Parallel()

	t.Run("valid tags", func(t *testing.T) {
		t.Parallel()

		names := []string{
			"tag-name",
			"-",
			"..",
			"foo:bar:baz",
			"__underscores__",
			"AaBb123",
		}

		for _, name := range names {
			name := name
			t.Run(name, func(t *testing.T) {
				tags := map[apitype.StackTagName]string{
					name: "tag-value",
				}

				err := ValidateStackTags(tags)
				assert.NoError(t, err)
			})
		}
	})

	t.Run("invalid stack tag names", func(t *testing.T) {
		t.Parallel()

		var names = []string{
			"tag!",
			"something with spaces",
			"escape\nsequences\there",
			"😄",
			"foo***bar",
		}

		for _, name := range names {
			name := name
			t.Run(name, func(t *testing.T) {
				t.Parallel()

				tags := map[apitype.StackTagName]string{
					name: "tag-value",
				}

				err := ValidateStackTags(tags)
				assert.Error(t, err)
				msg := "stack tag names may only contain alphanumerics, hyphens, underscores, periods, or colons"
				assert.Equal(t, err.Error(), msg)
			})
		}
	})

	t.Run("too long tag name", func(t *testing.T) {
		t.Parallel()

		tags := map[apitype.StackTagName]string{
			strings.Repeat("v", 41): "tag-value",
		}

		err := ValidateStackTags(tags)
		assert.Error(t, err)
		msg := fmt.Sprintf("stack tag %q is too long (max length %d characters)", strings.Repeat("v", 41), 40)
		assert.Equal(t, err.Error(), msg)
	})

	t.Run("too long tag value", func(t *testing.T) {
		t.Parallel()

		tags := map[apitype.StackTagName]string{
			"tag-name": strings.Repeat("v", 257),
		}

		err := ValidateStackTags(tags)
		assert.Error(t, err)
		msg := fmt.Sprintf("stack tag %q value is too long (max length %d characters)", "tag-name", 256)
		assert.Equal(t, err.Error(), msg)
	})
}
