// Copyright 2016-2018, Pulumi Corporation.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

package deploy

import (
	"context"

	"github.com/pulumi/pulumi/sdk/v2/go/common/tokens"
	"github.com/pulumi/pulumi/sdk/v2/go/common/util/contract"
	"github.com/pulumi/pulumi/sdk/v2/go/common/util/result"
)

// NewFixedSource returns a valid planning source that is comprised of a list of pre-computed steps.
func NewFixedSource(ctx tokens.PackageName, steps []SourceEvent) Source {
	return &fixedSource{ctx: ctx, steps: steps}
}

// A fixedSource just returns from a fixed set of resource states.
type fixedSource struct {
	ctx   tokens.PackageName
	steps []SourceEvent
}

func (src *fixedSource) Close() error                { return nil }
func (src *fixedSource) Project() tokens.PackageName { return src.ctx }

func (src *fixedSource) Iterate(
	ctx context.Context, opts Options, providers ProviderSource) (SourceIterator, result.Result) {

	contract.Ignore(ctx) // TODO[pulumi/pulumi#1714]
	return &fixedSourceIterator{
		src:     src,
		current: -1,
	}, nil
}

// fixedSourceIterator always returns nil, nil in response to Next, indicating that it is done.
type fixedSourceIterator struct {
	src     *fixedSource
	current int
}

func (iter *fixedSourceIterator) Close() error {
	return nil // nothing to do.
}

func (iter *fixedSourceIterator) Next() (SourceEvent, result.Result) {
	iter.current++
	if iter.current >= len(iter.src.steps) {
		return nil, nil
	}
	return iter.src.steps[iter.current], nil
}
