package python

import (
	"path/filepath"
	"testing"

	"github.com/pulumi/pulumi/pkg/v2/codegen/internal/test"
	"github.com/stretchr/testify/assert"
)

var pathTests = []struct {
	input    string
	expected string
}{
	{".", "."},
	{"", "."},
	{"../", ".."},
	{"../..", "..."},
	{"../../..", "...."},
	{"something", ".something"},
	{"../parent", "..parent"},
	{"../../module", "...module"},
}

func TestRelPathToRelImport(t *testing.T) {
	for _, tt := range pathTests {
		t.Run(tt.input, func(t *testing.T) {
			result := relPathToRelImport(tt.input)
			if result != tt.expected {
				t.Errorf("expected \"%s\"; got \"%s\"", tt.expected, result)
			}
		})
	}
}

func TestMakeSafeEnumName(t *testing.T) {
	tests := []struct {
		input    string
		expected string
		wantErr  bool
	}{
		{"red", "RED", false},
		{"snake_cased_name", "SNAKE_CASED_NAME", false},
		{"+", "", true},
		{"*", "ASTERISK", false},
		{"0", "ZERO", false},
		{"Microsoft-Windows-Shell-Startup", "MICROSOFT_WINDOWS_SHELL_STARTUP", false},
		{"Microsoft.Batch", "MICROSOFT_BATCH", false},
		{"readonly", "READONLY", false},
		{"SystemAssigned, UserAssigned", "SYSTEM_ASSIGNED_USER_ASSIGNED", false},
		{"Dev(NoSLA)_Standard_D11_v2", "DEV_NO_SL_A_STANDARD_D11_V2", false},
		{"Standard_E8as_v4+1TB_PS", "STANDARD_E8AS_V4_1_T_B_PS", false},
		{"Plants'R'Us", "PLANTS_R_US", false},
		{"Pulumi Planters Inc.", "PULUMI_PLANTERS_INC_", false},
		{"ZeroPointOne", "ZERO_POINT_ONE", false},
	}
	for _, tt := range tests {
		t.Run(tt.input, func(t *testing.T) {
			got, err := makeSafeEnumName(tt.input)
			if (err != nil) != tt.wantErr {
				t.Errorf("makeSafeEnumName() error = %v, wantErr %v", err, tt.wantErr)
				return
			}
			if got != tt.expected {
				t.Errorf("makeSafeEnumName() got = %v, want %v", got, tt.expected)
			}
		})
	}
}

func TestGeneratePackage(t *testing.T) {
	tests := []struct {
		name          string
		schemaDir     string
		expectedFiles []string
	}{
		{
			"Simple schema with local resource properties",
			"simple-resource-schema",
			[]string{
				filepath.Join("pulumi_example", "resource.py"),
				filepath.Join("pulumi_example", "other_resource.py"),
				filepath.Join("pulumi_example", "arg_function.py"),
			},
		},
		{
			"External resource schema",
			"external-resource-schema",
			[]string{
				filepath.Join("pulumi_example", "_inputs.py"),
				filepath.Join("pulumi_example", "arg_function.py"),
				filepath.Join("pulumi_example", "cat.py"),
				filepath.Join("pulumi_example", "component.py"),
				filepath.Join("pulumi_example", "workload.py"),
			},
		},
		{
			"Simple schema with enum types",
			"simple-enum-schema",
			[]string{
				filepath.Join("pulumi_plant_provider", "_enums.py"),
				filepath.Join("pulumi_plant_provider", "_inputs.py"),
				filepath.Join("pulumi_plant_provider", "outputs.py"),
				filepath.Join("pulumi_plant_provider", "__init__.py"),
				filepath.Join("pulumi_plant_provider", "tree", "__init__.py"),
				filepath.Join("pulumi_plant_provider", "tree", "v1", "_enums.py"),
				filepath.Join("pulumi_plant_provider", "tree", "v1", "__init__.py"),
				filepath.Join("pulumi_plant_provider", "tree", "v1", "rubber_tree.py"),
			},
		},
	}

	testDir := filepath.Join("..", "internal", "test", "testdata")

	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			files, err := test.GeneratePackageFilesFromSchema(
				filepath.Join(testDir, tt.schemaDir, "schema.json"), GeneratePackage)
			assert.NoError(t, err)

			expectedFiles, err := test.LoadFiles(filepath.Join(testDir, tt.schemaDir), "python", tt.expectedFiles)
			assert.NoError(t, err)

			test.ValidateFileEquality(t, files, expectedFiles)
		})
	}
}
