// Copyright 2016-2019, Pulumi Corporation.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

package main

import (
	"encoding/base64"

	"github.com/pulumi/pulumi/pkg/v2/secrets"
	"github.com/pulumi/pulumi/pkg/v2/secrets/cloud"
	"github.com/pulumi/pulumi/sdk/v2/go/common/tokens"
	"github.com/pulumi/pulumi/sdk/v2/go/common/util/contract"
	"github.com/pulumi/pulumi/sdk/v2/go/common/workspace"
)

func newCloudSecretsManager(stackName tokens.QName, configFile, secretsProvider string) (secrets.Manager, error) {
	contract.Assertf(stackName != "", "stackName %s", "!= \"\"")

	if configFile == "" {
		f, err := workspace.DetectProjectStackPath(stackName)
		if err != nil {
			return nil, err
		}
		configFile = f
	}

	info, err := workspace.LoadProjectStack(configFile)
	if err != nil {
		return nil, err
	}

	// Only a passphrase provider has an encryption salt. So changing a secrets provider
	// from passphrase to a cloud secrets provider should ensure that we remove the enryptionsalt
	// as it's a legacy artifact and needs to be removed
	if info.EncryptionSalt != "" {
		info.EncryptionSalt = ""
	}

	var secretsManager *cloud.Manager

	// if there is no key OR the secrets provider is changing
	// then we need to generate the new key based on the new secrets provider
	if info.EncryptedKey == "" || info.SecretsProvider != secretsProvider {
		dataKey, err := cloud.GenerateNewDataKey(secretsProvider)
		if err != nil {
			return nil, err
		}
		info.EncryptedKey = base64.StdEncoding.EncodeToString(dataKey)
	}
	info.SecretsProvider = secretsProvider
	if err = info.Save(configFile); err != nil {
		return nil, err
	}

	dataKey, err := base64.StdEncoding.DecodeString(info.EncryptedKey)
	if err != nil {
		return nil, err
	}
	secretsManager, err = cloud.NewCloudSecretsManager(secretsProvider, dataKey)
	if err != nil {
		return nil, err
	}

	return secretsManager, nil
}
