// Copyright 2016-2020, Pulumi Corporation.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

package python

import (
	"encoding/json"

	"github.com/pulumi/pulumi/pkg/v2/codegen/schema"
)

// Compatibility mode for Kubernetes 2.0 SDK
const kubernetes20 = "kubernetes20"

// PropertyInfo tracks Python-specific information associated with properties in a package.
type PropertyInfo struct {
	MapCase bool `json:"mapCase,omitempty"`
}

// PackageInfo tracks Python-specific information associated with a package.
type PackageInfo struct {
	Requires map[string]string `json:"requires,omitempty"`
	// Readme contains the text for the package's README.md files.
	Readme string `json:"readme,omitempty"`
	// Optional overrides for Pulumi module names
	//
	//    { "flowcontrol.apiserver.k8s.io/v1alpha1": "flowcontrol/v1alpha1" }
	//
	ModuleNameOverrides map[string]string `json:"moduleNameOverrides,omitempty"`
	// Toggle compatibility mode for a specified target.
	Compatibility string `json:"compatibility,omitempty"`
	// Indicates whether the package generates input/output classes.
	UsesIOClasses bool `json:"usesIOClasses,omitempty"`
	// Indicates whether the package should ignore if some name has a different
	// result for PyName and PyNameLegacy.
	IgnorePyNamePanic bool `json:"ignorePyNamePanic,omitempty"`
}

// Importer implements schema.Language for Python.
var Importer schema.Language = importer(0)

type importer int

// ImportDefaultSpec decodes language-specific metadata associated with a DefaultValue.
func (importer) ImportDefaultSpec(def *schema.DefaultValue, raw json.RawMessage) (interface{}, error) {
	return raw, nil
}

// ImportPropertySpec decodes language-specific metadata associated with a Property.
func (importer) ImportPropertySpec(property *schema.Property, raw json.RawMessage) (interface{}, error) {
	var info PropertyInfo
	if err := json.Unmarshal([]byte(raw), &info); err != nil {
		return nil, err
	}
	return info, nil
}

// ImportObjectTypeSpec decodes language-specific metadata associated with a ObjectType.
func (importer) ImportObjectTypeSpec(object *schema.ObjectType, raw json.RawMessage) (interface{}, error) {
	return raw, nil
}

// ImportResourceSpec decodes language-specific metadata associated with a Resource.
func (importer) ImportResourceSpec(resource *schema.Resource, raw json.RawMessage) (interface{}, error) {
	return raw, nil
}

// ImportFunctionSpec decodes language-specific metadata associated with a Function.
func (importer) ImportFunctionSpec(function *schema.Function, raw json.RawMessage) (interface{}, error) {
	return raw, nil
}

// ImportPackageSpec decodes language-specific metadata associated with a Package.
func (importer) ImportPackageSpec(pkg *schema.Package, raw json.RawMessage) (interface{}, error) {
	var info PackageInfo
	if err := json.Unmarshal([]byte(raw), &info); err != nil {
		return nil, err
	}
	return info, nil
}
