// Implementation of Step interface. Entry point to building code.
package build

import "fmt"
import "os"
import "os/exec"
import "path"
import "path/filepath"
import "runtime"
import "strings"
import "github.com/op/go-logging"

import "core"

var log = logging.MustGetLogger("build")

const dirPermissions = os.ModeDir | 0775


func Build(tid int, state *core.BuildState, label core.BuildLabel) {
    target := state.Graph.Target(label)
    if target == nil {
        state.Results <- core.NewBuildResult(tid, label, core.TargetBuildFailed, "Couldn't find build target")
        return
    }
    target.State = core.Building
    if err := buildTarget(tid, state, target); err != nil {
        state.Results <- core.NewBuildError(tid, label, core.TargetBuildFailed, err, "Build failed")
        target.State = core.Failed
        return
    }
    // Add any of the reverse deps that are now fully built to the queue.
    for reverseDepLabel, _ := range(target.ReverseDependencies) {
        reverseDep := state.Graph.Target(reverseDepLabel)
        if reverseDep.State == core.Active && reverseDep.AllDepsBuilt() && reverseDep.ActiveToPending() {
            state.PendingBuilds <- reverseDepLabel
        }
    }
    if target.IsTest {
        state.PendingTests <- target.Label
    }
}

// Builds a single target
func buildTarget(tid int, state *core.BuildState, target *core.BuildTarget) error {
    if err := target.CheckDependencyVisibility(); err != nil {
        return err
    }
    state.Results <- core.NewBuildResult(tid, target.Label, core.TargetBuilding, "Preparing...")
    // This must run before needsBuilding.
    if err := updateOutputs(state.Graph, target); err != nil {
        return fmt.Errorf("Error calculating outputs for %s: %s", target.Label, err)
    }
    if !needsBuilding(state, target) {
        log.Debug("Not rebuilding %s, nothing's changed", target.Label)
        target.State = core.Unchanged
        state.Results <- core.NewBuildResult(tid, target.Label, core.TargetCached, "Cached")
        return nil  // Nothing needs to be done.
    }
    if err := prepareDirectories(target); err != nil {
        return fmt.Errorf("Error preparing directories for %s: %s", target.Label, err)
    }
    if err := prepareSources(state.Graph, target, target, map[core.BuildLabel]bool{}); err != nil {
        return fmt.Errorf("Error preparing sources for %s: %s", target.Label, err)
    }
    state.Results <- core.NewBuildResult(tid, target.Label, core.TargetBuilding, target.BuildingDescription)
    replacedCmd := replaceSequences(target)
	cmd := exec.Command("bash", "-c", replacedCmd)
    cmd.Dir = target.TmpDir()
    cmd.Env = buildEnvironment(state.Graph, target)
    log.Debug("Building target %s\nENVIRONMENT:\n%s\n%s",
        target.Label, strings.Join(cmd.Env, "\n"), replacedCmd)
    out, err := cmd.CombinedOutput()
    if err != nil {
        if state.Verbose {
            return fmt.Errorf("Error building target %s: %s\nENVIRONMENT:\n%s\n%s\n%s",
                target.Label, err, strings.Join(cmd.Env, "\n"), target.Command, out)
        } else {
            return fmt.Errorf("Error building target %s: %s\n%s", target.Label, err, out)
        }
    }
    state.Results <- core.NewBuildResult(tid, target.Label, core.TargetBuilding, "Collecting outputs...")
    outputsChanged, err := moveOutputs(state, target)
    if err != nil {
        return fmt.Errorf("Error moving outputs for target %s: %s", target.Label, err)
    } else if outputsChanged {
        target.State = core.Built
    } else {
        target.State = core.Unchanged
    }
    state.Results <- core.NewBuildResult(tid, target.Label, core.TargetBuilt, "Built")
    return nil
}

// Prepares the output directories for a target
func prepareDirectories(target *core.BuildTarget) error {
    if err := prepareDirectory(target.TmpDir(), true); err != nil {
        return err
    }
    if err := prepareDirectory(target.OutDir(), false); err != nil {
        return err
    }
    // Nicety for the build rules: create any directories that it's
    // declared it'll create files in.
    for _, out := range(target.Outputs) {
        if dir := path.Dir(out); dir != "." {
            outPath := path.Join(target.TmpDir(), dir)
            if !core.PathExists(outPath) {
                if err := os.MkdirAll(outPath, dirPermissions); err != nil {
                    return err
                }
            }
        }
    }
    // Buck src dir is a link to this package in the source tree.
    return os.Symlink(target.Label.PackageName, target.BuckSrcDir())
}

func prepareDirectory(directory string, remove bool) error {
    if remove && core.PathExists(directory) {
        if err := os.RemoveAll(directory); err != nil {
            return err
        }
    }
    return os.MkdirAll(directory, dirPermissions)  // drwxrwxr-x
}

// Symlinks the source files of this rule into its temp directory.
func prepareSources(graph *core.BuildGraph, target *core.BuildTarget, dependency *core.BuildTarget, done map[core.BuildLabel]bool) error {
    for source := range iterSources(graph, target, true) {
        if err := prepareSource(source.src, source.tmp); err != nil {
            return err
        }
    }
    return nil
}

// Symlinks a single source file for a build rule.
func prepareSource(sourcePath string, tmpPath string) error {
    sourcePath = path.Join(core.RepoRoot, sourcePath)
    dir := path.Dir(tmpPath)
    if !core.PathExists(dir) {
        if err := os.MkdirAll(dir, dirPermissions); err != nil {
            return err
        }
    }
    return os.Symlink(sourcePath, tmpPath)
}

func moveOutputs(state *core.BuildState, target *core.BuildTarget) (bool, error) {
    // Before we write any outputs, we must remove the old hash file to avoid it being
    // left in an inconsistent state.
    if err := os.Remove(ruleHashFileName(target)); err != nil && !os.IsNotExist(err) {
        return true, err
    }
    changed := false
    for _, output := range(target.Outputs) {
        tmpOutput := path.Join(target.TmpDir(), output)
        realOutput := path.Join(target.OutDir(), output)
        if !core.PathExists(tmpOutput) {
            return true, fmt.Errorf("Rule %s failed to create output %s", target.Label, tmpOutput)
        }
        // If output is a symlink, dereference it. Otherwise, for efficiency,
        // we can just move it without a full copy (saves copying large .jar files etc).
        dereferencedPath, err := filepath.EvalSymlinks(tmpOutput)
        if err != nil {
            return true, err
        }
        // The tmp output can be a symlink back to the real one; this is allowed for rules like
        // filegroups that attempt to link outputs of other rules. In that case we can't
        // remove the original because that'd break the link, but by definition we don't need
        // to actually do anything more.
        if absOutput, _ := filepath.Abs(realOutput); dereferencedPath == absOutput {
            continue
        }
        if core.PathExists(realOutput) {
            if !hasFileChanged(realOutput, dereferencedPath) {
                continue
            } else if err := os.RemoveAll(realOutput); err != nil {
                return true, err
            }
        }
        changed = true
        // Check if we need a directory for this output.
        dir := path.Dir(realOutput)
        if !core.PathExists(dir) {
            if err := os.MkdirAll(dir, dirPermissions); err != nil {
                return true, err
            }
        }
        if dereferencedPath == tmpOutput {
            if err := os.Rename(tmpOutput, realOutput); err != nil {
                return true, err
            }
        } else {
            // I would just copy here, but Go doesn't have a built in copy function. ggrrrr...
            absPath, err := filepath.Abs(dereferencedPath)
            if err != nil {
                return true, err
            }
            if err := os.Symlink(absPath, realOutput); err != nil {
                return true, err
            }
        }
    }
    if err := writeRuleHashFile(state, target); err != nil {
        return changed, fmt.Errorf("Attempting to create hash file: %s", err)
    }
    return changed, nil
}

func buildEnvironment(graph *core.BuildGraph, target *core.BuildTarget) []string {
    sources := target.SourcePaths(graph)
    env := []string {
        // Need to know these for certain rules, particularly Go rules.
        "ARCH=" + runtime.GOARCH,
        "OS=" + runtime.GOOS,
        // Use a restricted PATH; it'd be easier for the user if we pass it through
        // but really external environment variables shouldn't affect this.
        "PATH=/usr/local/bin:/usr/bin:/bin",
        "TMP_DIR=" + path.Join(core.RepoRoot, target.TmpDir()),
        "OUT_DIR=" + path.Join(core.RepoRoot, target.OutDir()),
        "ROOT_DIR=" + core.RepoRoot,
        "SRCS=" + strings.Join(sources, " "),
        "OUTS=" + strings.Join(target.Outputs, " "),
        "PKG=" + target.Label.PackageName,
        "PY_PKG=" + strings.Replace(target.Label.PackageName, "/", ".", -1),
        // Provided for some form of partial Buck compatibility. Try not to use.
        "SRCDIR=" + path.Join(core.RepoRoot, target.BuckSrcDir()),
    }
    // The OUT variable is only available on rules that have a single output.
    if len(target.Outputs) == 1 {
        env = append(env, "OUT=" + path.Join(core.RepoRoot, target.TmpDir(), target.Outputs[0]))
    }
    // The SRC variable is only available on rules that have a single source file.
    if len(sources) == 1 {
        env = append(env, "SRC=" + sources[0])
    }
    return env
}

// It's convenient to allow targets to have outputs as source labels
// (eg. filegroup to collect the outputs of other rules). We have to change these into
// actual files, but we can't do that at parse time because at that point we
// don't necessarily know what the outputs of the other rules would be.
func updateOutputs(graph *core.BuildGraph, target *core.BuildTarget) error {
    new_outs := make([]string, 0, len(target.Outputs))
    for _, out := range(target.Outputs) {
        if core.LooksLikeABuildLabel(out) {
            label := core.ParseBuildLabel(out, target.Label.PackageName)
            target := graph.Target(label)
            if target == nil {
                return fmt.Errorf("No matching build target: %s", out)
            }
            new_outs = append(new_outs, target.Outputs...)
        } else {
            new_outs = append(new_outs, out)
        }
    }
    // Replace these shell-style placeholders.
    // Needed to allow rules (particularly Go rules) to specify architecture-specific outs.
    for i, out := range(new_outs) {
        tmp := strings.Replace(out, "${ARCH}", runtime.GOARCH, -1)
        new_outs[i] = strings.Replace(tmp, "${OS}", runtime.GOOS, -1)
    }
    target.Outputs = new_outs
    return nil
}
