// Copyright The OpenTelemetry Authors
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

package otelhttp // import "go.opentelemetry.io/contrib/instrumentation/net/http/otelhttp"

import (
	"net/http"

	"go.opentelemetry.io/otel/attribute"
	"go.opentelemetry.io/otel/trace"
)

// Attribute keys that can be added to a span.
const (
	ReadBytesKey  = attribute.Key("http.read_bytes")  // if anything was read from the request body, the total number of bytes read
	ReadErrorKey  = attribute.Key("http.read_error")  // If an error occurred while reading a request, the string of the error (io.EOF is not recorded)
	WroteBytesKey = attribute.Key("http.wrote_bytes") // if anything was written to the response writer, the total number of bytes written
	WriteErrorKey = attribute.Key("http.write_error") // if an error occurred while writing a reply, the string of the error (io.EOF is not recorded)
)

// Server HTTP metrics.
const (
	// Deprecated: This field is unused.
	RequestCount = "http.server.request_count" // Incoming request count total
	// Deprecated: Use of this field has been migrated to serverRequestSize. It will be removed in a future version.
	RequestContentLength = "http.server.request_content_length" // Incoming request bytes total
	// Deprecated: Use of this field has been migrated to serverResponseSize. It will be removed in a future version.
	ResponseContentLength = "http.server.response_content_length" // Incoming response bytes total
	// Deprecated: Use of this field has been migrated to serverDuration. It will be removed in a future version.
	ServerLatency      = "http.server.duration"      // Incoming end to end duration, milliseconds
	serverRequestSize  = "http.server.request.size"  // Incoming request bytes total
	serverResponseSize = "http.server.response.size" // Incoming response bytes total
	serverDuration     = "http.server.duration"      // Incoming end to end duration, milliseconds
)

// Client HTTP metrics.
const (
	clientRequestSize  = "http.client.request.size"  // Outgoing request bytes total
	clientResponseSize = "http.client.response.size" // Outgoing response bytes total
	clientDuration     = "http.client.duration"      // Outgoing end to end duration, milliseconds
)

// Filter is a predicate used to determine whether a given http.request should
// be traced. A Filter must return true if the request should be traced.
type Filter func(*http.Request) bool

func newTracer(tp trace.TracerProvider) trace.Tracer {
	return tp.Tracer(ScopeName, trace.WithInstrumentationVersion(Version()))
}
