package docs

func init() {

	Definition["tmp"] = "# _murex_ Shell Docs\n\n## Command Reference: `tmp`\n\n> Create a temporary file and write to it\n\n## Description\n\n`tmp` creates a temporary file, writes the contents of STDIN to it then returns\nits filename to STDOUT.\n\nYou can optionally specify a file extension, for example if the temporary file\nneeds to be read by `open` or an editor which uses extensions to define syntax\nhighlighting.\n\n## Usage\n\n    <stdin> -> tmp [ file-extension ] -> <stdout>\n\n## Examples\n\n    » out: \"Hello, world!\" -> set: tmp\n    \n    » out: $tmp\n    /var/folders/3t/267q_b0j27d29bnf6pf7m7vm0000gn/T/murex838290600/8ec6936c1ac1c347bf85675eab4a0877-13893\n    \n    » open: $tmp\n    Hello, world!\n\n## Detail\n\nThe temporary file name is a base64 encoded md5 hash of the time plus _murex_\nfunction ID with _murex_ process ID appended:\n\n```go\npackage io\n\nimport (\n\t\"crypto/md5\"\n\t\"encoding/hex\"\n\t\"io\"\n\t\"os\"\n\t\"strconv\"\n\t\"time\"\n\n\t\"github.com/lmorg/murex/lang\"\n\t\"github.com/lmorg/murex/lang/types\"\n\t\"github.com/lmorg/murex/utils/consts\"\n)\n\nfunc init() {\n\tlang.DefineMethod(\"tmp\", cmdTempFile, types.Any, types.String)\n}\n\nfunc cmdTempFile(p *lang.Process) error {\n\tp.Stdout.SetDataType(types.String)\n\n\text, _ := p.Parameters.String(0)\n\tif ext != \"\" {\n\t\text = \".\" + ext\n\t}\n\n\tfileId := time.Now().String() + \":\" + strconv.Itoa(int(p.Id))\n\n\th := md5.New()\n\t_, err := h.Write([]byte(fileId))\n\tif err != nil {\n\t\treturn err\n\t}\n\n\tname := consts.TempDir + hex.EncodeToString(h.Sum(nil)) + \"-\" + strconv.Itoa(os.Getpid()) + ext\n\n\tfile, err := os.Create(name)\n\tif err != nil {\n\t\treturn err\n\t}\n\n\tdefer file.Close()\n\n\t_, err = io.Copy(file, p.Stdin)\n\tif err != nil {\n\t\treturn err\n\t}\n\n\t_, err = p.Stdout.Write([]byte(name))\n\treturn err\n}\n```\n\nThis should should provide enough distance to run `tmp` in parallel....should\nyou ever want to.\n\n`tmp` files are also located inside a unique per-process _murex_ temp directory\nwhich itself is located in the appropriate temp directory for the host OS (eg\n`$TMPDIR` on macOS).\n\n## See Also\n\n* [`>>` (append file)](../commands/greater-than-greater-than.md):\n  Writes STDIN to disk - appending contents if file already exists\n* [`>` (truncate file)](../commands/greater-than.md):\n  Writes STDIN to disk - overwriting contents if file already exists\n* [`open`](../commands/open.md):\n  Open a file with a preferred handler\n* [`pipe`](../commands/pipe.md):\n  Manage _murex_ named pipes"

}
