package main

import (
	"bytes"
	"fmt"
	"os"
	"os/exec"
	"regexp"
	"text/template"

	"github.com/sirupsen/logrus"
)

func isExecutableInPath(candidates []string) string {
	for _, cmd := range candidates {
		if path, err := exec.LookPath(cmd); err == nil {
			return path
		}
	}
	return ""
}

type LauncherBlock struct {
	cmd       string // The command to execute for the infostring above
	extension string // The file extension for the language
}

// global storage for launchers
// the key is the infostring from the code fence
var launchers = map[string]LauncherBlock{}

func loadLaunchers() error {
	addedLaunchers := []string{}

	if env := isExecutableInPath([]string{"env"}); env != "" {
		launchers["env"] = LauncherBlock{cmd: env, extension: "env"}
		addedLaunchers = append(addedLaunchers, env)
	} else {
		return fmt.Errorf("%w: env must be available in PATH", ErrNoEnv)
	}
	if cmd := isExecutableInPath([]string{"sh"}); cmd != "" {
		launchers["sh"] = LauncherBlock{cmd: cmd, extension: "sh"}
		launchers["bash"] = LauncherBlock{cmd: cmd, extension: "sh"}
		addedLaunchers = append(addedLaunchers, cmd)
	}

	if cmd := isExecutableInPath([]string{"bash"}); cmd != "" {
		launchers["bash"] = LauncherBlock{cmd: cmd, extension: "bash"}
		addedLaunchers = append(addedLaunchers, cmd)
	}

	pythonCandidates := []string{"python", "python3"}
	if cmd := isExecutableInPath(pythonCandidates); cmd != "" {
		launchers["py"] = LauncherBlock{cmd: cmd, extension: "py"}
		addedLaunchers = append(addedLaunchers, cmd)
	}

	logrus.Debug("Added launchers: ", addedLaunchers)
	return nil
}

func listLaunchers() {
	fmt.Println("Available launchers:")
	for lang, launcher := range launchers {
		fmt.Printf("%s: %s (extension: %s)\n", lang, launcher.cmd, launcher.extension)
	}
}

/*
Before executing commandBlock, this function validates that all dependencies are present in the commands map.
*/
func validateDependencies(commands map[string]CommandBlock, commandBlock *CommandBlock) error {
	for _, dep := range commandBlock.Dependencies {
		if _, ok := commands[dep]; !ok {
			return fmt.Errorf("%w: %s", ErrDependencyNotFound, dep)
		}
		dependency := commands[dep]
		if err := validateDependencies(commands, &dependency); err != nil {
			return err
		}
	}
	return nil
}

func executeCommandBlock(commands map[string]CommandBlock, commandBlock *CommandBlock, args ...string) error {

	if err := validateDependencies(commands, commandBlock); err != nil {
		return err
	}

	for _, dep := range commandBlock.Dependencies {
		if _, ok := commands[dep]; !ok {
			return fmt.Errorf("%w: %s", ErrDependencyNotFound, dep)
		}
		dependency := commands[dep]
		if err := executeCommandBlock(commands, &dependency); err != nil {
			logrus.Debug(fmt.Sprintf("Executing command %s with args %v", dependency.Name, args))
			return err
		}
	}

	for i, codeBlock := range commandBlock.CodeBlocks {
		logrus.Debug(fmt.Sprintf("Executing Code Block #%d", i))

		var err error
		if i == 0 {
			err = executeCodeBlock(&codeBlock, args...)
		} else {
			err = executeCodeBlock(&codeBlock)
		}

		if err != nil {
			if codeBlock.Config.OnError == "ignore" {
				logrus.Debug(fmt.Sprintf("Ignoring error in command block %s (b/c on-error = ignore): %v", commandBlock.Name, err))
			} else {
				return err
			}
		}
	}

	return nil

}
func executeCodeBlock(codeBlock *CodeBlock, args ...string) error {

	// Create a map for the template arguments
	argMap := make(map[string]string)
	for i, arg := range args {
		argMap[fmt.Sprintf("arg%d", i+1)] = arg
	}

	// Validate that all placeholders in the template are provided in args and vice versa
	placeholderPattern := regexp.MustCompile(`{{\s*\.arg(\d+)\s*}}`)
	matches := placeholderPattern.FindAllStringSubmatch(codeBlock.Code, -1)

	placeholderSet := make(map[string]struct{})
	for _, match := range matches {
		placeholderSet[match[1]] = struct{}{}
	}

	for i := range args {
		argKey := fmt.Sprintf("%d", i+1)
		if _, ok := placeholderSet[argKey]; !ok {
			return fmt.Errorf("%w: argument: %d (\"%s\")", ErrArgProvidedButNotUsed, i+1, args[i])
		}
	}

	for placeholder := range placeholderSet {
		argIndex := fmt.Sprintf("arg%s", placeholder)
		if _, ok := argMap[argIndex]; !ok {
			return fmt.Errorf("%w: {{.arg%s}}", ErrArgUsedInTemplateNotProvided, placeholder)
		}
	}

	tmpl, err := template.New("command").Parse(codeBlock.Code)
	if err != nil {
		return fmt.Errorf("failed to parse template: %v", err)
	}

	var renderedCode bytes.Buffer
	err = tmpl.Execute(&renderedCode, argMap)
	if err != nil {
		return fmt.Errorf("failed to execute template: %v", err)
	}

	launcher, ok := launchers[codeBlock.Lang]
	if !ok && !codeBlock.Config.SheBang {
		return fmt.Errorf("%w: %s", ErrNoInfostringOrShebang, codeBlock.Lang)
	}

	// Write the rendered code to the temporary file
	tmpFile, err := os.CreateTemp("", fmt.Sprintf("mdx-*.%s", launcher.extension))
	if err != nil {
		return fmt.Errorf("failed to create temporary file: %v", err)
	}
	// Set the permissions of the temporary file to 755
	if err := os.Chmod(tmpFile.Name(), 0755); err != nil {
		return fmt.Errorf("failed to set permissions on temporary file: %v", err)
	}

	defer os.Remove(tmpFile.Name())

	if !codeBlock.Config.SheBang {
		env := launchers["env"]
		if _, err := tmpFile.Write([]byte(fmt.Sprintf("#!%s %s\n", env.cmd, launcher.cmd))); err != nil {
			return fmt.Errorf("failed to write to temporary file: %v", err)
		}

	}
	if _, err := tmpFile.Write(renderedCode.Bytes()); err != nil {
		return fmt.Errorf("failed to write to temporary file: %v", err)
	}
	if err := tmpFile.Close(); err != nil {
		return fmt.Errorf("failed to close temporary file: %v", err)
	}

	cmd := exec.Command(tmpFile.Name())
	cmd.Stdout = os.Stdout
	cmd.Stderr = os.Stderr

	cmd.Dir = os.Getenv("PWD")
	logrus.Debug(fmt.Sprintf("Executing command in directory: %s", cmd.Dir))

	if err := cmd.Run(); err != nil {
		content, readErr := os.ReadFile(tmpFile.Name())
		if readErr != nil {
			return fmt.Errorf("failed to execute command: %v, and failed to read temporary file: %v", err, readErr)
		}
		fmt.Fprintf(os.Stderr, "Error executing Code Block:\n%s\n", content)
		return fmt.Errorf("%w: %v", ErrCodeBlockExecFailed, err)
	}
	return nil
}
