---
title: "Generating ConfigMaps"
linkTitle: "Generating ConfigMaps"
weight: 1
date: 2023-10-20
description: >
    Working with the ConfigMap Generator
---

ConfigMap objects can be generated by adding a [`configMapGenerator`] entry to the `kustomization.yaml` file.

## Create ConfigMap from a file

ConfigMap Resources may be generated from files - such as a java `.properties` file.  To generate a ConfigMap Resource for a file, add an entry to [`configMapGenerator`] with the filename.

The ConfigMaps will have data values populated from the file contents. The contents of each file will appear as a single data item in the ConfigMap keyed by the filename.

The following example generates a ConfigMap with a data item containing the contents of a file.

1. Create a Kustomization file.
```yaml
# kustomization.yaml
apiVersion: kustomize.config.k8s.io/v1beta1
kind: Kustomization
configMapGenerator:
- name: my-application-properties
  files:
  - application.properties
```

2. Create a `.properties` file.
```yaml
# application.properties
FOO=Bar
```

3. Create the ConfigMap using `kustomize build`.
```bash
kustomize build .
```

The output is similar to:
```yaml
apiVersion: v1
data:
  application.properties: |-
    FOO=Bar
kind: ConfigMap
metadata:
  name: my-application-properties-f7mm6mhf59
```

It is also possible to [define a key](https://kubernetes.io/docs/tasks/configure-pod-container/configure-pod-configmap/#define-the-key-to-use-when-creating-a-configmap-from-a-file) to set a name different than the filename.

The example below creates a ConfigMap with the name of file as `myFileName.ini` while the _actual_ filename from which the configmap is created is `whatever.ini`.

```yaml
apiVersion: kustomize.config.k8s.io/v1beta1
kind: Kustomization

configMapGenerator:
- name: app-whatever
  files:
  - myFileName.ini=whatever.ini
```


## Create ConfigMap from literals

ConfigMap Resources may be generated from literal key-value pairs - such as `JAVA_HOME=/opt/java/jdk`.
To generate a ConfigMap Resource from literal key-value pairs, add an entry to [`configMapGenerator`] with a
list of `literals`.

{{< alert color="success" title="Literal Syntax" >}}
- The key/value are separated by a `=` sign (left side is the key).
- The value of each literal will appear as a data item in the ConfigMap keyed by its key.
{{< /alert >}}

The following example generates a ConfigMap with two data items generated from literals.

1. Create a Kustomization file.
```yaml
# kustomization.yaml
apiVersion: kustomize.config.k8s.io/v1beta1
kind: Kustomization
configMapGenerator:
- name: my-java-server-env-vars
  literals:
  - JAVA_HOME=/opt/java/jdk
  - JAVA_TOOL_OPTIONS=-agentlib:hprof
```

2. Create the ConfigMap using `kustomize build`.
```bash
kustomize build .
```

The output is similar to:
```yaml
apiVersion: v1
data:
  JAVA_HOME: /opt/java/jdk
  JAVA_TOOL_OPTIONS: -agentlib:hprof
kind: ConfigMap
metadata:
  name: my-java-server-env-vars-44k658k8gk
```

## Create ConfigMap from env file
ConfigMap Resources may be generated from key-value pairs much the same as using the literals option
but taking the key-value pairs from an environment file. These generally end in `.env`.
To generate a ConfigMap Resource from an environment file, add an entry to [`configMapGenerator`] with a
single `envs` entry, e.g. `envs: [ 'config.env' ]`.

{{< alert color="success" title="Environment File Syntax" >}}
- The key/value pairs inside of the environment file are separated by a `=` sign (left side is the key).
- The value of each line will appear as a data item in the ConfigMap keyed by its key.
- Pairs may span a single line only.
{{< /alert >}}

The following example generates a ConfigMap with three data items generated from an environment file.

1. Create a Kustomization file.
```yaml
# kustomization.yaml
apiVersion: kustomize.config.k8s.io/v1beta1
kind: Kustomization
configMapGenerator:
- name: tracing-options
  envs:
  - tracing.env
```

2. Create an environment file.
```bash
# tracing.env
ENABLE_TRACING=true
SAMPLER_TYPE=probabilistic
SAMPLER_PARAMETERS=0.1
```

3. Create the ConfigMap using `kustomize build`.
```bash
kustomize build .
```

The output is similar to:
```yaml
apiVersion: v1
kind: ConfigMap
metadata:
  # The name has had a suffix applied
  name: tracing-options-6bh8gkdf7k
# The data has been populated from each literal pair
data:
  ENABLE_TRACING: "true"
  SAMPLER_TYPE: "probabilistic"
  SAMPLER_PARAMETERS: "0.1"
```

## Create ConfigMap with options
The labels and annotations of a generated ConfigMap can be set with the `options` field. The name suffix hash can also be disabled.

Labels and annotations added with `options` will not be overwritten by values defind in the `generatorOptions` field. Note that `disableNameSuffixHash: true` defined in `globalOptions` will override the locally defined `options`. This is a result of boolean behavior.

The following example generates a ConfigMap with labels, annotations and does not add a suffix hash to the name.

1. Create a Kustomization file.
```yaml
# kustomization.yaml
apiVersion: kustomize.config.k8s.io/v1beta1
kind: Kustomization

generatorOptions:
  labels:
    fruit: apple

configMapGenerator:
- name: my-java-server-env-vars
  literals:
  - JAVA_HOME=/opt/java/jdk
  - JAVA_TOOL_OPTIONS=-agentlib:hprof
  options:
    disableNameSuffixHash: true
    labels:
      pet: dog
    annotations:
      dashboard: "1"
```

2. Create the ConfigMap using `kustomize build`.
```bash
kustomize build .
```

The ConfigMap manifest is created with labels and annotations.
```yaml
apiVersion: v1
kind: ConfigMap
metadata:
  name: my-java-server-env-vars
  labels:
    fruit: apple
    pet: dog
  annotations:
    dashboard: "1"
data:
  JAVA_HOME: /opt/java/jdk
  JAVA_TOOL_OPTIONS: -agentlib:hprof
```

## Override base ConfigMap values
ConfigMap values from bases may be overridden by adding another generator for the ConfigMap
in the overlay and specifying the `behavior` field.

`behavior` may be one of:
* `create` (default value): used to create a new ConfigMap. A name conflict error will be thrown if a ConfigMap with the same name and namespace already exists.
* `replace`: replace an existing ConfigMap from the base.
* `merge`: add or update the values in an existing ConfigMap from the base.

When updating an existing ConfigMap with the `merge` or `replace` strategies, you must ensure that both the name and namespace match the ConfigMap you are targeting. For example, if the namespace is unspecified in the base, you should not specify it in the overlay. Conversely, if it is specified in the base, you must specify it in the overlay as well. This is true even if the overlay Kustomization includes a namespace, because `configMapGenerator` runs before the namespace transformer.

```yaml
apiVersion: kustomize.config.k8s.io/v1beta1
kind: Kustomization

namespace: my-new-namespace

resources:
- ../base

configMapGenerator:
  - name: existing-name
    namespace: existing-ns # needs to match target ConfigMap from base
    behavior: replace
    literals:
      - ENV=dev
```

{{< alert color="warning" title="Name suffixing with overlay configMapGenerator" >}}
When using `configMapGenerator` to override values of an existing ConfigMap, the overlay `configMapGenerator` does not cause suffixing of the existing ConfigMap's name to occur. To take advantage of name suffixing, use `configMapGenerator` in the base, and the overlay generator will correctly update the suffix based on the new content.
{{< /alert >}}


## Propagate ConfigMap Name Suffix

Workloads that reference the ConfigMap or Secret will need to know the name of the generated Resource,
including the suffix. Kustomize takes care of this automatically by identifying
references to generated ConfigMaps and Secrets, and updating them.

In the following example, the generated ConfigMap name will be `my-java-server-env-vars` with a suffix unique to its contents. Changes to the contents will change the name suffix, resulting in the creation of a new ConfigMap, which Kustomize will transform Workloads to point to.

The PodTemplate volume references the ConfigMap by the name specified in the generator (excluding the suffix). Kustomize will update the name to include the suffix applied to the ConfigMap name.

The following example generates a ConfigMap and propagates the ConfigMap name, including the suffix, to a Deployment that mounts the ConfigMap.

1. Create a Kustomization file.
```yaml
# kustomization.yaml
apiVersion: kustomize.config.k8s.io/v1beta1
kind: Kustomization
configMapGenerator:
- name: my-java-server-env-vars
  literals:
  - JAVA_HOME=/opt/java/jdk
  - JAVA_TOOL_OPTIONS=-agentlib:hprof
resources:
- deployment.yaml
```

2. Create a Deployment manifest.
```yaml
# deployment.yaml
apiVersion: apps/v1
kind: Deployment
metadata:
  name: test-deployment
  labels:
    app: test
spec:
  selector:
    matchLabels:
      app: test
  template:
    metadata:
      labels:
        app: test
    spec:
      containers:
      - name: container
        image: registry.k8s.io/busybox
        command: [ "/bin/sh", "-c", "ls /etc/config/" ]
        volumeMounts:
        - name: config-volume
          mountPath: /etc/config
      volumes:
      - name: config-volume
        configMap:
          name: my-java-server-env-vars
```

3. Create the ConfigMap using `kustomize build`.
```bash
kustomize build .
```

The output is similar to:
```yaml
apiVersion: v1
kind: ConfigMap
metadata:
  # The name has been updated to include the suffix
  name: my-java-server-env-vars-k44mhd6h5f
data:
  JAVA_HOME: /opt/java/jdk
  JAVA_TOOL_OPTIONS: -agentlib:hprof
---
apiVersion: apps/v1
kind: Deployment
metadata:
  labels:
    app: test
  name: test-deployment
spec:
  selector:
    matchLabels:
      app: test
  template:
    metadata:
      labels:
        app: test
    spec:
      containers:
      - command:
        - /bin/sh
        - -c
        - ls /etc/config/
        image: registry.k8s.io/busybox
        name: container
        volumeMounts:
        - mountPath: /etc/config
          name: config-volume
      volumes:
      - configMap:
          # The name has been updated to include the
          # suffix matching the ConfigMap
          name: my-java-server-env-vars-k44mhd6h5f
        name: config-volume
```

[`configMapGenerator`]: /docs/reference/api/kustomization-file/configmapgenerator/
