// Copyright 2021 The Kubeswitch authors
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//      http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

package kubeconfigutil

import (
	"fmt"
	"io/ioutil"
	"os"
	"strings"

	"gopkg.in/yaml.v3"
)

const (
	// TemporaryKubeconfigDir is a constant for the directory where the switcher stores the temporary kubeconfig files
	TemporaryKubeconfigDir = "$HOME/.kube/.switch_tmp"
)

type Kubeconfig struct {
	path       string
	useTmpFile bool
	rootNode   *yaml.Node
}

// NewKubeconfigForPath creates a kubeconfig representation based on an existing kubeconfig
// given by the path argument
// This will overwrite the kubeconfig given by path when calling WriteKubeconfigFile()
func NewKubeconfigForPath(path string) (*Kubeconfig, error) {
	bytes, err := ioutil.ReadFile(path)
	if err != nil {
		return nil, fmt.Errorf("failed to read kubeconfig file: %v", err)
	}
	return newKubeconfig(bytes, path, false)
}

func NewKubeconfig(kubeconfigData []byte) (*Kubeconfig, error) {
	return newKubeconfig(kubeconfigData, os.ExpandEnv(TemporaryKubeconfigDir), true)
}

func newKubeconfig(kubeconfigData []byte, path string, useTmpFile bool) (*Kubeconfig, error) {
	n := &yaml.Node{}
	if err := yaml.Unmarshal(kubeconfigData, n); err != nil {
		return nil, err
	}

	k := &Kubeconfig{
		rootNode:   n.Content[0],
		path:       path,
		useTmpFile: useTmpFile,
	}

	if k.rootNode.Kind != yaml.MappingNode {
		return nil, fmt.Errorf("kubeconfig file does not have expected format")
	}
	return k, nil
}

// SetContext sets the given context as a current context on the kubeconfig
// if the given context is an alias, then it also modifies the kubeconfig so that both the current-context
// as well as the contexts.context name are set to the alias (otherwise the current-context
// would point to a non-existing context name)
func (k *Kubeconfig) SetContext(currentContext, originalContextBeforeAlias string, prefix string) error {
	if len(originalContextBeforeAlias) > 0 {
		// currentContext variable already has an alias
		// get the original currentContext name to find and replace it with the alias

		// can originalContextBeforeAlias  still contain the prefix?
		//  - yes if this store is configured with prefix (FIX: then need to remove prefix!)
		//  - no if store disabled prefix (works today)
		if len(prefix) > 0 && strings.HasPrefix(originalContextBeforeAlias, prefix) {
			originalContextBeforeAlias = strings.TrimPrefix(originalContextBeforeAlias, fmt.Sprintf("%s/", prefix))
		}

		if err := k.ModifyContextName(originalContextBeforeAlias, currentContext); err != nil {
			return fmt.Errorf("failed to set currentContext on selected kubeconfig: %v", err)
		}
	}

	if err := k.ModifyCurrentContext(currentContext); err != nil {
		return fmt.Errorf("failed to set current context on selected kubeconfig: %v", err)
	}
	return nil
}

func (k *Kubeconfig) SetKubeswitchContext(context string) error {
	if err := k.ModifyKubeswitchContext(context); err != nil {
		return fmt.Errorf("failed to set switch context on selected kubeconfig: %v", err)
	}
	return nil
}

func (k *Kubeconfig) SetNamespaceForCurrentContext(namespace string) error {
	currentContext := k.GetCurrentContext()
	if len(currentContext) == 0 {
		return fmt.Errorf("current-context is not set")
	}

	if err := k.SetNamespace(currentContext, namespace); err != nil {
		return fmt.Errorf("failed to set namespace %q: %v", namespace, err)
	}

	return nil
}

// WriteKubeconfigFile writes kubeconfig bytes to the local filesystem
// and returns the kubeconfig path
func (k *Kubeconfig) WriteKubeconfigFile() (string, error) {
	var (
		file *os.File
		err  error
	)
	// if we do not use a tmp file, then k.path is the path to a directory to create the tmp file in
	if k.useTmpFile {
		err = os.Mkdir(k.path, 0700)
		if err != nil && !os.IsExist(err) {
			return "", err
		}

		// write temporary kubeconfig file
		file, err = ioutil.TempFile(k.path, "config.*.tmp")
		if err != nil {
			return "", err
		}
	} else {
		file, err = os.OpenFile(k.path, os.O_CREATE|os.O_WRONLY|os.O_TRUNC, 0644)
		if err != nil {
			return "", fmt.Errorf("failed to open existing kubeconfig file: %v", err)
		}
	}

	enc := yaml.NewEncoder(file)
	enc.SetIndent(0)

	if err := enc.Encode(k.rootNode); err != nil {
		return "", err
	}

	return file.Name(), nil
}

func (k *Kubeconfig) GetBytes() ([]byte, error) {
	return yaml.Marshal(k.rootNode)
}
