package parquet

import "testing"

func TestUnpack8int64(t *testing.T) {
	for _, test := range unpack8int64Tests {
		unpacker := unpack8Int64FuncByWidth[test.width]
		if got := unpacker(test.data); got != test.values {
			t.Errorf("unpack for width %d: got %v, want %v", test.width, got, test.values)
		}
	}
}

var unpack8int64Tests = []struct {
	width  int
	data   []byte
	values [8]int64
}{
	// bit width = 0
	{
		0,
		[]byte{},
		[8]int64{0, 0, 0, 0, 0, 0, 0, 0},
	},

	// bit width = 1
	{
		1,
		[]byte{0x00},
		[8]int64{0, 0, 0, 0, 0, 0, 0, 0},
	},
	{
		1,
		[]byte{0xFF},
		[8]int64{1, 1, 1, 1, 1, 1, 1, 1},
	},
	{
		1,
		[]byte{0x4D},
		[8]int64{1, 0, 1, 1, 0, 0, 1, 0},
	},

	// bit width = 2
	{
		2,
		[]byte{0x55, 0x55},
		[8]int64{1, 1, 1, 1, 1, 1, 1, 1},
	},
	{
		2,
		[]byte{0xAA, 0xAA},
		[8]int64{2, 2, 2, 2, 2, 2, 2, 2},
	},
	{
		2,
		[]byte{0xA4, 0x41},
		[8]int64{0, 1, 2, 2, 1, 0, 0, 1},
	},

	// bit width = 3
	{
		3,
		[]byte{0x00, 0x00, 0x00},
		[8]int64{0, 0, 0, 0, 0, 0, 0, 0},
	},
	{
		3,
		[]byte{0x88, 0xC6, 0xFA},
		[8]int64{0, 1, 2, 3, 4, 5, 6, 7},
	},
	{
		3,
		[]byte{0x77, 0x39, 0x05},
		[8]int64{7, 6, 5, 4, 3, 2, 1, 0},
	},
	{
		3,
		[]byte{0x77, 0x39, 0x05},
		[8]int64{7, 6, 5, 4, 3, 2, 1, 0},
	},
	{
		3,
		[]byte{0x10, 0x31, 0x8E},
		[8]int64{0, 2, 4, 0, 3, 4, 3, 4},
	},

	// bit width = 4
	{
		4,
		[]byte{0x00, 0x00, 0x00, 0x00},
		[8]int64{0, 0, 0, 0, 0, 0, 0, 0},
	},
	{
		4,
		[]byte{0x10, 0x32, 0x54, 0x76},
		[8]int64{0, 1, 2, 3, 4, 5, 6, 7},
	},
	{
		4,
		[]byte{0x67, 0x45, 0x23, 0x01},
		[8]int64{7, 6, 5, 4, 3, 2, 1, 0},
	},
	{
		4,
		[]byte{0xEF, 0xCD, 0xAB, 0x89},
		[8]int64{15, 14, 13, 12, 11, 10, 9, 8},
	},
	{
		4,
		[]byte{0x38, 0x4D, 0x87, 0x80},
		[8]int64{8, 3, 13, 4, 7, 8, 0, 8},
	},

	// bit width = 5
	{
		5,
		[]byte{0x00, 0x00, 0x00, 0x00, 0x00},
		[8]int64{0, 0, 0, 0, 0, 0, 0, 0},
	},
	{
		5,
		[]byte{0x20, 0x88, 0x41, 0x8A, 0x39},
		[8]int64{0, 1, 2, 3, 4, 5, 6, 7},
	},
	{
		5,
		[]byte{0xC7, 0x14, 0x32, 0x44, 0x00},
		[8]int64{7, 6, 5, 4, 3, 2, 1, 0},
	},
	{
		5,
		[]byte{0xDF, 0x77, 0xBE, 0x75, 0xC6},
		[8]int64{31, 30, 29, 28, 27, 26, 25, 24},
	},
	{
		5,
		[]byte{0x08, 0xA8, 0x23, 0x63, 0x60},
		[8]int64{8, 0, 10, 7, 18, 17, 1, 12},
	},

	// bit width = 6
	{
		6,
		[]byte{0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
		[8]int64{0, 0, 0, 0, 0, 0, 0, 0},
	},
	{
		6,
		[]byte{0x40, 0x20, 0x0C, 0x44, 0x61, 0x1C},
		[8]int64{0, 1, 2, 3, 4, 5, 6, 7},
	},
	{
		6,
		[]byte{0x87, 0x51, 0x10, 0x83, 0x10, 0x00},
		[8]int64{7, 6, 5, 4, 3, 2, 1, 0},
	},
	{
		6,
		[]byte{0xBF, 0xDF, 0xF3, 0xBB, 0x9E, 0xE3},
		[8]int64{63, 62, 61, 60, 59, 58, 57, 56},
	},
	{
		6,
		[]byte{0xC8, 0x96, 0xE1, 0xE2, 0x19, 0x41},
		[8]int64{8, 27, 25, 56, 34, 39, 17, 16},
	},

	// bit width = 7
	{
		7,
		[]byte{0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
		[8]int64{0, 0, 0, 0, 0, 0, 0, 0},
	},
	{
		7,
		[]byte{0x80, 0x80, 0x60, 0x40, 0x28, 0x18, 0x0E},
		[8]int64{0, 1, 2, 3, 4, 5, 6, 7},
	},
	{
		7,
		[]byte{0x07, 0x43, 0x81, 0x30, 0x10, 0x04, 0x00},
		[8]int64{7, 6, 5, 4, 3, 2, 1, 0},
	},
	{
		7,
		[]byte{0x7F, 0x7F, 0x9F, 0xBF, 0xD7, 0xE7, 0xF1},
		[8]int64{127, 126, 125, 124, 123, 122, 121, 120},
	},
	{
		7,
		[]byte{0xE8, 0x63, 0xD7, 0x3D, 0x0A, 0x7D, 0x09},
		[8]int64{104, 71, 93, 110, 35, 33, 95, 4},
	},

	// bit width = 8
	{
		8,
		[]byte{0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
		[8]int64{0, 0, 0, 0, 0, 0, 0, 0},
	},
	{
		8,
		[]byte{0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07},
		[8]int64{0, 1, 2, 3, 4, 5, 6, 7},
	},
	{
		8,
		[]byte{0x07, 0x06, 0x05, 0x04, 0x03, 0x02, 0x01, 0x00},
		[8]int64{7, 6, 5, 4, 3, 2, 1, 0},
	},
	{
		8,
		[]byte{0xFF, 0xFE, 0xFD, 0xFC, 0xFB, 0xFA, 0xF9, 0xF8},
		[8]int64{255, 254, 253, 252, 251, 250, 249, 248},
	},
	{
		8,
		[]byte{0xC8, 0xA5, 0xED, 0x2A, 0xBF, 0x7C, 0xD3, 0x48},
		[8]int64{200, 165, 237, 42, 191, 124, 211, 72},
	},

	// bit width = 9
	{
		9,
		[]byte{0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
		[8]int64{0, 0, 0, 0, 0, 0, 0, 0},
	},
	{
		9,
		[]byte{0x00, 0x02, 0x08, 0x18, 0x40, 0xA0, 0x80, 0x81, 0x03},
		[8]int64{0, 1, 2, 3, 4, 5, 6, 7},
	},
	{
		9,
		[]byte{0x07, 0x0C, 0x14, 0x20, 0x30, 0x40, 0x40, 0x00, 0x00},
		[8]int64{7, 6, 5, 4, 3, 2, 1, 0},
	},
	{
		9,
		[]byte{0xFF, 0xFD, 0xF7, 0xE7, 0xBF, 0x5F, 0x7F, 0x7E, 0xFC},
		[8]int64{511, 510, 509, 508, 507, 506, 505, 504},
	},
	{
		9,
		[]byte{0x28, 0xA2, 0x77, 0x85, 0x5A, 0x9C, 0x09, 0x3F, 0x5E},
		[8]int64{40, 465, 349, 336, 453, 76, 252, 188},
	},

	// bit width = 10
	{
		10,
		[]byte{0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
		[8]int64{0, 0, 0, 0, 0, 0, 0, 0},
	},
	{
		10,
		[]byte{0x00, 0x04, 0x20, 0xC0, 0x00, 0x04, 0x14, 0x60, 0xC0, 0x01},
		[8]int64{0, 1, 2, 3, 4, 5, 6, 7},
	},
	{
		10,
		[]byte{0x07, 0x18, 0x50, 0x00, 0x01, 0x03, 0x08, 0x10, 0x00, 0x00},
		[8]int64{7, 6, 5, 4, 3, 2, 1, 0},
	},
	{
		10,
		[]byte{0xFF, 0xFB, 0xDF, 0x3F, 0xFF, 0xFB, 0xEB, 0x9F, 0x3F, 0xFE},
		[8]int64{1023, 1022, 1021, 1020, 1019, 1018, 1017, 1016},
	},
	{
		10,
		[]byte{0x08, 0xD1, 0x06, 0x48, 0x88, 0xB4, 0xDA, 0xFB, 0x18, 0xD8},
		[8]int64{264, 436, 128, 545, 692, 758, 399, 864},
	},

	// bit width = 11
	{
		11,
		[]byte{0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
		[8]int64{0, 0, 0, 0, 0, 0, 0, 0},
	},
	{
		11,
		[]byte{0x00, 0x08, 0x80, 0x00, 0x06, 0x40, 0x80, 0x02, 0x18, 0xE0, 0x00},
		[8]int64{0, 1, 2, 3, 4, 5, 6, 7},
	},
	{
		11,
		[]byte{0x07, 0x30, 0x40, 0x01, 0x08, 0x30, 0x00, 0x01, 0x04, 0x00, 0x00},
		[8]int64{7, 6, 5, 4, 3, 2, 1, 0},
	},
	{
		11,
		[]byte{0xFF, 0xF7, 0x7F, 0xFF, 0xF9, 0xBF, 0x7F, 0xFD, 0xE7, 0x1F, 0xFF},
		[8]int64{2047, 2046, 2045, 2044, 2043, 2042, 2041, 2040},
	},
	{
		11,
		[]byte{0x68, 0x3D, 0xB4, 0xB2, 0xF6, 0x75, 0xE4, 0x5E, 0x8B, 0x82, 0x7A},
		[8]int64{1384, 1671, 714, 763, 1607, 1725, 162, 980},
	},

	// bit width = 12
	{
		12,
		[]byte{0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
		[8]int64{0, 0, 0, 0, 0, 0, 0, 0},
	},
	{
		12,
		[]byte{0x00, 0x10, 0x00, 0x02, 0x30, 0x00, 0x04, 0x50, 0x00, 0x06, 0x70, 0x00},
		[8]int64{0, 1, 2, 3, 4, 5, 6, 7},
	},
	{
		12,
		[]byte{0x07, 0x60, 0x00, 0x05, 0x40, 0x00, 0x03, 0x20, 0x00, 0x01, 0x00, 0x00},
		[8]int64{7, 6, 5, 4, 3, 2, 1, 0},
	},
	{
		12,
		[]byte{0xFF, 0xEF, 0xFF, 0xFD, 0xCF, 0xFF, 0xFB, 0xAF, 0xFF, 0xF9, 0x8F, 0xFF},
		[8]int64{4095, 4094, 4093, 4092, 4091, 4090, 4089, 4088},
	},
	{
		12,
		[]byte{0x48, 0xA3, 0xAA, 0xC6, 0x18, 0x0C, 0x2B, 0x13, 0xDA, 0x96, 0x83, 0x9E},
		[8]int64{840, 2730, 2246, 193, 811, 3489, 918, 2536},
	},

	// bit width = 13
	{
		13,
		[]byte{0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
		[8]int64{0, 0, 0, 0, 0, 0, 0, 0},
	},
	{
		13,
		[]byte{0x00, 0x20, 0x00, 0x08, 0x80, 0x01, 0x40, 0x00, 0x0A, 0x80, 0x01, 0x38, 0x00},
		[8]int64{0, 1, 2, 3, 4, 5, 6, 7},
	},
	{
		13,
		[]byte{0x07, 0xC0, 0x00, 0x14, 0x00, 0x02, 0x30, 0x00, 0x04, 0x40, 0x00, 0x00, 0x00},
		[8]int64{7, 6, 5, 4, 3, 2, 1, 0},
	},
	{
		13,
		[]byte{0xFF, 0xDF, 0xFF, 0xF7, 0x7F, 0xFE, 0xBF, 0xFF, 0xF5, 0x7F, 0xFE, 0xC7, 0xFF},
		[8]int64{8191, 8190, 8189, 8188, 8187, 8186, 8185, 8184},
	},
	{
		13,
		[]byte{0xA8, 0xB7, 0x6F, 0xF0, 0x05, 0xF3, 0x57, 0x42, 0xD1, 0xAA, 0x8E, 0xA2, 0x4B},
		[8]int64{6056, 893, 380, 4070, 5157, 5480, 2618, 2420},
	},

	// bit width = 14
	{
		14,
		[]byte{0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
		[8]int64{0, 0, 0, 0, 0, 0, 0, 0},
	},
	{
		14,
		[]byte{0x00, 0x40, 0x00, 0x20, 0x00, 0x0C, 0x00, 0x04, 0x40, 0x01, 0x60, 0x00, 0x1C, 0x00},
		[8]int64{0, 1, 2, 3, 4, 5, 6, 7},
	},
	{
		14,
		[]byte{0x07, 0x80, 0x01, 0x50, 0x00, 0x10, 0x00, 0x03, 0x80, 0x00, 0x10, 0x00, 0x00, 0x00},
		[8]int64{7, 6, 5, 4, 3, 2, 1, 0},
	},
	{
		14,
		[]byte{0xFF, 0xBF, 0xFF, 0xDF, 0xFF, 0xF3, 0xFF, 0xFB, 0xBF, 0xFE, 0x9F, 0xFF, 0xE3, 0xFF},
		[8]int64{16383, 16382, 16381, 16380, 16379, 16378, 16377, 16376},
	},
	{
		14,
		[]byte{0x88, 0xBA, 0xBF, 0x9D, 0x75, 0x2C, 0xB6, 0x36, 0xF1, 0x84, 0x4F, 0xF3, 0xC1, 0xD4},
		[8]int64{14984, 14078, 1881, 11659, 12598, 15891, 7988, 13616},
	},

	// bit width = 15
	{
		15,
		[]byte{0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
		[8]int64{0, 0, 0, 0, 0, 0, 0, 0},
	},
	{
		15,
		[]byte{0x00, 0x80, 0x00, 0x80, 0x00, 0x60, 0x00, 0x40, 0x00, 0x28, 0x00, 0x18, 0x00, 0x0E, 0x00},
		[8]int64{0, 1, 2, 3, 4, 5, 6, 7},
	},
	{
		15,
		[]byte{0x07, 0x00, 0x03, 0x40, 0x01, 0x80, 0x00, 0x30, 0x00, 0x10, 0x00, 0x04, 0x00, 0x00, 0x00},
		[8]int64{7, 6, 5, 4, 3, 2, 1, 0},
	},
	{
		15,
		[]byte{0xFF, 0x7F, 0xFF, 0x7F, 0xFF, 0x9F, 0xFF, 0xBF, 0xFF, 0xD7, 0xFF, 0xE7, 0xFF, 0xF1, 0xFF},
		[8]int64{32767, 32766, 32765, 32764, 32763, 32762, 32761, 32760},
	},
	{
		15,
		[]byte{0xE8, 0x6E, 0xEB, 0x28, 0xFC, 0xBB, 0xCC, 0x09, 0xDD, 0x32, 0xD1, 0x6F, 0xBA, 0xB9, 0xD4},
		[8]int64{28392, 20950, 28656, 20069, 11728, 31270, 28315, 27228},
	},

	// bit width = 16
	{
		16,
		[]byte{0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
		[8]int64{0, 0, 0, 0, 0, 0, 0, 0},
	},
	{
		16,
		[]byte{0x00, 0x00, 0x01, 0x00, 0x02, 0x00, 0x03, 0x00, 0x04, 0x00, 0x05, 0x00, 0x06, 0x00, 0x07, 0x00},
		[8]int64{0, 1, 2, 3, 4, 5, 6, 7},
	},
	{
		16,
		[]byte{0x07, 0x00, 0x06, 0x00, 0x05, 0x00, 0x04, 0x00, 0x03, 0x00, 0x02, 0x00, 0x01, 0x00, 0x00, 0x00},
		[8]int64{7, 6, 5, 4, 3, 2, 1, 0},
	},
	{
		16,
		[]byte{0xFF, 0xFF, 0xFE, 0xFF, 0xFD, 0xFF, 0xFC, 0xFF, 0xFB, 0xFF, 0xFA, 0xFF, 0xF9, 0xFF, 0xF8, 0xFF},
		[8]int64{65535, 65534, 65533, 65532, 65531, 65530, 65529, 65528},
	},
	{
		16,
		[]byte{0xC7, 0x68, 0x7C, 0x1C, 0x2C, 0x9B, 0x5D, 0x17, 0x3A, 0x68, 0x28, 0x8C, 0x42, 0xC6, 0x68, 0xE9},
		[8]int64{26823, 7292, 39724, 5981, 26682, 35880, 50754, 59752},
	},

	// bit width = 17
	{
		17,
		[]byte{0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
		[8]int64{0, 0, 0, 0, 0, 0, 0, 0},
	},
	{
		17,
		[]byte{0x00, 0x00, 0x02, 0x00, 0x08, 0x00, 0x18, 0x00, 0x40, 0x00, 0xA0, 0x00, 0x80, 0x01, 0x80, 0x03, 0x00},
		[8]int64{0, 1, 2, 3, 4, 5, 6, 7},
	},
	{
		17,
		[]byte{0x07, 0x00, 0x0C, 0x00, 0x14, 0x00, 0x20, 0x00, 0x30, 0x00, 0x40, 0x00, 0x40, 0x00, 0x00, 0x00, 0x00},
		[8]int64{7, 6, 5, 4, 3, 2, 1, 0},
	},
	{
		17,
		[]byte{0xFF, 0xFF, 0xFD, 0xFF, 0xF7, 0xFF, 0xE7, 0xFF, 0xBF, 0xFF, 0x5F, 0xFF, 0x7F, 0xFE, 0x7F, 0xFC, 0xFF},
		[8]int64{131071, 131070, 131069, 131068, 131067, 131066, 131065, 131064},
	},
	{
		17,
		[]byte{0x27, 0x1D, 0x50, 0x27, 0x2D, 0xB6, 0x48, 0x3E, 0x14, 0x8C, 0xEF, 0x73, 0x67, 0xE3, 0x41, 0xEE, 0x24},
		[8]int64{7463, 37800, 11659, 34761, 63681, 80799, 67469, 18908},
	},

	// bit width = 18
	{
		18,
		[]byte{0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
		[8]int64{0, 0, 0, 0, 0, 0, 0, 0},
	},
	{
		18,
		[]byte{0x00, 0x00, 0x04, 0x00, 0x20, 0x00, 0xC0, 0x00, 0x00, 0x04, 0x00, 0x14, 0x00, 0x60, 0x00, 0xC0, 0x01, 0x00},
		[8]int64{0, 1, 2, 3, 4, 5, 6, 7},
	},
	{
		18,
		[]byte{0x07, 0x00, 0x18, 0x00, 0x50, 0x00, 0x00, 0x01, 0x00, 0x03, 0x00, 0x08, 0x00, 0x10, 0x00, 0x00, 0x00, 0x00},
		[8]int64{7, 6, 5, 4, 3, 2, 1, 0},
	},
	{
		18,
		[]byte{0xFF, 0xFF, 0xFB, 0xFF, 0xDF, 0xFF, 0x3F, 0xFF, 0xFF, 0xFB, 0xFF, 0xEB, 0xFF, 0x9F, 0xFF, 0x3F, 0xFE, 0xFF},
		[8]int64{262143, 262142, 262141, 262140, 262139, 262138, 262137, 262136},
	},
	{
		18,
		[]byte{0x07, 0x00, 0x76, 0x47, 0x8F, 0x36, 0x3D, 0x6A, 0xD5, 0x2F, 0x2A, 0x28, 0x91, 0xF7, 0xE1, 0x39, 0x20, 0xCA},
		[8]int64{131079, 250333, 250728, 218536, 10799, 123978, 237087, 206976},
	},

	// bit width = 19
	{
		19,
		[]byte{0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
		[8]int64{0, 0, 0, 0, 0, 0, 0, 0},
	},
	{
		19,
		[]byte{0x00, 0x00, 0x08, 0x00, 0x80, 0x00, 0x00, 0x06, 0x00, 0x40, 0x00, 0x80, 0x02, 0x00, 0x18, 0x00, 0xE0, 0x00, 0x00},
		[8]int64{0, 1, 2, 3, 4, 5, 6, 7},
	},
	{
		19,
		[]byte{0x07, 0x00, 0x30, 0x00, 0x40, 0x01, 0x00, 0x08, 0x00, 0x30, 0x00, 0x00, 0x01, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00},
		[8]int64{7, 6, 5, 4, 3, 2, 1, 0},
	},
	{
		19,
		[]byte{0xFF, 0xFF, 0xF7, 0xFF, 0x7F, 0xFF, 0xFF, 0xF9, 0xFF, 0xBF, 0xFF, 0x7F, 0xFD, 0xFF, 0xE7, 0xFF, 0x1F, 0xFF, 0xFF},
		[8]int64{524287, 524286, 524285, 524284, 524283, 524282, 524281, 524280},
	},
	{
		19,
		[]byte{0x67, 0xB4, 0x82, 0xD5, 0xA2, 0xF8, 0xD8, 0x05, 0x6D, 0xCD, 0x6C, 0xC3, 0xF7, 0x29, 0xBD, 0xAF, 0x95, 0x1E, 0xB1},
		[8]int64{177255, 285360, 484322, 439938, 276172, 152559, 355311, 362740},
	},

	// bit width = 20
	{
		20,
		[]byte{0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
		[8]int64{0, 0, 0, 0, 0, 0, 0, 0},
	},
	{
		20,
		[]byte{0x00, 0x00, 0x10, 0x00, 0x00, 0x02, 0x00, 0x30, 0x00, 0x00, 0x04, 0x00, 0x50, 0x00, 0x00, 0x06, 0x00, 0x70, 0x00, 0x00},
		[8]int64{0, 1, 2, 3, 4, 5, 6, 7},
	},
	{
		20,
		[]byte{0x07, 0x00, 0x60, 0x00, 0x00, 0x05, 0x00, 0x40, 0x00, 0x00, 0x03, 0x00, 0x20, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00},
		[8]int64{7, 6, 5, 4, 3, 2, 1, 0},
	},
	{
		20,
		[]byte{0xFF, 0xFF, 0xEF, 0xFF, 0xFF, 0xFD, 0xFF, 0xCF, 0xFF, 0xFF, 0xFB, 0xFF, 0xAF, 0xFF, 0xFF, 0xF9, 0xFF, 0x8F, 0xFF, 0xFF},
		[8]int64{1048575, 1048574, 1048573, 1048572, 1048571, 1048570, 1048569, 1048568},
	},
	{
		20,
		[]byte{0x47, 0x3A, 0x3E, 0x6D, 0x30, 0x52, 0x78, 0x6B, 0xAC, 0xB9, 0xB0, 0x1B, 0x5E, 0x2F, 0x89, 0x05, 0x8F, 0x85, 0xB3, 0x96},
		[8]int64{932423, 198355, 751698, 760518, 924592, 561909, 364293, 617272},
	},

	// bit width = 21
	{
		21,
		[]byte{0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
		[8]int64{0, 0, 0, 0, 0, 0, 0, 0},
	},
	{
		21,
		[]byte{0x00, 0x00, 0x20, 0x00, 0x00, 0x08, 0x00, 0x80, 0x01, 0x00, 0x40, 0x00, 0x00, 0x0A, 0x00, 0x80, 0x01, 0x00, 0x38, 0x00, 0x00},
		[8]int64{0, 1, 2, 3, 4, 5, 6, 7},
	},
	{
		21,
		[]byte{0x07, 0x00, 0xC0, 0x00, 0x00, 0x14, 0x00, 0x00, 0x02, 0x00, 0x30, 0x00, 0x00, 0x04, 0x00, 0x40, 0x00, 0x00, 0x00, 0x00, 0x00},
		[8]int64{7, 6, 5, 4, 3, 2, 1, 0},
	},
	{
		21,
		[]byte{0xFF, 0xFF, 0xDF, 0xFF, 0xFF, 0xF7, 0xFF, 0x7F, 0xFE, 0xFF, 0xBF, 0xFF, 0xFF, 0xF5, 0xFF, 0x7F, 0xFE, 0xFF, 0xC7, 0xFF, 0xFF},
		[8]int64{2097151, 2097150, 2097149, 2097148, 2097147, 2097146, 2097145, 2097144},
	},
	{
		21,
		[]byte{0xA7, 0xEE, 0xDE, 0xF4, 0xF1, 0x92, 0x41, 0xF5, 0x2F, 0x82, 0x56, 0x35, 0x18, 0x89, 0x4A, 0xA7, 0x4D, 0xF0, 0xA7, 0xA2, 0xA1},
		[8]int64{2027175, 1544102, 1921124, 853087, 1147733, 1287492, 2081078, 1324116},
	},

	// bit width = 22
	{
		22,
		[]byte{0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
		[8]int64{0, 0, 0, 0, 0, 0, 0, 0},
	},
	{
		22,
		[]byte{0x00, 0x00, 0x40, 0x00, 0x00, 0x20, 0x00, 0x00, 0x0C, 0x00, 0x00, 0x04, 0x00, 0x40, 0x01, 0x00, 0x60, 0x00, 0x00, 0x1C, 0x00, 0x00},
		[8]int64{0, 1, 2, 3, 4, 5, 6, 7},
	},
	{
		22,
		[]byte{0x07, 0x00, 0x80, 0x01, 0x00, 0x50, 0x00, 0x00, 0x10, 0x00, 0x00, 0x03, 0x00, 0x80, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x00, 0x00},
		[8]int64{7, 6, 5, 4, 3, 2, 1, 0},
	},
	{
		22,
		[]byte{0xFF, 0xFF, 0xBF, 0xFF, 0xFF, 0xDF, 0xFF, 0xFF, 0xF3, 0xFF, 0xFF, 0xFB, 0xFF, 0xBF, 0xFE, 0xFF, 0x9F, 0xFF, 0xFF, 0xE3, 0xFF, 0xFF},
		[8]int64{4194303, 4194302, 4194301, 4194300, 4194299, 4194298, 4194297, 4194296},
	},
	{
		22,
		[]byte{0x87, 0xD1, 0x5C, 0xB5, 0x82, 0xF9, 0x9B, 0xD0, 0xB9, 0x9B, 0x4C, 0xBB, 0x59, 0x8D, 0x26, 0xBF, 0x4F, 0xAA, 0x12, 0x42, 0xA9, 0x12},
		[8]int64{1888647, 2493141, 1903039, 1255150, 874939, 4127898, 2173604, 305744},
	},

	// bit width = 23
	{
		23,
		[]byte{0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
		[8]int64{0, 0, 0, 0, 0, 0, 0, 0},
	},
	{
		23,
		[]byte{0x00, 0x00, 0x80, 0x00, 0x00, 0x80, 0x00, 0x00, 0x60, 0x00, 0x00, 0x40, 0x00, 0x00, 0x28, 0x00, 0x00, 0x18, 0x00, 0x00, 0x0E, 0x00, 0x00},
		[8]int64{0, 1, 2, 3, 4, 5, 6, 7},
	},
	{
		23,
		[]byte{0x07, 0x00, 0x00, 0x03, 0x00, 0x40, 0x01, 0x00, 0x80, 0x00, 0x00, 0x30, 0x00, 0x00, 0x10, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00},
		[8]int64{7, 6, 5, 4, 3, 2, 1, 0},
	},
	{
		23,
		[]byte{0xFF, 0xFF, 0x7F, 0xFF, 0xFF, 0x7F, 0xFF, 0xFF, 0x9F, 0xFF, 0xFF, 0xBF, 0xFF, 0xFF, 0xD7, 0xFF, 0xFF, 0xE7, 0xFF, 0xFF, 0xF1, 0xFF, 0xFF},
		[8]int64{8388607, 8388606, 8388605, 8388604, 8388603, 8388602, 8388601, 8388600},
	},
	{
		23,
		[]byte{0xE7, 0x85, 0xFD, 0xFF, 0x3E, 0xD7, 0xBD, 0xEF, 0x82, 0x3D, 0x87, 0xA4, 0x54, 0x24, 0x95, 0xA6, 0xD9, 0xB2, 0x8C, 0xD2, 0x89, 0x15, 0x0E},
		[8]int64{8226279, 3046911, 769783, 2374124, 5391690, 5977298, 7643948, 461508},
	},

	// bit width = 24
	{
		24,
		[]byte{0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
		[8]int64{0, 0, 0, 0, 0, 0, 0, 0},
	},
	{
		24,
		[]byte{0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x02, 0x00, 0x00, 0x03, 0x00, 0x00, 0x04, 0x00, 0x00, 0x05, 0x00, 0x00, 0x06, 0x00, 0x00, 0x07, 0x00, 0x00},
		[8]int64{0, 1, 2, 3, 4, 5, 6, 7},
	},
	{
		24,
		[]byte{0x07, 0x00, 0x00, 0x06, 0x00, 0x00, 0x05, 0x00, 0x00, 0x04, 0x00, 0x00, 0x03, 0x00, 0x00, 0x02, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00},
		[8]int64{7, 6, 5, 4, 3, 2, 1, 0},
	},
	{
		24,
		[]byte{0xFF, 0xFF, 0xFF, 0xFE, 0xFF, 0xFF, 0xFD, 0xFF, 0xFF, 0xFC, 0xFF, 0xFF, 0xFB, 0xFF, 0xFF, 0xFA, 0xFF, 0xFF, 0xF9, 0xFF, 0xFF, 0xF8, 0xFF, 0xFF},
		[8]int64{16777215, 16777214, 16777213, 16777212, 16777211, 16777210, 16777209, 16777208},
	},
	{
		24,
		[]byte{0xC8, 0x0B, 0x21, 0xDF, 0xD5, 0x6A, 0x87, 0xAA, 0xD7, 0x30, 0x1E, 0x39, 0x66, 0x60, 0xF0, 0xEF, 0x4D, 0x68, 0x38, 0xA8, 0x61, 0xF8, 0x12, 0xAD},
		[8]int64{2165704, 7001567, 14133895, 3743280, 15753318, 6835695, 6400056, 11342584},
	},

	// bit width = 25
	{
		25,
		[]byte{0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
		[8]int64{0, 0, 0, 0, 0, 0, 0, 0},
	},
	{
		25,
		[]byte{0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x08, 0x00, 0x00, 0x18, 0x00, 0x00, 0x40, 0x00, 0x00, 0xA0, 0x00, 0x00, 0x80, 0x01, 0x00, 0x80, 0x03, 0x00, 0x00},
		[8]int64{0, 1, 2, 3, 4, 5, 6, 7},
	},
	{
		25,
		[]byte{0x07, 0x00, 0x00, 0x0C, 0x00, 0x00, 0x14, 0x00, 0x00, 0x20, 0x00, 0x00, 0x30, 0x00, 0x00, 0x40, 0x00, 0x00, 0x40, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
		[8]int64{7, 6, 5, 4, 3, 2, 1, 0},
	},
	{
		25,
		[]byte{0xFF, 0xFF, 0xFF, 0xFD, 0xFF, 0xFF, 0xF7, 0xFF, 0xFF, 0xE7, 0xFF, 0xFF, 0xBF, 0xFF, 0xFF, 0x5F, 0xFF, 0xFF, 0x7F, 0xFE, 0xFF, 0x7F, 0xFC, 0xFF, 0xFF},
		[8]int64{33554431, 33554430, 33554429, 33554428, 33554427, 33554426, 33554425, 33554424},
	},
	{
		25,
		[]byte{0x28, 0xC0, 0x41, 0xEB, 0x65, 0x53, 0x0F, 0x33, 0xFE, 0x57, 0xF8, 0x83, 0x47, 0x09, 0x00, 0xEF, 0xC4, 0x90, 0x00, 0x7E, 0x22, 0x62, 0x42, 0x59, 0xF5},
		[8]int64{21086248, 27898613, 33524931, 15761162, 15728788, 296487, 25725432, 32158340},
	},

	// bit width = 26
	{
		26,
		[]byte{0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
		[8]int64{0, 0, 0, 0, 0, 0, 0, 0},
	},
	{
		26,
		[]byte{0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x20, 0x00, 0x00, 0xC0, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x14, 0x00, 0x00, 0x60, 0x00, 0x00, 0xC0, 0x01, 0x00, 0x00},
		[8]int64{0, 1, 2, 3, 4, 5, 6, 7},
	},
	{
		26,
		[]byte{0x07, 0x00, 0x00, 0x18, 0x00, 0x00, 0x50, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x03, 0x00, 0x00, 0x08, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
		[8]int64{7, 6, 5, 4, 3, 2, 1, 0},
	},
	{
		26,
		[]byte{0xFF, 0xFF, 0xFF, 0xFB, 0xFF, 0xFF, 0xDF, 0xFF, 0xFF, 0x3F, 0xFF, 0xFF, 0xFF, 0xFB, 0xFF, 0xFF, 0xEB, 0xFF, 0xFF, 0x9F, 0xFF, 0xFF, 0x3F, 0xFE, 0xFF, 0xFF},
		[8]int64{67108863, 67108862, 67108861, 67108860, 67108859, 67108858, 67108857, 67108856},
	},
	{
		26,
		[]byte{0x07, 0xA3, 0xDF, 0x1B, 0x9E, 0x4F, 0x6E, 0x6E, 0x9E, 0x67, 0xE1, 0x68, 0x4D, 0x5B, 0x22, 0xB1, 0xF6, 0x75, 0xBF, 0xAC, 0xC9, 0x60, 0x23, 0xF8, 0x74, 0x4C},
		[8]int64{64987911, 60024710, 41543398, 20292485, 45163099, 53468541, 37096602, 20042720},
	},

	// bit width = 27
	{
		27,
		[]byte{0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
		[8]int64{0, 0, 0, 0, 0, 0, 0, 0},
	},
	{
		27,
		[]byte{0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x80, 0x00, 0x00, 0x00, 0x06, 0x00, 0x00, 0x40, 0x00, 0x00, 0x80, 0x02, 0x00, 0x00, 0x18, 0x00, 0x00, 0xE0, 0x00, 0x00, 0x00},
		[8]int64{0, 1, 2, 3, 4, 5, 6, 7},
	},
	{
		27,
		[]byte{0x07, 0x00, 0x00, 0x30, 0x00, 0x00, 0x40, 0x01, 0x00, 0x00, 0x08, 0x00, 0x00, 0x30, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
		[8]int64{7, 6, 5, 4, 3, 2, 1, 0},
	},
	{
		27,
		[]byte{0xFF, 0xFF, 0xFF, 0xF7, 0xFF, 0xFF, 0x7F, 0xFF, 0xFF, 0xFF, 0xF9, 0xFF, 0xFF, 0xBF, 0xFF, 0xFF, 0x7F, 0xFD, 0xFF, 0xFF, 0xE7, 0xFF, 0xFF, 0x1F, 0xFF, 0xFF, 0xFF},
		[8]int64{134217727, 134217726, 134217725, 134217724, 134217723, 134217722, 134217721, 134217720},
	},
	{
		27,
		[]byte{0x67, 0x57, 0x00, 0x9F, 0xF5, 0xFA, 0xD3, 0x8B, 0x47, 0x14, 0x57, 0x85, 0xFC, 0xF4, 0xE9, 0xF3, 0x32, 0x25, 0xF5, 0x99, 0x5F, 0x96, 0xD0, 0x86, 0x6D, 0xEE, 0xED},
		[8]int64{117462887, 41901747, 72425007, 41829035, 53427871, 120842826, 28583319, 124744556},
	},

	// bit width = 28
	{
		28,
		[]byte{0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
		[8]int64{0, 0, 0, 0, 0, 0, 0, 0},
	},
	{
		28,
		[]byte{0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x30, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x50, 0x00, 0x00, 0x00, 0x06, 0x00, 0x00, 0x70, 0x00, 0x00, 0x00},
		[8]int64{0, 1, 2, 3, 4, 5, 6, 7},
	},
	{
		28,
		[]byte{0x07, 0x00, 0x00, 0x60, 0x00, 0x00, 0x00, 0x05, 0x00, 0x00, 0x40, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x20, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
		[8]int64{7, 6, 5, 4, 3, 2, 1, 0},
	},
	{
		28,
		[]byte{0xFF, 0xFF, 0xFF, 0xEF, 0xFF, 0xFF, 0xFF, 0xFD, 0xFF, 0xFF, 0xCF, 0xFF, 0xFF, 0xFF, 0xFB, 0xFF, 0xFF, 0xAF, 0xFF, 0xFF, 0xFF, 0xF9, 0xFF, 0xFF, 0x8F, 0xFF, 0xFF, 0xFF},
		[8]int64{268435455, 268435454, 268435453, 268435452, 268435451, 268435450, 268435449, 268435448},
	},
	{
		28,
		[]byte{0x48, 0xDD, 0xA3, 0x0E, 0x29, 0x8B, 0x91, 0x61, 0xCD, 0x7A, 0x96, 0x19, 0x6A, 0x31, 0x84, 0x23, 0x91, 0x20, 0xED, 0x92, 0x8D, 0x8B, 0x20, 0x47, 0x8D, 0x12, 0x39, 0x2A},
		[8]int64{245620040, 152613520, 108711265, 51814809, 9511812, 148451026, 222765195, 44273960},
	},

	// bit width = 29
	{
		29,
		[]byte{0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
		[8]int64{0, 0, 0, 0, 0, 0, 0, 0},
	},
	{
		29,
		[]byte{0x00, 0x00, 0x00, 0x20, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x80, 0x01, 0x00, 0x00, 0x40, 0x00, 0x00, 0x00, 0x0A, 0x00, 0x00, 0x80, 0x01, 0x00, 0x00, 0x38, 0x00, 0x00, 0x00},
		[8]int64{0, 1, 2, 3, 4, 5, 6, 7},
	},
	{
		29,
		[]byte{0x07, 0x00, 0x00, 0xC0, 0x00, 0x00, 0x00, 0x14, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x30, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x40, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
		[8]int64{7, 6, 5, 4, 3, 2, 1, 0},
	},
	{
		29,
		[]byte{0xFF, 0xFF, 0xFF, 0xDF, 0xFF, 0xFF, 0xFF, 0xF7, 0xFF, 0xFF, 0x7F, 0xFE, 0xFF, 0xFF, 0xBF, 0xFF, 0xFF, 0xFF, 0xF5, 0xFF, 0xFF, 0x7F, 0xFE, 0xFF, 0xFF, 0xC7, 0xFF, 0xFF, 0xFF},
		[8]int64{536870911, 536870910, 536870909, 536870908, 536870907, 536870906, 536870905, 536870904},
	},
	{
		29,
		[]byte{0xA8, 0x91, 0xC4, 0xB9, 0x37, 0x85, 0x80, 0xA5, 0xA7, 0x71, 0x7D, 0x3A, 0x01, 0xE7, 0xEE, 0x47, 0x18, 0xE7, 0xEB, 0x31, 0x15, 0x5B, 0x51, 0xAB, 0xB3, 0xA1, 0x85, 0x09, 0x27},
		[8]int64{432312744, 201599421, 526150121, 500040308, 510755966, 227186933, 114208069, 81866932},
	},

	// bit width = 30
	{
		30,
		[]byte{0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
		[8]int64{0, 0, 0, 0, 0, 0, 0, 0},
	},
	{
		30,
		[]byte{0x00, 0x00, 0x00, 0x40, 0x00, 0x00, 0x00, 0x20, 0x00, 0x00, 0x00, 0x0C, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x40, 0x01, 0x00, 0x00, 0x60, 0x00, 0x00, 0x00, 0x1C, 0x00, 0x00, 0x00},
		[8]int64{0, 1, 2, 3, 4, 5, 6, 7},
	},
	{
		30,
		[]byte{0x07, 0x00, 0x00, 0x80, 0x01, 0x00, 0x00, 0x50, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x80, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
		[8]int64{7, 6, 5, 4, 3, 2, 1, 0},
	},
	{
		30,
		[]byte{0xFF, 0xFF, 0xFF, 0xBF, 0xFF, 0xFF, 0xFF, 0xDF, 0xFF, 0xFF, 0xFF, 0xF3, 0xFF, 0xFF, 0xFF, 0xFB, 0xFF, 0xFF, 0xBF, 0xFE, 0xFF, 0xFF, 0x9F, 0xFF, 0xFF, 0xFF, 0xE3, 0xFF, 0xFF, 0xFF},
		[8]int64{1073741823, 1073741822, 1073741821, 1073741820, 1073741819, 1073741818, 1073741817, 1073741816},
	},
	{
		30,
		[]byte{0x88, 0x74, 0x62, 0x08, 0xF2, 0x8E, 0x2F, 0x43, 0xBF, 0x93, 0x22, 0x6F, 0x80, 0x60, 0x89, 0x8F, 0x61, 0xD0, 0x34, 0x68, 0xD0, 0x97, 0x76, 0x3D, 0xD4, 0xA8, 0x42, 0x48, 0xA1, 0x38},
		[8]int64{140670088, 213793736, 841563124, 576200731, 886071695, 442450336, 713901015, 237523472},
	},

	// bit width = 31
	{
		31,
		[]byte{0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
		[8]int64{0, 0, 0, 0, 0, 0, 0, 0},
	},
	{
		31,
		[]byte{0x00, 0x00, 0x00, 0x80, 0x00, 0x00, 0x00, 0x80, 0x00, 0x00, 0x00, 0x60, 0x00, 0x00, 0x00, 0x40, 0x00, 0x00, 0x00, 0x28, 0x00, 0x00, 0x00, 0x18, 0x00, 0x00, 0x00, 0x0E, 0x00, 0x00, 0x00},
		[8]int64{0, 1, 2, 3, 4, 5, 6, 7},
	},
	{
		31,
		[]byte{0x07, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x40, 0x01, 0x00, 0x00, 0x80, 0x00, 0x00, 0x00, 0x30, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
		[8]int64{7, 6, 5, 4, 3, 2, 1, 0},
	},
	{
		31,
		[]byte{0xFF, 0xFF, 0xFF, 0x7F, 0xFF, 0xFF, 0xFF, 0x7F, 0xFF, 0xFF, 0xFF, 0x9F, 0xFF, 0xFF, 0xFF, 0xBF, 0xFF, 0xFF, 0xFF, 0xD7, 0xFF, 0xFF, 0xFF, 0xE7, 0xFF, 0xFF, 0xFF, 0xF1, 0xFF, 0xFF, 0xFF},
		[8]int64{2147483647, 2147483646, 2147483645, 2147483644, 2147483643, 2147483642, 2147483641, 2147483640},
	},
	{
		31,
		[]byte{0xE8, 0x28, 0x83, 0x03, 0x4E, 0x62, 0xE9, 0x8C, 0xD5, 0x7E, 0xEB, 0x34, 0xE8, 0x17, 0xF4, 0x72, 0xD7, 0xF3, 0xD4, 0xCD, 0xB4, 0xE7, 0xF9, 0x69, 0xB9, 0x73, 0x85, 0x39, 0xE3, 0xCB, 0x40},
		[8]int64{58927336, 433243292, 1403910998, 396410689, 1565474167, 1060959897, 1633480282, 543551900},
	},

	// bit width = 32
	{
		32,
		[]byte{0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
		[8]int64{0, 0, 0, 0, 0, 0, 0, 0},
	},
	{
		32,
		[]byte{0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x05, 0x00, 0x00, 0x00, 0x06, 0x00, 0x00, 0x00, 0x07, 0x00, 0x00, 0x00},
		[8]int64{0, 1, 2, 3, 4, 5, 6, 7},
	},
	{
		32,
		[]byte{0x07, 0x00, 0x00, 0x00, 0x06, 0x00, 0x00, 0x00, 0x05, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
		[8]int64{7, 6, 5, 4, 3, 2, 1, 0},
	},
	{
		32,
		[]byte{0xFF, 0xFF, 0xFF, 0xFF, 0xFE, 0xFF, 0xFF, 0xFF, 0xFD, 0xFF, 0xFF, 0xFF, 0xFC, 0xFF, 0xFF, 0xFF, 0xFB, 0xFF, 0xFF, 0xFF, 0xFA, 0xFF, 0xFF, 0xFF, 0xF9, 0xFF, 0xFF, 0xFF, 0xF8, 0xFF, 0xFF, 0xFF},
		[8]int64{4294967295, 4294967294, 4294967293, 4294967292, 4294967291, 4294967290, 4294967289, 4294967288},
	},
	{
		32,
		[]byte{0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
		[8]int64{0, 0, 0, 0, 0, 0, 0, 0},
	},

	// bit width = 33
	{
		33,
		[]byte{0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
		[8]int64{0, 0, 0, 0, 0, 0, 0, 0},
	},
	{
		33,
		[]byte{0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x18, 0x00, 0x00, 0x00, 0x40, 0x00, 0x00, 0x00, 0xA0, 0x00, 0x00, 0x00, 0x80, 0x01, 0x00, 0x00, 0x80, 0x03, 0x00, 0x00, 0x00},
		[8]int64{0, 1, 2, 3, 4, 5, 6, 7},
	},
	{
		33,
		[]byte{0x07, 0x00, 0x00, 0x00, 0x0C, 0x00, 0x00, 0x00, 0x14, 0x00, 0x00, 0x00, 0x20, 0x00, 0x00, 0x00, 0x30, 0x00, 0x00, 0x00, 0x40, 0x00, 0x00, 0x00, 0x40, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
		[8]int64{7, 6, 5, 4, 3, 2, 1, 0},
	},
	{
		33,
		[]byte{0xFF, 0xFF, 0xFF, 0xFF, 0xFD, 0xFF, 0xFF, 0xFF, 0xF7, 0xFF, 0xFF, 0xFF, 0xE7, 0xFF, 0xFF, 0xFF, 0xBF, 0xFF, 0xFF, 0xFF, 0x5F, 0xFF, 0xFF, 0xFF, 0x7F, 0xFE, 0xFF, 0xFF, 0x7F, 0xFC, 0xFF, 0xFF, 0xFF},
		[8]int64{8589934591, 8589934590, 8589934589, 8589934588, 8589934587, 8589934586, 8589934585, 8589934584},
	},
	{
		33,
		[]byte{0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
		[8]int64{1, 0, 1, 1, 0, 0, 0, 0},
	},

	// bit width = 34
	{
		34,
		[]byte{0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
		[8]int64{0, 0, 0, 0, 0, 0, 0, 0},
	},
	{
		34,
		[]byte{0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x20, 0x00, 0x00, 0x00, 0xC0, 0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x14, 0x00, 0x00, 0x00, 0x60, 0x00, 0x00, 0x00, 0xC0, 0x01, 0x00, 0x00, 0x00},
		[8]int64{0, 1, 2, 3, 4, 5, 6, 7},
	},
	{
		34,
		[]byte{0x07, 0x00, 0x00, 0x00, 0x18, 0x00, 0x00, 0x00, 0x50, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
		[8]int64{7, 6, 5, 4, 3, 2, 1, 0},
	},
	{
		34,
		[]byte{0xFF, 0xFF, 0xFF, 0xFF, 0xFB, 0xFF, 0xFF, 0xFF, 0xDF, 0xFF, 0xFF, 0xFF, 0x3F, 0xFF, 0xFF, 0xFF, 0xFF, 0xFB, 0xFF, 0xFF, 0xFF, 0xEB, 0xFF, 0xFF, 0xFF, 0x9F, 0xFF, 0xFF, 0xFF, 0x3F, 0xFE, 0xFF, 0xFF, 0xFF},
		[8]int64{17179869183, 17179869182, 17179869181, 17179869180, 17179869179, 17179869178, 17179869177, 17179869176},
	},
	{
		34,
		[]byte{0x01, 0x00, 0x00, 0x00, 0x0C, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x20, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
		[8]int64{1, 3, 0, 2, 1, 1, 2, 0},
	},

	// bit width = 35
	{
		35,
		[]byte{0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
		[8]int64{0, 0, 0, 0, 0, 0, 0, 0},
	},
	{
		35,
		[]byte{0x00, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x80, 0x00, 0x00, 0x00, 0x00, 0x06, 0x00, 0x00, 0x00, 0x40, 0x00, 0x00, 0x00, 0x80, 0x02, 0x00, 0x00, 0x00, 0x18, 0x00, 0x00, 0x00, 0xE0, 0x00, 0x00, 0x00, 0x00},
		[8]int64{0, 1, 2, 3, 4, 5, 6, 7},
	},
	{
		35,
		[]byte{0x07, 0x00, 0x00, 0x00, 0x30, 0x00, 0x00, 0x00, 0x40, 0x01, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x30, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
		[8]int64{7, 6, 5, 4, 3, 2, 1, 0},
	},
	{
		35,
		[]byte{0xFF, 0xFF, 0xFF, 0xFF, 0xF7, 0xFF, 0xFF, 0xFF, 0x7F, 0xFF, 0xFF, 0xFF, 0xFF, 0xF9, 0xFF, 0xFF, 0xFF, 0xBF, 0xFF, 0xFF, 0xFF, 0x7F, 0xFD, 0xFF, 0xFF, 0xFF, 0xE7, 0xFF, 0xFF, 0xFF, 0x1F, 0xFF, 0xFF, 0xFF, 0xFF},
		[8]int64{34359738367, 34359738366, 34359738365, 34359738364, 34359738363, 34359738362, 34359738361, 34359738360},
	},
	{
		35,
		[]byte{0x01, 0x00, 0x00, 0x00, 0x38, 0x00, 0x00, 0x00, 0xC0, 0x01, 0x00, 0x00, 0x00, 0x06, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x0C, 0x00, 0x00, 0x00, 0x80, 0x00, 0x00, 0x00, 0x00},
		[8]int64{1, 7, 7, 3, 1, 2, 3, 4},
	},

	// bit width = 36
	{
		36,
		[]byte{0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
		[8]int64{0, 0, 0, 0, 0, 0, 0, 0},
	},
	{
		36,
		[]byte{0x00, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x30, 0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x50, 0x00, 0x00, 0x00, 0x00, 0x06, 0x00, 0x00, 0x00, 0x70, 0x00, 0x00, 0x00, 0x00},
		[8]int64{0, 1, 2, 3, 4, 5, 6, 7},
	},
	{
		36,
		[]byte{0x07, 0x00, 0x00, 0x00, 0x60, 0x00, 0x00, 0x00, 0x00, 0x05, 0x00, 0x00, 0x00, 0x40, 0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x20, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
		[8]int64{7, 6, 5, 4, 3, 2, 1, 0},
	},
	{
		36,
		[]byte{0xFF, 0xFF, 0xFF, 0xFF, 0xEF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFD, 0xFF, 0xFF, 0xFF, 0xCF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFB, 0xFF, 0xFF, 0xFF, 0xAF, 0xFF, 0xFF, 0xFF, 0xFF, 0xF9, 0xFF, 0xFF, 0xFF, 0x8F, 0xFF, 0xFF, 0xFF, 0xFF},
		[8]int64{68719476735, 68719476734, 68719476733, 68719476732, 68719476731, 68719476730, 68719476729, 68719476728},
	},
	{
		36,
		[]byte{0x09, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x00, 0x0E, 0x00, 0x00, 0x00, 0x80, 0x00, 0x00, 0x00, 0x00, 0x0B, 0x00, 0x00, 0x00, 0x20, 0x00, 0x00, 0x00, 0x00, 0x06, 0x00, 0x00, 0x00, 0x80, 0x00, 0x00, 0x00, 0x00},
		[8]int64{9, 1, 14, 8, 11, 2, 6, 8},
	},

	// bit width = 37
	{
		37,
		[]byte{0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
		[8]int64{0, 0, 0, 0, 0, 0, 0, 0},
	},
	{
		37,
		[]byte{0x00, 0x00, 0x00, 0x00, 0x20, 0x00, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x80, 0x01, 0x00, 0x00, 0x00, 0x40, 0x00, 0x00, 0x00, 0x00, 0x0A, 0x00, 0x00, 0x00, 0x80, 0x01, 0x00, 0x00, 0x00, 0x38, 0x00, 0x00, 0x00, 0x00},
		[8]int64{0, 1, 2, 3, 4, 5, 6, 7},
	},
	{
		37,
		[]byte{0x07, 0x00, 0x00, 0x00, 0xC0, 0x00, 0x00, 0x00, 0x00, 0x14, 0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x30, 0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x40, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
		[8]int64{7, 6, 5, 4, 3, 2, 1, 0},
	},
	{
		37,
		[]byte{0xFF, 0xFF, 0xFF, 0xFF, 0xDF, 0xFF, 0xFF, 0xFF, 0xFF, 0xF7, 0xFF, 0xFF, 0xFF, 0x7F, 0xFE, 0xFF, 0xFF, 0xFF, 0xBF, 0xFF, 0xFF, 0xFF, 0xFF, 0xF5, 0xFF, 0xFF, 0xFF, 0x7F, 0xFE, 0xFF, 0xFF, 0xFF, 0xC7, 0xFF, 0xFF, 0xFF, 0xFF},
		[8]int64{137438953471, 137438953470, 137438953469, 137438953468, 137438953467, 137438953466, 137438953465, 137438953464},
	},
	{
		37,
		[]byte{0x09, 0x00, 0x00, 0x00, 0x80, 0x01, 0x00, 0x00, 0x00, 0x2C, 0x00, 0x00, 0x00, 0x80, 0x09, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x2A, 0x00, 0x00, 0x00, 0x40, 0x02, 0x00, 0x00, 0x00, 0xA0, 0x00, 0x00, 0x00, 0x00},
		[8]int64{9, 12, 11, 19, 0, 21, 9, 20},
	},

	// bit width = 38
	{
		38,
		[]byte{0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
		[8]int64{0, 0, 0, 0, 0, 0, 0, 0},
	},
	{
		38,
		[]byte{0x00, 0x00, 0x00, 0x00, 0x40, 0x00, 0x00, 0x00, 0x00, 0x20, 0x00, 0x00, 0x00, 0x00, 0x0C, 0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x40, 0x01, 0x00, 0x00, 0x00, 0x60, 0x00, 0x00, 0x00, 0x00, 0x1C, 0x00, 0x00, 0x00, 0x00},
		[8]int64{0, 1, 2, 3, 4, 5, 6, 7},
	},
	{
		38,
		[]byte{0x07, 0x00, 0x00, 0x00, 0x80, 0x01, 0x00, 0x00, 0x00, 0x50, 0x00, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x80, 0x00, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
		[8]int64{7, 6, 5, 4, 3, 2, 1, 0},
	},
	{
		38,
		[]byte{0xFF, 0xFF, 0xFF, 0xFF, 0xBF, 0xFF, 0xFF, 0xFF, 0xFF, 0xDF, 0xFF, 0xFF, 0xFF, 0xFF, 0xF3, 0xFF, 0xFF, 0xFF, 0xFF, 0xFB, 0xFF, 0xFF, 0xFF, 0xBF, 0xFE, 0xFF, 0xFF, 0xFF, 0x9F, 0xFF, 0xFF, 0xFF, 0xFF, 0xE3, 0xFF, 0xFF, 0xFF, 0xFF},
		[8]int64{274877906943, 274877906942, 274877906941, 274877906940, 274877906939, 274877906938, 274877906937, 274877906936},
	},
	{
		38,
		[]byte{0x09, 0x00, 0x00, 0x00, 0x40, 0x06, 0x00, 0x00, 0x00, 0x20, 0x03, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x00, 0x31, 0x00, 0x00, 0x00, 0xC0, 0x0F, 0x00, 0x00, 0x00, 0x50, 0x00, 0x00, 0x00, 0x00, 0xC0, 0x00, 0x00, 0x00, 0x00},
		[8]int64{9, 25, 50, 4, 49, 63, 5, 48},
	},

	// bit width = 39
	{
		39,
		[]byte{0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
		[8]int64{0, 0, 0, 0, 0, 0, 0, 0},
	},
	{
		39,
		[]byte{0x00, 0x00, 0x00, 0x00, 0x80, 0x00, 0x00, 0x00, 0x00, 0x80, 0x00, 0x00, 0x00, 0x00, 0x60, 0x00, 0x00, 0x00, 0x00, 0x40, 0x00, 0x00, 0x00, 0x00, 0x28, 0x00, 0x00, 0x00, 0x00, 0x18, 0x00, 0x00, 0x00, 0x00, 0x0E, 0x00, 0x00, 0x00, 0x00},
		[8]int64{0, 1, 2, 3, 4, 5, 6, 7},
	},
	{
		39,
		[]byte{0x07, 0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x40, 0x01, 0x00, 0x00, 0x00, 0x80, 0x00, 0x00, 0x00, 0x00, 0x30, 0x00, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
		[8]int64{7, 6, 5, 4, 3, 2, 1, 0},
	},
	{
		39,
		[]byte{0xFF, 0xFF, 0xFF, 0xFF, 0x7F, 0xFF, 0xFF, 0xFF, 0xFF, 0x7F, 0xFF, 0xFF, 0xFF, 0xFF, 0x9F, 0xFF, 0xFF, 0xFF, 0xFF, 0xBF, 0xFF, 0xFF, 0xFF, 0xFF, 0xD7, 0xFF, 0xFF, 0xFF, 0xFF, 0xE7, 0xFF, 0xFF, 0xFF, 0xFF, 0xF1, 0xFF, 0xFF, 0xFF, 0xFF},
		[8]int64{549755813887, 549755813886, 549755813885, 549755813884, 549755813883, 549755813882, 549755813881, 549755813880},
	},
	{
		39,
		[]byte{0x69, 0x00, 0x00, 0x00, 0x80, 0x16, 0x00, 0x00, 0x00, 0x00, 0x16, 0x00, 0x00, 0x00, 0x40, 0x04, 0x00, 0x00, 0x00, 0x50, 0x05, 0x00, 0x00, 0x00, 0xC8, 0x03, 0x00, 0x00, 0x00, 0x28, 0x00, 0x00, 0x00, 0x00, 0xF8, 0x00, 0x00, 0x00, 0x00},
		[8]int64{105, 45, 88, 34, 85, 121, 10, 124},
	},

	// bit width = 40
	{
		40,
		[]byte{0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
		[8]int64{0, 0, 0, 0, 0, 0, 0, 0},
	},
	{
		40,
		[]byte{0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x05, 0x00, 0x00, 0x00, 0x00, 0x06, 0x00, 0x00, 0x00, 0x00, 0x07, 0x00, 0x00, 0x00, 0x00},
		[8]int64{0, 1, 2, 3, 4, 5, 6, 7},
	},
	{
		40,
		[]byte{0x07, 0x00, 0x00, 0x00, 0x00, 0x06, 0x00, 0x00, 0x00, 0x00, 0x05, 0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
		[8]int64{7, 6, 5, 4, 3, 2, 1, 0},
	},
	{
		40,
		[]byte{0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFE, 0xFF, 0xFF, 0xFF, 0xFF, 0xFD, 0xFF, 0xFF, 0xFF, 0xFF, 0xFC, 0xFF, 0xFF, 0xFF, 0xFF, 0xFB, 0xFF, 0xFF, 0xFF, 0xFF, 0xFA, 0xFF, 0xFF, 0xFF, 0xFF, 0xF9, 0xFF, 0xFF, 0xFF, 0xFF, 0xF8, 0xFF, 0xFF, 0xFF, 0xFF},
		[8]int64{1099511627775, 1099511627774, 1099511627773, 1099511627772, 1099511627771, 1099511627770, 1099511627769, 1099511627768},
	},
	{
		40,
		[]byte{0xC9, 0x00, 0x00, 0x00, 0x00, 0xCF, 0x00, 0x00, 0x00, 0x00, 0xB8, 0x00, 0x00, 0x00, 0x00, 0xDE, 0x00, 0x00, 0x00, 0x00, 0x8F, 0x00, 0x00, 0x00, 0x00, 0x6A, 0x00, 0x00, 0x00, 0x00, 0x17, 0x00, 0x00, 0x00, 0x00, 0x38, 0x00, 0x00, 0x00, 0x00},
		[8]int64{201, 207, 184, 222, 143, 106, 23, 56},
	},

	// bit width = 41
	{
		41,
		[]byte{0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
		[8]int64{0, 0, 0, 0, 0, 0, 0, 0},
	},
	{
		41,
		[]byte{0x00, 0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x18, 0x00, 0x00, 0x00, 0x00, 0x40, 0x00, 0x00, 0x00, 0x00, 0xA0, 0x00, 0x00, 0x00, 0x00, 0x80, 0x01, 0x00, 0x00, 0x00, 0x80, 0x03, 0x00, 0x00, 0x00, 0x00},
		[8]int64{0, 1, 2, 3, 4, 5, 6, 7},
	},
	{
		41,
		[]byte{0x07, 0x00, 0x00, 0x00, 0x00, 0x0C, 0x00, 0x00, 0x00, 0x00, 0x14, 0x00, 0x00, 0x00, 0x00, 0x20, 0x00, 0x00, 0x00, 0x00, 0x30, 0x00, 0x00, 0x00, 0x00, 0x40, 0x00, 0x00, 0x00, 0x00, 0x40, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
		[8]int64{7, 6, 5, 4, 3, 2, 1, 0},
	},
	{
		41,
		[]byte{0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFD, 0xFF, 0xFF, 0xFF, 0xFF, 0xF7, 0xFF, 0xFF, 0xFF, 0xFF, 0xE7, 0xFF, 0xFF, 0xFF, 0xFF, 0xBF, 0xFF, 0xFF, 0xFF, 0xFF, 0x5F, 0xFF, 0xFF, 0xFF, 0xFF, 0x7F, 0xFE, 0xFF, 0xFF, 0xFF, 0x7F, 0xFC, 0xFF, 0xFF, 0xFF, 0xFF},
		[8]int64{2199023255551, 2199023255550, 2199023255549, 2199023255548, 2199023255547, 2199023255546, 2199023255545, 2199023255544},
	},
	{
		41,
		[]byte{0x29, 0x00, 0x00, 0x00, 0x00, 0x46, 0x01, 0x00, 0x00, 0x00, 0xB8, 0x05, 0x00, 0x00, 0x00, 0x20, 0x00, 0x00, 0x00, 0x00, 0x60, 0x07, 0x00, 0x00, 0x00, 0xC0, 0x19, 0x00, 0x00, 0x00, 0x80, 0x46, 0x00, 0x00, 0x00, 0x00, 0x62, 0x00, 0x00, 0x00, 0x00},
		[8]int64{41, 163, 366, 4, 118, 206, 282, 196},
	},

	// bit width = 42
	{
		42,
		[]byte{0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
		[8]int64{0, 0, 0, 0, 0, 0, 0, 0},
	},
	{
		42,
		[]byte{0x00, 0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x20, 0x00, 0x00, 0x00, 0x00, 0xC0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x14, 0x00, 0x00, 0x00, 0x00, 0x60, 0x00, 0x00, 0x00, 0x00, 0xC0, 0x01, 0x00, 0x00, 0x00, 0x00},
		[8]int64{0, 1, 2, 3, 4, 5, 6, 7},
	},
	{
		42,
		[]byte{0x07, 0x00, 0x00, 0x00, 0x00, 0x18, 0x00, 0x00, 0x00, 0x00, 0x50, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
		[8]int64{7, 6, 5, 4, 3, 2, 1, 0},
	},
	{
		42,
		[]byte{0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFB, 0xFF, 0xFF, 0xFF, 0xFF, 0xDF, 0xFF, 0xFF, 0xFF, 0xFF, 0x3F, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFB, 0xFF, 0xFF, 0xFF, 0xFF, 0xEB, 0xFF, 0xFF, 0xFF, 0xFF, 0x9F, 0xFF, 0xFF, 0xFF, 0xFF, 0x3F, 0xFE, 0xFF, 0xFF, 0xFF, 0xFF},
		[8]int64{4398046511103, 4398046511102, 4398046511101, 4398046511100, 4398046511099, 4398046511098, 4398046511097, 4398046511096},
	},
	{
		42,
		[]byte{0x09, 0x01, 0x00, 0x00, 0x00, 0x60, 0x0F, 0x00, 0x00, 0x00, 0xB0, 0x34, 0x00, 0x00, 0x00, 0x40, 0x5B, 0x00, 0x00, 0x00, 0x00, 0x66, 0x03, 0x00, 0x00, 0x00, 0x90, 0x00, 0x00, 0x00, 0x00, 0x80, 0x08, 0x00, 0x00, 0x00, 0x00, 0x88, 0x00, 0x00, 0x00, 0x00},
		[8]int64{265, 984, 843, 365, 870, 36, 136, 544},
	},

	// bit width = 43
	{
		43,
		[]byte{0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
		[8]int64{0, 0, 0, 0, 0, 0, 0, 0},
	},
	{
		43,
		[]byte{0x00, 0x00, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x06, 0x00, 0x00, 0x00, 0x00, 0x40, 0x00, 0x00, 0x00, 0x00, 0x80, 0x02, 0x00, 0x00, 0x00, 0x00, 0x18, 0x00, 0x00, 0x00, 0x00, 0xE0, 0x00, 0x00, 0x00, 0x00, 0x00},
		[8]int64{0, 1, 2, 3, 4, 5, 6, 7},
	},
	{
		43,
		[]byte{0x07, 0x00, 0x00, 0x00, 0x00, 0x30, 0x00, 0x00, 0x00, 0x00, 0x40, 0x01, 0x00, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x30, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
		[8]int64{7, 6, 5, 4, 3, 2, 1, 0},
	},
	{
		43,
		[]byte{0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xF7, 0xFF, 0xFF, 0xFF, 0xFF, 0x7F, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xF9, 0xFF, 0xFF, 0xFF, 0xFF, 0xBF, 0xFF, 0xFF, 0xFF, 0xFF, 0x7F, 0xFD, 0xFF, 0xFF, 0xFF, 0xFF, 0xE7, 0xFF, 0xFF, 0xFF, 0xFF, 0x1F, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF},
		[8]int64{8796093022207, 8796093022206, 8796093022205, 8796093022204, 8796093022203, 8796093022202, 8796093022201, 8796093022200},
	},
	{
		43,
		[]byte{0x69, 0x01, 0x00, 0x00, 0x00, 0x20, 0x38, 0x00, 0x00, 0x00, 0xC0, 0xFF, 0x00, 0x00, 0x00, 0x00, 0x90, 0x0C, 0x00, 0x00, 0x00, 0x50, 0x59, 0x00, 0x00, 0x00, 0x80, 0x2E, 0x00, 0x00, 0x00, 0x00, 0x20, 0x05, 0x00, 0x00, 0x00, 0x80, 0xB5, 0x00, 0x00, 0x00, 0x00},
		[8]int64{361, 1796, 1023, 1608, 1429, 93, 328, 1452},
	},

	// bit width = 44
	{
		44,
		[]byte{0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
		[8]int64{0, 0, 0, 0, 0, 0, 0, 0},
	},
	{
		44,
		[]byte{0x00, 0x00, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x30, 0x00, 0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x50, 0x00, 0x00, 0x00, 0x00, 0x00, 0x06, 0x00, 0x00, 0x00, 0x00, 0x70, 0x00, 0x00, 0x00, 0x00, 0x00},
		[8]int64{0, 1, 2, 3, 4, 5, 6, 7},
	},
	{
		44,
		[]byte{0x07, 0x00, 0x00, 0x00, 0x00, 0x60, 0x00, 0x00, 0x00, 0x00, 0x00, 0x05, 0x00, 0x00, 0x00, 0x00, 0x40, 0x00, 0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x00, 0x20, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
		[8]int64{7, 6, 5, 4, 3, 2, 1, 0},
	},
	{
		44,
		[]byte{0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xEF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFD, 0xFF, 0xFF, 0xFF, 0xFF, 0xCF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFB, 0xFF, 0xFF, 0xFF, 0xFF, 0xAF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xF9, 0xFF, 0xFF, 0xFF, 0xFF, 0x8F, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF},
		[8]int64{17592186044415, 17592186044414, 17592186044413, 17592186044412, 17592186044411, 17592186044410, 17592186044409, 17592186044408},
	},
	{
		44,
		[]byte{0x4A, 0x0F, 0x00, 0x00, 0x00, 0x20, 0xAE, 0x00, 0x00, 0x00, 0x00, 0x6F, 0x00, 0x00, 0x00, 0x00, 0x50, 0x57, 0x00, 0x00, 0x00, 0x00, 0x87, 0x0D, 0x00, 0x00, 0x00, 0x90, 0x97, 0x00, 0x00, 0x00, 0x00, 0x54, 0x06, 0x00, 0x00, 0x00, 0x80, 0xB6, 0x00, 0x00, 0x00, 0x00},
		[8]int64{3914, 2786, 111, 1397, 3463, 2425, 1620, 2920},
	},

	// bit width = 45
	{
		45,
		[]byte{0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
		[8]int64{0, 0, 0, 0, 0, 0, 0, 0},
	},
	{
		45,
		[]byte{0x00, 0x00, 0x00, 0x00, 0x00, 0x20, 0x00, 0x00, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x80, 0x01, 0x00, 0x00, 0x00, 0x00, 0x40, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0A, 0x00, 0x00, 0x00, 0x00, 0x80, 0x01, 0x00, 0x00, 0x00, 0x00, 0x38, 0x00, 0x00, 0x00, 0x00, 0x00},
		[8]int64{0, 1, 2, 3, 4, 5, 6, 7},
	},
	{
		45,
		[]byte{0x07, 0x00, 0x00, 0x00, 0x00, 0xC0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x30, 0x00, 0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x40, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
		[8]int64{7, 6, 5, 4, 3, 2, 1, 0},
	},
	{
		45,
		[]byte{0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xDF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xF7, 0xFF, 0xFF, 0xFF, 0xFF, 0x7F, 0xFE, 0xFF, 0xFF, 0xFF, 0xFF, 0xBF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xF5, 0xFF, 0xFF, 0xFF, 0xFF, 0x7F, 0xFE, 0xFF, 0xFF, 0xFF, 0xFF, 0xC7, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF},
		[8]int64{35184372088831, 35184372088830, 35184372088829, 35184372088828, 35184372088827, 35184372088826, 35184372088825, 35184372088824},
	},
	{
		45,
		[]byte{0xAA, 0x03, 0x00, 0x00, 0x00, 0xE0, 0x41, 0x00, 0x00, 0x00, 0x00, 0x1C, 0x61, 0x00, 0x00, 0x00, 0x00, 0x4D, 0x0A, 0x00, 0x00, 0x00, 0x30, 0x37, 0x01, 0x00, 0x00, 0x00, 0x64, 0x23, 0x00, 0x00, 0x00, 0x40, 0xF7, 0x06, 0x00, 0x00, 0x00, 0x60, 0x28, 0x00, 0x00, 0x00, 0x00},
		[8]int64{938, 527, 6215, 5274, 4979, 4530, 7133, 1292},
	},

	// bit width = 46
	{
		46,
		[]byte{0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
		[8]int64{0, 0, 0, 0, 0, 0, 0, 0},
	},
	{
		46,
		[]byte{0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0x00, 0x00, 0x00, 0x00, 0x00, 0x20, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0C, 0x00, 0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x40, 0x01, 0x00, 0x00, 0x00, 0x00, 0x60, 0x00, 0x00, 0x00, 0x00, 0x00, 0x1C, 0x00, 0x00, 0x00, 0x00, 0x00},
		[8]int64{0, 1, 2, 3, 4, 5, 6, 7},
	},
	{
		46,
		[]byte{0x07, 0x00, 0x00, 0x00, 0x00, 0x80, 0x01, 0x00, 0x00, 0x00, 0x00, 0x50, 0x00, 0x00, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x00, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
		[8]int64{7, 6, 5, 4, 3, 2, 1, 0},
	},
	{
		46,
		[]byte{0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xBF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xDF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xF3, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFB, 0xFF, 0xFF, 0xFF, 0xFF, 0xBF, 0xFE, 0xFF, 0xFF, 0xFF, 0xFF, 0x9F, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xE3, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF},
		[8]int64{70368744177663, 70368744177662, 70368744177661, 70368744177660, 70368744177659, 70368744177658, 70368744177657, 70368744177656},
	},
	{
		46,
		[]byte{0x8A, 0x06, 0x00, 0x00, 0x00, 0x80, 0xDD, 0x08, 0x00, 0x00, 0x00, 0xC0, 0x1D, 0x02, 0x00, 0x00, 0x00, 0x5C, 0x23, 0x00, 0x00, 0x00, 0x00, 0x84, 0x10, 0x00, 0x00, 0x00, 0xC0, 0x41, 0x02, 0x00, 0x00, 0x00, 0x60, 0xAB, 0x01, 0x00, 0x00, 0x00, 0x40, 0xF1, 0x00, 0x00, 0x00, 0x00},
		[8]int64{1674, 9078, 8668, 2263, 4228, 2311, 6838, 15440},
	},

	// bit width = 47
	{
		47,
		[]byte{0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
		[8]int64{0, 0, 0, 0, 0, 0, 0, 0},
	},
	{
		47,
		[]byte{0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x60, 0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0x00, 0x00, 0x00, 0x00, 0x00, 0x28, 0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0E, 0x00, 0x00, 0x00, 0x00, 0x00},
		[8]int64{0, 1, 2, 3, 4, 5, 6, 7},
	},
	{
		47,
		[]byte{0x07, 0x00, 0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x00, 0x40, 0x01, 0x00, 0x00, 0x00, 0x00, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x30, 0x00, 0x00, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
		[8]int64{7, 6, 5, 4, 3, 2, 1, 0},
	},
	{
		47,
		[]byte{0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0x7F, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0x7F, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0x9F, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xBF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xD7, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xE7, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xF1, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF},
		[8]int64{140737488355327, 140737488355326, 140737488355325, 140737488355324, 140737488355323, 140737488355322, 140737488355321, 140737488355320},
	},
	{
		47,
		[]byte{0xEA, 0x7A, 0x00, 0x00, 0x00, 0x00, 0xDB, 0x09, 0x00, 0x00, 0x00, 0xC0, 0xB6, 0x09, 0x00, 0x00, 0x00, 0x20, 0x36, 0x0D, 0x00, 0x00, 0x00, 0x20, 0xE8, 0x00, 0x00, 0x00, 0x00, 0x00, 0xF6, 0x03, 0x00, 0x00, 0x00, 0xEC, 0x5D, 0x01, 0x00, 0x00, 0x00, 0xB8, 0xB7, 0x00, 0x00, 0x00, 0x00},
		[8]int64{31466, 5046, 9947, 27057, 3714, 32448, 22395, 23516},
	},

	// bit width = 48
	{
		48,
		[]byte{0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
		[8]int64{0, 0, 0, 0, 0, 0, 0, 0},
	},
	{
		48,
		[]byte{0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x05, 0x00, 0x00, 0x00, 0x00, 0x00, 0x06, 0x00, 0x00, 0x00, 0x00, 0x00, 0x07, 0x00, 0x00, 0x00, 0x00, 0x00},
		[8]int64{0, 1, 2, 3, 4, 5, 6, 7},
	},
	{
		48,
		[]byte{0x07, 0x00, 0x00, 0x00, 0x00, 0x00, 0x06, 0x00, 0x00, 0x00, 0x00, 0x00, 0x05, 0x00, 0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
		[8]int64{7, 6, 5, 4, 3, 2, 1, 0},
	},
	{
		48,
		[]byte{0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFE, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFD, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFC, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFB, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFA, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xF9, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xF8, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF},
		[8]int64{281474976710655, 281474976710654, 281474976710653, 281474976710652, 281474976710651, 281474976710650, 281474976710649, 281474976710648},
	},
	{
		48,
		[]byte{0xC8, 0xF4, 0x00, 0x00, 0x00, 0x00, 0x08, 0x02, 0x00, 0x00, 0x00, 0x00, 0x61, 0x84, 0x00, 0x00, 0x00, 0x00, 0xA9, 0x32, 0x00, 0x00, 0x00, 0x00, 0xEB, 0x48, 0x00, 0x00, 0x00, 0x00, 0xBE, 0x6C, 0x00, 0x00, 0x00, 0x00, 0xA8, 0xF3, 0x00, 0x00, 0x00, 0x00, 0x18, 0x08, 0x00, 0x00, 0x00, 0x00},
		[8]int64{62664, 520, 33889, 12969, 18667, 27838, 62376, 2072},
	},

	// bit width = 49
	{
		49,
		[]byte{0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
		[8]int64{0, 0, 0, 0, 0, 0, 0, 0},
	},
	{
		49,
		[]byte{0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0x00, 0x00, 0x00, 0x00, 0x00, 0xA0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0x01, 0x00, 0x00, 0x00, 0x00, 0x80, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00},
		[8]int64{0, 1, 2, 3, 4, 5, 6, 7},
	},
	{
		49,
		[]byte{0x07, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0C, 0x00, 0x00, 0x00, 0x00, 0x00, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x20, 0x00, 0x00, 0x00, 0x00, 0x00, 0x30, 0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
		[8]int64{7, 6, 5, 4, 3, 2, 1, 0},
	},
	{
		49,
		[]byte{0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFD, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xF7, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xE7, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xBF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0x5F, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0x7F, 0xFE, 0xFF, 0xFF, 0xFF, 0xFF, 0x7F, 0xFC, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF},
		[8]int64{562949953421311, 562949953421310, 562949953421309, 562949953421308, 562949953421307, 562949953421306, 562949953421305, 562949953421304},
	},
	{
		49,
		[]byte{0x29, 0xA9, 0x00, 0x00, 0x00, 0x00, 0x98, 0x0D, 0x01, 0x00, 0x00, 0x00, 0x58, 0x11, 0x05, 0x00, 0x00, 0x00, 0xE8, 0x63, 0x03, 0x00, 0x00, 0x00, 0x10, 0x7F, 0x0E, 0x00, 0x00, 0x00, 0x60, 0x6F, 0x39, 0x00, 0x00, 0x00, 0x40, 0xA2, 0x4F, 0x00, 0x00, 0x00, 0x00, 0xD2, 0xD3, 0x00, 0x00, 0x00, 0x00},
		[8]int64{43305, 34508, 83030, 27773, 59377, 117627, 81545, 108452},
	},

	// bit width = 50
	{
		50,
		[]byte{0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
		[8]int64{0, 0, 0, 0, 0, 0, 0, 0},
	},
	{
		50,
		[]byte{0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x20, 0x00, 0x00, 0x00, 0x00, 0x00, 0xC0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x60, 0x00, 0x00, 0x00, 0x00, 0x00, 0xC0, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00},
		[8]int64{0, 1, 2, 3, 4, 5, 6, 7},
	},
	{
		50,
		[]byte{0x07, 0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0x00, 0x00, 0x00, 0x00, 0x00, 0x50, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
		[8]int64{7, 6, 5, 4, 3, 2, 1, 0},
	},
	{
		50,
		[]byte{0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFB, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xDF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0x3F, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFB, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xEB, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0x9F, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0x3F, 0xFE, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF},
		[8]int64{1125899906842623, 1125899906842622, 1125899906842621, 1125899906842620, 1125899906842619, 1125899906842618, 1125899906842617, 1125899906842616},
	},
	{
		50,
		[]byte{0x08, 0x8C, 0x00, 0x00, 0x00, 0x00, 0xBC, 0x4E, 0x0C, 0x00, 0x00, 0x00, 0xD0, 0x5C, 0x31, 0x00, 0x00, 0x00, 0x00, 0xC3, 0x23, 0x00, 0x00, 0x00, 0x00, 0xE0, 0x0A, 0x00, 0x00, 0x00, 0x00, 0x40, 0x8B, 0x04, 0x00, 0x00, 0x00, 0x70, 0x7F, 0x2A, 0x00, 0x00, 0x00, 0x00, 0x40, 0x32, 0x00, 0x00, 0x00, 0x00},
		[8]int64{35848, 201647, 202189, 36620, 2784, 74448, 174071, 51456},
	},

	// bit width = 51
	{
		51,
		[]byte{0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
		[8]int64{0, 0, 0, 0, 0, 0, 0, 0},
	},
	{
		51,
		[]byte{0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x06, 0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0x00, 0x00, 0x00, 0x00, 0x00, 0xE0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
		[8]int64{0, 1, 2, 3, 4, 5, 6, 7},
	},
	{
		51,
		[]byte{0x07, 0x00, 0x00, 0x00, 0x00, 0x00, 0x30, 0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x30, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
		[8]int64{7, 6, 5, 4, 3, 2, 1, 0},
	},
	{
		51,
		[]byte{0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xF7, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0x7F, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xF9, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xBF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0x7F, 0xFD, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xE7, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0x1F, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF},
		[8]int64{2251799813685247, 2251799813685246, 2251799813685245, 2251799813685244, 2251799813685243, 2251799813685242, 2251799813685241, 2251799813685240},
	},
	{
		51,
		[]byte{0x68, 0x40, 0x01, 0x00, 0x00, 0x00, 0xE0, 0x56, 0x14, 0x00, 0x00, 0x00, 0x40, 0xBA, 0x6C, 0x00, 0x00, 0x00, 0x00, 0x9C, 0xA3, 0x05, 0x00, 0x00, 0x00, 0xA0, 0x61, 0x05, 0x00, 0x00, 0x00, 0x80, 0x7B, 0x52, 0x03, 0x00, 0x00, 0x00, 0xA0, 0x68, 0x03, 0x00, 0x00, 0x00, 0x80, 0x11, 0xCD, 0x00, 0x00, 0x00, 0x00},
		[8]int64{82024, 166620, 111337, 184782, 22042, 435447, 55848, 419980},
	},

	// bit width = 52
	{
		52,
		[]byte{0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
		[8]int64{0, 0, 0, 0, 0, 0, 0, 0},
	},
	{
		52,
		[]byte{0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x30, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x50, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x06, 0x00, 0x00, 0x00, 0x00, 0x00, 0x70, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
		[8]int64{0, 1, 2, 3, 4, 5, 6, 7},
	},
	{
		52,
		[]byte{0x07, 0x00, 0x00, 0x00, 0x00, 0x00, 0x60, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x05, 0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x20, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
		[8]int64{7, 6, 5, 4, 3, 2, 1, 0},
	},
	{
		52,
		[]byte{0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xEF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFD, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xCF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFB, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xAF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xF9, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0x8F, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF},
		[8]int64{4503599627370495, 4503599627370494, 4503599627370493, 4503599627370492, 4503599627370491, 4503599627370490, 4503599627370489, 4503599627370488},
	},
	{
		52,
		[]byte{0x49, 0xC6, 0x04, 0x00, 0x00, 0x00, 0x90, 0xEB, 0xBD, 0x00, 0x00, 0x00, 0x00, 0x79, 0x9E, 0x0D, 0x00, 0x00, 0x00, 0x20, 0x61, 0x7B, 0x00, 0x00, 0x00, 0x00, 0xFE, 0x3A, 0x0A, 0x00, 0x00, 0x00, 0x50, 0x42, 0xE7, 0x00, 0x00, 0x00, 0x00, 0xE5, 0x2A, 0x04, 0x00, 0x00, 0x00, 0x80, 0x64, 0xA8, 0x00, 0x00, 0x00, 0x00},
		[8]int64{312905, 777913, 892537, 505362, 670462, 947237, 273125, 689736},
	},

	// bit width = 53
	{
		53,
		[]byte{0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
		[8]int64{0, 0, 0, 0, 0, 0, 0, 0},
	},
	{
		53,
		[]byte{0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x20, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0A, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x38, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
		[8]int64{0, 1, 2, 3, 4, 5, 6, 7},
	},
	{
		53,
		[]byte{0x07, 0x00, 0x00, 0x00, 0x00, 0x00, 0xC0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x30, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
		[8]int64{7, 6, 5, 4, 3, 2, 1, 0},
	},
	{
		53,
		[]byte{0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xDF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xF7, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0x7F, 0xFE, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xBF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xF5, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0x7F, 0xFE, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xC7, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF},
		[8]int64{9007199254740991, 9007199254740990, 9007199254740989, 9007199254740988, 9007199254740987, 9007199254740986, 9007199254740985, 9007199254740984},
	},
	{
		53,
		[]byte{0xA9, 0x7A, 0x15, 0x00, 0x00, 0x00, 0xC0, 0xBC, 0xEA, 0x02, 0x00, 0x00, 0x00, 0x44, 0x63, 0x26, 0x00, 0x00, 0x00, 0x80, 0x89, 0x74, 0x00, 0x00, 0x00, 0x00, 0x90, 0xBF, 0xA9, 0x01, 0x00, 0x00, 0x00, 0x44, 0xA7, 0x0A, 0x00, 0x00, 0x00, 0x00, 0x2D, 0x7E, 0x06, 0x00, 0x00, 0x00, 0xA0, 0x2E, 0x41, 0x00, 0x00, 0x00, 0x00},
		[8]int64{1407657, 1529318, 628945, 59667, 1743865, 349090, 1702068, 533972},
	},

	// bit width = 54
	{
		54,
		[]byte{0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
		[8]int64{0, 0, 0, 0, 0, 0, 0, 0},
	},
	{
		54,
		[]byte{0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x20, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0C, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x60, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x1C, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
		[8]int64{0, 1, 2, 3, 4, 5, 6, 7},
	},
	{
		54,
		[]byte{0x07, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x50, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
		[8]int64{7, 6, 5, 4, 3, 2, 1, 0},
	},
	{
		54,
		[]byte{0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xBF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xDF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xF3, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFB, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xBF, 0xFE, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0x9F, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xE3, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF},
		[8]int64{18014398509481983, 18014398509481982, 18014398509481981, 18014398509481980, 18014398509481979, 18014398509481978, 18014398509481977, 18014398509481976},
	},
	{
		54,
		[]byte{0x89, 0x5D, 0x33, 0x00, 0x00, 0x00, 0xC0, 0xE7, 0xC3, 0x0A, 0x00, 0x00, 0x00, 0x40, 0x2F, 0x3F, 0x00, 0x00, 0x00, 0x00, 0x88, 0x2E, 0x9C, 0x00, 0x00, 0x00, 0x00, 0x09, 0x79, 0x19, 0x00, 0x00, 0x00, 0x00, 0x13, 0x7F, 0x0E, 0x00, 0x00, 0x00, 0x60, 0xF4, 0x88, 0x01, 0x00, 0x00, 0x00, 0xC0, 0x1C, 0xBD, 0x00, 0x00, 0x00, 0x00},
		[8]int64{3366281, 2822047, 258804, 2558882, 1669385, 3800140, 1609542, 3098416},
	},

	// bit width = 55
	{
		55,
		[]byte{0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
		[8]int64{0, 0, 0, 0, 0, 0, 0, 0},
	},
	{
		55,
		[]byte{0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x60, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x28, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0E, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
		[8]int64{0, 1, 2, 3, 4, 5, 6, 7},
	},
	{
		55,
		[]byte{0x07, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x30, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
		[8]int64{7, 6, 5, 4, 3, 2, 1, 0},
	},
	{
		55,
		[]byte{0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0x7F, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0x7F, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0x9F, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xBF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xD7, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xE7, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xF1, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF},
		[8]int64{36028797018963967, 36028797018963966, 36028797018963965, 36028797018963964, 36028797018963963, 36028797018963962, 36028797018963961, 36028797018963960},
	},
	{
		55,
		[]byte{0xE9, 0x11, 0x14, 0x00, 0x00, 0x00, 0x80, 0x39, 0xCC, 0x1F, 0x00, 0x00, 0x00, 0x80, 0x70, 0x35, 0x09, 0x00, 0x00, 0x00, 0x00, 0xA7, 0x0A, 0x02, 0x00, 0x00, 0x00, 0xC0, 0x5F, 0x62, 0x04, 0x00, 0x00, 0x00, 0xA0, 0x20, 0xEE, 0x00, 0x00, 0x00, 0x00, 0xAC, 0x8B, 0x46, 0x01, 0x00, 0x00, 0x00, 0x78, 0xCD, 0x59, 0x00, 0x00, 0x00, 0x00},
		[8]int64{1315305, 4167795, 2414018, 1070392, 4597244, 1950740, 5350123, 2942652},
	},

	// bit width = 56
	{
		56,
		[]byte{0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
		[8]int64{0, 0, 0, 0, 0, 0, 0, 0},
	},
	{
		56,
		[]byte{0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x05, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x06, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x07, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
		[8]int64{0, 1, 2, 3, 4, 5, 6, 7},
	},
	{
		56,
		[]byte{0x07, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x06, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x05, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
		[8]int64{7, 6, 5, 4, 3, 2, 1, 0},
	},
	{
		56,
		[]byte{0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFE, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFD, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFC, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFB, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFA, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xF9, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xF8, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF},
		[8]int64{72057594037927935, 72057594037927934, 72057594037927933, 72057594037927932, 72057594037927931, 72057594037927930, 72057594037927929, 72057594037927928},
	},
	{
		56,
		[]byte{0xC9, 0x97, 0x37, 0x00, 0x00, 0x00, 0x00, 0x95, 0x44, 0x26, 0x00, 0x00, 0x00, 0x00, 0x52, 0xC1, 0xF0, 0x00, 0x00, 0x00, 0x00, 0x7C, 0x39, 0x25, 0x00, 0x00, 0x00, 0x00, 0xE8, 0xBE, 0x1B, 0x00, 0x00, 0x00, 0x00, 0xF7, 0xAA, 0x10, 0x00, 0x00, 0x00, 0x00, 0xDE, 0x9D, 0xE4, 0x00, 0x00, 0x00, 0x00, 0x88, 0xFB, 0x9E, 0x00, 0x00, 0x00, 0x00},
		[8]int64{3643337, 2507925, 15778130, 2439548, 1818344, 1092343, 14982622, 10419080},
	},

	// bit width = 57
	{
		57,
		[]byte{0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
		[8]int64{0, 0, 0, 0, 0, 0, 0, 0},
	},
	{
		57,
		[]byte{0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xA0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
		[8]int64{0, 1, 2, 3, 4, 5, 6, 7},
	},
	{
		57,
		[]byte{0x07, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0C, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x20, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x30, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
		[8]int64{7, 6, 5, 4, 3, 2, 1, 0},
	},
	{
		57,
		[]byte{0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFD, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xF7, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xE7, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xBF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0x5F, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0x7F, 0xFE, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0x7F, 0xFC, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF},
		[8]int64{144115188075855871, 144115188075855870, 144115188075855869, 144115188075855868, 144115188075855867, 144115188075855866, 144115188075855865, 144115188075855864},
	},
	{
		57,
		[]byte{0x29, 0x4C, 0x58, 0x01, 0x00, 0x00, 0x00, 0xD2, 0x9A, 0x75, 0x00, 0x00, 0x00, 0x00, 0xE0, 0xD7, 0x99, 0x03, 0x00, 0x00, 0x00, 0xB0, 0xD2, 0xE4, 0x0E, 0x00, 0x00, 0x00, 0xD0, 0x01, 0x3E, 0x10, 0x00, 0x00, 0x00, 0xE0, 0x57, 0x8E, 0x0E, 0x00, 0x00, 0x00, 0x40, 0xFC, 0x4B, 0x4C, 0x00, 0x00, 0x00, 0x00, 0xFE, 0xAD, 0x30, 0x00, 0x00, 0x00, 0x00},
		[8]int64{22563881, 3853673, 15103480, 31234646, 17031197, 7631551, 20000753, 6380540},
	},

	// bit width = 58
	{
		58,
		[]byte{0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
		[8]int64{0, 0, 0, 0, 0, 0, 0, 0},
	},
	{
		58,
		[]byte{0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x20, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xC0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x60, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xC0, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
		[8]int64{0, 1, 2, 3, 4, 5, 6, 7},
	},
	{
		58,
		[]byte{0x07, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x50, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
		[8]int64{7, 6, 5, 4, 3, 2, 1, 0},
	},
	{
		58,
		[]byte{0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFB, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xDF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0x3F, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFB, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xEB, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0x9F, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0x3F, 0xFE, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF},
		[8]int64{288230376151711743, 288230376151711742, 288230376151711741, 288230376151711740, 288230376151711739, 288230376151711738, 288230376151711737, 288230376151711736},
	},
	{
		58,
		[]byte{0x09, 0x2F, 0xF6, 0x03, 0x00, 0x00, 0x00, 0x48, 0x34, 0x0B, 0x04, 0x00, 0x00, 0x00, 0x50, 0xFE, 0xF2, 0x29, 0x00, 0x00, 0x00, 0x40, 0x0E, 0x62, 0x92, 0x00, 0x00, 0x00, 0x00, 0x0D, 0x03, 0xA5, 0x02, 0x00, 0x00, 0x00, 0xA4, 0x6D, 0x24, 0x05, 0x00, 0x00, 0x00, 0xD0, 0x97, 0x03, 0x31, 0x00, 0x00, 0x00, 0x00, 0xA8, 0x8E, 0xC6, 0x00, 0x00, 0x00, 0x00},
		[8]int64{66465545, 16960786, 43986917, 38373433, 44368653, 21568361, 51394941, 52050592},
	},

	// bit width = 59
	{
		59,
		[]byte{0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
		[8]int64{0, 0, 0, 0, 0, 0, 0, 0},
	},
	{
		59,
		[]byte{0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x06, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xE0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
		[8]int64{0, 1, 2, 3, 4, 5, 6, 7},
	},
	{
		59,
		[]byte{0x07, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x30, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x30, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
		[8]int64{7, 6, 5, 4, 3, 2, 1, 0},
	},
	{
		59,
		[]byte{0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xF7, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0x7F, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xF9, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xBF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0x7F, 0xFD, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xE7, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0x1F, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF},
		[8]int64{576460752303423487, 576460752303423486, 576460752303423485, 576460752303423484, 576460752303423483, 576460752303423482, 576460752303423481, 576460752303423480},
	},
	{
		59,
		[]byte{0x69, 0xE3, 0x16, 0x03, 0x00, 0x00, 0x00, 0x10, 0xDE, 0x8D, 0x00, 0x00, 0x00, 0x00, 0x00, 0xD9, 0x01, 0xCE, 0x01, 0x00, 0x00, 0x00, 0xBC, 0x4E, 0x24, 0x07, 0x00, 0x00, 0x00, 0xE0, 0xDE, 0xB5, 0x73, 0x00, 0x00, 0x00, 0x00, 0x68, 0x8E, 0x29, 0x02, 0x00, 0x00, 0x00, 0x4C, 0x51, 0x16, 0x04, 0x00, 0x00, 0x00, 0x80, 0x62, 0x93, 0x1B, 0x00, 0x00, 0x00, 0x00},
		[8]int64{51831657, 1162178, 121112420, 59909982, 121331182, 72555728, 17142867, 14457620},
	},

	// bit width = 60
	{
		60,
		[]byte{0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
		[8]int64{0, 0, 0, 0, 0, 0, 0, 0},
	},
	{
		60,
		[]byte{0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x30, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x50, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x06, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x70, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
		[8]int64{0, 1, 2, 3, 4, 5, 6, 7},
	},
	{
		60,
		[]byte{0x07, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x60, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x05, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x20, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
		[8]int64{7, 6, 5, 4, 3, 2, 1, 0},
	},
	{
		60,
		[]byte{0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xEF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFD, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xCF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFB, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xAF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xF9, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0x8F, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF},
		[8]int64{1152921504606846975, 1152921504606846974, 1152921504606846973, 1152921504606846972, 1152921504606846971, 1152921504606846970, 1152921504606846969, 1152921504606846968},
	},
	{
		60,
		[]byte{0x49, 0x69, 0xBA, 0x02, 0x00, 0x00, 0x00, 0xC0, 0x81, 0x79, 0x68, 0x00, 0x00, 0x00, 0x00, 0xD4, 0x1B, 0x6C, 0x04, 0x00, 0x00, 0x00, 0x50, 0xCE, 0x2B, 0xA0, 0x00, 0x00, 0x00, 0x00, 0x2E, 0xBD, 0x62, 0x03, 0x00, 0x00, 0x00, 0x40, 0xA1, 0xFC, 0x39, 0x00, 0x00, 0x00, 0x00, 0x5F, 0x99, 0x72, 0x05, 0x00, 0x00, 0x00, 0x80, 0xD5, 0x87, 0x5A, 0x00, 0x00, 0x00, 0x00},
		[8]int64{45771081, 109549596, 74193876, 167951589, 56802606, 60803604, 91396447, 94928216},
	},

	// bit width = 61
	{
		61,
		[]byte{0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
		[8]int64{0, 0, 0, 0, 0, 0, 0, 0},
	},
	{
		61,
		[]byte{0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x20, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0A, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x38, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
		[8]int64{0, 1, 2, 3, 4, 5, 6, 7},
	},
	{
		61,
		[]byte{0x07, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xC0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x30, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
		[8]int64{7, 6, 5, 4, 3, 2, 1, 0},
	},
	{
		61,
		[]byte{0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xDF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xF7, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0x7F, 0xFE, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xBF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xF5, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0x7F, 0xFE, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xC7, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF},
		[8]int64{2305843009213693951, 2305843009213693950, 2305843009213693949, 2305843009213693948, 2305843009213693947, 2305843009213693946, 2305843009213693945, 2305843009213693944},
	},
	{
		61,
		[]byte{0xA9, 0x1D, 0xDB, 0x1D, 0x00, 0x00, 0x00, 0xE0, 0x16, 0x9E, 0xD1, 0x02, 0x00, 0x00, 0x00, 0x88, 0xB3, 0xF2, 0x16, 0x00, 0x00, 0x00, 0x00, 0xE0, 0x0E, 0x5D, 0x0A, 0x00, 0x00, 0x00, 0xC0, 0x3C, 0xDA, 0xA7, 0x01, 0x00, 0x00, 0x00, 0x4A, 0xDC, 0xF8, 0x0B, 0x00, 0x00, 0x00, 0x40, 0x34, 0xE6, 0x9D, 0x05, 0x00, 0x00, 0x00, 0x60, 0xEE, 0x56, 0x9B, 0x00, 0x00, 0x00, 0x00},
		[8]int64{500899241, 378335415, 96251106, 347741632, 444441548, 100429349, 376936657, 325770700},
	},

	// bit width = 62
	{
		62,
		[]byte{0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
		[8]int64{0, 0, 0, 0, 0, 0, 0, 0},
	},
	{
		62,
		[]byte{0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x20, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0C, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x60, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x1C, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
		[8]int64{0, 1, 2, 3, 4, 5, 6, 7},
	},
	{
		62,
		[]byte{0x07, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x50, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
		[8]int64{7, 6, 5, 4, 3, 2, 1, 0},
	},
	{
		62,
		[]byte{0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xBF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xDF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xF3, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFB, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xBF, 0xFE, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0x9F, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xE3, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF},
		[8]int64{4611686018427387903, 4611686018427387902, 4611686018427387901, 4611686018427387900, 4611686018427387899, 4611686018427387898, 4611686018427387897, 4611686018427387896},
	},
	{
		62,
		[]byte{0x89, 0x00, 0x79, 0x2C, 0x00, 0x00, 0x00, 0x00, 0x71, 0x2A, 0x2C, 0x04, 0x00, 0x00, 0x00, 0xF0, 0x2B, 0x25, 0x74, 0x03, 0x00, 0x00, 0x00, 0x3C, 0x13, 0xB0, 0x2D, 0x00, 0x00, 0x00, 0x00, 0xDD, 0x80, 0xDC, 0x30, 0x00, 0x00, 0x00, 0x80, 0x5E, 0xF1, 0x49, 0x06, 0x00, 0x00, 0x00, 0xF0, 0x23, 0x90, 0x2B, 0x01, 0x00, 0x00, 0x00, 0xC0, 0xF1, 0x8E, 0xA8, 0x00, 0x00, 0x00, 0x00},
		[8]int64{746127497, 280013252, 927093439, 191628495, 819757277, 422036858, 314114623, 706985072},
	},

	// bit width = 63
	{
		63,
		[]byte{0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
		[8]int64{0, 0, 0, 0, 0, 0, 0, 0},
	},
	{
		63,
		[]byte{0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x60, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x28, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0E, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
		[8]int64{0, 1, 2, 3, 4, 5, 6, 7},
	},
	{
		63,
		[]byte{0x07, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x30, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
		[8]int64{7, 6, 5, 4, 3, 2, 1, 0},
	},
	{
		63,
		[]byte{0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0x7F, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0x7F, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0x9F, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xBF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xD7, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xE7, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xF1, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF},
		[8]int64{9223372036854775807, 9223372036854775806, 9223372036854775805, 9223372036854775804, 9223372036854775803, 9223372036854775802, 9223372036854775801, 9223372036854775800},
	},
	{
		63,
		[]byte{0xE9, 0xB4, 0x99, 0x27, 0x00, 0x00, 0x00, 0x00, 0x78, 0x10, 0xCE, 0x01, 0x00, 0x00, 0x00, 0x40, 0xDD, 0xC6, 0x5A, 0x0C, 0x00, 0x00, 0x00, 0xA0, 0x7E, 0x94, 0x16, 0x00, 0x00, 0x00, 0x00, 0x90, 0xE2, 0x31, 0x14, 0x02, 0x00, 0x00, 0x00, 0x98, 0xED, 0x1F, 0x96, 0x01, 0x00, 0x00, 0x00, 0xC0, 0xD1, 0x8C, 0x7A, 0x00, 0x00, 0x00, 0x00, 0xC8, 0x39, 0xCC, 0x2F, 0x00, 0x00, 0x00, 0x00},
		[8]int64{664384745, 60563696, 829102965, 11838453, 558046761, 851705267, 514012272, 400956644},
	},

	// bit width = 64
	{
		64,
		[]byte{0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
		[8]int64{0, 0, 0, 0, 0, 0, 0, 0},
	},
	{
		64,
		[]byte{0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x05, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x06, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x07, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
		[8]int64{0, 1, 2, 3, 4, 5, 6, 7},
	},
	{
		64,
		[]byte{0x07, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x06, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x05, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
		[8]int64{7, 6, 5, 4, 3, 2, 1, 0},
	},
	{
		64,
		[]byte{0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFE, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFD, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFC, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFB, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFA, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xF9, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF},
		[8]int64{0, -1, -2, -3, -4, -5, -6, -7},
	},
	{
		64,
		[]byte{0xCA, 0xC6, 0x53, 0x33, 0x00, 0x00, 0x00, 0x00, 0xA6, 0xA5, 0x5B, 0x20, 0x00, 0x00, 0x00, 0x00, 0xD0, 0x1D, 0x50, 0x19, 0x00, 0x00, 0x00, 0x00, 0x65, 0xA4, 0x33, 0x34, 0x00, 0x00, 0x00, 0x00, 0x0A, 0xE6, 0x2A, 0x00, 0x00, 0x00, 0x00, 0x00, 0x17, 0xE2, 0xA7, 0x19, 0x00, 0x00, 0x00, 0x00, 0x7A, 0x75, 0x7C, 0x17, 0x00, 0x00, 0x00, 0x00, 0x58, 0xF2, 0x01, 0x0B, 0x00, 0x00, 0x00, 0x00},
		[8]int64{861128394, 542877094, 424680912, 875799653, 2811402, 430432791, 394032506, 184676952},
	},
}
