// Copyright (C) 2022 The Flux authors
//
// This Source Code Form is subject to the terms of the Mozilla Public
// License, v. 2.0. If a copy of the MPL was not distributed with this
// file, You can obtain one at https://mozilla.org/MPL/2.0/.

package sops

import (
	"bytes"
	"encoding/binary"
	"fmt"
	"github.com/dimchansky/utfbom"
	"io/ioutil"
	"unicode/utf16"

	"github.com/Azure/azure-sdk-for-go/sdk/azcore"
	"github.com/Azure/azure-sdk-for-go/sdk/azcore/cloud"
	"github.com/Azure/azure-sdk-for-go/sdk/azidentity"
	"sigs.k8s.io/yaml"
)

// LoadAADConfigFromBytes attempts to load the given bytes into the given AADConfig.
// By first decoding it if UTF-16, and then unmarshalling it into the given struct.
// It returns an error for any failure.
func LoadAADConfigFromBytes(b []byte, s *AADConfig) error {
	b, err := decode(b)
	if err != nil {
		return fmt.Errorf("failed to decode Azure authentication file bytes: %w", err)
	}
	if err = yaml.Unmarshal(b, s); err != nil {
		err = fmt.Errorf("failed to unmarshal Azure authentication file: %w", err)
	}
	return err
}

// AADConfig contains the selection of fields from an Azure authentication file
// required for Active Directory authentication.
type AADConfig struct {
	AZConfig
	TenantID                   string `json:"tenantId,omitempty"`
	ClientID                   string `json:"clientId,omitempty"`
	ClientSecret               string `json:"clientSecret,omitempty"`
	ClientCertificate          string `json:"clientCertificate,omitempty"`
	ClientCertificatePassword  string `json:"clientCertificatePassword,omitempty"`
	ClientCertificateSendChain bool   `json:"clientCertificateSendChain,omitempty"`
	AuthorityHost              string `json:"authorityHost,omitempty"`
}

// AZConfig contains the Service Principal fields as generated by `az`.
// Ref: https://docs.microsoft.com/en-us/azure/aks/kubernetes-service-principal?tabs=azure-cli#manually-create-a-service-principal
type AZConfig struct {
	AppID    string `json:"appId,omitempty"`
	Tenant   string `json:"tenant,omitempty"`
	Password string `json:"password,omitempty"`
}

// TokenFromAADConfig attempts to construct a Token using the AADConfig values.
// It detects credentials in the following order:
//
//   - azidentity.ClientSecretCredential when `tenantId`, `clientId` and
//     `clientSecret` fields are found.
//   - azidentity.ClientCertificateCredential when `tenantId`,
//     `clientCertificate` (and optionally `clientCertificatePassword`) fields
//     are found.
//   - azidentity.ClientSecretCredential when AZConfig fields are found.
//   - azidentity.ManagedIdentityCredential for a User ID, when a `clientId`
//     field but no `tenantId` is found.
//
// If no set of credentials is found or the azcore.TokenCredential can not be
// created, an error is returned.
func TokenFromAADConfig(c AADConfig) (_ azcore.TokenCredential, err error) {
	var token azcore.TokenCredential
	if c.TenantID != "" && c.ClientID != "" {
		if c.ClientSecret != "" {
			if token, err = azidentity.NewClientSecretCredential(c.TenantID, c.ClientID, c.ClientSecret, &azidentity.ClientSecretCredentialOptions{
				ClientOptions: azcore.ClientOptions{
					Cloud: c.GetCloudConfig(),
				},
			}); err != nil {
				return
			}
			return token, nil
		}
		if c.ClientCertificate != "" {
			certs, pk, err := azidentity.ParseCertificates([]byte(c.ClientCertificate), []byte(c.ClientCertificatePassword))
			if err != nil {
				return nil, err
			}
			if token, err = azidentity.NewClientCertificateCredential(c.TenantID, c.ClientID, certs, pk, &azidentity.ClientCertificateCredentialOptions{
				SendCertificateChain: c.ClientCertificateSendChain,
				ClientOptions: azcore.ClientOptions{
					Cloud: c.GetCloudConfig(),
				},
			}); err != nil {
				return nil, err
			}
			return token, nil
		}
	}

	switch {
	case c.Tenant != "" && c.AppID != "" && c.Password != "":
		if token, err = azidentity.NewClientSecretCredential(c.Tenant, c.AppID, c.Password, &azidentity.ClientSecretCredentialOptions{
			ClientOptions: azcore.ClientOptions{
				Cloud: c.GetCloudConfig(),
			},
		}); err != nil {
			return
		}
		return token, nil
	case c.ClientID != "":
		if token, err = azidentity.NewManagedIdentityCredential(&azidentity.ManagedIdentityCredentialOptions{
			ID: azidentity.ClientID(c.ClientID),
		}); err != nil {
			return
		}
		return token, nil
	default:
		return nil, fmt.Errorf("invalid data: requires a '%s' field, a combination of '%s', '%s' and '%s', or '%s', '%s' and '%s'",
			"clientId", "tenantId", "clientId", "clientSecret", "tenantId", "clientId", "clientCertificate")
	}
}

// GetCloudConfig returns a cloud.Configuration with the AuthorityHost, or the
// Azure Public Cloud default.
func (s AADConfig) GetCloudConfig() cloud.Configuration {
	if s.AuthorityHost != "" {
		return cloud.Configuration{
			ActiveDirectoryAuthorityHost: s.AuthorityHost,
			Services:                     map[cloud.ServiceName]cloud.ServiceConfiguration{},
		}
	}
	return cloud.AzurePublic
}

func decode(b []byte) ([]byte, error) {
	reader, enc := utfbom.Skip(bytes.NewReader(b))
	switch enc {
	case utfbom.UTF16LittleEndian:
		u16 := make([]uint16, (len(b)/2)-1)
		err := binary.Read(reader, binary.LittleEndian, &u16)
		if err != nil {
			return nil, err
		}
		return []byte(string(utf16.Decode(u16))), nil
	case utfbom.UTF16BigEndian:
		u16 := make([]uint16, (len(b)/2)-1)
		err := binary.Read(reader, binary.BigEndian, &u16)
		if err != nil {
			return nil, err
		}
		return []byte(string(utf16.Decode(u16))), nil
	}
	return ioutil.ReadAll(reader)
}
