package cmd

import (
	"os"
	"strings"
	"testing"

	"github.com/gsamokovarov/assert"
	"github.com/gsamokovarov/jump/cli"
)

func TestShellCmd_Bash(t *testing.T) {
	output := capture(&os.Stdout, func() {
		shellCmd(cli.Args{"bash"}, nil)
	})

	expectedComment := `# Put the line below in ~/.bashrc or ~/bash_profile:
#
#   eval "$(jump shell bash)"
#
# The following lines are autogenerated:`

	assert.True(t, strings.HasPrefix(output, expectedComment))
}

func TestShellCmd_Zsh(t *testing.T) {
	output := capture(&os.Stdout, func() {
		shellCmd(cli.Args{"zsh"}, nil)
	})

	expectedComment := `# Put the line below in ~/.zshrc:
#
#   eval "$(jump shell zsh)"
#
# The following lines are autogenerated:`

	assert.True(t, strings.HasPrefix(output, expectedComment))
}

func TestShellCmd_Fish(t *testing.T) {
	output := capture(&os.Stdout, func() {
		shellCmd(cli.Args{"fish"}, nil)
	})

	expectedComment := `# Put the line below in ~/.config/fish/config.fish:
#
#   jump shell fish | source
#
# The following lines are autogenerated:`

	assert.True(t, strings.HasPrefix(output, expectedComment))
}

func TestShellCmd_Pwsh(t *testing.T) {
	output := capture(&os.Stdout, func() {
		shellCmd(cli.Args{"pwsh"}, nil)
	})

	expectedComment := `# Put the line below in your PowerShell profile (Microsoft.PowerShell_profile.ps1):
#
#   Invoke-Expression (&jump shell pwsh | Out-String)
#
# You can locate your profile by typing $PROFILE in PowerShell.
#
# The following lines are autogenerated:`

	assert.True(t, strings.HasPrefix(output, expectedComment))
}

func TestShellCmd_Nushell(t *testing.T) {
	output := capture(&os.Stdout, func() {
		shellCmd(cli.Args{"nu"}, nil)
	})

	expectedComment := `# Put the line below in your Nushell config file:
#
#   jump shell nushell | save --append $nu.config-path
#
# The following lines are autogenerated:`

	assert.True(t, strings.HasPrefix(output, expectedComment))
}

func TestShellCmd_Bind(t *testing.T) {
	output := capture(&os.Stdout, func() {
		shellCmd(cli.Args{"bash", "--bind=z"}, nil)
	})

	expectedComment := `# Put the line below in ~/.bashrc or ~/bash_profile:
#
#   eval "$(jump shell bash)"
#
# The following lines are autogenerated:`

	assert.True(t, strings.HasPrefix(output, expectedComment))
}
