/*
Copyright © 2018-2025 blacktop

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
THE SOFTWARE.
*/
package cmd

import (
	"fmt"
	"net/url"
	"os"
	"path/filepath"
	"slices"
	"strconv"
	"strings"
	"text/tabwriter"

	"github.com/MakeNowJust/heredoc/v2"
	"github.com/apex/log"
	"github.com/blacktop/ipsw/internal/demangle"
	"github.com/blacktop/ipsw/pkg/crashlog"
	"github.com/blacktop/ipsw/pkg/dyld"
	"github.com/blacktop/ipsw/pkg/info"
	"github.com/spf13/cobra"
	"github.com/spf13/viper"
)

func init() {
	rootCmd.AddCommand(symbolicateCmd)

	symbolicateCmd.Flags().BoolP("all", "a", false, "Show all threads in crashlog")
	symbolicateCmd.Flags().BoolP("running", "r", false, "Show all running (TH_RUN) threads in crashlog")
	symbolicateCmd.Flags().StringP("proc", "p", "", "Filter crashlog by process name")
	symbolicateCmd.Flags().BoolP("unslide", "u", false, "Unslide user-space addresses for static analysis (kernel frames are always unslid)")
	symbolicateCmd.Flags().String("kc-slide", "", "Apply custom KASLR slide to kernelcache frames for live debugging (hex, e.g. 0x14f74000)")
	symbolicateCmd.Flags().String("dsc-slide", "", "Apply custom slide to dyld_shared_cache frames for live debugging (hex, e.g. 0x1a000000)")
	symbolicateCmd.Flags().BoolP("demangle", "d", false, "Demangle symbol names")
	symbolicateCmd.Flags().Bool("hex", false, "Display function offsets in hexadecimal")
	symbolicateCmd.Flags().Bool("peek", false, "Show disassembly instructions around each panicked frame")
	symbolicateCmd.Flags().Int("peek-count", 5, "Number of instructions to show with --peek (centered on frame, respects function boundaries)")
	symbolicateCmd.Flags().StringP("server", "s", "", "Symbol Server DB URL")
	symbolicateCmd.Flags().String("pem-db", "", "AEA pem DB JSON file")
	symbolicateCmd.Flags().String("signatures", "", "Path to signatures folder")
	symbolicateCmd.Flags().StringP("extra", "x", "", "Path to folder with extra files for symbolication")
	symbolicateCmd.Flags().Bool("ida", false, "Generate IDAPython script to mark panic frames in IDA Pro")
	// symbolicateCmd.Flags().String("cache", "", "Path to .a2s addr to sym cache file (speeds up analysis)")
	symbolicateCmd.MarkZshCompPositionalArgumentFile(2, "dyld_shared_cache*")
	viper.BindPFlag("symbolicate.all", symbolicateCmd.Flags().Lookup("all"))
	viper.BindPFlag("symbolicate.running", symbolicateCmd.Flags().Lookup("running"))
	viper.BindPFlag("symbolicate.proc", symbolicateCmd.Flags().Lookup("proc"))
	viper.BindPFlag("symbolicate.unslide", symbolicateCmd.Flags().Lookup("unslide"))
	viper.BindPFlag("symbolicate.kc-slide", symbolicateCmd.Flags().Lookup("kc-slide"))
	viper.BindPFlag("symbolicate.dsc-slide", symbolicateCmd.Flags().Lookup("dsc-slide"))
	viper.BindPFlag("symbolicate.demangle", symbolicateCmd.Flags().Lookup("demangle"))
	viper.BindPFlag("symbolicate.hex", symbolicateCmd.Flags().Lookup("hex"))
	viper.BindPFlag("symbolicate.peek", symbolicateCmd.Flags().Lookup("peek"))
	viper.BindPFlag("symbolicate.peek-count", symbolicateCmd.Flags().Lookup("peek-count"))
	viper.BindPFlag("symbolicate.server", symbolicateCmd.Flags().Lookup("server"))
	viper.BindPFlag("symbolicate.pem-db", symbolicateCmd.Flags().Lookup("pem-db"))
	viper.BindPFlag("symbolicate.signatures", symbolicateCmd.Flags().Lookup("signatures"))
	viper.BindPFlag("symbolicate.extra", symbolicateCmd.Flags().Lookup("extra"))
	viper.BindPFlag("symbolicate.ida", symbolicateCmd.Flags().Lookup("ida"))
}

// TODO: handle all edge cases from `/Applications/Xcode.app/Contents/SharedFrameworks/DVTFoundation.framework/Versions/A/Resources/symbolicatecrash` and handle spindumps etc

// symbolicateCmd represents the symbolicate command
var symbolicateCmd = &cobra.Command{
	Use:     "symbolicate <CRASHLOG> [IPSW|DSC]",
	Aliases: []string{"sym"},
	Short:   "Symbolicate ARM 64-bit crash logs (similar to Apple's symbolicatecrash)",
	Example: heredoc.Doc(`
	# Symbolicate a panic crashlog (BugType=210) with an IPSW
	❯ ipsw symbolicate panic-full-2024-03-21-004704.000.ips iPad_Pro_HFR_17.4_21E219_Restore.ipsw

	# Show disassembly around panic frames with --peek (default 5 instructions)
	❯ ipsw symbolicate panic.ips firmware.ipsw --peek

	# Show more instructions around panic frames (10 instructions, centered on frame)
	❯ ipsw symbolicate panic.ips firmware.ipsw --peek --peek-count 10
	  # Note: If frame is at function start, extra instructions shift to after the frame

	# Unslide user-space addresses for static analysis (kernel frames are always unslid)
	❯ ipsw symbolicate panic.ips firmware.ipsw --unslide
	  # Note: Kernel frame addresses are already KASLR-unslid and match static disassemblers
	  # The --unslide flag only affects user-space frames (processes like launchd, SpringBoard, etc.)

	# Apply custom KASLR slide to kernelcache frames for lldb live debugging
	❯ ipsw symbolicate panic.ips firmware.ipsw --kc-slide 0x14f74000
	  # Useful when reproducing a crash with a different KASLR slide
	  # Shows runtime addresses you can use with lldb breakpoints

	# Apply custom slide to dyld_shared_cache frames for lldb live debugging
	❯ ipsw symbolicate panic.ips firmware.ipsw --dsc-slide 0x1a000000
	  # For debugging user-space crashes where DSC was loaded at a different address

	# Combine both slides for full runtime address mapping
	❯ ipsw symbolicate panic.ips firmware.ipsw --kc-slide 0x14f74000 --dsc-slide 0x1a000000

	# Generate IDAPython script to mark panic frames in IDA Pro
	❯ ipsw symbolicate panic.ips firmware.ipsw --ida
	  # Outputs panic.ips.kc.ida.py for kernel frames (load in IDA with kernelcache)
	  # Outputs panic.ips.dsc.ida.py for DSC frames if present (load in IDA with DSC image)

	# Pretty print a crashlog (BugType=309) these are usually symbolicated by the OS
	❯ ipsw symbolicate --color Delta-2024-04-20-135807.ips

	# Symbolicate an old style crashlog (BugType=109) requiring a dyld_shared_cache
	❯ ipsw symbolicate Delta-2024-04-20-135807.ips dyld_shared_cache
	  ⨯ please supply a dyld_shared_cache for iPhone13,3 running 14.5 (18E5154f)`),
	Args:          cobra.MinimumNArgs(1),
	SilenceErrors: true,
	RunE: func(cmd *cobra.Command, args []string) error {
		/* flags */
		all := viper.GetBool("symbolicate.all")
		running := viper.GetBool("symbolicate.running")
		proc := viper.GetString("symbolicate.proc")
		unslide := viper.GetBool("symbolicate.unslide")
		// cacheFile, _ := cmd.Flags().GetString("cache")
		demangleFlag := viper.GetBool("symbolicate.demangle")
		asHex := viper.GetBool("symbolicate.hex")
		peek := viper.GetBool("symbolicate.peek")
		peekCount := viper.GetInt("symbolicate.peek-count")
		if peekCount < 1 {
			peekCount = 1
		}
		pemDB := viper.GetString("symbolicate.pem-db")
		signaturesDir := viper.GetString("symbolicate.signatures")
		extrasDir := viper.GetString("symbolicate.extra")
		idaScript := viper.GetBool("symbolicate.ida")
		kcSlideStr := viper.GetString("symbolicate.kc-slide")
		dscSlideStr := viper.GetString("symbolicate.dsc-slide")
		/* parse slide values (base 0 auto-detects hex 0x, octal 0o, or decimal) */
		var kcSlide uint64
		if kcSlideStr != "" {
			var err error
			kcSlide, err = strconv.ParseUint(kcSlideStr, 0, 64)
			if err != nil {
				return fmt.Errorf("invalid --kc-slide value %q: must be a number (e.g., 0x14f74000 or 351748096): %v", kcSlideStr, err)
			}
		}
		var dscSlide uint64
		if dscSlideStr != "" {
			var err error
			dscSlide, err = strconv.ParseUint(dscSlideStr, 0, 64)
			if err != nil {
				return fmt.Errorf("invalid --dsc-slide value %q: must be a number (e.g., 0x1a000000 or 436207616): %v", dscSlideStr, err)
			}
		}
		/* validate flags */
		if (Verbose || all) && len(proc) > 0 {
			return fmt.Errorf("cannot use --verbose OR --all WITH --proc")
		}
		if unslide && (kcSlide != 0 || dscSlide != 0) {
			return fmt.Errorf("cannot use --unslide with --kc-slide or --dsc-slide (they are mutually exclusive)")
		}

		hdr, err := crashlog.ParseHeader(args[0])
		if err != nil {
			log.WithError(err).Error("failed to parse crashlog header")
			log.Warn("trying to parse as IPS crashlog (BugType=210)")
			hdr = &crashlog.IpsMetadata{
				BugType: "109",
			}
		}

		switch hdr.BugType {
		case "210", "288", "309": // NEW JSON STYLE CRASHLOG
			ips, err := crashlog.OpenIPS(args[0], &crashlog.Config{
				All:           all || Verbose,
				Running:       running,
				Process:       proc,
				Unslid:        unslide,
				KernelSlide:   kcSlide,
				DSCSlide:      dscSlide,
				Demangle:      demangleFlag,
				Hex:           asHex,
				Peek:          peek,
				PeekCount:     peekCount,
				PemDB:         pemDB,
				SignaturesDir: signaturesDir,
				ExtrasDir:     extrasDir,
				IDAScript:     idaScript,
				Verbose:       Verbose,
			})
			if err != nil {
				return fmt.Errorf("failed to parse IPS file: %v", err)
			}

			if len(args) < 2 && (hdr.BugType == "210" || hdr.BugType == "288") {
				// --peek and --signatures require an IPSW
				if peek {
					return fmt.Errorf("--peek requires an IPSW to show disassembly instructions")
				}
				if signaturesDir != "" {
					return fmt.Errorf("--signatures requires an IPSW to symbolicate")
				}
				if viper.GetString("symbolicate.server") != "" {
					u, err := url.ParseRequestURI(viper.GetString("symbolicate.server"))
					if err != nil {
						return fmt.Errorf("failed to parse symbol server URL: %v", err)
					}
					if u.Scheme == "" || u.Host == "" {
						return fmt.Errorf("invalid symbol server URL: %s (needs a valid schema AND host)", u.String())
					}
					log.WithField("server", u.String()).Info("Symbolicating 210 Panic with Symbol Server")
					if err := ips.Symbolicate210WithDatabase(u.String()); err != nil {
						return err
					}
				} else {
					log.Warnf("please supply %s %s IPSW for symbolication", ips.Payload.Product, ips.Header.OsVersion)
				}
			} else {
				// TODO: use IPSW to populate symbol server if both are supplied
				if hdr.BugType == "210" || hdr.BugType == "288" {
					/* validate IPSW */
					i, err := info.Parse(args[1])
					if err != nil {
						return err
					}
					if i.Plists.BuildManifest.ProductVersion != ips.Header.Version() ||
						i.Plists.BuildManifest.ProductBuildVersion != ips.Header.Build() ||
						!slices.Contains(i.Plists.Restore.SupportedProductTypes, ips.Payload.Product) {
						return fmt.Errorf("supplied IPSW %s does NOT match crashlog: NEED %s; %s (%s), GOT %s; %s (%s)",
							filepath.Base(args[1]),
							ips.Payload.Product, ips.Header.Version(), ips.Header.Build(),
							strings.Join(i.Plists.Restore.SupportedProductTypes, ", "),
							i.Plists.BuildManifest.ProductVersion, i.Plists.BuildManifest.ProductBuildVersion,
						)
					}
					if err := ips.Symbolicate210(filepath.Clean(args[1])); err != nil {
						return err
					}
				}
			}
			fmt.Println(ips)
		case "109": // OLD STYLE CRASHLOG
			crashLog, err := crashlog.Open(args[0])
			if err != nil {
				return err
			}
			defer crashLog.Close()

			if len(args) > 1 {
				dscPath := filepath.Clean(args[1])

				fileInfo, err := os.Lstat(dscPath)
				if err != nil {
					return fmt.Errorf("file %s does not exist", dscPath)
				}

				// Check if file is a symlink
				if fileInfo.Mode()&os.ModeSymlink != 0 {
					symlinkPath, err := os.Readlink(dscPath)
					if err != nil {
						return fmt.Errorf("failed to read symlink %s: %v", dscPath, err)
					}
					// TODO: this seems like it would break
					linkParent := filepath.Dir(dscPath)
					linkRoot := filepath.Dir(linkParent)

					dscPath = filepath.Join(linkRoot, symlinkPath)
				}

				f, err := dyld.Open(dscPath)
				if err != nil {
					return err
				}
				defer f.Close()

				// if len(cacheFile) == 0 {
				// 	cacheFile = dscPath + ".a2s"
				// }
				// if err := f.OpenOrCreateA2SCache(cacheFile); err != nil {
				// 	return err
				// }

				// Symbolicate the crashing thread's backtrace
				for idx, bt := range crashLog.Threads[crashLog.CrashedThread].BackTrace {
					image, err := f.Image(bt.Image.Name)
					if err != nil {
						log.Errorf(err.Error())
						crashLog.Threads[crashLog.CrashedThread].BackTrace[idx].Symbol = "?"
						continue
					}
					// calculate slide
					bt.Image.Slide = bt.Image.Start - image.CacheImageTextInfo.LoadAddress
					unslidAddr := bt.Address - bt.Image.Slide

					m, err := image.GetMacho()
					if err != nil {
						return err
					}
					defer m.Close()

					// check if symbol is cached
					if symName, ok := f.AddressToSymbol[unslidAddr]; ok {
						if demangleFlag {
							symName = demangle.Do(symName, false, false)
						}
						crashLog.Threads[crashLog.CrashedThread].BackTrace[idx].Symbol = symName
						continue
					}

					if fn, err := m.GetFunctionForVMAddr(unslidAddr); err == nil {
						if symName, ok := f.AddressToSymbol[fn.StartAddr]; ok {
							if demangleFlag {
								symName = demangle.Do(symName, false, false)
							}
							crashLog.Threads[crashLog.CrashedThread].BackTrace[idx].Symbol = fmt.Sprintf("%s + %d", symName, unslidAddr-fn.StartAddr)
							continue
						}
					}

					if err := image.Analyze(); err != nil {
						return fmt.Errorf("failed to analyze image %s; %v", image.Name, err)
					}

					for _, patch := range image.PatchableExports {
						addr, err := image.GetVMAddress(uint64(patch.GetImplOffset()))
						if err != nil {
							return err
						}
						f.AddressToSymbol[addr] = patch.GetName()
					}

					if symName, ok := f.AddressToSymbol[unslidAddr]; ok {
						if demangleFlag {
							symName = demangle.Do(symName, false, false)
						}
						crashLog.Threads[crashLog.CrashedThread].BackTrace[idx].Symbol = symName
						continue
					}

					if fn, err := m.GetFunctionForVMAddr(unslidAddr); err == nil {
						if symName, ok := f.AddressToSymbol[fn.StartAddr]; ok {
							if demangleFlag {
								symName = demangle.Do(symName, false, false)
							}
							crashLog.Threads[crashLog.CrashedThread].BackTrace[idx].Symbol = fmt.Sprintf("%s + %d", symName, unslidAddr-fn.StartAddr)
						}
					}

				}

				fmt.Println(crashLog)

				fmt.Printf("Thread %d name: %s\n",
					crashLog.CrashedThread,
					crashLog.Threads[crashLog.CrashedThread].Name)
				fmt.Printf("Thread %d Crashed:\n", crashLog.CrashedThread)
				w := tabwriter.NewWriter(os.Stdout, 0, 0, 1, ' ', 0)
				for _, bt := range crashLog.Threads[crashLog.CrashedThread].BackTrace {
					if unslide {
						fmt.Fprintf(w, "\t%2d: %s\t%#x\t%s\n", bt.FrameNum, bt.Image.Name, bt.Address-bt.Image.Slide, bt.Symbol)
					} else {
						fmt.Fprintf(w, "\t%2d: %s\t(slide=%#x)\t%#x\t%s\n", bt.FrameNum, bt.Image.Name, bt.Image.Slide, bt.Address, bt.Symbol)
					}
				}
				w.Flush()
				var note string
				if unslide {
					if len(crashLog.Threads[crashLog.CrashedThread].BackTrace) > 0 {
						var slide uint64
						if img, err := f.GetImageContainingVMAddr(crashLog.Threads[crashLog.CrashedThread].State["pc"]); err == nil {
							found := false
							for _, bt := range crashLog.Threads[crashLog.CrashedThread].BackTrace {
								if bt.Image.Name == img.Name {
									slide = bt.Image.Slide
									found = true
									break
								}
							}
							if !found {
								slide = crashLog.Threads[crashLog.CrashedThread].BackTrace[0].Image.Slide
							}
						}
						note = fmt.Sprintf(" (may contain slid addresses; slide=%#x)", slide)
					} else {
						note = " (may contain slid addresses)"
					}
				}
				fmt.Printf("\nThread %d State:%s\n%s\n", crashLog.CrashedThread, note, crashLog.Threads[crashLog.CrashedThread].State)
				// slide := crashLog.Threads[crashLog.CrashedThread].BackTrace[0].Image.Slide
				// for key, val := range crashLog.Threads[crashLog.CrashedThread].State {
				// 	unslid := val - slide
				// 	if sym, ok := f.AddressToSymbol[unslid]; ok {
				// 		fmt.Printf("%4v: %#016x %s\n", key, val, sym)
				// 	} else {
				// 		fmt.Printf("%4v: %#016x\n", key, val)
				// 	}
				// }
			} else {
				log.Errorf("please supply a dyld_shared_cache for %s running %s (%s)", crashLog.HardwareModel, crashLog.OSVersion, crashLog.OSBuild)
			}
		default:
			log.Errorf("unsupported crashlog type: %s - %s (notify author to add support)", hdr.BugType, hdr.BugTypeDesc)
		}

		return nil
	},
}
