package cmd

import (
	"github.com/spf13/cobra"

	"github.com/helmfile/helmfile/pkg/app"
	"github.com/helmfile/helmfile/pkg/config"
)

// NewApplyCmd returns apply subcmd
func NewApplyCmd(globalCfg *config.GlobalImpl) *cobra.Command {
	applyOptions := &config.ApplyOptions{}
	applyImpl := config.NewApplyImpl(globalCfg, applyOptions)

	cmd := &cobra.Command{
		Use:   "apply",
		Short: "Apply all resources from state file only when there are changes",
		RunE: func(cmd *cobra.Command, args []string) error {
			err := config.NewCLIConfigImpl(applyImpl.GlobalImpl)
			if err != nil {
				return err
			}

			if err := applyImpl.ValidateConfig(); err != nil {
				return err
			}

			a := app.New(applyImpl)
			return toCLIError(applyImpl.GlobalImpl, a.Apply(applyImpl))
		},
	}

	f := cmd.Flags()
	f.StringSliceVar(&applyImpl.ApplyOptions.Values, "values", nil, "additional value files to be merged into the command")
	f.IntVar(&applyImpl.ApplyOptions.Concurrency, "concurrency", 0, "maximum number of concurrent helm processes to run, 0 is unlimited")
	f.BoolVar(&applyImpl.ApplyOptions.Validate, "validate", false, "validate your manifests against the Kubernetes cluster you are currently pointing at. Note that this requires access to a Kubernetes cluster to obtain information necessary for validating, like the list of available API versions")
	f.IntVar(&applyImpl.ApplyOptions.Context, "context", 0, "output NUM lines of context around changes")
	f.StringVar(&applyImpl.ApplyOptions.Output, "output", "", "output format for diff plugin")
	f.BoolVar(&applyImpl.ApplyOptions.DetailedExitcode, "detailed-exitcode", false, "return a non-zero exit code 2 instead of 0 when there were changes detected AND the changes are synced successfully")
	f.StringVar(&applyImpl.ApplyOptions.Args, "args", "", "pass args to helm exec")
	f.BoolVar(&applyImpl.ApplyOptions.RetainValuesFiles, "retain-values-files", false, "DEPRECATED: Use skip-cleanup instead")
	f.BoolVar(&applyImpl.ApplyOptions.SkipCleanup, "skip-cleanup", false, "Stop cleaning up temporary values generated by helmfile and helm-secrets. Useful for debugging. Don't use in production for security")
	f.BoolVar(&applyImpl.ApplyOptions.SkipCRDs, "skip-crds", false, "if set, no CRDs will be installed on sync. By default, CRDs are installed if not already present")
	f.BoolVar(&applyImpl.ApplyOptions.SkipNeeds, "skip-needs", false, `do not automatically include releases from the target release's "needs" when --selector/-l flag is provided. Does nothing when when --selector/-l flag is not provided. Defaults to true when --include-needs or --include-transitive-needs is not provided`)
	f.BoolVar(&applyImpl.ApplyOptions.IncludeNeeds, "include-needs", false, `automatically include releases from the target release's "needs" when --selector/-l flag is provided. Does nothing when when --selector/-l flag is not provided`)
	f.BoolVar(&applyImpl.ApplyOptions.IncludeTransitiveNeeds, "include-transitive-needs", false, `like --include-needs, but also includes transitive needs (needs of needs). Does nothing when when --selector/-l flag is not provided. Overrides exclusions of other selectors and conditions.`)
	f.BoolVar(&applyImpl.ApplyOptions.SkipDiffOnInstall, "skip-diff-on-install", false, "Skips running helm-diff on releases being newly installed on this apply. Useful when the release manifests are too huge to be reviewed, or it's too time-consuming to diff at all0")
	f.BoolVar(&applyImpl.ApplyOptions.IncludeTests, "include-tests", false, "enable the diffing of the helm test hooks")
	f.StringArrayVar(&applyImpl.ApplyOptions.Suppress, "suppress", nil, "suppress specified Kubernetes objects in the diff output. Can be provided multiple times. For example: --suppress KeycloakClient --suppress VaultSecret")
	f.BoolVar(&applyImpl.ApplyOptions.SuppressSecrets, "suppress-secrets", false, "suppress secrets in the diff output. highly recommended to specify on CI/CD use-cases")
	f.BoolVar(&applyImpl.ApplyOptions.ShowSecrets, "show-secrets", false, "do not redact secret values in the diff output. should be used for debug purpose only")
	f.BoolVar(&applyImpl.ApplyOptions.SuppressDiff, "suppress-diff", false, "suppress diff in the output. Usable in new installs")
	f.BoolVar(&applyImpl.ApplyOptions.SkipDeps, "skip-deps", false, `skip running "helm repo update" and "helm dependency build"`)
	f.Bool("wait", false, `Override helmDefaults.wait setting "helm upgrade --install --wait"`)
	f.Bool("wait-for-jobs", false, `Override helmDefaults.waitForJobs setting "helm upgrade --install --wait-for-jobs"`)

	return cmd
}
