// Licensed to Elasticsearch B.V. under one or more contributor
// license agreements. See the NOTICE file distributed with
// this work for additional information regarding copyright
// ownership. Elasticsearch B.V. licenses this file to you under
// the Apache License, Version 2.0 (the "License"); you may
// not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing,
// software distributed under the License is distributed on an
// "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
// KIND, either express or implied.  See the License for the
// specific language governing permissions and limitations
// under the License.

package compare

import (
	"testing"

	"github.com/google/go-cmp/cmp"
	"github.com/google/go-cmp/cmp/cmpopts"

	bundlev1 "github.com/elastic/harp/api/gen/go/harp/bundle/v1"
)

var (
	ignoreOpts = []cmp.Option{
		cmpopts.IgnoreUnexported(bundlev1.Patch{}),
		cmpopts.IgnoreUnexported(bundlev1.PatchMeta{}),
		cmpopts.IgnoreUnexported(bundlev1.PatchSpec{}),
		cmpopts.IgnoreUnexported(bundlev1.PatchRule{}),
		cmpopts.IgnoreUnexported(bundlev1.PatchSecret{}),
		cmpopts.IgnoreUnexported(bundlev1.PatchSelector{}),
		cmpopts.IgnoreUnexported(bundlev1.PatchSelectorMatchPath{}),
		cmpopts.IgnoreUnexported(bundlev1.PatchPackage{}),
		cmpopts.IgnoreUnexported(bundlev1.PatchOperation{}),
	}
)

func TestToPatch(t *testing.T) {
	type args struct {
		oplog []DiffItem
	}
	tests := []struct {
		name    string
		args    args
		want    *bundlev1.Patch
		wantErr bool
	}{
		{
			name:    "nil",
			wantErr: true,
		},
		{
			name: "empty",
			args: args{
				oplog: []DiffItem{},
			},
			wantErr: true,
		},
		{
			name: "package added",
			args: args{
				oplog: []DiffItem{
					{Operation: Add, Type: "package", Path: "application/test"},
				},
			},
			wantErr: false,
			want: &bundlev1.Patch{
				ApiVersion: "harp.elastic.co/v1",
				Kind:       "BundlePatch",
				Meta: &bundlev1.PatchMeta{
					Name:        "autogenerated-patch",
					Description: "Patch generated from oplog",
				},
				Spec: &bundlev1.PatchSpec{
					Rules: []*bundlev1.PatchRule{},
				},
			},
		},
		{
			name: "package removed",
			args: args{
				oplog: []DiffItem{
					{Operation: Remove, Type: "package", Path: "application/test"},
				},
			},
			wantErr: false,
			want: &bundlev1.Patch{
				ApiVersion: "harp.elastic.co/v1",
				Kind:       "BundlePatch",
				Meta: &bundlev1.PatchMeta{
					Name:        "autogenerated-patch",
					Description: "Patch generated from oplog",
				},
				Spec: &bundlev1.PatchSpec{
					Rules: []*bundlev1.PatchRule{
						{
							Selector: &bundlev1.PatchSelector{
								MatchPath: &bundlev1.PatchSelectorMatchPath{
									Strict: "application/test",
								},
							},
							Package: &bundlev1.PatchPackage{
								Remove: true,
							},
						},
					},
				},
			},
		},
		{
			name: "secret added",
			args: args{
				oplog: []DiffItem{
					{Operation: Add, Type: "secret", Path: "application/test#key1", Value: "payload"},
					{Operation: Add, Type: "secret", Path: "application/test#key2", Value: "payload"},
				},
			},
			wantErr: false,
			want: &bundlev1.Patch{
				ApiVersion: "harp.elastic.co/v1",
				Kind:       "BundlePatch",
				Meta: &bundlev1.PatchMeta{
					Name:        "autogenerated-patch",
					Description: "Patch generated from oplog",
				},
				Spec: &bundlev1.PatchSpec{
					Rules: []*bundlev1.PatchRule{
						{
							Selector: &bundlev1.PatchSelector{
								MatchPath: &bundlev1.PatchSelectorMatchPath{
									Strict: "application/test",
								},
							},
							Package: &bundlev1.PatchPackage{
								Data: &bundlev1.PatchSecret{
									Kv: &bundlev1.PatchOperation{
										Add: map[string]string{
											"key1": "payload",
											"key2": "payload",
										},
									},
								},
							},
						},
					},
				},
			},
		},
		{
			name: "secret updated",
			args: args{
				oplog: []DiffItem{
					{Operation: Add, Type: "secret", Path: "application/test#key1", Value: "payload"},
					{Operation: Replace, Type: "secret", Path: "application/test#key2", Value: "payload"},
				},
			},
			wantErr: false,
			want: &bundlev1.Patch{
				ApiVersion: "harp.elastic.co/v1",
				Kind:       "BundlePatch",
				Meta: &bundlev1.PatchMeta{
					Name:        "autogenerated-patch",
					Description: "Patch generated from oplog",
				},
				Spec: &bundlev1.PatchSpec{
					Rules: []*bundlev1.PatchRule{
						{
							Selector: &bundlev1.PatchSelector{
								MatchPath: &bundlev1.PatchSelectorMatchPath{
									Strict: "application/test",
								},
							},
							Package: &bundlev1.PatchPackage{
								Data: &bundlev1.PatchSecret{
									Kv: &bundlev1.PatchOperation{
										Add: map[string]string{
											"key1": "payload",
										},
										Update: map[string]string{
											"key2": "payload",
										},
									},
								},
							},
						},
					},
				},
			},
		},
		{
			name: "secret removed",
			args: args{
				oplog: []DiffItem{
					{Operation: Add, Type: "secret", Path: "application/test#key1", Value: "payload"},
					{Operation: Replace, Type: "secret", Path: "application/test#key2", Value: "payload"},
					{Operation: Remove, Type: "secret", Path: "application/test#key3"},
				},
			},
			wantErr: false,
			want: &bundlev1.Patch{
				ApiVersion: "harp.elastic.co/v1",
				Kind:       "BundlePatch",
				Meta: &bundlev1.PatchMeta{
					Name:        "autogenerated-patch",
					Description: "Patch generated from oplog",
				},
				Spec: &bundlev1.PatchSpec{
					Rules: []*bundlev1.PatchRule{
						{
							Selector: &bundlev1.PatchSelector{
								MatchPath: &bundlev1.PatchSelectorMatchPath{
									Strict: "application/test",
								},
							},
							Package: &bundlev1.PatchPackage{
								Data: &bundlev1.PatchSecret{
									Kv: &bundlev1.PatchOperation{
										Add: map[string]string{
											"key1": "payload",
										},
										Update: map[string]string{
											"key2": "payload",
										},
										Remove: []string{
											"key3",
										},
									},
								},
							},
						},
					},
				},
			},
		},
	}
	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			got, err := ToPatch(tt.args.oplog)
			if (err != nil) != tt.wantErr {
				t.Errorf("ToPatch() error = %v, wantErr %v", err, tt.wantErr)
				return
			}
			if diff := cmp.Diff(got, tt.want, ignoreOpts...); diff != "" {
				t.Errorf("%q. ToPatch():\n-got/+want\ndiff %s", tt.name, diff)
			}
		})
	}
}
