package v3

// Vulnerability represents the minimum data fields necessary to perform package-to-vulnerability matching. This can represent a CVE, 3rd party advisory, or any source that relates back to a CVE.
type Vulnerability struct {
	ID                     string                   // The identifier of the vulnerability or advisory
	PackageName            string                   // The name of the package that is vulnerable
	Namespace              string                   // The ecosystem where the package resides
	VersionConstraint      string                   // The version range which the given package is vulnerable
	VersionFormat          string                   // The format which all version fields should be interpreted as
	CPEs                   []string                 // The CPEs which are considered vulnerable
	RelatedVulnerabilities []VulnerabilityReference // Other Vulnerabilities that are related to this one (e.g. GHSA relate to CVEs, or how distro CVE relates to NVD record)
	Fix                    Fix                      // All information about fixed versions
	Advisories             []Advisory               // Any vendor advisories about fixes or other notifications about this vulnerability
}

type VulnerabilityReference struct {
	ID        string
	Namespace string
}

//nolint:gocognit
func (v *Vulnerability) Equal(vv Vulnerability) bool {
	equal := v.ID == vv.ID &&
		v.PackageName == vv.PackageName &&
		v.Namespace == vv.Namespace &&
		v.VersionConstraint == vv.VersionConstraint &&
		v.VersionFormat == vv.VersionFormat &&
		len(v.CPEs) == len(vv.CPEs) &&
		len(v.RelatedVulnerabilities) == len(vv.RelatedVulnerabilities) &&
		len(v.Advisories) == len(vv.Advisories) &&
		v.Fix.State == vv.Fix.State &&
		len(v.Fix.Versions) == len(vv.Fix.Versions)

	if !equal {
		return false
	}
	for idx, cpe := range v.CPEs {
		if cpe != vv.CPEs[idx] {
			return false
		}
	}
	for idx, item := range v.CPEs {
		if item != vv.CPEs[idx] {
			return false
		}
	}
	for idx, item := range v.RelatedVulnerabilities {
		if item != vv.RelatedVulnerabilities[idx] {
			return false
		}
	}
	for idx, item := range v.Advisories {
		if item != vv.Advisories[idx] {
			return false
		}
	}
	for idx, item := range v.Fix.Versions {
		if item != vv.Fix.Versions[idx] {
			return false
		}
	}

	return true
}
