package enumerator

import (
	"io/fs"
	"os"
	"path/filepath"
	"strings"

	"github.com/cloudskiff/driftctl/pkg/iac/config"
)

type FileEnumeratorConfig struct {
	Bucket *string
	Prefix *string
}

type FileEnumerator struct {
	config config.SupplierConfig
}

func NewFileEnumerator(config config.SupplierConfig) *FileEnumerator {
	return &FileEnumerator{
		config,
	}
}

// File enumeration does not follow symlinks.
// We may use something like this https://pkg.go.dev/github.com/facebookgo/symwalk
// to follow links, but it allows infinite loop so be careful !
// If a symlink is given as root path, we will follow it, but symlinks under this path
// will not be resolved.
func (s *FileEnumerator) Enumerate() ([]string, error) {
	path := s.config.Path
	info, err := os.Lstat(path)
	if err != nil {
		return nil, err
	}

	// if we got a symlink, use its destination
	if info.Mode()&os.ModeSymlink != 0 {
		destination, err := filepath.EvalSymlinks(path)
		if err != nil {
			return nil, err
		}
		path = destination
		info, err = os.Stat(destination)
		if err != nil {
			return nil, err
		}
	}

	if info != nil && !info.IsDir() {
		return []string{path}, nil
	}

	keys := make([]string, 0)

	err = filepath.WalkDir(path, func(path string, d fs.DirEntry, err error) error {
		// Not tested, we should remove a file or folder after WalkDir has enumerated the whole tree in memory
		// This edge case does not really need to be covered by tests
		if err != nil {
			return err
		}
		// Skip symlinks
		if d.Type()&os.ModeSymlink != 0 {
			return nil
		}

		// Ignore .backup files generated by terraform
		if strings.HasSuffix(path, ".backup") {
			return nil
		}

		if !d.IsDir() {
			keys = append(keys, path)
		}
		return nil
	})

	return keys, err
}
