// Copyright 2019 Dolthub, Inc.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.
//
// This file incorporates work covered by the following copyright and
// permission notice:
//
// Copyright 2016 Attic Labs, Inc. All rights reserved.
// Licensed under the Apache License, version 2.0:
// http://www.apache.org/licenses/LICENSE-2.0

package nbs

import (
	"context"
	"errors"
	"fmt"
	"io"
	"io/ioutil"
	"os"
	"path/filepath"
	"strings"
	"time"

	"github.com/dolthub/fslock"

	"github.com/dolthub/dolt/go/libraries/utils/file"
	"github.com/dolthub/dolt/go/store/chunks"
	"github.com/dolthub/dolt/go/store/hash"
	"github.com/dolthub/dolt/go/store/util/tempfiles"
)

const (
	manifestFileName = "manifest"
	lockFileName     = "LOCK"

	storageVersion4 = "4"

	prefixLen = 5
)

var ErrUnreadableManifest = errors.New("could not read file manifest")

type manifestParser func(r io.Reader) (manifestContents, error)
type manifestWriter func(temp io.Writer, contents manifestContents) error
type manifestChecker func(upstream, contents manifestContents) error

// ParseManifest parses s a manifest file from the supplied reader
func ParseManifest(version int, r io.Reader) (ManifestInfo, error) {
	switch version {
	case 4:
		return fileManifestV4{}.parseManifest(r)
	case 5:
		return fileManifestV5{}.parseManifest(r)
	default:
		return nil, fmt.Errorf("unknown manifest version: %d", version)
	}
}

func MaybeMigrateFileManifest(ctx context.Context, dir string) (bool, error) {
	_, err := os.Stat(filepath.Join(dir, manifestFileName))
	if os.IsNotExist(err) {
		// no manifest exists, no need to migrate
		return false, nil
	} else if err != nil {
		return false, err
	}

	fm5 := fileManifestV5{dir}
	ok, _, err := fm5.ParseIfExists(ctx, &Stats{}, nil)
	if ok && err == nil {
		// on v5, no need to migrate
		return false, nil
	}

	fm4 := fileManifestV4{dir}
	ok, contents, err := fm4.ParseIfExists(ctx, &Stats{}, nil)
	if err != nil {
		return false, err
	}
	if !ok {
		// expected v4 or v5
		return false, ErrUnreadableManifest
	}

	check := func(_, contents manifestContents) error {
		var empty addr
		if contents.gcGen != empty {
			return errors.New("migrating from v4 to v5 should result in a manifest with a 0 gcGen")
		}

		return nil
	}

	_, err = updateWithParseWriterAndChecker(ctx, dir, fm5.writeManifest, fm4.parseManifest, check, contents.lock, contents, nil)

	if err != nil {
		return false, err
	}

	return true, err
}

// parse the manifest in its given format
func getFileManifest(ctx context.Context, dir string) (manifest, error) {
	f, err := openIfExists(filepath.Join(dir, manifestFileName))
	if err != nil {
		return nil, err
	}
	if f == nil {
		// initialize empty repos with v4
		return fileManifestV4{dir}, nil
	}
	defer func() {
		err = f.Close()
	}()

	fm5 := fileManifestV5{dir}
	ok, _, err := fm5.ParseIfExists(ctx, &Stats{}, nil)
	if ok && err == nil {
		return fm5, nil
	}

	fm4 := fileManifestV4{dir}
	ok, _, err = fm4.ParseIfExists(ctx, &Stats{}, nil)
	if ok && err == nil {
		return fm4, nil
	}

	return nil, ErrUnreadableManifest
}

// fileManifestV5 provides access to a NomsBlockStore manifest stored on disk in |dir|. The format
// is currently human readable. The prefix contains 5 strings, followed by pairs of table file
// hashes and their counts:
//
// |-- String --|-- String --|-------- String --------|-------- String --------|-------- String -----------------|
// | nbs version:Noms version:Base32-encoded lock hash:Base32-encoded root hash:Base32-encoded GC generation hash
//
// |-- String --|- String --|...|-- String --|- String --|
// :table 1 hash:table 1 cnt:...:table N hash:table N cnt|
type fileManifestV5 struct {
	dir string
}

func newLock(dir string) *fslock.Lock {
	lockPath := filepath.Join(dir, lockFileName)
	return fslock.New(lockPath)
}

func lockFileExists(dir string) (bool, error) {
	lockPath := filepath.Join(dir, lockFileName)
	info, err := os.Stat(lockPath)

	if err != nil {
		if os.IsNotExist(err) {
			return false, nil
		}

		return false, errors.New("failed to determine if lock file exists")
	} else if info.IsDir() {
		return false, errors.New("lock file is a directory")
	}

	return true, nil
}

// Returns nil if path does not exist
func openIfExists(path string) (*os.File, error) {
	f, err := os.Open(path)
	if os.IsNotExist(err) {
		return nil, nil
	} else if err != nil {
		return nil, err
	}

	return f, err
}

func (fm5 fileManifestV5) Name() string {
	return fm5.dir
}

// ParseIfExists looks for a LOCK and manifest file in fm.dir. If it finds
// them, it takes the lock, parses the manifest and returns its contents,
// setting |exists| to true. If not, it sets |exists| to false and returns. In
// that case, the other return values are undefined. If |readHook| is non-nil,
// it will be executed while ParseIfExists() holds the manifest file lock.
// This is to allow for race condition testing.
func (fm5 fileManifestV5) ParseIfExists(ctx context.Context, stats *Stats, readHook func() error) (exists bool, contents manifestContents, err error) {
	t1 := time.Now()
	defer func() {
		stats.ReadManifestLatency.SampleTimeSince(t1)
	}()

	return parseIfExistsWithParser(ctx, fm5.dir, fm5.parseManifest, readHook)
}

func (fm5 fileManifestV5) Update(ctx context.Context, lastLock addr, newContents manifestContents, stats *Stats, writeHook func() error) (mc manifestContents, err error) {
	t1 := time.Now()
	defer func() { stats.WriteManifestLatency.SampleTimeSince(t1) }()

	checker := func(upstream, contents manifestContents) error {
		if contents.gcGen != upstream.gcGen {
			return chunks.ErrGCGenerationExpired
		}
		return nil
	}

	return updateWithParseWriterAndChecker(ctx, fm5.dir, fm5.writeManifest, fm5.parseManifest, checker, lastLock, newContents, writeHook)
}

func (fm5 fileManifestV5) UpdateGCGen(ctx context.Context, lastLock addr, newContents manifestContents, stats *Stats, writeHook func() error) (mc manifestContents, err error) {
	t1 := time.Now()
	defer func() { stats.WriteManifestLatency.SampleTimeSince(t1) }()

	checker := func(upstream, contents manifestContents) error {
		if contents.gcGen == upstream.gcGen {
			return errors.New("UpdateGCGen() must update the garbage collection generation")
		}

		if contents.root != upstream.root {
			return errors.New("UpdateGCGen() cannot update the root")
		}
		return nil
	}

	return updateWithParseWriterAndChecker(ctx, fm5.dir, fm5.writeManifest, fm5.parseManifest, checker, lastLock, newContents, writeHook)
}

func (fm5 fileManifestV5) parseManifest(r io.Reader) (manifestContents, error) {
	manifest, err := ioutil.ReadAll(r)

	if err != nil {
		return manifestContents{}, err
	}

	slices := strings.Split(string(manifest), ":")
	if len(slices) < prefixLen || len(slices)%2 != 1 {
		return manifestContents{}, ErrCorruptManifest
	}

	if StorageVersion != string(slices[0]) {
		return manifestContents{}, errors.New("invalid storage version")
	}

	specs, err := parseSpecs(slices[prefixLen:])
	if err != nil {
		return manifestContents{}, err
	}

	lock, err := parseAddr(slices[2])
	if err != nil {
		return manifestContents{}, err
	}

	gcGen, err := parseAddr(slices[4])
	if err != nil {
		return manifestContents{}, err
	}

	return manifestContents{
		vers:  slices[1],
		lock:  lock,
		root:  hash.Parse(slices[3]),
		gcGen: gcGen,
		specs: specs,
	}, nil
}

func (fm5 fileManifestV5) writeManifest(temp io.Writer, contents manifestContents) error {
	strs := make([]string, 2*len(contents.specs)+prefixLen)
	strs[0], strs[1], strs[2], strs[3], strs[4] = StorageVersion, contents.vers, contents.lock.String(), contents.root.String(), contents.gcGen.String()
	tableInfo := strs[prefixLen:]
	formatSpecs(contents.specs, tableInfo)
	_, err := io.WriteString(temp, strings.Join(strs, ":"))

	return err
}

// fileManifestV4 is the previous versions of the NomsBlockStore manifest.
// The format is as follows:
//
// |-- String --|-- String --|-------- String --------|-------- String --------|-- String --|- String --|...|-- String --|- String --|
// | nbs version:Noms version:Base32-encoded lock hash:Base32-encoded root hash:table 1 hash:table 1 cnt:...:table N hash:table N cnt|
type fileManifestV4 struct {
	dir string
}

func (fm4 fileManifestV4) Name() string {
	return fm4.dir
}

func (fm4 fileManifestV4) ParseIfExists(ctx context.Context, stats *Stats, readHook func() error) (exists bool, contents manifestContents, err error) {
	t1 := time.Now()
	defer func() {
		stats.ReadManifestLatency.SampleTimeSince(t1)
	}()

	return parseIfExistsWithParser(ctx, fm4.dir, fm4.parseManifest, readHook)
}

func (fm4 fileManifestV4) Update(ctx context.Context, lastLock addr, newContents manifestContents, stats *Stats, writeHook func() error) (mc manifestContents, err error) {
	t1 := time.Now()
	defer func() { stats.WriteManifestLatency.SampleTimeSince(t1) }()

	noop := func(_, _ manifestContents) error {
		return nil
	}

	return updateWithParseWriterAndChecker(ctx, fm4.dir, fm4.writeManifest, fm4.parseManifest, noop, lastLock, newContents, writeHook)
}

func (fm4 fileManifestV4) parseManifest(r io.Reader) (manifestContents, error) {
	manifest, err := ioutil.ReadAll(r)

	if err != nil {
		return manifestContents{}, err
	}

	slices := strings.Split(string(manifest), ":")
	if len(slices) < 4 || len(slices)%2 == 1 {
		return manifestContents{}, ErrCorruptManifest
	}

	if storageVersion4 != string(slices[0]) {
		return manifestContents{}, errors.New("invalid storage version")
	}

	specs, err := parseSpecs(slices[4:])

	if err != nil {
		return manifestContents{}, err
	}

	ad, err := parseAddr(slices[2])

	if err != nil {
		return manifestContents{}, err
	}

	return manifestContents{
		vers:  slices[1],
		lock:  ad,
		root:  hash.Parse(slices[3]),
		specs: specs,
	}, nil
}

func (fm4 fileManifestV4) writeManifest(temp io.Writer, contents manifestContents) error {
	strs := make([]string, 2*len(contents.specs)+4)
	strs[0], strs[1], strs[2], strs[3] = storageVersion4, contents.vers, contents.lock.String(), contents.root.String()
	tableInfo := strs[4:]
	formatSpecs(contents.specs, tableInfo)
	_, err := io.WriteString(temp, strings.Join(strs, ":"))

	return err
}

func parseIfExistsWithParser(_ context.Context, dir string, parse manifestParser, readHook func() error) (exists bool, contents manifestContents, err error) {
	var locked bool
	locked, err = lockFileExists(dir)

	if err != nil {
		return false, manifestContents{}, err
	}

	// !exists(lockFileName) => uninitialized store
	if locked {
		var f *os.File
		err = func() (ferr error) {
			lck := newLock(dir)
			ferr = lck.Lock()
			if ferr != nil {
				return ferr
			}

			defer func() {
				unlockErr := lck.Unlock()
				if ferr == nil {
					ferr = unlockErr
				}
			}()

			if readHook != nil {
				ferr = readHook()

				if ferr != nil {
					return ferr
				}
			}

			f, ferr = openIfExists(filepath.Join(dir, manifestFileName))
			if ferr != nil {
				return ferr
			}
			return nil
		}()

		if err != nil {
			return exists, contents, err
		}

		if f != nil {
			defer func() {
				closeErr := f.Close()

				if err == nil {
					err = closeErr
				}
			}()

			exists = true

			contents, err = parse(f)

			if err != nil {
				return false, contents, err
			}
		}
	}
	return exists, contents, nil
}

func updateWithParseWriterAndChecker(_ context.Context, dir string, write manifestWriter, parse manifestParser, validate manifestChecker, lastLock addr, newContents manifestContents, writeHook func() error) (mc manifestContents, err error) {
	var tempManifestPath string

	// Write a temporary manifest file, to be renamed over manifestFileName upon success.
	// The closure here ensures this file is closed before moving on.
	tempManifestPath, err = func() (name string, ferr error) {
		var temp *os.File
		temp, ferr = tempfiles.MovableTempFileProvider.NewFile(dir, "nbs_manifest_")

		if ferr != nil {
			return "", ferr
		}

		defer func() {
			closeErr := temp.Close()

			if ferr == nil {
				ferr = closeErr
			}
		}()

		ferr = write(temp, newContents)

		if ferr != nil {
			return "", ferr
		}

		return temp.Name(), nil
	}()

	if err != nil {
		return manifestContents{}, err
	}

	defer file.Remove(tempManifestPath) // If we rename below, this will be a no-op

	// Take manifest file lock
	lck := newLock(dir)
	err = lck.Lock()

	if err != nil {
		return manifestContents{}, err
	}

	defer func() {
		unlockErr := lck.Unlock()

		if err == nil {
			err = unlockErr
		}
	}()

	// writeHook is for testing, allowing other code to slip in and try to do stuff while we hold the lock.
	if writeHook != nil {
		err = writeHook()

		if err != nil {
			return manifestContents{}, err
		}
	}

	var upstream manifestContents
	// Read current manifest (if it exists). The closure ensures that the file is closed before moving on, so we can rename over it later if need be.
	manifestPath := filepath.Join(dir, manifestFileName)
	upstream, err = func() (upstream manifestContents, ferr error) {
		if f, ferr := openIfExists(manifestPath); ferr == nil && f != nil {
			defer func() {
				closeErr := f.Close()

				if ferr == nil {
					ferr = closeErr
				}
			}()

			upstream, ferr = parse(f)

			if ferr != nil {
				return manifestContents{}, ferr
			}

			if newContents.vers != upstream.vers {
				return manifestContents{}, errors.New("Update cannot change manifest version")
			}

			return upstream, nil
		} else if ferr != nil {
			return manifestContents{}, ferr
		}

		if lastLock != (addr{}) {
			return manifestContents{}, errors.New("new manifest created with non 0 lock")
		}

		return manifestContents{}, nil
	}()

	if err != nil {
		return manifestContents{}, err
	}

	if lastLock != upstream.lock {
		return upstream, nil
	}

	// this is where we assert that gcGen is correct
	err = validate(upstream, newContents)

	if err != nil {
		return manifestContents{}, err
	}

	err = file.Rename(tempManifestPath, manifestPath)
	if err != nil {
		return manifestContents{}, err
	}

	return newContents, nil
}
