package gift

import (
	"image"
	"testing"
)

func TestConvolution(t *testing.T) {

	testData := []struct {
		desc                  string
		kernel                []float32
		normalize, alpha, abs bool
		delta                 float32
		srcb, dstb            image.Rectangle
		srcPix, dstPix        []uint8
	}{
		{
			"convolution (0x0, false, false, false, 0)",
			[]float32{},
			false, false, false, 0,
			image.Rect(-1, -1, 2, 2),
			image.Rect(0, 0, 3, 3),
			[]uint8{
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x20, 0x40, 0x60, 0x80,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x80, 0x60, 0x40, 0x20, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			[]uint8{
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x20, 0x40, 0x60, 0x80,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x80, 0x60, 0x40, 0x20, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
		},
		{
			"convolution (3x3, false, false, false, 0)",
			[]float32{
				0, 0, 1,
				0, 0, 0,
				1, 0, 0,
			},
			false, false, false, 0,
			image.Rect(-1, -1, 2, 2),
			image.Rect(0, 0, 3, 3),
			[]uint8{
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x20, 0x40, 0x60, 0x80,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x80, 0x60, 0x40, 0x20, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			[]uint8{
				0x00, 0x00, 0x00, 0x00, 0x20, 0x40, 0x60, 0x00, 0x20, 0x40, 0x60, 0x80,
				0x80, 0x60, 0x40, 0x00, 0xA0, 0xA0, 0xA0, 0x00, 0x20, 0x40, 0x60, 0x00,
				0x80, 0x60, 0x40, 0x20, 0x80, 0x60, 0x40, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
		},
		{
			"convolution (3x3, true, false, false, 0)",
			[]float32{
				0, 0, 1,
				0, 0, 0,
				1, 0, 0,
			},
			true, false, false, 0,
			image.Rect(-1, -1, 2, 2),
			image.Rect(0, 0, 3, 3),
			[]uint8{
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x20, 0x40, 0x60, 0x80,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x80, 0x60, 0x40, 0x20, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			[]uint8{
				0x00, 0x00, 0x00, 0x00, 0x10, 0x20, 0x30, 0x00, 0x10, 0x20, 0x30, 0x80,
				0x40, 0x30, 0x20, 0x00, 0x50, 0x50, 0x50, 0x00, 0x10, 0x20, 0x30, 0x00,
				0x40, 0x30, 0x20, 0x20, 0x40, 0x30, 0x20, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
		},
		{
			"convolution (3x3, false, true, false, 0)",
			[]float32{
				0, 0, 1,
				0, 0, 0,
				1, 0, 0,
			},
			false, true, false, 0,
			image.Rect(-1, -1, 2, 2),
			image.Rect(0, 0, 3, 3),
			[]uint8{
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x20, 0x40, 0x60, 0x80,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x80, 0x60, 0x40, 0x20, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			[]uint8{
				0x00, 0x00, 0x00, 0x00, 0x20, 0x40, 0x60, 0x80, 0x20, 0x40, 0x60, 0x80,
				0x80, 0x60, 0x40, 0x20, 0xA0, 0xA0, 0xA0, 0xA0, 0x20, 0x40, 0x60, 0x80,
				0x80, 0x60, 0x40, 0x20, 0x80, 0x60, 0x40, 0x20, 0x00, 0x00, 0x00, 0x00,
			},
		},
		{
			"convolution (3x3, true, true, true, 0)",
			[]float32{
				0, 0, 0,
				-0.5, 0, 0.5,
				0, 0, 0,
			},
			true, true, true, 0,
			image.Rect(-1, -1, 2, 2),
			image.Rect(0, 0, 3, 3),
			[]uint8{
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x20, 0x40, 0x60, 0x80,
				0x80, 0x60, 0x40, 0x20, 0x00, 0x00, 0x00, 0x00, 0x20, 0x40, 0x60, 0x80,
				0x80, 0x60, 0x40, 0x20, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			[]uint8{
				0x00, 0x00, 0x00, 0x00, 0x20, 0x40, 0x60, 0x80, 0x20, 0x40, 0x60, 0x80,
				0x80, 0x60, 0x40, 0x20, 0x60, 0x20, 0x20, 0x60, 0x20, 0x40, 0x60, 0x80,
				0x80, 0x60, 0x40, 0x20, 0x80, 0x60, 0x40, 0x20, 0x00, 0x00, 0x00, 0x00,
			},
		},
		{
			"convolution (3x3, false, true, false, 3 / 255.0)",
			[]float32{
				0, 0, 1,
				0, 0, 0,
				1, 0, 0,
			},
			false, true, false, 3 / 255.0,
			image.Rect(-1, -1, 2, 2),
			image.Rect(0, 0, 3, 3),
			[]uint8{
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x20, 0x40, 0x60, 0x80,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x80, 0x60, 0x40, 0x20, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			[]uint8{
				0x03, 0x03, 0x03, 0x03, 0x23, 0x43, 0x63, 0x83, 0x23, 0x43, 0x63, 0x83,
				0x83, 0x63, 0x43, 0x23, 0xA3, 0xA3, 0xA3, 0xA3, 0x23, 0x43, 0x63, 0x83,
				0x83, 0x63, 0x43, 0x23, 0x83, 0x63, 0x43, 0x23, 0x03, 0x03, 0x03, 0x03,
			},
		},
		{
			"convolution (3x3, false, false, true, 0)",
			[]float32{
				0, 0, -0.5,
				0, 0, 0,
				0, 0, 0,
			},
			false, false, true, 0,
			image.Rect(-1, -1, 2, 2),
			image.Rect(0, 0, 3, 3),
			[]uint8{
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x20, 0x40, 0x60, 0x80,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x80, 0x60, 0x40, 0x20, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			[]uint8{
				0x00, 0x00, 0x00, 0x00, 0x10, 0x20, 0x30, 0x00, 0x10, 0x20, 0x30, 0x80,
				0x00, 0x00, 0x00, 0x00, 0x10, 0x20, 0x30, 0x00, 0x10, 0x20, 0x30, 0x00,
				0x00, 0x00, 0x00, 0x20, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
		},
		{
			"convolution (7x7, false, true, false, 0)",
			[]float32{
				0, 0, 0, 0, 0, 0, 1,
				0, 0, 0, 0, 0, 0, 0,
				0, 0, 0, 0, 0, 0, 0,
				0, 0, 0, 0, 0, 0, 0,
				0, 0, 0, 0, 0, 0, 0,
				0, 0, 0, 0, 0, 0, 0,
				1, 0, 0, 0, 0, 0, 0,
			},
			false, true, false, 0,
			image.Rect(-1, -1, 2, 2),
			image.Rect(0, 0, 3, 3),
			[]uint8{
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x20, 0x40, 0x60, 0x80,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x80, 0x60, 0x40, 0x20, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			[]uint8{
				0xA0, 0xA0, 0xA0, 0xA0, 0xA0, 0xA0, 0xA0, 0xA0, 0xA0, 0xA0, 0xA0, 0xA0,
				0xA0, 0xA0, 0xA0, 0xA0, 0xA0, 0xA0, 0xA0, 0xA0, 0xA0, 0xA0, 0xA0, 0xA0,
				0xA0, 0xA0, 0xA0, 0xA0, 0xA0, 0xA0, 0xA0, 0xA0, 0xA0, 0xA0, 0xA0, 0xA0,
			},
		},
	}

	for _, d := range testData {
		for _, parallel := range []bool{true, false} {
			src := image.NewNRGBA(d.srcb)
			src.Pix = d.srcPix

			f := Convolution(d.kernel, d.normalize, d.alpha, d.abs, d.delta)
			dst := image.NewNRGBA(f.Bounds(src.Bounds()))
			f.Draw(dst, src, &Options{Parallelization: parallel})

			if !checkBoundsAndPix(dst.Bounds(), d.dstb, dst.Pix, d.dstPix) {
				t.Errorf("test [%s] failed: %#v, %#v", d.desc, dst.Bounds(), dst.Pix)
			}
		}
	}

	testKernelSizes := []struct {
		klen, size int
	}{
		{0, 0},
		{1, 1}, {3, 1}, {8, 1},
		{9, 3}, {16, 3}, {24, 3},
		{25, 5}, {40, 5}, {48, 5},
	}

	for _, d := range testKernelSizes {
		tmp := make([]float32, d.klen)
		sz, _ := prepareConvolutionWeights(tmp, true)
		if sz != d.size {
			t.Errorf("unexpected kernel size: %d %d", d.klen, sz)
		}
	}

	// check no panics
	Convolution([]float32{}, true, true, true, 1).Draw(image.NewGray(image.Rect(0, 0, 0, 0)), image.NewGray(image.Rect(0, 0, 0, 0)), nil)
	convolve1dh(image.NewGray(image.Rect(0, 0, 1, 1)), image.NewGray(image.Rect(0, 0, 1, 1)), []float32{}, nil)
	convolve1dv(image.NewGray(image.Rect(0, 0, 1, 1)), image.NewGray(image.Rect(0, 0, 1, 1)), []float32{}, nil)
	convolve1dh(image.NewGray(image.Rect(0, 0, 0, 0)), image.NewGray(image.Rect(0, 0, 0, 0)), []float32{}, nil)
	convolve1dv(image.NewGray(image.Rect(0, 0, 0, 0)), image.NewGray(image.Rect(0, 0, 0, 0)), []float32{}, nil)
	convolveLine([]pixel{}, []pixel{}, []uweight{})
	prepareConvolutionWeights1d([]float32{0, 0})
	prepareConvolutionWeights1d([]float32{})
}

func TestGaussianBlur(t *testing.T) {

	testData := []struct {
		desc           string
		sigma          float32
		srcb, dstb     image.Rectangle
		srcPix, dstPix []uint8
	}{
		{
			"blur (0)",
			0,
			image.Rect(-1, -1, 4, 2),
			image.Rect(0, 0, 5, 3),
			[]uint8{
				0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0xC0, 0x00, 0xC0, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00,
			},
			[]uint8{
				0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0xC0, 0x00, 0xC0, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00,
			},
		},
		{
			"blur (0.3)",
			0.3,
			image.Rect(-1, -1, 4, 2),
			image.Rect(0, 0, 5, 3),
			[]uint8{
				0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0xC0, 0x00, 0xC0, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00,
			},
			[]uint8{
				0x00, 0x01, 0x00, 0x01, 0x00,
				0x01, 0xBD, 0x01, 0xBD, 0x01,
				0x00, 0x01, 0x00, 0x01, 0x00,
			},
		},
		{
			"blur (1)",
			1,
			image.Rect(-1, -1, 4, 2),
			image.Rect(0, 0, 5, 3),
			[]uint8{
				0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0xC0, 0x00, 0xC0, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00,
			},
			[]uint8{
				0x0B, 0x15, 0x16, 0x15, 0x0B,
				0x13, 0x23, 0x25, 0x23, 0x13,
				0x0B, 0x15, 0x16, 0x15, 0x0B,
			},
		},
		{
			"blur (3)",
			3,
			image.Rect(-1, -1, 4, 2),
			image.Rect(0, 0, 5, 3),
			[]uint8{
				0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0xC0, 0x00, 0xC0, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00,
			},
			[]uint8{
				0x05, 0x06, 0x06, 0x06, 0x05,
				0x05, 0x06, 0x06, 0x06, 0x05,
				0x05, 0x06, 0x06, 0x06, 0x05,
			},
		},
	}

	for _, d := range testData {
		src := image.NewGray(d.srcb)
		src.Pix = d.srcPix

		f := GaussianBlur(d.sigma)
		dst := image.NewGray(f.Bounds(src.Bounds()))
		f.Draw(dst, src, nil)

		if !checkBoundsAndPix(dst.Bounds(), d.dstb, dst.Pix, d.dstPix) {
			t.Errorf("test [%s] failed: %#v, %#v", d.desc, dst.Bounds(), dst.Pix)
		}
	}

	// check no panics
	GaussianBlur(0.5).Draw(image.NewGray(image.Rect(0, 0, 0, 0)), image.NewGray(image.Rect(0, 0, 0, 0)), nil)
}

func TestUnsharpMask(t *testing.T) {

	testData := []struct {
		desc                     string
		sigma, amount, threshold float32
		srcb, dstb               image.Rectangle
		srcPix, dstPix           []uint8
	}{
		{
			"unsharp mask (0.3, 1, 0)",
			0.3, 1, 0,
			image.Rect(-1, -1, 4, 2),
			image.Rect(0, 0, 5, 3),
			[]uint8{
				0x00, 0x40, 0x00, 0x40, 0x00,
				0x60, 0xB0, 0xA0, 0xB0, 0x60,
				0x00, 0x80, 0x00, 0x80, 0x00,
			},
			[]uint8{
				0x00, 0x40, 0x00, 0x40, 0x00,
				0x60, 0xB1, 0xA1, 0xB1, 0x60,
				0x00, 0x81, 0x00, 0x81, 0x00,
			},
		},
		{
			"unsharp mask (1, 1, 0)",
			1, 1, 0,
			image.Rect(-1, -1, 4, 2),
			image.Rect(0, 0, 5, 3),
			[]uint8{
				0x00, 0x40, 0x00, 0x40, 0x00,
				0x60, 0xB0, 0xA0, 0xB0, 0x60,
				0x00, 0x80, 0x00, 0x80, 0x00,
			},
			[]uint8{
				0x00, 0x45, 0x00, 0x45, 0x00,
				0x82, 0xFF, 0xE4, 0xFF, 0x82,
				0x00, 0xB2, 0x00, 0xB2, 0x00,
			},
		},
		{
			"unsharp mask (1, 0.5, 0)",
			1, 0.5, 0,
			image.Rect(-1, -1, 4, 2),
			image.Rect(0, 0, 5, 3),
			[]uint8{
				0x00, 0x40, 0x00, 0x40, 0x00,
				0x60, 0xB0, 0xA0, 0xB0, 0x60,
				0x00, 0x80, 0x00, 0x80, 0x00,
			},
			[]uint8{
				0x00, 0x42, 0x00, 0x42, 0x00,
				0x71, 0xDD, 0xC2, 0xDD, 0x71,
				0x00, 0x99, 0x00, 0x99, 0x00,
			},
		},
		{
			"unsharp mask (1, 1, 0.05)",
			1, 1, 0.05,
			image.Rect(-1, -1, 4, 2),
			image.Rect(0, 0, 5, 3),
			[]uint8{
				0x00, 0x40, 0x00, 0x40, 0x00,
				0x60, 0xB0, 0xA0, 0xB0, 0x60,
				0x00, 0x80, 0x00, 0x80, 0x00,
			},
			[]uint8{
				0x00, 0x40, 0x00, 0x40, 0x00,
				0x82, 0xFF, 0xE4, 0xFF, 0x82,
				0x00, 0xB2, 0x00, 0xB2, 0x00,
			},
		},
	}

	for _, d := range testData {
		src := image.NewGray(d.srcb)
		src.Pix = d.srcPix

		f := UnsharpMask(d.sigma, d.amount, d.threshold)
		dst := image.NewGray(f.Bounds(src.Bounds()))
		f.Draw(dst, src, nil)

		if !checkBoundsAndPix(dst.Bounds(), d.dstb, dst.Pix, d.dstPix) {
			t.Errorf("test [%s] failed: %#v, %#v", d.desc, dst.Bounds(), dst.Pix)
		}
	}

	// check no panics
	UnsharpMask(0.5, 1, 0).Draw(image.NewGray(image.Rect(0, 0, 0, 0)), image.NewGray(image.Rect(0, 0, 0, 0)), nil)
}

func TestMean(t *testing.T) {

	testData := []struct {
		desc           string
		ksize          int
		disk           bool
		srcb, dstb     image.Rectangle
		srcPix, dstPix []uint8
	}{
		{
			"mean (0x0 false)",
			0, false,
			image.Rect(-1, -1, 4, 2),
			image.Rect(0, 0, 5, 3),
			[]uint8{
				0x00, 0x40, 0x00, 0x40, 0x00,
				0x60, 0xB0, 0xA0, 0xB0, 0x60,
				0x00, 0x80, 0x00, 0x80, 0x00,
			},
			[]uint8{
				0x00, 0x40, 0x00, 0x40, 0x00,
				0x60, 0xB0, 0xA0, 0xB0, 0x60,
				0x00, 0x80, 0x00, 0x80, 0x00,
			},
		},
		{
			"mean (1x1 false)",
			1, false,
			image.Rect(-1, -1, 4, 2),
			image.Rect(0, 0, 5, 3),
			[]uint8{
				0x00, 0x40, 0x00, 0x40, 0x00,
				0x60, 0xB0, 0xA0, 0xB0, 0x60,
				0x00, 0x80, 0x00, 0x80, 0x00,
			},
			[]uint8{
				0x00, 0x40, 0x00, 0x40, 0x00,
				0x60, 0xB0, 0xA0, 0xB0, 0x60,
				0x00, 0x80, 0x00, 0x80, 0x00,
			},
		},
		{
			"mean (2x2 true)",
			2, true,
			image.Rect(-1, -1, 4, 2),
			image.Rect(0, 0, 5, 3),
			[]uint8{
				0x00, 0x40, 0x00, 0x40, 0x00,
				0x60, 0xB0, 0xA0, 0xB0, 0x60,
				0x00, 0x80, 0x00, 0x80, 0x00,
			},
			[]uint8{
				0x00, 0x40, 0x00, 0x40, 0x00,
				0x60, 0xB0, 0xA0, 0xB0, 0x60,
				0x00, 0x80, 0x00, 0x80, 0x00,
			},
		},
		{
			"mean (3x3 false)",
			3, false,
			image.Rect(-1, -1, 4, 2),
			image.Rect(0, 0, 5, 3),
			[]uint8{
				0x10, 0x40, 0x00, 0x40, 0x10,
				0x20, 0x50, 0x00, 0x50, 0x20,
				0x30, 0x60, 0x00, 0x60, 0x30,
			},
			[]uint8{
				0x25, 0x1E, 0x2E, 0x1E, 0x25,
				0x30, 0x25, 0x35, 0x25, 0x30,
				0x3B, 0x2C, 0x3C, 0x2C, 0x3B,
			},
		},
		{
			"mean (3x3 true)",
			3, true,
			image.Rect(-1, -1, 4, 2),
			image.Rect(0, 0, 5, 3),
			[]uint8{
				0x10, 0x40, 0x00, 0x40, 0x10,
				0x20, 0x50, 0x00, 0x50, 0x20,
				0x30, 0x60, 0x00, 0x60, 0x30,
			},
			[]uint8{
				0x1D, 0x2D, 0x1A, 0x2D, 0x1D,
				0x2A, 0x36, 0x20, 0x36, 0x2A,
				0x36, 0x40, 0x26, 0x40, 0x36,
			},
		},
	}

	for _, d := range testData {
		src := image.NewGray(d.srcb)
		src.Pix = d.srcPix

		f := Mean(d.ksize, d.disk)
		dst := image.NewGray(f.Bounds(src.Bounds()))
		f.Draw(dst, src, nil)

		if !checkBoundsAndPix(dst.Bounds(), d.dstb, dst.Pix, d.dstPix) {
			t.Errorf("test [%s] failed: %#v, %#v", d.desc, dst.Bounds(), dst.Pix)
		}
	}

	// check no panics
	Mean(5, false).Draw(image.NewGray(image.Rect(0, 0, 0, 0)), image.NewGray(image.Rect(0, 0, 0, 0)), nil)
}

func TestSobel(t *testing.T) {
	testData := []struct {
		desc           string
		srcb, dstb     image.Rectangle
		srcPix, dstPix []uint8
	}{

		{
			"sobel 0x0",
			image.Rect(0, 0, 0, 0),
			image.Rect(0, 0, 0, 0),
			[]uint8{},
			[]uint8{},
		},
		{
			"sobel 6x6",
			image.Rect(-1, -1, 5, 5),
			image.Rect(0, 0, 6, 6),
			[]uint8{
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x99, 0x99, 0x99, 0x99,
				0x00, 0x00, 0x99, 0x99, 0x99, 0x99,
				0x00, 0x00, 0x99, 0x99, 0x99, 0x99,
				0x00, 0x00, 0x99, 0x99, 0x99, 0x99,
			},
			[]uint8{
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0xd8, 0xff, 0xff, 0xff, 0xff,
				0x00, 0xff, 0xff, 0xff, 0xff, 0xff,
				0x00, 0xff, 0xff, 0x00, 0x00, 0x00,
				0x00, 0xff, 0xff, 0x00, 0x00, 0x00,
				0x00, 0xff, 0xff, 0x00, 0x00, 0x00,
			},
		},
	}

	for _, d := range testData {
		src := image.NewGray(d.srcb)
		src.Pix = d.srcPix

		f := Sobel()
		dst := image.NewGray(f.Bounds(src.Bounds()))
		f.Draw(dst, src, nil)

		if !checkBoundsAndPix(dst.Bounds(), d.dstb, dst.Pix, d.dstPix) {
			t.Errorf("test [%s] failed: %#v, %#v", d.desc, dst.Bounds(), dst.Pix)
		}
	}
}
