DIRENV-STDLIB 1 "2019" direnv "User Manuals"
============================================

NAME
----

direnv-stdlib - functions for the `.envrc`

SYNOPSIS
--------

`direnv stdlib`

DESCRIPTION
-----------

Outputs a bash script called the *stdlib*. The following commands are included in that script and loaded in the context of an `.envrc`. In addition, it also loads the file in `~/.direnvrc` if it exists.

STDLIB
------

### `has <command>`

Returns 0 if the *command* is available. Returns 1 otherwise. It can be a binary in the PATH or a shell function.

Example:

    if has curl; then
      echo "Yes we do"
    fi

### `expand_path <rel_path> [<relative_to>]`

Outputs the absolute path of *rel_path* relative to *relative_to* or the current directory.

Example:

    cd /usr/local/games
    expand_path ../foo
    # output: /usr/local/foo

### `dotenv [<dotenv_path>]`

Loads a ".env" file into the current environment

### `user_rel_path <abs_path>`

Transforms an absolute path *abs_path* into a user-relative path if possible.

Example:

    echo $HOME
    # output: /home/user
    user_rel_path /home/user/my/project
    # output: ~/my/project
    user_rel_path /usr/local/lib
    # output: /usr/local/lib

### `find_up <filename>`

Outputs the path of *filename* when searched from the current directory up to /. Returns 1 if the file has not been found.

Example:

    cd /usr/local/my
    mkdir -p project/foo
    touch bar
    cd project/foo
    find_up bar
    # output: /usr/local/my/bar

### `source_env <file_or_dir_path>`

Loads another `.envrc` either by specifying its path or filename.

NOTE: the other `.envrc` is not checked by the security framework.

### `source_env_if_exists <filename>`

Loads another ".envrc", but only if it exists.

NOTE: contrary to `source_env`, this only works when passing a path to a file,
      not a directory.

Example:

    source_env_if_exists .envrc.private

### `source_up [<filename>]`

Loads another `.envrc` if found when searching from the parent directory up to /.

NOTE: the other `.envrc` is not checked by the security framework.

### `source_url <url> <integrity-hash>`

Loads another script from the given `url`. Before loading it it will check the
integrity using the provided `integrity-hash`.

To find the value of the `integrity-hash`, call `direnv fetchurl <url>` and
extract the hash from the outputted message.

See also `direnv-fetchurl(1)` for more details.

### `fetchurl <url> [<integrity-hash>]`

Fetches the given `url` onto disk and outputs it's path location on stdout.

If the `integrity-hash` argument is provided, it will also check the integrity
of the script.

See also `direnv-fetchurl(1)` for more details.

### `direnv_apply_dump <file>`

Loads the output of `direnv dump` that was stored in a file.

### `direnv_load [<command-generating-dump-output>]`

Applies the environment generated by running *argv* as a command. This is useful for adopting the environment of a child process - cause that process to run "direnv dump" and then wrap the results with direnv_load.

Example:

    direnv_load opam-env exec -- direnv dump

### `PATH_add <path>`

Prepends the expanded *path* to the PATH environment variable. It prevents a common mistake where PATH is replaced by only the new *path*.

Example:

    pwd
    # output: /home/user/my/project
    PATH_add bin
    echo $PATH
    # output: /home/user/my/project/bin:/usr/bin:/bin

### `MANPATH_add <path>`

Prepends the expanded *path* to the MANPATH environment variable. It takes care of man-specific heuritic.

### `path_add <varname> <path>`

Works like `PATH_add` except that it's for an arbitrary *varname*.

### `PATH_rm <pattern> [<pattern> ...]`

Removes directories that match any of the given shell patterns from the PATH environment variable. Order of the remaining directories is preserved in the resulting PATH.

Bash pattern syntax:
  https://www.gnu.org/software/bash/manual/html_node/Pattern-Matching.html

Example:

    echo $PATH
    # output: /dontremove/me:/remove/me:/usr/local/bin/:...
    PATH_rm '/remove/*'
    echo $PATH
    # output: /dontremove/me:/usr/local/bin/:...


### `load_prefix <prefix_path>`

Expands some common path variables for the given *prefix_path* prefix. This is useful if you installed something in the *prefix_path* using `./configure --prefix=$prefix_path && make install` and want to use it in the project.

Variables set:

    CPATH
    LD_LIBRARY_PATH
    LIBRARY_PATH
    MANPATH
    PATH
    PKG_CONFIG_PATH

Example:

    ./configure --prefix=$HOME/rubies/ruby-1.9.3
    make && make install
    # Then in the .envrc
    load_prefix ~/rubies/ruby-1.9.3


### `semver_search <directory> <folder_prefix> <partial_version>`

Search a directory for the highest version number in SemVer format (X.Y.Z).

Examples:

    $ tree .
    .
    |-- dir
        |-- program-1.4.0
        |-- program-1.4.1
        |-- program-1.5.0
    $ semver_search "dir" "program-" "1.4.0"
    1.4.0
    $ semver_search "dir" "program-" "1.4"
    1.4.1
    $ semver_search "dir" "program-" "1"
    1.5.0

### `layout <type>`

A semantic dispatch used to describe common project layouts.

### `layout go`

Adds "$(direnv_layout_dir)/go" to the GOPATH environment variable.
And also adds "$PWD/bin" to the PATH environment variable.

### `layout julia`

Sets the `JULIA_PROJECT` environment variable to the current directory.

### `layout node`

Adds "$PWD/node_modules/.bin" to the PATH environment variable.

### `layout php`

Adds "$PWD/vendor/bin" to the PATH environment variable.

### `layout perl`

Setup environment variables required by perl's local::lib See http://search.cpan.org/dist/local-lib/lib/local/lib.pm for more details.

### `layout python [<python_exe>]`

Creates and loads a virtualenv environment under `$PWD/.direnv/python-$python_version`. This forces the installation of any egg into the project's sub-folder.

It's possible to specify the python executable if you want to use different versions of python (eg: `layout python python3`).

Note that previously virtualenv was located under `$PWD/.direnv/virtualenv` and will be re-used by direnv if it exists.

### `layout python3`

A shortcut for `layout python python3`

### `layout ruby`

Sets the GEM_HOME environment variable to `$PWD/.direnv/ruby/RUBY_VERSION`. This forces the installation of any gems into the project's sub-folder. If you're using bundler it will create wrapper programs that can be invoked directly instead of using the `bundle exec` prefix.

### `use <program_name> [<version>]`

A semantic command dispatch intended for loading external dependencies into the environment.

Example:

    use_ruby() {
      echo "Ruby $1"
    }
    use ruby 1.9.3
    # output: Ruby 1.9.3

### `use julia <version>`

Loads the specified Julia version. You must specify a path to the directory with
installed Julia versions using $JULIA_VERSIONS. You can optionally override the
prefix for folders inside $JULIA_VERSIONS (default `julia-`) using $JULIA_VERSION_PREFIX.
If no exact match for `<version>` is found a search will be performed and the latest
version will be loaded.

Examples (.envrc):

    use julia 1.5.1   # loads $JULIA_VERSIONS/julia-1.5.1
    use julia 1.5     # loads $JULIA_VERSIONS/julia-1.5.1
    use julia master  # loads $JULIA_VERSIONS/julia-master

### `use rbenv`

Loads rbenv which add the ruby wrappers available on the PATH.

### `use nix [...]`

Load environment variables from `nix-shell`.

If you have a `default.nix` or `shell.nix` these will be used by default, but you can also specify packages directly (e.g `use nix -p ocaml`).

See http://nixos.org/nix/manual/#sec-nix-shell

### `use guix [...]`

Load environment variables from `guix environment`.

Any arguments given will be passed to guix environment. For example, `use guix hello` would setup an environment with the dependencies of the hello package. To create an environment including hello, the `--ad-hoc` flag is used `use guix --ad-hoc hello`. Other options include `--load` which allows loading an environment from a file.

See https://www.gnu.org/software/guix/manual/html_node/Invoking-guix-environment.html

### `rvm [...]`

Should work just like in the shell if you have rvm installed.

### `use node [<version>]`:

Loads NodeJS version from a `.node-version` or `.nvmrc` file.

If you specify a partial NodeJS version (i.e. `4.2`), a fuzzy match is performed and the highest matching version installed is selected.

Example (.envrc):

    set -e
    use node

Example (.node-version):

    4.2

### `use node <version>`

Loads specified NodeJS version.

Example (.envrc):

    set -e
    use node 4.2.2

### `use vim [<vimrc_file>]`

Prepends the specified vim script (or .vimrc.local by default) to the
`DIRENV_EXTRA_VIMRC` environment variable.

This variable is understood by the direnv/direnv.vim extension. When found,
it will source it after opening files in the directory.

### `watch_file <path> [<path> ...]`

Adds each file to direnv's watch-list. If the file changes direnv will reload the environment on the next prompt.

Example (.envrc):

    watch_file Gemfile

### `direnv_version <version_at_least>`

Checks that the direnv version is at least old as `version_at_least`. This can
be useful when sharing an `.envrc` and to make sure that the users are up to
date.

### `strict_env [<command> ...]`

Turns on shell execution strictness. This will force the .envrc
evaluation context to exit immediately if:

- any command in a pipeline returns a non-zero exit status that is not
  otherwise handled as part of `if`, `while`, or `until` tests,
  return value negation (`!`), or part of a boolean (`&&` or `||`)
  chain.
- any variable that has not explicitly been set or declared (with
  either `declare` or `local`) is referenced.

If followed by a command-line, the strictness applies for the duration
of the command.

Example (Whole Script):

    strict_env
    has curl

Example (Command):

    strict_env has curl

#### `unstrict_env [<command> ...]`

Turns off shell execution strictness. If followed by a command-line, the
strictness applies for the duration of the command.

Example (Whole Script):

    unstrict_env
    has curl

Example (Command):

    unstrict_env has curl


COPYRIGHT
---------

MIT licence - Copyright (C) 2019 @zimbatm and contributors

SEE ALSO
--------

direnv(1), direnv.toml(1)
