// Copyright © 2016-2023 Wei Shen <shenwei356@gmail.com>
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to deal
// in the Software without restriction, including without limitation the rights
// to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
// copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
// OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
// THE SOFTWARE.

package cmd

import (
	"fmt"
	"runtime"

	"github.com/shenwei356/xopen"
	"github.com/spf13/cobra"
)

// headersCmd represents the cut command
var headersCmd = &cobra.Command{
	GroupID: "info",

	Use:   "headers",
	Short: "print headers",
	Long: `print headers

`,
	Run: func(cmd *cobra.Command, args []string) {
		config := getConfigs(cmd)
		files := getFileListFromArgsAndFile(cmd, args, true, "infile-list", true)
		runtime.GOMAXPROCS(config.NumCPUs)

		verbose := getFlagBool(cmd, "verbose")

		if config.NoHeaderRow {
			if config.Verbose {
				log.Warningf("flag -H (--no-header-row) ignored")
			}
		}

		outfh, err := xopen.Wopen(config.OutFile)
		checkError(err)
		defer outfh.Close()

		for _, file := range files {
			csvReader, err := newCSVReaderByConfig(config, file)

			if err != nil {
				if err == xopen.ErrNoContent {
					if config.Verbose {
						log.Warningf("csvtk header: skipping empty input file: %s", file)
					}
					continue
				}
				checkError(err)
			}

			csvReader.Read(ReadOption{
				FieldStr: "1-",
			})

			if verbose {
				outfh.WriteString(fmt.Sprintf("# %s\n", file))
			}

			for record := range csvReader.Ch {
				if record.Err != nil {
					checkError(record.Err)
				}

				for i, n := range record.All {
					if verbose {
						outfh.WriteString(fmt.Sprintf("%d\t%s\n", i+1, n))
					} else {
						outfh.WriteString(n + "\n")
					}
				}

				break
			}

			readerReport(&config, csvReader, file)
		}
	},
}

func init() {
	headersCmd.Flags().BoolP("verbose", "v", false, `print verbose information`)

	RootCmd.AddCommand(headersCmd)
}
