package ccloudv2

import (
	"context"
	"net/http"

	ssov2 "github.com/confluentinc/ccloud-sdk-go-v2/sso/v2"

	"github.com/confluentinc/cli/v4/pkg/errors"
)

func newSsoClient(httpClient *http.Client, url, userAgent string, unsafeTrace bool) *ssov2.APIClient {
	cfg := ssov2.NewConfiguration()
	cfg.Debug = unsafeTrace
	cfg.HTTPClient = httpClient
	cfg.Servers = ssov2.ServerConfigurations{{URL: url}}
	cfg.UserAgent = userAgent

	return ssov2.NewAPIClient(cfg)
}

func (c *Client) ssoApiContext() context.Context {
	return context.WithValue(context.Background(), ssov2.ContextAccessToken, c.cfg.Context().GetAuthToken())
}

func (c *Client) CreateGroupMapping(groupMapping ssov2.IamV2SsoGroupMapping) (ssov2.IamV2SsoGroupMapping, error) {
	resp, httpResp, err := c.SsoClient.GroupMappingsIamV2SsoApi.CreateIamV2SsoGroupMapping(c.ssoApiContext()).IamV2SsoGroupMapping(groupMapping).Execute()
	return resp, errors.CatchCCloudV2Error(err, httpResp)
}

func (c *Client) DeleteGroupMapping(id string) error {
	httpResp, err := c.SsoClient.GroupMappingsIamV2SsoApi.DeleteIamV2SsoGroupMapping(c.ssoApiContext(), id).Execute()
	return errors.CatchCCloudV2Error(err, httpResp)
}

func (c *Client) GetGroupMapping(id string) (ssov2.IamV2SsoGroupMapping, error) {
	resp, httpResp, err := c.SsoClient.GroupMappingsIamV2SsoApi.GetIamV2SsoGroupMapping(c.ssoApiContext(), id).Execute()
	return resp, errors.CatchCCloudV2Error(err, httpResp)
}

func (c *Client) UpdateGroupMapping(update ssov2.IamV2SsoGroupMapping) (ssov2.IamV2SsoGroupMapping, error) {
	resp, httpResp, err := c.SsoClient.GroupMappingsIamV2SsoApi.UpdateIamV2SsoGroupMapping(c.ssoApiContext(), *update.Id).IamV2SsoGroupMapping(update).Execute()
	return resp, errors.CatchCCloudV2Error(err, httpResp)
}

func (c *Client) ListGroupMappings() ([]ssov2.IamV2SsoGroupMapping, error) {
	var list []ssov2.IamV2SsoGroupMapping

	done := false
	pageToken := ""
	for !done {
		page, httpResp, err := c.executeListGroupMappings(pageToken)
		if err != nil {
			return nil, errors.CatchCCloudV2Error(err, httpResp)
		}
		list = append(list, page.GetData()...)

		pageToken, done, err = extractNextPageToken(page.GetMetadata().Next)
		if err != nil {
			return nil, err
		}
	}
	return list, nil
}

func (c *Client) executeListGroupMappings(pageToken string) (ssov2.IamV2SsoGroupMappingList, *http.Response, error) {
	req := c.SsoClient.GroupMappingsIamV2SsoApi.ListIamV2SsoGroupMappings(c.ssoApiContext()).PageSize(ccloudV2ListPageSize)
	if pageToken != "" {
		req = req.PageToken(pageToken)
	}
	return req.Execute()
}
