package main

import (
	"regexp"
	"strings"
	"testing"
)

func TestIgnore(t *testing.T) {
	empty := Ignore{}
	dirs := Ignore{Dirs: regexp.MustCompile(`/bar$`)}
	files := Ignore{Files: regexp.MustCompile(`bar/gen\.go$`)}
	gen := Ignore{GeneratedFiles: true}

	for _, test := range []struct {
		FileName      string
		Contents      string
		DirsExpected  bool
		FilesExpected bool
		GenExpected   bool
	}{
		{
			FileName:     "foo/bar/zip.go",
			Contents:     "blah blah...",
			DirsExpected: true,
		},
		{
			FileName:     "foo/bar/zip/test.go",
			Contents:     "blah blah...",
			DirsExpected: true,
		},
		{
			FileName:      "zip/foobar/gen.go",
			Contents:      "blah blah...",
			FilesExpected: true,
		},
		{
			FileName: "foobar/test.go",
			Contents: `package test
// Code generated by zzz; DO NOT EDIT.`,
			GenExpected: true,
		},
		{
			FileName: "foobar/test.go",
			Contents: `package test
// AUTOGENERATED FILE: zzz stub code to make the package`,
			GenExpected: true,
		},
		{
			FileName: "foobar/test.go",
			Contents: `package test
// Code generated by zzz v1.0.0`,
			GenExpected: true,
		},
		{
			FileName: "foobar/test.go",
			Contents: strings.Repeat("x", 256) + `
// Code generated by zzz v1.0.0`,
		},
	} {
		c := []byte(test.Contents)

		if empty.Match(test.FileName, nil) {
			t.Errorf("empty should NOT match %s+NO contents", test.FileName)
		}
		empty.cache = nil
		if empty.Match(test.FileName, c) {
			t.Errorf("empty should NOT match %s+contents", test.FileName)
		}
		empty.cache = nil

		if dirs.Match(test.FileName, nil) != test.DirsExpected {
			t.Errorf("dirs.Match(%s+NO contents) == %t but should be %t",
				test.FileName, !test.DirsExpected, test.DirsExpected)
		}
		dirs.cache = nil
		if dirs.Match(test.FileName, c) != test.DirsExpected {
			t.Errorf("dirs.Match(%s+contents) == %t but should be %t",
				test.FileName, !test.DirsExpected, test.DirsExpected)
		}
		dirs.cache = nil

		if files.Match(test.FileName, nil) != test.FilesExpected {
			t.Errorf("files.Match(%s+NO contents) == %t but should be %t",
				test.FileName, !test.FilesExpected, test.FilesExpected)
		}
		files.cache = nil
		if files.Match(test.FileName, c) != test.FilesExpected {
			t.Errorf("files.Match(%s+contents) == %t but should be %t",
				test.FileName, !test.FilesExpected, test.FilesExpected)
		}
		files.cache = nil

		if gen.Match(test.FileName, nil) {
			t.Errorf("gen should NOT match %s with no contents", test.FileName)
		}
		// no cache reset needed here as previous call should not be cached
		if gen.Match(test.FileName, c) != test.GenExpected {
			t.Errorf("gen.Match(%s+contents) == %t but should be %t",
				test.FileName, !test.GenExpected, test.GenExpected)
		}
		gen.cache = nil
	}
}
