package kcdsa

import (
	"testing"

	"github.com/RyuaNerin/go-krypto/internal"
)

func Test_Verify_2048_256_SHA256(t *testing.T) { testVerify(t, testCase_2048_256_SHA256) }

var testCase_2048_256_SHA256 = []testCase{
	////////////////////////////////////////////////////////////////////////////////////////////////////
	// KCDSA_(2048)(256)(SHA-256)_SGT.txt
	{
		Sizes: L2048N256SHA256,
		P:     internal.HI(`8FC3DD39F1EB67436524503260C8F561C3DF7AAA55C6B7695ACED51D2ED107344C86290C46E749B3612A304FA9AEBCAC858F6F732A95F4A2471348AD1FC1BAF8708E9D7F3977F3EB978E50B426CF36ABE22DA6C4973489C8E561F092074DF25B91520B798AA46C7C3F95ECD5B5436F7F660FECF957168AF75DB5025BE1218CBBB6E8C22164154A4A107E5BDC80A26F7E58DED90EC0E5797EB1A335ADCA8C1C653ED0D28638730064A1A5622DBF7AEF2573D10D519C4ADA4572CAB694D9CFB228F678DD47A2D15FAAE1570FD0CED042775B3F6BBCCB24A994C27A27BF2CE832027F5B4E4BFCDC74B79D606FEE203F3962D7774201B81646F554DD765EBAE2C5EB`),
		Q:     internal.HI(`884460255C6079174FC9DF905F36387CF284202B089F5A1B5618E0047EDB965D`),
		G:     internal.HI(`75D6337A746CB90DD912AA5D308D05F82C3316E6D661D0BC0938E676474E65A6B5EE97B0E0CE402FB32362245FCE040619C0DB353BF2AC9DE98BBCF3C0EDD15C65FE3BED4C428DB9539961032125780EB4DC9EF59D9EB866F91AB4FF603DE81F14D6D9F473CEAC03DF848A102EB2B60E04DE7AD0A1CF8D1DE00C88E6AE962FD3906C8649A98DA4E5D99588F1593D84C0C636E0B1CE28C3CCABD6E35CC7EC788076B181CAE1F718768E6F90B74B0E7C870B0BE279B510BE79DC07819A4D6F89D5D799B3E007544564ABD435E98E9B5618AAA48F5527DD20E0676A31288269E3F1B7BC655A518AD896878591924344BB09D86785A9C7EF8159C5CF15791D542353`),
		M:     internal.HB(`0CCDB56BD732B08A3BEE2E1EE054A7C297BAEA44185E60DF1922835CC5BEBD1C5CFAF3FBED79FA068C414FD3E6F337C398F2B3B1B356DD693E769F1BB9F65E58610B0A023DDBA93133D6CDF034C68B58AFF76861E7544F08DE2E860EE0AD32707391DEE22CC0A6ED305360F568146EC9211E2C0E28A2F88C24F2E6C5A44C9A5F`),
		X:     internal.HI(`8FEC8607ED50FCD10BFED1255515389913720E647F3E875E289F318834E24609`),
		Y:     internal.HI(`ECEDAE2545B06FE59263185C995F507EF459FD902D3C4FC8545D2E702532B4E2FAFE549E64366890E5380E86631C7A5D605D74CD117C7D52A8F454EC7B5D88DD760BFB9F1125F5E1DF5443702689D8765353700A35B6DA73CFF202B800F984E62F2436C2740C68FCE4CFBDFF08BA44A5D257DA7CA06546623F50A3B9B88074D75E31B2252E4A2FC35A362C58F9F1DD1417782762C5EE2651A2DAF4BE37887D66BC03BF09C87C1B6C92E1343BBD74533BB27971E3032F2DAB81CEF02B5D688C36C691B27A1E0C9149FF67740C87F04199F923680723D72D6CAC04A010D6E7799752073ECF606E5346BC1967F7F29D0AA4AED69E97FEF45485751EAE2558864B`),
		R:     internal.HI(`9E67AA57CA3671898732C8A85C9F635138286560A1858F81AC58D043835AFF96`),
		S:     internal.HI(`4C9EA075978BF533092844F2E25E3A111AB88A549EF5CA79A7917C9B3B1D2123`),
	},
	{
		Sizes: L2048N256SHA256,
		P:     internal.HI(`F758845BEFAC328718F3422298689208184FB6D851020E4DAD3BC9991E1AC30591F190EF8829BE08543699108830758CBAE9A5100FF546D2AB69DA8F543386BDD4EA92BDF5D88728F7E537DFAE5434DB907DE3474729DEE0D6538D5BD86BFC458622A80ACE0C14EB5D42A1DC38E30E0EBD4562A473A52B1C6908C47C631501A4A05B9CE97EBC714A5B2C17C13D9DDAE1FBF6FC2807E3005636B27C05F63AE5C4F809BD97DA66E1791E93ABAE971FB1D4E35DBAF20319CC01ACFBC60BDBE9A6A1133517FD495B1719F30D5ACE8B5BC44352FEBE766F3FDB878214E0FB09102F56051DBE6422BE4C64097274E83027E55E59F51297D35D045327D16232C635F8E7`),
		Q:     internal.HI(`DF151FF03880D70302069CFAFDCF5B2B7934D07BACD568D5F7C7C01FA7FF932B`),
		G:     internal.HI(`64896F685EC3781A227510C7F1F7EDF0B8AD4CB0C70B9C91A5C77DFFDF66D4EDA02EB5EBBB3034D2E70BDB53D2DF1EB7D288643ABB93FAD49680D83BA210F6F8BDFF7B43ED1A45D8432CFA3EC71F4E937B03267DA3A2CD29C9FA146D0D17A11BBF43832452F10D6C9E984A1B61E86499F5CBB0B0A732D0311439737715EE7B93CF4BDC5318862802FC596E72D12656D69E793D0E753EAC594734CC776CE01F5005ABEE7B62349582BD9825C91EDDF35A9578221887DA9289A168BDD5C73CBCB65000A356EE391856FA3E4B118579D1BD4DE7F5E1B9049D394940A773524B6DAB63F8C0D4060AE918079FDFBB25EB91531359B6B35A44D095F5733E3662CC1A28`),
		M:     internal.HB(`41254EACB7531200D4E02E0632E8DA46833E9497FACD52AD896B627AFE02AD834CE02A0DEF82BC90C7A7566C5814FEDC44342BFCF9A6088761C512057F41B3226A6234D9F8812DAA05DAE795B32BBF8422E1EE6654427A96CE34006749BEAC62454DA5890C489F3F59AB69913285339731AB511DCAF93F3ECBECAB4213F5C6CE`),
		X:     internal.HI(`4B7549B70BCCF9819D59E3FD1CA491EAF69E9EFDE24FD525D4871D56ED6DBE66`),
		Y:     internal.HI(`8DD394362FD3A12041D275EBA48EF387E531CDDF0E04B4C2483A08F692081B735729754D5D2E9C76A0FE84442DA0678801A0A1BBD1A550A25D96E601EC2B4C0F9C5A37396D85A99B023F965AF26FBB52840160C9B90D459B9AC1432A8A9ABDD9CE312CD4E1F071CFE20DC26E1F587639E57C473E5D73FFBEC122D3FDF06BF2999E219C71388806E79B4BA4621260F0FC133788AB7A4DF6C69CD325E143B9986B331ECC3921A702DD1A3F39D41B5B6499C8DB09C862380F07A6ABF5E5EA3203F5F15369D5762F44485F409E46620497FB54B5E3AB76BD4582D045C15EFB9D7017700479D426D123C0CDA5D1C955B669787201B21870DF4E969AC42EB83F9D7B8D`),
		R:     internal.HI(`53EBE1EEC202E3F6F881D5609ACD30C2290DF05B33D744C6B294DCB3351FDBB1`),
		S:     internal.HI(`146BD1E6EDF72A4D73E1C17DE1F973CD0F684FA9477E49CA67E4F91B0DEDCA7C`),
	},
	{
		Sizes: L2048N256SHA256,
		P:     internal.HI(`E0B09144F0160B377246AC29E8B90BE3CF66B6EE8E70595B12AC322DA470485665CCBDE67D34219E712DB126F191C7A7D0C18D662A77E8F2B4B3FC7EFE734E4224FDB028E6F55D8F2AFB546D7F3C649F9A85BB1F69B30F794042530AABCB53A026DD1908CDC0EF93D91427CA24FBCB2535DC9E89133BAC7C38D700BDD894656DAFE5B83FF3708E29A7F2901500733CF98B4106EE19D020C590343B4B53A1E28764B518A885E40003D2EB3FD34729F3264521DB1B9BA8FC0676733A7FEB7D8F353727D9B73393E0E19D0C6ED609E716A2FD2C5E55A942D9271C611F1D2D770DF0B6B2B237447352DF112A49654BCB176BBD688C7B53061D5867045720E072CF83`),
		Q:     internal.HI(`CE058726D9C6F8C032BECBD4E040DC52EF9C37AF7755BB7AABE91C67D7D04679`),
		G:     internal.HI(`7F824D0A2FE3BF8BF7D286858D3EF755A14EDF417A945F2D0697D8DDF134A880592BC8319A5E9433290E06F79A0AEBAEF65F9B47F143BB8D811A358F16EBDEFFF6E208530F42DCB015C5D8E03C1144A1CBD0FA4C9763BF52A181CF1383D3A2B65615514915DF7F8B71C6504F65BA321081BEA6D9D18BA740C8EB01D35A89346D1BF65CD41E0B0A81E062026C840B67841EA04FD8BCF4850D26ED8513C21352D7DF769286CD36C4B749307AF8DFEB59CD09D1818B2B3431A2C6541381354C83C31DC831EF069CAEDCA53F289E7B509AB680691C2556E4126630B3416F56CD7995626CFCDAB74FBE535B6950CE62EAA833F31D278F32F50CCC78EE45D8970A0F6C`),
		M:     internal.HB(`D947AAB974910023E8D153955340A65159F9388EB4EB1E4424A887DD2AA36AD67B7CE229925BF38EAB0A927A2FBA97555539FE4C7CF613D3F925F9A3EB4BAFCFCE99E13C5B6A4F9D6A14883D6C983CA7B6D9710D20C2CF7BC0245FA44CB1E3EDCED2B0F956C41ED0C33289FEC94279B1DFD1D22B32B737AB43EB8691D2620ACE`),
		X:     internal.HI(`14B58380FD685412D79DA17552D98FC0BE748BFBC884D70823200A728656EFAAA`),
		Y:     internal.HI(`ACCFA233128BE1361BCE50B9B602FF81A2215DD2DE2CF80F252EC195DC16DCC9C9FD4E04E3A2B26A8ECEEF2DE8CD05EF5F5D999CC8C025D32C2C11C5767CC9D055862D3C3B12267AC9C90E044A19B0415DCAF820354953877A06D65C0BFCF680F8636FD1C2A152BF958CF5DE089CE65F4F6214222D02705645DDCF92D2AA21D7C7767EC521FAE74E20359B5407DA470EAE4F2748F0085FD4522373AD0C1E3422385C9FB3DE143D4DDE3FD796021CEA6EF89D20EB19BA9851E92AF9C2D54F1819A84EC5ADE4603E5F60E728F9F321D70260DB19A7C232E659C7F66F8856F7BF8B063B409B73EBC183A68DF76DF9FE522FCC8D1064580F4E34844AE67401208DD3`),
		R:     internal.HI(`7224B06213849E67528438C7408DDBAB476D507F262CDE96E0299EA55C50DB5C`),
		S:     internal.HI(`B42E9437EB2264AAA75ECB017BAAC64E7086E2E742C26AA406997BF209E39C4E`),
	},
	{
		Sizes: L2048N256SHA256,
		P:     internal.HI(`DF5AB8B4DE25684F5D4A5243B7F1909FA77D8BDD929B6626C71462EE7517CB814727ECB4A3CCD6ED946EDF2878B7FBCD70CBE42AF20A8BE51CAE88CB989F8DF84DCFDDC975EC8A15E8CEC361F9F8E7403ABF135F3B1BF5974719E209AEABCD1D940B4421D473BBA49402A6140201D3CA91FCCA96F6CC765B84FAF1FA48EC7BBA2D7C68BD31F2EB6CB744CFEDDD960F5199CD0F07D6757624B61CCC3E71A0A6B2ADF4C4F0829C689E1FF6F7516D7F2F518887ADD00FD08003CB0BB12C65334C5431DB601083279377BEEEE7203D4BBC2F39D24FA7AE2C93A0E91F7C59EBD6422E028E25D880D0A972920D0F73CC58619A4D1A7BF18E26A6CED8EB31C5BD3FCFE7`),
		Q:     internal.HI(`CB72BA022C2B127F3438C4D8EB38AE79AF7F451053C9CBF763F6D2155B2CA3F1`),
		G:     internal.HI(`80CCE2C9CBABF63C93553747ACE5E6E6AD60E738C15AE918791AB7C5A4D6D6F4B60E54C9D3CC2BC82F689570DD5C2228626B49878EFEBE32D5BF5CAA8BB0E765C28071ABE96E191AD4756C4F6B7017CECAAEA6F211935442A8303A6A1B88ECA359EA643B15534896F4A03E14376A00441A35398A87DE6EEE8F5E5F403BB071E5680973D1BF6D741A333100F2DF17478DEAA60FA26F8049588E98057E0261B84D3D33D9B65252CFF4CCDC10BAD6674DE55A0DD5ECB83E9232A154DDBC155AAF252CF4DC4AD40E9288F861FAD07E868BA0CB20C423355313B11066E0F54A14DD3EC3782829018323AD3D80ACE7863BAAAD09F3BC42F245D03AD5800D95ADAD2F02`),
		M:     internal.HB(`C75F4A0FA0B61E8D2CACDE8517E5ECC00918D823D702643D1CC5B1C11AABD5F0DB77204B69CDC19A69504637BB7064073E2AAE184D932169BB7E98B0511E33B8825C9422755EB0A417EBEF232F9865995F88704F5F21EFCE696966FDBC903C6C034E01AB9D80C4BDA25280F90154A5711FB8B2B370A484EFBB5A396AB29DC5B8`),
		X:     internal.HI(`173A44F90E2C991E0C4B442A376FAB35CEF34D286633E71DF7F4EDE6DFC7AFEDC`),
		Y:     internal.HI(`532C3381D96656F9C42D9437B36BB4609EECC8B8FC235B859554D1E7B9608586B29D4B78EC2D77BB35BD959AA55E6489AE19EFFAF4DEEAB74778031291CE751C7BFC832F27AC74E3ECF85AB4A69314E83DED7D1EF69D3D32E7AA6D6357E14F07F967ABD8541DD4A5C2866DFE9EACFD98DD870816AFE46F005792ED62226DC62D71060F16278F59652853C1D972BA1973A145D082624196A547F319BF25F76E7D4E45DDC91AF154B0EFC03D186C297F4B434D606CC9FA4E48ECF54904A2A69DE4F824A21CB5CE8E1C15062E1F4DE5FE6B528D04DA0066B5D83290FFB97EFE42D0748EEF04C0E94296B6BE9369F46EC473D26F0B7489698762B39B00F142DA61EE`),
		R:     internal.HI(`7E1286B8779A3B378C704D48EE1904D96555E4AEF74A6BE27C90582D652DADC6`),
		S:     internal.HI(`42DEDF134E330432A765343A2669EE60637C0DE16AC302A9DC19703DCBBBA749`),
	},
	{
		Sizes: L2048N256SHA256,
		P:     internal.HI(`A96AF2C692C57E6721E8335DF76975B35A472F64180C975B9215C7AAE5FE54000609CBABBACD2871ACDFABA5A1B5BF4A430ED1157ABF0CFEC280B343E0ACC891C74DD880B0045A8223C52E65335F054BECE3CC4DB4C7C02DCCF7916F359DC1C06AE3B97E1C50FB64FEE9AFD7F72A6E51A45C1265F4B7AB7D4328C20F330C84D023E397F1CA5E987645D99CF32C15A6C85A020C0B192406135825951F73A243AE66CF08BA50798D23563F0D3F88038D24762735CDEECD64007643F844694A828726B155E4FE0D8D0DB1973B15346CFA3BE49246EE2F14219B8082DADEE81488AD925A7F77DA0675E73580EB169A52AFCD1DEE8D61AAA53AC5C27EC07C39BF1FBB`),
		Q:     internal.HI(`A9398B11F06556020E250B4A1D05752A170788731C6E0A2B8D7F742AD7A22DD9`),
		G:     internal.HI(`9FDFE3B6D27A1EA202FA0246918491A8A731A769DE697C7C8065761D23C6A3322553B9ECB9D7D445A2F817E3B52427530DC251E3DE74CA45AC55E84087644013CF8EEF01064EB803FD87B538996909F470935F74BA0FD1B23D1CB5228751ED6BD79940D14A513C4DAABE48386888B108D82AA5A50DE7E68D506AF871A5412CBE26ECF3F68C3472D7D0AB570304A33E646A9C68D146BD03F91BBD7B2318B982026A948B177B11183AA659C769DA70D9A3EF61F8C2A4E59AC94E3D8D28F98EAE4957E3AC8AADEEBC2669D7FF27EE19D303A4DF097057E190E689865655AE84EFAA5940933AB35D830EB934B4456F9819F0BA446D1FAF0835F4A6A6D3D3945DB4B5`),
		M:     internal.HB(`FB16B0244D0D0C58CEDB920FCF6211EC07C2F751765A48B2212B255F9FA3D3AA607CE36C06A3C94E3464325DD041C64C71323DDA7D4455E159BBB0E502449FB8F4D1CE06D9A8745BBC4B600050351C368E997027A1267FABF86BD62E6D6517B85367191AF243B51F267592B9A9C617B2610CF0AE160BC824E820860A0832DBE5`),
		X:     internal.HI(`13537A9FF5EA1811216DBE099587BEB97F25ABC0CDF77875820F3BC9742C5A6B1`),
		Y:     internal.HI(`B7B5F6EE94B898393FD141F9F6EB5A04B94B2842B2BA67436CEB221DC7354D924ACBFEF1ADB2A0DC4837EABC42C47EEB45D98D785B3CB3C253B08663AEBF9E644997D25A3F45B014224932F36B2AA7CADD5471F4D9D25629132C4EA0085D2CDBB6293231DC450AE0C16A10BA15C5FB353AF355E8162632A326E0A7201AEF2D1B885FC11C1BCA3A2C3C8AD02A24EB1832BBF629E274D05A6E2259E708131DE6EE8F80683451FE675E2BDF8EA5DB610AF2451A4A11206992B1424AE8247221FC350FFBA4CF211752EE45D37A1C8BFB8CCDF7072A3E64DE470F0729EF6157B339AD77D808A925901104B70587BC5569DDD5D77310589FBD676E3E1DE2AC709F50E`),
		R:     internal.HI(`6DDBDEDE95BD55BFEB01BCEBBD9EFBD38113610015D6FD4CCBB5E5B1B6E46E9`),
		S:     internal.HI(`53DAF172423A441A39160235190E6B40ED7B13C03C8407E8340E2993A900C64D`),
	},
	{
		Sizes: L2048N256SHA256,
		P:     internal.HI(`B7C49481D51BF9FD0C127404202F22E78019BE35103D8BDC77E3F3E60ADF726DFFD731D1C1DDD251CDEFB65599052CFCD95FD38E8FACC8A8B4C63920A57BA32DC273A26D73B882EE2956FC317EEC64C697849DFF71332A4230F49778EE719C5D9CA4E020FD1A0FF88C6CAD826CF67D1C980791BAF0D655B623E63AAFF78068443E0E4102A10101D7BB173AA5F461BF22D537C07778BE39A35C31F9C60DB280AB6C02F4F1DFE6502561BA192A844A20001A167C447A007F5CF56085BE7AEC1FC164376FE04A97F8FAF25A00C15D02981310EF06859F265017DA438A2E02DC19C74C60BF0234BAB212B1AF000262F47BD3DB4C455A770E3F2800F8E0CFC8BFFC93`),
		Q:     internal.HI(`A76DA05A17BD266F5997E1DFD63FDFC520E37250921590E03A0965678BA6C6F3`),
		G:     internal.HI(`1FFC41A6A2EE9B55956E2437C3FF9B3C37F6E820962A00F1FAE5C6AEF687C6C9AD8A7C5674E6683BEB9368FEF1A4A05F0815B376F4EE147EFA546E591BEF1E50A7F67BED7CE4F2B12613547CFB4239B9CDEAB884E1F3443529657F3C950130784F6AB42450341E98029873712B36A3089C1D10F74A6B39DFFB093CBFB95F9D57635BB652D081785D43D209E6A6C6685EDE8E404C56B28E33DDBF64ECE9297F97664B42B9EDA6493D6B059B78BBF1E8020E8A88AB9B92946CE09CD5D8F90BD53481AB6F22B897FFEA59CD001CAB8831DCF55A2BD452F21255B3E06BB1FD46ACDA1EF7D3E018F2B22AA9D3DFF55964B296F2D96A8992BAA5B2516F6886E9EA80E9`),
		M:     internal.HB(`E915DE750D5E6D1F0448316E4DC0F3AFC3A39711223FEABEE6C4A3710E1844DEF9353106F8272CC3BDB199A6C136A000DE7B2E0E9E5352578245847D4F485829992692E11A93BB598D1F1C1020F7C256B435708DFA1D9EAD219573EE2E3A562BB24AFABFE859138F01067FF91521B14D18F59218B7B5A5617BA8302823A82F30`),
		X:     internal.HI(`1499A14C93155BA0B35D74A98692296F3A4753377D0827302B451F4AEE8901AAA`),
		Y:     internal.HI(`3C3A9626C2A287B51BAD99DA253660344F159406E4B62DCFA1D164E3C5C59DBE1BE8EA18E1E869F4FB28B75EDD3AAE7139E7EB177E2DC0507A9208BF9C8271EB21F9848BF1EB6156A5B95CE4C6302D49AC7E74AB2211FEE0645798DE755B9C593B7A8A46E346023AC6479E1F0570CAE0EAF9D0AEDF1555E2060FFACB37B588043B43116D3A6B52933869C807D74A66CA9E9889C5223F129A2F16526502626B3A4D7BC903377A4DC6CCE9360751FB58D877A4BB21C3E6021910703965B327624BC1CD9F19D1FA3311D544DE34099D79EEF6256B1D92865573758B3D008B353BB72603062B7AC50446CCAA013AD714D4EEBD29551D03BB4D5328F9213ACD8E1F94`),
		R:     internal.HI(`3A78C9E501DB135AC37BA112BB4C389098DB3CF8C5ECEEFBF738F01F8ED15D00`),
		S:     internal.HI(`26FB45EB47EBAF9AFCC8C899B21C34636F0B7018004DA195D0DC17E93ECAAA69`),
	},
	{
		Sizes: L2048N256SHA256,
		P:     internal.HI(`EC3C8A2FDB6D2292CE16537E6624313442B2213B44638690419217DC50D9DC200573009E324E28667214C486DB0E99BD37D0B5EB1E2AE65AF0896D4E906DC93405A0F7088F46F929B2DFDE16BD1A43F257EE5C9C65BF0BE1EDF8A3FD94EAEAA4DEE9F05317C2A5328BD45C49E9DBA30857512F07253769D00854625A530BC478CA0FDC409C5E9659AC03305728C74BCB04AF5FB6AE4500FA37D36349C826529861BE1A29ACADE11218757133FDD060EA57A132271E04BA67E9889820136EFEB44AC26E6DFED2B4C6A59923BA778C4DE5C89512285910ADAE0F253F94100B28FDDA1A565554DE30D2B54E79052B2D76759D346BF59BEEBA51B1B05B22E32FDA17`),
		Q:     internal.HI(`DDF09AAD8B048B1A62522BA4882E4FCC47046D38A3E1A5F8743B549DAFD23F81`),
		G:     internal.HI(`7138E4415F878CEAF4FBBD5C8902CE6DB5CB1A7F9FA276DE2F0B9C4BDE2E730C1A2EC369FB2911F22FC69D5EEE47D8707FD3E87B3577675AEE0BC5840A7BF641C144FB8F037DBC3A221F07BA6D48AC9F5C21578B8E9EC20FFAF89273000854A73BC846FD437CBE23F5E2D33247E31CB12D96E08C15431085DEE5E4EB9010452DD607FBBD380D148BD508BBAADE11602D637163BAE2D921BD9653C794359A7C891D77C8917BA98778A2F2CC890D2F8070193B41CCE8F8733B1A14582B3476FD4A2513EABFAEDA81FDC92B130E240946AFC7EBF1A5EDFE72B75F1B64CDDD406352EC08C9E774953DA7D18399F3BC4ECEF755F98E29A474FE63043FA2FD275C2BF1`),
		M:     internal.HB(`8288587AF273637B7DDD7B5B638BF6E33165BA5CECF9EE7B9EF96E3037100A671BCB099456A28D95B721BD4D5E5854F97B2F012CC30B37E46B8754318BB33E65E1036130C8E729B93B4FE68A70EBBC54C587757D7BCE70EB95527DFBD219DB2012202915128BFE29E5ED8BF4176E569D359E98E966EDBCC326DE7880D40B22F2`),
		X:     internal.HI(`1975E08F31CD4253A01695865E779B452966EE0591CD2118A5F71B27839E22AAD`),
		Y:     internal.HI(`A2CF4DA8D71D71D62B54BE17F09ECD01CB3F9B9333175707999A04D44914FF848393821C84A53B902291A2489D5A2DDED2BD6A611C72EE1B99C6C94CECFF57AE3BFACCC40A5D3C7D72DAE4BDCAF3EC29E4C2B8FDE643EC5B1E1DCDA90C376DE528616DAC0179992327EBEB5E3676CF155193D5C8C44DC4522C109647C2A1944AC6624A69A43B21A76C634ADACC4451E5435DC1898B7825EABFCB818CA6A337B2B0C73839E88D594A447E5B785BD04E109E249EB022AE78D363692E3A25D63E0E5F8FE0D6B91BFC3293B65BB0F8CD0878B19AE18928E8123BB5BF6A813CEAE141627793BA5AB3D05FF63C20CB1B6087A68D6F65264DAB2B4FDC5B8C8CE26CF765`),
		R:     internal.HI(`FA1483AD41B345EB6FA793377DE76ED99C699AF09786C54DB5BF87A827B9C125`),
		S:     internal.HI(`CB6F9BC374F7B640400F97669ACFDBF9894B00FAE31C31FC4FA700FFF7232594`),
	},
	{
		Sizes: L2048N256SHA256,
		P:     internal.HI(`FB18C7DA5DED5135431E9FD3F566F77B878FD47C7DED4D19922725A64BD5EDC9D6BD5D2D0BA98B6FD3D8A319913A1E4E5DD243A1A01087AFD0606CF3C5A18A1B461263F80E997BB8FC38C59CE424E5B408125EF0E271494BEEA7BA64B2911C24E4D0052F87CF8A00EB70AC906962071DE97830B277D037DBE49045950CF49650D039C8558BBA2FEBECCF9A270174C7D738C68A6E6AE913C5CD1A97282085CAEEECC33B2C5624E71583782E389EEE9B65C87937089DC6E3CDEC86516512A9C1D693874E615D212C77119D3D2B014FC3DC612977DB563858210B18AFC34BCD7A6B607D421DA9249A45AF08F482918FEFB0DEFE45CFE808904DD35D3BDF0CB0EF0B`),
		Q:     internal.HI(`F415AB07DB434A5F07B6FD38ED303FA17E5FA55B0DE1A82EC299FCC8493A15DD`),
		G:     internal.HI(`93EF9556D8FA2B06AF63EB9C8E912B18C90E3A6BBA85D01A94C3D22F90C76CCA06AA8D99BC0DD2EF526649CE119BA13A7BB654DA7EE300ECCB3D4A19231B61BB5A35177D8E1755C7E8F8BF18EAF1CFEC26F5909867DCFF4AD78DF073B85DFAF9A210DD65D9BA3D0FD29078E7D6861306E2FA06223F26B8443F4BAAB8E8E3A17C6970859C2E839DEFC691E10DD9B2AEF66681690235459171A3E05F6503CE8FD95062E5D1D6F5762A0C0D5712B6D8352699862BB93D5506D0501DF89AD1F399797126E6E70C48D0ECC66098FCBD4AA588C3A4C3BC7E16539124D044D71969C857B75F7E18F85AC73E86FE17893CF76165D34C591620E2274FAAA5B3857B43BAF`),
		M:     internal.HB(`38191DAF0E988F076C04B412E24B7EE2C231E12FE855F582FAB7C8D76C97871E35E9EC102F5F8DDC531CE0897AB3439536793A2F00B6A8A244EAE3BB8890354540913DE9F6F05D17F7C67FAA949AE90B33B77F70B7861582078A378CAD8B89EE6311A515FEA41B840718F563803968FBACB2841C337E30649AAAA04B71E41587`),
		X:     internal.HI(`16DA7C87F0A108697F8AB48889391DF66C013FFD8E9C51A1675451FDEEB0A8D30`),
		Y:     internal.HI(`3F14F527E50058F2B67198DB3F5B1FAE4B747E4BBBBBC299ACD5A7274ADA4CEDEEE1E26C54349215C61BE87DFD8B96524480B56773C670FE30FD0EED1ECCAC9E32E056F6DE4D84D1AD84298DB40560154FEC0AB98BF9C959799D6308A95262663B5255CF1EA70D0DB8911B4EB89F8B4830F336F2696601CB9CEE89142319AA152850162CEEDDE0DB74F6B68552EAEC1363175534167C816E2BFC909472A0A74956B01A2895A8A6081B2400AE65C07A809D6EE330E4E99310FBB6A6B012FF6B2E6C7D94E7CE207DFFAF3A4E8A3831F19B727A56D5D42FD800014882ED4C2904F5E9F2FEA37CEBCE5B5DD129B57418B6405DC7B0D0BD2A52A8ED7561B16E7A7554`),
		R:     internal.HI(`950064264C27E84857E2A26E39A9555BBBFEB296394CB2009A0A7624F41DF0CF`),
		S:     internal.HI(`9329114F1484060576C65D177034BF7863D8E1680C799B7D8D8A902D02D2177C`),
	},
	{
		Sizes: L2048N256SHA256,
		P:     internal.HI(`BE66A314B64965EF75BCB16C512F21FE31F3C29FEC63219CD386DA69B7B35B7D6C0C2C33895B1CB9FF2D4538FD110A14B820551C8F1DE61BCF65183FC8D558B7A0220B18538F9B2EA29DE8F4E5938C896590013B85EC9C380E6A91503D0A7DF358F07D2B639FB8632E0F494936277308E4D3E0D18A997978A9444871C0392700AAE73F19962D532497AD601F486D28B8CE30CD0732B0967B37C6F3D7D6D01F9EB1FA566178595FA469676057B03AC6C3EBA88BEF8E0DD097206646E1EBA78CD71DA12EDD1C382C8C40421EE93C938F31A3A72B706C3CF14908CAB9D5EADCA33A2F36B1FF68F7C75473A45978B5ACC1A07E2C3AF65116BB68FC988FAA4BC10DAF`),
		Q:     internal.HI(`BBB0A8CE033A1D66D455604BE393DB6D6081CFE51390EE8F3D25E1257EA2CA2F`),
		G:     internal.HI(`2646B14B1ACD5620F1723BAFF35D9D41CE18D8AF87416A932312187212F7326D81A15B5ADD512C4584E936B77A880AE8503400A5565C178A1A5B70CDCB4A208B6DA3E4A01E6E5E15CCE7F26E437578EDA05CF57E88B3C5F97F200D0687EA03E0EF8A0DE29E230D2603F2842B0A996D644A5EE90BB77A18C4B0E3EA355699919762FB047C7DEE1CCBD53F527D261C4A666F26298AA9EC30E476D8001A1673E3C8F17224B677517F770C2CC0F514CF1269EA96DCBABA2F7ECEDC1F3A557EBE852AE06DC1C04D33C771B61A1040FC0E19E743E10308CD87CC936FE72D3F536E49AA883D62292380D6BB81BD18E219E5FDCAB083F33711A97EE005D9187416159756`),
		M:     internal.HB(`7BF0310DF317165C04A59ECA1F8B6A0768B10F02281BA26EAD66722100379DDC3BB9E07394274EB3C40EC418E5D0CFAC04815911E21E48ABBF5A755618E89D24A7FB290A37F67C0AF5EE2A2ADD8DAC52EEF29060BE8C310AA629E25C8E1D40F31A4AF03A42EE0BBD94EC0401A38AC7426D5A58AB30AFA2DB0B776AC4C8BDEAC7`),
		X:     internal.HI(`1666814C215D1030EDFC91D6BD2224B332816C6816A1F4B8F389E1EBC88474877`),
		Y:     internal.HI(`2926CC20363695C4B99F0EB32E3DF77FA53F1661CC8D75CED6979F8D477FE07487166D587F8671D60E108114BB35CAFC978B40F021EFB2A8C02DABA6681B8382A3E1DE16D6D19CC92D396EB8C4B199B769FC6C96CEB255213AA452878030CC3E0C1CC00120CDA0E046C9B7E37A40C88AB6F22CAAD6DCCB39609883314A27DF156DA9CE3ACD879DB3F6D84A5B97FBC9292C9C9713A15D0D21F739CBF3B46E568CD30C401E75858DA024D299B8D04AAD9247BEE4EC6047FF145B9009D6D49C114504AE1A5106B247E5F30CB1BBBECBAAE07F79E816C7743B8583F9014B0126130BDF597B72A7B43B4C88ED5BFBD60B0FCE22E4A40EF30349F3BA78AF3462D6A6F5`),
		R:     internal.HI(`14D008484E138B4870B2C99C99B012BBC0120C0381D5098020EB1951BAB15AC8`),
		S:     internal.HI(`4DB3AB15FC71B5D476EE6D776137D23C1AE47CEB5959D111E5AEE02FDCE8DD70`),
	},
	{
		Sizes: L2048N256SHA256,
		P:     internal.HI(`9A7796FD58BE26B9223DA3F2DE5B53EB54DA5DD6C5A8651FD7573120950B121FE383942A6AA26271AF403D1592B4C3456130BEEFD696FB6FD1A0AA98A05979823617F58E128223FC1B35E283413FC727069864428B8478F22F51B9E0E7AE5C6DD9A700CA82EFBEF243469AD5A8C4666C2DC7D5F19FA12E6155B3265878441B3EAAD9DF44168EDB6E57E14954A280ADE37F4F1E14992243B02D5A1AA5EDAACDDBFAB13FB14EF0D14E7007F1BE951A9B025A74550D1F1117F2F39927A3BC836B688BAC6612EE94A95C2710CE2F22E2B9C15AB335676E9C213C54A4D18F801274B2B4DB2B8EF5A4E1D324FA8916755B143100C62995C7228D86195CCD3D6EA84ABF`),
		Q:     internal.HI(`9315C34E52AD8D10AFFA9800FDC92C7418FCBB09524CBA279670DD4CBC4B4B8D`),
		G:     internal.HI(`50B6765AA2A5FBF123BF1DE6DDB167C4DA7D67260A97F9F1C55454CEEFAD08B6F8A52F96AB8B7BC297E33516E7BFB29CDFC1A833F31F42C84C0C026CA452C73192053DFA051CEFFD41A0EE730E0D76E8CB378C58B8A543BCE484913E99CACA2D78C256C18893F157FE3FDCDB3F945FA6736B024765BD4E6DCE99150F73EAA4CEA379AD9992B61519A5D4AE9BC803143AA7FD0BEB49E08371B2CBE472C2503D218526EE780733A1074801731BF1455D26AD7541E38EBD340907DC22C111DB8F23AF76AD1C4136561693498AFBB3AA02E36F9DA04F39FCEEE6943D084C4287692D5197FA05690EA2CEEB8681CDE3901CAC9FF89CCE52BB1801B9D8C890974DD516`),
		M:     internal.HB(`41B9948E33B8969474AEFABF6A549CA9959045CDBC1516D8686F304543FA2F7D9E6565389AC473B43CE0AB32F4B9DB195572614BFE8C3699913FC8BB0D45D85B08EDA50A9A45A72DE6B1A8C39ECF6707EA602CC725AC649D299742A449556407A6F20EFC6E32EDECC1D6F58552EED74BEABF168FF04D3646292E19242FA1058C`),
		X:     internal.HI(`D2AA6108B53CD493EF37C62BCF365AB41A66D8B740EF541AD1BD4BC03DF87920`),
		Y:     internal.HI(`774E910F8C7BEC9242D51032B4EEC4FAF490F23445E5D920E97EFDD216C05A1ABA1CE045F3C545558F5384A60A83F39EC44006DB23D9B581A33FC21DF2D241F5102CCD7CC67FA98EDA2EF272C1F6138CA7154CE3C6757539367F0E4A73547BF80D7D0CB498CE1C1C8F8B343D00F89641DECBB195BD6E036ABBB22C3DE46897C366A2913753C3245E1251E78CE78DE4B40654EDFD20E97E57FEF6C0488E39FC1A706F2621FE85E527268C51B5CBE09A625FCFE4D44FB5774A5D66E75AF3A4A5F82A0A9AC4BA1FCCA551CF4D1878C34DFE69C1E5B877EFEB1FE4246BC3E7538BA7959E38BEB45AEF18C041965DB213810E958ED92E1097C359DE4536013D1A5799`),
		R:     internal.HI(`989D62ADBDBE4BE5E0EBED94544865CA970FF0E3815D7DB02B5E492F8B5C2B1`),
		S:     internal.HI(`418CCFB568F3F2D99356D91E07CF20851746A4C2047E4EB9137AB2A81758EDA6`),
	},

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// KCDSA_(2048)(256)(SHA-256)_SVT.txt
	{
		Sizes: L2048N256SHA256,
		P:     internal.HI(`D837AD8CB841C9B7632FA91C157BA723ADCDFDFC97ED24EDA2FEFC7CB50A84A885628D851BDC2225ADF6CB179CFCBC3E28CEEE4531C1FEFCD06E4908AB7429298492C74D6FE3B7BFD56206487A4C4F2497B1175C30F7C5F3EC333F4D1CF07E1AE09DE590321035025562708060DEBCAA543CFC8E3937643F156C448EBB7CC710B5C5DE216752DCFAEC0B5C342FA42A8B6BE0E10E3FB49D758A76F2CACA6A19F9B506BC6BD7C7C92C5AFB37A3F917C33AD5928F991DE4C8BF88783F1507A80495984019E9ABE3FE255C6BFF7EC895740BE187E3CB06BDF29BE3B41141D6A2BF7638F17D049E185C852FBBE358913CBFB2EDDF2E91F069B2D5C4908A390A3A5B5B`),
		Q:     internal.HI(`C2FBF4F74B4217A5A6BDEC4B4485D5D1F31976BCBE726BB2C276800D083BD341`),
		G:     internal.HI(`90AE2720DA48AB2B9C30EC6D05C3EBDA7B044AA3FBD5BB1C750C246CF54125F49EBF4A45B3A50493F76C9A94DC7F8BCCBC37B003AD914B5F5BB51428937FC116EDFAE6641641F54E96F8DE0027CA580821C5612F938D8A8A99D2E8D9C8F5B2263C4076D9E0E664D751F49E0F3FBEF889029C0448E1DD1F1A99FCC4051594F385EFAC480AD66DDB920C3069F81F24F1191F0E256DC9A3095ACA55081A1CA1F9B3D6654B83CFCDEFE83AE16D9E5562900F933F4258B1D48EFE5E7CA77A0550003B86D9023C641F9A80B97CD34C8F9CF3C2A350A27A74E7F438166C5DCB1C184E005C03E97EBD2494A1F849CD2A2B05AE9D3B793285E9B56AAB7FB305FBD1F6F09A`),
		M:     internal.HB(`C0A2E1E6BE81DBEAC94C0E678809141E8921EFCCC67FD9585120AA401E6DF5E6381D81BC6EE154CEC1BDE5C8882FABE2B00FADDC095D3971F1C36A635F2AAD98A704DD58E0BB2E315AF08A8A07E07D6842B36E7CD5FE39E8B53ED8E64A9949AFB6A355D6047C5B7B9D08CE73FA848DE1E949B9595E0532F92C286BA094D7B2B0132A957851FC96BB9C13D06EDAEDAA275AF9594B607E4087E900E4927B6998CA76B58CA9C97A62AA6F04F292AA5468BA4FD0674F1FDB4C1D8B3049E6C63D74D29E4677FBF39D0AA539B7DF1AF35A23C58CC43E8E68F704F0B607ECEBC2A7874552C6D52ADF74A0B621F08D6AC9080119DBB486548F16DFDA19BD641A4765DB48620772162876008B5B5B3710C5CFEE3011662C1A71E31C6068739014B39C3EABA5C45ECCEB82CB7C1F8D60C00A889F2C0987657C7070C1AD612F99A1ECD84CE1FA9FF47DD2E06E89AE00D456427590D5A5ADB0437B399E94C8E4EEB15F0F63094921D5840E3D1A5A521BA7D79F3D089DD258D25C052148916C67475E059EAEE981BDEC6555B2CB3D5B283670DCF3129F83ECDADE0B7320C8227BA4E85A491DED98CD6AC8EABD432A245B23753C10839BB2B81C0710E24B04C7C54CB7663F682B9093C88193460DC20ED25B638774F9FB63F2383F2288B9B940D8D05DB61311626E38C98BA29B7D4B828E11DE1168D7D1A2B71113`),
		X:     internal.HI(`28B14ED806D64755EF9192F36AEC7AADBB68666910A9BC039E532A322E7BFC91`),
		Y:     internal.HI(`8A249E41D76190398B78B15B2AFFAB64A433D692493063EDB3BCAF9A27A91C3A47F99E79EBD9CE5763ED898BBDB89D6013206A3CC16DB2D88D935D59A55C54322782BD9957613F2A3704C1472EC34183E39D613497F20915FC959FB564AF5EE3D47AB1E31674B275A5178A638A72F5FC4D783901B20C8BB56E124391D6B7C4551F32D41C0A39C2E8B5312BBEC18E17DF75A7F559EB3304FFC54D4FDE1CDBFD2C22DD5CB354F050E18691E91C12CC7B83E85AF38D892E26491B01D9D0A8590D64C290EEA34331A9C8AD59C4BEA966A89E54A25CAF22499DFA01845566D210CDBA04F3596698D211331045CC921895D81FB74587340CB426DD9E6ED36186B26FC4`),
		R:     internal.HI(`7695E32661D5B59A0B6F193457D61C7B4ECF53E7DFA21C5166836EEA26A8674F`),
		S:     internal.HI(`147DA0DD0D162E933FEDC4E784B7AC2C82BF9ECB2D9C313FED7E1AF128243C9D`),
	},
	{
		Sizes: L2048N256SHA256,
		P:     internal.HI(`D837AD8CB841C9B7632FA91C157BA723ADCDFDFC97ED24EDA2FEFC7CB50A84A885628D851BDC2225ADF6CB179CFCBC3E28CEEE4531C1FEFCD06E4908AB7429298492C74D6FE3B7BFD56206487A4C4F2497B1175C30F7C5F3EC333F4D1CF07E1AE09DE590321035025562708060DEBCAA543CFC8E3937643F156C448EBB7CC710B5C5DE216752DCFAEC0B5C342FA42A8B6BE0E10E3FB49D758A76F2CACA6A19F9B506BC6BD7C7C92C5AFB37A3F917C33AD5928F991DE4C8BF88783F1507A80495984019E9ABE3FE255C6BFF7EC895740BE187E3CB06BDF29BE3B41141D6A2BF7638F17D049E185C852FBBE358913CBFB2EDDF2E91F069B2D5C4908A390A3A5B5B`),
		Q:     internal.HI(`C2FBF4F74B4217A5A6BDEC4B4485D5D1F31976BCBE726BB2C276800D083BD341`),
		G:     internal.HI(`90AE2720DA48AB2B9C30EC6D05C3EBDA7B044AA3FBD5BB1C750C246CF54125F49EBF4A45B3A50493F76C9A94DC7F8BCCBC37B003AD914B5F5BB51428937FC116EDFAE6641641F54E96F8DE0027CA580821C5612F938D8A8A99D2E8D9C8F5B2263C4076D9E0E664D751F49E0F3FBEF889029C0448E1DD1F1A99FCC4051594F385EFAC480AD66DDB920C3069F81F24F1191F0E256DC9A3095ACA55081A1CA1F9B3D6654B83CFCDEFE83AE16D9E5562900F933F4258B1D48EFE5E7CA77A0550003B86D9023C641F9A80B97CD34C8F9CF3C2A350A27A74E7F438166C5DCB1C184E005C03E97EBD2494A1F849CD2A2B05AE9D3B793285E9B56AAB7FB305FBD1F6F09A`),
		M:     internal.HB(`027A2A079061C460F543D07707EF74ACB5F17EC2B2B49E4CD6820FD83A43F200A36C1C123305B567F425532441D9F07E98D47430D6D32F4B6C1EE7009313F29F0812F6574EC8B5FD1334A568AB4A959D33AD2E6864579B919C9A15492570A70B4371E8881159F8C4B1968F26E0B1527C6E064EBEA1B3CCA3182058B71D10341D1197099AD5EE1C25D27DC388A2078057D86F61CEFBE113CE39E45499D32D2371D5995BCF1A462450241EDC00DBCDDD2FA982DC7A0663292802229985C58C67709B95C5AF89A67F3FFDB95B479F0A8FD0C0E019F07D452E8C1B1C9D589A7B5B4816AF1708F2DE02B15B95AA5F284E27CDA6325EA34718AD9FD61EBC3820CCC3F0A2160AF34F42E930E2021C92D9B39A7F892AD64E6FF696CE2E7C3F934DDCCA2642FE3CD0BFB3DA0DE157EA723BD9490B418195F62982424FC290521E41910272A2A537468B95E1614DF437D902E7FA5B4EF998E7D4E6721DDDBF0BD842556823164F6B4524D7750CC43F0DE8088EDC22D75AC2B3F2D4500071726A07BE1D5C4E984B2E0623F071B68BA85D09500686DAAC76DF3831876C7F161E53384C66ACD4CEEEC30747DE1BD290A603EF0411F8C74427A59865C2BFF40F3E9540AB348A5A019550117A271F9868B9BF9277F598F2C04DAD408ACDAA7B4656E53FC113905026A5E534CCDA940851673C3523843630E6D27EE3`),
		X:     internal.HI(`14F5DA575D821A99053CE0CBFB27622F75A4B232851BE89D189613AFC40E9EC0`),
		Y:     internal.HI(`746DED590B2A5685B823F6B042BA7B4DCC2640EDA74FEAE13AC051CFCAA1317052BD69014822429220F3D1FD78E71139AF94326D3A19EFF11CAF4ED187047F64E172277811362EBA9FCD699FC85ABCEF157ED7190DCDBB6FF91E844032BB5D32A437BF040DBC44BAA3C9627E293AB38B462BD9939FA4F6F44A0D112D52CC1048732162D81BE96872D77B848507B325DA93A645B209FF2FF99A4EEE572363163A3DE4B799448BA55515C9BD50C3EF07D39C3388D9AC823921281A1769803CD24C57EA983F699E3E13C0B378418A005B02026F43A3CEC994C86C5A63151D25C044454CB8E47FAD4A3F9766D85B3CE7F488FD4E1A111AF54B25D3AEC963A47BA3E5`),
		R:     internal.HI(`D86891CFAC234DDDCBDA3C6F81A574495B087BE4F354804E02ED61FC28EBF11B`),
		S:     internal.HI(`8E5D1F7058740C2A72E5F3CCB4FE0356CE50BCB4156EE20A0E058B97F60FB647`),
	},
	{
		Sizes: L2048N256SHA256,
		P:     internal.HI(`D837AD8CB841C9B7632FA91C157BA723ADCDFDFC97ED24EDA2FEFC7CB50A84A885628D851BDC2225ADF6CB179CFCBC3E28CEEE4531C1FEFCD06E4908AB7429298492C74D6FE3B7BFD56206487A4C4F2497B1175C30F7C5F3EC333F4D1CF07E1AE09DE590321035025562708060DEBCAA543CFC8E3937643F156C448EBB7CC710B5C5DE216752DCFAEC0B5C342FA42A8B6BE0E10E3FB49D758A76F2CACA6A19F9B506BC6BD7C7C92C5AFB37A3F917C33AD5928F991DE4C8BF88783F1507A80495984019E9ABE3FE255C6BFF7EC895740BE187E3CB06BDF29BE3B41141D6A2BF7638F17D049E185C852FBBE358913CBFB2EDDF2E91F069B2D5C4908A390A3A5B5B`),
		Q:     internal.HI(`C2FBF4F74B4217A5A6BDEC4B4485D5D1F31976BCBE726BB2C276800D083BD341`),
		G:     internal.HI(`90AE2720DA48AB2B9C30EC6D05C3EBDA7B044AA3FBD5BB1C750C246CF54125F49EBF4A45B3A50493F76C9A94DC7F8BCCBC37B003AD914B5F5BB51428937FC116EDFAE6641641F54E96F8DE0027CA580821C5612F938D8A8A99D2E8D9C8F5B2263C4076D9E0E664D751F49E0F3FBEF889029C0448E1DD1F1A99FCC4051594F385EFAC480AD66DDB920C3069F81F24F1191F0E256DC9A3095ACA55081A1CA1F9B3D6654B83CFCDEFE83AE16D9E5562900F933F4258B1D48EFE5E7CA77A0550003B86D9023C641F9A80B97CD34C8F9CF3C2A350A27A74E7F438166C5DCB1C184E005C03E97EBD2494A1F849CD2A2B05AE9D3B793285E9B56AAB7FB305FBD1F6F09A`),
		M:     internal.HB(`F74BEFE3999C17C3ECD98B7CC7758CF6EB30410C62AB15081B26A7827D6529D716589F14FD74DD8D2E5CBEF36AD65D2AD393DDA9EADA89AE1597C60E53DBB7E34C4F00E65FA68AC1DB9CB9130578F27D17B0DB75E7A2492F49FA305EEFC58F1BD7B9B5AF00BAC89A842C038CC70C79A4DA64E2BABB1D44F9ABB6EE9626E8A4E9BBC869AD625ECEBB7E4729F62A69AABD0235A1AB108FF3E5F5DA11C21430701DC556D10755685730E8D2C1D1EB90C24C4350D161D7615630AA1DB3D61BB1F6F28AE3A8C9EFD7A66BAA57668514AB883E138B2FDE4B27F68214E1F6AEE9A9C007669BB0EC8DD28A48720AC062F4094AE9B563840DED9BE2EA472B020F717BDF677F4DB54BD5A75609B6C77AA12223DE0A30FE9EBF879FB4DF1DAC08A3EEB5EF82C1748AAFB6CDE65AB7124B60809AA3C5562955AF2B3E8B403ABB4300E50B1230E1320BC364E29E4D7A15F0AA33367FA8C05B897E32AB10540758F3A22259F3B45D4F1B205DAD6B5ECEA52D6CADE7E0A7CEB120B73F3F73CBB82A61EBB7A4C2B7773EA542681CC2704B3DD180A5C7BD20A9EF0ACC3B7F6CBB4881E02A02193B483F179D9092469ECE4E01B1A61B1693D942843E1758143DA37954C890A50CA1E2879EFE58316A842B00BCAA85573C6AFD5186BCD911D0AC6BD6437B3C8DFABC66EE3ACBD0E3EE81A14DE1A0AEE8CACF2259B18B3E`),
		X:     internal.HI(`1180232BB85CF7149E8BE4D15B61B02B5565B298276083AEA6AD3DD41D60D5B4`),
		Y:     internal.HI(`19A40B118CE05EE0ACB0F98DB0879590B76E0A86A53CC7BA3F1DB39859BBD68C3AD749B63FEE4FF9CF13E652408FD53F547CEEE435B95D9C41EBEBC8ADC206DCF97A0BDA7FE200F270FCFECA6B3C8A3DE9BEB2ACD964A48EE442B4E422B5A07BCA7CDA14200465B787BC96D5609E1D5427A86717F4DB4E3A6493F0D250F9F50DB3EB83D519BF2FA9D8B4752985E5D720DEEDEE39C7407A0B9607B5CACE912ABC17600DD390AF851C033C74B304585371708B0A1A79DD0CF9DFBCA30827F44CE08D8D6697C8A8384FEF5154011CB5AE12533CF8CDBA1A2082A620708D85EC76F51E91CD4A6FFC98EC141BF443D400A0BFBAA1A42137B22118808898F0AA14EEFA`),
		R:     internal.HI(`80713F78D3E493F9D127270910661D6189D6D075A853E14B72F3B35ADFED96A9`),
		S:     internal.HI(`57F0E3CDDF1157217F02E610277B144DED59BA68F4E8F5AE7AD586D848AC75A5`),
		Fail:  true,
	},
	{
		Sizes: L2048N256SHA256,
		P:     internal.HI(`D837AD8CB841C9B7632FA91C157BA723ADCDFDFC97ED24EDA2FEFC7CB50A84A885628D851BDC2225ADF6CB179CFCBC3E28CEEE4531C1FEFCD06E4908AB7429298492C74D6FE3B7BFD56206487A4C4F2497B1175C30F7C5F3EC333F4D1CF07E1AE09DE590321035025562708060DEBCAA543CFC8E3937643F156C448EBB7CC710B5C5DE216752DCFAEC0B5C342FA42A8B6BE0E10E3FB49D758A76F2CACA6A19F9B506BC6BD7C7C92C5AFB37A3F917C33AD5928F991DE4C8BF88783F1507A80495984019E9ABE3FE255C6BFF7EC895740BE187E3CB06BDF29BE3B41141D6A2BF7638F17D049E185C852FBBE358913CBFB2EDDF2E91F069B2D5C4908A390A3A5B5B`),
		Q:     internal.HI(`C2FBF4F74B4217A5A6BDEC4B4485D5D1F31976BCBE726BB2C276800D083BD341`),
		G:     internal.HI(`90AE2720DA48AB2B9C30EC6D05C3EBDA7B044AA3FBD5BB1C750C246CF54125F49EBF4A45B3A50493F76C9A94DC7F8BCCBC37B003AD914B5F5BB51428937FC116EDFAE6641641F54E96F8DE0027CA580821C5612F938D8A8A99D2E8D9C8F5B2263C4076D9E0E664D751F49E0F3FBEF889029C0448E1DD1F1A99FCC4051594F385EFAC480AD66DDB920C3069F81F24F1191F0E256DC9A3095ACA55081A1CA1F9B3D6654B83CFCDEFE83AE16D9E5562900F933F4258B1D48EFE5E7CA77A0550003B86D9023C641F9A80B97CD34C8F9CF3C2A350A27A74E7F438166C5DCB1C184E005C03E97EBD2494A1F849CD2A2B05AE9D3B793285E9B56AAB7FB305FBD1F6F09A`),
		M:     internal.HB(`EC5995BC0484DA3F9A2F361D025C2FA52F1E3ED47BEB024EED6105C0261F627FA69D47B07FAAAEAF347F4113D3F8C2030E01EF3860B84161FB4A6A3D9B170631EA2B60999987AFAFE347B93E8E3747EA21CD4B6975211B5742DEBF07D7BD9E0DA08C88C2AA4F7BA0C27A06A55DD9C297F51FBB68F9B0348500A7B05967A6AEC4F771EDE2D2E0BACD313318B934473B2F5EBC32A0F195362CBC5172D4A20CC4EF69B14516FCC01C66DCB4694FCC8EC3BC78902BE028ABD56F42B4C185A9D3750CB34DCFE1D61C5887B268F8A7AA687433A6AFA6623473CD5EA8CDE2DE60875E84DF6F5032DBCA2D31F0E14D6817337071935730C77216E0EE4DC5A0BB785B26A83C66175E4B47624F15DA7AA127F7DE3835EBDA180667DBFDD5E8505E662B78AF61ABFA212FBAC6B2ED3615CAB464F236C7F93EC7DDDE8F5230AECF736A7A0BB930DE58A057EF321488FE40C061D0E300CC3580ABAD9EDA9B92B6800C8B749BCECEC9176A5D5C84184167A2CB993DF40D107B4807F1709D6D7AC550A697EDEFDFAD5BA572A11EA348B9C96D9B8F506EC6A8D0C982EFC5C547AECBB422255FD3C384AEF9164CFC8016DBA858463D5BA176EE61BF30B4B6448F3ADEA6CC95F01C3B5302901A4F6211DBD8AEA44C6754E85087816A8714071693763CAD560B69A9EF62BF72AD636755DB2AAE189297DBA3715FAF966A`),
		X:     internal.HI(`478DA75AE6CCB3F82403F9E2219C99A9F0396A5EA1D553D873CD382B000E1D2E`),
		Y:     internal.HI(`83A365DFA5536C8D14DE524F3631BC9F420322EF6F4EC5498FC5DD9207B06C1994465378E2A24050910DE88E624A629188BD424F99A4B1EAFB013FB95F5F8DDFC4D18B0794F34064AF9DA0C0F134308E78E328323E36549FAA402CD4DD32DBCC4F77B6A918C92E5F9F1D8DAC6DC0AF9633D9C5C5C2CDF4CC816589665235B9DA2DB2A4B06893AE7C400CDBA5EA26BFB5684A43C7EB6E5B47D79DB23C304A7B2B30D2D3517D8282AFFA10B601C7FCCCFD2B8B416C36B458539609E6FB9A76B3CE1F6069ACE0A416AE0C47A4D2E9F7304008162DA5B08E3002B75A558818D1F56E4D6A8C0ADCD3F0056F1FBCE8233B238BD77DC64292C3B8DA8BE45C93F528DC60`),
		R:     internal.HI(`71B11FCDA6E2840EAD7232CB56A4962FFCD0A009D274EF493550742DBBA8F672`),
		S:     internal.HI(`56357EFFB650482FCC325EC25E1D6C9530D462B6DC4EB72208E75FA05B8D621E`),
	},
	{
		Sizes: L2048N256SHA256,
		P:     internal.HI(`D837AD8CB841C9B7632FA91C157BA723ADCDFDFC97ED24EDA2FEFC7CB50A84A885628D851BDC2225ADF6CB179CFCBC3E28CEEE4531C1FEFCD06E4908AB7429298492C74D6FE3B7BFD56206487A4C4F2497B1175C30F7C5F3EC333F4D1CF07E1AE09DE590321035025562708060DEBCAA543CFC8E3937643F156C448EBB7CC710B5C5DE216752DCFAEC0B5C342FA42A8B6BE0E10E3FB49D758A76F2CACA6A19F9B506BC6BD7C7C92C5AFB37A3F917C33AD5928F991DE4C8BF88783F1507A80495984019E9ABE3FE255C6BFF7EC895740BE187E3CB06BDF29BE3B41141D6A2BF7638F17D049E185C852FBBE358913CBFB2EDDF2E91F069B2D5C4908A390A3A5B5B`),
		Q:     internal.HI(`C2FBF4F74B4217A5A6BDEC4B4485D5D1F31976BCBE726BB2C276800D083BD341`),
		G:     internal.HI(`90AE2720DA48AB2B9C30EC6D05C3EBDA7B044AA3FBD5BB1C750C246CF54125F49EBF4A45B3A50493F76C9A94DC7F8BCCBC37B003AD914B5F5BB51428937FC116EDFAE6641641F54E96F8DE0027CA580821C5612F938D8A8A99D2E8D9C8F5B2263C4076D9E0E664D751F49E0F3FBEF889029C0448E1DD1F1A99FCC4051594F385EFAC480AD66DDB920C3069F81F24F1191F0E256DC9A3095ACA55081A1CA1F9B3D6654B83CFCDEFE83AE16D9E5562900F933F4258B1D48EFE5E7CA77A0550003B86D9023C641F9A80B97CD34C8F9CF3C2A350A27A74E7F438166C5DCB1C184E005C03E97EBD2494A1F849CD2A2B05AE9D3B793285E9B56AAB7FB305FBD1F6F09A`),
		M:     internal.HB(`88FE4DD4DD793F5E6E40762C6309C8543E92A3076A20373DDCA98E9422E112C88EB99EAFE3948E2831C5E0D63888EA1D171BAE46891C59267FE2EB405FE097C7A9D4E65C53DB18FD9812DD0C93F3DBC861161B380E70D7D72CCCBAFA3B9D1156C9C6047D80954ADBEA8D76F540225174FC1D5CC482C4F2D2A07518AF36D68A98C34B004025D137C659E7DA4174B6A847B4130B5BD28D90A07F2F49CF3A0FB2B9556B045865669ACCFB186424CD17E8723A20ECF4550541D35596BB559895E4EC25726B03CCF2D802D15F935E4F77BD2C28B6E911CC2F3E03968F02C4097C216CC2F7B2034DDCFB85C445103567CE7CB4009014B95DD664D39E45D926AEA3137DA1D67FA74154B87BA398AB76EBDD24512DB0A87D968C3DEBB12C270D10AD0B6A2037A2C26E4F691226715E79172D5A53015E077570ACF8FDFC00F69320070287868711AFFD8D127DED30481C910E6C11B52DBC42495A6BC292C37D5C37E69B2E017CE955819560FC817CB2C4669A4EEA6BF77A0BE87F18FB6FC3179017471EC3A613731EF6B5A5D251460D610AB29F462FDD1B817CCF2472789449E3E7EF7EB174931C00BF04DD4EB5C4F1685BFDA594F149A2DD9DC561F87810C08E3A6A546C50897B76A761ACC5EE7720D99EED4C4B8CEE39DF49A34467245D5055080DE26F0ACA4E85E1735C932428803880BB2AEAC1C432CE`),
		X:     internal.HI(`9B94C0D28EE7DD649E95071A71E3B0EA25A8EF593642E9E65BB0521865F1B877`),
		Y:     internal.HI(`D5F31B4B5C1A02DFE1DDA702C7F5F726A16C115F1CFE626672A0ECEDCCEA33BF1B6571C52C7C96B1E59442F37C4538141D32B24594188FD4D598F7E4A2096249482C3C3703C12E1F7B2BAE65DD044880CB43FF62632B67E2AE12C9F2DC39EE924FDAF5F5DCAB794D75483FAC896B0E722D25B53C560EF6FC839DBEE6AFCCEAD4EFE6507109493E541C736E9F285DF523BD0B3C3A873536ACDC8D44ED94A54A55136F8BE3B72B72FA7E95371515F66DC47DB840E16C0589D25E603835D7D14734619A30F2F2F6BD42F9492DCEA05B2906F59FE4AE03E79442FF2FEDD912F90279CA97EF91B80E1DE3537D4559F44C557726A5B1F09C68E9F41C23599E84D5A66A`),
		R:     internal.HI(`5BA474F6000CB69EF1A687ACACAA3B96C97F5944222BF22E16C29471EDF1BA24`),
		S:     internal.HI(`B45DD1BCE9497F380F72DD2681FE5093962051759ED580447A1CB2F95D45EFAD`),
	},
	{
		Sizes: L2048N256SHA256,
		P:     internal.HI(`D837AD8CB841C9B7632FA91C157BA723ADCDFDFC97ED24EDA2FEFC7CB50A84A885628D851BDC2225ADF6CB179CFCBC3E28CEEE4531C1FEFCD06E4908AB7429298492C74D6FE3B7BFD56206487A4C4F2497B1175C30F7C5F3EC333F4D1CF07E1AE09DE590321035025562708060DEBCAA543CFC8E3937643F156C448EBB7CC710B5C5DE216752DCFAEC0B5C342FA42A8B6BE0E10E3FB49D758A76F2CACA6A19F9B506BC6BD7C7C92C5AFB37A3F917C33AD5928F991DE4C8BF88783F1507A80495984019E9ABE3FE255C6BFF7EC895740BE187E3CB06BDF29BE3B41141D6A2BF7638F17D049E185C852FBBE358913CBFB2EDDF2E91F069B2D5C4908A390A3A5B5B`),
		Q:     internal.HI(`C2FBF4F74B4217A5A6BDEC4B4485D5D1F31976BCBE726BB2C276800D083BD341`),
		G:     internal.HI(`90AE2720DA48AB2B9C30EC6D05C3EBDA7B044AA3FBD5BB1C750C246CF54125F49EBF4A45B3A50493F76C9A94DC7F8BCCBC37B003AD914B5F5BB51428937FC116EDFAE6641641F54E96F8DE0027CA580821C5612F938D8A8A99D2E8D9C8F5B2263C4076D9E0E664D751F49E0F3FBEF889029C0448E1DD1F1A99FCC4051594F385EFAC480AD66DDB920C3069F81F24F1191F0E256DC9A3095ACA55081A1CA1F9B3D6654B83CFCDEFE83AE16D9E5562900F933F4258B1D48EFE5E7CA77A0550003B86D9023C641F9A80B97CD34C8F9CF3C2A350A27A74E7F438166C5DCB1C184E005C03E97EBD2494A1F849CD2A2B05AE9D3B793285E9B56AAB7FB305FBD1F6F09A`),
		M:     internal.HB(`9F17E4F581B3F6113D56767DB908A8E31F66E4249516687DF93C0E0A7C470C2989D65CD5E404097261D4F115271F38DBD5AD054386DDBC2B088E5661419F046019CDBE3CD92BF23F7008EA4E6B9121882BD92A3F28BA48C1AD316D02FD90BFAD4550509E43C1275B03625585A8312AD59FB23B9F24FA657487866DE1171E69CF119B9C988CC8E44DBF3772428C3AE256B5E76814A65334223FD9B81782BBF52E84D177EFA5A92F494FC4C6334B4FA047FB12287765E19D52845CF8E8B43C1CDAB200FD8E0935D32765321E2EA17C828B05B13CC89F29522F0C271EBDAEE16389B21B908CBBA56769BF8C9132D2346FAD732EAA311B19CB90973E652AF952129CA6FDDE23459A46391DCA7C66AB5016E2E8D8C401270549F3EC8B4EE8A59C3F19B76CDAB9B91C95684DC986188016EE0512E721B19AAAD57CDBDFA3DB4A37C3B21814BFDBB29E426F21509BBE2D2F3698A77CA0E1D12E40FA3BF4760D090241BC0087133D51F80171750DF2F817AEF5C9639D6A5BB31E22E2EC6D20B189452338B2439FBB416B4E372C95088C2910F96B0D3DEE0EAE6FDD52D4D44421FBAE9ECE76AC785AB5A170323167A468D936D9F87131E414B97D980FE29ACCDF1854ABCC9F0DFB4767A9727C77E8D3A4236CF495643C4CAD520F4587101BE8971FB60E2B859ACAD598FE29D7FB68EB7C8A64700DC4036D41`),
		X:     internal.HI(`83133444539A7A5401E8701F35135608FA16C9788629468CD5C6417BA3657B3D`),
		Y:     internal.HI(`7A098E6329085EFC2258704FC95952B87ECCBF9024B4BBC0AB39AD6EA1285AA82F8A089AE4C3F94D648D76C1F92A89CC5508179B9947D3C3CAE07D3AE704D5403F2372B30C79F93932E39C1285A80EFE6FA2B010613D710E518E24DD45E419D502D6005709D0C293211D145A2BDA4FFDCC07583D4C3087DF5C170B6D475EE76D0B2D8734B85A2CFC5F6D90A0CD6A570E686921AF3856B4FFFE70620CAF1C7CEE6EAE6D04234732FC3CBB88D646F84FC1284AFDC61FD5276FA8EEC951C8248F467A1B5E2843BEADAF29A382BA7EB126B0AFDB8944EBE964F0D7BCC7570B1D61DB6DACB3F34F6437B24D64DD5D126207D67D3B257C583B3C032B8D88E1C0B0DDE8`),
		R:     internal.HI(`C6C3A45592F1CF6E75E7854B2C287DE0C5F846248096CDA24F50CD53F839F7E8`),
		S:     internal.HI(`6EF41A8F460796E5AF98CF3CCE4CF46036D964236003ECB0F44752D010FE2057`),
		Fail:  true,
	},
	{
		Sizes: L2048N256SHA256,
		P:     internal.HI(`D837AD8CB841C9B7632FA91C157BA723ADCDFDFC97ED24EDA2FEFC7CB50A84A885628D851BDC2225ADF6CB179CFCBC3E28CEEE4531C1FEFCD06E4908AB7429298492C74D6FE3B7BFD56206487A4C4F2497B1175C30F7C5F3EC333F4D1CF07E1AE09DE590321035025562708060DEBCAA543CFC8E3937643F156C448EBB7CC710B5C5DE216752DCFAEC0B5C342FA42A8B6BE0E10E3FB49D758A76F2CACA6A19F9B506BC6BD7C7C92C5AFB37A3F917C33AD5928F991DE4C8BF88783F1507A80495984019E9ABE3FE255C6BFF7EC895740BE187E3CB06BDF29BE3B41141D6A2BF7638F17D049E185C852FBBE358913CBFB2EDDF2E91F069B2D5C4908A390A3A5B5B`),
		Q:     internal.HI(`C2FBF4F74B4217A5A6BDEC4B4485D5D1F31976BCBE726BB2C276800D083BD341`),
		G:     internal.HI(`90AE2720DA48AB2B9C30EC6D05C3EBDA7B044AA3FBD5BB1C750C246CF54125F49EBF4A45B3A50493F76C9A94DC7F8BCCBC37B003AD914B5F5BB51428937FC116EDFAE6641641F54E96F8DE0027CA580821C5612F938D8A8A99D2E8D9C8F5B2263C4076D9E0E664D751F49E0F3FBEF889029C0448E1DD1F1A99FCC4051594F385EFAC480AD66DDB920C3069F81F24F1191F0E256DC9A3095ACA55081A1CA1F9B3D6654B83CFCDEFE83AE16D9E5562900F933F4258B1D48EFE5E7CA77A0550003B86D9023C641F9A80B97CD34C8F9CF3C2A350A27A74E7F438166C5DCB1C184E005C03E97EBD2494A1F849CD2A2B05AE9D3B793285E9B56AAB7FB305FBD1F6F09A`),
		M:     internal.HB(`BDE0047C3D8E3301471C0A55A55B16663A122AA9627C285851D460F0C3D73AE1FB77391367861D2ED9A3353DC53494B358B34E4CE01269EA6F859EDF0D74A6B87EE0230EE506E568497CA1B24EA7C95116AFAD5FACE38F15E8B0CDED2B0AC981E086BFAF2DEC2B10B58EDF41D72C870C55A4A9FAC041DF4DA0EC61FBB808999B463CD6E3BC01D7AFC32549200127E9D67D9A6EDC1C69E32F013B989157C2558C5D3EF63D18F519F6A2F6FE2B71E14FCA7F00EE6CCB7F707FFC0576E3B4788103582D75F8CE6068BC091EE7EAD78C652AD3ACE5BBDD8FA22B0D1DC7C9824FECD7F51673F973C285023423B788EC421B6179DED4806C8DDC4635BA22C67D55AB05776CD6CAA48376F1ECF1E4DD6D06AB01FA3D061B9955C90E007EE20267811DB5AA094DA007F38DDA7DDEB06423C197C566D990928DAB5FE67D732D4F0CB0E736E4304F55494A6135BFA62545DD46AA9054274A967A3CD95C4A08EF2640A9F6FD008D1B6E20A9D1A30F6F114C714EF46BE607D97E9A9BBA238718DDABDD1982A96233B715481706EB54C510F0C07CD08AC4C1A64A2C44CE19E0E274A4C7970702F79CF31F87856F00FC9F814DB05AE0471E03E6A17C9C2A438511F986E9A14BF434AC6407ABCAC4F9FE588C2931590C25ACE491D3D9EE28CB33B67868379C5D9814AE6BF188A60516D8B624F23BD488CEAFE46DD8`),
		X:     internal.HI(`0C3C61AC3E602925032DE9B557C5CBBA9A6CF5BF7C12705CB1F8421BB53514C3`),
		Y:     internal.HI(`1C9E25FA888DF46F6E7680B4C5617285100DD0A4AD9CA6BFD0103E577358864A7E7F503D6C35F9D05A8C791D695C9476A676DFC59CFCF646CAAC5D53900DF5A315BFB67B77A83E64A486BEF095CDF9C80272E9122C2084CA6318D0B0B0737AA28B566C76E02BAA4EE3CDE1E6088D8321DA95AF3A4D7229C190B5106CBFD746AC738D29771A36C1444B13B8A6CBF21C5BA40E546B660FFB64CB0DCACBCC98D68AF2D74AAF8D442336838046DE2C1CD6387346593BE25AFD5072536BC18345095033F4D7B07F36CF06F26C729409E39F30FB1EAB92C16AD73AD1742DB33E1743B879F05E1D756C0CC0A111F68851437C1C61F3721F5DEA9FDFFE9554C7932961C1`),
		R:     internal.HI(`236D5A08E14B4B979480935B03D790BBB75FC83D7DA7ACCCCE9BF89ECDC9FEAB`),
		S:     internal.HI(`3460C5BEC2CE608E1396283DB1FAB9DBA4EDC3FBD5824C9C63A1253D19C66795`),
		Fail:  true,
	},
	{
		Sizes: L2048N256SHA256,
		P:     internal.HI(`D837AD8CB841C9B7632FA91C157BA723ADCDFDFC97ED24EDA2FEFC7CB50A84A885628D851BDC2225ADF6CB179CFCBC3E28CEEE4531C1FEFCD06E4908AB7429298492C74D6FE3B7BFD56206487A4C4F2497B1175C30F7C5F3EC333F4D1CF07E1AE09DE590321035025562708060DEBCAA543CFC8E3937643F156C448EBB7CC710B5C5DE216752DCFAEC0B5C342FA42A8B6BE0E10E3FB49D758A76F2CACA6A19F9B506BC6BD7C7C92C5AFB37A3F917C33AD5928F991DE4C8BF88783F1507A80495984019E9ABE3FE255C6BFF7EC895740BE187E3CB06BDF29BE3B41141D6A2BF7638F17D049E185C852FBBE358913CBFB2EDDF2E91F069B2D5C4908A390A3A5B5B`),
		Q:     internal.HI(`C2FBF4F74B4217A5A6BDEC4B4485D5D1F31976BCBE726BB2C276800D083BD341`),
		G:     internal.HI(`90AE2720DA48AB2B9C30EC6D05C3EBDA7B044AA3FBD5BB1C750C246CF54125F49EBF4A45B3A50493F76C9A94DC7F8BCCBC37B003AD914B5F5BB51428937FC116EDFAE6641641F54E96F8DE0027CA580821C5612F938D8A8A99D2E8D9C8F5B2263C4076D9E0E664D751F49E0F3FBEF889029C0448E1DD1F1A99FCC4051594F385EFAC480AD66DDB920C3069F81F24F1191F0E256DC9A3095ACA55081A1CA1F9B3D6654B83CFCDEFE83AE16D9E5562900F933F4258B1D48EFE5E7CA77A0550003B86D9023C641F9A80B97CD34C8F9CF3C2A350A27A74E7F438166C5DCB1C184E005C03E97EBD2494A1F849CD2A2B05AE9D3B793285E9B56AAB7FB305FBD1F6F09A`),
		M:     internal.HB(`A1002FA700A3619F99CDA3B57A9C484FD2E916ACA8C2EE9203A240815BAB8106CDAE3828FA40D879E2AB0BDAC606937656326FF43209627A0B91967662626AC43E27292AC762C9C53359575EDE5A3D33062F0B490904E6B79188450B918E2F4081FA8C50BF41CFF47C43818C1BC8F1C1736799E2CA1E768D0B69ECA106155F082FA23265C27B4C1DB6BCCFDA61E9240139480D1F992ADBE8F6FA73BF68425552E682355A38186CFEE401CBE41AC0147CFC28A7842264A25AD8ED0A16E4CC30FB4FE2F74C128620010D364A6F38B4C5646646ECC29A6E23213ED8287F31CFDA8818F6227BC79E253344686867349704922BC7A9AEBD537D1EBC3C8EFA8ED00428F9D8A753599DFD4EA28987E013A2668907B9F547D08696DEF28143B0C0BD25B0B088C1674F2CF3B0487755175D764771BD122EB49EE31D9483F897F217EA809D07F2F170BA5819615FF4C572251CCD1C18B1F8427CAD881C1DD8223868151C15CAE8015233994A1119AE137B0605E504EB5C426A468E18F97541C3231162CCE5D3230317D8CF2917B038C2E9220A424B11C042C96199D258493BA37AF95F2683004651F254402273630F9E97226C62B96F747626B94B850A5EB731308F018D0E39DA8C3CD59B66CB7D96BC8A3AD58AC0EEF5A26FC385C98D828C255BC8A429496D529E7915F7F071501A76EE2454C77B84ABD6BB`),
		X:     internal.HI(`84B66BA83F41AA1AA7A6EED5712CD4C409069CB7783F0FC98387DDFFF6238478`),
		Y:     internal.HI(`61E9D567BA0370A20B80FCFC1FC42A0CDDC1905EFB43AE8C41910F09BE005420AF301ABBDD41A624DBDC3119E19C5FDF4B4E76912C6868E6D824ACC48B26D34E13301DE599A4DE14F43D3DE811ADE684ADEFB45E915814187678ADDA2B7B311395E588AA7BD978166B9F7640B8D0D43B38A507E3C240826B1056B0694E99DCCA0BDFCD7A654869DA580B42664882A58C0C9122C49B785DCA48F0F5CA49D936CCA9B7439E3A1AEE1D89CE83162815F0512A43AB2E1A10F952C505E55E986E1446A8FB8DE486366DB5CFEB1A5936DDDF56E60816CB22098CD2CE37A88B7D43B351F6B0C9D64510700423D56ABFF52970B73F0D9E9BD1A52A77986AA4C7EAB2AFDB`),
		R:     internal.HI(`AF7DF97C6E29FCB5B426E3AAEC3662073B384688E8A65E531E3A4783C6D115D0`),
		S:     internal.HI(`0CFE7D848F3B3F965F1CEC66687C8C7FE9D5E57360FA23A01C0260F1457E2A3E`),
	},
	{
		Sizes: L2048N256SHA256,
		P:     internal.HI(`D837AD8CB841C9B7632FA91C157BA723ADCDFDFC97ED24EDA2FEFC7CB50A84A885628D851BDC2225ADF6CB179CFCBC3E28CEEE4531C1FEFCD06E4908AB7429298492C74D6FE3B7BFD56206487A4C4F2497B1175C30F7C5F3EC333F4D1CF07E1AE09DE590321035025562708060DEBCAA543CFC8E3937643F156C448EBB7CC710B5C5DE216752DCFAEC0B5C342FA42A8B6BE0E10E3FB49D758A76F2CACA6A19F9B506BC6BD7C7C92C5AFB37A3F917C33AD5928F991DE4C8BF88783F1507A80495984019E9ABE3FE255C6BFF7EC895740BE187E3CB06BDF29BE3B41141D6A2BF7638F17D049E185C852FBBE358913CBFB2EDDF2E91F069B2D5C4908A390A3A5B5B`),
		Q:     internal.HI(`C2FBF4F74B4217A5A6BDEC4B4485D5D1F31976BCBE726BB2C276800D083BD341`),
		G:     internal.HI(`90AE2720DA48AB2B9C30EC6D05C3EBDA7B044AA3FBD5BB1C750C246CF54125F49EBF4A45B3A50493F76C9A94DC7F8BCCBC37B003AD914B5F5BB51428937FC116EDFAE6641641F54E96F8DE0027CA580821C5612F938D8A8A99D2E8D9C8F5B2263C4076D9E0E664D751F49E0F3FBEF889029C0448E1DD1F1A99FCC4051594F385EFAC480AD66DDB920C3069F81F24F1191F0E256DC9A3095ACA55081A1CA1F9B3D6654B83CFCDEFE83AE16D9E5562900F933F4258B1D48EFE5E7CA77A0550003B86D9023C641F9A80B97CD34C8F9CF3C2A350A27A74E7F438166C5DCB1C184E005C03E97EBD2494A1F849CD2A2B05AE9D3B793285E9B56AAB7FB305FBD1F6F09A`),
		M:     internal.HB(`04897B7F3E230FEBA39308B95454D4845B33CE71192265ECAA2AE2684962284339512C8640A66D8C31AA3E579E5A18AD2989CF9986FEE798B7E4D1F37E895F90A1C02ECBF2E0558CB2BA1E3A537712BC23BB2022DB0B232FB4BE9660EB82BB9958C2FA260DE7C23589D3B96C95837ACD373BAE86E423CBDD5ACB37DB456A27DB214DEFB570581B96C60B0B3DAEE06FC200804D8B95A8F7990B26A2DA6AE6F69E675E57E2265B2B8A1D82F54912757847B80EB93B0B85291DD0F2AD1B6220E4F23EFB605B5D9E29B1ED5E41715AB283CF4B6D96EB8A2B4AF05B5C17A45BCF13AC613024197057B2733AD1A2DF4991E893482F70378096AB5E059585C3AB2D0E6B3514A15BDE44C901B210B004CA906697E6ECBB018047057BD1D9850DD100C897C4C3C1A94FACDD55AC5BEF9BEFB924A60446D377474A7825676D8D6074939D5EC26353D4965DC32F23F9C9A4F39CB15229E7FD0CBB2F8F01199AFCE362BC50B68C210FF2A9AEB717BF3A8F6B395104F78780637CE6123A7BE2B5180292D60B6023329564AA7C5E60CC757B814A797CB7F4C91BCA0094D2C861180E7422C562E035D56DD1E12EC7204009B0BFDB3BE07DEF85204261E118E4E127F43759F64F88144D082ABFB2653BB95E3848C4495FFBA17D577991AA3596544ECA90A55E366CBCE8BEA5DC7F16577BE3058509D991ADD8828B49`),
		X:     internal.HI(`3EEE05EDA472459D100D7543C836EE1468849E9F8B09A6D4EFA27CFE8D7B26B4`),
		Y:     internal.HI(`4B89DC0D6097A78DF19EA784B0A6AD7C4B73BBC524CD3ADE436A0698D18D2627602777E92B48DBF59C8C3AB430D6D996BB7361882E03CFD36E3D1A194E6C5325F3EE48ABD57724D6A05A8B504D6B85E828D0BFD08818046B635686BF99F11E21E304238E0F499B9411275DFD46F7F8E5780595CBE700B66EBF9743E6387328163085A4A98191D12D34FE577DB14E317F0BB12C72146DA70D2F8F919BF4CF05F7E6C86A83E26C1BF1DB12FF23F2919B6F81E5787BD5C253A046423D17BB90109199E053CCD093B3EBCBFF49A29C4171D063DB9D7F7ECAD138204FEF1909F63A60C41E6D246D4D3530E492DBD763F0000C114541CEE7F3E7D27F03E619BAF7454C`),
		R:     internal.HI(`A5795E2C0694DF9F241E60984261355226C6C5D68BF4A766BA174B0EA9947C04`),
		S:     internal.HI(`04503B1BE3E7A3050A8B6BDA35D56A836C9DE71F59A4A4D4FA52B66ED50D329A`),
		Fail:  true,
	},
	{
		Sizes: L2048N256SHA256,
		P:     internal.HI(`D837AD8CB841C9B7632FA91C157BA723ADCDFDFC97ED24EDA2FEFC7CB50A84A885628D851BDC2225ADF6CB179CFCBC3E28CEEE4531C1FEFCD06E4908AB7429298492C74D6FE3B7BFD56206487A4C4F2497B1175C30F7C5F3EC333F4D1CF07E1AE09DE590321035025562708060DEBCAA543CFC8E3937643F156C448EBB7CC710B5C5DE216752DCFAEC0B5C342FA42A8B6BE0E10E3FB49D758A76F2CACA6A19F9B506BC6BD7C7C92C5AFB37A3F917C33AD5928F991DE4C8BF88783F1507A80495984019E9ABE3FE255C6BFF7EC895740BE187E3CB06BDF29BE3B41141D6A2BF7638F17D049E185C852FBBE358913CBFB2EDDF2E91F069B2D5C4908A390A3A5B5B`),
		Q:     internal.HI(`C2FBF4F74B4217A5A6BDEC4B4485D5D1F31976BCBE726BB2C276800D083BD341`),
		G:     internal.HI(`90AE2720DA48AB2B9C30EC6D05C3EBDA7B044AA3FBD5BB1C750C246CF54125F49EBF4A45B3A50493F76C9A94DC7F8BCCBC37B003AD914B5F5BB51428937FC116EDFAE6641641F54E96F8DE0027CA580821C5612F938D8A8A99D2E8D9C8F5B2263C4076D9E0E664D751F49E0F3FBEF889029C0448E1DD1F1A99FCC4051594F385EFAC480AD66DDB920C3069F81F24F1191F0E256DC9A3095ACA55081A1CA1F9B3D6654B83CFCDEFE83AE16D9E5562900F933F4258B1D48EFE5E7CA77A0550003B86D9023C641F9A80B97CD34C8F9CF3C2A350A27A74E7F438166C5DCB1C184E005C03E97EBD2494A1F849CD2A2B05AE9D3B793285E9B56AAB7FB305FBD1F6F09A`),
		M:     internal.HB(`DC538CC8A50390E4B03A865EB45D14410C541BA4CC67BBB924321B9B9B17B77B280D8655B545B7D6087E549BAD74876ED45DDEB2C1C34093BDC1E8402DF529B2B61849501BA42EEEB66A3578368AF3D197AE251EDAEF9825985060DEE23608E981C5250B1E89F6605D51735A61A9784EE2DC73A033D93C476EF580BF21E86528AAEBD0A3AF65FEA9C90DBE6EE8E61B136C03B3B790768C171E50ADF3F72539C17CEC6AFA64B2188BF00030A82B5ACA9714C738A85DC3D000B386B453190A684E6BAE79B97DF10213EE154AC6342A5B96E054BE83ADC538AD5B46CB80E6BFBBB211A3EE55E2AB5E9607BEF54CB5808C18005F671F3B8BDC1A71C68DE26274E4152FC31E092271B9ADA9F5818708910469C824BE1A6B28BC8574C502AB3C5F7EEAB28FC9D575DD873E673CA88C2E975020B968B7DB49BBBF770C8894D3C8C109EAAA0D1687BB902574009E8B37D0D5E61A7876AC928869B8BE0CDE1A1B05DFF21754D094B07B34D6C354AEB42E4198257DD4236034845F5C736A1ED10C980B88BB11EF3AACE479C5E6738512DD793252B7C5CB008141D24CF648255DF6CE9A08686C0A679CCE1A09E292C600791B009C7E67531C00690011EEEE5ACCF39DED92F8184AE26DB9D79B030C9B3D006F6119CF0327B200512E1B4CCDAB93E4A36A308CEF5FDBD0CD7961DB65B6F43468C5C6F0063B2395`),
		X:     internal.HI(`28180D2BCEE64942D1FA1160FDFBD876DEB8630AC020889DAE3F235959015CFF`),
		Y:     internal.HI(`0F778D958A6F937F862A4D5BE2A0B0577030521D23C9C8CD8AECA2F17CD185AAFA1F8215C0A6E3B850608F7359F7C55FA77CE13D0539EA1643C83222269BF001878CFC2B9FDA8B814D2C6ABE83C361BA52B107BA91A1E091A3A8BB5BED0F4517E9F15259C323C8084DBC5836B65A451B97C688789D1DAF769F0D1601D95FF1AD7170733BC8CC29797ADF38763C1E07076EE3ACD970119D9C79069302EC1B28A6586352B42EA71E88FDAB49296A38FDE3EE65BFA1807519B3492D5DCF4BBEFA6A5CA3EA55999CBF63A950BCD4B67082BE4E133209863F576354A42585A830EF7F9C7483BFDD7840FC745C74A1F387DD0C2835BFC0DEEB285DBC9F68CD2DC8D4D9`),
		R:     internal.HI(`CE85C1D1F3777D5845DD1153A479F9E232523EE6EEB90B5387BBDE1E81D7D7E9`),
		S:     internal.HI(`5B2A28435BEFCC970408E04F3B45A90DFB36C74ABAA1C4ADB07B69A5770508C6`),
	},
	{
		Sizes: L2048N256SHA256,
		P:     internal.HI(`D837AD8CB841C9B7632FA91C157BA723ADCDFDFC97ED24EDA2FEFC7CB50A84A885628D851BDC2225ADF6CB179CFCBC3E28CEEE4531C1FEFCD06E4908AB7429298492C74D6FE3B7BFD56206487A4C4F2497B1175C30F7C5F3EC333F4D1CF07E1AE09DE590321035025562708060DEBCAA543CFC8E3937643F156C448EBB7CC710B5C5DE216752DCFAEC0B5C342FA42A8B6BE0E10E3FB49D758A76F2CACA6A19F9B506BC6BD7C7C92C5AFB37A3F917C33AD5928F991DE4C8BF88783F1507A80495984019E9ABE3FE255C6BFF7EC895740BE187E3CB06BDF29BE3B41141D6A2BF7638F17D049E185C852FBBE358913CBFB2EDDF2E91F069B2D5C4908A390A3A5B5B`),
		Q:     internal.HI(`C2FBF4F74B4217A5A6BDEC4B4485D5D1F31976BCBE726BB2C276800D083BD341`),
		G:     internal.HI(`90AE2720DA48AB2B9C30EC6D05C3EBDA7B044AA3FBD5BB1C750C246CF54125F49EBF4A45B3A50493F76C9A94DC7F8BCCBC37B003AD914B5F5BB51428937FC116EDFAE6641641F54E96F8DE0027CA580821C5612F938D8A8A99D2E8D9C8F5B2263C4076D9E0E664D751F49E0F3FBEF889029C0448E1DD1F1A99FCC4051594F385EFAC480AD66DDB920C3069F81F24F1191F0E256DC9A3095ACA55081A1CA1F9B3D6654B83CFCDEFE83AE16D9E5562900F933F4258B1D48EFE5E7CA77A0550003B86D9023C641F9A80B97CD34C8F9CF3C2A350A27A74E7F438166C5DCB1C184E005C03E97EBD2494A1F849CD2A2B05AE9D3B793285E9B56AAB7FB305FBD1F6F09A`),
		M:     internal.HB(`467D8F8E702581D68FF381E841D8D327484C52B5A49F9E8A70060C3BA1F5C8435EBF04040A11AA5CB25071080253BDA06C94317AA555663324C49E2DCAA1E6CA5A7B90A0E9FD6385EE5877A31E5C1869B4399DA35375A2772A247D957FEA69654BC341780E2534FB8685648769BCCF05D36169FA938FDCF3E736BD74F66DBF0E88CEAE09436FAC50047A8FEC7CE3D1E2C657EFAF11DDA82F0A31FB362E6D5DA3B1FDF639186A62FE3901D970BAE61A52CF91125C4341A19887725BAEEED5C0EDAED9BD53E74AF7673F0DAB1B5087A9927CAB3B0165456A849E808A18C5396D9CAF12340ECEED10D678BAF35C2E2D88C69F695D097D19AF31D40BBB1609D4F3482B811086B9D95E7B8E482C0A10E7C9FA55B9F243579824C6F7E8ABB3DA8BE571E3268F41573B9872732355637A6D842100AEFCE9874382501B169F631FE8E27FDD2A772B21E97CBC5FDEF710B51FDC1A4E85079C6B438FC5A1BB630085208FC26BDE18995861D344D632211ED404F9A632A42365286B14032C274ACD840D997224BA474905C76CDBA1036D06B3CB0D73E895ECB5A933E6D2ABD032775B34B5B0E75E6360F9EA1D3CD25BFAA5F4CC5113F30F8465A5BEDFDE54537E0E11C0A183DD94516ACC3FC708C36C7144020706051FEC95B798D4E3BDA5791B10748422DC754B815DE0E351CA15920191102376AA81335153`),
		X:     internal.HI(`18FE60DC6C61BC23E515EE8F13377095BD2F50E124109B9F4DEB33D41F77B805`),
		Y:     internal.HI(`670CCBC1B3B9B72D3FE3CB245B4750BF1EA0091ACC75A0AF53D66AAD2C9A63EE22A27C90327637C3B91ACE398CC07B9409FEC1E62F4424E0F1C27A2046D0F1D36074D598FE8525C84C2575CA98C850218734DE7CC4EB6622F5C05D91278806E6F36EBAEBB58DCC9906B007D7B0F1C622B22677B94697B839C3F8B8B9A6C3A6870DADB9C62D6D1B42EFB5926F698EAD806DBBBB8A99D2EFA9DB64BC810F1FE135B38080D977E3B3720EE427AF7C369179E1024C7EB8BDB3847EF2F50A22C94B93F81EBB934AD9A84FBC8A707CD8D1D6FFDF8763D69E77867D921B2803A941499DE6C723EA3DEFEC8736C7B8B121A0E447A6BD0F88811BEB6C847886D1C6F57B49`),
		R:     internal.HI(`95CCC50ED90E24260F462062ECFCCF0E64CCBB1126179F88D0144E064EE99D53`),
		S:     internal.HI(`A5D543E45F91B7EA82FF66B6E02421380AF53A6DB8D4844AA703F40B45AF18A7`),
	},
	{
		Sizes: L2048N256SHA256,
		P:     internal.HI(`D837AD8CB841C9B7632FA91C157BA723ADCDFDFC97ED24EDA2FEFC7CB50A84A885628D851BDC2225ADF6CB179CFCBC3E28CEEE4531C1FEFCD06E4908AB7429298492C74D6FE3B7BFD56206487A4C4F2497B1175C30F7C5F3EC333F4D1CF07E1AE09DE590321035025562708060DEBCAA543CFC8E3937643F156C448EBB7CC710B5C5DE216752DCFAEC0B5C342FA42A8B6BE0E10E3FB49D758A76F2CACA6A19F9B506BC6BD7C7C92C5AFB37A3F917C33AD5928F991DE4C8BF88783F1507A80495984019E9ABE3FE255C6BFF7EC895740BE187E3CB06BDF29BE3B41141D6A2BF7638F17D049E185C852FBBE358913CBFB2EDDF2E91F069B2D5C4908A390A3A5B5B`),
		Q:     internal.HI(`C2FBF4F74B4217A5A6BDEC4B4485D5D1F31976BCBE726BB2C276800D083BD341`),
		G:     internal.HI(`90AE2720DA48AB2B9C30EC6D05C3EBDA7B044AA3FBD5BB1C750C246CF54125F49EBF4A45B3A50493F76C9A94DC7F8BCCBC37B003AD914B5F5BB51428937FC116EDFAE6641641F54E96F8DE0027CA580821C5612F938D8A8A99D2E8D9C8F5B2263C4076D9E0E664D751F49E0F3FBEF889029C0448E1DD1F1A99FCC4051594F385EFAC480AD66DDB920C3069F81F24F1191F0E256DC9A3095ACA55081A1CA1F9B3D6654B83CFCDEFE83AE16D9E5562900F933F4258B1D48EFE5E7CA77A0550003B86D9023C641F9A80B97CD34C8F9CF3C2A350A27A74E7F438166C5DCB1C184E005C03E97EBD2494A1F849CD2A2B05AE9D3B793285E9B56AAB7FB305FBD1F6F09A`),
		M:     internal.HB(`EE632F7E5E1BFC0592679DE9965C9DA9F4B24E65A41870F2517310734A3D13C2D79C8CA245654C2065B3C906D933D9C8E0CD4BDBFBDDE6D42B9BD63E15D2178BC6A6CF0C76793F50FEA139E0551CAEACCFDE5D2C7DEB237C57E3047A96205C3CF11F5DC80D45497BEDC07D049D99D95DEDA87CA87A947053D77F3C7B158DF6E175ECC76B8D81858CAAE94D88B0F2BFEA4A34CD28A8F2BCCC98EB4D1B44469C8D5A3CC110E1ABB77B943A8B0CF2386E6BE2D5990C27E99F606EEC2ABD3C40B15A8D842C5D5D0A4A44F41D3EB62F449C009524533C7E245890148CF04A7D373F6DE4820D7CBAAD52EDFA4098339EB7A6CC2F0396299B15CEE43021E335F2B0F6F11D3C92241D6A8B82BE9CF0BADCFA9103619B24CAD6F792F04D447178E9F93119DE00139011E158194070C80AED3D0CDBC560E8A1EECCDB4BDEDB2D951E25F121B6630F848778C5D06944C8683E796C93DE09F4B60A6189974111D498B111E04B1A442A4DDC5E85CC07E9C0A3A470AE72159A8398B947227DB95A4B112B644FE369C735BED38BF43AD475A9125EA977C8BD5CF860F2D9D7B072719F1D7D88373CE85C263696BC154F6E49A4920F7617EC0FE1DDAF153A7FE7815B035C00B33BB2C5B71B96B9789449600CF98AC4EF813D2E02E4ADE7559AE5E164D5F975E3A3A818D75B4D370FC46D17AF19E9F3F6552323E2E809`),
		X:     internal.HI(`0CA53B16E482784B7E4167E59F5D9422167A0D2A252F3A20BE0897640251A9DA`),
		Y:     internal.HI(`003000AE2C3CD43E3CEC0CB3C72BF21D4A577765D8F7E5DBFEBC58799E99937C01C10E9588BDBE94736E3208DB36F8670F8CCD674F31A10C9EB5131006B4246389E4C4DC7FA539423AC9A89C6D11535F79417B42B785402A3E9B08A9A91D795C8E6E21B487B96264D02191B881A4F62C79D1456AC9C4F2612933D763CB67B3D3AEFE065817E451FA358E142FE9FC6FC9E6E5B0495BCFCDAEC208E336562DFC92D70FB04EFF18FA0EA0CBB7A10800F3B8FE8EDE2D9ADA3981689A398D1C3ED1A0928C2B91C474940A7CBED8ECFC41804C310FD92A888F2F5D6E50D69F45481B55F8B8BBE6F1C2B953B611B9D31DACA25D636D927C82FB8EE7C0CAF479D1159016`),
		R:     internal.HI(`8205951BB5732D4FB354BB497FBCA1FBAFA395404E76EA023AE3F4D7FD8F6E7B`),
		S:     internal.HI(`7F8607DDCDCDA13C43FA80FA12C1EA42BCA24609A09B27428F4EA733A74685D1`),
		Fail:  true,
	},
	{
		Sizes: L2048N256SHA256,
		P:     internal.HI(`D837AD8CB841C9B7632FA91C157BA723ADCDFDFC97ED24EDA2FEFC7CB50A84A885628D851BDC2225ADF6CB179CFCBC3E28CEEE4531C1FEFCD06E4908AB7429298492C74D6FE3B7BFD56206487A4C4F2497B1175C30F7C5F3EC333F4D1CF07E1AE09DE590321035025562708060DEBCAA543CFC8E3937643F156C448EBB7CC710B5C5DE216752DCFAEC0B5C342FA42A8B6BE0E10E3FB49D758A76F2CACA6A19F9B506BC6BD7C7C92C5AFB37A3F917C33AD5928F991DE4C8BF88783F1507A80495984019E9ABE3FE255C6BFF7EC895740BE187E3CB06BDF29BE3B41141D6A2BF7638F17D049E185C852FBBE358913CBFB2EDDF2E91F069B2D5C4908A390A3A5B5B`),
		Q:     internal.HI(`C2FBF4F74B4217A5A6BDEC4B4485D5D1F31976BCBE726BB2C276800D083BD341`),
		G:     internal.HI(`90AE2720DA48AB2B9C30EC6D05C3EBDA7B044AA3FBD5BB1C750C246CF54125F49EBF4A45B3A50493F76C9A94DC7F8BCCBC37B003AD914B5F5BB51428937FC116EDFAE6641641F54E96F8DE0027CA580821C5612F938D8A8A99D2E8D9C8F5B2263C4076D9E0E664D751F49E0F3FBEF889029C0448E1DD1F1A99FCC4051594F385EFAC480AD66DDB920C3069F81F24F1191F0E256DC9A3095ACA55081A1CA1F9B3D6654B83CFCDEFE83AE16D9E5562900F933F4258B1D48EFE5E7CA77A0550003B86D9023C641F9A80B97CD34C8F9CF3C2A350A27A74E7F438166C5DCB1C184E005C03E97EBD2494A1F849CD2A2B05AE9D3B793285E9B56AAB7FB305FBD1F6F09A`),
		M:     internal.HB(`5626F8DF3AF2F276F139516EC4B496B40D71D6C892CC75AA5F09E5B2D4497B403AF3D1AA9D35CE4B3EE0D21E5C230ED2A06699B4F5E73A65113E82B77A6E2C9E4AB35EA583AFBDBD321EEDDF2AC08362A991E76EA00E3870F098B4FB67AFB303B421B6D030DA9193B37F420F4B78702F099129FBC8DDD3D61804EE632C9BA0FC0DBC9631CC77471F1154191AC27E7B8A0AC970658D3B9A2E0A3A4DDC410BACA052CD63D96A91003903BA60707D4F6F4E5D6877C0F3564692B2B7935D061DB68DE9662CDE04790440A792B18B4EAD42DE1A619E27EAA2B2A764C42DE2C23BC8EA9D5EA5607DCACA1E848749EBF2A61023C371F0BE46DFE899D96F2F70A7131364A5582C869D64EA438B0D121B0C8C1D90411D1EAFC711141E368E5215CB9EEE329DBAC77F187228A6115D99AB2AFCD71FE4B3812E13888D7006C1FCE7301BDB138AB8228487666DC9D67C1835BFD8D15367461B77B6D7D3553A6E3702BC13824DDA4895D26DF9CDB301336D5A274EC836E9B496CD28DE8B1830C4B88B4154B3AD612D1CB1332E81F4211520C3A7D1A159F5A2437AC7C2858FA8BAD9B176F7678B5DEE5E722B4DEDA62E7E6123691D69D97A7B1CD3D8F2B614D00E9FA8FB5CBFC472DDA66A90E99B698591073183355356D376C23289213D8E3949B5AE592B03BFAE13ECF984DC3E64EE3791AFF165BDFBC08F7EFC`),
		X:     internal.HI(`73ADF376A493DC795DE1D26232DE02294766A00674C58DA49F5E07A3BA10D33D`),
		Y:     internal.HI(`6BEFFB9425E926BC401178C801FF7AD68C672A829196FFC2D00EAEFA194F2567C3026F9F31709CBB95535690361E807D5C62E05721515FDAF9A9A0D2245A9104F8C57B0E436DEE7B5B1F09385D356C42E9A0A82735156347227E60E1921B9C8054A03048FF4DBA5B533B4D7DE5CB1DEB244B89F9ADA805F7522468D79EE2FA93EB21B58F98D5D3F7DEFCA5E571398D5B3E4E7E44DCC3FBC2FA40457A67312773FE93B806151844A93A9A2B3E017A7F6C68902D2560AF116D6DCFE5F563A2EF0B0ABAE3EFB19DF57EA61D6BCE40DA7E8FBD86648CD8B32570864A20BC9F2671DFECC04CAB92C405E93FEA44153344B6227514AD07C63B20E8B46A9D99B052C93E`),
		R:     internal.HI(`35F0115668115708963CE1E87DB8AE9D2FCBEADA4C7D9677EAE99B505ED8CBC8`),
		S:     internal.HI(`88FC577C385CFBD21CB5B5C21BCC906435E6811A0EC382AE51931086FD4D0F05`),
	},
	{
		Sizes: L2048N256SHA256,
		P:     internal.HI(`D837AD8CB841C9B7632FA91C157BA723ADCDFDFC97ED24EDA2FEFC7CB50A84A885628D851BDC2225ADF6CB179CFCBC3E28CEEE4531C1FEFCD06E4908AB7429298492C74D6FE3B7BFD56206487A4C4F2497B1175C30F7C5F3EC333F4D1CF07E1AE09DE590321035025562708060DEBCAA543CFC8E3937643F156C448EBB7CC710B5C5DE216752DCFAEC0B5C342FA42A8B6BE0E10E3FB49D758A76F2CACA6A19F9B506BC6BD7C7C92C5AFB37A3F917C33AD5928F991DE4C8BF88783F1507A80495984019E9ABE3FE255C6BFF7EC895740BE187E3CB06BDF29BE3B41141D6A2BF7638F17D049E185C852FBBE358913CBFB2EDDF2E91F069B2D5C4908A390A3A5B5B`),
		Q:     internal.HI(`C2FBF4F74B4217A5A6BDEC4B4485D5D1F31976BCBE726BB2C276800D083BD341`),
		G:     internal.HI(`90AE2720DA48AB2B9C30EC6D05C3EBDA7B044AA3FBD5BB1C750C246CF54125F49EBF4A45B3A50493F76C9A94DC7F8BCCBC37B003AD914B5F5BB51428937FC116EDFAE6641641F54E96F8DE0027CA580821C5612F938D8A8A99D2E8D9C8F5B2263C4076D9E0E664D751F49E0F3FBEF889029C0448E1DD1F1A99FCC4051594F385EFAC480AD66DDB920C3069F81F24F1191F0E256DC9A3095ACA55081A1CA1F9B3D6654B83CFCDEFE83AE16D9E5562900F933F4258B1D48EFE5E7CA77A0550003B86D9023C641F9A80B97CD34C8F9CF3C2A350A27A74E7F438166C5DCB1C184E005C03E97EBD2494A1F849CD2A2B05AE9D3B793285E9B56AAB7FB305FBD1F6F09A`),
		M:     internal.HB(`67DFB76E080052A36B8670CC871047B0489923286895422B7B6A8A449A0EC291942471C603450B393DD79D8D832795F34E1CD29B2B3DA7490F62F5AB82C6076392D45461F56CD48534E50B1B65A3763336778461D39DB7ED7AA224F62DE6F4956523098A3A9C7670D65FF20DAA46E6C1470F469A4051DF12078A340D1F26A70F99AAFD3539E401093258559FDCB9251953B6EA0D5CDCD2DE87E40B84E42285DE436D67FD5F3DCC8ADE4FFAB88990BEDDB1A10B2B19AB8D9C55E054930F623C39FED8472621877753F72774E5484482D7436F0B0A6F1252C25418841B3A52C37FF0BC629CFB8AEAED252F1A5CB9378BFB6F29166B604CAFECED8DD9A70B4A5737C35649F175F55508961B0FF984B33B62273C1EB4F57F76DE13A757682B877E0EAE4851621A622F7D00083A4456EA3C4FE37FDEF53FB5C3854139CB374E2F06DC6C9D9AD2804F394CA17C4C68E9F6802DA433D8E759E4FBF6787ACA972F50049F41C90ACD45267A9E4063BF3DFADA408EF1FE59E862E7C96E420CB1AF92E1D67DFAA64F860D3642C42B12D33E5280FE2EDAEE720085822150B3F8A45442C022C6EA78E2D987B7293539479093BFB984EDF97CFEDEF3613F2964B090FC13B1D7EFEFE9014967C90E93C727C908193FE4D86D86A1DBE418A6AE790B2BCBDF6529986D0DB2026B751AA7BC3F141440B57720E053C4F5`),
		X:     internal.HI(`1382E2B8DF0C9FDABEDCFA1DE198DCDD4CFC0555C0D2B5B905E6D9FC389CA35E`),
		Y:     internal.HI(`0028FE1834863FC135E3D624AA793380C76AAC792D56F787C40B1CF1E9C8DA3BA78A8D262248A4932BA544C71F4F781995C96F2345C7990F55BB5A79291C4B3538D0F062D598332FF76A6B8290FD342EECE561F7FDB08A09F7F860DFC7B37DD7B5EF331D344E4122AA32A5B6E5615657563430AFE25464301A0C7DE308736981130228E582C5F13163A0333EA0E9391F81B55711C134D469ED4DC043ED6F66092D94361BA6E71432A69C4D4815C13E63E921C05B7C9F7EED8C4442373777EB9838AFCB4FCFEDDE17A44B9B370CC7E2A80324219C2C8E199D30A48AB87EE829C84E5F1C3D36FEB0CD4C321B8CE3F4921AE0D9F00D2D33992BDB07E9EDCCBE51BF`),
		R:     internal.HI(`BA9ED1E46500A76FB81C51B47353B0186DC30C5ACB0996D5E1697A30A3A5CD8A`),
		S:     internal.HI(`4A32186E6505654EECB0E26AA1A9C412F9FF341A7926C004EBF443D9ECBE934B`),
	},
	{
		Sizes: L2048N256SHA256,
		P:     internal.HI(`D837AD8CB841C9B7632FA91C157BA723ADCDFDFC97ED24EDA2FEFC7CB50A84A885628D851BDC2225ADF6CB179CFCBC3E28CEEE4531C1FEFCD06E4908AB7429298492C74D6FE3B7BFD56206487A4C4F2497B1175C30F7C5F3EC333F4D1CF07E1AE09DE590321035025562708060DEBCAA543CFC8E3937643F156C448EBB7CC710B5C5DE216752DCFAEC0B5C342FA42A8B6BE0E10E3FB49D758A76F2CACA6A19F9B506BC6BD7C7C92C5AFB37A3F917C33AD5928F991DE4C8BF88783F1507A80495984019E9ABE3FE255C6BFF7EC895740BE187E3CB06BDF29BE3B41141D6A2BF7638F17D049E185C852FBBE358913CBFB2EDDF2E91F069B2D5C4908A390A3A5B5B`),
		Q:     internal.HI(`C2FBF4F74B4217A5A6BDEC4B4485D5D1F31976BCBE726BB2C276800D083BD341`),
		G:     internal.HI(`90AE2720DA48AB2B9C30EC6D05C3EBDA7B044AA3FBD5BB1C750C246CF54125F49EBF4A45B3A50493F76C9A94DC7F8BCCBC37B003AD914B5F5BB51428937FC116EDFAE6641641F54E96F8DE0027CA580821C5612F938D8A8A99D2E8D9C8F5B2263C4076D9E0E664D751F49E0F3FBEF889029C0448E1DD1F1A99FCC4051594F385EFAC480AD66DDB920C3069F81F24F1191F0E256DC9A3095ACA55081A1CA1F9B3D6654B83CFCDEFE83AE16D9E5562900F933F4258B1D48EFE5E7CA77A0550003B86D9023C641F9A80B97CD34C8F9CF3C2A350A27A74E7F438166C5DCB1C184E005C03E97EBD2494A1F849CD2A2B05AE9D3B793285E9B56AAB7FB305FBD1F6F09A`),
		M:     internal.HB(`BC9A4CF08A8A7096864F5FC4D758AABC608683D4D31AA5E02180929BD561E9C857FA1B25AB0051E5A90BC2CF45FB41ACD81B51310B9880095C065B21C987B2F2AB5589D646326D23179D80306E29F9EA4EFB6D1C2AAFA1227CB32C18CA66B3460EEB0FD91E1E14A60F884E101FC17BBE2C2B064E45604BA65550EC0B776009AEFB888FCA9CC67A0CF6D9455CCAA756B70239B0471AB2A657DF2BEB4E18DDB85E187D520ED136BE3B5D25E9CB8BC903AD883C67510FB4C53F3F1EE0FA9B50ACD330A608D17880E760F88826DC271BE2C1A0D2927C307EA2AEBE8AE9F29932B8CF3963CB0AF2C1E1F1A8A64DD7079C3B5A5122FDA1342E1F40CF558DE2510399604FA11B73481A84AC71AB1BC941503E27CCDADE6078EC06D60DF1BD3DAB4EF4D8B6D0E3932CB98D97844CB28A8F4503216B2FD02302E80A9A3A56CE3C37A353D4DBEB72B6F7D0A20036C59DB956918987ACE1DA1B8059C4FD410481E32B9B2B36547582F0AC9A517C06DAD0BCA04FBAE13A366840467DB6BA3404FBFD68D9D92CDB783420F35B11E99BD8A5C121A76500E4FB5053529D4BD24EE5CC1D75069F29568611E81E5E3F6EC393E0C135C443FAA388CFDC4A08D48FF1C1EB9D81D2ABE9D1B90AB727F622787467AC7AE0DDF23098BB892E7C148C82A937B89F65F7106F82B578C1AF7FE7BCCE399B72CC2FFC490BF98B60`),
		X:     internal.HI(`282B78E9964836E990091696D3C51699D4934D8CD841B675169A67D78980E7C7`),
		Y:     internal.HI(`9880B677174A225CF53C6113E77F17ADA5A604E54D2351B828950DBD243B5951099284BCE4FA790A8A237463B472F589D31513671517F7694F593640B5761C86E594623E6A631C565305615DA62577197336294A09A72BCF18F41C722462ECC2C1B3390D614EDDCD59A44B7E3EE219F47AFB42BA685356D8D21564996DBE765FD2DE5E46AED965E2BE7D04F36D69E310E396C47394F84F59708E3C81E6C272679A5D438E2F74D8E2823BDFE96EF1ACD8AC45F5C8B9A675BFC7E8B0D0424796D7BD0CD6AE99C3F1FC875C6992990322B2632B02E7DBAC6E16899EB4CE52A835B168B73BE436FDE311735BFDF0E4904FD0152EBD74BB498DA0851C32B0EB2CF486`),
		R:     internal.HI(`E4E61502BF8162EDBD6CE828666045BA6209658A743391370400B831AEA9B27D`),
		S:     internal.HI(`51A738D5B25072A5F9C94DE3F63EF5A441A071C6A66A3136B8E21D6477E4E47C`),
	},
}
