// Copyright 2019 The gVisor Authors.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

//go:build arm64
// +build arm64

package strace

import (
	"github.com/metacubex/gvisor/pkg/abi"
	"github.com/metacubex/gvisor/pkg/sentry/arch"
)

// linuxARM64 provides a mapping of the Linux arm64 syscalls and their argument
// types for display / formatting.
var linuxARM64 = SyscallMap{
	0:   makeSyscallInfo("io_setup", Hex, Hex),
	1:   makeSyscallInfo("io_destroy", Hex),
	2:   makeSyscallInfo("io_submit", Hex, Hex, Hex),
	3:   makeSyscallInfo("io_cancel", Hex, Hex, Hex),
	4:   makeSyscallInfo("io_getevents", Hex, Hex, Hex, Hex, Timespec),
	5:   makeSyscallInfo("setxattr", Path, Path, Hex, Hex, Hex),
	6:   makeSyscallInfo("lsetxattr", Path, Path, Hex, Hex, Hex),
	7:   makeSyscallInfo("fsetxattr", FD, Path, Hex, Hex, Hex),
	8:   makeSyscallInfo("getxattr", Path, Path, Hex, Hex),
	9:   makeSyscallInfo("lgetxattr", Path, Path, Hex, Hex),
	10:  makeSyscallInfo("fgetxattr", FD, Path, Hex, Hex),
	11:  makeSyscallInfo("listxattr", Path, Path, Hex),
	12:  makeSyscallInfo("llistxattr", Path, Path, Hex),
	13:  makeSyscallInfo("flistxattr", FD, Path, Hex),
	14:  makeSyscallInfo("removexattr", Path, Path),
	15:  makeSyscallInfo("lremovexattr", Path, Path),
	16:  makeSyscallInfo("fremovexattr", FD, Path),
	17:  makeSyscallInfo("getcwd", PostPath, Hex),
	18:  makeSyscallInfo("lookup_dcookie", Hex, Hex, Hex),
	19:  makeSyscallInfo("eventfd2", Hex, Hex),
	20:  makeSyscallInfo("epoll_create1", Hex),
	21:  makeSyscallInfo("epoll_ctl", FD, EpollCtlOp, FD, EpollEvent),
	22:  makeSyscallInfo("epoll_pwait", FD, EpollEvents, Hex, Hex, SigSet, Hex),
	23:  makeSyscallInfo("dup", FD),
	24:  makeSyscallInfo("dup3", FD, FD, Hex),
	25:  makeSyscallInfo("fcntl", FD, Hex, Hex),
	26:  makeSyscallInfo("inotify_init1", Hex),
	27:  makeSyscallInfo("inotify_add_watch", Hex, Path, Hex),
	28:  makeSyscallInfo("inotify_rm_watch", Hex, Hex),
	29:  makeSyscallInfo("ioctl", FD, Hex, Hex),
	30:  makeSyscallInfo("ioprio_set", Hex, Hex, Hex),
	31:  makeSyscallInfo("ioprio_get", Hex, Hex),
	32:  makeSyscallInfo("flock", FD, Hex),
	33:  makeSyscallInfo("mknodat", FD, Path, Mode, Hex),
	34:  makeSyscallInfo("mkdirat", FD, Path, Hex),
	35:  makeSyscallInfo("unlinkat", FD, Path, Hex),
	36:  makeSyscallInfo("symlinkat", Path, FD, Path),
	37:  makeSyscallInfo("linkat", FD, Path, Hex, Path, Hex),
	38:  makeSyscallInfo("renameat", FD, Path, Hex, Path),
	39:  makeSyscallInfo("umount2", Path, Hex),
	40:  makeSyscallInfo("mount", Path, Path, Path, Hex, Path),
	41:  makeSyscallInfo("pivot_root", Path, Path),
	42:  makeSyscallInfo("nfsservctl", Hex, Hex, Hex),
	43:  makeSyscallInfo("statfs", Path, Hex),
	44:  makeSyscallInfo("fstatfs", FD, Hex),
	45:  makeSyscallInfo("truncate", Path, Hex),
	46:  makeSyscallInfo("ftruncate", FD, Hex),
	47:  makeSyscallInfo("fallocate", FD, Hex, Hex, Hex),
	48:  makeSyscallInfo("faccessat", FD, Path, Oct, Hex),
	49:  makeSyscallInfo("chdir", Path),
	50:  makeSyscallInfo("fchdir", FD),
	51:  makeSyscallInfo("chroot", Path),
	52:  makeSyscallInfo("fchmod", FD, Mode),
	53:  makeSyscallInfo("fchmodat", FD, Path, Mode),
	54:  makeSyscallInfo("fchownat", FD, Path, Hex, Hex, Hex),
	55:  makeSyscallInfo("fchown", FD, Hex, Hex),
	56:  makeSyscallInfo("openat", FD, Path, OpenFlags, Mode),
	57:  makeSyscallInfo("close", FD),
	58:  makeSyscallInfo("vhangup"),
	59:  makeSyscallInfo("pipe2", PipeFDs, Hex),
	60:  makeSyscallInfo("quotactl", Hex, Hex, Hex, Hex),
	61:  makeSyscallInfo("getdents64", FD, Hex, Hex),
	62:  makeSyscallInfo("lseek", Hex, Hex, Hex),
	63:  makeSyscallInfo("read", FD, ReadBuffer, Hex),
	64:  makeSyscallInfo("write", FD, WriteBuffer, Hex),
	65:  makeSyscallInfo("readv", FD, ReadIOVec, Hex),
	66:  makeSyscallInfo("writev", FD, WriteIOVec, Hex),
	67:  makeSyscallInfo("pread64", FD, ReadBuffer, Hex, Hex),
	68:  makeSyscallInfo("pwrite64", FD, WriteBuffer, Hex, Hex),
	69:  makeSyscallInfo("preadv", FD, ReadIOVec, Hex, Hex),
	70:  makeSyscallInfo("pwritev", FD, WriteIOVec, Hex, Hex),
	71:  makeSyscallInfo("sendfile", FD, FD, Hex, Hex),
	72:  makeSyscallInfo("pselect6", Hex, Hex, Hex, Hex, Hex, Hex),
	73:  makeSyscallInfo("ppoll", PollFDs, Hex, Timespec, SigSet, Hex),
	74:  makeSyscallInfo("signalfd4", Hex, Hex, Hex, Hex),
	75:  makeSyscallInfo("vmsplice", FD, Hex, Hex, Hex),
	76:  makeSyscallInfo("splice", FD, Hex, FD, Hex, Hex, Hex),
	77:  makeSyscallInfo("tee", FD, FD, Hex, Hex),
	78:  makeSyscallInfo("readlinkat", FD, Path, ReadBuffer, Hex),
	79:  makeSyscallInfo("fstatat", FD, Path, Stat, Hex),
	80:  makeSyscallInfo("fstat", FD, Stat),
	81:  makeSyscallInfo("sync"),
	82:  makeSyscallInfo("fsync", FD),
	83:  makeSyscallInfo("fdatasync", FD),
	84:  makeSyscallInfo("sync_file_range", FD, Hex, Hex, Hex),
	85:  makeSyscallInfo("timerfd_create", Hex, Hex),
	86:  makeSyscallInfo("timerfd_settime", FD, Hex, ItimerSpec, PostItimerSpec),
	87:  makeSyscallInfo("timerfd_gettime", FD, PostItimerSpec),
	88:  makeSyscallInfo("utimensat", FD, Path, UTimeTimespec, Hex),
	89:  makeSyscallInfo("acct", Hex),
	90:  makeSyscallInfo("capget", CapHeader, PostCapData),
	91:  makeSyscallInfo("capset", CapHeader, CapData),
	92:  makeSyscallInfo("personality", Hex),
	93:  makeSyscallInfo("exit", Hex),
	94:  makeSyscallInfo("exit_group", Hex),
	95:  makeSyscallInfo("waitid", Hex, Hex, Hex, Hex, Rusage),
	96:  makeSyscallInfo("set_tid_address", Hex),
	97:  makeSyscallInfo("unshare", CloneFlags),
	98:  makeSyscallInfo("futex", Hex, FutexOp, Hex, Timespec, Hex, Hex),
	99:  makeSyscallInfo("set_robust_list", Hex, Hex),
	100: makeSyscallInfo("get_robust_list", Hex, Hex, Hex),
	101: makeSyscallInfo("nanosleep", Timespec, PostTimespec),
	102: makeSyscallInfo("getitimer", ItimerType, PostItimerVal),
	103: makeSyscallInfo("setitimer", ItimerType, ItimerVal, PostItimerVal),
	104: makeSyscallInfo("kexec_load", Hex, Hex, Hex, Hex),
	105: makeSyscallInfo("init_module", Hex, Hex, Hex),
	106: makeSyscallInfo("delete_module", Hex, Hex),
	107: makeSyscallInfo("timer_create", Hex, Hex, Hex),
	108: makeSyscallInfo("timer_gettime", Hex, PostItimerSpec),
	109: makeSyscallInfo("timer_getoverrun", Hex),
	110: makeSyscallInfo("timer_settime", Hex, Hex, ItimerSpec, PostItimerSpec),
	111: makeSyscallInfo("timer_delete", Hex),
	112: makeSyscallInfo("clock_settime", Hex, Timespec),
	113: makeSyscallInfo("clock_gettime", Hex, PostTimespec),
	114: makeSyscallInfo("clock_getres", Hex, PostTimespec),
	115: makeSyscallInfo("clock_nanosleep", Hex, Hex, Timespec, PostTimespec),
	116: makeSyscallInfo("syslog", Hex, Hex, Hex),
	117: makeSyscallInfo("ptrace", PtraceRequest, Hex, Hex, Hex),
	118: makeSyscallInfo("sched_setparam", Hex, Hex),
	119: makeSyscallInfo("sched_setscheduler", Hex, Hex, Hex),
	120: makeSyscallInfo("sched_getscheduler", Hex),
	121: makeSyscallInfo("sched_getparam", Hex, Hex),
	122: makeSyscallInfo("sched_setaffinity", Hex, Hex, Hex),
	123: makeSyscallInfo("sched_getaffinity", Hex, Hex, Hex),
	124: makeSyscallInfo("sched_yield"),
	125: makeSyscallInfo("sched_get_priority_max", Hex),
	126: makeSyscallInfo("sched_get_priority_min", Hex),
	127: makeSyscallInfo("sched_rr_get_interval", Hex, Hex),
	128: makeSyscallInfo("restart_syscall"),
	129: makeSyscallInfo("kill", Hex, Signal),
	130: makeSyscallInfo("tkill", Hex, Signal),
	131: makeSyscallInfo("tgkill", Hex, Hex, Signal),
	132: makeSyscallInfo("sigaltstack", Hex, Hex),
	133: makeSyscallInfo("rt_sigsuspend", Hex),
	134: makeSyscallInfo("rt_sigaction", Signal, SigAction, PostSigAction, Hex),
	135: makeSyscallInfo("rt_sigprocmask", SignalMaskAction, SigSet, PostSigSet, Hex),
	136: makeSyscallInfo("rt_sigpending", Hex),
	137: makeSyscallInfo("rt_sigtimedwait", SigSet, Hex, Timespec, Hex),
	138: makeSyscallInfo("rt_sigqueueinfo", Hex, Signal, Hex),
	139: makeSyscallInfo("rt_sigreturn"),
	140: makeSyscallInfo("setpriority", Hex, Hex, Hex),
	141: makeSyscallInfo("getpriority", Hex, Hex),
	142: makeSyscallInfo("reboot", Hex, Hex, Hex, Hex),
	143: makeSyscallInfo("setregid", Hex, Hex),
	144: makeSyscallInfo("setgid", Hex),
	145: makeSyscallInfo("setreuid", Hex, Hex),
	146: makeSyscallInfo("setuid", Hex),
	147: makeSyscallInfo("setresuid", Hex, Hex, Hex),
	148: makeSyscallInfo("getresuid", Hex, Hex, Hex),
	149: makeSyscallInfo("setresgid", Hex, Hex, Hex),
	150: makeSyscallInfo("getresgid", Hex, Hex, Hex),
	151: makeSyscallInfo("setfsuid", Hex),
	152: makeSyscallInfo("setfsgid", Hex),
	153: makeSyscallInfo("times", Hex),
	154: makeSyscallInfo("setpgid", Hex, Hex),
	155: makeSyscallInfo("getpgid", Hex),
	156: makeSyscallInfo("getsid", Hex),
	157: makeSyscallInfo("setsid"),
	158: makeSyscallInfo("getgroups", Hex, Hex),
	159: makeSyscallInfo("setgroups", Hex, Hex),
	160: makeSyscallInfo("uname", Uname),
	161: makeSyscallInfo("sethostname", Hex, Hex),
	162: makeSyscallInfo("setdomainname", Hex, Hex),
	163: makeSyscallInfo("getrlimit", Hex, Hex),
	164: makeSyscallInfo("setrlimit", Hex, Hex),
	165: makeSyscallInfo("getrusage", Hex, Rusage),
	166: makeSyscallInfo("umask", Hex),
	167: makeSyscallInfo("prctl", Hex, Hex, Hex, Hex, Hex),
	168: makeSyscallInfo("getcpu", Hex, Hex, Hex),
	169: makeSyscallInfo("gettimeofday", Timeval, Hex),
	170: makeSyscallInfo("settimeofday", Timeval, Hex),
	171: makeSyscallInfo("adjtimex", Hex),
	172: makeSyscallInfo("getpid"),
	173: makeSyscallInfo("getppid"),
	174: makeSyscallInfo("getuid"),
	175: makeSyscallInfo("geteuid"),
	176: makeSyscallInfo("getgid"),
	177: makeSyscallInfo("getegid"),
	178: makeSyscallInfo("gettid"),
	179: makeSyscallInfo("sysinfo", Hex),
	180: makeSyscallInfo("mq_open", Hex, Hex, Hex, Hex),
	181: makeSyscallInfo("mq_unlink", Hex),
	182: makeSyscallInfo("mq_timedsend", Hex, Hex, Hex, Hex, Hex),
	183: makeSyscallInfo("mq_timedreceive", Hex, Hex, Hex, Hex, Hex),
	184: makeSyscallInfo("mq_notify", Hex, Hex),
	185: makeSyscallInfo("mq_getsetattr", Hex, Hex, Hex),
	186: makeSyscallInfo("msgget", Hex, Hex),
	187: makeSyscallInfo("msgctl", Hex, Hex, Hex),
	188: makeSyscallInfo("msgrcv", Hex, Hex, Hex, Hex, Hex),
	189: makeSyscallInfo("msgsnd", Hex, Hex, Hex, Hex),
	190: makeSyscallInfo("semget", Hex, Hex, Hex),
	191: makeSyscallInfo("semctl", Hex, Hex, Hex, Hex),
	192: makeSyscallInfo("semtimedop", Hex, Hex, Hex, Hex),
	193: makeSyscallInfo("semop", Hex, Hex, Hex),
	194: makeSyscallInfo("shmget", Hex, Hex, Hex),
	195: makeSyscallInfo("shmctl", Hex, Hex, Hex),
	196: makeSyscallInfo("shmat", Hex, Hex, Hex),
	197: makeSyscallInfo("shmdt", Hex),
	198: makeSyscallInfo("socket", SockFamily, SockType, SockProtocol),
	199: makeSyscallInfo("socketpair", SockFamily, SockType, SockProtocol, Hex),
	200: makeSyscallInfo("bind", FD, SockAddr, Hex),
	201: makeSyscallInfo("listen", FD, Hex),
	202: makeSyscallInfo("accept", FD, PostSockAddr, SockLen),
	203: makeSyscallInfo("connect", FD, SockAddr, Hex),
	204: makeSyscallInfo("getsockname", FD, PostSockAddr, SockLen),
	205: makeSyscallInfo("getpeername", FD, PostSockAddr, SockLen),
	206: makeSyscallInfo("sendto", FD, Hex, Hex, Hex, SockAddr, Hex),
	207: makeSyscallInfo("recvfrom", FD, Hex, Hex, Hex, PostSockAddr, SockLen),
	208: makeSyscallInfo("setsockopt", FD, Hex, Hex, Hex, Hex),
	209: makeSyscallInfo("getsockopt", FD, Hex, Hex, Hex, Hex),
	210: makeSyscallInfo("shutdown", FD, Hex),
	211: makeSyscallInfo("sendmsg", FD, SendMsgHdr, Hex),
	212: makeSyscallInfo("recvmsg", FD, RecvMsgHdr, Hex),
	213: makeSyscallInfo("readahead", Hex, Hex, Hex),
	214: makeSyscallInfo("brk", Hex),
	215: makeSyscallInfo("munmap", Hex, Hex),
	216: makeSyscallInfo("mremap", Hex, Hex, Hex, Hex, Hex),
	217: makeSyscallInfo("add_key", Hex, Hex, Hex, Hex, Hex),
	218: makeSyscallInfo("request_key", Hex, Hex, Hex, Hex),
	219: makeSyscallInfo("keyctl", Hex, Hex, Hex, Hex, Hex),
	220: makeSyscallInfo("clone", CloneFlags, Hex, Hex, Hex, Hex),
	221: makeSyscallInfo("execve", Path, ExecveStringVector, ExecveStringVector),
	222: makeSyscallInfo("mmap", Hex, Hex, MmapProt, MmapFlags, FD, Hex),
	223: makeSyscallInfo("fadvise64", FD, Hex, Hex, Hex),
	224: makeSyscallInfo("swapon", Hex, Hex),
	225: makeSyscallInfo("swapoff", Hex),
	226: makeSyscallInfo("mprotect", Hex, Hex, Hex),
	227: makeSyscallInfo("msync", Hex, Hex, Hex),
	228: makeSyscallInfo("mlock", Hex, Hex),
	229: makeSyscallInfo("munlock", Hex, Hex),
	230: makeSyscallInfo("mlockall", Hex),
	231: makeSyscallInfo("munlockall"),
	232: makeSyscallInfo("mincore", Hex, Hex, Hex),
	233: makeSyscallInfo("madvise", Hex, Hex, Hex),
	234: makeSyscallInfo("remap_file_pages", Hex, Hex, Hex, Hex, Hex),
	235: makeSyscallInfo("mbind", Hex, Hex, Hex, Hex, Hex, Hex),
	236: makeSyscallInfo("get_mempolicy", Hex, Hex, Hex, Hex, Hex),
	237: makeSyscallInfo("set_mempolicy", Hex, Hex, Hex),
	238: makeSyscallInfo("migrate_pages", Hex, Hex, Hex, Hex),
	239: makeSyscallInfo("move_pages", Hex, Hex, Hex, Hex, Hex, Hex),
	240: makeSyscallInfo("rt_tgsigqueueinfo", Hex, Hex, Signal, Hex),
	241: makeSyscallInfo("perf_event_open", Hex, Hex, Hex, Hex, Hex),
	242: makeSyscallInfo("accept4", FD, PostSockAddr, SockLen, SockFlags),
	243: makeSyscallInfo("recvmmsg", FD, Hex, Hex, Hex, Hex),

	260: makeSyscallInfo("wait4", Hex, Hex, Hex, Rusage),
	261: makeSyscallInfo("prlimit64", Hex, Hex, Hex, Hex),
	262: makeSyscallInfo("fanotify_init", Hex, Hex),
	263: makeSyscallInfo("fanotify_mark", Hex, Hex, Hex, Hex, Hex),
	264: makeSyscallInfo("name_to_handle_at", FD, Hex, Hex, Hex, Hex),
	265: makeSyscallInfo("open_by_handle_at", FD, Hex, Hex),
	266: makeSyscallInfo("clock_adjtime", Hex, Hex),
	267: makeSyscallInfo("syncfs", FD),
	268: makeSyscallInfo("setns", FD, Hex),
	269: makeSyscallInfo("sendmmsg", FD, Hex, Hex, Hex),
	270: makeSyscallInfo("process_vm_readv", Hex, ReadIOVec, Hex, IOVec, Hex, Hex),
	271: makeSyscallInfo("process_vm_writev", Hex, IOVec, Hex, WriteIOVec, Hex, Hex),
	272: makeSyscallInfo("kcmp", Hex, Hex, Hex, Hex, Hex),
	273: makeSyscallInfo("finit_module", Hex, Hex, Hex),
	274: makeSyscallInfo("sched_setattr", Hex, Hex, Hex),
	275: makeSyscallInfo("sched_getattr", Hex, Hex, Hex),
	276: makeSyscallInfo("renameat2", FD, Path, Hex, Path, Hex),
	277: makeSyscallInfo("seccomp", Hex, Hex, Hex),
	278: makeSyscallInfo("getrandom", Hex, Hex, Hex),
	279: makeSyscallInfo("memfd_create", Path, Hex),
	280: makeSyscallInfo("bpf", Hex, Hex, Hex),
	281: makeSyscallInfo("execveat", FD, Path, Hex, Hex, Hex),
	282: makeSyscallInfo("userfaultfd", Hex),
	283: makeSyscallInfo("membarrier", Hex),
	284: makeSyscallInfo("mlock2", Hex, Hex, Hex),
	285: makeSyscallInfo("copy_file_range", FD, Hex, FD, Hex, Hex, Hex),
	286: makeSyscallInfo("preadv2", FD, ReadIOVec, Hex, Hex, Hex),
	287: makeSyscallInfo("pwritev2", FD, WriteIOVec, Hex, Hex, Hex),
	291: makeSyscallInfo("statx", FD, Path, Hex, Hex, Hex),
	292: makeSyscallInfo("io_pgetevents", Hex, Hex, Hex, Hex, Timespec, SigSet),
	293: makeSyscallInfo("rseq", Hex, Hex, Hex, Hex),
	424: makeSyscallInfo("pidfd_send_signal", FD, Signal, Hex, Hex),
	425: makeSyscallInfo("io_uring_setup", Hex, Hex),
	426: makeSyscallInfo("io_uring_enter", FD, Hex, Hex, Hex, SigSet, Hex),
	427: makeSyscallInfo("io_uring_register", FD, Hex, Hex, Hex),
	428: makeSyscallInfo("open_tree", FD, Path, Hex),
	429: makeSyscallInfo("move_mount", FD, Path, FD, Path, Hex),
	430: makeSyscallInfo("fsopen", Path, Hex), // Not quite a path, but close.
	431: makeSyscallInfo("fsconfig", FD, Hex, Hex, Hex, Hex),
	432: makeSyscallInfo("fsmount", FD, Hex, Hex),
	433: makeSyscallInfo("fspick", FD, Path, Hex),
	434: makeSyscallInfo("pidfd_open", Hex, Hex),
	435: makeSyscallInfo("clone3", Hex, Hex),
	436: makeSyscallInfo("close_range", FD, FD, CloseRangeFlags),
	441: makeSyscallInfo("epoll_pwait2", FD, EpollEvents, Hex, Timespec, SigSet),
}

func init() {
	syscallTables = append(syscallTables,
		syscallTable{
			os:       abi.Linux,
			arch:     arch.ARM64,
			syscalls: linuxARM64})
}
