/*!
 * \file gnss_block_factory.h
 * \brief Interface of a factory that returns smart pointers to GNSS blocks.
 * \author Carlos Aviles, 2010. carlos.avilesr(at)googlemail.com
 *         Luis Esteve, 2011. luis(at)epsilon-formacion.com
 *         Javier Arribas, 2011. jarribas(at)cttc.es
 *         Carles Fernandez-Prades, 2014-2020. cfernandez(at)cttc.es
 *
 * This class encapsulates the complexity behind the instantiation
 * of GNSS blocks.
 *
 *
 * -----------------------------------------------------------------------------
 *
 * Copyright (C) 2010-2020  (see AUTHORS file for a list of contributors)
 *
 * GNSS-SDR is a software defined Global Navigation
 *          Satellite Systems receiver
 *
 * This file is part of GNSS-SDR.
 *
 * SPDX-License-Identifier: GPL-3.0-or-later
 *
 * -----------------------------------------------------------------------------
 */

#ifndef GNSS_SDR_BLOCK_FACTORY_H
#define GNSS_SDR_BLOCK_FACTORY_H

#include "concurrent_queue.h"
#include <pmt/pmt.h>
#include <memory>  // for unique_ptr
#include <string>  // for string
#include <vector>  // for vector


class ConfigurationInterface;
class GNSSBlockInterface;
class AcquisitionInterface;
class TrackingInterface;
class TelemetryDecoderInterface;

/*!
 * \brief Class that produces all kinds of GNSS blocks
 */
class GNSSBlockFactory
{
public:
    GNSSBlockFactory() = default;
    ~GNSSBlockFactory() = default;

    std::unique_ptr<GNSSBlockInterface> GetSignalSource(ConfigurationInterface* configuration,
        Concurrent_Queue<pmt::pmt_t>* queue, int ID = -1);  // NOLINT(performance-unnecessary-value-param)

    std::unique_ptr<GNSSBlockInterface> GetSignalConditioner(ConfigurationInterface* configuration, int ID = -1);

    std::unique_ptr<std::vector<std::unique_ptr<GNSSBlockInterface>>> GetChannels(ConfigurationInterface* configuration,
        Concurrent_Queue<pmt::pmt_t>* queue);  // NOLINT(performance-unnecessary-value-param)

    std::unique_ptr<GNSSBlockInterface> GetObservables(ConfigurationInterface* configuration);

    std::unique_ptr<GNSSBlockInterface> GetPVT(ConfigurationInterface* configuration);

    /*!
     * \brief Returns the block with the required configuration and implementation
     */
    std::unique_ptr<GNSSBlockInterface> GetBlock(ConfigurationInterface* configuration,
        const std::string& role, const std::string& implementation,
        unsigned int in_streams, unsigned int out_streams,
        Concurrent_Queue<pmt::pmt_t>* queue = nullptr);  // NOLINT(performance-unnecessary-value-param)

private:
    std::unique_ptr<GNSSBlockInterface> GetChannel_1C(ConfigurationInterface* configuration,
        const std::string& acq, const std::string& trk, const std::string& tlm, int channel,
        Concurrent_Queue<pmt::pmt_t>* queue);

    std::unique_ptr<GNSSBlockInterface> GetChannel_2S(ConfigurationInterface* configuration,
        const std::string& acq, const std::string& trk, const std::string& tlm, int channel,
        Concurrent_Queue<pmt::pmt_t>* queue);

    std::unique_ptr<GNSSBlockInterface> GetChannel_1B(ConfigurationInterface* configuration,
        const std::string& acq, const std::string& trk, const std::string& tlm, int channel,
        Concurrent_Queue<pmt::pmt_t>* queue);

    std::unique_ptr<GNSSBlockInterface> GetChannel_5X(ConfigurationInterface* configuration,
        const std::string& acq, const std::string& trk, const std::string& tlm, int channel,
        Concurrent_Queue<pmt::pmt_t>* queue);

    std::unique_ptr<GNSSBlockInterface> GetChannel_L5(ConfigurationInterface* configuration,
        const std::string& acq, const std::string& trk, const std::string& tlm, int channel,
        Concurrent_Queue<pmt::pmt_t>* queue);

    std::unique_ptr<GNSSBlockInterface> GetChannel_1G(ConfigurationInterface* configuration,
        const std::string& acq, const std::string& trk, const std::string& tlm, int channel,
        Concurrent_Queue<pmt::pmt_t>* queue);

    std::unique_ptr<GNSSBlockInterface> GetChannel_2G(ConfigurationInterface* configuration,
        const std::string& acq, const std::string& trk, const std::string& tlm, int channel,
        Concurrent_Queue<pmt::pmt_t>* queue);

    std::unique_ptr<GNSSBlockInterface> GetChannel_B1(ConfigurationInterface* configuration,
        const std::string& acq, const std::string& trk, const std::string& tlm, int channel,
        Concurrent_Queue<pmt::pmt_t>* queue);

    std::unique_ptr<GNSSBlockInterface> GetChannel_B3(ConfigurationInterface* configuration,
        const std::string& acq, const std::string& trk, const std::string& tlm, int channel,
        Concurrent_Queue<pmt::pmt_t>* queue);

    std::unique_ptr<AcquisitionInterface> GetAcqBlock(
        ConfigurationInterface* configuration,
        const std::string& role,
        const std::string& implementation, unsigned int in_streams,
        unsigned int out_streams);

    std::unique_ptr<TrackingInterface> GetTrkBlock(
        ConfigurationInterface* configuration,
        const std::string& role,
        const std::string& implementation, unsigned int in_streams,
        unsigned int out_streams);

    std::unique_ptr<TelemetryDecoderInterface> GetTlmBlock(
        ConfigurationInterface* configuration,
        const std::string& role,
        const std::string& implementation, unsigned int in_streams,
        unsigned int out_streams);
};

#endif  // GNSS_SDR_BLOCK_FACTORY_H
