# Gmsh - Copyright (C) 1997-2020 C. Geuzaine, J.-F. Remacle
#
# See the LICENSE.txt file for license information. Please report all
# issues on https://gitlab.onelab.info/gmsh/gmsh/issues.

# This file defines the Gmsh Python API (v4.5).
#
# Do not edit it directly: it is automatically generated by `api/gen.py'.
#
# By design, the Gmsh Python API is purely functional, and only uses elementary
# types (as well as `numpy' arrays if `numpy' is avaiable). See
# `tutorial/python' and `demos/api' for examples.

from ctypes import *
from ctypes.util import find_library
import signal
import os
import platform
from math import pi

GMSH_API_VERSION = "4.5"
GMSH_API_VERSION_MAJOR = 4
GMSH_API_VERSION_MINOR = 5

__version__ = GMSH_API_VERSION

signal.signal(signal.SIGINT, signal.SIG_DFL)
libdir = os.path.dirname(os.path.realpath(__file__))
if platform.system() == "Windows":
    libpath = os.path.join(libdir, "gmsh-4.5.dll")
elif platform.system() == "Darwin":
    libpath = os.path.join(libdir, "libgmsh.dylib")
else:
    libpath = os.path.join(libdir, "libgmsh.so")

if not os.path.exists(libpath):
    libpath = find_library("gmsh")

lib = CDLL(libpath)

use_numpy = False
try:
    import numpy
    try:
        from weakref import finalize as weakreffinalize
    except:
        from backports.weakref import finalize as weakreffinalize
    use_numpy = True
except:
    pass

# Utility functions, not part of the Gmsh Python API

def _ostring(s):
    sp = s.value.decode("utf-8")
    lib.gmshFree(s)
    return sp

def _ovectorpair(ptr, size):
    v = list((ptr[i * 2], ptr[i * 2 + 1]) for i in range(size//2))
    lib.gmshFree(ptr)
    return v

def _ovectorint(ptr, size):
    if use_numpy:
        v = numpy.ctypeslib.as_array(ptr, (size, ))
        weakreffinalize(v, lib.gmshFree, ptr)
    else:
        v = list(ptr[i] for i in range(size))
        lib.gmshFree(ptr)
    return v

def _ovectorsize(ptr, size):
    if use_numpy:
        v = numpy.ctypeslib.as_array(ptr, (size, ))
        weakreffinalize(v, lib.gmshFree, ptr)
    else:
        v = list(ptr[i] for i in range(size))
        lib.gmshFree(ptr)
    return v

def _ovectordouble(ptr, size):
    if use_numpy:
        v = numpy.ctypeslib.as_array(ptr, (size, ))
        weakreffinalize(v, lib.gmshFree, ptr)
    else:
        v = list(ptr[i] for i in range(size))
        lib.gmshFree(ptr)
    return v

def _ovectorstring(ptr, size):
    v = list(_ostring(cast(ptr[i], c_char_p)) for i in range(size))
    lib.gmshFree(ptr)
    return v

def _ovectorvectorint(ptr, size, n):
    v = [_ovectorint(pointer(ptr[i].contents), size[i]) for i in range(n.value)]
    lib.gmshFree(size)
    lib.gmshFree(ptr)
    return v

def _ovectorvectorsize(ptr, size, n):
    v = [_ovectorsize(pointer(ptr[i].contents), size[i]) for i in range(n.value)]
    lib.gmshFree(size)
    lib.gmshFree(ptr)
    return v

def _ovectorvectordouble(ptr, size, n):
    v = [_ovectordouble(pointer(ptr[i].contents), size[i]) for i in range(n.value)]
    lib.gmshFree(size)
    lib.gmshFree(ptr)
    return v

def _ovectorvectorpair(ptr, size, n):
    v = [_ovectorpair(pointer(ptr[i].contents), size[i]) for i in range(n.value)]
    lib.gmshFree(size)
    lib.gmshFree(ptr)
    return v

def _ivectorint(o):
    if use_numpy:
        return numpy.ascontiguousarray(o, numpy.int32).ctypes, c_size_t(len(o))
    else:
        return (c_int * len(o))(*o), c_size_t(len(o))

def _ivectorsize(o):
    if use_numpy:
        return numpy.ascontiguousarray(o, numpy.uintp).ctypes, c_size_t(len(o))
    else:
        return (c_size_t * len(o))(*o), c_size_t(len(o))

def _ivectordouble(o):
    if use_numpy:
        array = numpy.ascontiguousarray(o, numpy.float64)
        ct = array.ctypes
        ct.array = array
        return  ct, c_size_t(len(o))
    else:
        return (c_double * len(o))(*o), c_size_t(len(o))

def _ivectorpair(o):
    if use_numpy:
        array = numpy.ascontiguousarray(o, numpy.int32)
        ct = array.ctypes
        ct.array = array
        return  ct, c_size_t(len(o) * 2)
    else:
        return ((c_int * 2) * len(o))(*o), c_size_t(len(o) * 2)

def _ivectorstring(o):
    return (c_char_p * len(o))(*(s.encode() for s in o)), c_size_t(len(o))

def _ivectorvectorint(os):
    n = len(os)
    parrays = [_ivectorint(o) for o in os]
    sizes = (c_size_t * n)(*(a[1] for a in parrays))
    arrays = (POINTER(c_int) * n)(*(cast(a[0], POINTER(c_int)) for a in parrays))
    arrays.ref = [a[0] for a in parrays]
    size = c_size_t(n)
    return arrays, sizes, size

def _ivectorvectorsize(os):
    n = len(os)
    parrays = [_ivectorsize(o) for o in os]
    sizes = (c_size_t * n)(*(a[1] for a in parrays))
    arrays = (POINTER(c_size_t) * n)(*(cast(a[0], POINTER(c_size_t)) for a in parrays))
    arrays.ref = [a[0] for a in parrays]
    size = c_size_t(n)
    return arrays, sizes, size

def _ivectorvectordouble(os):
    n = len(os)
    parrays = [_ivectordouble(o) for o in os]
    sizes = (c_size_t * n)(*(a[1] for a in parrays))
    arrays = (POINTER(c_double) * n)(*(cast(a[0], POINTER(c_double)) for a in parrays))
    arrays.ref = [a[0] for a in parrays]
    size = c_size_t(n)
    return arrays, sizes, size

def _iargcargv(o):
    return c_int(len(o)), (c_char_p * len(o))(*(s.encode() for s in o))

# Gmsh Python API begins here

def initialize(argv=[], readConfigFiles=True):
    """
    gmsh.initialize(argv=[], readConfigFiles=True)

    Initialize Gmsh. This must be called before any call to the other functions
    in the API. If `argc' and `argv' (or just `argv' in Python or Julia) are
    provided, they will be handled in the same way as the command line
    arguments in the Gmsh app. If `readConfigFiles' is set, read system Gmsh
    configuration files (gmshrc and gmsh-options).
    """
    api_argc_, api_argv_ = _iargcargv(argv)
    ierr = c_int()
    lib.gmshInitialize(
        api_argc_, api_argv_,
        c_int(bool(readConfigFiles)),
        byref(ierr))
    if ierr.value != 0:
        raise ValueError(
            "gmshInitialize returned non-zero error code: ",
            ierr.value)

def finalize():
    """
    gmsh.finalize()

    Finalize Gmsh. This must be called when you are done using the Gmsh API.
    """
    ierr = c_int()
    lib.gmshFinalize(
        byref(ierr))
    if ierr.value != 0:
        raise ValueError(
            "gmshFinalize returned non-zero error code: ",
            ierr.value)

def open(fileName):
    """
    gmsh.open(fileName)

    Open a file. Equivalent to the `File->Open' menu in the Gmsh app. Handling
    of the file depends on its extension and/or its contents: opening a file
    with model data will create a new model.
    """
    ierr = c_int()
    lib.gmshOpen(
        c_char_p(fileName.encode()),
        byref(ierr))
    if ierr.value != 0:
        raise ValueError(
            "gmshOpen returned non-zero error code: ",
            ierr.value)

def merge(fileName):
    """
    gmsh.merge(fileName)

    Merge a file. Equivalent to the `File->Merge' menu in the Gmsh app.
    Handling of the file depends on its extension and/or its contents. Merging
    a file with model data will add the data to the current model.
    """
    ierr = c_int()
    lib.gmshMerge(
        c_char_p(fileName.encode()),
        byref(ierr))
    if ierr.value != 0:
        raise ValueError(
            "gmshMerge returned non-zero error code: ",
            ierr.value)

def write(fileName):
    """
    gmsh.write(fileName)

    Write a file. The export format is determined by the file extension.
    """
    ierr = c_int()
    lib.gmshWrite(
        c_char_p(fileName.encode()),
        byref(ierr))
    if ierr.value != 0:
        raise ValueError(
            "gmshWrite returned non-zero error code: ",
            ierr.value)

def clear():
    """
    gmsh.clear()

    Clear all loaded models and post-processing data, and add a new empty
    model.
    """
    ierr = c_int()
    lib.gmshClear(
        byref(ierr))
    if ierr.value != 0:
        raise ValueError(
            "gmshClear returned non-zero error code: ",
            ierr.value)


class option:
    """
    Option handling functions
    """

    @staticmethod
    def setNumber(name, value):
        """
        gmsh.option.setNumber(name, value)

        Set a numerical option to `value'. `name' is of the form "category.option"
        or "category[num].option". Available categories and options are listed in
        the Gmsh reference manual.
        """
        ierr = c_int()
        lib.gmshOptionSetNumber(
            c_char_p(name.encode()),
            c_double(value),
            byref(ierr))
        if ierr.value != 0:
            raise ValueError(
                "gmshOptionSetNumber returned non-zero error code: ",
                ierr.value)

    @staticmethod
    def getNumber(name):
        """
        gmsh.option.getNumber(name)

        Get the `value' of a numerical option. `name' is of the form
        "category.option" or "category[num].option". Available categories and
        options are listed in the Gmsh reference manual.

        Return `value'.
        """
        api_value_ = c_double()
        ierr = c_int()
        lib.gmshOptionGetNumber(
            c_char_p(name.encode()),
            byref(api_value_),
            byref(ierr))
        if ierr.value != 0:
            raise ValueError(
                "gmshOptionGetNumber returned non-zero error code: ",
                ierr.value)
        return api_value_.value

    @staticmethod
    def setString(name, value):
        """
        gmsh.option.setString(name, value)

        Set a string option to `value'. `name' is of the form "category.option" or
        "category[num].option". Available categories and options are listed in the
        Gmsh reference manual.
        """
        ierr = c_int()
        lib.gmshOptionSetString(
            c_char_p(name.encode()),
            c_char_p(value.encode()),
            byref(ierr))
        if ierr.value != 0:
            raise ValueError(
                "gmshOptionSetString returned non-zero error code: ",
                ierr.value)

    @staticmethod
    def getString(name):
        """
        gmsh.option.getString(name)

        Get the `value' of a string option. `name' is of the form "category.option"
        or "category[num].option". Available categories and options are listed in
        the Gmsh reference manual.

        Return `value'.
        """
        api_value_ = c_char_p()
        ierr = c_int()
        lib.gmshOptionGetString(
            c_char_p(name.encode()),
            byref(api_value_),
            byref(ierr))
        if ierr.value != 0:
            raise ValueError(
                "gmshOptionGetString returned non-zero error code: ",
                ierr.value)
        return _ostring(api_value_)

    @staticmethod
    def setColor(name, r, g, b, a=0):
        """
        gmsh.option.setColor(name, r, g, b, a=0)

        Set a color option to the RGBA value (`r', `g', `b', `a'), where where `r',
        `g', `b' and `a' should be integers between 0 and 255. `name' is of the
        form "category.option" or "category[num].option". Available categories and
        options are listed in the Gmsh reference manual, with the "Color." middle
        string removed.
        """
        ierr = c_int()
        lib.gmshOptionSetColor(
            c_char_p(name.encode()),
            c_int(r),
            c_int(g),
            c_int(b),
            c_int(a),
            byref(ierr))
        if ierr.value != 0:
            raise ValueError(
                "gmshOptionSetColor returned non-zero error code: ",
                ierr.value)

    @staticmethod
    def getColor(name):
        """
        gmsh.option.getColor(name)

        Get the `r', `g', `b', `a' value of a color option. `name' is of the form
        "category.option" or "category[num].option". Available categories and
        options are listed in the Gmsh reference manual, with the "Color." middle
        string removed.

        Return `r', `g', `b', `a'.
        """
        api_r_ = c_int()
        api_g_ = c_int()
        api_b_ = c_int()
        api_a_ = c_int()
        ierr = c_int()
        lib.gmshOptionGetColor(
            c_char_p(name.encode()),
            byref(api_r_),
            byref(api_g_),
            byref(api_b_),
            byref(api_a_),
            byref(ierr))
        if ierr.value != 0:
            raise ValueError(
                "gmshOptionGetColor returned non-zero error code: ",
                ierr.value)
        return (
            api_r_.value,
            api_g_.value,
            api_b_.value,
            api_a_.value)


class model:
    """
    Model functions
    """

    @staticmethod
    def add(name):
        """
        gmsh.model.add(name)

        Add a new model, with name `name', and set it as the current model.
        """
        ierr = c_int()
        lib.gmshModelAdd(
            c_char_p(name.encode()),
            byref(ierr))
        if ierr.value != 0:
            raise ValueError(
                "gmshModelAdd returned non-zero error code: ",
                ierr.value)

    @staticmethod
    def remove():
        """
        gmsh.model.remove()

        Remove the current model.
        """
        ierr = c_int()
        lib.gmshModelRemove(
            byref(ierr))
        if ierr.value != 0:
            raise ValueError(
                "gmshModelRemove returned non-zero error code: ",
                ierr.value)

    @staticmethod
    def list():
        """
        gmsh.model.list()

        List the names of all models.

        Return `names'.
        """
        api_names_, api_names_n_ = POINTER(POINTER(c_char))(), c_size_t()
        ierr = c_int()
        lib.gmshModelList(
            byref(api_names_), byref(api_names_n_),
            byref(ierr))
        if ierr.value != 0:
            raise ValueError(
                "gmshModelList returned non-zero error code: ",
                ierr.value)
        return _ovectorstring(api_names_, api_names_n_.value)

    @staticmethod
    def getCurrent():
        """
        gmsh.model.getCurrent()

        Get the name of the current model.

        Return `name'.
        """
        api_name_ = c_char_p()
        ierr = c_int()
        lib.gmshModelGetCurrent(
            byref(api_name_),
            byref(ierr))
        if ierr.value != 0:
            raise ValueError(
                "gmshModelGetCurrent returned non-zero error code: ",
                ierr.value)
        return _ostring(api_name_)

    @staticmethod
    def setCurrent(name):
        """
        gmsh.model.setCurrent(name)

        Set the current model to the model with name `name'. If several models have
        the same name, select the one that was added first.
        """
        ierr = c_int()
        lib.gmshModelSetCurrent(
            c_char_p(name.encode()),
            byref(ierr))
        if ierr.value != 0:
            raise ValueError(
                "gmshModelSetCurrent returned non-zero error code: ",
                ierr.value)

    @staticmethod
    def getEntities(dim=-1):
        """
        gmsh.model.getEntities(dim=-1)

        Get all the entities in the current model. If `dim' is >= 0, return only
        the entities of the specified dimension (e.g. points if `dim' == 0). The
        entities are returned as a vector of (dim, tag) integer pairs.

        Return `dimTags'.
        """
        api_dimTags_, api_dimTags_n_ = POINTER(c_int)(), c_size_t()
        ierr = c_int()
        lib.gmshModelGetEntities(
            byref(api_dimTags_), byref(api_dimTags_n_),
            c_int(dim),
            byref(ierr))
        if ierr.value != 0:
            raise ValueError(
                "gmshModelGetEntities returned non-zero error code: ",
                ierr.value)
        return _ovectorpair(api_dimTags_, api_dimTags_n_.value)

    @staticmethod
    def setEntityName(dim, tag, name):
        """
        gmsh.model.setEntityName(dim, tag, name)

        Set the name of the entity of dimension `dim' and tag `tag'.
        """
        ierr = c_int()
        lib.gmshModelSetEntityName(
            c_int(dim),
            c_int(tag),
            c_char_p(name.encode()),
            byref(ierr))
        if ierr.value != 0:
            raise ValueError(
                "gmshModelSetEntityName returned non-zero error code: ",
                ierr.value)

    @staticmethod
    def getEntityName(dim, tag):
        """
        gmsh.model.getEntityName(dim, tag)

        Get the name of the entity of dimension `dim' and tag `tag'.

        Return `name'.
        """
        api_name_ = c_char_p()
        ierr = c_int()
        lib.gmshModelGetEntityName(
            c_int(dim),
            c_int(tag),
            byref(api_name_),
            byref(ierr))
        if ierr.value != 0:
            raise ValueError(
                "gmshModelGetEntityName returned non-zero error code: ",
                ierr.value)
        return _ostring(api_name_)

    @staticmethod
    def getPhysicalGroups(dim=-1):
        """
        gmsh.model.getPhysicalGroups(dim=-1)

        Get all the physical groups in the current model. If `dim' is >= 0, return
        only the entities of the specified dimension (e.g. physical points if `dim'
        == 0). The entities are returned as a vector of (dim, tag) integer pairs.

        Return `dimTags'.
        """
        api_dimTags_, api_dimTags_n_ = POINTER(c_int)(), c_size_t()
        ierr = c_int()
        lib.gmshModelGetPhysicalGroups(
            byref(api_dimTags_), byref(api_dimTags_n_),
            c_int(dim),
            byref(ierr))
        if ierr.value != 0:
            raise ValueError(
                "gmshModelGetPhysicalGroups returned non-zero error code: ",
                ierr.value)
        return _ovectorpair(api_dimTags_, api_dimTags_n_.value)

    @staticmethod
    def getEntitiesForPhysicalGroup(dim, tag):
        """
        gmsh.model.getEntitiesForPhysicalGroup(dim, tag)

        Get the tags of the model entities making up the physical group of
        dimension `dim' and tag `tag'.

        Return `tags'.
        """
        api_tags_, api_tags_n_ = POINTER(c_int)(), c_size_t()
        ierr = c_int()
        lib.gmshModelGetEntitiesForPhysicalGroup(
            c_int(dim),
            c_int(tag),
            byref(api_tags_), byref(api_tags_n_),
            byref(ierr))
        if ierr.value != 0:
            raise ValueError(
                "gmshModelGetEntitiesForPhysicalGroup returned non-zero error code: ",
                ierr.value)
        return _ovectorint(api_tags_, api_tags_n_.value)

    @staticmethod
    def getPhysicalGroupsForEntity(dim, tag):
        """
        gmsh.model.getPhysicalGroupsForEntity(dim, tag)

        Get the tags of the physical groups (if any) to which the model entity of
        dimension `dim' and tag `tag' belongs.

        Return `physicalTags'.
        """
        api_physicalTags_, api_physicalTags_n_ = POINTER(c_int)(), c_size_t()
        ierr = c_int()
        lib.gmshModelGetPhysicalGroupsForEntity(
            c_int(dim),
            c_int(tag),
            byref(api_physicalTags_), byref(api_physicalTags_n_),
            byref(ierr))
        if ierr.value != 0:
            raise ValueError(
                "gmshModelGetPhysicalGroupsForEntity returned non-zero error code: ",
                ierr.value)
        return _ovectorint(api_physicalTags_, api_physicalTags_n_.value)

    @staticmethod
    def addPhysicalGroup(dim, tags, tag=-1):
        """
        gmsh.model.addPhysicalGroup(dim, tags, tag=-1)

        Add a physical group of dimension `dim', grouping the model entities with
        tags `tags'. Return the tag of the physical group, equal to `tag' if `tag'
        is positive, or a new tag if `tag' < 0.

        Return an integer value.
        """
        api_tags_, api_tags_n_ = _ivectorint(tags)
        ierr = c_int()
        api__result__ = lib.gmshModelAddPhysicalGroup(
            c_int(dim),
            api_tags_, api_tags_n_,
            c_int(tag),
            byref(ierr))
        if ierr.value != 0:
            raise ValueError(
                "gmshModelAddPhysicalGroup returned non-zero error code: ",
                ierr.value)
        return api__result__

    @staticmethod
    def setPhysicalName(dim, tag, name):
        """
        gmsh.model.setPhysicalName(dim, tag, name)

        Set the name of the physical group of dimension `dim' and tag `tag'.
        """
        ierr = c_int()
        lib.gmshModelSetPhysicalName(
            c_int(dim),
            c_int(tag),
            c_char_p(name.encode()),
            byref(ierr))
        if ierr.value != 0:
            raise ValueError(
                "gmshModelSetPhysicalName returned non-zero error code: ",
                ierr.value)

    @staticmethod
    def getPhysicalName(dim, tag):
        """
        gmsh.model.getPhysicalName(dim, tag)

        Get the name of the physical group of dimension `dim' and tag `tag'.

        Return `name'.
        """
        api_name_ = c_char_p()
        ierr = c_int()
        lib.gmshModelGetPhysicalName(
            c_int(dim),
            c_int(tag),
            byref(api_name_),
            byref(ierr))
        if ierr.value != 0:
            raise ValueError(
                "gmshModelGetPhysicalName returned non-zero error code: ",
                ierr.value)
        return _ostring(api_name_)

    @staticmethod
    def getBoundary(dimTags, combined=True, oriented=True, recursive=False):
        """
        gmsh.model.getBoundary(dimTags, combined=True, oriented=True, recursive=False)

        Get the boundary of the model entities `dimTags'. Return in `outDimTags'
        the boundary of the individual entities (if `combined' is false) or the
        boundary of the combined geometrical shape formed by all input entities (if
        `combined' is true). Return tags multiplied by the sign of the boundary
        entity if `oriented' is true. Apply the boundary operator recursively down
        to dimension 0 (i.e. to points) if `recursive' is true.

        Return `outDimTags'.
        """
        api_dimTags_, api_dimTags_n_ = _ivectorpair(dimTags)
        api_outDimTags_, api_outDimTags_n_ = POINTER(c_int)(), c_size_t()
        ierr = c_int()
        lib.gmshModelGetBoundary(
            api_dimTags_, api_dimTags_n_,
            byref(api_outDimTags_), byref(api_outDimTags_n_),
            c_int(bool(combined)),
            c_int(bool(oriented)),
            c_int(bool(recursive)),
            byref(ierr))
        if ierr.value != 0:
            raise ValueError(
                "gmshModelGetBoundary returned non-zero error code: ",
                ierr.value)
        return _ovectorpair(api_outDimTags_, api_outDimTags_n_.value)

    @staticmethod
    def getEntitiesInBoundingBox(xmin, ymin, zmin, xmax, ymax, zmax, dim=-1):
        """
        gmsh.model.getEntitiesInBoundingBox(xmin, ymin, zmin, xmax, ymax, zmax, dim=-1)

        Get the model entities in the bounding box defined by the two points
        (`xmin', `ymin', `zmin') and (`xmax', `ymax', `zmax'). If `dim' is >= 0,
        return only the entities of the specified dimension (e.g. points if `dim'
        == 0).

        Return `tags'.
        """
        api_tags_, api_tags_n_ = POINTER(c_int)(), c_size_t()
        ierr = c_int()
        lib.gmshModelGetEntitiesInBoundingBox(
            c_double(xmin),
            c_double(ymin),
            c_double(zmin),
            c_double(xmax),
            c_double(ymax),
            c_double(zmax),
            byref(api_tags_), byref(api_tags_n_),
            c_int(dim),
            byref(ierr))
        if ierr.value != 0:
            raise ValueError(
                "gmshModelGetEntitiesInBoundingBox returned non-zero error code: ",
                ierr.value)
        return _ovectorpair(api_tags_, api_tags_n_.value)

    @staticmethod
    def getBoundingBox(dim, tag):
        """
        gmsh.model.getBoundingBox(dim, tag)

        Get the bounding box (`xmin', `ymin', `zmin'), (`xmax', `ymax', `zmax') of
        the model entity of dimension `dim' and tag `tag'. If `dim' and `tag' are
        negative, get the bounding box of the whole model.

        Return `xmin', `ymin', `zmin', `xmax', `ymax', `zmax'.
        """
        api_xmin_ = c_double()
        api_ymin_ = c_double()
        api_zmin_ = c_double()
        api_xmax_ = c_double()
        api_ymax_ = c_double()
        api_zmax_ = c_double()
        ierr = c_int()
        lib.gmshModelGetBoundingBox(
            c_int(dim),
            c_int(tag),
            byref(api_xmin_),
            byref(api_ymin_),
            byref(api_zmin_),
            byref(api_xmax_),
            byref(api_ymax_),
            byref(api_zmax_),
            byref(ierr))
        if ierr.value != 0:
            raise ValueError(
                "gmshModelGetBoundingBox returned non-zero error code: ",
                ierr.value)
        return (
            api_xmin_.value,
            api_ymin_.value,
            api_zmin_.value,
            api_xmax_.value,
            api_ymax_.value,
            api_zmax_.value)

    @staticmethod
    def getDimension():
        """
        gmsh.model.getDimension()

        Get the geometrical dimension of the current model.

        Return an integer value.
        """
        ierr = c_int()
        api__result__ = lib.gmshModelGetDimension(
            byref(ierr))
        if ierr.value != 0:
            raise ValueError(
                "gmshModelGetDimension returned non-zero error code: ",
                ierr.value)
        return api__result__

    @staticmethod
    def addDiscreteEntity(dim, tag=-1, boundary=[]):
        """
        gmsh.model.addDiscreteEntity(dim, tag=-1, boundary=[])

        Add a discrete model entity (defined by a mesh) of dimension `dim' in the
        current model. Return the tag of the new discrete entity, equal to `tag' if
        `tag' is positive, or a new tag if `tag' < 0. `boundary' specifies the tags
        of the entities on the boundary of the discrete entity, if any. Specifying
        `boundary' allows Gmsh to construct the topology of the overall model.

        Return an integer value.
        """
        api_boundary_, api_boundary_n_ = _ivectorint(boundary)
        ierr = c_int()
        api__result__ = lib.gmshModelAddDiscreteEntity(
            c_int(dim),
            c_int(tag),
            api_boundary_, api_boundary_n_,
            byref(ierr))
        if ierr.value != 0:
            raise ValueError(
                "gmshModelAddDiscreteEntity returned non-zero error code: ",
                ierr.value)
        return api__result__

    @staticmethod
    def removeEntities(dimTags, recursive=False):
        """
        gmsh.model.removeEntities(dimTags, recursive=False)

        Remove the entities `dimTags' of the current model. If `recursive' is true,
        remove all the entities on their boundaries, down to dimension 0.
        """
        api_dimTags_, api_dimTags_n_ = _ivectorpair(dimTags)
        ierr = c_int()
        lib.gmshModelRemoveEntities(
            api_dimTags_, api_dimTags_n_,
            c_int(bool(recursive)),
            byref(ierr))
        if ierr.value != 0:
            raise ValueError(
                "gmshModelRemoveEntities returned non-zero error code: ",
                ierr.value)

    @staticmethod
    def removeEntityName(name):
        """
        gmsh.model.removeEntityName(name)

        Remove the entity name `name' from the current model.
        """
        ierr = c_int()
        lib.gmshModelRemoveEntityName(
            c_char_p(name.encode()),
            byref(ierr))
        if ierr.value != 0:
            raise ValueError(
                "gmshModelRemoveEntityName returned non-zero error code: ",
                ierr.value)

    @staticmethod
    def removePhysicalGroups(dimTags=[]):
        """
        gmsh.model.removePhysicalGroups(dimTags=[])

        Remove the physical groups `dimTags' of the current model. If `dimTags' is
        empty, remove all groups.
        """
        api_dimTags_, api_dimTags_n_ = _ivectorpair(dimTags)
        ierr = c_int()
        lib.gmshModelRemovePhysicalGroups(
            api_dimTags_, api_dimTags_n_,
            byref(ierr))
        if ierr.value != 0:
            raise ValueError(
                "gmshModelRemovePhysicalGroups returned non-zero error code: ",
                ierr.value)

    @staticmethod
    def removePhysicalName(name):
        """
        gmsh.model.removePhysicalName(name)

        Remove the physical name `name' from the current model.
        """
        ierr = c_int()
        lib.gmshModelRemovePhysicalName(
            c_char_p(name.encode()),
            byref(ierr))
        if ierr.value != 0:
            raise ValueError(
                "gmshModelRemovePhysicalName returned non-zero error code: ",
                ierr.value)

    @staticmethod
    def getType(dim, tag):
        """
        gmsh.model.getType(dim, tag)

        Get the type of the entity of dimension `dim' and tag `tag'.

        Return `entityType'.
        """
        api_entityType_ = c_char_p()
        ierr = c_int()
        lib.gmshModelGetType(
            c_int(dim),
            c_int(tag),
            byref(api_entityType_),
            byref(ierr))
        if ierr.value != 0:
            raise ValueError(
                "gmshModelGetType returned non-zero error code: ",
                ierr.value)
        return _ostring(api_entityType_)

    @staticmethod
    def getParent(dim, tag):
        """
        gmsh.model.getParent(dim, tag)

        In a partitioned model, get the parent of the entity of dimension `dim' and
        tag `tag', i.e. from which the entity is a part of, if any. `parentDim' and
        `parentTag' are set to -1 if the entity has no parent.

        Return `parentDim', `parentTag'.
        """
        api_parentDim_ = c_int()
        api_parentTag_ = c_int()
        ierr = c_int()
        lib.gmshModelGetParent(
            c_int(dim),
            c_int(tag),
            byref(api_parentDim_),
            byref(api_parentTag_),
            byref(ierr))
        if ierr.value != 0:
            raise ValueError(
                "gmshModelGetParent returned non-zero error code: ",
                ierr.value)
        return (
            api_parentDim_.value,
            api_parentTag_.value)

    @staticmethod
    def getPartitions(dim, tag):
        """
        gmsh.model.getPartitions(dim, tag)

        In a partitioned model, return the tags of the partition(s) to which the
        entity belongs.

        Return `partitions'.
        """
        api_partitions_, api_partitions_n_ = POINTER(c_int)(), c_size_t()
        ierr = c_int()
        lib.gmshModelGetPartitions(
            c_int(dim),
            c_int(tag),
            byref(api_partitions_), byref(api_partitions_n_),
            byref(ierr))
        if ierr.value != 0:
            raise ValueError(
                "gmshModelGetPartitions returned non-zero error code: ",
                ierr.value)
        return _ovectorint(api_partitions_, api_partitions_n_.value)

    @staticmethod
    def getValue(dim, tag, parametricCoord):
        """
        gmsh.model.getValue(dim, tag, parametricCoord)

        Evaluate the parametrization of the entity of dimension `dim' and tag `tag'
        at the parametric coordinates `parametricCoord'. Only valid for `dim' equal
        to 0 (with empty `parametricCoord'), 1 (with `parametricCoord' containing
        parametric coordinates on the curve) or 2 (with `parametricCoord'
        containing pairs of u, v parametric coordinates on the surface,
        concatenated: [p1u, p1v, p2u, ...]). Return triplets of x, y, z coordinates
        in `points', concatenated: [p1x, p1y, p1z, p2x, ...].

        Return `points'.
        """
        api_parametricCoord_, api_parametricCoord_n_ = _ivectordouble(parametricCoord)
        api_points_, api_points_n_ = POINTER(c_double)(), c_size_t()
        ierr = c_int()
        lib.gmshModelGetValue(
            c_int(dim),
            c_int(tag),
            api_parametricCoord_, api_parametricCoord_n_,
            byref(api_points_), byref(api_points_n_),
            byref(ierr))
        if ierr.value != 0:
            raise ValueError(
                "gmshModelGetValue returned non-zero error code: ",
                ierr.value)
        return _ovectordouble(api_points_, api_points_n_.value)

    @staticmethod
    def getDerivative(dim, tag, parametricCoord):
        """
        gmsh.model.getDerivative(dim, tag, parametricCoord)

        Evaluate the derivative of the parametrization of the entity of dimension
        `dim' and tag `tag' at the parametric coordinates `parametricCoord'. Only
        valid for `dim' equal to 1 (with `parametricCoord' containing parametric
        coordinates on the curve) or 2 (with `parametricCoord' containing pairs of
        u, v parametric coordinates on the surface, concatenated: [p1u, p1v, p2u,
        ...]). For `dim' equal to 1 return the x, y, z components of the derivative
        with respect to u [d1ux, d1uy, d1uz, d2ux, ...]; for `dim' equal to 2
        return the x, y, z components of the derivate with respect to u and v:
        [d1ux, d1uy, d1uz, d1vx, d1vy, d1vz, d2ux, ...].

        Return `derivatives'.
        """
        api_parametricCoord_, api_parametricCoord_n_ = _ivectordouble(parametricCoord)
        api_derivatives_, api_derivatives_n_ = POINTER(c_double)(), c_size_t()
        ierr = c_int()
        lib.gmshModelGetDerivative(
            c_int(dim),
            c_int(tag),
            api_parametricCoord_, api_parametricCoord_n_,
            byref(api_derivatives_), byref(api_derivatives_n_),
            byref(ierr))
        if ierr.value != 0:
            raise ValueError(
                "gmshModelGetDerivative returned non-zero error code: ",
                ierr.value)
        return _ovectordouble(api_derivatives_, api_derivatives_n_.value)

    @staticmethod
    def getCurvature(dim, tag, parametricCoord):
        """
        gmsh.model.getCurvature(dim, tag, parametricCoord)

        Evaluate the (maximum) curvature of the entity of dimension `dim' and tag
        `tag' at the parametric coordinates `parametricCoord'. Only valid for `dim'
        equal to 1 (with `parametricCoord' containing parametric coordinates on the
        curve) or 2 (with `parametricCoord' containing pairs of u, v parametric
        coordinates on the surface, concatenated: [p1u, p1v, p2u, ...]).

        Return `curvatures'.
        """
        api_parametricCoord_, api_parametricCoord_n_ = _ivectordouble(parametricCoord)
        api_curvatures_, api_curvatures_n_ = POINTER(c_double)(), c_size_t()
        ierr = c_int()
        lib.gmshModelGetCurvature(
            c_int(dim),
            c_int(tag),
            api_parametricCoord_, api_parametricCoord_n_,
            byref(api_curvatures_), byref(api_curvatures_n_),
            byref(ierr))
        if ierr.value != 0:
            raise ValueError(
                "gmshModelGetCurvature returned non-zero error code: ",
                ierr.value)
        return _ovectordouble(api_curvatures_, api_curvatures_n_.value)

    @staticmethod
    def getPrincipalCurvatures(tag, parametricCoord):
        """
        gmsh.model.getPrincipalCurvatures(tag, parametricCoord)

        Evaluate the principal curvatures of the surface with tag `tag' at the
        parametric coordinates `parametricCoord', as well as their respective
        directions. `parametricCoord' are given by pair of u and v coordinates,
        concatenated: [p1u, p1v, p2u, ...].

        Return `curvatureMax', `curvatureMin', `directionMax', `directionMin'.
        """
        api_parametricCoord_, api_parametricCoord_n_ = _ivectordouble(parametricCoord)
        api_curvatureMax_, api_curvatureMax_n_ = POINTER(c_double)(), c_size_t()
        api_curvatureMin_, api_curvatureMin_n_ = POINTER(c_double)(), c_size_t()
        api_directionMax_, api_directionMax_n_ = POINTER(c_double)(), c_size_t()
        api_directionMin_, api_directionMin_n_ = POINTER(c_double)(), c_size_t()
        ierr = c_int()
        lib.gmshModelGetPrincipalCurvatures(
            c_int(tag),
            api_parametricCoord_, api_parametricCoord_n_,
            byref(api_curvatureMax_), byref(api_curvatureMax_n_),
            byref(api_curvatureMin_), byref(api_curvatureMin_n_),
            byref(api_directionMax_), byref(api_directionMax_n_),
            byref(api_directionMin_), byref(api_directionMin_n_),
            byref(ierr))
        if ierr.value != 0:
            raise ValueError(
                "gmshModelGetPrincipalCurvatures returned non-zero error code: ",
                ierr.value)
        return (
            _ovectordouble(api_curvatureMax_, api_curvatureMax_n_.value),
            _ovectordouble(api_curvatureMin_, api_curvatureMin_n_.value),
            _ovectordouble(api_directionMax_, api_directionMax_n_.value),
            _ovectordouble(api_directionMin_, api_directionMin_n_.value))

    @staticmethod
    def getNormal(tag, parametricCoord):
        """
        gmsh.model.getNormal(tag, parametricCoord)

        Get the normal to the surface with tag `tag' at the parametric coordinates
        `parametricCoord'. `parametricCoord' are given by pairs of u and v
        coordinates, concatenated: [p1u, p1v, p2u, ...]. `normals' are returned as
        triplets of x, y, z components, concatenated: [n1x, n1y, n1z, n2x, ...].

        Return `normals'.
        """
        api_parametricCoord_, api_parametricCoord_n_ = _ivectordouble(parametricCoord)
        api_normals_, api_normals_n_ = POINTER(c_double)(), c_size_t()
        ierr = c_int()
        lib.gmshModelGetNormal(
            c_int(tag),
            api_parametricCoord_, api_parametricCoord_n_,
            byref(api_normals_), byref(api_normals_n_),
            byref(ierr))
        if ierr.value != 0:
            raise ValueError(
                "gmshModelGetNormal returned non-zero error code: ",
                ierr.value)
        return _ovectordouble(api_normals_, api_normals_n_.value)

    @staticmethod
    def getParametrization(dim, tag, points):
        """
        gmsh.model.getParametrization(dim, tag, points)

        Get the parametric coordinates `parametricCoord' for the points `points' on
        the entity of dimension `dim' and tag `tag'. `points' are given as triplets
        of x, y, z coordinates, concatenated: [p1x, p1y, p1z, p2x, ...].
        `parametricCoord' returns the parametric coordinates t on the curve (if
        `dim' = 1) or pairs of u and v coordinates concatenated on the surface (if
        `dim' = 2), i.e. [p1t, p2t, ...] or [p1u, p1v, p2u, ...].

        Return `parametricCoord'.
        """
        api_points_, api_points_n_ = _ivectordouble(points)
        api_parametricCoord_, api_parametricCoord_n_ = POINTER(c_double)(), c_size_t()
        ierr = c_int()
        lib.gmshModelGetParametrization(
            c_int(dim),
            c_int(tag),
            api_points_, api_points_n_,
            byref(api_parametricCoord_), byref(api_parametricCoord_n_),
            byref(ierr))
        if ierr.value != 0:
            raise ValueError(
                "gmshModelGetParametrization returned non-zero error code: ",
                ierr.value)
        return _ovectordouble(api_parametricCoord_, api_parametricCoord_n_.value)

    @staticmethod
    def setVisibility(dimTags, value, recursive=False):
        """
        gmsh.model.setVisibility(dimTags, value, recursive=False)

        Set the visibility of the model entities `dimTags' to `value'. Apply the
        visibility setting recursively if `recursive' is true.
        """
        api_dimTags_, api_dimTags_n_ = _ivectorpair(dimTags)
        ierr = c_int()
        lib.gmshModelSetVisibility(
            api_dimTags_, api_dimTags_n_,
            c_int(value),
            c_int(bool(recursive)),
            byref(ierr))
        if ierr.value != 0:
            raise ValueError(
                "gmshModelSetVisibility returned non-zero error code: ",
                ierr.value)

    @staticmethod
    def getVisibility(dim, tag):
        """
        gmsh.model.getVisibility(dim, tag)

        Get the visibility of the model entity of dimension `dim' and tag `tag'.

        Return `value'.
        """
        api_value_ = c_int()
        ierr = c_int()
        lib.gmshModelGetVisibility(
            c_int(dim),
            c_int(tag),
            byref(api_value_),
            byref(ierr))
        if ierr.value != 0:
            raise ValueError(
                "gmshModelGetVisibility returned non-zero error code: ",
                ierr.value)
        return api_value_.value

    @staticmethod
    def setColor(dimTags, r, g, b, a=0, recursive=False):
        """
        gmsh.model.setColor(dimTags, r, g, b, a=0, recursive=False)

        Set the color of the model entities `dimTags' to the RGBA value (`r', `g',
        `b', `a'), where `r', `g', `b' and `a' should be integers between 0 and
        255. Apply the color setting recursively if `recursive' is true.
        """
        api_dimTags_, api_dimTags_n_ = _ivectorpair(dimTags)
        ierr = c_int()
        lib.gmshModelSetColor(
            api_dimTags_, api_dimTags_n_,
            c_int(r),
            c_int(g),
            c_int(b),
            c_int(a),
            c_int(bool(recursive)),
            byref(ierr))
        if ierr.value != 0:
            raise ValueError(
                "gmshModelSetColor returned non-zero error code: ",
                ierr.value)

    @staticmethod
    def getColor(dim, tag):
        """
        gmsh.model.getColor(dim, tag)

        Get the color of the model entity of dimension `dim' and tag `tag'.

        Return `r', `g', `b', `a'.
        """
        api_r_ = c_int()
        api_g_ = c_int()
        api_b_ = c_int()
        api_a_ = c_int()
        ierr = c_int()
        lib.gmshModelGetColor(
            c_int(dim),
            c_int(tag),
            byref(api_r_),
            byref(api_g_),
            byref(api_b_),
            byref(api_a_),
            byref(ierr))
        if ierr.value != 0:
            raise ValueError(
                "gmshModelGetColor returned non-zero error code: ",
                ierr.value)
        return (
            api_r_.value,
            api_g_.value,
            api_b_.value,
            api_a_.value)

    @staticmethod
    def setCoordinates(tag, x, y, z):
        """
        gmsh.model.setCoordinates(tag, x, y, z)

        Set the `x', `y', `z' coordinates of a geometrical point.
        """
        ierr = c_int()
        lib.gmshModelSetCoordinates(
            c_int(tag),
            c_double(x),
            c_double(y),
            c_double(z),
            byref(ierr))
        if ierr.value != 0:
            raise ValueError(
                "gmshModelSetCoordinates returned non-zero error code: ",
                ierr.value)


    class mesh:
        """
        Mesh functions
        """

        @staticmethod
        def generate(dim=3):
            """
            gmsh.model.mesh.generate(dim=3)

            Generate a mesh of the current model, up to dimension `dim' (0, 1, 2 or 3).
            """
            ierr = c_int()
            lib.gmshModelMeshGenerate(
                c_int(dim),
                byref(ierr))
            if ierr.value != 0:
                raise ValueError(
                    "gmshModelMeshGenerate returned non-zero error code: ",
                    ierr.value)

        @staticmethod
        def partition(numPart):
            """
            gmsh.model.mesh.partition(numPart)

            Partition the mesh of the current model into `numPart' partitions.
            """
            ierr = c_int()
            lib.gmshModelMeshPartition(
                c_int(numPart),
                byref(ierr))
            if ierr.value != 0:
                raise ValueError(
                    "gmshModelMeshPartition returned non-zero error code: ",
                    ierr.value)

        @staticmethod
        def unpartition():
            """
            gmsh.model.mesh.unpartition()

            Unpartition the mesh of the current model.
            """
            ierr = c_int()
            lib.gmshModelMeshUnpartition(
                byref(ierr))
            if ierr.value != 0:
                raise ValueError(
                    "gmshModelMeshUnpartition returned non-zero error code: ",
                    ierr.value)

        @staticmethod
        def optimize(method, force=False, niter=1, dimTags=[]):
            """
            gmsh.model.mesh.optimize(method, force=False, niter=1, dimTags=[])

            Optimize the mesh of the current model using `method' (empty for default
            tetrahedral mesh optimizer, "Netgen" for Netgen optimizer, "HighOrder" for
            direct high-order mesh optimizer, "HighOrderElastic" for high-order elastic
            smoother, "HighOrderFastCurving" for fast curving algorithm, "Laplace2D"
            for Laplace smoothing, "Relocate2D" and "Relocate3D" for node relocation).
            If `force' is set apply the optimization also to discrete entities. If
            `dimTags' is given, only apply the optimizer to the given entities.
            """
            api_dimTags_, api_dimTags_n_ = _ivectorpair(dimTags)
            ierr = c_int()
            lib.gmshModelMeshOptimize(
                c_char_p(method.encode()),
                c_int(bool(force)),
                c_int(niter),
                api_dimTags_, api_dimTags_n_,
                byref(ierr))
            if ierr.value != 0:
                raise ValueError(
                    "gmshModelMeshOptimize returned non-zero error code: ",
                    ierr.value)

        @staticmethod
        def recombine():
            """
            gmsh.model.mesh.recombine()

            Recombine the mesh of the current model.
            """
            ierr = c_int()
            lib.gmshModelMeshRecombine(
                byref(ierr))
            if ierr.value != 0:
                raise ValueError(
                    "gmshModelMeshRecombine returned non-zero error code: ",
                    ierr.value)

        @staticmethod
        def refine():
            """
            gmsh.model.mesh.refine()

            Refine the mesh of the current model by uniformly splitting the elements.
            """
            ierr = c_int()
            lib.gmshModelMeshRefine(
                byref(ierr))
            if ierr.value != 0:
                raise ValueError(
                    "gmshModelMeshRefine returned non-zero error code: ",
                    ierr.value)

        @staticmethod
        def setOrder(order):
            """
            gmsh.model.mesh.setOrder(order)

            Set the order of the elements in the mesh of the current model to `order'.
            """
            ierr = c_int()
            lib.gmshModelMeshSetOrder(
                c_int(order),
                byref(ierr))
            if ierr.value != 0:
                raise ValueError(
                    "gmshModelMeshSetOrder returned non-zero error code: ",
                    ierr.value)

        @staticmethod
        def getLastEntityError():
            """
            gmsh.model.mesh.getLastEntityError()

            Get the last entities (if any) where a meshing error occurred. Currently
            only populated by the new 3D meshing algorithms.

            Return `dimTags'.
            """
            api_dimTags_, api_dimTags_n_ = POINTER(c_int)(), c_size_t()
            ierr = c_int()
            lib.gmshModelMeshGetLastEntityError(
                byref(api_dimTags_), byref(api_dimTags_n_),
                byref(ierr))
            if ierr.value != 0:
                raise ValueError(
                    "gmshModelMeshGetLastEntityError returned non-zero error code: ",
                    ierr.value)
            return _ovectorpair(api_dimTags_, api_dimTags_n_.value)

        @staticmethod
        def getLastNodeError():
            """
            gmsh.model.mesh.getLastNodeError()

            Get the last nodes (if any) where a meshing error occurred. Currently only
            populated by the new 3D meshing algorithms.

            Return `nodeTags'.
            """
            api_nodeTags_, api_nodeTags_n_ = POINTER(c_size_t)(), c_size_t()
            ierr = c_int()
            lib.gmshModelMeshGetLastNodeError(
                byref(api_nodeTags_), byref(api_nodeTags_n_),
                byref(ierr))
            if ierr.value != 0:
                raise ValueError(
                    "gmshModelMeshGetLastNodeError returned non-zero error code: ",
                    ierr.value)
            return _ovectorsize(api_nodeTags_, api_nodeTags_n_.value)

        @staticmethod
        def clear():
            """
            gmsh.model.mesh.clear()

            Clear the mesh, i.e. delete all the nodes and elements.
            """
            ierr = c_int()
            lib.gmshModelMeshClear(
                byref(ierr))
            if ierr.value != 0:
                raise ValueError(
                    "gmshModelMeshClear returned non-zero error code: ",
                    ierr.value)

        @staticmethod
        def getNodes(dim=-1, tag=-1, includeBoundary=False, returnParametricCoord=True):
            """
            gmsh.model.mesh.getNodes(dim=-1, tag=-1, includeBoundary=False, returnParametricCoord=True)

            Get the nodes classified on the entity of dimension `dim' and tag `tag'. If
            `tag' < 0, get the nodes for all entities of dimension `dim'. If `dim' and
            `tag' are negative, get all the nodes in the mesh. `nodeTags' contains the
            node tags (their unique, strictly positive identification numbers). `coord'
            is a vector of length 3 times the length of `nodeTags' that contains the x,
            y, z coordinates of the nodes, concatenated: [n1x, n1y, n1z, n2x, ...]. If
            `dim' >= 0 and `returnParamtricCoord' is set, `parametricCoord' contains
            the parametric coordinates ([u1, u2, ...] or [u1, v1, u2, ...]) of the
            nodes, if available. The length of `parametricCoord' can be 0 or `dim'
            times the length of `nodeTags'. If `includeBoundary' is set, also return
            the nodes classified on the boundary of the entity (which will be
            reparametrized on the entity if `dim' >= 0 in order to compute their
            parametric coordinates).

            Return `nodeTags', `coord', `parametricCoord'.
            """
            api_nodeTags_, api_nodeTags_n_ = POINTER(c_size_t)(), c_size_t()
            api_coord_, api_coord_n_ = POINTER(c_double)(), c_size_t()
            api_parametricCoord_, api_parametricCoord_n_ = POINTER(c_double)(), c_size_t()
            ierr = c_int()
            lib.gmshModelMeshGetNodes(
                byref(api_nodeTags_), byref(api_nodeTags_n_),
                byref(api_coord_), byref(api_coord_n_),
                byref(api_parametricCoord_), byref(api_parametricCoord_n_),
                c_int(dim),
                c_int(tag),
                c_int(bool(includeBoundary)),
                c_int(bool(returnParametricCoord)),
                byref(ierr))
            if ierr.value != 0:
                raise ValueError(
                    "gmshModelMeshGetNodes returned non-zero error code: ",
                    ierr.value)
            return (
                _ovectorsize(api_nodeTags_, api_nodeTags_n_.value),
                _ovectordouble(api_coord_, api_coord_n_.value),
                _ovectordouble(api_parametricCoord_, api_parametricCoord_n_.value))

        @staticmethod
        def getNodesByElementType(elementType, tag=-1, returnParametricCoord=True):
            """
            gmsh.model.mesh.getNodesByElementType(elementType, tag=-1, returnParametricCoord=True)

            Get the nodes classified on the entity of tag `tag', for all the elements
            of type `elementType'. The other arguments are treated as in `getNodes'.

            Return `nodeTags', `coord', `parametricCoord'.
            """
            api_nodeTags_, api_nodeTags_n_ = POINTER(c_size_t)(), c_size_t()
            api_coord_, api_coord_n_ = POINTER(c_double)(), c_size_t()
            api_parametricCoord_, api_parametricCoord_n_ = POINTER(c_double)(), c_size_t()
            ierr = c_int()
            lib.gmshModelMeshGetNodesByElementType(
                c_int(elementType),
                byref(api_nodeTags_), byref(api_nodeTags_n_),
                byref(api_coord_), byref(api_coord_n_),
                byref(api_parametricCoord_), byref(api_parametricCoord_n_),
                c_int(tag),
                c_int(bool(returnParametricCoord)),
                byref(ierr))
            if ierr.value != 0:
                raise ValueError(
                    "gmshModelMeshGetNodesByElementType returned non-zero error code: ",
                    ierr.value)
            return (
                _ovectorsize(api_nodeTags_, api_nodeTags_n_.value),
                _ovectordouble(api_coord_, api_coord_n_.value),
                _ovectordouble(api_parametricCoord_, api_parametricCoord_n_.value))

        @staticmethod
        def getNode(nodeTag):
            """
            gmsh.model.mesh.getNode(nodeTag)

            Get the coordinates and the parametric coordinates (if any) of the node
            with tag `tag'. This function relies on an internal cache (a vector in case
            of dense node numbering, a map otherwise); for large meshes accessing nodes
            in bulk is often preferable.

            Return `coord', `parametricCoord'.
            """
            api_coord_, api_coord_n_ = POINTER(c_double)(), c_size_t()
            api_parametricCoord_, api_parametricCoord_n_ = POINTER(c_double)(), c_size_t()
            ierr = c_int()
            lib.gmshModelMeshGetNode(
                c_size_t(nodeTag),
                byref(api_coord_), byref(api_coord_n_),
                byref(api_parametricCoord_), byref(api_parametricCoord_n_),
                byref(ierr))
            if ierr.value != 0:
                raise ValueError(
                    "gmshModelMeshGetNode returned non-zero error code: ",
                    ierr.value)
            return (
                _ovectordouble(api_coord_, api_coord_n_.value),
                _ovectordouble(api_parametricCoord_, api_parametricCoord_n_.value))

        @staticmethod
        def setNode(nodeTag, coord, parametricCoord):
            """
            gmsh.model.mesh.setNode(nodeTag, coord, parametricCoord)

            Set the coordinates and the parametric coordinates (if any) of the node
            with tag `tag'. This function relies on an internal cache (a vector in case
            of dense node numbering, a map otherwise); for large meshes accessing nodes
            in bulk is often preferable.
            """
            api_coord_, api_coord_n_ = _ivectordouble(coord)
            api_parametricCoord_, api_parametricCoord_n_ = _ivectordouble(parametricCoord)
            ierr = c_int()
            lib.gmshModelMeshSetNode(
                c_size_t(nodeTag),
                api_coord_, api_coord_n_,
                api_parametricCoord_, api_parametricCoord_n_,
                byref(ierr))
            if ierr.value != 0:
                raise ValueError(
                    "gmshModelMeshSetNode returned non-zero error code: ",
                    ierr.value)

        @staticmethod
        def rebuildNodeCache(onlyIfNecessary=True):
            """
            gmsh.model.mesh.rebuildNodeCache(onlyIfNecessary=True)

            Rebuild the node cache.
            """
            ierr = c_int()
            lib.gmshModelMeshRebuildNodeCache(
                c_int(bool(onlyIfNecessary)),
                byref(ierr))
            if ierr.value != 0:
                raise ValueError(
                    "gmshModelMeshRebuildNodeCache returned non-zero error code: ",
                    ierr.value)

        @staticmethod
        def getNodesForPhysicalGroup(dim, tag):
            """
            gmsh.model.mesh.getNodesForPhysicalGroup(dim, tag)

            Get the nodes from all the elements belonging to the physical group of
            dimension `dim' and tag `tag'. `nodeTags' contains the node tags; `coord'
            is a vector of length 3 times the length of `nodeTags' that contains the x,
            y, z coordinates of the nodes, concatenated: [n1x, n1y, n1z, n2x, ...].

            Return `nodeTags', `coord'.
            """
            api_nodeTags_, api_nodeTags_n_ = POINTER(c_size_t)(), c_size_t()
            api_coord_, api_coord_n_ = POINTER(c_double)(), c_size_t()
            ierr = c_int()
            lib.gmshModelMeshGetNodesForPhysicalGroup(
                c_int(dim),
                c_int(tag),
                byref(api_nodeTags_), byref(api_nodeTags_n_),
                byref(api_coord_), byref(api_coord_n_),
                byref(ierr))
            if ierr.value != 0:
                raise ValueError(
                    "gmshModelMeshGetNodesForPhysicalGroup returned non-zero error code: ",
                    ierr.value)
            return (
                _ovectorsize(api_nodeTags_, api_nodeTags_n_.value),
                _ovectordouble(api_coord_, api_coord_n_.value))

        @staticmethod
        def addNodes(dim, tag, nodeTags, coord, parametricCoord=[]):
            """
            gmsh.model.mesh.addNodes(dim, tag, nodeTags, coord, parametricCoord=[])

            Add nodes classified on the model entity of dimension `dim' and tag `tag'.
            `nodeTags' contains the node tags (their unique, strictly positive
            identification numbers). `coord' is a vector of length 3 times the length
            of `nodeTags' that contains the x, y, z coordinates of the nodes,
            concatenated: [n1x, n1y, n1z, n2x, ...]. The optional `parametricCoord'
            vector contains the parametric coordinates of the nodes, if any. The length
            of `parametricCoord' can be 0 or `dim' times the length of `nodeTags'. If
            the `nodeTags' vector is empty, new tags are automatically assigned to the
            nodes.
            """
            api_nodeTags_, api_nodeTags_n_ = _ivectorsize(nodeTags)
            api_coord_, api_coord_n_ = _ivectordouble(coord)
            api_parametricCoord_, api_parametricCoord_n_ = _ivectordouble(parametricCoord)
            ierr = c_int()
            lib.gmshModelMeshAddNodes(
                c_int(dim),
                c_int(tag),
                api_nodeTags_, api_nodeTags_n_,
                api_coord_, api_coord_n_,
                api_parametricCoord_, api_parametricCoord_n_,
                byref(ierr))
            if ierr.value != 0:
                raise ValueError(
                    "gmshModelMeshAddNodes returned non-zero error code: ",
                    ierr.value)

        @staticmethod
        def reclassifyNodes():
            """
            gmsh.model.mesh.reclassifyNodes()

            Reclassify all nodes on their associated model entity, based on the
            elements. Can be used when importing nodes in bulk (e.g. by associating
            them all to a single volume), to reclassify them correctly on model
            surfaces, curves, etc. after the elements have been set.
            """
            ierr = c_int()
            lib.gmshModelMeshReclassifyNodes(
                byref(ierr))
            if ierr.value != 0:
                raise ValueError(
                    "gmshModelMeshReclassifyNodes returned non-zero error code: ",
                    ierr.value)

        @staticmethod
        def relocateNodes(dim=-1, tag=-1):
            """
            gmsh.model.mesh.relocateNodes(dim=-1, tag=-1)

            Relocate the nodes classified on the entity of dimension `dim' and tag
            `tag' using their parametric coordinates. If `tag' < 0, relocate the nodes
            for all entities of dimension `dim'. If `dim' and `tag' are negative,
            relocate all the nodes in the mesh.
            """
            ierr = c_int()
            lib.gmshModelMeshRelocateNodes(
                c_int(dim),
                c_int(tag),
                byref(ierr))
            if ierr.value != 0:
                raise ValueError(
                    "gmshModelMeshRelocateNodes returned non-zero error code: ",
                    ierr.value)

        @staticmethod
        def getElements(dim=-1, tag=-1):
            """
            gmsh.model.mesh.getElements(dim=-1, tag=-1)

            Get the elements classified on the entity of dimension `dim' and tag `tag'.
            If `tag' < 0, get the elements for all entities of dimension `dim'. If
            `dim' and `tag' are negative, get all the elements in the mesh.
            `elementTypes' contains the MSH types of the elements (e.g. `2' for 3-node
            triangles: see `getElementProperties' to obtain the properties for a given
            element type). `elementTags' is a vector of the same length as
            `elementTypes'; each entry is a vector containing the tags (unique,
            strictly positive identifiers) of the elements of the corresponding type.
            `nodeTags' is also a vector of the same length as `elementTypes'; each
            entry is a vector of length equal to the number of elements of the given
            type times the number N of nodes for this type of element, that contains
            the node tags of all the elements of the given type, concatenated: [e1n1,
            e1n2, ..., e1nN, e2n1, ...].

            Return `elementTypes', `elementTags', `nodeTags'.
            """
            api_elementTypes_, api_elementTypes_n_ = POINTER(c_int)(), c_size_t()
            api_elementTags_, api_elementTags_n_, api_elementTags_nn_ = POINTER(POINTER(c_size_t))(), POINTER(c_size_t)(), c_size_t()
            api_nodeTags_, api_nodeTags_n_, api_nodeTags_nn_ = POINTER(POINTER(c_size_t))(), POINTER(c_size_t)(), c_size_t()
            ierr = c_int()
            lib.gmshModelMeshGetElements(
                byref(api_elementTypes_), byref(api_elementTypes_n_),
                byref(api_elementTags_), byref(api_elementTags_n_), byref(api_elementTags_nn_),
                byref(api_nodeTags_), byref(api_nodeTags_n_), byref(api_nodeTags_nn_),
                c_int(dim),
                c_int(tag),
                byref(ierr))
            if ierr.value != 0:
                raise ValueError(
                    "gmshModelMeshGetElements returned non-zero error code: ",
                    ierr.value)
            return (
                _ovectorint(api_elementTypes_, api_elementTypes_n_.value),
                _ovectorvectorsize(api_elementTags_, api_elementTags_n_, api_elementTags_nn_),
                _ovectorvectorsize(api_nodeTags_, api_nodeTags_n_, api_nodeTags_nn_))

        @staticmethod
        def getElement(elementTag):
            """
            gmsh.model.mesh.getElement(elementTag)

            Get the type and node tags of the element with tag `tag'. This function
            relies on an internal cache (a vector in case of dense element numbering, a
            map otherwise); for large meshes accessing elements in bulk is often
            preferable.

            Return `elementType', `nodeTags'.
            """
            api_elementType_ = c_int()
            api_nodeTags_, api_nodeTags_n_ = POINTER(c_size_t)(), c_size_t()
            ierr = c_int()
            lib.gmshModelMeshGetElement(
                c_size_t(elementTag),
                byref(api_elementType_),
                byref(api_nodeTags_), byref(api_nodeTags_n_),
                byref(ierr))
            if ierr.value != 0:
                raise ValueError(
                    "gmshModelMeshGetElement returned non-zero error code: ",
                    ierr.value)
            return (
                api_elementType_.value,
                _ovectorsize(api_nodeTags_, api_nodeTags_n_.value))

        @staticmethod
        def getElementByCoordinates(x, y, z, dim=-1, strict=False):
            """
            gmsh.model.mesh.getElementByCoordinates(x, y, z, dim=-1, strict=False)

            Search the mesh for an element located at coordinates (`x', `y', `z'). This
            function performs a search in a spatial octree. If an element is found,
            return its tag, type and node tags, as well as the local coordinates (`u',
            `v', `w') within the element corresponding to search location. If `dim' is
            >= 0, only search for elements of the given dimension. If `strict' is not
            set, use a tolerance to find elements near the search location.

            Return `elementTag', `elementType', `nodeTags', `u', `v', `w'.
            """
            api_elementTag_ = c_size_t()
            api_elementType_ = c_int()
            api_nodeTags_, api_nodeTags_n_ = POINTER(c_size_t)(), c_size_t()
            api_u_ = c_double()
            api_v_ = c_double()
            api_w_ = c_double()
            ierr = c_int()
            lib.gmshModelMeshGetElementByCoordinates(
                c_double(x),
                c_double(y),
                c_double(z),
                byref(api_elementTag_),
                byref(api_elementType_),
                byref(api_nodeTags_), byref(api_nodeTags_n_),
                byref(api_u_),
                byref(api_v_),
                byref(api_w_),
                c_int(dim),
                c_int(bool(strict)),
                byref(ierr))
            if ierr.value != 0:
                raise ValueError(
                    "gmshModelMeshGetElementByCoordinates returned non-zero error code: ",
                    ierr.value)
            return (
                api_elementTag_.value,
                api_elementType_.value,
                _ovectorsize(api_nodeTags_, api_nodeTags_n_.value),
                api_u_.value,
                api_v_.value,
                api_w_.value)

        @staticmethod
        def getElementsByCoordinates(x, y, z, dim=-1, strict=False):
            """
            gmsh.model.mesh.getElementsByCoordinates(x, y, z, dim=-1, strict=False)

            Search the mesh for element(s) located at coordinates (`x', `y', `z'). This
            function performs a search in a spatial octree. Return the tags of all
            found elements in `elementTags'. Additional information about the elements
            can be accessed through `getElement' and `getLocalCoordinatesInElement'. If
            `dim' is >= 0, only search for elements of the given dimension. If `strict'
            is not set, use a tolerance to find elements near the search location.

            Return `elementTags'.
            """
            api_elementTags_, api_elementTags_n_ = POINTER(c_size_t)(), c_size_t()
            ierr = c_int()
            lib.gmshModelMeshGetElementsByCoordinates(
                c_double(x),
                c_double(y),
                c_double(z),
                byref(api_elementTags_), byref(api_elementTags_n_),
                c_int(dim),
                c_int(bool(strict)),
                byref(ierr))
            if ierr.value != 0:
                raise ValueError(
                    "gmshModelMeshGetElementsByCoordinates returned non-zero error code: ",
                    ierr.value)
            return _ovectorsize(api_elementTags_, api_elementTags_n_.value)

        @staticmethod
        def getLocalCoordinatesInElement(elementTag, x, y, z):
            """
            gmsh.model.mesh.getLocalCoordinatesInElement(elementTag, x, y, z)

            Return the local coordinates (`u', `v', `w') within the element
            `elementTag' corresponding to the model coordinates (`x', `y', `z'). This
            function relies on an internal cache (a vector in case of dense element
            numbering, a map otherwise); for large meshes accessing elements in bulk is
            often preferable.

            Return `u', `v', `w'.
            """
            api_u_ = c_double()
            api_v_ = c_double()
            api_w_ = c_double()
            ierr = c_int()
            lib.gmshModelMeshGetLocalCoordinatesInElement(
                c_size_t(elementTag),
                c_double(x),
                c_double(y),
                c_double(z),
                byref(api_u_),
                byref(api_v_),
                byref(api_w_),
                byref(ierr))
            if ierr.value != 0:
                raise ValueError(
                    "gmshModelMeshGetLocalCoordinatesInElement returned non-zero error code: ",
                    ierr.value)
            return (
                api_u_.value,
                api_v_.value,
                api_w_.value)

        @staticmethod
        def getElementTypes(dim=-1, tag=-1):
            """
            gmsh.model.mesh.getElementTypes(dim=-1, tag=-1)

            Get the types of elements in the entity of dimension `dim' and tag `tag'.
            If `tag' < 0, get the types for all entities of dimension `dim'. If `dim'
            and `tag' are negative, get all the types in the mesh.

            Return `elementTypes'.
            """
            api_elementTypes_, api_elementTypes_n_ = POINTER(c_int)(), c_size_t()
            ierr = c_int()
            lib.gmshModelMeshGetElementTypes(
                byref(api_elementTypes_), byref(api_elementTypes_n_),
                c_int(dim),
                c_int(tag),
                byref(ierr))
            if ierr.value != 0:
                raise ValueError(
                    "gmshModelMeshGetElementTypes returned non-zero error code: ",
                    ierr.value)
            return _ovectorint(api_elementTypes_, api_elementTypes_n_.value)

        @staticmethod
        def getElementType(familyName, order, serendip=False):
            """
            gmsh.model.mesh.getElementType(familyName, order, serendip=False)

            Return an element type given its family name `familyName' ("point", "line",
            "triangle", "quadrangle", "tetrahedron", "pyramid", "prism", "hexahedron")
            and polynomial order `order'. If `serendip' is true, return the
            corresponding serendip element type (element without interior nodes).

            Return an integer value.
            """
            ierr = c_int()
            api__result__ = lib.gmshModelMeshGetElementType(
                c_char_p(familyName.encode()),
                c_int(order),
                c_int(bool(serendip)),
                byref(ierr))
            if ierr.value != 0:
                raise ValueError(
                    "gmshModelMeshGetElementType returned non-zero error code: ",
                    ierr.value)
            return api__result__

        @staticmethod
        def getElementProperties(elementType):
            """
            gmsh.model.mesh.getElementProperties(elementType)

            Get the properties of an element of type `elementType': its name
            (`elementName'), dimension (`dim'), order (`order'), number of nodes
            (`numNodes'), coordinates of the nodes in the reference element
            (`nodeCoord' vector, of length `dim' times `numNodes') and number of
            primary (first order) nodes (`numPrimaryNodes').

            Return `elementName', `dim', `order', `numNodes', `nodeCoord', `numPrimaryNodes'.
            """
            api_elementName_ = c_char_p()
            api_dim_ = c_int()
            api_order_ = c_int()
            api_numNodes_ = c_int()
            api_nodeCoord_, api_nodeCoord_n_ = POINTER(c_double)(), c_size_t()
            api_numPrimaryNodes_ = c_int()
            ierr = c_int()
            lib.gmshModelMeshGetElementProperties(
                c_int(elementType),
                byref(api_elementName_),
                byref(api_dim_),
                byref(api_order_),
                byref(api_numNodes_),
                byref(api_nodeCoord_), byref(api_nodeCoord_n_),
                byref(api_numPrimaryNodes_),
                byref(ierr))
            if ierr.value != 0:
                raise ValueError(
                    "gmshModelMeshGetElementProperties returned non-zero error code: ",
                    ierr.value)
            return (
                _ostring(api_elementName_),
                api_dim_.value,
                api_order_.value,
                api_numNodes_.value,
                _ovectordouble(api_nodeCoord_, api_nodeCoord_n_.value),
                api_numPrimaryNodes_.value)

        @staticmethod
        def getElementsByType(elementType, tag=-1, task=0, numTasks=1):
            """
            gmsh.model.mesh.getElementsByType(elementType, tag=-1, task=0, numTasks=1)

            Get the elements of type `elementType' classified on the entity of tag
            `tag'. If `tag' < 0, get the elements for all entities. `elementTags' is a
            vector containing the tags (unique, strictly positive identifiers) of the
            elements of the corresponding type. `nodeTags' is a vector of length equal
            to the number of elements of the given type times the number N of nodes for
            this type of element, that contains the node tags of all the elements of
            the given type, concatenated: [e1n1, e1n2, ..., e1nN, e2n1, ...]. If
            `numTasks' > 1, only compute and return the part of the data indexed by
            `task'.

            Return `elementTags', `nodeTags'.
            """
            api_elementTags_, api_elementTags_n_ = POINTER(c_size_t)(), c_size_t()
            api_nodeTags_, api_nodeTags_n_ = POINTER(c_size_t)(), c_size_t()
            ierr = c_int()
            lib.gmshModelMeshGetElementsByType(
                c_int(elementType),
                byref(api_elementTags_), byref(api_elementTags_n_),
                byref(api_nodeTags_), byref(api_nodeTags_n_),
                c_int(tag),
                c_size_t(task),
                c_size_t(numTasks),
                byref(ierr))
            if ierr.value != 0:
                raise ValueError(
                    "gmshModelMeshGetElementsByType returned non-zero error code: ",
                    ierr.value)
            return (
                _ovectorsize(api_elementTags_, api_elementTags_n_.value),
                _ovectorsize(api_nodeTags_, api_nodeTags_n_.value))

        @staticmethod
        def addElements(dim, tag, elementTypes, elementTags, nodeTags):
            """
            gmsh.model.mesh.addElements(dim, tag, elementTypes, elementTags, nodeTags)

            Add elements classified on the entity of dimension `dim' and tag `tag'.
            `types' contains the MSH types of the elements (e.g. `2' for 3-node
            triangles: see the Gmsh reference manual). `elementTags' is a vector of the
            same length as `types'; each entry is a vector containing the tags (unique,
            strictly positive identifiers) of the elements of the corresponding type.
            `nodeTags' is also a vector of the same length as `types'; each entry is a
            vector of length equal to the number of elements of the given type times
            the number N of nodes per element, that contains the node tags of all the
            elements of the given type, concatenated: [e1n1, e1n2, ..., e1nN, e2n1,
            ...].
            """
            api_elementTypes_, api_elementTypes_n_ = _ivectorint(elementTypes)
            api_elementTags_, api_elementTags_n_, api_elementTags_nn_ = _ivectorvectorsize(elementTags)
            api_nodeTags_, api_nodeTags_n_, api_nodeTags_nn_ = _ivectorvectorsize(nodeTags)
            ierr = c_int()
            lib.gmshModelMeshAddElements(
                c_int(dim),
                c_int(tag),
                api_elementTypes_, api_elementTypes_n_,
                api_elementTags_, api_elementTags_n_, api_elementTags_nn_,
                api_nodeTags_, api_nodeTags_n_, api_nodeTags_nn_,
                byref(ierr))
            if ierr.value != 0:
                raise ValueError(
                    "gmshModelMeshAddElements returned non-zero error code: ",
                    ierr.value)

        @staticmethod
        def addElementsByType(tag, elementType, elementTags, nodeTags):
            """
            gmsh.model.mesh.addElementsByType(tag, elementType, elementTags, nodeTags)

            Add elements of type `elementType' classified on the entity of tag `tag'.
            `elementTags' contains the tags (unique, strictly positive identifiers) of
            the elements of the corresponding type. `nodeTags' is a vector of length
            equal to the number of elements times the number N of nodes per element,
            that contains the node tags of all the elements, concatenated: [e1n1, e1n2,
            ..., e1nN, e2n1, ...]. If the `elementTag' vector is empty, new tags are
            automatically assigned to the elements.
            """
            api_elementTags_, api_elementTags_n_ = _ivectorsize(elementTags)
            api_nodeTags_, api_nodeTags_n_ = _ivectorsize(nodeTags)
            ierr = c_int()
            lib.gmshModelMeshAddElementsByType(
                c_int(tag),
                c_int(elementType),
                api_elementTags_, api_elementTags_n_,
                api_nodeTags_, api_nodeTags_n_,
                byref(ierr))
            if ierr.value != 0:
                raise ValueError(
                    "gmshModelMeshAddElementsByType returned non-zero error code: ",
                    ierr.value)

        @staticmethod
        def getIntegrationPoints(elementType, integrationType):
            """
            gmsh.model.mesh.getIntegrationPoints(elementType, integrationType)

            Get the numerical quadrature information for the given element type
            `elementType' and integration rule `integrationType' (e.g. "Gauss4" for a
            Gauss quadrature suited for integrating 4th order polynomials).
            `integrationPoints' contains the u, v, w coordinates of the G integration
            points in the reference element: [g1u, g1v, g1w, ..., gGu, gGv, gGw].
            `integrationWeigths' contains the associated weights: [g1q, ..., gGq].

            Return `integrationPoints', `integrationWeights'.
            """
            api_integrationPoints_, api_integrationPoints_n_ = POINTER(c_double)(), c_size_t()
            api_integrationWeights_, api_integrationWeights_n_ = POINTER(c_double)(), c_size_t()
            ierr = c_int()
            lib.gmshModelMeshGetIntegrationPoints(
                c_int(elementType),
                c_char_p(integrationType.encode()),
                byref(api_integrationPoints_), byref(api_integrationPoints_n_),
                byref(api_integrationWeights_), byref(api_integrationWeights_n_),
                byref(ierr))
            if ierr.value != 0:
                raise ValueError(
                    "gmshModelMeshGetIntegrationPoints returned non-zero error code: ",
                    ierr.value)
            return (
                _ovectordouble(api_integrationPoints_, api_integrationPoints_n_.value),
                _ovectordouble(api_integrationWeights_, api_integrationWeights_n_.value))

        @staticmethod
        def getJacobians(elementType, integrationPoints, tag=-1, task=0, numTasks=1):
            """
            gmsh.model.mesh.getJacobians(elementType, integrationPoints, tag=-1, task=0, numTasks=1)

            Get the Jacobians of all the elements of type `elementType' classified on
            the entity of tag `tag', at the G integration points `integrationPoints'
            given as concatenated triplets of coordinates in the reference element
            [g1u, g1v, g1w, ..., gGu, gGv, gGw]. Data is returned by element, with
            elements in the same order as in `getElements' and `getElementsByType'.
            `jacobians' contains for each element the 9 entries of the 3x3 Jacobian
            matrix at each integration point. The matrix is returned by column:
            [e1g1Jxu, e1g1Jyu, e1g1Jzu, e1g1Jxv, ..., e1g1Jzw, e1g2Jxu, ..., e1gGJzw,
            e2g1Jxu, ...], with Jxu=dx/du, Jyu=dy/du, etc. `determinants' contains for
            each element the determinant of the Jacobian matrix at each integration
            point: [e1g1, e1g2, ... e1gG, e2g1, ...]. `points' contains for each
            element the x, y, z coordinates of the integration points. If `tag' < 0,
            get the Jacobian data for all entities. If `numTasks' > 1, only compute and
            return the part of the data indexed by `task'.

            Return `jacobians', `determinants', `points'.
            """
            api_integrationPoints_, api_integrationPoints_n_ = _ivectordouble(integrationPoints)
            api_jacobians_, api_jacobians_n_ = POINTER(c_double)(), c_size_t()
            api_determinants_, api_determinants_n_ = POINTER(c_double)(), c_size_t()
            api_points_, api_points_n_ = POINTER(c_double)(), c_size_t()
            ierr = c_int()
            lib.gmshModelMeshGetJacobians(
                c_int(elementType),
                api_integrationPoints_, api_integrationPoints_n_,
                byref(api_jacobians_), byref(api_jacobians_n_),
                byref(api_determinants_), byref(api_determinants_n_),
                byref(api_points_), byref(api_points_n_),
                c_int(tag),
                c_size_t(task),
                c_size_t(numTasks),
                byref(ierr))
            if ierr.value != 0:
                raise ValueError(
                    "gmshModelMeshGetJacobians returned non-zero error code: ",
                    ierr.value)
            return (
                _ovectordouble(api_jacobians_, api_jacobians_n_.value),
                _ovectordouble(api_determinants_, api_determinants_n_.value),
                _ovectordouble(api_points_, api_points_n_.value))

        @staticmethod
        def getBasisFunctions(elementType, integrationPoints, functionSpaceType):
            """
            gmsh.model.mesh.getBasisFunctions(elementType, integrationPoints, functionSpaceType)

            Get the basis functions of the element of type `elementType' at the
            integration points `integrationPoints' (given as concatenated triplets of
            coordinates in the reference element [g1u, g1v, g1w, ..., gGu, gGv, gGw]),
            for the function space `functionSpaceType' (e.g. "Lagrange" or
            "GradLagrange" for Lagrange basis functions or their gradient, in the u, v,
            w coordinates of the reference element). `numComponents' returns the number
            C of components of a basis function. `basisFunctions' returns the value of
            the N basis functions at the integration points, i.e. [g1f1, g1f2, ...,
            g1fN, g2f1, ...] when C == 1 or [g1f1u, g1f1v, g1f1w, g1f2u, ..., g1fNw,
            g2f1u, ...] when C == 3.

            Return `numComponents', `basisFunctions'.
            """
            api_integrationPoints_, api_integrationPoints_n_ = _ivectordouble(integrationPoints)
            api_numComponents_ = c_int()
            api_basisFunctions_, api_basisFunctions_n_ = POINTER(c_double)(), c_size_t()
            ierr = c_int()
            lib.gmshModelMeshGetBasisFunctions(
                c_int(elementType),
                api_integrationPoints_, api_integrationPoints_n_,
                c_char_p(functionSpaceType.encode()),
                byref(api_numComponents_),
                byref(api_basisFunctions_), byref(api_basisFunctions_n_),
                byref(ierr))
            if ierr.value != 0:
                raise ValueError(
                    "gmshModelMeshGetBasisFunctions returned non-zero error code: ",
                    ierr.value)
            return (
                api_numComponents_.value,
                _ovectordouble(api_basisFunctions_, api_basisFunctions_n_.value))

        @staticmethod
        def getEdgeNumber(edgeNodes):
            """
            gmsh.model.mesh.getEdgeNumber(edgeNodes)

            Get the global edge identifier `edgeNum' for an input list of node pairs,
            concatenated in the vector `edgeNodes'.  Warning: this is an experimental
            feature and will probably change in a future release.

            Return `edgeNum'.
            """
            api_edgeNodes_, api_edgeNodes_n_ = _ivectorint(edgeNodes)
            api_edgeNum_, api_edgeNum_n_ = POINTER(c_int)(), c_size_t()
            ierr = c_int()
            lib.gmshModelMeshGetEdgeNumber(
                api_edgeNodes_, api_edgeNodes_n_,
                byref(api_edgeNum_), byref(api_edgeNum_n_),
                byref(ierr))
            if ierr.value != 0:
                raise ValueError(
                    "gmshModelMeshGetEdgeNumber returned non-zero error code: ",
                    ierr.value)
            return _ovectorint(api_edgeNum_, api_edgeNum_n_.value)

        @staticmethod
        def getLocalMultipliersForHcurl0(elementType, tag=-1):
            """
            gmsh.model.mesh.getLocalMultipliersForHcurl0(elementType, tag=-1)

            Get the local multipliers (to guarantee H(curl)-conformity) of the order 0
            H(curl) basis functions. Warning: this is an experimental feature and will
            probably change in a future release.

            Return `localMultipliers'.
            """
            api_localMultipliers_, api_localMultipliers_n_ = POINTER(c_int)(), c_size_t()
            ierr = c_int()
            lib.gmshModelMeshGetLocalMultipliersForHcurl0(
                c_int(elementType),
                byref(api_localMultipliers_), byref(api_localMultipliers_n_),
                c_int(tag),
                byref(ierr))
            if ierr.value != 0:
                raise ValueError(
                    "gmshModelMeshGetLocalMultipliersForHcurl0 returned non-zero error code: ",
                    ierr.value)
            return _ovectorint(api_localMultipliers_, api_localMultipliers_n_.value)

        @staticmethod
        def getBasisFunctionsForElements(elementType, integrationPoints, functionSpaceType, tag=-1, task=0, numTasks=1):
            """
            gmsh.model.mesh.getBasisFunctionsForElements(elementType, integrationPoints, functionSpaceType, tag=-1, task=0, numTasks=1)

            Get the element-dependent basis functions of the elements of type
            `elementType' in the entity of tag `tag'at the integration points
            `integrationPoints' (given as concatenated triplets of coordinates in the
            reference element [g1u, g1v, g1w, ..., gGu, gGv, gGw]), for the function
            space `functionSpaceType' (e.g. "H1Legendre3" or "GradH1Legendre3" for 3rd
            order hierarchical H1 Legendre functions or their gradient, in the u, v, w
            coordinates of the reference elements). `numComponents' returns the number
            C of components of a basis function. `numBasisFunctions' returns the number
            N of basis functions per element. `basisFunctions' returns the value of the
            basis functions at the integration points for each element: [e1g1f1,...,
            e1g1fN, e1g2f1,..., e2g1f1, ...] when C == 1 or [e1g1f1u, e1g1f1v,...,
            e1g1fNw, e1g2f1u,..., e2g1f1u, ...]. Warning: this is an experimental
            feature and will probably change in a future release. If `numTasks' > 1,
            only compute and return the part of the data indexed by `task'.

            Return `numComponents', `numFunctionsPerElements', `basisFunctions'.
            """
            api_integrationPoints_, api_integrationPoints_n_ = _ivectordouble(integrationPoints)
            api_numComponents_ = c_int()
            api_numFunctionsPerElements_ = c_int()
            api_basisFunctions_, api_basisFunctions_n_ = POINTER(c_double)(), c_size_t()
            ierr = c_int()
            lib.gmshModelMeshGetBasisFunctionsForElements(
                c_int(elementType),
                api_integrationPoints_, api_integrationPoints_n_,
                c_char_p(functionSpaceType.encode()),
                byref(api_numComponents_),
                byref(api_numFunctionsPerElements_),
                byref(api_basisFunctions_), byref(api_basisFunctions_n_),
                c_int(tag),
                c_size_t(task),
                c_size_t(numTasks),
                byref(ierr))
            if ierr.value != 0:
                raise ValueError(
                    "gmshModelMeshGetBasisFunctionsForElements returned non-zero error code: ",
                    ierr.value)
            return (
                api_numComponents_.value,
                api_numFunctionsPerElements_.value,
                _ovectordouble(api_basisFunctions_, api_basisFunctions_n_.value))

        @staticmethod
        def getKeysForElements(elementType, functionSpaceType, tag=-1, returnCoord=True):
            """
            gmsh.model.mesh.getKeysForElements(elementType, functionSpaceType, tag=-1, returnCoord=True)

            Generate the `keys' for the elements of type `elementType' in the entity of
            tag `tag', for the `functionSpaceType' function space. Each key uniquely
            identifies a basis function in the function space. If `returnCoord' is set,
            the `coord' vector contains the x, y, z coordinates locating basis
            functions for sorting purposes. Warning: this is an experimental feature
            and will probably change in a future release.

            Return `keys', `coord'.
            """
            api_keys_, api_keys_n_ = POINTER(c_int)(), c_size_t()
            api_coord_, api_coord_n_ = POINTER(c_double)(), c_size_t()
            ierr = c_int()
            lib.gmshModelMeshGetKeysForElements(
                c_int(elementType),
                c_char_p(functionSpaceType.encode()),
                byref(api_keys_), byref(api_keys_n_),
                byref(api_coord_), byref(api_coord_n_),
                c_int(tag),
                c_int(bool(returnCoord)),
                byref(ierr))
            if ierr.value != 0:
                raise ValueError(
                    "gmshModelMeshGetKeysForElements returned non-zero error code: ",
                    ierr.value)
            return (
                _ovectorpair(api_keys_, api_keys_n_.value),
                _ovectordouble(api_coord_, api_coord_n_.value))

        @staticmethod
        def getNumberOfKeysForElements(elementType, functionSpaceType):
            """
            gmsh.model.mesh.getNumberOfKeysForElements(elementType, functionSpaceType)

            Get the number of keys by elements of type `elementType' for function space
            named `functionSpaceType'.

            Return an integer value.
            """
            ierr = c_int()
            api__result__ = lib.gmshModelMeshGetNumberOfKeysForElements(
                c_int(elementType),
                c_char_p(functionSpaceType.encode()),
                byref(ierr))
            if ierr.value != 0:
                raise ValueError(
                    "gmshModelMeshGetNumberOfKeysForElements returned non-zero error code: ",
                    ierr.value)
            return api__result__

        @staticmethod
        def getInformationForElements(keys, elementType, functionSpaceType):
            """
            gmsh.model.mesh.getInformationForElements(keys, elementType, functionSpaceType)

            Get information about the `keys'. `infoKeys' returns information about the
            functions associated with the `keys'. `infoKeys[0].first' describes the
            type of function (0 for  vertex function, 1 for edge function, 2 for face
            function and 3 for bubble function). `infoKeys[0].second' gives the order
            of the function associated with the key. Warning: this is an experimental
            feature and will probably change in a future release.

            Return `infoKeys'.
            """
            api_keys_, api_keys_n_ = _ivectorpair(keys)
            api_infoKeys_, api_infoKeys_n_ = POINTER(c_int)(), c_size_t()
            ierr = c_int()
            lib.gmshModelMeshGetInformationForElements(
                api_keys_, api_keys_n_,
                c_int(elementType),
                c_char_p(functionSpaceType.encode()),
                byref(api_infoKeys_), byref(api_infoKeys_n_),
                byref(ierr))
            if ierr.value != 0:
                raise ValueError(
                    "gmshModelMeshGetInformationForElements returned non-zero error code: ",
                    ierr.value)
            return _ovectorpair(api_infoKeys_, api_infoKeys_n_.value)

        @staticmethod
        def precomputeBasisFunctions(elementType):
            """
            gmsh.model.mesh.precomputeBasisFunctions(elementType)

            Precomputes the basis functions corresponding to `elementType'.
            """
            ierr = c_int()
            lib.gmshModelMeshPrecomputeBasisFunctions(
                c_int(elementType),
                byref(ierr))
            if ierr.value != 0:
                raise ValueError(
                    "gmshModelMeshPrecomputeBasisFunctions returned non-zero error code: ",
                    ierr.value)

        @staticmethod
        def getBarycenters(elementType, tag, fast, primary, task=0, numTasks=1):
            """
            gmsh.model.mesh.getBarycenters(elementType, tag, fast, primary, task=0, numTasks=1)

            Get the barycenters of all elements of type `elementType' classified on the
            entity of tag `tag'. If `primary' is set, only the primary nodes of the
            elements are taken into account for the barycenter calculation. If `fast'
            is set, the function returns the sum of the primary node coordinates
            (without normalizing by the number of nodes). If `tag' < 0, get the
            barycenters for all entities. If `numTasks' > 1, only compute and return
            the part of the data indexed by `task'.

            Return `barycenters'.
            """
            api_barycenters_, api_barycenters_n_ = POINTER(c_double)(), c_size_t()
            ierr = c_int()
            lib.gmshModelMeshGetBarycenters(
                c_int(elementType),
                c_int(tag),
                c_int(bool(fast)),
                c_int(bool(primary)),
                byref(api_barycenters_), byref(api_barycenters_n_),
                c_size_t(task),
                c_size_t(numTasks),
                byref(ierr))
            if ierr.value != 0:
                raise ValueError(
                    "gmshModelMeshGetBarycenters returned non-zero error code: ",
                    ierr.value)
            return _ovectordouble(api_barycenters_, api_barycenters_n_.value)

        @staticmethod
        def getElementEdgeNodes(elementType, tag=-1, primary=False, task=0, numTasks=1):
            """
            gmsh.model.mesh.getElementEdgeNodes(elementType, tag=-1, primary=False, task=0, numTasks=1)

            Get the nodes on the edges of all elements of type `elementType' classified
            on the entity of tag `tag'. `nodeTags' contains the node tags of the edges
            for all the elements: [e1a1n1, e1a1n2, e1a2n1, ...]. Data is returned by
            element, with elements in the same order as in `getElements' and
            `getElementsByType'. If `primary' is set, only the primary (begin/end)
            nodes of the edges are returned. If `tag' < 0, get the edge nodes for all
            entities. If `numTasks' > 1, only compute and return the part of the data
            indexed by `task'.

            Return `nodeTags'.
            """
            api_nodeTags_, api_nodeTags_n_ = POINTER(c_size_t)(), c_size_t()
            ierr = c_int()
            lib.gmshModelMeshGetElementEdgeNodes(
                c_int(elementType),
                byref(api_nodeTags_), byref(api_nodeTags_n_),
                c_int(tag),
                c_int(bool(primary)),
                c_size_t(task),
                c_size_t(numTasks),
                byref(ierr))
            if ierr.value != 0:
                raise ValueError(
                    "gmshModelMeshGetElementEdgeNodes returned non-zero error code: ",
                    ierr.value)
            return _ovectorsize(api_nodeTags_, api_nodeTags_n_.value)

        @staticmethod
        def getElementFaceNodes(elementType, faceType, tag=-1, primary=False, task=0, numTasks=1):
            """
            gmsh.model.mesh.getElementFaceNodes(elementType, faceType, tag=-1, primary=False, task=0, numTasks=1)

            Get the nodes on the faces of type `faceType' (3 for triangular faces, 4
            for quadrangular faces) of all elements of type `elementType' classified on
            the entity of tag `tag'. `nodeTags' contains the node tags of the faces for
            all elements: [e1f1n1, ..., e1f1nFaceType, e1f2n1, ...]. Data is returned
            by element, with elements in the same order as in `getElements' and
            `getElementsByType'. If `primary' is set, only the primary (corner) nodes
            of the faces are returned. If `tag' < 0, get the face nodes for all
            entities. If `numTasks' > 1, only compute and return the part of the data
            indexed by `task'.

            Return `nodeTags'.
            """
            api_nodeTags_, api_nodeTags_n_ = POINTER(c_size_t)(), c_size_t()
            ierr = c_int()
            lib.gmshModelMeshGetElementFaceNodes(
                c_int(elementType),
                c_int(faceType),
                byref(api_nodeTags_), byref(api_nodeTags_n_),
                c_int(tag),
                c_int(bool(primary)),
                c_size_t(task),
                c_size_t(numTasks),
                byref(ierr))
            if ierr.value != 0:
                raise ValueError(
                    "gmshModelMeshGetElementFaceNodes returned non-zero error code: ",
                    ierr.value)
            return _ovectorsize(api_nodeTags_, api_nodeTags_n_.value)

        @staticmethod
        def getGhostElements(dim, tag):
            """
            gmsh.model.mesh.getGhostElements(dim, tag)

            Get the ghost elements `elementTags' and their associated `partitions'
            stored in the ghost entity of dimension `dim' and tag `tag'.

            Return `elementTags', `partitions'.
            """
            api_elementTags_, api_elementTags_n_ = POINTER(c_size_t)(), c_size_t()
            api_partitions_, api_partitions_n_ = POINTER(c_int)(), c_size_t()
            ierr = c_int()
            lib.gmshModelMeshGetGhostElements(
                c_int(dim),
                c_int(tag),
                byref(api_elementTags_), byref(api_elementTags_n_),
                byref(api_partitions_), byref(api_partitions_n_),
                byref(ierr))
            if ierr.value != 0:
                raise ValueError(
                    "gmshModelMeshGetGhostElements returned non-zero error code: ",
                    ierr.value)
            return (
                _ovectorsize(api_elementTags_, api_elementTags_n_.value),
                _ovectorint(api_partitions_, api_partitions_n_.value))

        @staticmethod
        def setSize(dimTags, size):
            """
            gmsh.model.mesh.setSize(dimTags, size)

            Set a mesh size constraint on the model entities `dimTags'. Currently only
            entities of dimension 0 (points) are handled.
            """
            api_dimTags_, api_dimTags_n_ = _ivectorpair(dimTags)
            ierr = c_int()
            lib.gmshModelMeshSetSize(
                api_dimTags_, api_dimTags_n_,
                c_double(size),
                byref(ierr))
            if ierr.value != 0:
                raise ValueError(
                    "gmshModelMeshSetSize returned non-zero error code: ",
                    ierr.value)

        @staticmethod
        def setTransfiniteCurve(tag, numNodes, meshType="Progression", coef=1.):
            """
            gmsh.model.mesh.setTransfiniteCurve(tag, numNodes, meshType="Progression", coef=1.)

            Set a transfinite meshing constraint on the curve `tag', with `numNodes'
            nodes distributed according to `meshType' and `coef'. Currently supported
            types are "Progression" (geometrical progression with power `coef') and
            "Bump" (refinement toward both extremities of the curve).
            """
            ierr = c_int()
            lib.gmshModelMeshSetTransfiniteCurve(
                c_int(tag),
                c_int(numNodes),
                c_char_p(meshType.encode()),
                c_double(coef),
                byref(ierr))
            if ierr.value != 0:
                raise ValueError(
                    "gmshModelMeshSetTransfiniteCurve returned non-zero error code: ",
                    ierr.value)

        @staticmethod
        def setTransfiniteSurface(tag, arrangement="Left", cornerTags=[]):
            """
            gmsh.model.mesh.setTransfiniteSurface(tag, arrangement="Left", cornerTags=[])

            Set a transfinite meshing constraint on the surface `tag'. `arrangement'
            describes the arrangement of the triangles when the surface is not flagged
            as recombined: currently supported values are "Left", "Right",
            "AlternateLeft" and "AlternateRight". `cornerTags' can be used to specify
            the (3 or 4) corners of the transfinite interpolation explicitly;
            specifying the corners explicitly is mandatory if the surface has more that
            3 or 4 points on its boundary.
            """
            api_cornerTags_, api_cornerTags_n_ = _ivectorint(cornerTags)
            ierr = c_int()
            lib.gmshModelMeshSetTransfiniteSurface(
                c_int(tag),
                c_char_p(arrangement.encode()),
                api_cornerTags_, api_cornerTags_n_,
                byref(ierr))
            if ierr.value != 0:
                raise ValueError(
                    "gmshModelMeshSetTransfiniteSurface returned non-zero error code: ",
                    ierr.value)

        @staticmethod
        def setTransfiniteVolume(tag, cornerTags=[]):
            """
            gmsh.model.mesh.setTransfiniteVolume(tag, cornerTags=[])

            Set a transfinite meshing constraint on the surface `tag'. `cornerTags' can
            be used to specify the (6 or 8) corners of the transfinite interpolation
            explicitly.
            """
            api_cornerTags_, api_cornerTags_n_ = _ivectorint(cornerTags)
            ierr = c_int()
            lib.gmshModelMeshSetTransfiniteVolume(
                c_int(tag),
                api_cornerTags_, api_cornerTags_n_,
                byref(ierr))
            if ierr.value != 0:
                raise ValueError(
                    "gmshModelMeshSetTransfiniteVolume returned non-zero error code: ",
                    ierr.value)

        @staticmethod
        def setRecombine(dim, tag):
            """
            gmsh.model.mesh.setRecombine(dim, tag)

            Set a recombination meshing constraint on the model entity of dimension
            `dim' and tag `tag'. Currently only entities of dimension 2 (to recombine
            triangles into quadrangles) are supported.
            """
            ierr = c_int()
            lib.gmshModelMeshSetRecombine(
                c_int(dim),
                c_int(tag),
                byref(ierr))
            if ierr.value != 0:
                raise ValueError(
                    "gmshModelMeshSetRecombine returned non-zero error code: ",
                    ierr.value)

        @staticmethod
        def setSmoothing(dim, tag, val):
            """
            gmsh.model.mesh.setSmoothing(dim, tag, val)

            Set a smoothing meshing constraint on the model entity of dimension `dim'
            and tag `tag'. `val' iterations of a Laplace smoother are applied.
            """
            ierr = c_int()
            lib.gmshModelMeshSetSmoothing(
                c_int(dim),
                c_int(tag),
                c_int(val),
                byref(ierr))
            if ierr.value != 0:
                raise ValueError(
                    "gmshModelMeshSetSmoothing returned non-zero error code: ",
                    ierr.value)

        @staticmethod
        def setReverse(dim, tag, val=True):
            """
            gmsh.model.mesh.setReverse(dim, tag, val=True)

            Set a reverse meshing constraint on the model entity of dimension `dim' and
            tag `tag'. If `val' is true, the mesh orientation will be reversed with
            respect to the natural mesh orientation (i.e. the orientation consistent
            with the orientation of the geometry). If `val' is false, the mesh is left
            as-is.
            """
            ierr = c_int()
            lib.gmshModelMeshSetReverse(
                c_int(dim),
                c_int(tag),
                c_int(bool(val)),
                byref(ierr))
            if ierr.value != 0:
                raise ValueError(
                    "gmshModelMeshSetReverse returned non-zero error code: ",
                    ierr.value)

        @staticmethod
        def setAlgorithm(dim, tag, val):
            """
            gmsh.model.mesh.setAlgorithm(dim, tag, val)

            Set the meshing algorithm on the model entity of dimension `dim' and tag
            `tag'. Currently only supported for `dim' == 2.
            """
            ierr = c_int()
            lib.gmshModelMeshSetAlgorithm(
                c_int(dim),
                c_int(tag),
                c_int(val),
                byref(ierr))
            if ierr.value != 0:
                raise ValueError(
                    "gmshModelMeshSetAlgorithm returned non-zero error code: ",
                    ierr.value)

        @staticmethod
        def setSizeFromBoundary(dim, tag, val):
            """
            gmsh.model.mesh.setSizeFromBoundary(dim, tag, val)

            Force the mesh size to be extended from the boundary, or not, for the model
            entity of dimension `dim' and tag `tag'. Currently only supported for `dim'
            == 2.
            """
            ierr = c_int()
            lib.gmshModelMeshSetSizeFromBoundary(
                c_int(dim),
                c_int(tag),
                c_int(val),
                byref(ierr))
            if ierr.value != 0:
                raise ValueError(
                    "gmshModelMeshSetSizeFromBoundary returned non-zero error code: ",
                    ierr.value)

        @staticmethod
        def setCompound(dim, tags):
            """
            gmsh.model.mesh.setCompound(dim, tags)

            Set a compound meshing constraint on the model entities of dimension `dim'
            and tags `tags'. During meshing, compound entities are treated as a single
            discrete entity, which is automatically reparametrized.
            """
            api_tags_, api_tags_n_ = _ivectorint(tags)
            ierr = c_int()
            lib.gmshModelMeshSetCompound(
                c_int(dim),
                api_tags_, api_tags_n_,
                byref(ierr))
            if ierr.value != 0:
                raise ValueError(
                    "gmshModelMeshSetCompound returned non-zero error code: ",
                    ierr.value)

        @staticmethod
        def setOutwardOrientation(tag):
            """
            gmsh.model.mesh.setOutwardOrientation(tag)

            Set meshing constraints on the bounding surfaces of the volume of tag `tag'
            so that all surfaces are oriented with outward pointing normals. Currently
            only available with the OpenCASCADE kernel, as it relies on the STL
            triangulation.
            """
            ierr = c_int()
            lib.gmshModelMeshSetOutwardOrientation(
                c_int(tag),
                byref(ierr))
            if ierr.value != 0:
                raise ValueError(
                    "gmshModelMeshSetOutwardOrientation returned non-zero error code: ",
                    ierr.value)

        @staticmethod
        def embed(dim, tags, inDim, inTag):
            """
            gmsh.model.mesh.embed(dim, tags, inDim, inTag)

            Embed the model entities of dimension `dim' and tags `tags' in the
            (`inDim', `inTag') model entity. The dimension `dim' can 0, 1 or 2 and must
            be strictly smaller than `inDim', which must be either 2 or 3. The embedded
            entities should not be part of the boundary of the entity `inTag', whose
            mesh will conform to the mesh of the embedded entities.
            """
            api_tags_, api_tags_n_ = _ivectorint(tags)
            ierr = c_int()
            lib.gmshModelMeshEmbed(
                c_int(dim),
                api_tags_, api_tags_n_,
                c_int(inDim),
                c_int(inTag),
                byref(ierr))
            if ierr.value != 0:
                raise ValueError(
                    "gmshModelMeshEmbed returned non-zero error code: ",
                    ierr.value)

        @staticmethod
        def removeEmbedded(dimTags, dim=-1):
            """
            gmsh.model.mesh.removeEmbedded(dimTags, dim=-1)

            Remove embedded entities from the model entities `dimTags'. if `dim' is >=
            0, only remove embedded entities of the given dimension (e.g. embedded
            points if `dim' == 0).
            """
            api_dimTags_, api_dimTags_n_ = _ivectorpair(dimTags)
            ierr = c_int()
            lib.gmshModelMeshRemoveEmbedded(
                api_dimTags_, api_dimTags_n_,
                c_int(dim),
                byref(ierr))
            if ierr.value != 0:
                raise ValueError(
                    "gmshModelMeshRemoveEmbedded returned non-zero error code: ",
                    ierr.value)

        @staticmethod
        def reorderElements(elementType, tag, ordering):
            """
            gmsh.model.mesh.reorderElements(elementType, tag, ordering)

            Reorder the elements of type `elementType' classified on the entity of tag
            `tag' according to `ordering'.
            """
            api_ordering_, api_ordering_n_ = _ivectorsize(ordering)
            ierr = c_int()
            lib.gmshModelMeshReorderElements(
                c_int(elementType),
                c_int(tag),
                api_ordering_, api_ordering_n_,
                byref(ierr))
            if ierr.value != 0:
                raise ValueError(
                    "gmshModelMeshReorderElements returned non-zero error code: ",
                    ierr.value)

        @staticmethod
        def renumberNodes():
            """
            gmsh.model.mesh.renumberNodes()

            Renumber the node tags in a continuous sequence.
            """
            ierr = c_int()
            lib.gmshModelMeshRenumberNodes(
                byref(ierr))
            if ierr.value != 0:
                raise ValueError(
                    "gmshModelMeshRenumberNodes returned non-zero error code: ",
                    ierr.value)

        @staticmethod
        def renumberElements():
            """
            gmsh.model.mesh.renumberElements()

            Renumber the element tags in a continuous sequence.
            """
            ierr = c_int()
            lib.gmshModelMeshRenumberElements(
                byref(ierr))
            if ierr.value != 0:
                raise ValueError(
                    "gmshModelMeshRenumberElements returned non-zero error code: ",
                    ierr.value)

        @staticmethod
        def setPeriodic(dim, tags, tagsMaster, affineTransform):
            """
            gmsh.model.mesh.setPeriodic(dim, tags, tagsMaster, affineTransform)

            Set the meshes of the entities of dimension `dim' and tag `tags' as
            periodic copies of the meshes of entities `tagsMaster', using the affine
            transformation specified in `affineTransformation' (16 entries of a 4x4
            matrix, by row). If used after meshing, generate the periodic node
            correspondence information assuming the meshes of entities `tags'
            effectively match the meshes of entities `tagsMaster' (useful for
            structured and extruded meshes). Currently only available for @code{dim} ==
            1 and @code{dim} == 2.
            """
            api_tags_, api_tags_n_ = _ivectorint(tags)
            api_tagsMaster_, api_tagsMaster_n_ = _ivectorint(tagsMaster)
            api_affineTransform_, api_affineTransform_n_ = _ivectordouble(affineTransform)
            ierr = c_int()
            lib.gmshModelMeshSetPeriodic(
                c_int(dim),
                api_tags_, api_tags_n_,
                api_tagsMaster_, api_tagsMaster_n_,
                api_affineTransform_, api_affineTransform_n_,
                byref(ierr))
            if ierr.value != 0:
                raise ValueError(
                    "gmshModelMeshSetPeriodic returned non-zero error code: ",
                    ierr.value)

        @staticmethod
        def getPeriodicNodes(dim, tag):
            """
            gmsh.model.mesh.getPeriodicNodes(dim, tag)

            Get the master entity `tagMaster', the node tags `nodeTags' and their
            corresponding master node tags `nodeTagsMaster', and the affine transform
            `affineTransform' for the entity of dimension `dim' and tag `tag'.

            Return `tagMaster', `nodeTags', `nodeTagsMaster', `affineTransform'.
            """
            api_tagMaster_ = c_int()
            api_nodeTags_, api_nodeTags_n_ = POINTER(c_size_t)(), c_size_t()
            api_nodeTagsMaster_, api_nodeTagsMaster_n_ = POINTER(c_size_t)(), c_size_t()
            api_affineTransform_, api_affineTransform_n_ = POINTER(c_double)(), c_size_t()
            ierr = c_int()
            lib.gmshModelMeshGetPeriodicNodes(
                c_int(dim),
                c_int(tag),
                byref(api_tagMaster_),
                byref(api_nodeTags_), byref(api_nodeTags_n_),
                byref(api_nodeTagsMaster_), byref(api_nodeTagsMaster_n_),
                byref(api_affineTransform_), byref(api_affineTransform_n_),
                byref(ierr))
            if ierr.value != 0:
                raise ValueError(
                    "gmshModelMeshGetPeriodicNodes returned non-zero error code: ",
                    ierr.value)
            return (
                api_tagMaster_.value,
                _ovectorsize(api_nodeTags_, api_nodeTags_n_.value),
                _ovectorsize(api_nodeTagsMaster_, api_nodeTagsMaster_n_.value),
                _ovectordouble(api_affineTransform_, api_affineTransform_n_.value))

        @staticmethod
        def removeDuplicateNodes():
            """
            gmsh.model.mesh.removeDuplicateNodes()

            Remove duplicate nodes in the mesh of the current model.
            """
            ierr = c_int()
            lib.gmshModelMeshRemoveDuplicateNodes(
                byref(ierr))
            if ierr.value != 0:
                raise ValueError(
                    "gmshModelMeshRemoveDuplicateNodes returned non-zero error code: ",
                    ierr.value)

        @staticmethod
        def splitQuadrangles(quality=1., tag=-1):
            """
            gmsh.model.mesh.splitQuadrangles(quality=1., tag=-1)

            Split (into two triangles) all quadrangles in surface `tag' whose quality
            is lower than `quality'. If `tag' < 0, split quadrangles in all surfaces.
            """
            ierr = c_int()
            lib.gmshModelMeshSplitQuadrangles(
                c_double(quality),
                c_int(tag),
                byref(ierr))
            if ierr.value != 0:
                raise ValueError(
                    "gmshModelMeshSplitQuadrangles returned non-zero error code: ",
                    ierr.value)

        @staticmethod
        def classifySurfaces(angle, boundary=True, forReparametrization=False, curveAngle=pi):
            """
            gmsh.model.mesh.classifySurfaces(angle, boundary=True, forReparametrization=False, curveAngle=pi)

            Classify ("color") the surface mesh based on the angle threshold `angle'
            (in radians), and create new discrete surfaces, curves and points
            accordingly. If `boundary' is set, also create discrete curves on the
            boundary if the surface is open. If `forReparametrization' is set, create
            edges and surfaces that can be reparametrized using a single map. If
            `curveAngle' is less than Pi, also force curves to be split according to
            `curveAngle'.
            """
            ierr = c_int()
            lib.gmshModelMeshClassifySurfaces(
                c_double(angle),
                c_int(bool(boundary)),
                c_int(bool(forReparametrization)),
                c_double(curveAngle),
                byref(ierr))
            if ierr.value != 0:
                raise ValueError(
                    "gmshModelMeshClassifySurfaces returned non-zero error code: ",
                    ierr.value)

        @staticmethod
        def createGeometry():
            """
            gmsh.model.mesh.createGeometry()

            Create a parametrization for discrete curves and surfaces (i.e. curves and
            surfaces represented solely by a mesh, without an underlying CAD
            description), assuming that each can be parametrized with a single map.
            """
            ierr = c_int()
            lib.gmshModelMeshCreateGeometry(
                byref(ierr))
            if ierr.value != 0:
                raise ValueError(
                    "gmshModelMeshCreateGeometry returned non-zero error code: ",
                    ierr.value)

        @staticmethod
        def createTopology():
            """
            gmsh.model.mesh.createTopology()

            Create a boundary representation from the mesh if the model does not have
            one (e.g. when imported from mesh file formats with no BRep representation
            of the underlying model).
            """
            ierr = c_int()
            lib.gmshModelMeshCreateTopology(
                byref(ierr))
            if ierr.value != 0:
                raise ValueError(
                    "gmshModelMeshCreateTopology returned non-zero error code: ",
                    ierr.value)

        @staticmethod
        def computeHomology(domainTags=[], subdomainTags=[], dims=[]):
            """
            gmsh.model.mesh.computeHomology(domainTags=[], subdomainTags=[], dims=[])

            Compute a basis representation for homology spaces after a mesh has been
            generated. The computation domain is given in a list of physical group tags
            `domainTags'; if empty, the whole mesh is the domain. The computation
            subdomain for relative homology computation is given in a list of physical
            group tags `subdomainTags'; if empty, absolute homology is computed. The
            dimensions homology bases to be computed are given in the list `dim'; if
            empty, all bases are computed. Resulting basis representation chains are
            stored as physical groups in the mesh.
            """
            api_domainTags_, api_domainTags_n_ = _ivectorint(domainTags)
            api_subdomainTags_, api_subdomainTags_n_ = _ivectorint(subdomainTags)
            api_dims_, api_dims_n_ = _ivectorint(dims)
            ierr = c_int()
            lib.gmshModelMeshComputeHomology(
                api_domainTags_, api_domainTags_n_,
                api_subdomainTags_, api_subdomainTags_n_,
                api_dims_, api_dims_n_,
                byref(ierr))
            if ierr.value != 0:
                raise ValueError(
                    "gmshModelMeshComputeHomology returned non-zero error code: ",
                    ierr.value)

        @staticmethod
        def computeCohomology(domainTags=[], subdomainTags=[], dims=[]):
            """
            gmsh.model.mesh.computeCohomology(domainTags=[], subdomainTags=[], dims=[])

            Compute a basis representation for cohomology spaces after a mesh has been
            generated. The computation domain is given in a list of physical group tags
            `domainTags'; if empty, the whole mesh is the domain. The computation
            subdomain for relative cohomology computation is given in a list of
            physical group tags `subdomainTags'; if empty, absolute cohomology is
            computed. The dimensions homology bases to be computed are given in the
            list `dim'; if empty, all bases are computed. Resulting basis
            representation cochains are stored as physical groups in the mesh.
            """
            api_domainTags_, api_domainTags_n_ = _ivectorint(domainTags)
            api_subdomainTags_, api_subdomainTags_n_ = _ivectorint(subdomainTags)
            api_dims_, api_dims_n_ = _ivectorint(dims)
            ierr = c_int()
            lib.gmshModelMeshComputeCohomology(
                api_domainTags_, api_domainTags_n_,
                api_subdomainTags_, api_subdomainTags_n_,
                api_dims_, api_dims_n_,
                byref(ierr))
            if ierr.value != 0:
                raise ValueError(
                    "gmshModelMeshComputeCohomology returned non-zero error code: ",
                    ierr.value)

        @staticmethod
        def computeCrossField():
            """
            gmsh.model.mesh.computeCrossField()

            Compute a cross field for the current mesh. The function creates 3 views:
            the H function, the Theta function and cross directions. Return the tags of
            the views

            Return `viewTags'.
            """
            api_viewTags_, api_viewTags_n_ = POINTER(c_int)(), c_size_t()
            ierr = c_int()
            lib.gmshModelMeshComputeCrossField(
                byref(api_viewTags_), byref(api_viewTags_n_),
                byref(ierr))
            if ierr.value != 0:
                raise ValueError(
                    "gmshModelMeshComputeCrossField returned non-zero error code: ",
                    ierr.value)
            return _ovectorint(api_viewTags_, api_viewTags_n_.value)


        class field:
            """
            Mesh size field functions
            """

            @staticmethod
            def add(fieldType, tag=-1):
                """
                gmsh.model.mesh.field.add(fieldType, tag=-1)

                Add a new mesh size field of type `fieldType'. If `tag' is positive, assign
                the tag explicitly; otherwise a new tag is assigned automatically. Return
                the field tag.

                Return an integer value.
                """
                ierr = c_int()
                api__result__ = lib.gmshModelMeshFieldAdd(
                    c_char_p(fieldType.encode()),
                    c_int(tag),
                    byref(ierr))
                if ierr.value != 0:
                    raise ValueError(
                        "gmshModelMeshFieldAdd returned non-zero error code: ",
                        ierr.value)
                return api__result__

            @staticmethod
            def remove(tag):
                """
                gmsh.model.mesh.field.remove(tag)

                Remove the field with tag `tag'.
                """
                ierr = c_int()
                lib.gmshModelMeshFieldRemove(
                    c_int(tag),
                    byref(ierr))
                if ierr.value != 0:
                    raise ValueError(
                        "gmshModelMeshFieldRemove returned non-zero error code: ",
                        ierr.value)

            @staticmethod
            def setNumber(tag, option, value):
                """
                gmsh.model.mesh.field.setNumber(tag, option, value)

                Set the numerical option `option' to value `value' for field `tag'.
                """
                ierr = c_int()
                lib.gmshModelMeshFieldSetNumber(
                    c_int(tag),
                    c_char_p(option.encode()),
                    c_double(value),
                    byref(ierr))
                if ierr.value != 0:
                    raise ValueError(
                        "gmshModelMeshFieldSetNumber returned non-zero error code: ",
                        ierr.value)

            @staticmethod
            def setString(tag, option, value):
                """
                gmsh.model.mesh.field.setString(tag, option, value)

                Set the string option `option' to value `value' for field `tag'.
                """
                ierr = c_int()
                lib.gmshModelMeshFieldSetString(
                    c_int(tag),
                    c_char_p(option.encode()),
                    c_char_p(value.encode()),
                    byref(ierr))
                if ierr.value != 0:
                    raise ValueError(
                        "gmshModelMeshFieldSetString returned non-zero error code: ",
                        ierr.value)

            @staticmethod
            def setNumbers(tag, option, value):
                """
                gmsh.model.mesh.field.setNumbers(tag, option, value)

                Set the numerical list option `option' to value `value' for field `tag'.
                """
                api_value_, api_value_n_ = _ivectordouble(value)
                ierr = c_int()
                lib.gmshModelMeshFieldSetNumbers(
                    c_int(tag),
                    c_char_p(option.encode()),
                    api_value_, api_value_n_,
                    byref(ierr))
                if ierr.value != 0:
                    raise ValueError(
                        "gmshModelMeshFieldSetNumbers returned non-zero error code: ",
                        ierr.value)

            @staticmethod
            def setAsBackgroundMesh(tag):
                """
                gmsh.model.mesh.field.setAsBackgroundMesh(tag)

                Set the field `tag' as the background mesh size field.
                """
                ierr = c_int()
                lib.gmshModelMeshFieldSetAsBackgroundMesh(
                    c_int(tag),
                    byref(ierr))
                if ierr.value != 0:
                    raise ValueError(
                        "gmshModelMeshFieldSetAsBackgroundMesh returned non-zero error code: ",
                        ierr.value)

            @staticmethod
            def setAsBoundaryLayer(tag):
                """
                gmsh.model.mesh.field.setAsBoundaryLayer(tag)

                Set the field `tag' as a boundary layer size field.
                """
                ierr = c_int()
                lib.gmshModelMeshFieldSetAsBoundaryLayer(
                    c_int(tag),
                    byref(ierr))
                if ierr.value != 0:
                    raise ValueError(
                        "gmshModelMeshFieldSetAsBoundaryLayer returned non-zero error code: ",
                        ierr.value)


    class geo:
        """
        Built-in CAD kernel functions
        """

        @staticmethod
        def addPoint(x, y, z, meshSize=0., tag=-1):
            """
            gmsh.model.geo.addPoint(x, y, z, meshSize=0., tag=-1)

            Add a geometrical point in the built-in CAD representation, at coordinates
            (`x', `y', `z'). If `meshSize' is > 0, add a meshing constraint at that
            point. If `tag' is positive, set the tag explicitly; otherwise a new tag is
            selected automatically. Return the tag of the point. (Note that the point
            will be added in the current model only after `synchronize' is called. This
            behavior holds for all the entities added in the geo module.)

            Return an integer value.
            """
            ierr = c_int()
            api__result__ = lib.gmshModelGeoAddPoint(
                c_double(x),
                c_double(y),
                c_double(z),
                c_double(meshSize),
                c_int(tag),
                byref(ierr))
            if ierr.value != 0:
                raise ValueError(
                    "gmshModelGeoAddPoint returned non-zero error code: ",
                    ierr.value)
            return api__result__

        @staticmethod
        def addLine(startTag, endTag, tag=-1):
            """
            gmsh.model.geo.addLine(startTag, endTag, tag=-1)

            Add a straight line segment between the two points with tags `startTag' and
            `endTag'. If `tag' is positive, set the tag explicitly; otherwise a new tag
            is selected automatically. Return the tag of the line.

            Return an integer value.
            """
            ierr = c_int()
            api__result__ = lib.gmshModelGeoAddLine(
                c_int(startTag),
                c_int(endTag),
                c_int(tag),
                byref(ierr))
            if ierr.value != 0:
                raise ValueError(
                    "gmshModelGeoAddLine returned non-zero error code: ",
                    ierr.value)
            return api__result__

        @staticmethod
        def addCircleArc(startTag, centerTag, endTag, tag=-1, nx=0., ny=0., nz=0.):
            """
            gmsh.model.geo.addCircleArc(startTag, centerTag, endTag, tag=-1, nx=0., ny=0., nz=0.)

            Add a circle arc (strictly smaller than Pi) between the two points with
            tags `startTag' and `endTag', with center `centertag'. If `tag' is
            positive, set the tag explicitly; otherwise a new tag is selected
            automatically. If (`nx', `ny', `nz') != (0, 0, 0), explicitly set the plane
            of the circle arc. Return the tag of the circle arc.

            Return an integer value.
            """
            ierr = c_int()
            api__result__ = lib.gmshModelGeoAddCircleArc(
                c_int(startTag),
                c_int(centerTag),
                c_int(endTag),
                c_int(tag),
                c_double(nx),
                c_double(ny),
                c_double(nz),
                byref(ierr))
            if ierr.value != 0:
                raise ValueError(
                    "gmshModelGeoAddCircleArc returned non-zero error code: ",
                    ierr.value)
            return api__result__

        @staticmethod
        def addEllipseArc(startTag, centerTag, majorTag, endTag, tag=-1, nx=0., ny=0., nz=0.):
            """
            gmsh.model.geo.addEllipseArc(startTag, centerTag, majorTag, endTag, tag=-1, nx=0., ny=0., nz=0.)

            Add an ellipse arc (strictly smaller than Pi) between the two points
            `startTag' and `endTag', with center `centerTag' and major axis point
            `majorTag'. If `tag' is positive, set the tag explicitly; otherwise a new
            tag is selected automatically. If (`nx', `ny', `nz') != (0, 0, 0),
            explicitly set the plane of the circle arc. Return the tag of the ellipse
            arc.

            Return an integer value.
            """
            ierr = c_int()
            api__result__ = lib.gmshModelGeoAddEllipseArc(
                c_int(startTag),
                c_int(centerTag),
                c_int(majorTag),
                c_int(endTag),
                c_int(tag),
                c_double(nx),
                c_double(ny),
                c_double(nz),
                byref(ierr))
            if ierr.value != 0:
                raise ValueError(
                    "gmshModelGeoAddEllipseArc returned non-zero error code: ",
                    ierr.value)
            return api__result__

        @staticmethod
        def addSpline(pointTags, tag=-1):
            """
            gmsh.model.geo.addSpline(pointTags, tag=-1)

            Add a spline (Catmull-Rom) curve going through the points `pointTags'. If
            `tag' is positive, set the tag explicitly; otherwise a new tag is selected
            automatically. Create a periodic curve if the first and last points are the
            same. Return the tag of the spline curve.

            Return an integer value.
            """
            api_pointTags_, api_pointTags_n_ = _ivectorint(pointTags)
            ierr = c_int()
            api__result__ = lib.gmshModelGeoAddSpline(
                api_pointTags_, api_pointTags_n_,
                c_int(tag),
                byref(ierr))
            if ierr.value != 0:
                raise ValueError(
                    "gmshModelGeoAddSpline returned non-zero error code: ",
                    ierr.value)
            return api__result__

        @staticmethod
        def addBSpline(pointTags, tag=-1):
            """
            gmsh.model.geo.addBSpline(pointTags, tag=-1)

            Add a cubic b-spline curve with `pointTags' control points. If `tag' is
            positive, set the tag explicitly; otherwise a new tag is selected
            automatically. Creates a periodic curve if the first and last points are
            the same. Return the tag of the b-spline curve.

            Return an integer value.
            """
            api_pointTags_, api_pointTags_n_ = _ivectorint(pointTags)
            ierr = c_int()
            api__result__ = lib.gmshModelGeoAddBSpline(
                api_pointTags_, api_pointTags_n_,
                c_int(tag),
                byref(ierr))
            if ierr.value != 0:
                raise ValueError(
                    "gmshModelGeoAddBSpline returned non-zero error code: ",
                    ierr.value)
            return api__result__

        @staticmethod
        def addBezier(pointTags, tag=-1):
            """
            gmsh.model.geo.addBezier(pointTags, tag=-1)

            Add a Bezier curve with `pointTags' control points. If `tag' is positive,
            set the tag explicitly; otherwise a new tag is selected automatically.
            Return the tag of the Bezier curve.

            Return an integer value.
            """
            api_pointTags_, api_pointTags_n_ = _ivectorint(pointTags)
            ierr = c_int()
            api__result__ = lib.gmshModelGeoAddBezier(
                api_pointTags_, api_pointTags_n_,
                c_int(tag),
                byref(ierr))
            if ierr.value != 0:
                raise ValueError(
                    "gmshModelGeoAddBezier returned non-zero error code: ",
                    ierr.value)
            return api__result__

        @staticmethod
        def addCompoundSpline(curveTags, numIntervals=5, tag=-1):
            """
            gmsh.model.geo.addCompoundSpline(curveTags, numIntervals=5, tag=-1)

            Add a spline (Catmull-Rom) going through points sampling the curves in
            `curveTags'. The density of sampling points on each curve is governed by
            `numIntervals'. If `tag' is positive, set the tag explicitly; otherwise a
            new tag is selected automatically. Return the tag of the spline.

            Return an integer value.
            """
            api_curveTags_, api_curveTags_n_ = _ivectorint(curveTags)
            ierr = c_int()
            api__result__ = lib.gmshModelGeoAddCompoundSpline(
                api_curveTags_, api_curveTags_n_,
                c_int(numIntervals),
                c_int(tag),
                byref(ierr))
            if ierr.value != 0:
                raise ValueError(
                    "gmshModelGeoAddCompoundSpline returned non-zero error code: ",
                    ierr.value)
            return api__result__

        @staticmethod
        def addCompoundBSpline(curveTags, numIntervals=20, tag=-1):
            """
            gmsh.model.geo.addCompoundBSpline(curveTags, numIntervals=20, tag=-1)

            Add a b-spline with control points sampling the curves in `curveTags'. The
            density of sampling points on each curve is governed by `numIntervals'. If
            `tag' is positive, set the tag explicitly; otherwise a new tag is selected
            automatically. Return the tag of the b-spline.

            Return an integer value.
            """
            api_curveTags_, api_curveTags_n_ = _ivectorint(curveTags)
            ierr = c_int()
            api__result__ = lib.gmshModelGeoAddCompoundBSpline(
                api_curveTags_, api_curveTags_n_,
                c_int(numIntervals),
                c_int(tag),
                byref(ierr))
            if ierr.value != 0:
                raise ValueError(
                    "gmshModelGeoAddCompoundBSpline returned non-zero error code: ",
                    ierr.value)
            return api__result__

        @staticmethod
        def addCurveLoop(curveTags, tag=-1):
            """
            gmsh.model.geo.addCurveLoop(curveTags, tag=-1)

            Add a curve loop (a closed wire) formed by the curves `curveTags'.
            `curveTags' should contain (signed) tags of model enties of dimension 1
            forming a closed loop: a negative tag signifies that the underlying curve
            is considered with reversed orientation. If `tag' is positive, set the tag
            explicitly; otherwise a new tag is selected automatically. Return the tag
            of the curve loop.

            Return an integer value.
            """
            api_curveTags_, api_curveTags_n_ = _ivectorint(curveTags)
            ierr = c_int()
            api__result__ = lib.gmshModelGeoAddCurveLoop(
                api_curveTags_, api_curveTags_n_,
                c_int(tag),
                byref(ierr))
            if ierr.value != 0:
                raise ValueError(
                    "gmshModelGeoAddCurveLoop returned non-zero error code: ",
                    ierr.value)
            return api__result__

        @staticmethod
        def addPlaneSurface(wireTags, tag=-1):
            """
            gmsh.model.geo.addPlaneSurface(wireTags, tag=-1)

            Add a plane surface defined by one or more curve loops `wireTags'. The
            first curve loop defines the exterior contour; additional curve loop define
            holes. If `tag' is positive, set the tag explicitly; otherwise a new tag is
            selected automatically. Return the tag of the surface.

            Return an integer value.
            """
            api_wireTags_, api_wireTags_n_ = _ivectorint(wireTags)
            ierr = c_int()
            api__result__ = lib.gmshModelGeoAddPlaneSurface(
                api_wireTags_, api_wireTags_n_,
                c_int(tag),
                byref(ierr))
            if ierr.value != 0:
                raise ValueError(
                    "gmshModelGeoAddPlaneSurface returned non-zero error code: ",
                    ierr.value)
            return api__result__

        @staticmethod
        def addSurfaceFilling(wireTags, tag=-1, sphereCenterTag=-1):
            """
            gmsh.model.geo.addSurfaceFilling(wireTags, tag=-1, sphereCenterTag=-1)

            Add a surface filling the curve loops in `wireTags'. Currently only a
            single curve loop is supported; this curve loop should be composed by 3 or
            4 curves only. If `tag' is positive, set the tag explicitly; otherwise a
            new tag is selected automatically. Return the tag of the surface.

            Return an integer value.
            """
            api_wireTags_, api_wireTags_n_ = _ivectorint(wireTags)
            ierr = c_int()
            api__result__ = lib.gmshModelGeoAddSurfaceFilling(
                api_wireTags_, api_wireTags_n_,
                c_int(tag),
                c_int(sphereCenterTag),
                byref(ierr))
            if ierr.value != 0:
                raise ValueError(
                    "gmshModelGeoAddSurfaceFilling returned non-zero error code: ",
                    ierr.value)
            return api__result__

        @staticmethod
        def addSurfaceLoop(surfaceTags, tag=-1):
            """
            gmsh.model.geo.addSurfaceLoop(surfaceTags, tag=-1)

            Add a surface loop (a closed shell) formed by `surfaceTags'.  If `tag' is
            positive, set the tag explicitly; otherwise a new tag is selected
            automatically. Return the tag of the shell.

            Return an integer value.
            """
            api_surfaceTags_, api_surfaceTags_n_ = _ivectorint(surfaceTags)
            ierr = c_int()
            api__result__ = lib.gmshModelGeoAddSurfaceLoop(
                api_surfaceTags_, api_surfaceTags_n_,
                c_int(tag),
                byref(ierr))
            if ierr.value != 0:
                raise ValueError(
                    "gmshModelGeoAddSurfaceLoop returned non-zero error code: ",
                    ierr.value)
            return api__result__

        @staticmethod
        def addVolume(shellTags, tag=-1):
            """
            gmsh.model.geo.addVolume(shellTags, tag=-1)

            Add a volume (a region) defined by one or more shells `shellTags'. The
            first surface loop defines the exterior boundary; additional surface loop
            define holes. If `tag' is positive, set the tag explicitly; otherwise a new
            tag is selected automatically. Return the tag of the volume.

            Return an integer value.
            """
            api_shellTags_, api_shellTags_n_ = _ivectorint(shellTags)
            ierr = c_int()
            api__result__ = lib.gmshModelGeoAddVolume(
                api_shellTags_, api_shellTags_n_,
                c_int(tag),
                byref(ierr))
            if ierr.value != 0:
                raise ValueError(
                    "gmshModelGeoAddVolume returned non-zero error code: ",
                    ierr.value)
            return api__result__

        @staticmethod
        def extrude(dimTags, dx, dy, dz, numElements=[], heights=[], recombine=False):
            """
            gmsh.model.geo.extrude(dimTags, dx, dy, dz, numElements=[], heights=[], recombine=False)

            Extrude the model entities `dimTags' by translation along (`dx', `dy',
            `dz'). Return extruded entities in `outDimTags'. If `numElements' is not
            empty, also extrude the mesh: the entries in `numElements' give the number
            of elements in each layer. If `height' is not empty, it provides the
            (cumulative) height of the different layers, normalized to 1. If `dx' ==
            `dy' == `dz' == 0, the entities are extruded along their normal.

            Return `outDimTags'.
            """
            api_dimTags_, api_dimTags_n_ = _ivectorpair(dimTags)
            api_outDimTags_, api_outDimTags_n_ = POINTER(c_int)(), c_size_t()
            api_numElements_, api_numElements_n_ = _ivectorint(numElements)
            api_heights_, api_heights_n_ = _ivectordouble(heights)
            ierr = c_int()
            lib.gmshModelGeoExtrude(
                api_dimTags_, api_dimTags_n_,
                c_double(dx),
                c_double(dy),
                c_double(dz),
                byref(api_outDimTags_), byref(api_outDimTags_n_),
                api_numElements_, api_numElements_n_,
                api_heights_, api_heights_n_,
                c_int(bool(recombine)),
                byref(ierr))
            if ierr.value != 0:
                raise ValueError(
                    "gmshModelGeoExtrude returned non-zero error code: ",
                    ierr.value)
            return _ovectorpair(api_outDimTags_, api_outDimTags_n_.value)

        @staticmethod
        def revolve(dimTags, x, y, z, ax, ay, az, angle, numElements=[], heights=[], recombine=False):
            """
            gmsh.model.geo.revolve(dimTags, x, y, z, ax, ay, az, angle, numElements=[], heights=[], recombine=False)

            Extrude the model entities `dimTags' by rotation of `angle' radians around
            the axis of revolution defined by the point (`x', `y', `z') and the
            direction (`ax', `ay', `az'). The angle should be strictly smaller than Pi.
            Return extruded entities in `outDimTags'. If `numElements' is not empty,
            also extrude the mesh: the entries in `numElements' give the number of
            elements in each layer. If `height' is not empty, it provides the
            (cumulative) height of the different layers, normalized to 1.

            Return `outDimTags'.
            """
            api_dimTags_, api_dimTags_n_ = _ivectorpair(dimTags)
            api_outDimTags_, api_outDimTags_n_ = POINTER(c_int)(), c_size_t()
            api_numElements_, api_numElements_n_ = _ivectorint(numElements)
            api_heights_, api_heights_n_ = _ivectordouble(heights)
            ierr = c_int()
            lib.gmshModelGeoRevolve(
                api_dimTags_, api_dimTags_n_,
                c_double(x),
                c_double(y),
                c_double(z),
                c_double(ax),
                c_double(ay),
                c_double(az),
                c_double(angle),
                byref(api_outDimTags_), byref(api_outDimTags_n_),
                api_numElements_, api_numElements_n_,
                api_heights_, api_heights_n_,
                c_int(bool(recombine)),
                byref(ierr))
            if ierr.value != 0:
                raise ValueError(
                    "gmshModelGeoRevolve returned non-zero error code: ",
                    ierr.value)
            return _ovectorpair(api_outDimTags_, api_outDimTags_n_.value)

        @staticmethod
        def twist(dimTags, x, y, z, dx, dy, dz, ax, ay, az, angle, numElements=[], heights=[], recombine=False):
            """
            gmsh.model.geo.twist(dimTags, x, y, z, dx, dy, dz, ax, ay, az, angle, numElements=[], heights=[], recombine=False)

            Extrude the model entities `dimTags' by a combined translation and rotation
            of `angle' radians, along (`dx', `dy', `dz') and around the axis of
            revolution defined by the point (`x', `y', `z') and the direction (`ax',
            `ay', `az'). The angle should be strictly smaller than Pi. Return extruded
            entities in `outDimTags'. If `numElements' is not empty, also extrude the
            mesh: the entries in `numElements' give the number of elements in each
            layer. If `height' is not empty, it provides the (cumulative) height of the
            different layers, normalized to 1.

            Return `outDimTags'.
            """
            api_dimTags_, api_dimTags_n_ = _ivectorpair(dimTags)
            api_outDimTags_, api_outDimTags_n_ = POINTER(c_int)(), c_size_t()
            api_numElements_, api_numElements_n_ = _ivectorint(numElements)
            api_heights_, api_heights_n_ = _ivectordouble(heights)
            ierr = c_int()
            lib.gmshModelGeoTwist(
                api_dimTags_, api_dimTags_n_,
                c_double(x),
                c_double(y),
                c_double(z),
                c_double(dx),
                c_double(dy),
                c_double(dz),
                c_double(ax),
                c_double(ay),
                c_double(az),
                c_double(angle),
                byref(api_outDimTags_), byref(api_outDimTags_n_),
                api_numElements_, api_numElements_n_,
                api_heights_, api_heights_n_,
                c_int(bool(recombine)),
                byref(ierr))
            if ierr.value != 0:
                raise ValueError(
                    "gmshModelGeoTwist returned non-zero error code: ",
                    ierr.value)
            return _ovectorpair(api_outDimTags_, api_outDimTags_n_.value)

        @staticmethod
        def translate(dimTags, dx, dy, dz):
            """
            gmsh.model.geo.translate(dimTags, dx, dy, dz)

            Translate the model entities `dimTags' along (`dx', `dy', `dz').
            """
            api_dimTags_, api_dimTags_n_ = _ivectorpair(dimTags)
            ierr = c_int()
            lib.gmshModelGeoTranslate(
                api_dimTags_, api_dimTags_n_,
                c_double(dx),
                c_double(dy),
                c_double(dz),
                byref(ierr))
            if ierr.value != 0:
                raise ValueError(
                    "gmshModelGeoTranslate returned non-zero error code: ",
                    ierr.value)

        @staticmethod
        def rotate(dimTags, x, y, z, ax, ay, az, angle):
            """
            gmsh.model.geo.rotate(dimTags, x, y, z, ax, ay, az, angle)

            Rotate the model entities `dimTags' of `angle' radians around the axis of
            revolution defined by the point (`x', `y', `z') and the direction (`ax',
            `ay', `az').
            """
            api_dimTags_, api_dimTags_n_ = _ivectorpair(dimTags)
            ierr = c_int()
            lib.gmshModelGeoRotate(
                api_dimTags_, api_dimTags_n_,
                c_double(x),
                c_double(y),
                c_double(z),
                c_double(ax),
                c_double(ay),
                c_double(az),
                c_double(angle),
                byref(ierr))
            if ierr.value != 0:
                raise ValueError(
                    "gmshModelGeoRotate returned non-zero error code: ",
                    ierr.value)

        @staticmethod
        def dilate(dimTags, x, y, z, a, b, c):
            """
            gmsh.model.geo.dilate(dimTags, x, y, z, a, b, c)

            Scale the model entities `dimTag' by factors `a', `b' and `c' along the
            three coordinate axes; use (`x', `y', `z') as the center of the homothetic
            transformation.
            """
            api_dimTags_, api_dimTags_n_ = _ivectorpair(dimTags)
            ierr = c_int()
            lib.gmshModelGeoDilate(
                api_dimTags_, api_dimTags_n_,
                c_double(x),
                c_double(y),
                c_double(z),
                c_double(a),
                c_double(b),
                c_double(c),
                byref(ierr))
            if ierr.value != 0:
                raise ValueError(
                    "gmshModelGeoDilate returned non-zero error code: ",
                    ierr.value)

        @staticmethod
        def mirror(dimTags, a, b, c, d):
            """
            gmsh.model.geo.mirror(dimTags, a, b, c, d)

            Mirror the model entities `dimTag', with respect to the plane of equation
            `a' * x + `b' * y + `c' * z + `d' = 0.
            """
            api_dimTags_, api_dimTags_n_ = _ivectorpair(dimTags)
            ierr = c_int()
            lib.gmshModelGeoMirror(
                api_dimTags_, api_dimTags_n_,
                c_double(a),
                c_double(b),
                c_double(c),
                c_double(d),
                byref(ierr))
            if ierr.value != 0:
                raise ValueError(
                    "gmshModelGeoMirror returned non-zero error code: ",
                    ierr.value)

        @staticmethod
        def symmetrize(dimTags, a, b, c, d):
            """
            gmsh.model.geo.symmetrize(dimTags, a, b, c, d)

            Mirror the model entities `dimTag', with respect to the plane of equation
            `a' * x + `b' * y + `c' * z + `d' = 0. (This is a synonym for `mirror',
            which will be deprecated in a future release.)
            """
            api_dimTags_, api_dimTags_n_ = _ivectorpair(dimTags)
            ierr = c_int()
            lib.gmshModelGeoSymmetrize(
                api_dimTags_, api_dimTags_n_,
                c_double(a),
                c_double(b),
                c_double(c),
                c_double(d),
                byref(ierr))
            if ierr.value != 0:
                raise ValueError(
                    "gmshModelGeoSymmetrize returned non-zero error code: ",
                    ierr.value)

        @staticmethod
        def copy(dimTags):
            """
            gmsh.model.geo.copy(dimTags)

            Copy the entities `dimTags'; the new entities are returned in `outDimTags'.

            Return `outDimTags'.
            """
            api_dimTags_, api_dimTags_n_ = _ivectorpair(dimTags)
            api_outDimTags_, api_outDimTags_n_ = POINTER(c_int)(), c_size_t()
            ierr = c_int()
            lib.gmshModelGeoCopy(
                api_dimTags_, api_dimTags_n_,
                byref(api_outDimTags_), byref(api_outDimTags_n_),
                byref(ierr))
            if ierr.value != 0:
                raise ValueError(
                    "gmshModelGeoCopy returned non-zero error code: ",
                    ierr.value)
            return _ovectorpair(api_outDimTags_, api_outDimTags_n_.value)

        @staticmethod
        def remove(dimTags, recursive=False):
            """
            gmsh.model.geo.remove(dimTags, recursive=False)

            Remove the entities `dimTags'. If `recursive' is true, remove all the
            entities on their boundaries, down to dimension 0.
            """
            api_dimTags_, api_dimTags_n_ = _ivectorpair(dimTags)
            ierr = c_int()
            lib.gmshModelGeoRemove(
                api_dimTags_, api_dimTags_n_,
                c_int(bool(recursive)),
                byref(ierr))
            if ierr.value != 0:
                raise ValueError(
                    "gmshModelGeoRemove returned non-zero error code: ",
                    ierr.value)

        @staticmethod
        def removeAllDuplicates():
            """
            gmsh.model.geo.removeAllDuplicates()

            Remove all duplicate entities (different entities at the same geometrical
            location).
            """
            ierr = c_int()
            lib.gmshModelGeoRemoveAllDuplicates(
                byref(ierr))
            if ierr.value != 0:
                raise ValueError(
                    "gmshModelGeoRemoveAllDuplicates returned non-zero error code: ",
                    ierr.value)

        @staticmethod
        def splitCurve(tag, pointTags):
            """
            gmsh.model.geo.splitCurve(tag, pointTags)

            Split the model curve of tag `tag' on the control points `pointTags'.
            Return the tags `curveTags' of the newly created curves.

            Return `curveTags'.
            """
            api_pointTags_, api_pointTags_n_ = _ivectorint(pointTags)
            api_curveTags_, api_curveTags_n_ = POINTER(c_int)(), c_size_t()
            ierr = c_int()
            lib.gmshModelGeoSplitCurve(
                c_int(tag),
                api_pointTags_, api_pointTags_n_,
                byref(api_curveTags_), byref(api_curveTags_n_),
                byref(ierr))
            if ierr.value != 0:
                raise ValueError(
                    "gmshModelGeoSplitCurve returned non-zero error code: ",
                    ierr.value)
            return _ovectorint(api_curveTags_, api_curveTags_n_.value)

        @staticmethod
        def synchronize():
            """
            gmsh.model.geo.synchronize()

            Synchronize the built-in CAD representation with the current Gmsh model.
            This can be called at any time, but since it involves a non trivial amount
            of processing, the number of synchronization points should normally be
            minimized.
            """
            ierr = c_int()
            lib.gmshModelGeoSynchronize(
                byref(ierr))
            if ierr.value != 0:
                raise ValueError(
                    "gmshModelGeoSynchronize returned non-zero error code: ",
                    ierr.value)


        class mesh:
            """
            Built-in CAD kernel meshing constraints
            """

            @staticmethod
            def setSize(dimTags, size):
                """
                gmsh.model.geo.mesh.setSize(dimTags, size)

                Set a mesh size constraint on the model entities `dimTags'. Currently only
                entities of dimension 0 (points) are handled.
                """
                api_dimTags_, api_dimTags_n_ = _ivectorpair(dimTags)
                ierr = c_int()
                lib.gmshModelGeoMeshSetSize(
                    api_dimTags_, api_dimTags_n_,
                    c_double(size),
                    byref(ierr))
                if ierr.value != 0:
                    raise ValueError(
                        "gmshModelGeoMeshSetSize returned non-zero error code: ",
                        ierr.value)

            @staticmethod
            def setTransfiniteCurve(tag, nPoints, meshType="Progression", coef=1.):
                """
                gmsh.model.geo.mesh.setTransfiniteCurve(tag, nPoints, meshType="Progression", coef=1.)

                Set a transfinite meshing constraint on the curve `tag', with `numNodes'
                nodes distributed according to `meshType' and `coef'. Currently supported
                types are "Progression" (geometrical progression with power `coef') and
                "Bump" (refinement toward both extremities of the curve).
                """
                ierr = c_int()
                lib.gmshModelGeoMeshSetTransfiniteCurve(
                    c_int(tag),
                    c_int(nPoints),
                    c_char_p(meshType.encode()),
                    c_double(coef),
                    byref(ierr))
                if ierr.value != 0:
                    raise ValueError(
                        "gmshModelGeoMeshSetTransfiniteCurve returned non-zero error code: ",
                        ierr.value)

            @staticmethod
            def setTransfiniteSurface(tag, arrangement="Left", cornerTags=[]):
                """
                gmsh.model.geo.mesh.setTransfiniteSurface(tag, arrangement="Left", cornerTags=[])

                Set a transfinite meshing constraint on the surface `tag'. `arrangement'
                describes the arrangement of the triangles when the surface is not flagged
                as recombined: currently supported values are "Left", "Right",
                "AlternateLeft" and "AlternateRight". `cornerTags' can be used to specify
                the (3 or 4) corners of the transfinite interpolation explicitly;
                specifying the corners explicitly is mandatory if the surface has more that
                3 or 4 points on its boundary.
                """
                api_cornerTags_, api_cornerTags_n_ = _ivectorint(cornerTags)
                ierr = c_int()
                lib.gmshModelGeoMeshSetTransfiniteSurface(
                    c_int(tag),
                    c_char_p(arrangement.encode()),
                    api_cornerTags_, api_cornerTags_n_,
                    byref(ierr))
                if ierr.value != 0:
                    raise ValueError(
                        "gmshModelGeoMeshSetTransfiniteSurface returned non-zero error code: ",
                        ierr.value)

            @staticmethod
            def setTransfiniteVolume(tag, cornerTags=[]):
                """
                gmsh.model.geo.mesh.setTransfiniteVolume(tag, cornerTags=[])

                Set a transfinite meshing constraint on the surface `tag'. `cornerTags' can
                be used to specify the (6 or 8) corners of the transfinite interpolation
                explicitly.
                """
                api_cornerTags_, api_cornerTags_n_ = _ivectorint(cornerTags)
                ierr = c_int()
                lib.gmshModelGeoMeshSetTransfiniteVolume(
                    c_int(tag),
                    api_cornerTags_, api_cornerTags_n_,
                    byref(ierr))
                if ierr.value != 0:
                    raise ValueError(
                        "gmshModelGeoMeshSetTransfiniteVolume returned non-zero error code: ",
                        ierr.value)

            @staticmethod
            def setRecombine(dim, tag, angle=45.):
                """
                gmsh.model.geo.mesh.setRecombine(dim, tag, angle=45.)

                Set a recombination meshing constraint on the model entity of dimension
                `dim' and tag `tag'. Currently only entities of dimension 2 (to recombine
                triangles into quadrangles) are supported.
                """
                ierr = c_int()
                lib.gmshModelGeoMeshSetRecombine(
                    c_int(dim),
                    c_int(tag),
                    c_double(angle),
                    byref(ierr))
                if ierr.value != 0:
                    raise ValueError(
                        "gmshModelGeoMeshSetRecombine returned non-zero error code: ",
                        ierr.value)

            @staticmethod
            def setSmoothing(dim, tag, val):
                """
                gmsh.model.geo.mesh.setSmoothing(dim, tag, val)

                Set a smoothing meshing constraint on the model entity of dimension `dim'
                and tag `tag'. `val' iterations of a Laplace smoother are applied.
                """
                ierr = c_int()
                lib.gmshModelGeoMeshSetSmoothing(
                    c_int(dim),
                    c_int(tag),
                    c_int(val),
                    byref(ierr))
                if ierr.value != 0:
                    raise ValueError(
                        "gmshModelGeoMeshSetSmoothing returned non-zero error code: ",
                        ierr.value)

            @staticmethod
            def setReverse(dim, tag, val=True):
                """
                gmsh.model.geo.mesh.setReverse(dim, tag, val=True)

                Set a reverse meshing constraint on the model entity of dimension `dim' and
                tag `tag'. If `val' is true, the mesh orientation will be reversed with
                respect to the natural mesh orientation (i.e. the orientation consistent
                with the orientation of the geometry). If `val' is false, the mesh is left
                as-is.
                """
                ierr = c_int()
                lib.gmshModelGeoMeshSetReverse(
                    c_int(dim),
                    c_int(tag),
                    c_int(bool(val)),
                    byref(ierr))
                if ierr.value != 0:
                    raise ValueError(
                        "gmshModelGeoMeshSetReverse returned non-zero error code: ",
                        ierr.value)

            @staticmethod
            def setAlgorithm(dim, tag, val):
                """
                gmsh.model.geo.mesh.setAlgorithm(dim, tag, val)

                Set the meshing algorithm on the model entity of dimension `dim' and tag
                `tag'. Currently only supported for `dim' == 2.
                """
                ierr = c_int()
                lib.gmshModelGeoMeshSetAlgorithm(
                    c_int(dim),
                    c_int(tag),
                    c_int(val),
                    byref(ierr))
                if ierr.value != 0:
                    raise ValueError(
                        "gmshModelGeoMeshSetAlgorithm returned non-zero error code: ",
                        ierr.value)

            @staticmethod
            def setSizeFromBoundary(dim, tag, val):
                """
                gmsh.model.geo.mesh.setSizeFromBoundary(dim, tag, val)

                Force the mesh size to be extended from the boundary, or not, for the model
                entity of dimension `dim' and tag `tag'. Currently only supported for `dim'
                == 2.
                """
                ierr = c_int()
                lib.gmshModelGeoMeshSetSizeFromBoundary(
                    c_int(dim),
                    c_int(tag),
                    c_int(val),
                    byref(ierr))
                if ierr.value != 0:
                    raise ValueError(
                        "gmshModelGeoMeshSetSizeFromBoundary returned non-zero error code: ",
                        ierr.value)


    class occ:
        """
        OpenCASCADE CAD kernel functions
        """

        @staticmethod
        def addPoint(x, y, z, meshSize=0., tag=-1):
            """
            gmsh.model.occ.addPoint(x, y, z, meshSize=0., tag=-1)

            Add a geometrical point in the OpenCASCADE CAD representation, at
            coordinates (`x', `y', `z'). If `meshSize' is > 0, add a meshing constraint
            at that point. If `tag' is positive, set the tag explicitly; otherwise a
            new tag is selected automatically. Return the tag of the point. (Note that
            the point will be added in the current model only after `synchronize' is
            called. This behavior holds for all the entities added in the occ module.)

            Return an integer value.
            """
            ierr = c_int()
            api__result__ = lib.gmshModelOccAddPoint(
                c_double(x),
                c_double(y),
                c_double(z),
                c_double(meshSize),
                c_int(tag),
                byref(ierr))
            if ierr.value != 0:
                raise ValueError(
                    "gmshModelOccAddPoint returned non-zero error code: ",
                    ierr.value)
            return api__result__

        @staticmethod
        def addLine(startTag, endTag, tag=-1):
            """
            gmsh.model.occ.addLine(startTag, endTag, tag=-1)

            Add a straight line segment between the two points with tags `startTag' and
            `endTag'. If `tag' is positive, set the tag explicitly; otherwise a new tag
            is selected automatically. Return the tag of the line.

            Return an integer value.
            """
            ierr = c_int()
            api__result__ = lib.gmshModelOccAddLine(
                c_int(startTag),
                c_int(endTag),
                c_int(tag),
                byref(ierr))
            if ierr.value != 0:
                raise ValueError(
                    "gmshModelOccAddLine returned non-zero error code: ",
                    ierr.value)
            return api__result__

        @staticmethod
        def addCircleArc(startTag, centerTag, endTag, tag=-1):
            """
            gmsh.model.occ.addCircleArc(startTag, centerTag, endTag, tag=-1)

            Add a circle arc between the two points with tags `startTag' and `endTag',
            with center `centerTag'. If `tag' is positive, set the tag explicitly;
            otherwise a new tag is selected automatically. Return the tag of the circle
            arc.

            Return an integer value.
            """
            ierr = c_int()
            api__result__ = lib.gmshModelOccAddCircleArc(
                c_int(startTag),
                c_int(centerTag),
                c_int(endTag),
                c_int(tag),
                byref(ierr))
            if ierr.value != 0:
                raise ValueError(
                    "gmshModelOccAddCircleArc returned non-zero error code: ",
                    ierr.value)
            return api__result__

        @staticmethod
        def addCircle(x, y, z, r, tag=-1, angle1=0., angle2=2*pi):
            """
            gmsh.model.occ.addCircle(x, y, z, r, tag=-1, angle1=0., angle2=2*pi)

            Add a circle of center (`x', `y', `z') and radius `r'. If `tag' is
            positive, set the tag explicitly; otherwise a new tag is selected
            automatically. If `angle1' and `angle2' are specified, create a circle arc
            between the two angles. Return the tag of the circle.

            Return an integer value.
            """
            ierr = c_int()
            api__result__ = lib.gmshModelOccAddCircle(
                c_double(x),
                c_double(y),
                c_double(z),
                c_double(r),
                c_int(tag),
                c_double(angle1),
                c_double(angle2),
                byref(ierr))
            if ierr.value != 0:
                raise ValueError(
                    "gmshModelOccAddCircle returned non-zero error code: ",
                    ierr.value)
            return api__result__

        @staticmethod
        def addEllipseArc(startTag, centerTag, majorTag, endTag, tag=-1):
            """
            gmsh.model.occ.addEllipseArc(startTag, centerTag, majorTag, endTag, tag=-1)

            Add an ellipse arc between the two points `startTag' and `endTag', with
            center `centerTag' and major axis point `majorTag'. If `tag' is positive,
            set the tag explicitly; otherwise a new tag is selected automatically.
            Return the tag of the ellipse arc. Note that OpenCASCADE does not allow
            creating ellipse arcs with the major radius smaller than the minor radius.

            Return an integer value.
            """
            ierr = c_int()
            api__result__ = lib.gmshModelOccAddEllipseArc(
                c_int(startTag),
                c_int(centerTag),
                c_int(majorTag),
                c_int(endTag),
                c_int(tag),
                byref(ierr))
            if ierr.value != 0:
                raise ValueError(
                    "gmshModelOccAddEllipseArc returned non-zero error code: ",
                    ierr.value)
            return api__result__

        @staticmethod
        def addEllipse(x, y, z, r1, r2, tag=-1, angle1=0., angle2=2*pi):
            """
            gmsh.model.occ.addEllipse(x, y, z, r1, r2, tag=-1, angle1=0., angle2=2*pi)

            Add an ellipse of center (`x', `y', `z') and radii `r1' and `r2' along the
            x- and y-axes respectively. If `tag' is positive, set the tag explicitly;
            otherwise a new tag is selected automatically. If `angle1' and `angle2' are
            specified, create an ellipse arc between the two angles. Return the tag of
            the ellipse. Note that OpenCASCADE does not allow creating ellipses with
            the major radius (along the x-axis) smaller than or equal to the minor
            radius (along the y-axis): rotate the shape or use `addCircle' in such
            cases.

            Return an integer value.
            """
            ierr = c_int()
            api__result__ = lib.gmshModelOccAddEllipse(
                c_double(x),
                c_double(y),
                c_double(z),
                c_double(r1),
                c_double(r2),
                c_int(tag),
                c_double(angle1),
                c_double(angle2),
                byref(ierr))
            if ierr.value != 0:
                raise ValueError(
                    "gmshModelOccAddEllipse returned non-zero error code: ",
                    ierr.value)
            return api__result__

        @staticmethod
        def addSpline(pointTags, tag=-1):
            """
            gmsh.model.occ.addSpline(pointTags, tag=-1)

            Add a spline (C2 b-spline) curve going through the points `pointTags'. If
            `tag' is positive, set the tag explicitly; otherwise a new tag is selected
            automatically. Create a periodic curve if the first and last points are the
            same. Return the tag of the spline curve.

            Return an integer value.
            """
            api_pointTags_, api_pointTags_n_ = _ivectorint(pointTags)
            ierr = c_int()
            api__result__ = lib.gmshModelOccAddSpline(
                api_pointTags_, api_pointTags_n_,
                c_int(tag),
                byref(ierr))
            if ierr.value != 0:
                raise ValueError(
                    "gmshModelOccAddSpline returned non-zero error code: ",
                    ierr.value)
            return api__result__

        @staticmethod
        def addBSpline(pointTags, tag=-1, degree=3, weights=[], knots=[], multiplicities=[]):
            """
            gmsh.model.occ.addBSpline(pointTags, tag=-1, degree=3, weights=[], knots=[], multiplicities=[])

            Add a b-spline curve of degree `degree' with `pointTags' control points. If
            `weights', `knots' or `multiplicities' are not provided, default parameters
            are computed automatically. If `tag' is positive, set the tag explicitly;
            otherwise a new tag is selected automatically. Create a periodic curve if
            the first and last points are the same. Return the tag of the b-spline
            curve.

            Return an integer value.
            """
            api_pointTags_, api_pointTags_n_ = _ivectorint(pointTags)
            api_weights_, api_weights_n_ = _ivectordouble(weights)
            api_knots_, api_knots_n_ = _ivectordouble(knots)
            api_multiplicities_, api_multiplicities_n_ = _ivectorint(multiplicities)
            ierr = c_int()
            api__result__ = lib.gmshModelOccAddBSpline(
                api_pointTags_, api_pointTags_n_,
                c_int(tag),
                c_int(degree),
                api_weights_, api_weights_n_,
                api_knots_, api_knots_n_,
                api_multiplicities_, api_multiplicities_n_,
                byref(ierr))
            if ierr.value != 0:
                raise ValueError(
                    "gmshModelOccAddBSpline returned non-zero error code: ",
                    ierr.value)
            return api__result__

        @staticmethod
        def addBezier(pointTags, tag=-1):
            """
            gmsh.model.occ.addBezier(pointTags, tag=-1)

            Add a Bezier curve with `pointTags' control points. If `tag' is positive,
            set the tag explicitly; otherwise a new tag is selected automatically.
            Return the tag of the Bezier curve.

            Return an integer value.
            """
            api_pointTags_, api_pointTags_n_ = _ivectorint(pointTags)
            ierr = c_int()
            api__result__ = lib.gmshModelOccAddBezier(
                api_pointTags_, api_pointTags_n_,
                c_int(tag),
                byref(ierr))
            if ierr.value != 0:
                raise ValueError(
                    "gmshModelOccAddBezier returned non-zero error code: ",
                    ierr.value)
            return api__result__

        @staticmethod
        def addWire(curveTags, tag=-1, checkClosed=False):
            """
            gmsh.model.occ.addWire(curveTags, tag=-1, checkClosed=False)

            Add a wire (open or closed) formed by the curves `curveTags'. Note that an
            OpenCASCADE wire can be made of curves that share geometrically identical
            (but topologically different) points. If `tag' is positive, set the tag
            explicitly; otherwise a new tag is selected automatically. Return the tag
            of the wire.

            Return an integer value.
            """
            api_curveTags_, api_curveTags_n_ = _ivectorint(curveTags)
            ierr = c_int()
            api__result__ = lib.gmshModelOccAddWire(
                api_curveTags_, api_curveTags_n_,
                c_int(tag),
                c_int(bool(checkClosed)),
                byref(ierr))
            if ierr.value != 0:
                raise ValueError(
                    "gmshModelOccAddWire returned non-zero error code: ",
                    ierr.value)
            return api__result__

        @staticmethod
        def addCurveLoop(curveTags, tag=-1):
            """
            gmsh.model.occ.addCurveLoop(curveTags, tag=-1)

            Add a curve loop (a closed wire) formed by the curves `curveTags'.
            `curveTags' should contain tags of curves forming a closed loop. Note that
            an OpenCASCADE curve loop can be made of curves that share geometrically
            identical (but topologically different) points. If `tag' is positive, set
            the tag explicitly; otherwise a new tag is selected automatically. Return
            the tag of the curve loop.

            Return an integer value.
            """
            api_curveTags_, api_curveTags_n_ = _ivectorint(curveTags)
            ierr = c_int()
            api__result__ = lib.gmshModelOccAddCurveLoop(
                api_curveTags_, api_curveTags_n_,
                c_int(tag),
                byref(ierr))
            if ierr.value != 0:
                raise ValueError(
                    "gmshModelOccAddCurveLoop returned non-zero error code: ",
                    ierr.value)
            return api__result__

        @staticmethod
        def addRectangle(x, y, z, dx, dy, tag=-1, roundedRadius=0.):
            """
            gmsh.model.occ.addRectangle(x, y, z, dx, dy, tag=-1, roundedRadius=0.)

            Add a rectangle with lower left corner at (`x', `y', `z') and upper right
            corner at (`x' + `dx', `y' + `dy', `z'). If `tag' is positive, set the tag
            explicitly; otherwise a new tag is selected automatically. Round the
            corners if `roundedRadius' is nonzero. Return the tag of the rectangle.

            Return an integer value.
            """
            ierr = c_int()
            api__result__ = lib.gmshModelOccAddRectangle(
                c_double(x),
                c_double(y),
                c_double(z),
                c_double(dx),
                c_double(dy),
                c_int(tag),
                c_double(roundedRadius),
                byref(ierr))
            if ierr.value != 0:
                raise ValueError(
                    "gmshModelOccAddRectangle returned non-zero error code: ",
                    ierr.value)
            return api__result__

        @staticmethod
        def addDisk(xc, yc, zc, rx, ry, tag=-1):
            """
            gmsh.model.occ.addDisk(xc, yc, zc, rx, ry, tag=-1)

            Add a disk with center (`xc', `yc', `zc') and radius `rx' along the x-axis
            and `ry' along the y-axis. If `tag' is positive, set the tag explicitly;
            otherwise a new tag is selected automatically. Return the tag of the disk.

            Return an integer value.
            """
            ierr = c_int()
            api__result__ = lib.gmshModelOccAddDisk(
                c_double(xc),
                c_double(yc),
                c_double(zc),
                c_double(rx),
                c_double(ry),
                c_int(tag),
                byref(ierr))
            if ierr.value != 0:
                raise ValueError(
                    "gmshModelOccAddDisk returned non-zero error code: ",
                    ierr.value)
            return api__result__

        @staticmethod
        def addPlaneSurface(wireTags, tag=-1):
            """
            gmsh.model.occ.addPlaneSurface(wireTags, tag=-1)

            Add a plane surface defined by one or more curve loops (or closed wires)
            `wireTags'. The first curve loop defines the exterior contour; additional
            curve loop define holes. If `tag' is positive, set the tag explicitly;
            otherwise a new tag is selected automatically. Return the tag of the
            surface.

            Return an integer value.
            """
            api_wireTags_, api_wireTags_n_ = _ivectorint(wireTags)
            ierr = c_int()
            api__result__ = lib.gmshModelOccAddPlaneSurface(
                api_wireTags_, api_wireTags_n_,
                c_int(tag),
                byref(ierr))
            if ierr.value != 0:
                raise ValueError(
                    "gmshModelOccAddPlaneSurface returned non-zero error code: ",
                    ierr.value)
            return api__result__

        @staticmethod
        def addSurfaceFilling(wireTag, tag=-1, pointTags=[]):
            """
            gmsh.model.occ.addSurfaceFilling(wireTag, tag=-1, pointTags=[])

            Add a surface filling the curve loops in `wireTags'. If `tag' is positive,
            set the tag explicitly; otherwise a new tag is selected automatically.
            Return the tag of the surface. If `pointTags' are provided, force the
            surface to pass through the given points.

            Return an integer value.
            """
            api_pointTags_, api_pointTags_n_ = _ivectorint(pointTags)
            ierr = c_int()
            api__result__ = lib.gmshModelOccAddSurfaceFilling(
                c_int(wireTag),
                c_int(tag),
                api_pointTags_, api_pointTags_n_,
                byref(ierr))
            if ierr.value != 0:
                raise ValueError(
                    "gmshModelOccAddSurfaceFilling returned non-zero error code: ",
                    ierr.value)
            return api__result__

        @staticmethod
        def addSurfaceLoop(surfaceTags, tag=-1, sewing=False):
            """
            gmsh.model.occ.addSurfaceLoop(surfaceTags, tag=-1, sewing=False)

            Add a surface loop (a closed shell) formed by `surfaceTags'.  If `tag' is
            positive, set the tag explicitly; otherwise a new tag is selected
            automatically. Return the tag of the surface loop. Setting `sewing' allows
            to build a shell made of surfaces that share geometrically identical (but
            topologically different) curves.

            Return an integer value.
            """
            api_surfaceTags_, api_surfaceTags_n_ = _ivectorint(surfaceTags)
            ierr = c_int()
            api__result__ = lib.gmshModelOccAddSurfaceLoop(
                api_surfaceTags_, api_surfaceTags_n_,
                c_int(tag),
                c_int(bool(sewing)),
                byref(ierr))
            if ierr.value != 0:
                raise ValueError(
                    "gmshModelOccAddSurfaceLoop returned non-zero error code: ",
                    ierr.value)
            return api__result__

        @staticmethod
        def addVolume(shellTags, tag=-1):
            """
            gmsh.model.occ.addVolume(shellTags, tag=-1)

            Add a volume (a region) defined by one or more surface loops `shellTags'.
            The first surface loop defines the exterior boundary; additional surface
            loop define holes. If `tag' is positive, set the tag explicitly; otherwise
            a new tag is selected automatically. Return the tag of the volume.

            Return an integer value.
            """
            api_shellTags_, api_shellTags_n_ = _ivectorint(shellTags)
            ierr = c_int()
            api__result__ = lib.gmshModelOccAddVolume(
                api_shellTags_, api_shellTags_n_,
                c_int(tag),
                byref(ierr))
            if ierr.value != 0:
                raise ValueError(
                    "gmshModelOccAddVolume returned non-zero error code: ",
                    ierr.value)
            return api__result__

        @staticmethod
        def addSphere(xc, yc, zc, radius, tag=-1, angle1=-pi/2, angle2=pi/2, angle3=2*pi):
            """
            gmsh.model.occ.addSphere(xc, yc, zc, radius, tag=-1, angle1=-pi/2, angle2=pi/2, angle3=2*pi)

            Add a sphere of center (`xc', `yc', `zc') and radius `r'. The optional
            `angle1' and `angle2' arguments define the polar angle opening (from -Pi/2
            to Pi/2). The optional `angle3' argument defines the azimuthal opening
            (from 0 to 2*Pi). If `tag' is positive, set the tag explicitly; otherwise a
            new tag is selected automatically. Return the tag of the sphere.

            Return an integer value.
            """
            ierr = c_int()
            api__result__ = lib.gmshModelOccAddSphere(
                c_double(xc),
                c_double(yc),
                c_double(zc),
                c_double(radius),
                c_int(tag),
                c_double(angle1),
                c_double(angle2),
                c_double(angle3),
                byref(ierr))
            if ierr.value != 0:
                raise ValueError(
                    "gmshModelOccAddSphere returned non-zero error code: ",
                    ierr.value)
            return api__result__

        @staticmethod
        def addBox(x, y, z, dx, dy, dz, tag=-1):
            """
            gmsh.model.occ.addBox(x, y, z, dx, dy, dz, tag=-1)

            Add a parallelepipedic box defined by a point (`x', `y', `z') and the
            extents along the x-, y- and z-axes. If `tag' is positive, set the tag
            explicitly; otherwise a new tag is selected automatically. Return the tag
            of the box.

            Return an integer value.
            """
            ierr = c_int()
            api__result__ = lib.gmshModelOccAddBox(
                c_double(x),
                c_double(y),
                c_double(z),
                c_double(dx),
                c_double(dy),
                c_double(dz),
                c_int(tag),
                byref(ierr))
            if ierr.value != 0:
                raise ValueError(
                    "gmshModelOccAddBox returned non-zero error code: ",
                    ierr.value)
            return api__result__

        @staticmethod
        def addCylinder(x, y, z, dx, dy, dz, r, tag=-1, angle=2*pi):
            """
            gmsh.model.occ.addCylinder(x, y, z, dx, dy, dz, r, tag=-1, angle=2*pi)

            Add a cylinder, defined by the center (`x', `y', `z') of its first circular
            face, the 3 components (`dx', `dy', `dz') of the vector defining its axis
            and its radius `r'. The optional `angle' argument defines the angular
            opening (from 0 to 2*Pi). If `tag' is positive, set the tag explicitly;
            otherwise a new tag is selected automatically. Return the tag of the
            cylinder.

            Return an integer value.
            """
            ierr = c_int()
            api__result__ = lib.gmshModelOccAddCylinder(
                c_double(x),
                c_double(y),
                c_double(z),
                c_double(dx),
                c_double(dy),
                c_double(dz),
                c_double(r),
                c_int(tag),
                c_double(angle),
                byref(ierr))
            if ierr.value != 0:
                raise ValueError(
                    "gmshModelOccAddCylinder returned non-zero error code: ",
                    ierr.value)
            return api__result__

        @staticmethod
        def addCone(x, y, z, dx, dy, dz, r1, r2, tag=-1, angle=2*pi):
            """
            gmsh.model.occ.addCone(x, y, z, dx, dy, dz, r1, r2, tag=-1, angle=2*pi)

            Add a cone, defined by the center (`x', `y', `z') of its first circular
            face, the 3 components of the vector (`dx', `dy', `dz') defining its axis
            and the two radii `r1' and `r2' of the faces (these radii can be zero). If
            `tag' is positive, set the tag explicitly; otherwise a new tag is selected
            automatically. `angle' defines the optional angular opening (from 0 to
            2*Pi). Return the tag of the cone.

            Return an integer value.
            """
            ierr = c_int()
            api__result__ = lib.gmshModelOccAddCone(
                c_double(x),
                c_double(y),
                c_double(z),
                c_double(dx),
                c_double(dy),
                c_double(dz),
                c_double(r1),
                c_double(r2),
                c_int(tag),
                c_double(angle),
                byref(ierr))
            if ierr.value != 0:
                raise ValueError(
                    "gmshModelOccAddCone returned non-zero error code: ",
                    ierr.value)
            return api__result__

        @staticmethod
        def addWedge(x, y, z, dx, dy, dz, tag=-1, ltx=0.):
            """
            gmsh.model.occ.addWedge(x, y, z, dx, dy, dz, tag=-1, ltx=0.)

            Add a right angular wedge, defined by the right-angle point (`x', `y', `z')
            and the 3 extends along the x-, y- and z-axes (`dx', `dy', `dz'). If `tag'
            is positive, set the tag explicitly; otherwise a new tag is selected
            automatically. The optional argument `ltx' defines the top extent along the
            x-axis. Return the tag of the wedge.

            Return an integer value.
            """
            ierr = c_int()
            api__result__ = lib.gmshModelOccAddWedge(
                c_double(x),
                c_double(y),
                c_double(z),
                c_double(dx),
                c_double(dy),
                c_double(dz),
                c_int(tag),
                c_double(ltx),
                byref(ierr))
            if ierr.value != 0:
                raise ValueError(
                    "gmshModelOccAddWedge returned non-zero error code: ",
                    ierr.value)
            return api__result__

        @staticmethod
        def addTorus(x, y, z, r1, r2, tag=-1, angle=2*pi):
            """
            gmsh.model.occ.addTorus(x, y, z, r1, r2, tag=-1, angle=2*pi)

            Add a torus, defined by its center (`x', `y', `z') and its 2 radii `r' and
            `r2'. If `tag' is positive, set the tag explicitly; otherwise a new tag is
            selected automatically. The optional argument `angle' defines the angular
            opening (from 0 to 2*Pi). Return the tag of the wedge.

            Return an integer value.
            """
            ierr = c_int()
            api__result__ = lib.gmshModelOccAddTorus(
                c_double(x),
                c_double(y),
                c_double(z),
                c_double(r1),
                c_double(r2),
                c_int(tag),
                c_double(angle),
                byref(ierr))
            if ierr.value != 0:
                raise ValueError(
                    "gmshModelOccAddTorus returned non-zero error code: ",
                    ierr.value)
            return api__result__

        @staticmethod
        def addThruSections(wireTags, tag=-1, makeSolid=True, makeRuled=False):
            """
            gmsh.model.occ.addThruSections(wireTags, tag=-1, makeSolid=True, makeRuled=False)

            Add a volume (if the optional argument `makeSolid' is set) or surfaces
            defined through the open or closed wires `wireTags'. If `tag' is positive,
            set the tag explicitly; otherwise a new tag is selected automatically. The
            new entities are returned in `outDimTags'. If the optional argument
            `makeRuled' is set, the surfaces created on the boundary are forced to be
            ruled surfaces.

            Return `outDimTags'.
            """
            api_wireTags_, api_wireTags_n_ = _ivectorint(wireTags)
            api_outDimTags_, api_outDimTags_n_ = POINTER(c_int)(), c_size_t()
            ierr = c_int()
            lib.gmshModelOccAddThruSections(
                api_wireTags_, api_wireTags_n_,
                byref(api_outDimTags_), byref(api_outDimTags_n_),
                c_int(tag),
                c_int(bool(makeSolid)),
                c_int(bool(makeRuled)),
                byref(ierr))
            if ierr.value != 0:
                raise ValueError(
                    "gmshModelOccAddThruSections returned non-zero error code: ",
                    ierr.value)
            return _ovectorpair(api_outDimTags_, api_outDimTags_n_.value)

        @staticmethod
        def addThickSolid(volumeTag, excludeSurfaceTags, offset, tag=-1):
            """
            gmsh.model.occ.addThickSolid(volumeTag, excludeSurfaceTags, offset, tag=-1)

            Add a hollowed volume built from an initial volume `volumeTag' and a set of
            faces from this volume `excludeSurfaceTags', which are to be removed. The
            remaining faces of the volume become the walls of the hollowed solid, with
            thickness `offset'. If `tag' is positive, set the tag explicitly; otherwise
            a new tag is selected automatically.

            Return `outDimTags'.
            """
            api_excludeSurfaceTags_, api_excludeSurfaceTags_n_ = _ivectorint(excludeSurfaceTags)
            api_outDimTags_, api_outDimTags_n_ = POINTER(c_int)(), c_size_t()
            ierr = c_int()
            lib.gmshModelOccAddThickSolid(
                c_int(volumeTag),
                api_excludeSurfaceTags_, api_excludeSurfaceTags_n_,
                c_double(offset),
                byref(api_outDimTags_), byref(api_outDimTags_n_),
                c_int(tag),
                byref(ierr))
            if ierr.value != 0:
                raise ValueError(
                    "gmshModelOccAddThickSolid returned non-zero error code: ",
                    ierr.value)
            return _ovectorpair(api_outDimTags_, api_outDimTags_n_.value)

        @staticmethod
        def extrude(dimTags, dx, dy, dz, numElements=[], heights=[], recombine=False):
            """
            gmsh.model.occ.extrude(dimTags, dx, dy, dz, numElements=[], heights=[], recombine=False)

            Extrude the model entities `dimTags' by translation along (`dx', `dy',
            `dz'). Return extruded entities in `outDimTags'. If `numElements' is not
            empty, also extrude the mesh: the entries in `numElements' give the number
            of elements in each layer. If `height' is not empty, it provides the
            (cumulative) height of the different layers, normalized to 1.

            Return `outDimTags'.
            """
            api_dimTags_, api_dimTags_n_ = _ivectorpair(dimTags)
            api_outDimTags_, api_outDimTags_n_ = POINTER(c_int)(), c_size_t()
            api_numElements_, api_numElements_n_ = _ivectorint(numElements)
            api_heights_, api_heights_n_ = _ivectordouble(heights)
            ierr = c_int()
            lib.gmshModelOccExtrude(
                api_dimTags_, api_dimTags_n_,
                c_double(dx),
                c_double(dy),
                c_double(dz),
                byref(api_outDimTags_), byref(api_outDimTags_n_),
                api_numElements_, api_numElements_n_,
                api_heights_, api_heights_n_,
                c_int(bool(recombine)),
                byref(ierr))
            if ierr.value != 0:
                raise ValueError(
                    "gmshModelOccExtrude returned non-zero error code: ",
                    ierr.value)
            return _ovectorpair(api_outDimTags_, api_outDimTags_n_.value)

        @staticmethod
        def revolve(dimTags, x, y, z, ax, ay, az, angle, numElements=[], heights=[], recombine=False):
            """
            gmsh.model.occ.revolve(dimTags, x, y, z, ax, ay, az, angle, numElements=[], heights=[], recombine=False)

            Extrude the model entities `dimTags' by rotation of `angle' radians around
            the axis of revolution defined by the point (`x', `y', `z') and the
            direction (`ax', `ay', `az'). Return extruded entities in `outDimTags'. If
            `numElements' is not empty, also extrude the mesh: the entries in
            `numElements' give the number of elements in each layer. If `height' is not
            empty, it provides the (cumulative) height of the different layers,
            normalized to 1. When the mesh is extruded the angle should be strictly
            smaller than 2*Pi.

            Return `outDimTags'.
            """
            api_dimTags_, api_dimTags_n_ = _ivectorpair(dimTags)
            api_outDimTags_, api_outDimTags_n_ = POINTER(c_int)(), c_size_t()
            api_numElements_, api_numElements_n_ = _ivectorint(numElements)
            api_heights_, api_heights_n_ = _ivectordouble(heights)
            ierr = c_int()
            lib.gmshModelOccRevolve(
                api_dimTags_, api_dimTags_n_,
                c_double(x),
                c_double(y),
                c_double(z),
                c_double(ax),
                c_double(ay),
                c_double(az),
                c_double(angle),
                byref(api_outDimTags_), byref(api_outDimTags_n_),
                api_numElements_, api_numElements_n_,
                api_heights_, api_heights_n_,
                c_int(bool(recombine)),
                byref(ierr))
            if ierr.value != 0:
                raise ValueError(
                    "gmshModelOccRevolve returned non-zero error code: ",
                    ierr.value)
            return _ovectorpair(api_outDimTags_, api_outDimTags_n_.value)

        @staticmethod
        def addPipe(dimTags, wireTag):
            """
            gmsh.model.occ.addPipe(dimTags, wireTag)

            Add a pipe by extruding the entities `dimTags' along the wire `wireTag'.
            Return the pipe in `outDimTags'.

            Return `outDimTags'.
            """
            api_dimTags_, api_dimTags_n_ = _ivectorpair(dimTags)
            api_outDimTags_, api_outDimTags_n_ = POINTER(c_int)(), c_size_t()
            ierr = c_int()
            lib.gmshModelOccAddPipe(
                api_dimTags_, api_dimTags_n_,
                c_int(wireTag),
                byref(api_outDimTags_), byref(api_outDimTags_n_),
                byref(ierr))
            if ierr.value != 0:
                raise ValueError(
                    "gmshModelOccAddPipe returned non-zero error code: ",
                    ierr.value)
            return _ovectorpair(api_outDimTags_, api_outDimTags_n_.value)

        @staticmethod
        def fillet(volumeTags, curveTags, radii, removeVolume=True):
            """
            gmsh.model.occ.fillet(volumeTags, curveTags, radii, removeVolume=True)

            Fillet the volumes `volumeTags' on the curves `curveTags' with radii
            `radii'. The `radii' vector can either contain a single radius, as many
            radii as `curveTags', or twice as many as `curveTags' (in which case
            different radii are provided for the begin and end points of the curves).
            Return the filleted entities in `outDimTags'. Remove the original volume if
            `removeVolume' is set.

            Return `outDimTags'.
            """
            api_volumeTags_, api_volumeTags_n_ = _ivectorint(volumeTags)
            api_curveTags_, api_curveTags_n_ = _ivectorint(curveTags)
            api_radii_, api_radii_n_ = _ivectordouble(radii)
            api_outDimTags_, api_outDimTags_n_ = POINTER(c_int)(), c_size_t()
            ierr = c_int()
            lib.gmshModelOccFillet(
                api_volumeTags_, api_volumeTags_n_,
                api_curveTags_, api_curveTags_n_,
                api_radii_, api_radii_n_,
                byref(api_outDimTags_), byref(api_outDimTags_n_),
                c_int(bool(removeVolume)),
                byref(ierr))
            if ierr.value != 0:
                raise ValueError(
                    "gmshModelOccFillet returned non-zero error code: ",
                    ierr.value)
            return _ovectorpair(api_outDimTags_, api_outDimTags_n_.value)

        @staticmethod
        def chamfer(volumeTags, curveTags, surfaceTags, distances, removeVolume=True):
            """
            gmsh.model.occ.chamfer(volumeTags, curveTags, surfaceTags, distances, removeVolume=True)

            Chamfer the volumes `volumeTags' on the curves `curveTags' with distances
            `distances' measured on surfaces `surfaceTags'. The `distances' vector can
            either contain a single distance, as many distances as `curveTags' and
            `surfaceTags', or twice as many as `curveTags' and `surfaceTags' (in which
            case the first in each pair is measured on the corresponding surface in
            `surfaceTags', the other on the other adjacent surface). Return the
            chamfered entities in `outDimTags'. Remove the original volume if
            `removeVolume' is set.

            Return `outDimTags'.
            """
            api_volumeTags_, api_volumeTags_n_ = _ivectorint(volumeTags)
            api_curveTags_, api_curveTags_n_ = _ivectorint(curveTags)
            api_surfaceTags_, api_surfaceTags_n_ = _ivectorint(surfaceTags)
            api_distances_, api_distances_n_ = _ivectordouble(distances)
            api_outDimTags_, api_outDimTags_n_ = POINTER(c_int)(), c_size_t()
            ierr = c_int()
            lib.gmshModelOccChamfer(
                api_volumeTags_, api_volumeTags_n_,
                api_curveTags_, api_curveTags_n_,
                api_surfaceTags_, api_surfaceTags_n_,
                api_distances_, api_distances_n_,
                byref(api_outDimTags_), byref(api_outDimTags_n_),
                c_int(bool(removeVolume)),
                byref(ierr))
            if ierr.value != 0:
                raise ValueError(
                    "gmshModelOccChamfer returned non-zero error code: ",
                    ierr.value)
            return _ovectorpair(api_outDimTags_, api_outDimTags_n_.value)

        @staticmethod
        def fuse(objectDimTags, toolDimTags, tag=-1, removeObject=True, removeTool=True):
            """
            gmsh.model.occ.fuse(objectDimTags, toolDimTags, tag=-1, removeObject=True, removeTool=True)

            Compute the boolean union (the fusion) of the entities `objectDimTags' and
            `toolDimTags'. Return the resulting entities in `outDimTags'. If `tag' is
            positive, try to set the tag explicitly (only valid if the boolean
            operation results in a single entity). Remove the object if `removeObject'
            is set. Remove the tool if `removeTool' is set.

            Return `outDimTags', `outDimTagsMap'.
            """
            api_objectDimTags_, api_objectDimTags_n_ = _ivectorpair(objectDimTags)
            api_toolDimTags_, api_toolDimTags_n_ = _ivectorpair(toolDimTags)
            api_outDimTags_, api_outDimTags_n_ = POINTER(c_int)(), c_size_t()
            api_outDimTagsMap_, api_outDimTagsMap_n_, api_outDimTagsMap_nn_ = POINTER(POINTER(c_int))(), POINTER(c_size_t)(), c_size_t()
            ierr = c_int()
            lib.gmshModelOccFuse(
                api_objectDimTags_, api_objectDimTags_n_,
                api_toolDimTags_, api_toolDimTags_n_,
                byref(api_outDimTags_), byref(api_outDimTags_n_),
                byref(api_outDimTagsMap_), byref(api_outDimTagsMap_n_), byref(api_outDimTagsMap_nn_),
                c_int(tag),
                c_int(bool(removeObject)),
                c_int(bool(removeTool)),
                byref(ierr))
            if ierr.value != 0:
                raise ValueError(
                    "gmshModelOccFuse returned non-zero error code: ",
                    ierr.value)
            return (
                _ovectorpair(api_outDimTags_, api_outDimTags_n_.value),
                _ovectorvectorpair(api_outDimTagsMap_, api_outDimTagsMap_n_, api_outDimTagsMap_nn_))

        @staticmethod
        def intersect(objectDimTags, toolDimTags, tag=-1, removeObject=True, removeTool=True):
            """
            gmsh.model.occ.intersect(objectDimTags, toolDimTags, tag=-1, removeObject=True, removeTool=True)

            Compute the boolean intersection (the common parts) of the entities
            `objectDimTags' and `toolDimTags'. Return the resulting entities in
            `outDimTags'. If `tag' is positive, try to set the tag explicitly (only
            valid if the boolean operation results in a single entity). Remove the
            object if `removeObject' is set. Remove the tool if `removeTool' is set.

            Return `outDimTags', `outDimTagsMap'.
            """
            api_objectDimTags_, api_objectDimTags_n_ = _ivectorpair(objectDimTags)
            api_toolDimTags_, api_toolDimTags_n_ = _ivectorpair(toolDimTags)
            api_outDimTags_, api_outDimTags_n_ = POINTER(c_int)(), c_size_t()
            api_outDimTagsMap_, api_outDimTagsMap_n_, api_outDimTagsMap_nn_ = POINTER(POINTER(c_int))(), POINTER(c_size_t)(), c_size_t()
            ierr = c_int()
            lib.gmshModelOccIntersect(
                api_objectDimTags_, api_objectDimTags_n_,
                api_toolDimTags_, api_toolDimTags_n_,
                byref(api_outDimTags_), byref(api_outDimTags_n_),
                byref(api_outDimTagsMap_), byref(api_outDimTagsMap_n_), byref(api_outDimTagsMap_nn_),
                c_int(tag),
                c_int(bool(removeObject)),
                c_int(bool(removeTool)),
                byref(ierr))
            if ierr.value != 0:
                raise ValueError(
                    "gmshModelOccIntersect returned non-zero error code: ",
                    ierr.value)
            return (
                _ovectorpair(api_outDimTags_, api_outDimTags_n_.value),
                _ovectorvectorpair(api_outDimTagsMap_, api_outDimTagsMap_n_, api_outDimTagsMap_nn_))

        @staticmethod
        def cut(objectDimTags, toolDimTags, tag=-1, removeObject=True, removeTool=True):
            """
            gmsh.model.occ.cut(objectDimTags, toolDimTags, tag=-1, removeObject=True, removeTool=True)

            Compute the boolean difference between the entities `objectDimTags' and
            `toolDimTags'. Return the resulting entities in `outDimTags'. If `tag' is
            positive, try to set the tag explicitly (only valid if the boolean
            operation results in a single entity). Remove the object if `removeObject'
            is set. Remove the tool if `removeTool' is set.

            Return `outDimTags', `outDimTagsMap'.
            """
            api_objectDimTags_, api_objectDimTags_n_ = _ivectorpair(objectDimTags)
            api_toolDimTags_, api_toolDimTags_n_ = _ivectorpair(toolDimTags)
            api_outDimTags_, api_outDimTags_n_ = POINTER(c_int)(), c_size_t()
            api_outDimTagsMap_, api_outDimTagsMap_n_, api_outDimTagsMap_nn_ = POINTER(POINTER(c_int))(), POINTER(c_size_t)(), c_size_t()
            ierr = c_int()
            lib.gmshModelOccCut(
                api_objectDimTags_, api_objectDimTags_n_,
                api_toolDimTags_, api_toolDimTags_n_,
                byref(api_outDimTags_), byref(api_outDimTags_n_),
                byref(api_outDimTagsMap_), byref(api_outDimTagsMap_n_), byref(api_outDimTagsMap_nn_),
                c_int(tag),
                c_int(bool(removeObject)),
                c_int(bool(removeTool)),
                byref(ierr))
            if ierr.value != 0:
                raise ValueError(
                    "gmshModelOccCut returned non-zero error code: ",
                    ierr.value)
            return (
                _ovectorpair(api_outDimTags_, api_outDimTags_n_.value),
                _ovectorvectorpair(api_outDimTagsMap_, api_outDimTagsMap_n_, api_outDimTagsMap_nn_))

        @staticmethod
        def fragment(objectDimTags, toolDimTags, tag=-1, removeObject=True, removeTool=True):
            """
            gmsh.model.occ.fragment(objectDimTags, toolDimTags, tag=-1, removeObject=True, removeTool=True)

            Compute the boolean fragments (general fuse) of the entities
            `objectDimTags' and `toolDimTags'. Return the resulting entities in
            `outDimTags'. If `tag' is positive, try to set the tag explicitly (only
            valid if the boolean operation results in a single entity). Remove the
            object if `removeObject' is set. Remove the tool if `removeTool' is set.

            Return `outDimTags', `outDimTagsMap'.
            """
            api_objectDimTags_, api_objectDimTags_n_ = _ivectorpair(objectDimTags)
            api_toolDimTags_, api_toolDimTags_n_ = _ivectorpair(toolDimTags)
            api_outDimTags_, api_outDimTags_n_ = POINTER(c_int)(), c_size_t()
            api_outDimTagsMap_, api_outDimTagsMap_n_, api_outDimTagsMap_nn_ = POINTER(POINTER(c_int))(), POINTER(c_size_t)(), c_size_t()
            ierr = c_int()
            lib.gmshModelOccFragment(
                api_objectDimTags_, api_objectDimTags_n_,
                api_toolDimTags_, api_toolDimTags_n_,
                byref(api_outDimTags_), byref(api_outDimTags_n_),
                byref(api_outDimTagsMap_), byref(api_outDimTagsMap_n_), byref(api_outDimTagsMap_nn_),
                c_int(tag),
                c_int(bool(removeObject)),
                c_int(bool(removeTool)),
                byref(ierr))
            if ierr.value != 0:
                raise ValueError(
                    "gmshModelOccFragment returned non-zero error code: ",
                    ierr.value)
            return (
                _ovectorpair(api_outDimTags_, api_outDimTags_n_.value),
                _ovectorvectorpair(api_outDimTagsMap_, api_outDimTagsMap_n_, api_outDimTagsMap_nn_))

        @staticmethod
        def translate(dimTags, dx, dy, dz):
            """
            gmsh.model.occ.translate(dimTags, dx, dy, dz)

            Translate the model entities `dimTags' along (`dx', `dy', `dz').
            """
            api_dimTags_, api_dimTags_n_ = _ivectorpair(dimTags)
            ierr = c_int()
            lib.gmshModelOccTranslate(
                api_dimTags_, api_dimTags_n_,
                c_double(dx),
                c_double(dy),
                c_double(dz),
                byref(ierr))
            if ierr.value != 0:
                raise ValueError(
                    "gmshModelOccTranslate returned non-zero error code: ",
                    ierr.value)

        @staticmethod
        def rotate(dimTags, x, y, z, ax, ay, az, angle):
            """
            gmsh.model.occ.rotate(dimTags, x, y, z, ax, ay, az, angle)

            Rotate the model entities `dimTags' of `angle' radians around the axis of
            revolution defined by the point (`x', `y', `z') and the direction (`ax',
            `ay', `az').
            """
            api_dimTags_, api_dimTags_n_ = _ivectorpair(dimTags)
            ierr = c_int()
            lib.gmshModelOccRotate(
                api_dimTags_, api_dimTags_n_,
                c_double(x),
                c_double(y),
                c_double(z),
                c_double(ax),
                c_double(ay),
                c_double(az),
                c_double(angle),
                byref(ierr))
            if ierr.value != 0:
                raise ValueError(
                    "gmshModelOccRotate returned non-zero error code: ",
                    ierr.value)

        @staticmethod
        def dilate(dimTags, x, y, z, a, b, c):
            """
            gmsh.model.occ.dilate(dimTags, x, y, z, a, b, c)

            Scale the model entities `dimTag' by factors `a', `b' and `c' along the
            three coordinate axes; use (`x', `y', `z') as the center of the homothetic
            transformation.
            """
            api_dimTags_, api_dimTags_n_ = _ivectorpair(dimTags)
            ierr = c_int()
            lib.gmshModelOccDilate(
                api_dimTags_, api_dimTags_n_,
                c_double(x),
                c_double(y),
                c_double(z),
                c_double(a),
                c_double(b),
                c_double(c),
                byref(ierr))
            if ierr.value != 0:
                raise ValueError(
                    "gmshModelOccDilate returned non-zero error code: ",
                    ierr.value)

        @staticmethod
        def mirror(dimTags, a, b, c, d):
            """
            gmsh.model.occ.mirror(dimTags, a, b, c, d)

            Apply a symmetry transformation to the model entities `dimTag', with
            respect to the plane of equation `a' * x + `b' * y + `c' * z + `d' = 0.
            """
            api_dimTags_, api_dimTags_n_ = _ivectorpair(dimTags)
            ierr = c_int()
            lib.gmshModelOccMirror(
                api_dimTags_, api_dimTags_n_,
                c_double(a),
                c_double(b),
                c_double(c),
                c_double(d),
                byref(ierr))
            if ierr.value != 0:
                raise ValueError(
                    "gmshModelOccMirror returned non-zero error code: ",
                    ierr.value)

        @staticmethod
        def symmetrize(dimTags, a, b, c, d):
            """
            gmsh.model.occ.symmetrize(dimTags, a, b, c, d)

            Apply a symmetry transformation to the model entities `dimTag', with
            respect to the plane of equation `a' * x + `b' * y + `c' * z + `d' = 0.
            (This is a synonym for `mirror', which will be deprecated in a future
            release.)
            """
            api_dimTags_, api_dimTags_n_ = _ivectorpair(dimTags)
            ierr = c_int()
            lib.gmshModelOccSymmetrize(
                api_dimTags_, api_dimTags_n_,
                c_double(a),
                c_double(b),
                c_double(c),
                c_double(d),
                byref(ierr))
            if ierr.value != 0:
                raise ValueError(
                    "gmshModelOccSymmetrize returned non-zero error code: ",
                    ierr.value)

        @staticmethod
        def affineTransform(dimTags, a):
            """
            gmsh.model.occ.affineTransform(dimTags, a)

            Apply a general affine transformation matrix `a' (16 entries of a 4x4
            matrix, by row; only the 12 first can be provided for convenience) to the
            model entities `dimTag'.
            """
            api_dimTags_, api_dimTags_n_ = _ivectorpair(dimTags)
            api_a_, api_a_n_ = _ivectordouble(a)
            ierr = c_int()
            lib.gmshModelOccAffineTransform(
                api_dimTags_, api_dimTags_n_,
                api_a_, api_a_n_,
                byref(ierr))
            if ierr.value != 0:
                raise ValueError(
                    "gmshModelOccAffineTransform returned non-zero error code: ",
                    ierr.value)

        @staticmethod
        def copy(dimTags):
            """
            gmsh.model.occ.copy(dimTags)

            Copy the entities `dimTags'; the new entities are returned in `outDimTags'.

            Return `outDimTags'.
            """
            api_dimTags_, api_dimTags_n_ = _ivectorpair(dimTags)
            api_outDimTags_, api_outDimTags_n_ = POINTER(c_int)(), c_size_t()
            ierr = c_int()
            lib.gmshModelOccCopy(
                api_dimTags_, api_dimTags_n_,
                byref(api_outDimTags_), byref(api_outDimTags_n_),
                byref(ierr))
            if ierr.value != 0:
                raise ValueError(
                    "gmshModelOccCopy returned non-zero error code: ",
                    ierr.value)
            return _ovectorpair(api_outDimTags_, api_outDimTags_n_.value)

        @staticmethod
        def remove(dimTags, recursive=False):
            """
            gmsh.model.occ.remove(dimTags, recursive=False)

            Remove the entities `dimTags'. If `recursive' is true, remove all the
            entities on their boundaries, down to dimension 0.
            """
            api_dimTags_, api_dimTags_n_ = _ivectorpair(dimTags)
            ierr = c_int()
            lib.gmshModelOccRemove(
                api_dimTags_, api_dimTags_n_,
                c_int(bool(recursive)),
                byref(ierr))
            if ierr.value != 0:
                raise ValueError(
                    "gmshModelOccRemove returned non-zero error code: ",
                    ierr.value)

        @staticmethod
        def removeAllDuplicates():
            """
            gmsh.model.occ.removeAllDuplicates()

            Remove all duplicate entities (different entities at the same geometrical
            location) after intersecting (using boolean fragments) all highest
            dimensional entities.
            """
            ierr = c_int()
            lib.gmshModelOccRemoveAllDuplicates(
                byref(ierr))
            if ierr.value != 0:
                raise ValueError(
                    "gmshModelOccRemoveAllDuplicates returned non-zero error code: ",
                    ierr.value)

        @staticmethod
        def healShapes(dimTags=[], tolerance=1e-8, fixDegenerated=True, fixSmallEdges=True, fixSmallFaces=True, sewFaces=True, makeSolids=True):
            """
            gmsh.model.occ.healShapes(dimTags=[], tolerance=1e-8, fixDegenerated=True, fixSmallEdges=True, fixSmallFaces=True, sewFaces=True, makeSolids=True)

            Apply various healing procedures to the entities `dimTags' (or to all the
            entities in the model if `dimTags' is empty). Return the healed entities in
            `outDimTags'. Available healing options are listed in the Gmsh reference
            manual.

            Return `outDimTags'.
            """
            api_outDimTags_, api_outDimTags_n_ = POINTER(c_int)(), c_size_t()
            api_dimTags_, api_dimTags_n_ = _ivectorpair(dimTags)
            ierr = c_int()
            lib.gmshModelOccHealShapes(
                byref(api_outDimTags_), byref(api_outDimTags_n_),
                api_dimTags_, api_dimTags_n_,
                c_double(tolerance),
                c_int(bool(fixDegenerated)),
                c_int(bool(fixSmallEdges)),
                c_int(bool(fixSmallFaces)),
                c_int(bool(sewFaces)),
                c_int(bool(makeSolids)),
                byref(ierr))
            if ierr.value != 0:
                raise ValueError(
                    "gmshModelOccHealShapes returned non-zero error code: ",
                    ierr.value)
            return _ovectorpair(api_outDimTags_, api_outDimTags_n_.value)

        @staticmethod
        def importShapes(fileName, highestDimOnly=True, format=""):
            """
            gmsh.model.occ.importShapes(fileName, highestDimOnly=True, format="")

            Import BREP, STEP or IGES shapes from the file `fileName'. The imported
            entities are returned in `outDimTags'. If the optional argument
            `highestDimOnly' is set, only import the highest dimensional entities in
            the file. The optional argument `format' can be used to force the format of
            the file (currently "brep", "step" or "iges").

            Return `outDimTags'.
            """
            api_outDimTags_, api_outDimTags_n_ = POINTER(c_int)(), c_size_t()
            ierr = c_int()
            lib.gmshModelOccImportShapes(
                c_char_p(fileName.encode()),
                byref(api_outDimTags_), byref(api_outDimTags_n_),
                c_int(bool(highestDimOnly)),
                c_char_p(format.encode()),
                byref(ierr))
            if ierr.value != 0:
                raise ValueError(
                    "gmshModelOccImportShapes returned non-zero error code: ",
                    ierr.value)
            return _ovectorpair(api_outDimTags_, api_outDimTags_n_.value)

        @staticmethod
        def setMeshSize(dimTags, size):
            """
            gmsh.model.occ.setMeshSize(dimTags, size)

            Set a mesh size constraint on the model entities `dimTags'. Currently only
            entities of dimension 0 (points) are handled.
            """
            api_dimTags_, api_dimTags_n_ = _ivectorpair(dimTags)
            ierr = c_int()
            lib.gmshModelOccSetMeshSize(
                api_dimTags_, api_dimTags_n_,
                c_double(size),
                byref(ierr))
            if ierr.value != 0:
                raise ValueError(
                    "gmshModelOccSetMeshSize returned non-zero error code: ",
                    ierr.value)

        @staticmethod
        def getMass(dim, tag):
            """
            gmsh.model.occ.getMass(dim, tag)

            Get the mass of the model entity of dimension `dim' and tag `tag'.

            Return `mass'.
            """
            api_mass_ = c_double()
            ierr = c_int()
            lib.gmshModelOccGetMass(
                c_int(dim),
                c_int(tag),
                byref(api_mass_),
                byref(ierr))
            if ierr.value != 0:
                raise ValueError(
                    "gmshModelOccGetMass returned non-zero error code: ",
                    ierr.value)
            return api_mass_.value

        @staticmethod
        def getCenterOfMass(dim, tag):
            """
            gmsh.model.occ.getCenterOfMass(dim, tag)

            Get the center of mass of the model entity of dimension `dim' and tag
            `tag'.

            Return `x', `y', `z'.
            """
            api_x_ = c_double()
            api_y_ = c_double()
            api_z_ = c_double()
            ierr = c_int()
            lib.gmshModelOccGetCenterOfMass(
                c_int(dim),
                c_int(tag),
                byref(api_x_),
                byref(api_y_),
                byref(api_z_),
                byref(ierr))
            if ierr.value != 0:
                raise ValueError(
                    "gmshModelOccGetCenterOfMass returned non-zero error code: ",
                    ierr.value)
            return (
                api_x_.value,
                api_y_.value,
                api_z_.value)

        @staticmethod
        def getMatrixOfInertia(dim, tag):
            """
            gmsh.model.occ.getMatrixOfInertia(dim, tag)

            Get the matrix of inertia (by row) of the model entity of dimension `dim'
            and tag `tag'.

            Return `mat'.
            """
            api_mat_, api_mat_n_ = POINTER(c_double)(), c_size_t()
            ierr = c_int()
            lib.gmshModelOccGetMatrixOfInertia(
                c_int(dim),
                c_int(tag),
                byref(api_mat_), byref(api_mat_n_),
                byref(ierr))
            if ierr.value != 0:
                raise ValueError(
                    "gmshModelOccGetMatrixOfInertia returned non-zero error code: ",
                    ierr.value)
            return _ovectordouble(api_mat_, api_mat_n_.value)

        @staticmethod
        def synchronize():
            """
            gmsh.model.occ.synchronize()

            Synchronize the OpenCASCADE CAD representation with the current Gmsh model.
            This can be called at any time, but since it involves a non trivial amount
            of processing, the number of synchronization points should normally be
            minimized.
            """
            ierr = c_int()
            lib.gmshModelOccSynchronize(
                byref(ierr))
            if ierr.value != 0:
                raise ValueError(
                    "gmshModelOccSynchronize returned non-zero error code: ",
                    ierr.value)


class view:
    """
    Post-processing view functions
    """

    @staticmethod
    def add(name, tag=-1):
        """
        gmsh.view.add(name, tag=-1)

        Add a new post-processing view, with name `name'. If `tag' is positive use
        it (and remove the view with that tag if it already exists), otherwise
        associate a new tag. Return the view tag.

        Return an integer value.
        """
        ierr = c_int()
        api__result__ = lib.gmshViewAdd(
            c_char_p(name.encode()),
            c_int(tag),
            byref(ierr))
        if ierr.value != 0:
            raise ValueError(
                "gmshViewAdd returned non-zero error code: ",
                ierr.value)
        return api__result__

    @staticmethod
    def remove(tag):
        """
        gmsh.view.remove(tag)

        Remove the view with tag `tag'.
        """
        ierr = c_int()
        lib.gmshViewRemove(
            c_int(tag),
            byref(ierr))
        if ierr.value != 0:
            raise ValueError(
                "gmshViewRemove returned non-zero error code: ",
                ierr.value)

    @staticmethod
    def getIndex(tag):
        """
        gmsh.view.getIndex(tag)

        Get the index of the view with tag `tag' in the list of currently loaded
        views. This dynamic index (it can change when views are removed) is used to
        access view options.

        Return an integer value.
        """
        ierr = c_int()
        api__result__ = lib.gmshViewGetIndex(
            c_int(tag),
            byref(ierr))
        if ierr.value != 0:
            raise ValueError(
                "gmshViewGetIndex returned non-zero error code: ",
                ierr.value)
        return api__result__

    @staticmethod
    def getTags():
        """
        gmsh.view.getTags()

        Get the tags of all views.

        Return `tags'.
        """
        api_tags_, api_tags_n_ = POINTER(c_int)(), c_size_t()
        ierr = c_int()
        lib.gmshViewGetTags(
            byref(api_tags_), byref(api_tags_n_),
            byref(ierr))
        if ierr.value != 0:
            raise ValueError(
                "gmshViewGetTags returned non-zero error code: ",
                ierr.value)
        return _ovectorint(api_tags_, api_tags_n_.value)

    @staticmethod
    def addModelData(tag, step, modelName, dataType, tags, data, time=0., numComponents=-1, partition=0):
        """
        gmsh.view.addModelData(tag, step, modelName, dataType, tags, data, time=0., numComponents=-1, partition=0)

        Add model-based post-processing data to the view with tag `tag'.
        `modelName' identifies the model the data is attached to. `dataType'
        specifies the type of data, currently either "NodeData", "ElementData" or
        "ElementNodeData". `step' specifies the identifier (>= 0) of the data in a
        sequence. `tags' gives the tags of the nodes or elements in the mesh to
        which the data is associated. `data' is a vector of the same length as
        `tags': each entry is the vector of double precision numbers representing
        the data associated with the corresponding tag. The optional `time'
        argument associate a time value with the data. `numComponents' gives the
        number of data components (1 for scalar data, 3 for vector data, etc.) per
        entity; if negative, it is automatically inferred (when possible) from the
        input data. `partition' allows to specify data in several sub-sets.
        """
        api_tags_, api_tags_n_ = _ivectorsize(tags)
        api_data_, api_data_n_, api_data_nn_ = _ivectorvectordouble(data)
        ierr = c_int()
        lib.gmshViewAddModelData(
            c_int(tag),
            c_int(step),
            c_char_p(modelName.encode()),
            c_char_p(dataType.encode()),
            api_tags_, api_tags_n_,
            api_data_, api_data_n_, api_data_nn_,
            c_double(time),
            c_int(numComponents),
            c_int(partition),
            byref(ierr))
        if ierr.value != 0:
            raise ValueError(
                "gmshViewAddModelData returned non-zero error code: ",
                ierr.value)

    @staticmethod
    def getModelData(tag, step):
        """
        gmsh.view.getModelData(tag, step)

        Get model-based post-processing data from the view with tag `tag' at step
        `step'. Return the `data' associated to the nodes or the elements with tags
        `tags', as well as the `dataType' and the number of components
        `numComponents'.

        Return `dataType', `tags', `data', `time', `numComponents'.
        """
        api_dataType_ = c_char_p()
        api_tags_, api_tags_n_ = POINTER(c_size_t)(), c_size_t()
        api_data_, api_data_n_, api_data_nn_ = POINTER(POINTER(c_double))(), POINTER(c_size_t)(), c_size_t()
        api_time_ = c_double()
        api_numComponents_ = c_int()
        ierr = c_int()
        lib.gmshViewGetModelData(
            c_int(tag),
            c_int(step),
            byref(api_dataType_),
            byref(api_tags_), byref(api_tags_n_),
            byref(api_data_), byref(api_data_n_), byref(api_data_nn_),
            byref(api_time_),
            byref(api_numComponents_),
            byref(ierr))
        if ierr.value != 0:
            raise ValueError(
                "gmshViewGetModelData returned non-zero error code: ",
                ierr.value)
        return (
            _ostring(api_dataType_),
            _ovectorsize(api_tags_, api_tags_n_.value),
            _ovectorvectordouble(api_data_, api_data_n_, api_data_nn_),
            api_time_.value,
            api_numComponents_.value)

    @staticmethod
    def addListData(tag, dataType, numEle, data):
        """
        gmsh.view.addListData(tag, dataType, numEle, data)

        Add list-based post-processing data to the view with tag `tag'. `dataType'
        identifies the data: "SP" for scalar points, "VP", for vector points, etc.
        `numEle' gives the number of elements in the data. `data' contains the data
        for the `numEle' elements.
        """
        api_data_, api_data_n_ = _ivectordouble(data)
        ierr = c_int()
        lib.gmshViewAddListData(
            c_int(tag),
            c_char_p(dataType.encode()),
            c_int(numEle),
            api_data_, api_data_n_,
            byref(ierr))
        if ierr.value != 0:
            raise ValueError(
                "gmshViewAddListData returned non-zero error code: ",
                ierr.value)

    @staticmethod
    def getListData(tag):
        """
        gmsh.view.getListData(tag)

        Get list-based post-processing data from the view with tag `tag'. Return
        the types `dataTypes', the number of elements `numElements' for each data
        type and the `data' for each data type.

        Return `dataType', `numElements', `data'.
        """
        api_dataType_, api_dataType_n_ = POINTER(POINTER(c_char))(), c_size_t()
        api_numElements_, api_numElements_n_ = POINTER(c_int)(), c_size_t()
        api_data_, api_data_n_, api_data_nn_ = POINTER(POINTER(c_double))(), POINTER(c_size_t)(), c_size_t()
        ierr = c_int()
        lib.gmshViewGetListData(
            c_int(tag),
            byref(api_dataType_), byref(api_dataType_n_),
            byref(api_numElements_), byref(api_numElements_n_),
            byref(api_data_), byref(api_data_n_), byref(api_data_nn_),
            byref(ierr))
        if ierr.value != 0:
            raise ValueError(
                "gmshViewGetListData returned non-zero error code: ",
                ierr.value)
        return (
            _ovectorstring(api_dataType_, api_dataType_n_.value),
            _ovectorint(api_numElements_, api_numElements_n_.value),
            _ovectorvectordouble(api_data_, api_data_n_, api_data_nn_))

    @staticmethod
    def addAlias(refTag, copyOptions=False, tag=-1):
        """
        gmsh.view.addAlias(refTag, copyOptions=False, tag=-1)

        Add a post-processing view as an `alias' of the reference view with tag
        `refTag'. If `copyOptions' is set, copy the options of the reference view.
        If `tag' is positive use it (and remove the view with that tag if it
        already exists), otherwise associate a new tag. Return the view tag.

        Return an integer value.
        """
        ierr = c_int()
        api__result__ = lib.gmshViewAddAlias(
            c_int(refTag),
            c_int(bool(copyOptions)),
            c_int(tag),
            byref(ierr))
        if ierr.value != 0:
            raise ValueError(
                "gmshViewAddAlias returned non-zero error code: ",
                ierr.value)
        return api__result__

    @staticmethod
    def copyOptions(refTag, tag):
        """
        gmsh.view.copyOptions(refTag, tag)

        Copy the options from the view with tag `refTag' to the view with tag
        `tag'.
        """
        ierr = c_int()
        lib.gmshViewCopyOptions(
            c_int(refTag),
            c_int(tag),
            byref(ierr))
        if ierr.value != 0:
            raise ValueError(
                "gmshViewCopyOptions returned non-zero error code: ",
                ierr.value)

    @staticmethod
    def combine(what, how, remove=True, copyOptions=True):
        """
        gmsh.view.combine(what, how, remove=True, copyOptions=True)

        Combine elements (if `what' == "elements") or steps (if `what' == "steps")
        of all views (`how' == "all"), all visible views (`how' == "visible") or
        all views having the same name (`how' == "name"). Remove original views if
        `remove' is set.
        """
        ierr = c_int()
        lib.gmshViewCombine(
            c_char_p(what.encode()),
            c_char_p(how.encode()),
            c_int(bool(remove)),
            c_int(bool(copyOptions)),
            byref(ierr))
        if ierr.value != 0:
            raise ValueError(
                "gmshViewCombine returned non-zero error code: ",
                ierr.value)

    @staticmethod
    def probe(tag, x, y, z, step=-1, numComp=-1, gradient=False, tolerance=0., xElemCoord=[], yElemCoord=[], zElemCoord=[]):
        """
        gmsh.view.probe(tag, x, y, z, step=-1, numComp=-1, gradient=False, tolerance=0., xElemCoord=[], yElemCoord=[], zElemCoord=[])

        Probe the view `tag' for its `value' at point (`x', `y', `z'). Return only
        the value at step `step' is `step' is positive. Return only values with
        `numComp' if `numComp' is positive. Return the gradient of the `value' if
        `gradient' is set. Probes with a geometrical tolerance (in the reference
        unit cube) of `tolerance' if `tolerance' is not zero. Return the result
        from the element described by its coordinates if `xElementCoord',
        `yElementCoord' and `zElementCoord' are provided.

        Return `value'.
        """
        api_value_, api_value_n_ = POINTER(c_double)(), c_size_t()
        api_xElemCoord_, api_xElemCoord_n_ = _ivectordouble(xElemCoord)
        api_yElemCoord_, api_yElemCoord_n_ = _ivectordouble(yElemCoord)
        api_zElemCoord_, api_zElemCoord_n_ = _ivectordouble(zElemCoord)
        ierr = c_int()
        lib.gmshViewProbe(
            c_int(tag),
            c_double(x),
            c_double(y),
            c_double(z),
            byref(api_value_), byref(api_value_n_),
            c_int(step),
            c_int(numComp),
            c_int(bool(gradient)),
            c_double(tolerance),
            api_xElemCoord_, api_xElemCoord_n_,
            api_yElemCoord_, api_yElemCoord_n_,
            api_zElemCoord_, api_zElemCoord_n_,
            byref(ierr))
        if ierr.value != 0:
            raise ValueError(
                "gmshViewProbe returned non-zero error code: ",
                ierr.value)
        return _ovectordouble(api_value_, api_value_n_.value)

    @staticmethod
    def write(tag, fileName, append=False):
        """
        gmsh.view.write(tag, fileName, append=False)

        Write the view to a file `fileName'. The export format is determined by the
        file extension. Append to the file if `append' is set.
        """
        ierr = c_int()
        lib.gmshViewWrite(
            c_int(tag),
            c_char_p(fileName.encode()),
            c_int(bool(append)),
            byref(ierr))
        if ierr.value != 0:
            raise ValueError(
                "gmshViewWrite returned non-zero error code: ",
                ierr.value)


class plugin:
    """
    Plugin functions
    """

    @staticmethod
    def setNumber(name, option, value):
        """
        gmsh.plugin.setNumber(name, option, value)

        Set the numerical option `option' to the value `value' for plugin `name'.
        """
        ierr = c_int()
        lib.gmshPluginSetNumber(
            c_char_p(name.encode()),
            c_char_p(option.encode()),
            c_double(value),
            byref(ierr))
        if ierr.value != 0:
            raise ValueError(
                "gmshPluginSetNumber returned non-zero error code: ",
                ierr.value)

    @staticmethod
    def setString(name, option, value):
        """
        gmsh.plugin.setString(name, option, value)

        Set the string option `option' to the value `value' for plugin `name'.
        """
        ierr = c_int()
        lib.gmshPluginSetString(
            c_char_p(name.encode()),
            c_char_p(option.encode()),
            c_char_p(value.encode()),
            byref(ierr))
        if ierr.value != 0:
            raise ValueError(
                "gmshPluginSetString returned non-zero error code: ",
                ierr.value)

    @staticmethod
    def run(name):
        """
        gmsh.plugin.run(name)

        Run the plugin `name'.
        """
        ierr = c_int()
        lib.gmshPluginRun(
            c_char_p(name.encode()),
            byref(ierr))
        if ierr.value != 0:
            raise ValueError(
                "gmshPluginRun returned non-zero error code: ",
                ierr.value)


class graphics:
    """
    Graphics functions
    """

    @staticmethod
    def draw():
        """
        gmsh.graphics.draw()

        Draw all the OpenGL scenes.
        """
        ierr = c_int()
        lib.gmshGraphicsDraw(
            byref(ierr))
        if ierr.value != 0:
            raise ValueError(
                "gmshGraphicsDraw returned non-zero error code: ",
                ierr.value)


class fltk:
    """
    FLTK graphical user interface functions
    """

    @staticmethod
    def initialize():
        """
        gmsh.fltk.initialize()

        Create the FLTK graphical user interface. Can only be called in the main
        thread.
        """
        ierr = c_int()
        lib.gmshFltkInitialize(
            byref(ierr))
        if ierr.value != 0:
            raise ValueError(
                "gmshFltkInitialize returned non-zero error code: ",
                ierr.value)

    @staticmethod
    def wait(time=-1.):
        """
        gmsh.fltk.wait(time=-1.)

        Wait at most `time' seconds for user interface events and return. If `time'
        < 0, wait indefinitely. First automatically create the user interface if it
        has not yet been initialized. Can only be called in the main thread.
        """
        ierr = c_int()
        lib.gmshFltkWait(
            c_double(time),
            byref(ierr))
        if ierr.value != 0:
            raise ValueError(
                "gmshFltkWait returned non-zero error code: ",
                ierr.value)

    @staticmethod
    def update():
        """
        gmsh.fltk.update()

        Update the user interface (potentially creating new widgets and windows).
        First automatically create the user interface if it has not yet been
        initialized. Can only be called in the main thread: use `awake("update")'
        to trigger an update of the user interface from another thread.
        """
        ierr = c_int()
        lib.gmshFltkUpdate(
            byref(ierr))
        if ierr.value != 0:
            raise ValueError(
                "gmshFltkUpdate returned non-zero error code: ",
                ierr.value)

    @staticmethod
    def awake(action=""):
        """
        gmsh.fltk.awake(action="")

        Awake the main user interface thread and process pending events, and
        optionally perform an action (currently the only `action' allowed is
        "update").
        """
        ierr = c_int()
        lib.gmshFltkAwake(
            c_char_p(action.encode()),
            byref(ierr))
        if ierr.value != 0:
            raise ValueError(
                "gmshFltkAwake returned non-zero error code: ",
                ierr.value)

    @staticmethod
    def lock():
        """
        gmsh.fltk.lock()

        Block the current thread until it can safely modify the user interface.
        """
        ierr = c_int()
        lib.gmshFltkLock(
            byref(ierr))
        if ierr.value != 0:
            raise ValueError(
                "gmshFltkLock returned non-zero error code: ",
                ierr.value)

    @staticmethod
    def unlock():
        """
        gmsh.fltk.unlock()

        Release the lock that was set using lock.
        """
        ierr = c_int()
        lib.gmshFltkUnlock(
            byref(ierr))
        if ierr.value != 0:
            raise ValueError(
                "gmshFltkUnlock returned non-zero error code: ",
                ierr.value)

    @staticmethod
    def run():
        """
        gmsh.fltk.run()

        Run the event loop of the graphical user interface, i.e. repeatedly call
        `wait()'. First automatically create the user interface if it has not yet
        been initialized. Can only be called in the main thread.
        """
        ierr = c_int()
        lib.gmshFltkRun(
            byref(ierr))
        if ierr.value != 0:
            raise ValueError(
                "gmshFltkRun returned non-zero error code: ",
                ierr.value)

    @staticmethod
    def isAvailable():
        """
        gmsh.fltk.isAvailable()

        Check if the user interface is available (e.g. to detect if it has been
        closed).

        Return an integer value.
        """
        ierr = c_int()
        api__result__ = lib.gmshFltkIsAvailable(
            byref(ierr))
        if ierr.value != 0:
            raise ValueError(
                "gmshFltkIsAvailable returned non-zero error code: ",
                ierr.value)
        return api__result__

    @staticmethod
    def selectEntities(dim=-1):
        """
        gmsh.fltk.selectEntities(dim=-1)

        Select entities in the user interface. If `dim' is >= 0, return only the
        entities of the specified dimension (e.g. points if `dim' == 0).

        Return an integer value, `dimTags'.
        """
        api_dimTags_, api_dimTags_n_ = POINTER(c_int)(), c_size_t()
        ierr = c_int()
        api__result__ = lib.gmshFltkSelectEntities(
            byref(api_dimTags_), byref(api_dimTags_n_),
            c_int(dim),
            byref(ierr))
        if ierr.value != 0:
            raise ValueError(
                "gmshFltkSelectEntities returned non-zero error code: ",
                ierr.value)
        return (
            api__result__,
            _ovectorpair(api_dimTags_, api_dimTags_n_.value))

    @staticmethod
    def selectElements():
        """
        gmsh.fltk.selectElements()

        Select elements in the user interface.

        Return an integer value, `elementTags'.
        """
        api_elementTags_, api_elementTags_n_ = POINTER(c_size_t)(), c_size_t()
        ierr = c_int()
        api__result__ = lib.gmshFltkSelectElements(
            byref(api_elementTags_), byref(api_elementTags_n_),
            byref(ierr))
        if ierr.value != 0:
            raise ValueError(
                "gmshFltkSelectElements returned non-zero error code: ",
                ierr.value)
        return (
            api__result__,
            _ovectorsize(api_elementTags_, api_elementTags_n_.value))

    @staticmethod
    def selectViews():
        """
        gmsh.fltk.selectViews()

        Select views in the user interface.

        Return an integer value, `viewTags'.
        """
        api_viewTags_, api_viewTags_n_ = POINTER(c_int)(), c_size_t()
        ierr = c_int()
        api__result__ = lib.gmshFltkSelectViews(
            byref(api_viewTags_), byref(api_viewTags_n_),
            byref(ierr))
        if ierr.value != 0:
            raise ValueError(
                "gmshFltkSelectViews returned non-zero error code: ",
                ierr.value)
        return (
            api__result__,
            _ovectorint(api_viewTags_, api_viewTags_n_.value))


class onelab:
    """
    ONELAB server functions
    """

    @staticmethod
    def set(data, format="json"):
        """
        gmsh.onelab.set(data, format="json")

        Set one or more parameters in the ONELAB database, encoded in `format'.
        """
        ierr = c_int()
        lib.gmshOnelabSet(
            c_char_p(data.encode()),
            c_char_p(format.encode()),
            byref(ierr))
        if ierr.value != 0:
            raise ValueError(
                "gmshOnelabSet returned non-zero error code: ",
                ierr.value)

    @staticmethod
    def get(name="", format="json"):
        """
        gmsh.onelab.get(name="", format="json")

        Get all the parameters (or a single one if `name' is specified) from the
        ONELAB database, encoded in `format'.

        Return `data'.
        """
        api_data_ = c_char_p()
        ierr = c_int()
        lib.gmshOnelabGet(
            byref(api_data_),
            c_char_p(name.encode()),
            c_char_p(format.encode()),
            byref(ierr))
        if ierr.value != 0:
            raise ValueError(
                "gmshOnelabGet returned non-zero error code: ",
                ierr.value)
        return _ostring(api_data_)

    @staticmethod
    def setNumber(name, value):
        """
        gmsh.onelab.setNumber(name, value)

        Set the value of the number parameter `name' in the ONELAB database. Create
        the parameter if it does not exist; update the value if the parameter
        exists.
        """
        api_value_, api_value_n_ = _ivectordouble(value)
        ierr = c_int()
        lib.gmshOnelabSetNumber(
            c_char_p(name.encode()),
            api_value_, api_value_n_,
            byref(ierr))
        if ierr.value != 0:
            raise ValueError(
                "gmshOnelabSetNumber returned non-zero error code: ",
                ierr.value)

    @staticmethod
    def setString(name, value):
        """
        gmsh.onelab.setString(name, value)

        Set the value of the string parameter `name' in the ONELAB database. Create
        the parameter if it does not exist; update the value if the parameter
        exists.
        """
        api_value_, api_value_n_ = _ivectorstring(value)
        ierr = c_int()
        lib.gmshOnelabSetString(
            c_char_p(name.encode()),
            api_value_, api_value_n_,
            byref(ierr))
        if ierr.value != 0:
            raise ValueError(
                "gmshOnelabSetString returned non-zero error code: ",
                ierr.value)

    @staticmethod
    def getNumber(name):
        """
        gmsh.onelab.getNumber(name)

        Get the value of the number parameter `name' from the ONELAB database.
        Return an empty vector if the parameter does not exist.

        Return `value'.
        """
        api_value_, api_value_n_ = POINTER(c_double)(), c_size_t()
        ierr = c_int()
        lib.gmshOnelabGetNumber(
            c_char_p(name.encode()),
            byref(api_value_), byref(api_value_n_),
            byref(ierr))
        if ierr.value != 0:
            raise ValueError(
                "gmshOnelabGetNumber returned non-zero error code: ",
                ierr.value)
        return _ovectordouble(api_value_, api_value_n_.value)

    @staticmethod
    def getString(name):
        """
        gmsh.onelab.getString(name)

        Get the value of the string parameter `name' from the ONELAB database.
        Return an empty vector if the parameter does not exist.

        Return `value'.
        """
        api_value_, api_value_n_ = POINTER(POINTER(c_char))(), c_size_t()
        ierr = c_int()
        lib.gmshOnelabGetString(
            c_char_p(name.encode()),
            byref(api_value_), byref(api_value_n_),
            byref(ierr))
        if ierr.value != 0:
            raise ValueError(
                "gmshOnelabGetString returned non-zero error code: ",
                ierr.value)
        return _ovectorstring(api_value_, api_value_n_.value)

    @staticmethod
    def clear(name=""):
        """
        gmsh.onelab.clear(name="")

        Clear the ONELAB database, or remove a single parameter if `name' is given.
        """
        ierr = c_int()
        lib.gmshOnelabClear(
            c_char_p(name.encode()),
            byref(ierr))
        if ierr.value != 0:
            raise ValueError(
                "gmshOnelabClear returned non-zero error code: ",
                ierr.value)

    @staticmethod
    def run(name="", command=""):
        """
        gmsh.onelab.run(name="", command="")

        Run a ONELAB client. If `name' is provided, create a new ONELAB client with
        name `name' and executes `command'. If not, try to run a client that might
        be linked to the processed input files.
        """
        ierr = c_int()
        lib.gmshOnelabRun(
            c_char_p(name.encode()),
            c_char_p(command.encode()),
            byref(ierr))
        if ierr.value != 0:
            raise ValueError(
                "gmshOnelabRun returned non-zero error code: ",
                ierr.value)


class logger:
    """
    Information logging functions
    """

    @staticmethod
    def write(message, level="info"):
        """
        gmsh.logger.write(message, level="info")

        Write a `message'. `level' can be "info", "warning" or "error".
        """
        ierr = c_int()
        lib.gmshLoggerWrite(
            c_char_p(message.encode()),
            c_char_p(level.encode()),
            byref(ierr))
        if ierr.value != 0:
            raise ValueError(
                "gmshLoggerWrite returned non-zero error code: ",
                ierr.value)

    @staticmethod
    def start():
        """
        gmsh.logger.start()

        Start logging messages.
        """
        ierr = c_int()
        lib.gmshLoggerStart(
            byref(ierr))
        if ierr.value != 0:
            raise ValueError(
                "gmshLoggerStart returned non-zero error code: ",
                ierr.value)

    @staticmethod
    def get():
        """
        gmsh.logger.get()

        Get logged messages.

        Return `log'.
        """
        api_log_, api_log_n_ = POINTER(POINTER(c_char))(), c_size_t()
        ierr = c_int()
        lib.gmshLoggerGet(
            byref(api_log_), byref(api_log_n_),
            byref(ierr))
        if ierr.value != 0:
            raise ValueError(
                "gmshLoggerGet returned non-zero error code: ",
                ierr.value)
        return _ovectorstring(api_log_, api_log_n_.value)

    @staticmethod
    def stop():
        """
        gmsh.logger.stop()

        Stop logging messages.
        """
        ierr = c_int()
        lib.gmshLoggerStop(
            byref(ierr))
        if ierr.value != 0:
            raise ValueError(
                "gmshLoggerStop returned non-zero error code: ",
                ierr.value)

    @staticmethod
    def getWallTime():
        """
        gmsh.logger.getWallTime()

        Return wall clock time.

        Return a floating point value.
        """
        ierr = c_int()
        api__result__ = lib.gmshLoggerGetWallTime(
            byref(ierr))
        if ierr.value != 0:
            raise ValueError(
                "gmshLoggerGetWallTime returned non-zero error code: ",
                ierr.value)
        return api__result__

    @staticmethod
    def getCpuTime():
        """
        gmsh.logger.getCpuTime()

        Return CPU time.

        Return a floating point value.
        """
        ierr = c_int()
        api__result__ = lib.gmshLoggerGetCpuTime(
            byref(ierr))
        if ierr.value != 0:
            raise ValueError(
                "gmshLoggerGetCpuTime returned non-zero error code: ",
                ierr.value)
        return api__result__
