/*******************************<GINKGO LICENSE>******************************
Copyright (c) 2017-2023, the Ginkgo authors
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions
are met:

1. Redistributions of source code must retain the above copyright
notice, this list of conditions and the following disclaimer.

2. Redistributions in binary form must reproduce the above copyright
notice, this list of conditions and the following disclaimer in the
documentation and/or other materials provided with the distribution.

3. Neither the name of the copyright holder nor the names of its
contributors may be used to endorse or promote products derived from
this software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS
IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
******************************<GINKGO LICENSE>*******************************/

#include <memory>
#include <random>


#include <gtest/gtest.h>


#include <ginkgo/core/base/exception.hpp>
#include <ginkgo/core/base/executor.hpp>
#include <ginkgo/core/matrix/csr.hpp>
#include <ginkgo/core/matrix/dense.hpp>
#include <ginkgo/core/solver/triangular.hpp>


#include "core/solver/lower_trs_kernels.hpp"
#include "hip/test/utils.hip.hpp"


namespace {


class LowerTrs : public HipTestFixture {
protected:
    using CsrMtx = gko::matrix::Csr<double, gko::int32>;
    using Mtx = gko::matrix::Dense<>;

    LowerTrs() : rand_engine(30) {}

    std::unique_ptr<Mtx> gen_mtx(int num_rows, int num_cols)
    {
        return gko::test::generate_random_matrix<Mtx>(
            num_rows, num_cols,
            std::uniform_int_distribution<>(num_cols, num_cols),
            std::normal_distribution<>(-1.0, 1.0), rand_engine, ref);
    }

    std::unique_ptr<Mtx> gen_l_mtx(int size)
    {
        return gko::test::generate_random_lower_triangular_matrix<Mtx>(
            size, false, std::uniform_int_distribution<>(size, size),
            std::normal_distribution<>(-1.0, 1.0), rand_engine, ref);
    }

    void initialize_data(int m, int n)
    {
        mtx = gen_l_mtx(m);
        b = gen_mtx(m, n);
        x = gen_mtx(m, n);
        csr_mtx = CsrMtx::create(ref);
        mtx->convert_to(csr_mtx);
        d_csr_mtx = CsrMtx::create(exec);
        d_x = gko::clone(exec, x);
        d_csr_mtx->copy_from(csr_mtx);
        b2 = Mtx::create(ref);
        d_b2 = gko::clone(exec, b);
        b2->copy_from(b);
    }

    std::shared_ptr<Mtx> b;
    std::shared_ptr<Mtx> b2;
    std::shared_ptr<Mtx> x;
    std::shared_ptr<Mtx> mtx;
    std::shared_ptr<CsrMtx> csr_mtx;
    std::shared_ptr<Mtx> d_b;
    std::shared_ptr<Mtx> d_b2;
    std::shared_ptr<Mtx> d_x;
    std::shared_ptr<CsrMtx> d_csr_mtx;
    std::default_random_engine rand_engine;
};


TEST_F(LowerTrs, HipLowerTrsFlagCheckIsCorrect)
{
    bool trans_flag = false;
    bool expected_flag = true;
    gko::kernels::hip::lower_trs::should_perform_transpose(exec, trans_flag);

    ASSERT_EQ(expected_flag, trans_flag);
}


TEST_F(LowerTrs, HipSingleRhsApplyIsEquivalentToRef)
{
    initialize_data(50, 1);
    auto lower_trs_factory = gko::solver::LowerTrs<>::build().on(ref);
    auto d_lower_trs_factory = gko::solver::LowerTrs<>::build().on(exec);
    auto solver = lower_trs_factory->generate(csr_mtx);
    auto d_solver = d_lower_trs_factory->generate(d_csr_mtx);

    solver->apply(b2, x);
    d_solver->apply(d_b2, d_x);

    GKO_ASSERT_MTX_NEAR(d_x, x, 1e-14);
}


TEST_F(LowerTrs, HipMultipleRhsApplyIsEquivalentToRef)
{
    initialize_data(50, 3);
    auto lower_trs_factory =
        gko::solver::LowerTrs<>::build().with_num_rhs(3u).on(ref);
    auto d_lower_trs_factory =
        gko::solver::LowerTrs<>::build().with_num_rhs(3u).on(exec);
    auto solver = lower_trs_factory->generate(csr_mtx);
    auto d_solver = d_lower_trs_factory->generate(d_csr_mtx);

    solver->apply(b2, x);
    d_solver->apply(d_b2, d_x);

    GKO_ASSERT_MTX_NEAR(d_x, x, 1e-14);
}


}  // namespace
