!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
!
module cubemain_compare
  use cube_types
  use cubetools_structure
  use cubeadm_cubeid_types
  use cubeadm_cubeprod_types
  use cubemain_messaging
  !
  public :: compare
  public :: cubemain_compare_command
  private
  !
  type :: compare_comm_t
     type(option_t),     pointer :: comm
     type(cubeid_arg_t), pointer :: one
     type(cubeid_arg_t), pointer :: two
     type(option_t),     pointer :: significant
     type(cube_prod_t),  pointer :: residuals
     type(cube_prod_t),  pointer :: fidelity
   contains
     procedure, public  :: register          => cubemain_compare_register
     procedure, private :: parse             => cubemain_compare_parse
     procedure, private :: parse_significant => cubemain_compare_parse_significant
     procedure, private :: main              => cubemain_compare_main
  end type compare_comm_t
  type(compare_comm_t) :: compare
  !
  type compare_user_t
     real(kind=sign_k)   :: significant ! Significant signal level for the residuals
     type(cubeid_user_t) :: cubeids
   contains
     procedure, private :: toprog => cubemain_compare_user_toprog
  end type compare_user_t
  !
  type compare_prog_t
     real(kind=sign_k)     :: significant ! Significant signal level for the residuals
     type(cube_t), pointer :: one
     type(cube_t), pointer :: two
     type(cube_t), pointer :: residuals
     type(cube_t), pointer :: fidelity
   contains
     procedure, private :: header => cubemain_compare_prog_header
     procedure, private :: data   => cubemain_compare_prog_data
     procedure, private :: loop   => cubemain_compare_prog_loop
     procedure, private :: act    => cubemain_compare_prog_act
  end type compare_prog_t
  !
contains
  !
  subroutine cubemain_compare_command(line,error)
    !----------------------------------------------------------------------
    !
    !----------------------------------------------------------------------
    character(len=*), intent(in)    :: line
    logical,          intent(inout) :: error
    !
    type(compare_user_t) :: user
    character(len=*), parameter :: rname='COMPARE>COMMAND'
    !
    call cubemain_message(mainseve%trace,rname,'Welcome')
    !
    call compare%parse(line,user,error)
    if (error) return
    call compare%main(user,error)
    if (error) continue
  end subroutine cubemain_compare_command
  !
  !----------------------------------------------------------------------
  !
  subroutine cubemain_compare_register(compare,error)
    use cubedag_allflags
    !----------------------------------------------------------------------
    !
    !----------------------------------------------------------------------
    class(compare_comm_t), intent(inout) :: compare
    logical,               intent(inout) :: error
    !
    type(cubeid_arg_t) :: cubearg
    type(cube_prod_t) :: oucube
    type(standard_arg_t) :: stdarg
    character(len=*), parameter :: comm_abstract='Compute cube1-cube2 and cube1/(cube1-cube2)'
    character(len=*), parameter :: comm_help=strg_id
    character(len=*), parameter :: rname='COMPARE>REGISTER'
    !
    call cubemain_message(mainseve%trace,rname,'Welcome')
    !
    ! Command
    call cubetools_register_command(&
         'COMPARE','cube1 cube2',&
         comm_abstract,&
         comm_help,&
         cubemain_compare_command,&
         compare%comm,error)
    call cubearg%register(&
         'CUBE1',&
         'Input cube #1',  &
         strg_id,&
         code_arg_mandatory,  &
         [flag_cube],& ! *** JP: Should it be flag_any?
         code_read, &
         code_access_subset, &
         compare%one, &
         error)
    call cubearg%register(&
         'CUBE2',&
         'Input cube #2',&
         strg_id,&
         code_arg_mandatory,&
         [flag_cube],& ! *** JP: Should it be flag_any?
         code_read, &
         code_access_subset, &
         compare%two, &
         error)
    ! Option #1
    call cubetools_register_option(&
         'SIGNIFICANT','level',&
         'Define a level of significance for the residuals',&
         strg_id,&
         compare%significant,error)
    call stdarg%register(&
         'LEVEL',&
         'Significant signal value (typical 3 to 5 sigma)',&
         strg_id,&
         code_arg_mandatory,&
         error)
    !
    ! Products
    call oucube%register(&
         'RESIDUALS',&
         'Cube of residuals',&
         strg_id,&
         [flag_compare,flag_residuals],&
         compare%residuals,&
         error)
    if (error) return
    call oucube%register(&
         'FIDELITY',&
         'Fidelity cube',&
         strg_id,&
         [flag_compare,flag_fidelity],&
         compare%fidelity,&
         error)
    if (error) return
  end subroutine cubemain_compare_register
  !
  subroutine cubemain_compare_parse(compare,line,user,error)
    !----------------------------------------------------------------------
    ! COMPARE id1 id2
    !----------------------------------------------------------------------
    class(compare_comm_t), intent(in)    :: compare
    character(len=*),      intent(in)    :: line
    type(compare_user_t),  intent(out)   :: user
    logical,               intent(inout) :: error
    !
    character(len=*), parameter :: rname='COMPARE>PARSE'
    !
    call cubemain_message(mainseve%trace,rname,'Welcome')
    !
    call cubeadm_cubeid_parse(line,compare%comm,user%cubeids,error)
    if (error) return
    call compare%parse_significant(line,user,error)
    if (error) return
  end subroutine cubemain_compare_parse
  !
  subroutine cubemain_compare_parse_significant(comm,line,user,error)
    !----------------------------------------------------------------------
    ! /SIGNIFICANT rms
    !----------------------------------------------------------------------
    class(compare_comm_t), intent(in)    :: comm
    character(len=*),      intent(in)    :: line
    type(compare_user_t),  intent(inout) :: user
    logical,               intent(inout) :: error
    !
    logical :: present
    character(len=*), parameter :: rname='COMPARE>PARSE>SIGNIFICANT'
    !
    call cubemain_message(mainseve%trace,rname,'Welcome')
    !
    call comm%significant%present(line,present,error)
    if (error) return
    if (.not.present) then
       user%significant = tiny(0.0)
    else
       call cubetools_getarg(line,compare%significant,1,user%significant,mandatory,error)
       if (error) return
       !
       if (user%significant.lt.0) then
          call cubemain_message(seve%e,rname,'Significant must be positive')
          error = .true.
          return
       endif
    endif
  end subroutine cubemain_compare_parse_significant
  !
  subroutine cubemain_compare_main(comm,user,error) 
    use cubeadm_timing
    !----------------------------------------------------------------------
    !
    !----------------------------------------------------------------------
    class(compare_comm_t), intent(in)    :: comm
    type(compare_user_t),  intent(inout) :: user
    logical,               intent(inout) :: error
    !
    type(compare_prog_t) :: prog
    character(len=*), parameter :: rname='COMPARE>MAIN'
    !
    call cubemain_message(mainseve%trace,rname,'Welcome')
    !
    call user%toprog(comm,prog,error)
    if (error) return
    call prog%header(comm,error)
    if (error) return
    call cubeadm_timing_prepro2process()
    call prog%data(error)
    if (error) return
    call cubeadm_timing_process2postpro()
  end subroutine cubemain_compare_main
  !
  !----------------------------------------------------------------------
  !
  subroutine cubemain_compare_user_toprog(user,comm,prog,error)
    use cubetools_user2prog
    use cubeadm_get
    !----------------------------------------------------------------------
    !
    !----------------------------------------------------------------------
    class(compare_user_t), intent(in)    :: user
    type(compare_comm_t),  intent(in)    :: comm 
    type(compare_prog_t),  intent(out)   :: prog
    logical,                       intent(inout) :: error
    !
    real(kind=sign_k), parameter :: default=1.0
    character(len=*), parameter :: rname='COMPARE>USER>TOPROG'
    !
    call cubemain_message(mainseve%trace,rname,'Welcome')
    !
    prog%significant = user%significant
    call cubeadm_get_header(comm%one,user%cubeids,prog%one,error)
    if (error) return
    call cubeadm_get_header(comm%two,user%cubeids,prog%two,error)
    if (error) return
  end subroutine cubemain_compare_user_toprog
  !
  !----------------------------------------------------------------------
  !
  subroutine cubemain_compare_prog_header(prog,comm,error)
    use cubetools_consistency_methods
    use cubetools_header_methods
    use cubedag_allflags
    use cubeadm_clone
    !----------------------------------------------------------------------
    !
    !----------------------------------------------------------------------
    class(compare_prog_t), intent(inout) :: prog
    type(compare_comm_t),  intent(in)    :: comm
    logical,               intent(inout) :: error
    !
    logical :: conspb
    character(len=*), parameter :: rname='COMPARE>PROG>HEADER'
    !
    call cubemain_message(mainseve%trace,rname,'Welcome')
    !
    conspb = .false.
    call cubetools_consistency_grid('Input cube #1',prog%one%head,'Input cube #2',prog%two%head,conspb,error)
    if (error) return
    if (cubetools_consistency_failed(rname,conspb,error)) return
    call cubeadm_clone_header(comm%residuals,prog%one,prog%residuals,error)
    if (error) return
    call cubeadm_clone_header(comm%fidelity,prog%one,prog%fidelity,error)
    if (error) return
    call cubetools_header_put_array_unit('---', prog%fidelity%head,error)
    if (error) return
  end subroutine cubemain_compare_prog_header
  !
  subroutine cubemain_compare_prog_data(prog,error)
    use cubeadm_opened
    !----------------------------------------------------------------------
    ! 
    !----------------------------------------------------------------------
    class(compare_prog_t), intent(inout) :: prog
    logical,               intent(inout) :: error
    !
    type(cubeadm_iterator_t) :: itertask
    character(len=*), parameter :: rname='COMPARE>PROG>DATA'
    !
    call cubemain_message(mainseve%trace,rname,'Welcome')
    !
    call cubeadm_datainit_all(itertask,error)
    if (error) return
    !$OMP PARALLEL DEFAULT(none) SHARED(prog,error) FIRSTPRIVATE(itertask)
    !$OMP SINGLE
    do while (cubeadm_dataiterate_all(itertask,error))
       if (error) exit
       !$OMP TASK SHARED(prog,error) FIRSTPRIVATE(itertask)
       if (.not.error) &
         call prog%loop(itertask,error)
       !$OMP END TASK
    enddo ! itertask
    !$OMP END SINGLE
    !$OMP END PARALLEL
  end subroutine cubemain_compare_prog_data
  !   
  subroutine cubemain_compare_prog_loop(prog,itertask,error)
    use cubeadm_taskloop
    !----------------------------------------------------------------------
    ! The subcube iterator will be shared by all input and output subcubes
    !----------------------------------------------------------------------
    class(compare_prog_t),    intent(inout) :: prog
    type(cubeadm_iterator_t), intent(inout) :: itertask
    logical,                  intent(inout) :: error
    !
    character(len=*), parameter :: rname='COMPARE>PROG>LOOP'
    !
    do while (itertask%iterate_entry(error))
       call prog%act(itertask,error)
       if (error) return
    enddo  ! ientry
  end subroutine cubemain_compare_prog_loop
  !   
  subroutine cubemain_compare_prog_act(prog,itertask,error)
    use cubeadm_taskloop
    use cubeadm_subcube_types
    !----------------------------------------------------------------------
    !
    !----------------------------------------------------------------------
    class(compare_prog_t),    intent(inout) :: prog
    type(cubeadm_iterator_t), intent(in)    :: itertask
    logical,                  intent(inout) :: error
    !
    integer(kind=indx_k) :: ix,iy,iz
    type(subcube_t) :: one,two,residuals,fidelity
    character(len=*), parameter :: rname='COMPARE>PROG>ACT'
    !
    ! Compares are initialized here as their size (3rd dim) may change from
    ! from one compare to another.
    call one%associate('one',prog%one,itertask,error)
    if (error) return
    call two%associate('two',prog%two,itertask,error)
    if (error) return
    call residuals%allocate('residuals',prog%residuals,itertask,error)
    if (error) return
    call fidelity%allocate('fidelity',prog%fidelity,itertask,error)
    if (error) return
    !
    call one%get(error)
    if (error) return
    call two%get(error)
    if (error) return
    do iz=1,one%nz
       do iy=1,one%ny
          do ix=1,one%nx
             residuals%val(ix,iy,iz) = one%val(ix,iy,iz)-two%val(ix,iy,iz)
             fidelity%val(ix,iy,iz) = abs(one%val(ix,iy,iz))/max(abs(residuals%val(ix,iy,iz)),prog%significant)
          enddo ! ix
       enddo ! iy
    enddo ! iz
    call residuals%put(error)
    if (error) return
    call fidelity%put(error)
    if (error) return
  end subroutine cubemain_compare_prog_act
end module cubemain_compare
!
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
