!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
!
module cubemain_chebyshev_svd
  use cubemain_messaging
  !
  public :: chebyshev_t,svd_t
  public :: cubemain_chebyshev_fit,cubemain_chebyshev_subtract
  public :: cubemain_free_chebyshev
  private
  !
  integer(kind=4), parameter :: coef_k = 4
  integer(kind=4), parameter :: degr_k = 4
  !
  type chebyshev_t
     integer(kind=degr_k) :: n = 0 ! Polynomial degree
     real(kind=4), pointer :: coeff(:) => null() ! Coefficients
     real(kind=4), pointer :: cheby(:) => null() ! Chebyshev polynomial values
     real(kind=coor_k) :: xval = 0.0 ! X value where Chebyshev polynomial values are computed
     real(kind=coor_k) :: xmin = 0.0 ! Minimum value of the x-axis computation interval
     real(kind=coor_k) :: xmax = 0.0 ! Maximum value of the x-axis computation interval
     real(kind=coor_k) :: xcen = 0.0 ! Its center 
     real(kind=coor_k) :: xsiz = 0.0 ! Its size
  end type chebyshev_t
  type svd_t
     integer(kind=chan_k) :: nchan = 0
     integer(kind=coef_k) :: ncoef = 0
     real(kind=4), pointer :: u(:,:) => null() ! nchan x ncoef
     real(kind=4), pointer :: v(:,:) => null() ! ncoef x ncoef
     real(kind=4), pointer :: w(:)   => null() ! ncoef
  end type svd_t
  !
contains
  !
  subroutine cubemain_reallocate_chebyshev(degree,poly,error)
    use gkernel_interfaces
    !----------------------------------------------------------------------
    !
    !----------------------------------------------------------------------
    integer(kind=degr_k),intent(in)    :: degree
    type(chebyshev_t),   intent(inout) :: poly
    logical,             intent(inout) :: error
    !
    logical :: alloc
    integer(kind=4) :: ier
    integer(kind=coef_k)  :: ncoeff
    character(len=mess_l) :: mess
    character(len=*), parameter :: rname='REALLOCATE>CHEBYSHEV'
    !
    call cubemain_message(mainseve%trace,rname,'Welcome')
    !
    ! Sanity check
    ncoeff = degree+1
    if (ncoeff.le.0) then
       call cubemain_message(seve%e,rname,'Negative or zero number of channels')
       error = .true.
       return
    endif
    alloc = .true.
    if (associated(poly%coeff)) then
       if (poly%n.eq.ncoeff) then
          write(mess,'(a,i0)')  &
               'Chebishev coefficients already associated at the right size: ',ncoeff
          call cubemain_message(mainseve%alloc,rname,mess)
          alloc = .false.
       else
          write(mess,'(a)') &
               'Chebishev coefficients already associated but with a different size => Freeing it first'
          call cubemain_message(mainseve%alloc,rname,mess)
          call cubemain_free_chebyshev(poly,error)
          if (error)  return
       endif
    endif
    if (alloc) then
       allocate(poly%coeff(ncoeff),poly%cheby(ncoeff),stat=ier)
       if (failed_allocate(rname,'Chebyshev coefficients',ier,error)) return
    endif
    ! Allocation success => poly%n may be updated
    poly%n = ncoeff
  end subroutine cubemain_reallocate_chebyshev
  !
  subroutine cubemain_free_chebyshev(poly,error)
    !----------------------------------------------------------------------
    ! 
    !----------------------------------------------------------------------
    type(chebyshev_t), intent(inout) :: poly
    logical,           intent(inout) :: error
    !
    character(len=*), parameter :: rname='FREE>CHEBYSHEV'
    !
    call cubemain_message(mainseve%trace,rname,'Welcome')
    !
    poly%n = 0
    if (associated(poly%coeff)) deallocate(poly%coeff)
    if (associated(poly%cheby)) deallocate(poly%cheby)
  end subroutine cubemain_free_chebyshev
  !
  subroutine cubemain_reallocate_svd(nchan,ncoef,svd,error)
    use gkernel_interfaces
    !----------------------------------------------------------------------
    ! 
    !----------------------------------------------------------------------
    integer(kind=chan_k), intent(in)    :: nchan
    integer(kind=coef_k), intent(in)    :: ncoef
    type(svd_t),          intent(inout) :: svd
    logical,              intent(inout) :: error
    !
    logical :: alloc
    integer(kind=4) :: ier
    character(len=mess_l) :: mess
    character(len=*), parameter :: rname='REALLOCATE>SVD'
    !
    call cubemain_message(mainseve%trace,rname,'Welcome')
    !
    ! Sanity check
    if (nchan.le.0) then
       call cubemain_message(seve%e,rname,'Negative or zero number of channels')
       error = .true.
       return
    endif
    if (ncoef.le.0) then
       call cubemain_message(seve%e,rname,'Negative or zero number of coefficients')
       error = .true.
       return
    endif
    alloc = .true.
    if (associated(svd%u)) then
       if ((svd%nchan.eq.nchan).and.(svd%ncoef.eq.ncoef)) then
          write(mess,'(a,i0,a,i0)')  &
               'SVD pointers already associated at the right size: ',nchan,' x ',ncoef
          call cubemain_message(mainseve%alloc,rname,mess)
          alloc = .false.
       else
          if (svd%ncoef.ne.ncoef) then
             ! At least ncoef changed => Reallocate everybody
             write(mess,'(a)') &
                  'SVD pointers already associated but with a different size => Freeing them first'
             call cubemain_message(mainseve%alloc,rname,mess)
             call cubemain_free_svd(svd,error)
             if (error)  return
             alloc = .true.
          else 
             ! Only nchan has changed => Reallocate only U
             write(mess,'(a)') &
                  'SVD%U pointer already associated but with a different size => Freeing it first'
             call cubemain_message(mainseve%alloc,rname,mess)
             if (associated(svd%u)) deallocate(svd%u)
             allocate(svd%u(nchan,ncoef),stat=ier)
             if (failed_allocate(rname,' svd%u',ier,error)) return
             alloc = .false.
          endif
       endif
    endif
    if (alloc) then
       allocate(svd%u(nchan,ncoef),svd%v(ncoef,ncoef),svd%w(ncoef),stat=ier)
       if (failed_allocate(rname,'SVD pointers',ier,error)) return
    endif
    ! Allocation success => svd%nchan and svd%ncoef may be updated
    svd%nchan = nchan
    svd%ncoef = ncoef
  end subroutine cubemain_reallocate_svd
  !
  subroutine cubemain_free_svd(svd,error)
    !----------------------------------------------------------------------
    ! 
    !----------------------------------------------------------------------
    type(svd_t), intent(inout) :: svd
    logical,     intent(inout) :: error
    !
    character(len=*), parameter :: rname='FREE>SVD'
    !
    call cubemain_message(mainseve%trace,rname,'Welcome')
    !
    svd%nchan = 0
    svd%ncoef = 0
    if (associated(svd%u)) deallocate(svd%u)
    if (associated(svd%v)) deallocate(svd%v)
    if (associated(svd%w)) deallocate(svd%w)
  end subroutine cubemain_free_svd
  !
  subroutine cubemain_chebyshev_polynomials(x,t,np)
    !---------------------------------------------------------------------
    ! Compute the first NP-1 Chebyshev polynomial of first kind at x
    !    T_n(x), n=0...np-1
    ! Recurrence relation : T_(n+1)(x) - 2xT_(n)(x) + T_(n-1)(x) = 0
    ! Initialization      : T_0 = 1
    !                     : T_1 = x
    !---------------------------------------------------------------------
    real(kind=coor_k),    intent(in)  :: x      ! Variable value
    integer(kind=degr_k), intent(in)  :: np     ! Polynomial degree
    real(kind=sign_k),    intent(out) :: t(np)  ! NP values of Chebyshev polynomial
    !
    integer :: j
    !
    t(1) = 1.
    if (np.gt.1) then
       t(2) = x
       do j=3,np
          t(j)=2*t(j-1)*x-t(j-2)
       enddo
    endif
  end subroutine cubemain_chebyshev_polynomials
  !
  function cubemain_chebyshev_approximation(poly,x)
    !---------------------------------------------------------------------
    ! 
    !---------------------------------------------------------------------
    real(kind=sign_k)                :: cubemain_chebyshev_approximation
    type(chebyshev_t), intent(inout) :: poly
    real(kind=coor_k), intent(in)    :: x
    !
    real(kind=sign_k) :: val
    integer(kind=coef_k)   :: icoeff,ncoeff
    !
    ncoeff = poly%n
    poly%xval = x
    call cubemain_chebyshev_polynomials(x,poly%cheby,ncoeff)
    val = 0
    do icoeff=1,ncoeff
       val = val+poly%coeff(icoeff)*poly%cheby(icoeff)
    enddo ! icoeff
    cubemain_chebyshev_approximation = val
  end function cubemain_chebyshev_approximation
  !
  subroutine cubemain_chebyshev_check_degree(rname,svd)
    !---------------------------------------------------------------------
    ! 
    !---------------------------------------------------------------------
    character,   intent(in) :: rname
    type(svd_t), intent(in) :: svd
    !
    integer(kind=degr_k)  :: imin
    character(len=mess_l) :: mess
    !
    imin = minloc(svd%w,1)
    if (imin.lt.svd%ncoef) then
       write(mess,'(A,I2,A)') 'Degree ',imin-1,' would be even better'
       call cubemain_message(seve%i,rname,mess)
    endif
  end subroutine cubemain_chebyshev_check_degree
  !
  subroutine cubemain_chebyshev_fit(degree,spec,poly,svd,error)
    use gkernel_interfaces
    use cubemain_spectrum_real
    !---------------------------------------------------------------------
    ! Fit a Chebyshev polynomial baseline using singular value decomposition
    ! This one assumes that spec does not contain blanked values
    !---------------------------------------------------------------------
    integer(kind=degr_K),intent(in)    :: degree
    type(spectrum_t),    intent(in)    :: spec
    type(chebyshev_t),   intent(inout) :: poly
    type(svd_t),         intent(inout) :: svd
    logical,             intent(inout) :: error
    !
    integer(kind=chan_k) :: ichan
    real(kind=sign_k) :: chisq
    character(len=mess_l) :: mess
    character(len=*), parameter :: rname='CHEBYSHEV>FIT'
    !
    ! Check that we have enough data points
    ! Note: polynomial degree >= 0 implies nchan > 0
    if (spec%n.le.degree) then
       error = .true.
       write(mess,*) 'Not enough channels to fit baseline: ',spec%n,degree
       call cubemain_message(seve%e,rname,mess)
       return
    endif
    !
    call cubemain_reallocate_chebyshev(degree,poly,error)
    if (error) return
    ! Normalize range to [-1,1]
    ! *** JP: What happens when npoint = 1 is unclear... *** JP
    ! VVV The following line implies the spectrum header is filled
    poly%xmin = spec%v(1)
    poly%xmax = spec%v(spec%n)
    poly%xcen = 0.5*(poly%xmax+poly%xmin)
    poly%xsiz = 0.5*(poly%xmax-poly%xmin)
    do ichan=1,spec%n
       spec%v(ichan) = (spec%v(ichan)-poly%xcen) / poly%xsiz
    enddo ! ichan
    !
    call cubemain_reallocate_svd(spec%n,poly%n,svd,error)
    if (error) return
    call cubemain_svdfit(spec%v,spec%t,spec%w,spec%n,&
         poly%coeff,poly%n,&
         svd%u,svd%v,svd%w,svd%nchan,svd%ncoef,&
         chisq,cubemain_chebyshev_polynomials,&
         error)
    if (error) then
       call cubemain_message(seve%e,rname,'Error in singular value decomposition')
       return
    endif
    call cubemain_free_svd(svd,error)
    if (error) return
  end subroutine cubemain_chebyshev_fit
  !
  subroutine cubemain_chebyshev_subtract(poly,spec,ifirst,ilast,base,resi,error)
    use cubemain_spectrum_real
    !---------------------------------------------------------------------
    ! Subtract a Chebyshev approximation of the baseline to the spectrum
    !---------------------------------------------------------------------
    type(chebyshev_t),    intent(inout) :: poly
    type(spectrum_t),     intent(in)    :: spec
    integer(kind=chan_k), intent(in)    :: ifirst
    integer(kind=chan_k), intent(in)    :: ilast
    type(spectrum_t),     intent(inout) :: base
    type(spectrum_t),     intent(inout) :: resi
    logical,              intent(inout) :: error
    !
    integer(kind=chan_k) :: ic
    real(kind=coor_k)    :: xic
    real(kind=sign_k) :: yic,ymin,ymax
    character(len=*), parameter :: rname='CHEBYSHEV>SUBTRACT'
    !
    ! Compute baseline value
    ! First at interval edges
    ymin = cubemain_chebyshev_approximation(poly,-1d0)
    ymax = cubemain_chebyshev_approximation(poly,+1d0)
    ! Then everywhere
    if (poly%n.gt.2) then
       ! Degree > 1 => poly%n.gt.2
       ! Compute polynomial inside fitting range and constant or zero outside
       do ic=ifirst,ilast
          xic = (spec%v(ic)-poly%xcen) /poly%xsiz
          if (xic.le.-1.0) then
             yic = ymin
          elseif (xic.ge.1.0) then
             yic = ymax
          else
             yic = cubemain_chebyshev_approximation(poly,xic)
          endif
          base%t(ic) = yic
          resi%t(ic) = spec%t(ic)-yic
       enddo ! if
    else
       ! Degree 0 or 1 : compute polynomial everywhere (including
       ! extrapolation beyond edges)
       do ic=ifirst,ilast
          xic = (spec%v(ic)-poly%xcen) /poly%xsiz
          yic = ymin + (xic+1.0)*(ymax-ymin)/2.0
          base%t(ic) = yic
          resi%t(ic) = spec%t(ic)-yic
       enddo ! if
    endif
  end subroutine cubemain_chebyshev_subtract
  !
  subroutine cubemain_svdfit(x,y,weight,ndata,a,ma,u,v,w,mp,np,chisq,funcs,error)
    !---------------------------------------------------------------------
    !
    ! Singular value decomposition
    !
    ! Given a set of NDATA points X(I), Y(I) with individual weights
    ! WEIGHT(I), use Chi2 minimization to determine the MA coefficients A of
    ! the fitting function. Here we solve the fitting equations using
    ! singular value decomposition of the NDATA by MA matrix.
    !
    ! The user supplies a subroutine FUNCS(X,AFUNC,MA) that returns the MA
    ! basis functions evaluated at x=X in the array AFUNC.
    !
    ! The programs returns values for the MA fit parameters and Chi2, CHISQ. 
    !
    ! Arrays U,V,W provide workspace on input. On output they define the
    ! singular value decomposition and can be used to obtain the covariance
    ! matrix.
    !
    ! It is necessary that MP>=NDATA, NP>=MA.
    !---------------------------------------------------------------------
    integer(kind=chan_k), intent(in)    :: ndata          !
    real(kind=coor_k),    intent(in)    :: x(ndata)       !
    real(kind=sign_k),    intent(in)    :: y(ndata)       !
    real(kind=sign_k),    intent(in)    :: weight(ndata)  !
    integer(kind=4),      intent(in)    :: ma             !
    real(kind=4),         intent(out)   :: a(ma)          !
    integer(kind=chan_k), intent(in)    :: mp             !
    integer(kind=4),      intent(in)    :: np             !
    real(kind=4),         intent(inout) :: u(mp,np)       !
    real(kind=4),         intent(inout) :: v(np,np)       !
    real(kind=4),         intent(inout) :: w(np)          !
    real(kind=sign_k),    intent(out)   :: chisq          ! Chi^2
    external                            :: funcs          !
    logical,              intent(out)   :: error          !
    !
    real(kind=4), parameter :: tol=1.e-5
    real(kind=4) :: b(ndata)
    real(kind=4) :: afunc(ma)
    integer(kind=4) :: i,j
    real(kind=4) :: wmax,thresh,sum,tmp
    !
    error = .false.
    do i=1,ndata
       call funcs(x(i),afunc,ma)
       tmp = sqrt(weight(i))
       do j=1,ma
          u(i,j) = afunc(j)*tmp
       enddo
       b(i) = y(i)*tmp
    enddo
    call cubemain_svdcmp(u,ndata,ma,mp,np,w,v,error)
    if (error) return
    ! Edit out the (nearly) singular values
    wmax = 0.
    do j=1,ma
       if (w(j).gt.wmax) wmax=w(j)
    enddo
    thresh = tol*wmax
    do j=1,ma
       if (w(j).lt.thresh) w(j) = 0
    enddo
    ! Solve the equations
    call cubemain_svbksb(u,w,v,ndata,ma,mp,np,b,a,error)
    if (error) return
    ! Evaluate chi-square
    chisq = 0.
    do i=1,ndata
       call funcs(x(i),afunc,ma)
       sum=0.
       do j=1,ma
          sum = sum+a(j)*afunc(j)
       enddo
       chisq = chisq+(y(i)-sum)**2*weight(i)
    enddo
  end subroutine cubemain_svdfit
  !
  subroutine cubemain_svdcmp(a,m,n,mp,np,w,v,error)
    use gkernel_interfaces
    !---------------------------------------------------------------------
    !
    ! Singular value decomposition
    !
    ! Given a matrix A with logical dimensions M by N and physical dimensions
    ! MP by NP, this routine computes its singular value decomposition
    !	A = U.W.Vt
    !
    ! The matrix U replaces A on output. The diagonal matrix of singular
    ! values W is output as vector W. The matrix V (not the transpose Vt) is
    ! output as V.
    !
    ! M must be greater or equal to N. If it is smaller, then A should be
    ! filled up to square with zero rows.
    ! ---------------------------------------------------------------------
    integer(kind=chan_k), intent(in)    :: mp        !
    integer(kind=4),      intent(in)    :: np        !
    real(kind=4),         intent(inout) :: a(mp,np)  !
    integer(kind=chan_k), intent(in)    :: m         !
    integer(kind=4),      intent(in)    :: n         !
    real(kind=4),         intent(out)   :: w(np)     !
    real(kind=4),         intent(out)   :: v(np,np)  !
    logical,              intent(out)   :: error     !
    !
    integer(kind=4), parameter :: nmax=100 ! Maximum anticipated value of N
    real(kind=4) ::  rv1(nmax)
    integer(kind=4) :: i,j,k,l,nm,its,jj
    real(kind=4) :: scale,anorm,c,s,x,y,z
    real(kind=4) :: f,g,h
    !
    if (n.gt.nmax) then
       call cubemain_message(seve%e,'SVDCMP','NMAX dimension too small => Recompilation needed')
       error = .true.
       return
    elseif (m.lt.n) then
       call cubemain_message(seve%e,'SVDCMP','You must add extra zero rows to A')
       error = .true.
       return
    endif
    ! Householder reduction to diagonal form
    g = 0.0
    scale = 0.0
    anorm = 0.0
    do i=1,n
       l=i+1
       rv1(i) = scale*g
       g = 0.0
       s = 0.0
       scale = 0.0
       if (i.le.m) then
          do k=i,m
             scale = scale+abs(a(k,i))
          enddo
          if (scale.ne.0.0) then
             do k=i,m
                a(k,i) = a(k,i)/scale
                s = s+a(k,i)*a(k,i)
             enddo
             f = a(i,i)
             g = -sign(sqrt(s),f)
             h = f*g-s
             a(i,i) = f-g
             if(i.ne.n) then
                do j=l,n
                   s=0.0
                   do k=i,m
                      s = s+a(k,i)*a(k,j)
                   enddo
                   f = s/h
                   do k=i,m
                      a(k,j) = a(k,j)+f*a(k,i)
                   enddo
                enddo
             endif
             do k=i,m
                a(k,i) = scale*a(k,i)
             enddo
          endif
       endif
       w(i) = scale*g
       g = 0.0
       s = 0.0
       scale = 0.0
       if ((i.le.m).and.(i.ne.n)) then
          do k=l,n
             scale = scale+abs(a(i,k))
          enddo
          if (scale.ne.0) then
             do k=l,n
                a(i,k)=a(i,k)/scale
                s=s+a(i,k)*a(i,k)
             enddo
             f=a(i,l)
             g=-sign(sqrt(s),f)
             h=f*g-s
             a(i,l)=f-g
             do k=l,n
                rv1(k)=a(i,k)/h
             enddo
             if (i.ne.m) then
                do j=l,m
                   s=0.0
                   do k=l,n
                      s=s+a(j,k)*a(i,k)
                   enddo
                   do k=l,n
                      a(j,k)=a(j,k)+s*rv1(k)
                   enddo
                enddo
             endif
             do k=l,n
                a(i,k)=scale*a(i,k)
             enddo
          endif
       endif
       anorm = max(anorm,(abs(w(i))+abs(rv1(i))))
    enddo
    ! Accumulation of right hand transformation
    do i=n,1,-1
       if (i.lt.n) then
          if (g.ne.0.0) then
             do j=l,n               ! Double division avoids possible underflow
                v(j,i) = (a(i,j)/a(i,l))/g
             enddo
             do j=l,n
                s=0.0
                do k=l,n
                   s=s+a(i,k)*v(k,j)
                enddo
                do k=l,n
                   v(k,j)=v(k,j)+s*v(k,i)
                enddo
             enddo
          endif
          do j=l,n
             v(i,j) = 0.0
             v(j,i) = 0.0
          enddo
       endif
       v(i,i) = 1.0
       g = rv1(i)
       l = i
    enddo
    ! Accumulation of left hand transformations.
    do i=n,1,-1
       l = i+1
       g = w(i)
       if (i.lt.n) then
          do j=l,n
             a(i,j) = 0.0
          enddo
       endif
       if (g.ne.0.0) then
          g = 1.0/g
          if (i.ne.n) then
             do j=l,n
                s = 0.0
                do k=l,m
                   s = s+a(k,i)*a(k,j)
                enddo
                f = (s/a(i,i))*g
                do k=i,m
                   a(k,j)=a(k,j)+f*a(k,i)
                enddo
             enddo
          endif
          do j=i,m
             a(j,i) = a(j,i)*g
          enddo
       else
          do j=i,m
             a(j,i) = 0.0
          enddo
       endif
       a(i,i) = a(i,i)+1.0
    enddo
    ! Diagonalization of the bidiagonal form
    do k=n,1,-1 ! Loop over singular values
       do its=1,30 ! Loop over allowed iterations
          do l=k,1,-1 ! Test for splitting:
             nm=l-1 ! Note that RV1(1) is always zero
             if ((abs(rv1(l))+anorm).eq.anorm) goto 2
             if ((abs(w(nm))+anorm).eq.anorm) goto 1
          enddo
1         continue
          c=0.0 ! Cancellation of RV1(L), if L>1
          s=1.0
          do i=l,k
             f=s*rv1(i)
             if ((abs(f)+anorm).ne.anorm) then
                g = w(i)
                h = sqrt(f*f+g*g)
                w(i) = h
                h = 1.0/h
                c = (g*h)
                s = -f*h
                do j=1,m
                   y = a(j,nm)
                   z = a(j,i)
                   a(j,nm)=(y*c)+(z*s)
                   a(j,i)=-(y*s)+(z*c)
                enddo
             endif
          enddo
2         continue
          z=w(k)
          if (l.eq.k) then ! Convergence
             if (z.lt.0.0) then ! Singular value is made non negative
                w(k)=-z
                do j=1,n
                   v(j,k)=-v(j,k)
                enddo
             endif
             goto 3
          endif
          if (its.eq.30) then
             call cubemain_message(seve%e,'SVDCMP','No convergence in 30 iterations.')
             error = .true.
             return
          endif
          x=w(l) ! Shift from bottom 2-by-2 minor
          nm = k-1
          y = w(nm)
          g = rv1(nm)
          h = rv1(k)
          f = ((y-z)*(y+z)+(g-h)*(g+h))/(2.0*h*y)
          g = sqrt(f*f+1.0)
          f = ((x-z)*(x+z)+h*((y/(f+sign(g,f)))-h))/x
          ! Next QR transformation
          c=1.0
          s=1.0
          do j=l,nm
             i=j+1
             g=rv1(i)
             y=w(i)
             h=s*g
             g=c*g
             z=sqrt(f*f+h*h)
             rv1(j) = z
             c=f/z
             s=h/z
             f= (x*c)+(g*s)
             g=-(x*s)+(g*c)
             h=y*s
             y=y*c
             do jj=1,n
                x=v(jj,j)
                z=v(jj,i)
                v(jj,j)= (x*c)+(z*s)
                v(jj,i)=-(x*s)+(z*c)
             enddo
             z=sqrt(f*f+h*h)
             w(j) = z
             if (z.ne.0) then ! Rotation can be arbitrary if Z=0
                z=1.0/z
                c=f*z
                s=h*z
             endif
             f= (c*g)+(s*y)
             x=-(s*g)+(c*y)
             do jj=1,m
                y=a(jj,j)
                z=a(jj,i)
                a(jj,j) = (y*c)+(z*s)
                a(jj,i) =-(y*s)+(z*c)
             enddo
          enddo
          rv1(l)=0.0
          rv1(k)=f
          w(k)=x
       enddo
3      continue
    enddo
  end subroutine cubemain_svdcmp
  !
  subroutine cubemain_svbksb(u,w,v,m,n,mp,np,b,x,error)
    !---------------------------------------------------------------------
    ! Singular value decomposition
    !
    ! Solves A.X = B for a vector X, where A is specified by the arrays U,W,V
    ! as returned by SVDCMP.
    !
    ! M and N are the logical dimensions of A, and will be equal for square
    ! matrices. MP and NP are the physical dimensions of A.
    !
    ! B is the input right hand side. X is the output solution vector. No
    ! input quantities are destroyed, so the routine may be called
    ! sequentially with different B's.
    ! ---------------------------------------------------------------------
    integer(kind=chan_k), intent(in)  :: mp        !
    integer(kind=4),      intent(in)  :: np        !
    real(kind=4),         intent(in)  :: u(mp,np)  !
    real(kind=4),         intent(in)  :: w(np)     !
    real(kind=4),         intent(in)  :: v(np,np)  !
    integer(kind=chan_k), intent(in)  :: m         !
    integer(kind=4),      intent(in)  :: n         !
    real(kind=4),         intent(in)  :: b(mp)     !
    real(kind=4),         intent(out) :: x(np)     !
    logical,              intent(out) :: error     !
    !
    integer(kind=4), parameter :: nmax=100
    real(kind=4) :: tmp(nmax),s
    integer(kind=4) :: i,j,jj
    !
    if (n.gt.nmax) then
       call cubemain_message(seve%e,'SVDCMP','NMAX dimension too small -- Will need to recompile.')
       error = .true.
       return
    endif
    ! Calculate UtB
    do j=1,n
       s = 0.
       if (w(j).ne.0.) then
          do i=1,m
             s=s+u(i,j)*b(i)
          enddo
          s = s/w(j)
       endif
       tmp(j) = s
    enddo
    ! Matrix multiply by V to get answer
    do j=1,n
       s = 0.0
       do jj=1,n
          s = s+v(j,jj)*tmp(jj)
       enddo
       x(j) = s
    enddo
  end subroutine cubemain_svbksb
end module cubemain_chebyshev_svd
!
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
