subroutine greg3_spectrum_compute(line,error)
  use image_def
  use sic_types
  use greg_interfaces, except_this=>greg3_spectrum_compute
  use greg_dependencies_interfaces
  use greg_poly
  use gbl_message
  !-----------------------------------------------------------------
  ! @ private
  !
  ! SPECTRUM OutVar Invar [Mask] /MEAN  [Mean|Sum]
  !     /CORNER Blc Trc /Plane First Last
  !
  !   Compute the mean spectrum over the specified region,
  !   bounded by the GreG polygon.
  !
  ! Invar is a SIC Image Variable
  ! Return this in the specified Outvar SIC variable
  !
  ! The idea is that this routine will be callable by GO VIEW as
  !   SPECTRUM MSPECTRE &1 /CORNER IBLC[1] IBLC[2] ITRC[1] ITRC[2]
  !     /PLANE MIPLANE[1] MIPLANE[2 [/MEAN] [/SUM]
  ! Invar will then be VUE, while for command VIEW in Mapping, Invar
  ! is  W. Outvar is MSPECTRE in the GO VIEW application.
  ! Loops can be parallelized for speed.
  !------------------------------------------------------------------
  character(len=*), intent(in) :: line
  logical, intent(out) :: error
  !
  character(len=*), parameter :: rname='SPECTRUM'
  type(gildas) :: hin
  integer :: n, m, nc
  integer(kind=address_length) :: ipin,ipou,ipma,ip
  type(sic_descriptor_t) :: descr, desco, descm
  integer :: ibound(6), ier
  character(len=varname_length) :: invar,ouvar
  integer(kind=4) :: memory(2)
  integer, allocatable :: s0(:)
  real, allocatable :: mask(:,:)
  integer :: nm
  logical :: found
  !
  real(4), save, target :: aire
  integer(kind=4), parameter :: opt_corner=1
  integer(kind=4), parameter :: opt_mean=2
  integer(kind=4), parameter :: opt_plane=3
  integer(kind=4), parameter :: opt_sum=4
  logical :: do_sum
  !
  error = .false.
  if (sic_present(opt_mean,0)) then
    if (sic_present(opt_sum,0)) then
      call greg_message(seve%e,rname,'Conflicting option /MEAN and /SUM')
      error = .true.
      return
    endif
    do_sum = .false.
  else if (sic_present(opt_sum,0)) then
    do_sum = .true.
  else
    call greg_message(seve%e,rname,'Missing option /MEAN or /SUM')
    error = .true.
    return
  endif
  !
  call gildas_null(hin)
  !
  invar = 'W'
  call sic_ch(line,0,2,invar,n,.false.,error)
  m = len_trim(invar)
  call sic_upper(invar)
  ! Allow the name to have the %data postfix
  if (m.gt.5 .and. invar(m-4:m).eq."%DATA") then
    n = m-5
  else
    n = m
  endif
  !
  ! Get dimensions
  error = .true.
  hin%gil%ndim = 3
  call sic_descriptor(invar(1:n)//'%DIM',descr,found)
  if (.not.found) then
    call greg_message(seve%e,rname,'Input variable '//invar(1:m)//' not found')
    return
  endif
  ip = gag_pointer(descr%addr,memory)
  if (index_length.eq.4) then
    call i4toi4(memory(ip),hin%gil%dim,3)
  else
    call i8toi8(memory(ip),hin%gil%dim,3)
  endif
  !
  ! Get conversion formula
  call sic_descriptor(invar(1:n)//'%CONVERT',descr,found)
  if (.not.found) return
  ip = gag_pointer(descr%addr,memory)
  call r8tor8(memory(ip),hin%gil%convert,12)
  !
  ! Get Blanking
  call sic_descriptor(invar(1:n)//'%BLANK',descr,found)
  if (.not.found) return
  ip = gag_pointer(descr%addr,memory)
  call r4tor4(memory(ip),hin%gil%bval,2)
  !
  ! Get input array
  call sic_descriptor(invar(1:m),descr,found)
  if (.not.found) return
  ipin = gag_pointer(descr%addr,memory)
  !
  ! Get output array
  call sic_ch(line,0,1,ouvar,n,.true.,error)
  if (error) return
  call sic_descriptor(ouvar(1:n),desco,found)
  if (.not.found) then
    call greg_message(seve%e,rname,'Output variable not found')
    error = .true.
    return
  endif
  ipou = gag_pointer(desco%addr,memory)
  !
  ! Get mask if any
  if (sic_present(0,3)) then
    call sic_ch(line,0,3,ouvar,n,.true.,error)
    if (error) return
    call sic_descriptor(ouvar(1:n),descm,found)
    if (.not.found) then
      call greg_message(seve%e,rname,'Mask variable not found')
      error = .true.
      return
    endif
    ipma = gag_pointer(descm%addr,memory)
    nm = max(descm%dims(3),1)
  else
    allocate(mask(hin%gil%dim(1),hin%gil%dim(2)),stat=ier)
    if (ier.ne.0) then
      call greg_message(seve%e,rname,'Mask allocation error')
      error = .true.
      return
    endif
    mask = 1.0
    nm = 1
    ipma = gag_pointer(locwrd(mask),memory)
  endif
  !
  ! /CORNER option handling
  ibound(1:2) = 0
  call sic_i4(line,opt_corner,1,ibound(1),.false.,error)
  if (error) return
  call sic_i4(line,opt_corner,2,ibound(3),.false.,error)
  if (error) return
  ier = gdf_range(ibound(1:2),hin%gil%dim(1))
  !
  ibound(3:4) = 0
  call sic_i4(line,opt_corner,3,ibound(3),.false.,error)
  if (error) return
  call sic_i4(line,opt_corner,4,ibound(4),.false.,error)
  if (error) return
  ier = gdf_range(ibound(3:4),hin%gil%dim(2))
  !
  ! /PLANE option handling
  ibound(5:6) = 0
  call sic_i4(line,opt_plane,1,ibound(5),.false.,error)
  if (error) return
  call sic_i4(line,opt_plane,2,ibound(6),.false.,error)
  if (error) return
  ier = gdf_range(ibound(5:6),hin%gil%dim(3))
  !
  nc = ibound(6)-ibound(5)+1
  if (nc.ne.desco%size) then
    call greg_message(seve%e,rname,'Spectrum dimension mismatch')
    !!Print *,'nc ',nc,ibound(5:6),desco%size
    error = .true.
    return
  else
    write(*,'(A,I6)') 'I-SPECTRUM,  Output number of channels ',nc
    write(*,'(A,I6,I6)') 'I-SPECTRUM,  Number of pixels ',hin%gil%dim(1),hin%gil%dim(2)
  endif
  !
  allocate(s0(nc),stat=ier)
  if (ier.ne.0) then
    call greg_message(seve%e,rname,'Allocation error')
    error = .true.
    return
  endif
  !
  call greg_drive_spectre(nc,memory(ipou),s0,aire,hin,memory(ipin),ibound, &
    &   memory(ipma),nm,gpoly,do_sum)
  if (sic_varexist('POLY%AREA')) call sic_delvariable('POLY%AREA',.false.,error)
  call sic_def_real ('POLY%AREA',aire,0,1,.true.,error)
  !
end subroutine greg3_spectrum_compute
!
subroutine greg_drive_spectre(nc,s1,s0,aire,hin,din,ibound,rmask,nm,poly,do_sum)
  use image_def
  use greg_interfaces
  use greg_types
  use gbl_message
  !$ use omp_lib
  !
  ! @ no-interfaces
  !
  integer, intent(in) :: nc                          ! Number of channels
  integer(4), intent(out) :: s0(nc)                  ! Area of valid pixels per plane
  real(4), intent(out) :: s1(nc)                     ! Integrated area per plane
  real(4), intent(out) :: aire                       ! Area of support
  type(gildas), intent(inout) :: hin                 ! Image header
  real(4), intent(in) :: din(hin%gil%dim(1),hin%gil%dim(2),hin%gil%dim(3)) ! Cube
  integer(kind=4), intent(inout) :: ibound(6)        ! Cube boundaries
  integer(kind=4), intent(in) :: nm                  ! Mask last dim
  real, intent(in) :: rmask(hin%gil%dim(1),hin%gil%dim(2),nm) ! Real mask (0 or 1)
  type (polygon_t), intent(in) :: poly               ! Polygon
  logical, intent(in) :: do_sum                      ! Sum or Mean
  !
  integer(kind=index_length) :: i,j,imin,imax,jmin,jmax,kmin,kmax,nk
  real(8) :: x,y
  integer :: k
  !
  real, allocatable :: t_s1(:,:),t_s0(:,:)
  integer, allocatable :: t_nk(:)
  integer :: nt, ithread, ier
  character(len=60) :: mess
  !
  if (poly%ngon.gt.0) then
    !
    ! Avoid exploring all the Map by finding IMIN,IMAX,JMIN,JMAX
    if (hin%gil%inc(1).gt.0.) then
      imin = max (1,    int((poly%xgon1-hin%gil%val(1))/hin%gil%inc(1)+hin%gil%ref(1)) )
      imax = min (hin%gil%dim(1),int((poly%xgon2-hin%gil%val(1))/hin%gil%inc(1)+hin%gil%ref(1))+1 )
    else
      imin = max (1,    int((poly%xgon2-hin%gil%val(1))/hin%gil%inc(1)+hin%gil%ref(1)) )
      imax = min (hin%gil%dim(1),int((poly%xgon1-hin%gil%val(1))/hin%gil%inc(1)+hin%gil%ref(1))+1 )
    endif
    if (hin%gil%inc(2).gt.0.) then
      jmin = max (1,    int((poly%ygon1-hin%gil%val(2))/hin%gil%inc(2)+hin%gil%ref(2)) )
      jmax = min (hin%gil%dim(2),int((poly%ygon2-hin%gil%val(2))/hin%gil%inc(2)+hin%gil%ref(2))+1 )
    else
      jmin = max (1,    int((poly%ygon2-hin%gil%val(2))/hin%gil%inc(2)+hin%gil%ref(2)) )
      jmax = min (hin%gil%dim(2),int((poly%ygon1-hin%gil%val(2))/hin%gil%inc(2)+hin%gil%ref(2))+1 )
    endif
    ibound(1) = max(ibound(1),imin)
    ibound(2) = min(ibound(2),imax)
    ibound(3) = max(ibound(3),jmin)
    ibound(4) = min(ibound(4),jmax)
  endif
  !
  s0 = 0.
  s1 = 0.
  nk = 0
  !
  ! Avoid exploring all the Map by finding IMIN,IMAX,JMIN,JMAX
  imin = ibound(1)
  imax = ibound(2)
  jmin = ibound(3)
  jmax = ibound(4)
  kmin = ibound(5)
  kmax = ibound(6)
  !
  ! Now explore a reasonable part of the map
  if (hin%gil%eval.lt.0.0) then
    !
    ! No blanking: just go fast...
    nt = 1
    !$ nt = omp_get_max_threads()
    allocate(t_s1(nc,nt),t_nk(nt),stat=ier)
    if (nm.gt.1) then
      allocate(t_s0(nc,nt),stat=ier)
      t_s0 = 0.0
    endif
    t_nk = 0
    t_s1 = 0.0
    ithread = 1
    !$OMP PARALLEL DEFAULT(none) SHARED(t_s0,t_s1,poly,rmask,din,nt) &
    !$OMP  & SHARED(kmin,kmax,jmin,jmax,imin,imax) &
    !$OMP  & PRIVATE(i,j,x,y,ithread) &
    !$OMP  & SHARED(nm, hin, t_nk, mess)
    !$  ithread = omp_get_thread_num()+1
    !$  if (ithread.eq.1) then
    !$    nt = omp_get_num_threads()
    !$    if (nt.gt.1) then
    !$      write(mess,'(A,I0,A)') 'Using ',nt,' threads'
    !$      call greg_message(seve%d,'SPECTRUM',mess)
    !$    endif
    !$  endif
    if (nm.le.1) then
      ! Global mask
      !$OMP DO COLLAPSE(2)
      do j=jmin,jmax
        do i=imin,imax
          x = (i-hin%gil%ref(1))*hin%gil%inc(1) + hin%gil%val(1)
          y = (j-hin%gil%ref(2))*hin%gil%inc(2) + hin%gil%val(2)
          if (greg_poly_inside(x,y,poly).and.rmask(i,j,1).ne.0.0) then
            t_nk(ithread) = t_nk(ithread)+1
            t_s1(:,ithread) = t_s1(:,ithread) + din(i,j,kmin:kmax)
          endif
        enddo
      enddo
      !$OMP END DO
    else
      ! Per plane mask
      !$OMP DO COLLAPSE(2)
      do j=jmin,jmax
        do i=imin,imax
          x = (i-hin%gil%ref(1))*hin%gil%inc(1) + hin%gil%val(1)
          y = (j-hin%gil%ref(2))*hin%gil%inc(2) + hin%gil%val(2)
          if (greg_poly_inside(x,y,poly)) then
            t_s0(:,ithread) = t_s0(:,ithread) + rmask(i,j,kmin:kmax)
            t_nk(ithread) = t_nk(ithread)+1
            t_s1(:,ithread) = t_s1(:,ithread) + rmask(i,j,kmin:kmax)*din(i,j,kmin:kmax)
          endif
        enddo
      enddo
      !$OMP END DO
    endif
    !$OMP END PARALLEL
    nk = 0
    do i=1,nt
      nk = nk + t_nk(i)
      s1 = s1 + t_s1(:,i)
    enddo
    if (.not.do_sum) then
      ! Renormalize to mean
      if (nm.eq.1) then
        s1 = s1/nk
      else
        ! Per plane mask
        do i=1,nt
          s0 = s0 + t_s0(:,i)
        enddo
        where (s0.ne.0) s1 = s1/s0
      endif
    endif
  else
    nt = 1
    !$ nt = omp_get_max_threads()
    allocate(t_s1(nc,nt),t_nk(nt),t_s0(nc,nt),stat=ier)
    t_s0 = 0.0
    t_nk = 0
    t_s1 = 0.0
    ithread = 1
    !$OMP PARALLEL DEFAULT(none) SHARED(t_s0,t_s1,poly,rmask,din,nt) &
    !$OMP  & SHARED(kmin,kmax,jmin,jmax,imin,imax) &
    !$OMP  & PRIVATE(i,j,x,y,ithread) &
    !$OMP  & SHARED(nm, hin, t_nk, nc, mess)
    !$  ithread = omp_get_thread_num()+1
    !$  if (ithread.eq.1) then
    !$    nt = omp_get_num_threads()
    !$    if (nt.gt.1) then
    !$      write(mess,'(A,I0,A)') 'Using ',nt,' threads'
    !$      call greg_message(seve%d,'SPECTRUM',mess)
    !$    endif
    !$  endif
    ! With blanking value
    if (nm.le.1) then
      !$OMP DO COLLAPSE(2)
      do j=jmin,jmax
        do i=imin,imax
          x = (i-hin%gil%ref(1))*hin%gil%inc(1) + hin%gil%val(1)
          y = (j-hin%gil%ref(2))*hin%gil%inc(2) + hin%gil%val(2)
          if (rmask(i,j,1).ne.0.0) then
            if (greg_poly_inside(x,y,poly)) then
              t_nk(ithread) = t_nk(ithread)+1
              do k=1,nc
                if (abs(din(i,j,kmin+k-1)-hin%gil%bval).gt.hin%gil%eval) then
                  t_s0(k,ithread) = t_s0(k,ithread) + 1
                  t_s1(k,ithread) = t_s1(k,ithread) + din(i,j,kmin+k-1)
                endif
              enddo
            endif
          endif
        enddo
      enddo
      !$OMP END DO
    else
      !$OMP DO COLLAPSE(2)
      do j=jmin,jmax
        do i=imin,imax
          x = (i-hin%gil%ref(1))*hin%gil%inc(1) + hin%gil%val(1)
          y = (j-hin%gil%ref(2))*hin%gil%inc(2) + hin%gil%val(2)
          if (greg_poly_inside(x,y,poly)) then
            t_nk(ithread) = t_nk(ithread)+1
            do k=1,nc
              if (abs(din(i,j,kmin+k-1)-hin%gil%bval).gt.hin%gil%eval) then
                t_s0(k,ithread) = t_s0(k,ithread) + rmask(i,j,kmin+k-1)
                t_s1(k,ithread) = t_s1(k,ithread) + din(i,j,kmin+k-1)*rmask(i,j,kmin+k-1)
              endif
            enddo
          endif
        enddo
      enddo
      !$OMP END DO
    endif
    !$OMP END PARALLEL
    nk = 0
    do i=1,nt
      nk = nk + t_nk(i)
      s1 = s1 + t_s1(:,i)
    enddo
    !
    ! Get sum or Mean - Either case is valid, depending on do_sum...
    if (.not.do_sum) then
      if (nm.eq.1) then
        s1 = s1/nk
      else
        ! Per plane mask
        do i=1,nt
          s0 = s0 + t_s0(:,i)
        enddo
        where (s0.ne.0) s1 = s1/s0
      endif
    endif
  endif
  !
  if (do_sum) s1 = s1 * abs(hin%gil%inc(1) * hin%gil%inc(2))
  aire = nk * abs(hin%gil%inc(1) * hin%gil%inc(2))
end subroutine greg_drive_spectre
