!
!     $Id: modulePakoGlobalParameters.f90,v 1.2 2005/06/22 14:46:29 ungerech Exp $
!     
!----------------------------------------------------------------------
!     PAKO 
!
! <DOCUMENTATION name="moduleGlobalParameters">
!
! constants  (parameters in the FORTRAN sense),
!    including unit conversion factors
!
! integer codes for SIC error messages
! 
! character strings for "choice" keywords
!    lenCH : length of these strings 
!    they come in 2 flavors:
!    *%* :  
!       derived data types, 
!          e.g., rec%Bolo                =   "Bolometer"
!          short, recognizable by their peculiar construction
!    *Choices(i) :
!       elements of arrays of char strings, 
!          e.g., receiverChoices(iBolo)  =   "Bolometer"
!          more elaborate, suitable for general matching routine
!    nDim*Choices : number of available choices in each case
!    i*           : predefined indices 
!    The strings normally are in mixed case, e.g., "trueHorizon",
!    but use standard spelling, e.g., "Doppler" or standard 
!    abbreviations, e.g., "HERA", where appropriate.
!
! print* subroutines to print the choice keywords
!
! </DOCUMENTATION> <!-- name="moduleGlobalParameters" -->
!
Module modulePakoGlobalParameters
  !
  Implicit None
  Save
  Public
  !
  ! *** Sizes, kinds for variables ***     
  Include 'inc/parameters/sizes.inc'
  !
  Real(Kind=kindDouble),  Parameter :: Pi     = 3.141592653589793D0
  !
  ! *** conversion factors
  !   * for frequencies:
  Real(Kind=kindDouble),  Parameter :: MHz      = 1.0D6
  Real(Kind=kindDouble),  Parameter :: MHzToGhz = 1.0D-3
  !
  !   * to rad:
  Real(Kind=kindDouble),  Parameter :: rad    = 1.0D0
  Real(Kind=kindDouble),  Parameter :: radian = 1.0D0
  Real(Kind=kindDouble),  Parameter :: arcsec = Pi/(180.0D0*60.0D0*60.0D0)
  Real(Kind=kindDouble),  Parameter :: arcmin = Pi/(180.0D0*60.0D0)
  Real(Kind=kindDouble),  Parameter :: deg    = Pi/(180.0D0)
  Real(Kind=kindDouble),  Parameter :: degree = Pi/(180.0D0)
  Real(Kind=kindDouble),  Parameter :: degs   = Pi/(180.0D0)
  Real(Kind=kindDouble),  Parameter :: second = Pi/( 12.0D0*60.0D0*60.0D0)
  Real(Kind=kindDouble),  Parameter :: minute = Pi/( 12.0D0*60.0D0)
  Real(Kind=kindDouble),  Parameter :: hour   = Pi/( 12.0D0)
  !
  ! *** for SIC messages:
  !
  Integer, Parameter :: severityI = 1
  Integer, Parameter :: severityW = 2
  Integer, Parameter :: severityE = 3
  Integer, Parameter :: severityF = 4
  !
  Integer, Parameter :: priorityI = severityI*2
  Integer, Parameter :: priorityW = severityW*2
  Integer, Parameter :: priorityE = severityE*2
  Integer, Parameter :: priorityF = severityF*2
  !
  ! *** CHARACTER STRINGS / KEYWORDS FOR CHOICES
  !
  ! *** receivers
  !
  Integer, Parameter      :: nDimReceiverChoices  =  12
  Integer, Parameter      :: iA100          =  1       !   "A100"            
  Integer, Parameter      :: iA230          =  2       !   "A230"              
  Integer, Parameter      :: iB100          =  3       !   "B100"              
  Integer, Parameter      :: iB230          =  4       !   "B230"              
  Integer, Parameter      :: iC150          =  5       !   "C150"              
  Integer, Parameter      :: iC270          =  6       !   "C270"              
  Integer, Parameter      :: iD150          =  7       !   "D150"              
  Integer, Parameter      :: iD270          =  8       !   "D270"              
  Integer, Parameter      :: iBolo          =  9       !   "Bolometer"         
  Integer, Parameter      :: iHERA          = 10       !   "HERAall"              
  Integer, Parameter      :: iHERA1         = 11       !   "HERA1vertical"    
  Integer, Parameter      :: iHERA2         = 12       !   "HERA2horizontal"  
  !
  Type :: receiverChoicesType
     Character(len=lenCh) ::  A100                     =   "A100"            
     Character(len=lenCh) ::  A230                     =   "A230"            
     Character(len=lenCh) ::  B100                     =   "B100"            
     Character(len=lenCh) ::  B230                     =   "B230"            
     Character(len=lenCh) ::  C150                     =   "C150"            
     Character(len=lenCh) ::  C270                     =   "C270"            
     Character(len=lenCh) ::  D150                     =   "D150"            
     Character(len=lenCh) ::  D270                     =   "D270"            
     Character(len=lenCh) ::  Bolo                     =   "Bolometer"       
     Character(len=lenCh) ::  HERA                     =   "HERAall"            
     Character(len=lenCh) ::  HERA1                    =   "HERA1vertical"  
     Character(len=lenCh) ::  HERA2                    =   "HERA2horizontal"
  End Type receiverChoicesType
  !
  Type (receiverChoicesType), Parameter ::                                   &
       &  rec =                                                              &
       &                         receiverChoicesType (     "A100"          , &
       &                                                   "A230"          , &
       &                                                   "B100"          , &
       &                                                   "B230"          , &
       &                                                   "C150"          , &
       &                                                   "C270"          , &
       &                                                   "D150"          , &
       &                                                   "D270"          , &
       &                                                   "Bolometer"     , &
       &                                                   "HERAall"       , &
       &                                                   "HERA1vertical" , &
       &                                                   "HERA2horizontal")
  !
  Character(len=lenCh), Dimension(nDimReceiverChoices), Parameter ::         &
       &  receiverChoices =                                                  &
       &               (/ rec%A100   ,                                       &
       &                  rec%A230   ,                                       &
       &                  rec%B100   ,                                       &
       &                  rec%B230   ,                                       &
       &                  rec%C150   ,                                       &
       &                  rec%C270   ,                                       &
       &                  rec%D150   ,                                       &
       &                  rec%D270   ,                                       &
       &                  rec%Bolo   ,                                       &
       &                  rec%HERA   ,                                       &
       &                  rec%HERA1  ,                                       &
       &                  rec%HERA2  /)
  !
  ! *   sidebands
  !
  Integer, Parameter      :: nDimSidebandChoices  =  2
  Integer, Parameter      :: ilsb           =  1       !   "LSB"
  Integer, Parameter      :: iusb           =  2       !   "USB"
  !
  Type :: sidebandChoicesType
     Character(len=lenCh) ::  lsb                      =   "LSB"
     Character(len=lenCh) ::  usb                      =   "USB"
  End Type sidebandChoicesType
  !
  Type (sidebandChoicesType), Parameter ::                                   &
       &  sb =                                                               &
       &                         sidebandChoicesType (     "LSB"     ,       &
       &                                                   "USB"  )
  !
  Character(len=lenCh), Dimension(nDimSidebandChoices), Parameter ::         &
       &  sidebandChoices =                                                  &
       &                (/ sb%lsb    ,                                       &
       &                   sb%usb   /)
  !
  ! *   doppler
  !
  Integer, Parameter      :: nDimDopplerChoices  =  2
  Integer, Parameter      :: idoppler       =  1       !   "Doppler"
  Integer, Parameter      :: ifixed         =  2       !   "fixed"  
  !
  Type :: dopplerChoicesType
     Character(len=lenCh) ::  doppler                  =   "Doppler"
     Character(len=lenCh) ::  fixed                    =   "fixed"  
  End Type dopplerChoicesType
  !
  Type (dopplerChoicesType), Parameter ::                                    &
       &  doppler =                                                          &
       &                          dopplerChoicesType (     "Doppler"  ,      &
       &                                                   "fixed"   )
  !
  Character(len=lenCh), Dimension(nDimDopplerChoices), Parameter ::          &
       &  dopplerChoices =                                                   &
       &           (/ doppler%doppler ,                                      &
       &              doppler%fixed   /)
  !
  ! *   width
  !
  Integer, Parameter      :: nDimWidthChoices  =  2
  Integer, Parameter      :: inarrow        =  1       !   "narrow"
  Integer, Parameter      :: iwide          =  2       !   "wide"  
  !
  Type :: widthChoicesType
     Character(len=lenCh) ::  narrow                   =   "narrow" 
     Character(len=lenCh) ::  wide                     =   "wide"  
  End Type widthChoicesType
  !
  Type (widthChoicesType), Parameter ::                                      &
       &  width =                                                            &
       &                            widthChoicesType (     "narrow"  ,       &
       &                                                   "wide"    )
  !
  Character(len=lenCh), Dimension(nDimWidthChoices), Parameter ::            &
       &  widthChoices =                                                     &
       &             (/ width%narrow ,                                       &
       &                width%wide   /)
  !
  ! *   bandWidth
  !
  Integer, Parameter      :: nDimBWChoices  =  2
  Integer, Parameter      :: iBWNarrow      =  1
  Integer, Parameter      :: iBWWide        =  2
  !
  Type :: BWChoicesType
     Real(kind=kindDouble) ::   narrow      =  0.500D0  ! narrow mode high freq.      
     Real(kind=kindDouble) ::   wide        =  1.000D0  ! wide   mode high freq.      
  End Type BWChoicesType
  !
  Type (BWChoicesType), Parameter ::                                         &
       &  BW =                                                               &
       &                       BWChoicesType ( 0.500D0,                      &
       &                                       1.000D0  )
  !
  Real(kind=kindDouble), Dimension(nDimBWChoices), Parameter ::              &
       &  BWChoices =                                                        &
       &           (/        BW%narrow,                                      &
       &                     BW%wide /)
  !
  ! *   scale
  !
  Integer, Parameter      :: nDimScaleChoices  =  2
  Integer, Parameter      :: iScaleAntenna  =  1       !   "antenna"
  Integer, Parameter      :: iScaleBeam     =  2       !   "beam"  
  !
  Type :: scaleChoicesType
     Character(len=lenCh) ::        antenna            =   "antenna" 
     Character(len=lenCh) ::        beam               =   "beam"  
  End Type scaleChoicesType
  !
  Type (scaleChoicesType), Parameter ::                                      &
       &  scale =                                                            &
       &                            scaleChoicesType (     "antenna" ,       &
       &                                                   "beam"    )
  !
  Character(len=lenCh), Dimension(nDimScaleChoices), Parameter ::            &
       &  scaleChoices =                                                     &
       &             (/       scale%antenna ,                                &
       &                      scale%beam   /)
  !
  ! *   polarization
  !
  Integer, Parameter      :: nDimPolarizationChoices  =  2
  Integer, Parameter      :: ihorizontal    =  1       !   "horizontal"
  Integer, Parameter      :: ivertical      =  2       !   "vertical"  
  !
  Type :: polarizationChoicesType
     Character(len=lenCh) ::  hor                      =   "horizontal"
     Character(len=lenCh) ::  ver                      =   "vertical"  
  End Type polarizationChoicesType
  !
  Type (polarizationChoicesType), Parameter ::                               &
       &  pol =                                                              &
       &                     polarizationChoicesType (     "horizontal",     &
       &                                                   "vertical"   )
  !
  Character(len=lenCh), Dimension(nDimPolarizationChoices), Parameter ::     &
       &  polarizationChoices =                                              &
       &           (/     pol%hor      ,                                     &
       &                  pol%ver      /)
  !
  ! *   IF
  !
  Integer, Parameter      :: nDimIFChoices  =  3
  Integer, Parameter      :: iIFNarrow      =  1
  Integer, Parameter      :: iIFWide        =  2
  Integer, Parameter      :: iIFlow         =  3
  !
  Type :: IFChoicesType
     Real(kind=kindDouble) ::   narrow      =  4.258D0  ! narrow mode high freq.      
     Real(kind=kindDouble) ::   wide        =  4.002D0  ! wide   mode high freq.      
     Real(kind=kindDouble) ::   low         =  1.498D0  !             low  freq. (3mm)
  End Type IFChoicesType
  !
  Type (IFChoicesType), Parameter ::                                         &
       &  If =                                                               &
       &                       IFChoicesType ( 4.258D0,                      &
       &                                       4.002D0,                      &
       &                                       1.498D0  )
  !
  Real(kind=kindDouble), Dimension(nDimIFChoices), Parameter ::              &
       &  IFChoices =                                                        &
       &           (/      IF%narrow,                                        &
       &                   IF%wide,                                          &
       &                   IF%low  /)
  !
  ! *   IF2
  !
  Integer, Parameter      :: nDimIF2Choices  =  3
  Integer, Parameter      :: iIF2Narrow     =  1
  Integer, Parameter      :: iIF2Wide       =  2
  Integer, Parameter      :: iIF2low        =  3
  !
  Type :: IF2ChoicesType
     Real(kind=kindDouble) ::    narrow     =  0.342D0  ! narrow mode high freq.      
     Real(kind=kindDouble) ::    wide       =  0.598D0  ! wide   mode high freq.      
     Real(kind=kindDouble) ::    low        =  0.342D0  !             low  freq. (3mm)
  End Type IF2ChoicesType
  !
  Type (IF2ChoicesType), Parameter ::                                        &
       &  IF2 =                                                              &
       &                      IF2ChoicesType ( 0.342D0,                      &
       &                                       0.598D0,                      &
       &                                       0.342D0  )
  !
  Real(kind=kindDouble), Dimension(nDimIF2Choices), Parameter ::             &
       &  IF2Choices =                                                       &
       &           (/        IF2%narrow,                                     &
       &                     IF2%wide,                                       &
       &                     IF2%low  /)
  !
  ! *   Bolometers
  !
  Integer, Parameter      :: nDimBolometerChoices  =  2
  Integer, Parameter      :: ib37           =  1       !   "MAMBO1"
  Integer, Parameter      :: ib117          =  2       !   "MAMBO2"
  !
  Type :: bolometerChoicesType
     Character(len=lenCh) ::  b37                      =   "MAMBO1" 
     Character(len=lenCh) ::  b117                     =   "MAMBO2"
  End Type bolometerChoicesType
  !
  Type (bolometerChoicesType), Parameter ::                                  &
       &  bolo =                                                             &
       &                        bolometerChoicesType (     "MAMBO1"     ,    &  
       &                                                   "MAMBO2"  )
  !
  Character(len=lenCh), Dimension(nDimBolometerChoices), Parameter ::        &
       &  bolometerChoices =                                                 &
       &              (/ bolo%b37    ,                                       &
       &                 bolo%b117   /)
  !
  ! *** Backends
  !
  Integer, Parameter      :: nDimBackendChoices  =  7
  Integer, Parameter      :: iContinuum     =  1       !   "Continuum"
  Integer, Parameter      :: iFB4MHz        =  2       !   "4MHz"     
  Integer, Parameter      :: iFB1MHz        =  3       !   "1MHz"     
  Integer, Parameter      :: iFB100kHz      =  4       !   "100kHz"   
  Integer, Parameter      :: iWILMA         =  5       !   "WILMA"    
  Integer, Parameter      :: iVESPA         =  6       !   "VESPA"
  Integer, Parameter      :: iABBA          =  7       !   "ABBA"
  !
  Type :: backendChoicesType
     Character(len=lenCh) ::  Continuum                =   "Continuum"
     Character(len=lenCh) ::  FB4MHz                   =   "4MHz"     
     Character(len=lenCh) ::  FB1MHz                   =   "1MHz"     
     Character(len=lenCh) ::  FB100kHz                 =   "100kHz"   
     Character(len=lenCh) ::  WILMA                    =   "WILMA"    
     Character(len=lenCh) ::  VESPA                    =   "VESPA"
     Character(len=lenCh) ::  ABBA                     =   "ABBA"
  End Type backendChoicesType
  !
  Type (backendChoicesType), Parameter ::                                    &
       &  bac =                                                              &
       &                          backendChoicesType (     "Continuum"   ,   &
       &                                                   "4MHz"        ,   &
       &                                                   "1MHz"        ,   &
       &                                                   "100kHz"      ,   &
       &                                                   "WILMA"       ,   &
       &                                                   "VESPA"       ,   &
       &                                                   "ABBA"    )          ! not directly used in paKo
  !
  Character(len=lenCh), Dimension(nDimBackendChoices), Parameter ::          &
       &  backendChoices =                                                   &
       &           (/     bac%Continuum ,                                    &
       &                  bac%FB4MHz    ,                                    &
       &                  bac%FB1MHz    ,                                    &
       &                  bac%FB100kHz  ,                                    &
       &                  bac%WILMA     ,                                    &
       &                  bac%VESPA     ,                                    &
       &                  bac%ABBA       /)
  !
  ! *   backendModes (VESPA)
  !
  Integer, Parameter      :: nDimBEmodeChoices  =  3
  Integer, Parameter      :: iSimp          =  1       !   "simple"      
  Integer, Parameter      :: iPara          =  2       !   "parallel"    
  Integer, Parameter      :: iPola          =  3       !   "polarization"
  !
  Type :: BEmodeChoicesType
     Character(len=lenCh) ::  Simp                     =   "simple"      
     Character(len=lenCh) ::  Para                     =   "parallel"    
     Character(len=lenCh) ::  Pola                     =   "polarization"
  End Type BEmodeChoicesType
  !
  Type (BEmodeChoicesType), Parameter ::                                     &
       &  BEmode =                                                           &
       &                          BEmodeChoicesType (       "simple"      ,  &
       &                                                    "parallel"    ,  &
       &                                                    "polarization" )
  !
  Character(len=lenCh), Dimension(nDimBEmodeChoices), Parameter ::           &
       &  BEmodeChoices =                                                    &
       &           (/  BEmode%Simp      ,                                    &
       &               BEmode%Para      ,                                    &
       &               BEmode%Pola      /)
  !
  ! *** switchingModes
  !
  Integer, Parameter      :: nDimSwModeChoices  =  4
  Integer, Parameter      :: iSwTotal       =  1       !   "totalPower"        
  Integer, Parameter      :: iSwBeam        =  2       !   "beamSwitching"     
  Integer, Parameter      :: iSwWobb        =  3       !   "wobblerSwitching"  
  Integer, Parameter      :: iSwFreq        =  4       !   "frequencySwitching"
  !
  Type :: swModeChoicesType
     Character(len=lenCh) ::    Total                  =   "totalPower"        
     Character(len=lenCh) ::    Beam                   =   "beamSwitching"     
     Character(len=lenCh) ::    Wobb                   =   "wobblerSwitching"  
     Character(len=lenCh) ::    Freq                   =   "frequencySwitching"
  End Type swModeChoicesType
  !
  Type (swModeChoicesType), Parameter ::                                         &
       &  swMode =                                                               &
       &                          swModeChoicesType (      "totalPower"      ,   &
       &                                                   "beamSwitching"   ,   &
       &                                                   "wobblerSwitching",   &
       &                                                   "frequencySwitching" )
  !
  Character(len=lenCh), Dimension(nDimswModeChoices), Parameter ::           &
       &  swModeChoices =                                                    &
       &           (/    swMode%Total     ,                                  &
       &                 swMode%Beam      ,                                  &
       &                 swMode%Wobb      ,                                  &
       &                 swMode%Freq      /)
  !
  ! *** observingModes
  !
  Integer, Parameter      :: nDimOMChoices  =  13
  Integer, Parameter      :: iOmCal         =  1       !   "calibrate"    
  Integer, Parameter      :: iOmPointing    =  2       !   "pointing"     
  Integer, Parameter      :: iOmFocus       =  3       !   "focus"        
  Integer, Parameter      :: iOmTip         =  4       !   "tip"          
  Integer, Parameter      :: iOmTrack       =  5       !   "track"        
  Integer, Parameter      :: iOmOnOff       =  6       !   "onOff"        
  Integer, Parameter      :: iOmOtfMap      =  7       !   "onTheFlyMap"  
  Integer, Parameter      :: iOmRaster      =  8       !   "raster"       
  Integer, Parameter      :: iOmOtfTip      =  9       !   "onTheFlyTip"  
  Integer, Parameter      :: iOmCross       = 10       !   "crossRaster"  
  Integer, Parameter      :: iOmOtfCross    = 11       !   "onTheFlyCross"
  Integer, Parameter      :: iOmOtfFocus    = 12       !   "onTheFlyFocus"
  Integer, Parameter      :: iOmVLBI        = 13       !   "VLBI"
  !
  Type :: OMChoicesType
     Character(len=lenCh) ::    Cal                    =   "calibrate"    
     Character(len=lenCh) ::    Pointing               =   "pointing"     
     Character(len=lenCh) ::    Focus                  =   "focus"        
     Character(len=lenCh) ::    Tip                    =   "tip"          
     Character(len=lenCh) ::    Track                  =   "track"        
     Character(len=lenCh) ::    OnOff                  =   "onOff"        
     Character(len=lenCh) ::    OtfMap                 =   "onTheFlyMap"  
     Character(len=lenCh) ::    Raster                 =   "raster"       
     Character(len=lenCh) ::    OtfTip                 =   "onTheFlyTip"  
     Character(len=lenCh) ::    Cross                  =   "crossRaster"  
     Character(len=lenCh) ::    OtfCross               =   "onTheFlyCross"
     Character(len=lenCh) ::    OtfFocus               =   "onTheFlyFocus"
     Character(len=lenCh) ::    VLBI                   =   "VLBI"
  End Type OMChoicesType
  !
  Type (OMChoicesType), Parameter ::                                         &
       &  OM =                                                               &
       &                        OMChoicesType (            "calibrate"     , &
       &                                                   "pointing"      , &                   
       &                                                   "focus"         , &                    
       &                                                   "tip"           , &                   
       &                                                   "track"         , &                   
       &                                                   "onOff"         , &                   
       &                                                   "onTheFlyMap"   , &                
       &                                                   "raster"        , &   !  TBD soon   
       &                                                   "onTheFlyTip"   , &   !  TBD soon ?  
       &                                                   "crossRaster"   , &   !  future
       &                                                   "onTheFlyCross" , &   !  future   
       &                                                   "onTheFlyFocus" , &   !  future?
       &                                                   "VLBI" )     
  !
  Character(len=lenCh), Dimension(nDimOMChoices), Parameter ::               &
       &  OMChoices =                                                        &
       &           (/        OM%Cal                 ,                        &
       &                     OM%Pointing            ,                        &
       &                     OM%Focus               ,                        &
       &                     OM%Tip                 ,                        &
       &                     OM%Track               ,                        &
       &                     OM%OnOff               ,                        &
       &                     OM%OtfMap              ,                        &
       &                     OM%Raster              ,                        &
       &                     OM%OtfTip              ,                        &
       &                     OM%Cross               ,                        &
       &                     OM%OtfCross            ,                        &
       &                     OM%OtfFocus            ,                        &
       &                     OM%VLBI                /)
  !
  ! *   subscans
  !
  Integer, Parameter      :: nDimSubscanChoices  =  11
  Integer, Parameter      :: iSSca          =  1       !   "calAmbient"  
  Integer, Parameter      :: iSScc          =  2       !   "calCold"     
  Integer, Parameter      :: iSScs          =  3       !   "calSky"      
  Integer, Parameter      :: iSScp          =  4       !   "calPaddle"   
  Integer, Parameter      :: iSSam          =  5       !   "airmass"     
  Integer, Parameter      :: iSSonFocus     =  6       !   "onFocus"     
  Integer, Parameter      :: iSStrack       =  7       !   "track"       
  Integer, Parameter      :: iSSotf         =  8       !   "onTheFly"    
  Integer, Parameter      :: iSSon          =  9       !   "on"          
  Integer, Parameter      :: iSSref         = 10       !   "reference" 
  Integer, Parameter      :: iSSVLBI        = 11       !   "VLBI" 
  !
  Type :: subscanChoicesType
     Character(len=lenCh) ::    ca                     =   "calAmbient"
     Character(len=lenCh) ::    cc                     =   "calCold"     
     Character(len=lenCh) ::    cs                     =   "calSky"      
     Character(len=lenCh) ::    cp                     =   "calPaddle"   
     Character(len=lenCh) ::    am                     =   "airmass"   
     Character(len=lenCh) ::    onFocus                =   "onFocus"     
     Character(len=lenCh) ::    track                  =   "track"       
     Character(len=lenCh) ::    otf                    =   "onTheFly"    
     Character(len=lenCh) ::    on                     =   "on"            
     Character(len=lenCh) ::    ref                    =   "reference" 
     Character(len=lenCh) ::    VLBI                   =   "VLBI"
  End Type subscanChoicesType
  !
  Type (subscanChoicesType), Parameter ::                                    &
       &  ss =                                                               &
       &                          subscanChoicesType (     "calAmbient"  ,   &
       &                                                   "calCold"     ,   &
       &                                                   "calSky"      ,   &
       &                                                   "calPaddle"   ,   &
       &                                                   "airmass"     ,   &
       &                                                   "onFocus"     ,   &
       &                                                   "track"       ,   &
       &                                                   "onTheFly"    ,   &
       &                                                   "on"          ,   &  
       &                                                   "reference"   ,   &
       &                                                   "VLBI"        )
  !
  Character(len=lenCh), Dimension(nDimSubscanChoices), Parameter ::          &
       &  subscanChoices =                                                   &
       &                (/   ss%ca      ,                                    &
       &                     ss%cc      ,                                    &
       &                     ss%cs      ,                                    &
       &                     ss%cp      ,                                    &
       &                     ss%am      ,                                    &
       &                     ss%onFocus ,                                    &
       &                     ss%track   ,                                    &
       &                     ss%otf     ,                                    &
       &                     ss%on      ,                                    &
       &                     ss%ref     ,                                    &
       &                     ss%VLBI    /)
  !
  ! *   segments
  !
  Integer, Parameter      :: nDimSegmentChoices  =  4
  Integer, Parameter      :: iSegtrack      =  1       !   "track" 
  Integer, Parameter      :: iSeglinear     =  2       !   "linear"
  Integer, Parameter      :: iSegcircle     =  3       !   "circle"
  Integer, Parameter      :: iSegcurve      =  4       !   "curve" 
  !
  Type :: segmentChoicesType
     Character(len=lenCh) ::     track                 =   "track" 
     Character(len=lenCh) ::     linear                =   "linear"
     Character(len=lenCh) ::     circle                =   "circle"
     Character(len=lenCh) ::     curve                 =   "curve" 
  End Type segmentChoicesType
  !
  Type (segmentChoicesType), Parameter ::                                    &
       &  seg =                                                              &
       &                          segmentChoicesType (     "track"       ,   &
       &                                                   "linear"      ,   &
       &                                                   "circle"      ,   &
       &                                                   "curve"  )
  !
  Character(len=lenCh), Dimension(nDimSegmentChoices), Parameter ::          &
       &  segmentChoices =                                                   &
       &                (/   seg%track   ,                                   &
       &                     seg%linear  ,                                   &
       &                     seg%circle  ,                                   &
       &                     seg%curve   /)
  !
  ! *   traceFlag
  !
  Integer, Parameter      :: nDimtraceflagChoices  =  2
  Integer, Parameter      ::   itraceon     =  1       !   "on"       
  Integer, Parameter      ::   itraceref    =  2       !   "reference"
  !
  Type :: traceflagChoicesType
     Character(len=lenCh) ::         on                =   "on"       
     Character(len=lenCh) ::         ref               =   "reference"
  End Type traceflagChoicesType
  !
  Type (traceflagChoicesType), Parameter ::                                  &
       &  traceflag =                                                        &
       &                          traceflagChoicesType (   "on"         ,    &
       &                                                   "reference"  )
  !
  Character(len=lenCh), Dimension(nDimtraceflagChoices), Parameter ::        &
       &  traceflagChoices =                                                 &
       &                (/ traceflag%on   ,                                  &
       &                   traceflag%ref   /)
  !
  ! *   units for angles
  !
  Integer, Parameter      :: nDimAngleChoices  =  7
  Integer, Parameter      :: iradian        =  1       !      
  Integer, Parameter      :: iarcsec        =  2       !   
  Integer, Parameter      :: iarcmin        =  3       !   
  Integer, Parameter      :: idegree        =  4       !    
  Integer, Parameter      :: isecond        =  5       !   
  Integer, Parameter      :: iminute        =  6       !   
  Integer, Parameter      :: ihour          =  7       !      
  !
  Type :: angleChoicesType
     Character(len=lenCh) ::  radian                   =   "radian"   
     Character(len=lenCh) ::  arcsec                   =   "arcsec"
     Character(len=lenCh) ::  arcmin                   =   "arcmin"
     Character(len=lenCh) ::  degree                   =   "degree"
     Character(len=lenCh) ::  second                   =   "second"     
     Character(len=lenCh) ::  minute                   =   "minute"     
     Character(len=lenCh) ::  hour                     =   "hour"     
  End Type angleChoicesType
  !
  Type (angleChoicesType), Parameter ::                                      &
       &  au =                                                               &
       &                            angleChoicesType (     "radian"  ,       &
       &                                                   "arcsec"  ,       &
       &                                                   "arcmin"  ,       &
       &                                                   "degree"  ,       &
       &                                                   "second"  ,       &
       &                                                   "minute"  ,       &
       &                                                   "hour"    )
  !
  Character(len=lenCh), Dimension(nDimAngleChoices), Parameter ::            &
       &  angleChoices =                                                     &
       &                (/ au%radian ,                                       &
       &                   au%arcsec ,                                       &
       &                   au%arcmin ,                                       &
       &                   au%degree ,                                       &
       &                   au%second ,                                       &
       &                   au%minute ,                                       &
       &                   au%hour   /)
  !
  Real (Kind=kindDouble), Dimension(nDimAngleChoices), Parameter ::          &
       &  auConv =                                                           &
       &                (/    radian    ,                                    &
       &                      arcsec    ,                                    &
       &                      arcmin    ,                                    &
       &                      degree    ,                                    &
       &                      second    ,                                    &
       &                      minute    ,                                    &
       &                      hour      /)
  !
  !  ** following are our XML conventions (~VOTable)
  Type (angleChoicesType), Parameter ::                                      &
       &  auXML =                                                            &
       &                            angleChoicesType (     "rad"       ,     &
       &                                                   "arcsec"    ,     &
       &                                                   "arcmin"    ,     &
       &                                                   "deg"       ,     &
       &                                                   "s"         ,     &
       &                                                   "min"       ,     &
       &                                                   "h"         )

  !
  Character(len=lenCh), Dimension(nDimAngleChoices), Parameter ::            &
       &  angleChoicesXML =                                                  &
       &             (/ auXML%radian ,                                       &
       &                auXML%arcsec ,                                       &
       &                auXML%arcmin ,                                       &
       &                auXML%degree ,                                       &
       &                auXML%second ,                                       &
       &                auXML%minute ,                                       &
       &                auXML%hour   /)
  !
  ! *   units for angular speeds
  !
  Integer, Parameter      :: nDimSpeedChoices  =  nDimAngleChoices
  ! NB: indices i* are the same as for  angle units above
  !
  Type :: speedChoicesType
     Character(len=lenCh) ::  radian                   =   "radian/s"   
     Character(len=lenCh) ::  arcsec                   =   "arcsec/s"
     Character(len=lenCh) ::  arcmin                   =   "arcmin/s"
     Character(len=lenCh) ::  degree                   =   "degree/s"
     Character(len=lenCh) ::  second                   =   "second/s"     
     Character(len=lenCh) ::  minute                   =   "minute/s"     
     Character(len=lenCh) ::  hour                     =   "hour/s"     
  End Type speedChoicesType
  !
  Type (speedChoicesType), Parameter ::                                      &
       &  su =                                                               &
       &                            speedChoicesType (     "radian/s"  ,     &
       &                                                   "arcsec/s"  ,     &
       &                                                   "arcmin/s"  ,     &
       &                                                   "degree/s"  ,     &
       &                                                   "second/s"  ,     &
       &                                                   "minute/s"  ,     &
       &                                                   "hour/s"     )
  !
  !  ** following are our XML conventions (~VOTable)
  Type (speedChoicesType), Parameter ::                                      &
       &  suXML =                                                            &
       &                            speedChoicesType (     "rad/s"     ,     &
       &                                                   "arcsec/s"  ,     &
       &                                                   "arcmin/s"  ,     &
       &                                                   "deg/s"     ,     &
       &                                                   "s/s"       ,     &
       &                                                   "min/s"     ,     &
       &                                                   "h/s"       )
  !
  Character(len=lenCh), Dimension(nDimSpeedChoices), Parameter ::            &
       &  speedChoices =                                                     &
       &                (/ su%radian ,                                       &
       &                   su%arcsec ,                                       &
       &                   su%arcmin ,                                       &
       &                   su%degree ,                                       &
       &                   su%second ,                                       &
       &                   su%minute ,                                       &
       &                   su%hour   /)
  !
  Character(len=lenCh), Dimension(nDimSpeedChoices), Parameter ::            &
       &  speedChoicesXML =                                                  &
       &             (/ suXML%radian ,                                       &
       &                suXML%arcsec ,                                       &
       &                suXML%arcmin ,                                       &
       &                suXML%degree ,                                       &
       &                suXML%second ,                                       &
       &                suXML%minute ,                                       &
       &                suXML%hour   /)
  !
  ! *   units for Frequencies
  !
  Integer, Parameter      :: nDimFrequencyChoices  =  5
  Integer, Parameter      :: iTHz           =  1       !   "THz"
  Integer, Parameter      :: iGHz           =  2       !   "GHz"   
  Integer, Parameter      :: iMHz           =  3       !   "MHz"   
  Integer, Parameter      :: ikHz           =  4       !   "kHz"   
  Integer, Parameter      :: iHz            =  5       !   "Hz" 
  !                                                         
  Type :: frequencyChoicesType
     Character(len=lench) ::  THz                      =   "THz"
     Character(len=lench) ::  GHz                      =   "GHz"
     Character(len=lench) ::  MHz                      =   "MHz"
     Character(len=lench) ::  kHz                      =   "kHz"
     Character(len=lench) ::  Hz                       =   "Hz" 
  End Type frequencyChoicesType
  !
  Type (FREQUENCYCHOICESTYPE), Parameter ::                                  &
       &  FU =                                                               &
       &                        frequencyChoicesType (     "THz"      ,      &
       &                                                   "GHz"      ,      &
       &                                                   "MHz"      ,      &
       &                                                   "kHz"      ,      &
       &                                                   "Hz" )
  !
  Character(len=lenCh), Dimension(nDimFrequencyChoices), Parameter ::        &
       &  frequencyChoices =                                                 &
       &                (/ fu%THz ,                                          &
       &                   fu%GHz ,                                          &
       &                   fu%MHz ,                                          &
       &                   fu%kHz ,                                          &
       &                   fu%Hz  /)
  !
  ! *** systems for offsets
  !
  Integer, Parameter      :: nDimOffsetChoices  = 8
  Integer, Parameter      :: ipro           =  1       !   "projection" 
  Integer, Parameter      :: ides           =  2       !   "descriptive"
  Integer, Parameter      :: ibas           =  3       !   "basis"      
  Integer, Parameter      :: iequ           =  4       !   "equatorial" 
  Integer, Parameter      :: ihad           =  5       !   "haDec "     
  Integer, Parameter      :: itru           =  6       !   "trueHorizon"
  Integer, Parameter      :: ihor           =  7       !   "horizon" 
  Integer, Parameter      :: inas           =  8       !   "Nasmyth"    
  !
  Type :: offsetChoicesType
     Character(len=lenCh) ::  pro                      =   "projection" 
     Character(len=lenCh) ::  des                      =   "descriptive"
     Character(len=lenCh) ::  bas                      =   "basis"      
     Character(len=lenCh) ::  equ                      =   "equatorial" 
     Character(len=lenCh) ::  had                      =   "haDec "     
     Character(len=lenCh) ::  tru                      =   "trueHorizon"
     Character(len=lenCh) ::  hor                      =   "horizon" 
     Character(len=lenCh) ::  nas                      =   "Nasmyth"    
  End Type offsetChoicesType
  !
  Type (offsetChoicesType), Parameter ::                                     &
       &  offs =                                                             &
       &                          offsetChoicesType (      "projection"    , &
       &                                                   "descriptive"   , &
       &                                                   "basis"         , &
       &                                                   "equatorial"    , &
       &                                                   "haDec "        , &
       &                                                   "horizontalTrue", &
       &                                                   "horizontal"    , &
       &                                                   "Nasmyth"     )
  !
  Character(len=lenCh), Dimension(nDimOffsetChoices), Parameter ::           &
       &  offsetChoices =                                                    &
       &              (/ offs%pro          ,                                 &
       &                 offs%des          ,                                 &
       &                 offs%bas          ,                                 &
       &                 offs%equ          ,                                 &
       &                 offs%had          ,                                 &
       &                 offs%tru          ,                                 &
       &                 offs%hor          ,                                 &
       &                 offs%nas          /)
  !
  ! *   topology
  !
  Integer, Parameter      :: nDimTopologyChoices  = 5
  Integer, Parameter      :: ilow           = 1        !   "low"    
  Integer, Parameter      :: ihigh          = 2        !   "high"   
  Integer, Parameter      :: inearest       = 3        !   "nearest"
  Integer, Parameter      :: isouth         = 4        !   "south"  
  Integer, Parameter      :: inorth         = 5        !   "north"  
  !
  Type :: topologyChoicesType
     Character(len=lenCh) ::  low                      =   "low"    
     Character(len=lenCh) ::  high                     =   "high"   
     Character(len=lenCh) ::  nearest                  =   "nearest"
     Character(len=lenCh) ::  south                    =   "south"  
     Character(len=lenCh) ::  north                    =   "north"  
  End Type topologyChoicesType
  !
  Type   (topologyChoicesType), Parameter ::                                 &
       &  topo =                                                             &
       &                         topologyChoicesType (     "low"       ,     &
       &                                                   "high"      ,     &
       &                                                   "nearest"   ,     &
       &                                                   "south"     ,     &
       &                                                   "north"  )
  !
  Character(len=lenCh), Dimension(nDimTopologyChoices), Parameter ::         &
       &  topologyChoices =                                                  &
       &              (/ topo%low     ,                                      &
       &                 topo%high    ,                                      &
       &                 topo%nearest ,                                      &
       &                 topo%south   ,                                      &
       &                 topo%north   /)
  !
  ! *** set keywords
  !
  Integer, Parameter      :: nDimSetChoices  =  9
  Integer, Parameter      :: iproject       =  1       !   "project" 
  Integer, Parameter      :: iPI            =  2       !   "PI"      
  Integer, Parameter      :: iobserver      =  3       !   "observer"
  Integer, Parameter      :: ioperator      =  4       !   "operator"
  Integer, Parameter      :: ilevel         =  5       !   "level"   
  Integer, Parameter      :: ipointing      =  6       !   "pointing"
  Integer, Parameter      :: ifocus         =  7       !   "focus"   
  Integer, Parameter      :: itopology      =  8       !   "topology"
  Integer, Parameter      :: iAngleUnit     =  9       !   "angleUnit"
  !
  Type :: setChoicesType
     Character(len=lench) ::  project                  =   "project" 
     Character(len=lench) ::  PI                       =   "PI"      
     Character(len=lench) ::  observer                 =   "observer"
     Character(len=lench) ::  Operator                 =   "operator"
     Character(len=lench) ::  level                    =   "level"   
     Character(len=lench) ::  pointing                 =   "pointing"
     Character(len=lench) ::  focus                    =   "focus"   
     Character(len=lench) ::  topology                 =   "topology"
     Character(len=lench) ::  angleUnit                =   "angleUnit"
  End Type setChoicesType
  !
  Type (setChoicesType), Parameter ::                                        &
       &  set =                                                              &
       &                              setChoicesType (     "project"   ,     &
       &                                                   "PI"        ,     &
       &                                                   "observer"  ,     &
       &                                                   "operator"  ,     &
       &                                                   "level"     ,     &
       &                                                   "pointing"  ,     &
       &                                                   "focus"     ,     &
       &                                                   "topology"  ,     &
       &                                                   "angleUnit" )
  !
  Character(len=lenCh), Dimension(nDimSetChoices), Parameter ::              &
       &  setChoices =                                                       &
       &               (/ set%project  ,                                     &
       &                  set%PI       ,                                     &
       &                  set%observer ,                                     &
       &                  set%operator ,                                     &
       &                  set%level    ,                                     &
       &                  set%pointing ,                                     &
       &                  set%focus    ,                                     &
       &                  set%topology ,                                     &
       &                  set%angleUnit /)
  !
  ! ***
  Integer :: ii
!!!
!!! >>> TBD: the rest down to <<< is deprecated or obsolete
  Integer, Parameter :: nDimSegmentTypes = 4
  Character(len=lenCh),  Dimension(nDimSegmentTypes)                      &
       &                                   :: segmentTypeChoices="none"
  !
  Integer, Parameter :: nDimSystemNameChoices = 7
  Character(len=lenCh),  Dimension(nDimSystemNameChoices)                 &
       &                                   :: systemNameChoices="none"
  !
  Character(len=lenCh),  Dimension(nDimOffsetChoices)                     &
       &     ::  offsetSystemChoices="none"
  Integer, Parameter :: iPROJECTION   = 1
  Integer, Parameter :: iDESCRIPTIVE  = 2
  Integer, Parameter :: iBASIS        = 3
  Integer, Parameter :: iEQUATORIAL   = 4
  Integer, Parameter :: iHADEC        = 5
  Integer, Parameter :: iHORIZONTRUE  = 6
  Integer, Parameter :: iHORIZON      = 7
  Integer, Parameter :: iNASMYTH      = 8
  !
!!! <<<
!!!
!!!
Contains
!!!
!!!
  Subroutine printAngleChoices
    !
    Write (6,*) " "
    Do ii = 1, nDimAngleChoices, 1
       Write (6,*) "    angleChoices   ( ",ii,") = ", angleChoices(ii)
    End Do
    !
  End Subroutine printAngleChoices
!!!
!!!
  Subroutine printSpeedChoices
    !
    Write (6,*) " "
    Do ii = 1, nDimSpeedChoices, 1
       Write (6,*) "    speedChoices   ( ",ii,") = ", speedChoices(ii)
    End Do
    !
  End Subroutine printSpeedChoices
!!!
!!!
  Subroutine printReceiverChoices
    !
    Write (6,*) " "
    Do ii = 1, nDimReceiverChoices, 1
       Write (6,*) " receiverChoices   ( ",ii,") = ", receiverChoices(ii)
    End Do
    !
    Write (6,*) " "
    Do ii = 1, nDimBolometerChoices, 1
       Write (6,*) "bolometerChoices   ( ",ii,") = ", bolometerChoices(ii)
    End Do
    !
    Write (6,*) " "
    Do ii = 1, nDimSidebandChoices, 1
       Write (6,*) " sidebandChoices   ( ",ii,") = ", sidebandChoices(ii)
    End Do
    !
    Write (6,*) " "
    Do ii = 1, nDimWidthChoices, 1
       Write (6,*) "    widthChoices   ( ",ii,") = ", widthChoices(ii)
    End Do
    !
    Write (6,*) " "
    Do ii = 1, nDimScaleChoices, 1
       Write (6,*) "    scaleChoices   ( ",ii,") = ", scaleChoices(ii)
    End Do
    !
    Write (6,*) " "
    Do ii = 1, nDimDopplerChoices, 1
       Write (6,*) "  dopplerChoices   ( ",ii,") = ", dopplerChoices(ii)
    End Do
    !
  End Subroutine printReceiverChoices
!!!
!!!
  Subroutine printOffsetChoices
    !
    Write (6,*) " "
    Do ii = 1, nDimOffsetChoices, 1
       Write (6,*) "   offsetChoices   ( ",ii,") = ", offsetChoices(ii)
    End Do
    !
  End Subroutine printOffsetChoices
!!!
!!!
  Subroutine printTopologyChoices
    !
    Write (6,*) " "
    Do ii = 1, nDimTopologyChoices, 1
       Write (6,*) "   topologyChoices ( ",ii,") = ", topologyChoices(ii)
    End Do
    !
  End Subroutine printTopologyChoices
!!!
!!!
  Subroutine printSubscanChoices
    !
    Write (6,*) " "
    Do ii = 1, nDimSubscanChoices, 1
       Write (6,*) "   subscanChoices  ( ",ii,") = ", subscanChoices(ii)
    End Do
    !
  End Subroutine printSubscanChoices
!!!
!!!
  Subroutine printSegmentChoices
    !
    Write (6,*) " "
    Do ii = 1, nDimSegmentChoices, 1
       Write (6,*) "   segmentChoices  ( ",ii,") = ", segmentChoices(ii)
    End Do
    !
  End Subroutine printSegmentChoices
!!!
!!!
  Subroutine printSetChoices
    !
    Write (6,*) " "
    Do ii = 1, nDimSetChoices, 1
       Write (6,*) "       setChoices  ( ",ii,") = ", setChoices(ii)
    End Do
    !
  End Subroutine printSetChoices
!!!
!!!
  Subroutine initializeReceiverChoices
    ! TBD: SR IS OBSOLETE
    !
!!$    receiverNameChoices(iA100)       =  "A100"      
!!$    receiverNameChoices(iA230)       =  "A230"      
!!$    receiverNameChoices(iB100)       =  "B100"      
!!$    receiverNameChoices(iB230)       =  "B230"      
!!$    receiverNameChoices(iC150)       =  "C150"      
!!$    receiverNameChoices(iC270)       =  "C270"      
!!$    receiverNameChoices(iD150)       =  "D150"      
!!$    receiverNameChoices(iD270)       =  "D270"      
!!$    receiverNameChoices(iBOLO)       =  "BOLOMETER" 
!!$    receiverNameChoices(iHERA)       =  "HERA"      
!!$    receiverNameChoices(iVERTICAL)   =  "VERTICAL"  
!!$    receiverNameChoices(iHORIZONTAL) =  "HORIZONTAL"
!!$    !
!!$    bolometerNameChoices(i37)        =  "37"
!!$    bolometerNameChoices(i117)       =  "117"
!!$    !
!!$    dopplerChoices(1)     =  "DOPPLER"
!!$    dopplerChoices(2)     =  "FIXED"
!!$    !
!!$    scaleChoices(1)       =  "ANTENNA"
!!$    scaleChoices(2)       =  "BEAM"
!!$    !
!!$    widthChoices(1)       =  "NARROW"
!!$    widthChoices(2)       =  "WIDE"
    !
  End Subroutine initializeReceiverChoices
!!!
!!!
  Subroutine initializeSystemNameChoices
    ! TBD: SR IS DEPRECATED
    Integer ii
    !
    !D          Write (6,*) "   moduleGlobalParameters: SR initializeSystemNameChoices"
    !
    ! TBD: SYSTEMNAMECHOICES IS OBSOLETE
    systemNameChoices(1)    = "PROJECTION"
    systemNameChoices(2)    = "DESCRIPTIVE"
    systemNameChoices(3)    = "BASIS"
    systemNameChoices(4)    = "EQUATORIAL"
    systemNameChoices(5)    = "HA/DECL"
    systemNameChoices(6)    = "HORIZONTAL"
    systemNameChoices(7)    = "NASMYTH"
    !
    ! TBD: OFFSETSYSTEMNAMECHOICES IS OBSOLETE
    offsetSystemChoices(iPROJECTION ) = "PROJECTION" 
    offsetSystemChoices(iDESCRIPTIVE) = "DESCRIPTIVE"
    offsetSystemChoices(iBASIS      ) = "BASIS"      
    offsetSystemChoices(iEQUATORIAL ) = "EQUATORIAL" 
    offsetSystemChoices(iHADEC      ) = "HADEC "     
    offsetSystemChoices(iHORIZONTRUE) = "TRUEHORIZON"
    offsetSystemChoices(iHORIZON    ) = "HORIZONTAL" 
    offsetSystemChoices(iNASMYTH    ) = "NASMYTH"    
    !
    !D          do ii = 1, nDimOffsets, 1
    !D             Write (6,*) ii, offsetSystemChoices(ii)
    !D          end do
    !
  End Subroutine initializeSystemNameChoices
!!!
!!!
End Module modulePakoGlobalParameters





