module mod_clean_task
  use gildas_def
  type clean_files
    character(len=filename_length) :: dirty
    character(len=filename_length) :: beam
    character(len=filename_length) :: clean
    character(len=filename_length) :: cct
    character(len=filename_length) :: residual
  end type
end module mod_clean_task
!
program clean
  use gildas_def
  use gkernel_interfaces
  use mapping_interfaces
  use image_def
  use clean_def
  use mod_clean_task
  use gbl_message
  !---------------------------------------------------------------------
  ! GILDAS	Standalone program
  !	Implementation of several CLEAN deconvolution algorithms :
  !	- The simple-minded one, taken out from the Berkeley package.
  !	  Beam and maps can be of different sizes. Cleaned area
  !	  depends of their respective sizes.
  !	- Clark's CLEAN. Beam cannot be different from map.
  !	- Multi-Resolution Clean, with a Clark-like major cycle
  !	  implementation. Beam cannot be different from map.
  !     - Multi-Scale Clean, with 3 scales, no major cycle.
  !       Beam cannot be different from map.
  ! Peculiarities
  !	- Little assumptions about beam maximum
  !	- Cubes can be cleaned with a unique beam
  ! Restriction
  !	- Cannot be restarted (presently)
  !---------------------------------------------------------------------
  ! Local
  type(gildas) :: hclean,hresid,hcct,hbeam,hdirty,hmask
  type(clean_par) :: method
  type(clean_files) :: names
  !
  logical, allocatable :: mask(:,:)
  integer, allocatable :: list(:)
  integer :: ier
  logical :: error
  character(len=12) :: cmethod
  integer :: nx,ny
  !
  error = .false.
  call gildas_null(hclean)
  call gildas_null(hresid)
  call gildas_null(hcct)
  call gildas_null(hbeam)
  call gildas_null(hdirty)
  call gildas_null(hmask)
  !
  ! Reads command file
  call get_parameters (cmethod,names,method)
  !!Print *,'Got parameters ',method%method
  !
  ! Opens input maps & create output maps
  call open_maps(names,hdirty,hresid,hclean,hcct,hbeam,method,error)
  !!Print *,'Done open_maps ',method%method
  !
  ! Check cleaning box
  nx = hdirty%gil%dim(1)
  ny = hdirty%gil%dim(2)
  call check_box(nx,ny,method%blc,method%trc)
  allocate(list(nx*ny),mask(nx,ny),stat=ier)
  if (ier.ne.0) then
    call map_message(seve%e,'CLEAN','Mask allocation error')
    call sysexi(fatale)
  endif
  call check_list(method,nx,ny,list,mask,error)
  !!Print *,'Done check list ',error
  if (error) call sysexi(fatale)
  !
  ! Perform the cleaning
  call clean_dispatch(method,hdirty,hresid,hclean,hcct,hbeam,hmask,&
  & mask,list,error) 
  !!Print *,'Done dispatch  ',error
  !
  ! Update clean header (& residual if extrema computed on-the-fly)
  call gdf_update_header(hclean,error)
  !
  if (error) call sysexi(fatale)
  call map_message(seve%i,'CLEAN','Successful completion')
end program clean
!
subroutine check_list(method,nx,ny,list,mask,error)
  use clean_def
  use gbl_message
  type(clean_par), intent(inout) :: method
  integer, intent(in) :: nx,ny
  logical, intent(out) :: mask(nx,ny)
  integer, intent(out) :: list(*)
  logical, intent(out) :: error
  !
  integer i,j,k,ier
  !
  method%box = [method%blc(1),method%blc(2),method%trc(1),method%trc(2)]
  !
  method%nlist = (method%trc(2)-method%blc(2)+1) * &
      & (method%trc(1)-method%blc(1)+1)
  !
  mask = .false.
  k = 0
  do j=method%blc(2),method%trc(2)
    do i=method%blc(1),method%trc(1)
      k = k+1
      list(k) = i+nx*(j-1)
      mask(i,j) = .true.
    enddo
  enddo
  error = .false.
end subroutine check_list
!
subroutine clean_dispatch (method,hdirty,hresid,hclean,hcct,hbeam,hmask,& 
  & mask,list,error)
  use gkernel_interfaces
  use mapping_interfaces
  use clean_def
  use image_def
  use gbl_message
  !----------------------------------------------------------------------
  ! @ private
  !
  ! MAPPING Internal routine
  !     Implementation of all standard CLEAN deconvolution algorithms,
  !----------------------------------------------------------------------
  type(clean_par), intent(inout) :: method
  type(gildas), intent(inout) :: hdirty
  type(gildas), intent(inout) :: hresid
  type(gildas), intent(inout) :: hclean
  type(gildas), intent(inout) :: hcct
  type(gildas), intent(inout) :: hbeam
  type(gildas), intent(in) :: hmask
  logical, intent(in) :: mask(hdirty%gil%dim(1),hdirty%gil%dim(2))
  integer, intent(in) :: list(hdirty%gil%dim(1)*hdirty%gil%dim(2))
  !
  logical, intent(out) :: error
  !
  real(8), parameter :: pi=3.14159265358979323846d0
  external noplot_mrc
  !
  integer :: ier, nx, ny, np, nplane, nbeam, nz
  integer :: l, j, i
  real, allocatable :: dcct(:,:,:)
  type(gildas) :: hprim
  !
  call gildas_null(hprim)
  !
  if (method%first.eq.0) method%first = 1
  if (method%last.eq.0) method%last = hdirty%gil%dim(3)
  method%first = max(1,min(method%first,hdirty%gil%dim(3)))
  method%last = max(method%first,min(method%last,hdirty%gil%dim(3)))
  !
  call check_area(method,hdirty,.false.)
  !
  ! Usefull variables
  nx = hdirty%gil%dim(1)
  ny = hdirty%gil%dim(2)
  np = 1  !! No primary beams ... max(1,hprim%gil%dim(1))
  !
  ! Beam patch according to Method
  if (method%method.eq.'CLARK'.or.   &
     &    method%method.eq.'MRC') then
    if (method%patch(1).ne.0) then
      method%patch(1) = min(method%patch(1),nx)
    else
      method%patch(1) = min(nx,max(32,nx/4))
    endif
    if (method%patch(2).ne.0) then
      method%patch(2) = min(method%patch(2),ny)
    else
      method%patch(2) = min(ny,max(32,ny/4))
    endif
  elseif (method%method.eq.'SDI') then
    if (method%patch(1).ne.0) then
      method%patch(1) = min(method%patch(1),nx/4)
    else
      method%patch(1) = min(nx/2,max(16,nx/8))
    endif
    if (method%patch(2).ne.0) then
      method%patch(2) = min(method%patch(2),ny/4)
    else
      method%patch(2) = min(ny,max(16,ny/8))
    endif
  endif
  !
  ! Here put the blocking loops
  nplane = max((64*64*64)/(nx*ny),1)
  nbeam = hbeam%gil%dim(3)
  np = min(nplane,nbeam)
  !
  ! Clean component 
  allocate(dcct(3,nplane,method%m_iter),stat=ier)
  if (ier.ne.0) then
    call map_message(seve%e,'CLEAN','Memory allocation failure for CCT')
    error = .true.
    return
  endif
  !
  ! Avoid array constructors (too touchy about )
  method%bzone(1:2) = 1
  method%bzone(3:4) = hdirty%gil%dim(1:2) 
  !
  allocate (hclean%r3d(nx,ny,nplane), &
      &   hresid%r3d(nx,ny,nplane),   &
      &   hdirty%r3d(nx,ny,nplane),   &
      &   hbeam%r4d(nx,ny,np,1),        &
      &      stat=ier)
  if (ier.ne.0) then
    call map_message(seve%e,'CLEAN','Memory allocation failure for Images')
    error = .true.
    return
  endif
  !
  if (method%method.ne.'MRC') then
    allocate (hcct%r3d(3,method%m_iter,nplane), &
      &      stat=ier)
    if (ier.ne.0) then
      call map_message(seve%e,'CLEAN','Memory allocation failure for Images')
      error = .true.
      return
    endif
  endif
  !
  if (nbeam.le.1) then
    hbeam%blc = 0 
    hbeam%trc = 0 
    call gdf_read_data(hbeam,hbeam%r4d,error)
    if (error) then
      call map_message(seve%e,'CLEAN','Could not read beam')
      return
    endif
  endif
  !
  do j=1,hdirty%gil%dim(4)
    hdirty%blc(4) = j
    hdirty%trc(4) = j
    do l=1,hdirty%gil%dim(3),nplane
      hdirty%blc(3) = l
      hdirty%trc(3) = min(hdirty%gil%dim(3),l+nplane-1)
      nz = hdirty%trc(3)-hdirty%blc(3)+1
      hresid%blc = hdirty%blc
      hclean%blc = hdirty%blc
      hresid%trc = hdirty%trc
      hclean%trc = hdirty%trc
      !
      ! Read dirty image
      call gdf_read_data(hdirty,hdirty%r3d,error)
      !
      ! Read beam if needed
      if (nbeam.gt.1) then
        hbeam%blc(3) = hdirty%blc(3)
        hbeam%trc(3) = hdirty%trc(3)
        call gdf_read_data(hbeam,hbeam%r4d,error)
        hbeam%gil%dim(3) = nz
      endif
      !   
      ! Specify subset to be cleaned
      method%first = 1
      method%last = nz
      !
      ! Need to pass the GET_BEAM routine as argument
      ! in order to fit beam size
      if (method%method.eq.'MRC') then
        call sub_mrc('MRC',method,hdirty,hresid,hclean,hbeam,hprim, &
        &   mask,error,noplot_mrc)
      else
        call sub_major(method,hdirty,hresid,hclean,hbeam,hprim,hmask, &
        &   dcct,mask,list,error,no_major_plot,no_next_flux)
      endif
      !
      call gdf_write_data(hresid,hresid%r3d,error)
      call gdf_write_data(hclean,hclean%r3d,error)
      !
      ! Write CCT if possible
      if (method%method.ne.'MRC') then
        hcct%r3d = 0.0
        do i=1,nz
          hcct%r3d(:,:,i) = dcct(:,i,:)
        enddo
        hcct%blc(3) = hdirty%blc(3)
        hcct%trc(3) = hdirty%trc(3)      
        call gdf_write_data(hcct,hcct%r3d,error)
      endif
      !
      hbeam%gil%dim(3) = nbeam
    enddo
  enddo
  !
  ! Reset extrema (could be computed on-the-fly if needed)
  hresid%gil%extr_words = 0
  hclean%gil%extr_words = 0
  !
  ! Specify clean beam parameters
  hclean%gil%reso_words = 3
  hclean%gil%majo = method%major
  hclean%gil%mino = method%minor
  hclean%gil%posa = pi*method%angle/180.0
  !
end subroutine clean_dispatch
!
subroutine noplot_mrc(method,head,array,code)
  use image_def
  use clean_def
  use mapping_interfaces, except_this=>plot_mrc
  !----------------------------------------------------------------
  ! 
  ! @ private
  !
  ! MAPPING
  !   Dispatch the various plotting actions in MRC
  !----------------------------------------------------------------
  type(clean_par), intent(in) :: method
  type(gildas), intent(in) :: head
  integer, intent(in) :: code
  real, intent(in) :: array(head%gil%dim(1),head%gil%dim(2))
  !
  return
end subroutine
!
subroutine get_parameters (cmethod,names,method)
  use gkernel_interfaces
  use clean_def
  use mod_clean_task
  !---------------------------------------------------------------------
  ! GILDAS:	CLEAN 	Internal routine
  !	Retrieves input parameters for CLEAN
  !---------------------------------------------------------------------
  character(len=*) :: cmethod        !
  type(clean_par) :: method
  type(clean_files) :: names
  !
  call gildas_open
  call gildas_char('METHOD$',method%method)
  call gildas_char('DIRTY$',names%dirty)
  call gildas_char('BEAM$',names%beam)
  call gildas_char('RESIDUAL$',names%residual)
  call gildas_char('CLEAN$',names%clean)
  call gildas_char('COMPONENT$',names%cct)
  call gildas_real('GAIN$',method%gain,1)
  call gildas_inte('NITER$',method%m_iter,1)
  call gildas_real('FRES$',method%fres,1)
  call gildas_real('ARES$',method%ares,1)
  call gildas_inte('BLC$',method%blc,2)
  call gildas_inte('TRC$',method%trc,2)
  call gildas_inte('POSITIVE$',method%p_iter,1)
  call gildas_logi('KEEP$',method%keep,1)
  call gildas_real('MAJOR$',method%major,1)
  call gildas_real('MINOR$',method%minor,1)
  call gildas_real('PA$',method%angle,1)
  call gildas_inte('BEAM_PATCH$',method%patch,2)
  call gildas_close
  !
  call sic_upper(method%method)
  if (method%method.eq.'SIMPLE') method%method='HOGBOM'
  if (method%method.eq.'THRESOLD') method%method='SDI'
  !
  method%pflux = .false.
  method%pcycle = .false.
  method%qcycle = .false.
  method%pflux = .false.
  method%mosaic = .false.
  method%n_major = 100
  !
  method%bshift = 0
  method%nker = [1,7,11]
  method%smooth = sqrt(3.0)
  method%ninflate = 100
  method%worry = 0.0
  method%verbose = .true.
  method%gains(:) = method%gain
  !
  call beam_unit_conversion(method)  
  !
  method%first = 0
  method%last = 0
end subroutine get_parameters
!
subroutine open_maps(names,hdirty,hresid,hclean,hcct,hbeam,method,error)
  use gildas_def
  use gkernel_interfaces
  use mod_clean_task
  use clean_def
  use image_def
  use gbl_message
  !---------------------------------------------------------------------
  ! GILDAS:	CLEAN 	Internal routine
  !	Open input maps
  !	Dirty beam is copied into hbeam scratch area
  !	Clean Component Table in Z, originally undefined
  !	Residual map in hresid, originally loaded with dirty map
  !	Cleaned map in X, originally undefined
  !---------------------------------------------------------------------
  type(clean_files), intent(inout) :: names
  type(gildas), intent(inout) :: hdirty,hresid,hclean,hcct,hbeam           !
  type(clean_par), intent(inout)  :: method        !
  logical, intent(out) :: error
  ! Local
  real(8), parameter :: epsilon=1d-7
  integer :: n,max_iter0,nx,ny
  character(len=12) :: rname
  character(len=80) :: chain
  !
  rname = method%method
  !
  ! Read beam
  n = len_trim(names%beam)
  if (n.eq.0) goto 99
  call gildas_null(hbeam)
  call gdf_read_gildas(hbeam,  names%beam, '.beam', error, rank=3, data=.false.)
  if (error) then
    call map_message(seve%e,rname,'Cannot read dirty beam '//trim(hbeam%file))
    goto 99
  endif
  !
  nx = hbeam%gil%dim(1)
  ny = hbeam%gil%dim(2)
  !
  if ( (power_of_two(nx).lt.0) .or. (power_of_two(ny).lt.0) ) then
    if (method%method.eq.'MRC') then
      call map_message(seve%e,rname,'Dimensions must be powers of two')
      goto 99
    elseif (method%method.eq.'CLARK') then
      call map_message(seve%w,rname,'Dimensions not powers of two, may be slow')
    endif
  endif
  !
  if (hbeam%gil%dim(3)*hbeam%gil%dim(4) .ne.1 ) then
    call map_message(seve%w,rname,'Using a cube as a beam')
  endif
  !
  call gildas_null(hdirty)
  !
  ! Read dirty map header
  n = len_trim(names%dirty)
  if (n.eq.0) goto 99
  call gildas_null(hdirty)
  call gdf_read_gildas(hdirty,  names%dirty, '.lmv', error, rank=3, data=.false.)
  if (error) then
    call map_message(seve%e,rname,'Cannot read dirty map '//trim(hdirty%file))
    goto 99
  endif
  !
  ! Compute MAX_ITER if 0
  if (hdirty%gil%extr_words.eq.0) then
    call map_message(seve%e,rname,'Missing extrema in Dirty map')
    goto 99
  endif
  !
  if (method%ares.eq.0) then
    method%ares = max(hdirty%gil%noise,method%fres*hdirty%gil%rmax)
  endif
  write(chain,*) 'Cleaning down to ',method%ares
  call map_message(seve%i,rname,chain)
  !
  ! Assume only Clean BOX is cleaned, and beam area is about 20 pixels
  call check_box(hdirty%gil%dim(1),hdirty%gil%dim(2),method%blc,method%trc)
  max_iter0 = log(method%ares/hdirty%gil%rmax)/log(1.0-method%gain)*   &
     &    (method%trc(1)-method%blc(1)+1)*(method%trc(2)-method%blc(2)+1)/20.0
  write(chain,*) 'Could need ',max_iter0,' iterations'
  call map_message(seve%i,rname,chain)
  !
  if (method%m_iter.eq.0) then
    method%m_iter = max_iter0
  endif
  !
  names%dirty = hdirty%file
  !
  ! Check input dimensions
  if (hdirty%gil%dim(1).ne.hbeam%gil%dim(1) .or. &
    & hdirty%gil%dim(2).ne.hbeam%gil%dim(2)) then
    call map_message(seve%e,rname,'Beam and map do not match')
    goto 99
  endif
  if (hbeam%gil%dim(3).le.1) then
    if (hdirty%gil%dim(3)*hdirty%gil%dim(4).gt.1 )  then
      call map_message(seve%w,rname,'Combining a cube with a plane')
    endif
  else
    if (hbeam%gil%dim(3).ne.hdirty%gil%dim(3)*hdirty%gil%dim(4)) then
      call map_message(seve%e,rname,'Image and Beam cube have '   &
     &        //' different number of channels')
      goto 99
    endif
  endif
  !
  ! Check pixel size
  if ( (abs(hbeam%gil%inc(1)-hdirty%gil%inc(1)).gt.abs(hdirty%gil%inc(1)*epsilon)) .or.   &
     &    (abs(hbeam%gil%inc(2)-hdirty%gil%inc(2)).gt.abs(hdirty%gil%inc(2)*epsilon)) ) then
    call map_message(seve%e,rname,'Pixel sizes in map and beam'//   &
     &      'do not match')
    goto 99
  endif
  !
  ! Create Clean Component Table (in hcct)
  if (method%method.ne.'MRC') then
    n = len_trim(names%cct)
    if (n.eq.0) goto 99
    call gildas_null(hcct)
    call gdf_copy_header (hdirty,hcct,error)
    hcct%gil%dim(1) = 3
    hcct%gil%dim(2) = method%m_iter
    !	hcct%gil%dim(3) and hcct%gil%dim(4) as before
    hcct%loca%size = hcct%gil%dim(1)*hcct%gil%dim(2)*hcct%gil%dim(3)*hcct%gil%dim(4)
    call sic_parsef(names%cct(1:n),hcct%file,' ','.cct')
    hcct%char%code(1) = 'IJV'
    hcct%char%code(2) = 'COMPONENT'
    hcct%gil%val(1) = 0.d0
    hcct%gil%val(2) = 0.d0
    hcct%gil%extr_words = 0
    !
    call gdf_create_image(hcct,error)
    if (error) then
      call map_message(seve%e,rname,'Cannot create clean component table')
      goto 99
    endif
  else
    !
    ! MRC specific parameters
    nx = hdirty%gil%dim(1)
    ny = hdirty%gil%dim(2)
    if (nx*ny.gt.512*512) then
      method%ratio = 8
    elseif (nx*ny.gt.128*128) then
      method%ratio = 4
    else
      method%ratio = 2
    endif
  endif
  !
  ! Create Residual Map (in hresi)
  n = len_trim(names%residual)
  if (n.eq.0) goto 99
  call gildas_null(hresid)
  call gdf_copy_header(hdirty,hresid,error)
  call sic_parsef(names%residual(1:n),hresid%file,' ','.lmv-res')
  hresid%gil%extr_words = 0
  call gdf_create_image (hresid,error)
  if (error) then
    call map_message(seve%e,rname,'Cannot create residual image')
    goto 99
  endif
  !
  ! Prepare Cleaned Map 
  !
  n = len_trim(names%clean)
  if (n.eq.0) goto 99
  call gildas_null(hclean)
  call gdf_copy_header(hdirty,hclean,error)
  call sic_parsef(names%clean(1:n),hclean%file,' ','.lmv-clean')
  hclean%gil%reso_words = 3
  hclean%gil%extr_words = 0
  !
  call gdf_create_image(hclean,error)
  if (error) then
    call map_message(seve%e,rname,'Cannot create cleaned image')
    goto 99
  endif
  return
  !
  99    error =.true.
end subroutine open_maps
!
! Dummies for subroutines
subroutine sic_wprn
end subroutine sic_wprn
!
function sic_ctrlc()
  logical sic_ctrlc
  sic_ctrlc = .false.
end function sic_ctrlc
!
subroutine plot_multi90(niter,flux,is)
  integer, intent(in) :: is
  integer, intent(in) :: niter
  real, intent(in) :: flux
end subroutine plot_multi90

subroutine init_plot(method,head,pdata)
  use clean_def
  use image_def
  type (clean_par), intent(in) :: method
  type (gildas), intent(in)    :: head
  real, intent(in)             :: pdata(head%gil%dim(1),head%gil%dim(2))
end subroutine init_plot

