!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
!
module cubemain_average
  use cube_types
  use cubemain_messaging
  use cubetools_structure
  use cubeadm_cubeid_types
  !
  public :: average
  public :: cubemain_average_command
  private
  !
  type :: average_comm_t
     type(option_t), pointer :: comm
     type(option_t), pointer :: noise
     type(option_t), pointer :: weight
     type(option_t), pointer :: family
   contains
     procedure, public  :: register     => cubemain_average_register
     procedure, private :: parse        => cubemain_average_parse
     procedure, private :: parse_weight => cubemain_average_parse_weight
     procedure, private :: parse_family => cubemain_average_parse_family
     procedure, private :: parse_noise  => cubemain_average_parse_noise
     procedure, private :: main         => cubemain_average_main
  end type average_comm_t
  type(average_comm_t) :: average
  !
  integer(kind=4), parameter :: ione = 1 
  integer(kind=4), parameter :: itwo = 2
  type average_user_t
     type(cubeid_user_t)   :: cubeids
     type(cubeid_user_t)   :: noiseids
     real(kind=sign_k)     :: weights(2)
     character(len=base_l) :: family
     logical               :: donoise
     logical               :: doweight
     logical               :: dofamily
   contains
     procedure, private :: toprog => cubemain_average_user_toprog
  end type average_user_t
  type average_prog_t
     real(kind=sign_k)     :: weights(2)
     character(len=base_l) :: family
     logical               :: donoise
     logical               :: dofamily
     type(cube_t), pointer :: one
     type(cube_t), pointer :: two
     type(cube_t), pointer :: noise1
     type(cube_t), pointer :: noise2
     type(cube_t), pointer :: averaged
     type(cube_t), pointer :: weight
     real(kind=sign_k), allocatable :: wei1(:,:), wei2(:,:)
     procedure(cubemain_average_prog_noise_loop), private, pointer :: loop => null()
   contains
     procedure, private :: header          => cubemain_average_prog_header
     procedure, private :: consistency     => cubemain_average_prog_consistency
     procedure, private :: data            => cubemain_average_prog_data
     procedure, private :: act_noise       => cubemain_average_prog_act_noise
     procedure, private :: act_weight      => cubemain_average_prog_act_weight
     procedure, private :: compute_average => cubemain_average_prog_compute_average
  end type average_prog_t
  !
contains
  !
  subroutine cubemain_average_command(line,error)
    !----------------------------------------------------------------------
    !
    !----------------------------------------------------------------------
    character(len=*), intent(in)    :: line
    logical,          intent(inout) :: error
    !
    type(average_user_t) :: user
    character(len=*), parameter :: rname='AVERAGE>COMMAND'
    !
    call cubemain_message(mainseve%trace,rname,'Welcome')
    !
    call average%parse(line,user,error)
    if (error) return
    call average%main(user,error)
    if (error) continue
  end subroutine cubemain_average_command
  !
  !----------------------------------------------------------------------
  !
  subroutine cubemain_average_register(average,error)
    use cubedag_allflags
    !----------------------------------------------------------------------
    !
    !----------------------------------------------------------------------
    class(average_comm_t), intent(inout) :: average
    logical,               intent(inout) :: error
    !
    type(cubeid_arg_t) :: cubearg
    type(standard_arg_t) :: stdarg
    character(len=*), parameter :: comm_abstract = &
         'Average two cubes together'
    character(len=*), parameter :: comm_help = &
         'Average two cubes together with different weighting&
         & schemes. By default both cubes are assumed to have the&
         & same weight. This can be changed by using options /NOISE&
         & and /WEIGHT. These two options can be combined and the&
         & resulting weight will be w_i/(noise_i)**2. By default the&
         & resulting cube will have the same family name as cube1.&
         & This can be changed by using option /FAMILY.'
    character(len=*), parameter :: rname='AVERAGE>REGISTER'
    !
    call cubemain_message(mainseve%trace,rname,'Welcome')
    !
    ! Command
    call cubetools_register_command(&
         'AVERAGE','cube1 cube2',&
         comm_abstract,&
         comm_help,&
         cubemain_average_command,&
         average%comm,error)
    if (error) return
    call cubearg%register(&
         'CUBE1',&
         'Input cube #1',  &
         strg_id,&
         code_arg_mandatory,  &
         [flag_cube],&
         error)
    if (error) return
    call cubearg%register(&
         'CUBE2',&
         'Input cube #2',&
         strg_id,&
         code_arg_mandatory,&
         [flag_cube],&
         error)
    if (error) return
    !
    call cubetools_register_option(&
         'WEIGHT','w1 w2',&
         'Define explicit real weights',&
         strg_id,&
         average%weight,error)
    if (error) return
    call stdarg%register(&
         'w1',&
         'Weight for Input cube #1',&
         strg_id,&
         code_arg_mandatory,&
         error)
    if (error) return
    call stdarg%register(&
         'w2',&
         'Weight for Input cube #2',&
         strg_id,&
         code_arg_mandatory,&
         error)
    if (error) return
    !
    call cubetools_register_option(&
         'FAMILY','newfamily',&
         'Define the new family name for products',&
         strg_id,&
         average%family,error)
    if (error) return
    call stdarg%register(&
         'newfamily',&
         'New family name for products',&
         strg_id,&
         code_arg_mandatory,&
         error)
    if (error) return
    !
    call cubetools_register_option(&
         'NOISE','noise1 noise2',&
         'Cubes are weighted by noise',&
         'The weight is computed from the noise images as: 1/(noise&
         &**2). Currently only single noise per spectrum is supported',&
         average%noise,error)
    if (error) return
    call cubearg%register(&
         'NOISE1',&
         'Noise for Input cube #1',  &
         strg_id,&
         code_arg_mandatory,  &
         [flag_noise],&
         error)
    if (error) return
    call cubearg%register(&
         'NOISE2',&
         'Noise for Input cube #2',&
         strg_id,&
         code_arg_mandatory,&
         [flag_noise],&
         error)
    if (error) return
  end subroutine cubemain_average_register
  !
  subroutine cubemain_average_parse(average,line,user,error)
    !----------------------------------------------------------------------
    ! 
    !----------------------------------------------------------------------
    class(average_comm_t), intent(in)    :: average 
    character(len=*),      intent(in)    :: line
    type(average_user_t),  intent(out)   :: user
    logical,               intent(inout) :: error
    !
    character(len=*), parameter :: rname='AVERAGE>PARSE'
    !
    call cubemain_message(mainseve%trace,rname,'Welcome')
    !
    call cubeadm_cubeid_parse(line,average%comm,user%cubeids,error)
    if (error) return
    call average%parse_weight(line,user,error)
    if (error) return
    call average%parse_noise(line,user,error)
    if (error) return
    call average%parse_family(line,user,error)
    if (error) return
  end subroutine cubemain_average_parse
  !
  subroutine cubemain_average_parse_weight(average,line,user,error)
    !----------------------------------------------------------------------
    !
    !----------------------------------------------------------------------
    class(average_comm_t), intent(in)    :: average 
    character(len=*),      intent(in)    :: line
    type(average_user_t),  intent(inout) :: user
    logical,               intent(inout) :: error
    !
    integer(kind=4) :: iw
    character(len=*), parameter :: rname='AVERAGE>PARSE>WEIGHT'
    !
    call cubemain_message(mainseve%trace,rname,'Welcome')
    !
    call average%weight%present(line,user%doweight,error)
    if (error) return
    if (user%doweight) then
       do iw=1,2
          call cubetools_getarg(line,average%weight,iw,user%weights(iw),mandatory,error)
          if (error) return
          if (user%weights(iw).le.0) then
             call cubemain_message(seve%e,rname,'Weights must be positive')
             error = .true.
             return
          endif
       enddo
    else
       ! Do nothing
    endif
  end subroutine cubemain_average_parse_weight
  !
  subroutine cubemain_average_parse_family(average,line,user,error)
    !----------------------------------------------------------------------
    !
    !----------------------------------------------------------------------
    class(average_comm_t), intent(in)    :: average 
    character(len=*),      intent(in)    :: line
    type(average_user_t),  intent(inout) :: user
    logical,               intent(inout) :: error
    !
    character(len=*), parameter :: rname='AVERAGE>PARSE>FAMILY'
    !
    call cubemain_message(mainseve%trace,rname,'Welcome')
    !
    call average%family%present(line,user%dofamily,error)
    if (error) return
    if (user%dofamily) then
       call cubetools_getarg(line,average%family,1,user%family,mandatory,error)
       if (error) return
    else
       ! Do nothing
    endif
  end subroutine cubemain_average_parse_family
  !
  subroutine cubemain_average_parse_noise(average,line,user,error)
    !----------------------------------------------------------------------
    !
    !----------------------------------------------------------------------
    class(average_comm_t), intent(in)    :: average 
    character(len=*),      intent(in)    :: line
    type(average_user_t),  intent(inout) :: user
    logical,               intent(inout) :: error
    !
    character(len=*), parameter :: rname='AVERAGE>PARSE>NOISE'
    !
    call cubemain_message(mainseve%trace,rname,'Welcome')
    !
    call average%noise%present(line,user%donoise,error)
    if (error) return
    if (user%donoise) then
       call cubeadm_cubeid_parse(line,average%noise,user%noiseids,error)
       if (error) return
    else
       ! Do nothing
    endif
  end subroutine cubemain_average_parse_noise
  !
  subroutine cubemain_average_main(average,user,error)
    use cubeadm_timing
    !----------------------------------------------------------------------
    !
    !----------------------------------------------------------------------
    class(average_comm_t), intent(in)    :: average
    type(average_user_t),  intent(inout) :: user
    logical,               intent(inout) :: error
    !
    type(average_prog_t) :: prog
    character(len=*), parameter :: rname='AVERAGE>MAIN'
    !
    call cubemain_message(mainseve%trace,rname,'Welcome')
    !
    call user%toprog(prog,error)
    if (error) return
    call prog%header(error)
    if (error) return
    call cubeadm_timing_prepro2process()
    call prog%data(error)
    if (error) return
    call cubeadm_timing_process2postpro()
  end subroutine cubemain_average_main
  !
  !----------------------------------------------------------------------
  !
  subroutine cubemain_average_user_toprog(user,prog,error)
    use cubeadm_get
    use cubetools_header_methods
    !----------------------------------------------------------------------
    !
    !----------------------------------------------------------------------
    class(average_user_t), intent(in)    :: user
    type(average_prog_t),  intent(out)   :: prog
    logical,               intent(inout) :: error
    !
    integer(kind=chan_k) :: nnoi1,nnoi2
    character(len=*), parameter :: rname='AVERAGE>USER>TOPROG'
    !
    call cubemain_message(mainseve%trace,rname,'Welcome')
    !
    call cubeadm_cubeid_get_header(average%comm,ione,user%cubeids,code_access_imaset,code_read,prog%one,error)
    if (error) return
    call cubeadm_cubeid_get_header(average%comm,itwo,user%cubeids,code_access_imaset,code_read,prog%two,error)
    if (error) return
    !
    prog%donoise = user%donoise
    if (prog%donoise) then
       call cubeadm_cubeid_get_header(average%noise,ione,user%noiseids,code_access_imaset,code_read,&
            prog%noise1,error)
       if (error) return
       call cubeadm_cubeid_get_header(average%noise,itwo,user%noiseids,code_access_imaset,code_read,&
            prog%noise2,error)
       if (error) return
       !
       call cubetools_header_get_nchan(prog%noise1%head,nnoi1,error)
       if (error) return       
       call cubetools_header_get_nchan(prog%noise2%head,nnoi2,error)
       if (error) return       
       if (nnoi2.gt.1) then
          error = .true.
          call cubemain_message(seve%e,rname,'Noise for first cube has more than 1 channel, not supported')
       endif
       if (nnoi2.gt.1) then
          error = .true.
          call cubemain_message(seve%e,rname,'Noise for second cube has more than 1 channel, not supported')
       endif
       if (error) return
       !
       prog%loop => cubemain_average_prog_noise_loop
    else
       prog%loop => cubemain_average_prog_weight_loop
    endif
    !
    if (user%doweight) then
       prog%weights = user%weights
    else
       prog%weights = 1.0
    endif
    !
    prog%dofamily = user%dofamily
    prog%family   = user%family
  end subroutine cubemain_average_user_toprog
  !
  !----------------------------------------------------------------------
  !
  subroutine cubemain_average_prog_header(prog,error)
    use cubetools_header_methods
    use cubedag_allflags 
    use cubedag_node
    use cubeadm_clone
    !----------------------------------------------------------------------
    !
    !----------------------------------------------------------------------
    class(average_prog_t), intent(inout) :: prog
    logical,               intent(inout) :: error
    !
    character(len=*), parameter :: rname='AVERAGE>PROG>HEADER'
    !
    call cubemain_message(mainseve%trace,rname,'Welcome')
    !
    call prog%consistency(error)
    if (error) return
    call cubeadm_clone_header(prog%one,[flag_average,flag_cube],prog%averaged,error)
    if (error) return
    call cubeadm_clone_header(prog%one,[flag_average,flag_weight],prog%weight,error)
    if (error) return
    !
    if (prog%dofamily) then
       call cubedag_node_set_family(prog%averaged,prog%family,error)
       if (error)  return
       call cubedag_node_set_family(prog%weight,prog%family,error)
       if (error)  return
    endif
    !
    call cubetools_header_add_observatories(prog%two%head,prog%averaged%head,error)
    if (error) return
    call cubetools_header_add_observatories(prog%two%head,prog%weight%head,error)
    if (error) return    
    call cubetools_header_put_array_unit('---', prog%weight%head,error)
    if (error) return
  end subroutine cubemain_average_prog_header
  !
  subroutine cubemain_average_prog_consistency(prog,error)
    use cubetools_consistency_methods
    !----------------------------------------------------------------------
    !
    !----------------------------------------------------------------------
    class(average_prog_t), intent(inout) :: prog
    logical,               intent(inout) :: error
    !
    logical :: prob
    character(len=*), parameter :: rname='AVERAGE>PROG>CONSISTENCY'
    !
    call cubemain_message(mainseve%trace,rname,'Welcome')
    !
    prob = .false.
    call cubetools_consistency_grid('Input cube #1',prog%one%head,'Input cube #2',prog%two%head,prob,error)
    if(error) return
    !
    if (prog%donoise) then
       call cubetools_consistency_signal_noise('Input cube #1',prog%one%head,&
            'Noise #1',prog%noise1%head,prob,error)
       if (error) return
       call cubetools_consistency_signal_noise('Input cube #2',prog%two%head,&
            'Noise #2',prog%noise2%head,prob,error)
       if (error) return
    endif
    if (cubetools_consistency_failed(rname,prob,error)) return
    !
  end subroutine cubemain_average_prog_consistency
  !
  subroutine cubemain_average_prog_data(prog,error)
    use cubeadm_opened
    !----------------------------------------------------------------------
    !
    !----------------------------------------------------------------------
    class(average_prog_t), intent(inout) :: prog
    logical,               intent(inout) :: error
    !
    type(cubeadm_iterator_t) :: iter
    character(len=*), parameter :: rname='AVERAGE>PROG>DATA'
    !
    call cubemain_message(mainseve%trace,rname,'Welcome')
    !
    call cubeadm_datainit_all(iter,error)
    if (error) return
    !$OMP PARALLEL DEFAULT(none) SHARED(prog,error) FIRSTPRIVATE(iter)
    !$OMP SINGLE
    do while (cubeadm_dataiterate_all(iter,error))
       if (error) exit
       !$OMP TASK SHARED(prog) FIRSTPRIVATE(iter,error)
       if (.not.error) &
         call prog%loop(iter%first,iter%last,error)
       !$OMP END TASK
    enddo ! ie
    !$OMP END SINGLE
    !$OMP END PARALLEL
  end subroutine cubemain_average_prog_data
  !
  subroutine cubemain_average_prog_noise_loop(prog,first,last,error)
    use cubeadm_entryloop
    use cubemain_image_real
    !----------------------------------------------------------------------
    !
    !----------------------------------------------------------------------
    class(average_prog_t), intent(inout) :: prog
    integer(kind=entr_k),  intent(in)    :: first
    integer(kind=entr_k),  intent(in)    :: last
    logical,               intent(inout) :: error
    !
    integer(kind=entr_k) :: ie
    type(image_t) :: one,two,noi1,noi2,averaged,weight
    character(len=*), parameter :: rname='AVERAGE>NOISE>LOOP'
    !
    call one%init(prog%one,error)
    if (error) return
    call two%init(prog%two,error)
    if (error) return
    call noi1%init(prog%noise1,error)
    if (error) return
    call noi2%init(prog%noise2,error)
    if (error) return
    
    call averaged%reallocate('averaged',prog%one%head%arr%n%l,prog%one%head%arr%n%m,error)
    if (error) return
    call weight%reallocate('weight',prog%one%head%arr%n%l,prog%one%head%arr%n%m,error)
    if (error) return
    !
    do ie=first,last
      call cubeadm_entryloop_iterate(ie,error)
      if (error) return
      call prog%act_noise(ie,one,noi1,two,noi2,averaged,weight,error)
      if (error) return
    enddo
  end subroutine cubemain_average_prog_noise_loop
  !
  subroutine cubemain_average_prog_act_noise(prog,ie,one,noi1,two,noi2,averaged,weight,error)
    use cubemain_image_real
    !----------------------------------------------------------------------
    !
    !----------------------------------------------------------------------
    class(average_prog_t), intent(inout) :: prog
    integer(kind=entr_k),  intent(in)    :: ie
    type(image_t),         intent(inout) :: one
    type(image_t),         intent(inout) :: noi1
    type(image_t),         intent(inout) :: two
    type(image_t),         intent(inout) :: noi2
    type(image_t),         intent(inout) :: averaged
    type(image_t),         intent(inout) :: weight
    logical,               intent(inout) :: error
    !
    integer(kind=pixe_k) :: il,im
    real(kind=4) :: wei1,wei2
    character(len=*), parameter :: rname='AVERAGE>PROG>ACT>NOISE'
    !
    call one%get(prog%one,ie,error)
    if (error) return
    call two%get(prog%two,ie,error)
    if (error) return
    call noi1%get(prog%noise1,ie,error)
    if (error) return
    call noi2%get(prog%noise2,ie,error)
    if (error) return
    do im=1,prog%one%head%arr%n%m
       do il=1,prog%one%head%arr%n%l
          wei1 = prog%weights(1)/noi1%z(il,im)**2
          wei2 = prog%weights(2)/noi2%z(il,im)**2
          call prog%compute_average(one%z(il,im),wei1,two%z(il,im),wei2,&
               averaged%z(il,im),weight%z(il,im),error)
          if (error) return
       enddo ! il
    enddo ! im
    call averaged%put(prog%averaged,ie,error)
    if (error) return
    call weight%put(prog%weight,ie,error)
    if (error) return
  end subroutine cubemain_average_prog_act_noise
  !
  subroutine cubemain_average_prog_weight_loop(prog,first,last,error)
    use cubeadm_entryloop
    use cubemain_image_real
    !----------------------------------------------------------------------
    !
    !----------------------------------------------------------------------
    class(average_prog_t), intent(inout) :: prog
    integer(kind=entr_k),  intent(in)    :: first
    integer(kind=entr_k),  intent(in)    :: last
    logical,               intent(inout) :: error
    !
    integer(kind=entr_k) :: ie
    type(image_t) :: one,two,averaged,weight
    character(len=*), parameter :: rname='AVERAGE>WEIGHT>LOOP'
    !
    call one%init(prog%one,error)
    if (error) return
    call two%init(prog%two,error)
    if (error) return
    call averaged%reallocate('averaged',prog%one%head%arr%n%l,prog%one%head%arr%n%m,error)
    if (error) return
    call weight%reallocate('weight',prog%one%head%arr%n%l,prog%one%head%arr%n%m,error)
    if (error) return
    !
    do ie=first,last
      call cubeadm_entryloop_iterate(ie,error)
      if (error) return
      call prog%act_weight(ie,one,two,averaged,weight,error)
      if (error) return
    enddo
  end subroutine cubemain_average_prog_weight_loop
  !
  subroutine cubemain_average_prog_act_weight(prog,ie,one,two,averaged,weight,error)
    use cubemain_image_real
    !----------------------------------------------------------------------
    !
    !----------------------------------------------------------------------
    class(average_prog_t), intent(inout) :: prog
    integer(kind=entr_k),  intent(in)    :: ie
    type(image_t),         intent(inout) :: one
    type(image_t),         intent(inout) :: two
    type(image_t),         intent(inout) :: averaged
    type(image_t),         intent(inout) :: weight
    logical,               intent(inout) :: error
    !
    integer(kind=pixe_k) :: il,im
    character(len=*), parameter :: rname='AVERAGE>PROG>ACT>WEIGHT'
    !
    call one%get(prog%one,ie,error)
    if (error) return
    call two%get(prog%two,ie,error)
    if (error) return
    do im=1,prog%one%head%arr%n%m
       do il=1,prog%one%head%arr%n%l
          call prog%compute_average(one%z(il,im),prog%weights(1),two%z(il,im),prog%weights(2),&
               averaged%z(il,im),weight%z(il,im),error)
          if (error) return
       enddo ! il
    enddo ! im
    call averaged%put(prog%averaged,ie,error)
    if (error) return
    call weight%put(prog%weight,ie,error)
    if (error) return
  end subroutine cubemain_average_prog_act_weight
  !
  subroutine cubemain_average_prog_compute_average(prog,val1,wei1,val2,wei2,ave,wei,error)
    use cubetools_nan
    !----------------------------------------------------------------------
    !
    !----------------------------------------------------------------------
    class(average_prog_t), intent(inout) :: prog
    real(kind=sign_k),     intent(in)    :: val1
    real(kind=sign_k),     intent(in)    :: wei1
    real(kind=sign_k),     intent(in)    :: val2
    real(kind=sign_k),     intent(in)    :: wei2
    real(kind=sign_k),     intent(out)   :: ave
    real(kind=sign_k),     intent(out)   :: wei
    logical,               intent(inout) :: error
    !
    character(len=*), parameter :: rname='AVERAGE>PROG>COMPUTE>AVERAGE'
    !
    if (ieee_is_nan(val1).and.ieee_is_nan(val2)) then
       ave = gr4nan
       wei = 0
    else if (ieee_is_nan(val1)) then
       ave = val2
       wei = wei2
    else if (ieee_is_nan(val2)) then
       ave = val1
       wei = wei1
    else
       wei = wei1+wei2
       ave = (val1*wei1+val2*wei2)/wei
    endif
  end subroutine cubemain_average_prog_compute_average
end module cubemain_average
!
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
