module cubedag_walker
  use cubedag_messaging
  use cubedag_parameters
  use cubedag_node
  use cubedag_link
  use cubedag_dag
  use cubedag_types

  integer(kind=4), parameter :: topomarker_null=0
  integer(kind=4), parameter :: topomarker_done=1
  type(cubedag_link_t) :: link
  integer(kind=entr_k) :: current

contains
  !
  subroutine cubedag_childwalker_reset(start,callback,error)
    !-------------------------------------------------------------------
    ! 
    !-------------------------------------------------------------------
    class(cubedag_node_object_t), pointer       :: start
    external                                    :: callback
    logical,                      intent(inout) :: error
    ! Local
    character(len=*), parameter :: rname='CHILDWALKER>RESET'
    !
    call cubedag_walker_reset(cubedag_walker_getchildren,start,callback,error)
  end subroutine cubedag_childwalker_reset
  !
  subroutine cubedag_parentwalker_reset(start,callback,error)
    !-------------------------------------------------------------------
    ! 
    !-------------------------------------------------------------------
    class(cubedag_node_object_t), pointer       :: start
    external                                    :: callback
    logical,                      intent(inout) :: error
    ! Local
    character(len=*), parameter :: rname='PARENTWALKER>RESET'
    !
    call cubedag_walker_reset(cubedag_walker_getparent,start,callback,error)
  end subroutine cubedag_parentwalker_reset
  !
  subroutine cubedag_walker_reset(cubedag_walker_getlink,start,callback,error)
    !-------------------------------------------------------------------
    ! Set the current walker in memory to start from given commit
    ! The callback is called at load time, on each parent-child pair,
    ! with the signature:
    !    call callback(pid,cid,error)
    ! If the callback raises an error, the load stops and
    ! cubedag_walker_reset returns an error
    !-------------------------------------------------------------------
    interface
      recursive subroutine cubedag_walker_getlink(obj,callback,error)
        use cubedag_types
        class(cubedag_node_object_t), pointer       :: obj
        external                                    :: callback
        logical,                      intent(inout) :: error
      end subroutine cubedag_walker_getlink
      subroutine callback(par,chi,error)
        use cubedag_types
        class(cubedag_node_object_t), pointer       :: par
        class(cubedag_node_object_t), pointer       :: chi
        logical,                      intent(inout) :: error
      end subroutine callback
    end interface
    external                                    :: cubedag_walker_getlink
    class(cubedag_node_object_t), pointer       :: start
    logical,                      intent(inout) :: error
    ! Local
    character(len=*), parameter :: rname='WALKER>RESET'
    !
    ! ZZZ Unclear if collection should be done here or at first
    !     cubedag_walker_next
    !
    call cubedag_link_reallocate(link,ix%next-1,error)
    if (error)  return
    !
    ! Prepare topomarkers
    ix%topomarker(1:ix%next-1) = topomarker_null
    !
    ! Insert start point
    link%n = 1
    link%list(link%n)%p => start
    !
    ! ZZZ Might implement several sorting modes (like git_revwalk_sorting)
    call cubedag_walker_getlink(start,callback,error)
    if (error)  return
    !
    current = 0
  end subroutine cubedag_walker_reset
  !
  recursive subroutine cubedag_walker_getchildren(par,callback,error)
    !---------------------------------------------------------------------
    ! Recursive childwalking
    !---------------------------------------------------------------------
    class(cubedag_node_object_t), pointer       :: par       ! Starting point (already in list)
    interface
      subroutine callback(par,chi,error)
        use cubedag_types
        class(cubedag_node_object_t), pointer       :: par
        class(cubedag_node_object_t), pointer       :: chi
        logical,                      intent(inout) :: error
      end subroutine callback
    end interface
    logical,                      intent(inout) :: error
    ! Local
    character(len=*), parameter :: rname='WALKER>GETCHILDREN'
    integer(kind=entr_k) :: first,last,ilist,cent
    class(cubedag_node_object_t), pointer :: chi
    !
    first = link%n+1
    do ilist=1,par%node%children%n
      chi => par%node%children%list(ilist)%p
      !
      call callback(par,chi,error)
      if (error)  return
      !
      cent = chi%node%ient
      if (ix%topomarker(cent).eq.topomarker_done)  cycle  ! Do not store duplicate nodes
      if (link%n.ge.size(link%list)) then
        call cubedag_message(seve%e,rname,'Internal error: list exhausted')
        error = .true.
        return
      endif
      link%n = link%n+1
      link%list(link%n)%p => chi
      ix%topomarker(cent) = topomarker_done
    enddo
    last = link%n
    !
    do ilist=first,last
      call cubedag_walker_getchildren(link%list(ilist)%p,callback,error)
      if (error)  return
    enddo
  end subroutine cubedag_walker_getchildren
  !
  recursive subroutine cubedag_walker_getparent(chi,callback,error)
    !---------------------------------------------------------------------
    ! Recursive parentwalking
    !---------------------------------------------------------------------
    class(cubedag_node_object_t), pointer       :: chi       ! Starting point (already in list)
    interface
      subroutine callback(par,chi,error)
        use cubedag_types
        class(cubedag_node_object_t), pointer       :: par
        class(cubedag_node_object_t), pointer       :: chi
        logical,                      intent(inout) :: error
      end subroutine callback
    end interface
    logical,                      intent(inout) :: error
    ! Local
    character(len=*), parameter :: rname='WALKER>GETPARENT'
    integer(kind=entr_k) :: first,last,ilist,pent
    class(cubedag_node_object_t), pointer :: par
    !
    first = link%n+1
    do ilist=1,chi%node%parents%n
      par => chi%node%parents%list(ilist)%p
      !
      call callback(par,chi,error)
      if (error)  return
      !
      pent = par%node%ient
      if (ix%topomarker(pent).eq.topomarker_done)  cycle  ! Do not store duplicate nodes
      if (link%n.ge.size(link%list)) then
        call cubedag_message(seve%e,rname,'Internal error: list exhausted')
        error = .true.
        return
      endif
      link%n = link%n+1
      link%list(link%n)%p => par
      ix%topomarker(pent) = topomarker_done
    enddo
    last = link%n
    !
    do ilist=first,last
      call cubedag_walker_getparent(link%list(ilist)%p,callback,error)
      if (error)  return
    enddo
  end subroutine cubedag_walker_getparent
  !
  function cubedag_childwalker_next(next)
    !-------------------------------------------------------------------
    ! Get the next commit ID. Evaluate to .false. if all done.
    !-------------------------------------------------------------------
    logical :: cubedag_childwalker_next
    class(cubedag_node_object_t), pointer :: next
    cubedag_childwalker_next = cubedag_walker_next(next)
  end function cubedag_childwalker_next
  !
  function cubedag_parentwalker_next(next)
    !-------------------------------------------------------------------
    ! Get the next commit ID. Evaluate to .false. if all done.
    !-------------------------------------------------------------------
    logical :: cubedag_parentwalker_next
    class(cubedag_node_object_t), pointer :: next
    cubedag_parentwalker_next = cubedag_walker_next(next)
  end function cubedag_parentwalker_next
  !
  function cubedag_walker_next(next)
    !-------------------------------------------------------------------
    ! Get the next commit ID. Evaluate to .false. if all done.
    !-------------------------------------------------------------------
    logical :: cubedag_walker_next
    class(cubedag_node_object_t), pointer :: next
    !
    if (current.ge.link%n) then
      cubedag_walker_next = .false.
      next => null()
      return
    endif
    cubedag_walker_next = .true.
    current = current+1
    next => link%list(current)%p
    !
  end function cubedag_walker_next
  !
  function cubedag_walker_contains(link,obj)
    !-------------------------------------------------------------------
    ! Return .true. if the list contains the named ID
    ! ZZZ This is unefficient (N^2)
    !-------------------------------------------------------------------
    logical :: cubedag_walker_contains
    type(cubedag_link_t),         intent(in) :: link
    class(cubedag_node_object_t), pointer    :: obj
    ! Local
    integer(kind=entr_k) :: ilist
    do ilist=1,link%n
      if (associated(link%list(ilist)%p,obj)) then
        cubedag_walker_contains = .true.
        return
      endif
    enddo
    cubedag_walker_contains = .false.
  end function cubedag_walker_contains
  !
  subroutine cubedag_walker_null(par,chi,error)
    !-------------------------------------------------------------------
    ! Does nothing
    !-------------------------------------------------------------------
    class(cubedag_node_object_t), pointer       :: par
    class(cubedag_node_object_t), pointer       :: chi
    logical,                      intent(inout) :: error
    !
    return
  end subroutine cubedag_walker_null
  !
end module cubedag_walker
