subroutine feather_comm(line,error)
  use image_def
  use gkernel_interfaces
  use gbl_message
  use clean_arrays
  !---------------------------------------------------------------------
  ! @ private
  !
  ! IMAGER
  !   Support routine for command
  !
  ! FEATHER [/FILE FileMerge FileHigh FileLow] [/REPROJECT]
  !   Uses  FEATHER_RADIUS
  !         FEATHER_SCALE
  !         FEATHER_RANGE[2]
  !---------------------------------------------------------------------
  character(len=*), intent(in) :: line
  logical, intent(out) :: error
  !
  integer, parameter :: o_file=1
  integer, parameter :: o_reproject=2
  character(len=1), parameter :: question_mark='?'
  character(len=*), parameter :: rname='FEATHER'
  !
  type(gildas), save :: hall
  real, save :: feather_radius=15.0
  real, save :: feather_ratio=1.0
  real, save :: feather_expo=8.0
  real, save :: feather_range(2)=[0.,0.]
  real :: actual_range(2)
  logical, save :: feather_init=.true.
  logical :: auto
  character(len=filename_length) :: nameh,namel,name_out
  character(len=60) :: argum
  integer :: narg
  integer :: n, ier
  !
  error = .false.
  !
  if (feather_init) then
    ! At first call, just initialize and do nothing else
    call sic_def_real('FEATHER_RADIUS',feather_radius,0,0,.false.,error)  
    call sic_def_real('FEATHER_RATIO', feather_ratio,0,0,.false.,error)  
    call sic_def_real('FEATHER_EXPO', feather_expo,0,0,.false.,error)  
    call sic_def_real('FEATHER_RANGE', feather_range,1,2,.false.,error)  
    feather_init = .false.
    return
  endif
  !
  narg = sic_narg(0)
  if (narg.eq.1) then
    call sic_ch(line,0,1,argum,n,.true.,error)
    if (argum(1:1).eq.question_mark) then
      if (argum.eq.'?' .or. argum.eq.'??' .or. argum.eq.'???') then
        call exec_program('@ i_feather')
        return
      endif
    endif
  endif
  !
  if (narg.ne.0) then
    call map_message(seve%e,rname,'FEATHER accepts no argument (except ?)')
    error = .true.
    return
  endif
  !
  ! Set current range
  actual_range = feather_range
  if (feather_range(1).eq.0) actual_range(1) = feather_radius/1.15
  if (feather_range(2).eq.0) actual_range(2) = feather_radius*1.15
  !
  ! Auto reproject
  auto = sic_present(o_reproject,0)
  !
  if (sic_present(o_file,0)) then
    ! File version
    call sic_ch(line,o_file,3,namel,n,.true.,error)
    if (error) return
    call sic_ch(line,o_file,1,name_out,n,.true.,error)
    call sic_ch(line,o_file,2,nameh,n,.true.,error)
    !
    call t_uv_feather(nameh,namel,name_out, &
      & feather_radius,feather_ratio,feather_expo,actual_range,auto,error)
  else 
    ! Default buffer version
    ! Ideally, one would like to specify any buffer, but
    ! we do not know how to set a pointer to a Memory region,
    ! except using Fortran 2008 (which is marginally accepted yet ?)
    if (.not.allocated(dsky)) then
      call map_message(seve%e,rname,'SKY image is not defined')
      error = .true.
    else
      hsky%r3d => dsky
    endif
    if (.not.associated(hshort%r3d)) then
      call map_message(seve%e,rname,'SHORT image is not defined')
      error = .true.
    endif
    if (error) return
    !
    if (associated(hall%r3d)) deallocate(hall%r3d)
    call sic_delvariable ('FEATHERED',.false.,error)
    !
    call gildas_null(hall)
    call gdf_copy_header(hsky,hall,error)
    call gdf_allocate(hall,error)
    if (error) return
    !
    ! Do the job
    call c_uv_feather(hall,hsky,hshort, &
      & feather_radius,feather_ratio,feather_expo,actual_range,auto,error)
    if (error) then
      deallocate(hall%r3d,stat=ier)
      return
    endif
    call sic_mapgildas('FEATHERED',hall,error,hall%r3d)
  endif
end subroutine feather_comm
!
subroutine t_uv_feather(nameh,namel,name_out,uvradius,scale,expo,range, &
  & auto,error)
  use image_def
  use gkernel_interfaces
  use gbl_message
  !---------------------------------------------------------------------
  ! @ private
  !
  ! IMAGER
  !   Support routine for command
  !
  ! FEATHER /FILE FileMerge FileHigh FileLow [/REPROJECT]
  !   Uses  FEATHER_RADIUS
  !         FEATHER_SCALE
  !         FEATHER_RANGE[2]
  ! "Feather" (in the UV plane) two data cubes
  !---------------------------------------------------------------------
  character(len=*), intent(in) :: nameh    ! HIGHres file
  character(len=*), intent(in) :: namel    ! LOWres file
  character(len=*), intent(in) :: name_out ! Merged file
  real, intent(in) ::  uvradius            ! Transition radius
  real, intent(in) ::  scale               ! Scale factor LOW / HIGH
  real, intent(in) ::  expo                ! Sharpness of transition
  real, intent(in) ::  range(2)            ! Range of overlap (m) 
  logical, intent(in) :: auto              ! Auto reproject
  logical, intent(out) :: error            ! Error flag
  !
  character(len=*), parameter :: rname='FEATHER'
  type(gildas) :: high, low, all, htmp
  real :: ratio
  integer :: ier
  integer :: sys_code
  logical :: auto_space
  logical :: err
  !
  error = .false.
  ratio = scale
  auto_space = auto
  !
  if (uvradius.eq.0) error = .true.
  if (ratio.eq.0) ratio = 1.0
  if (range(2).le.range(1)) error = .true.
  if (error) then
    call map_message(seve%e,rname,'Invalid FEATHER_* input values')
    return
  endif
  !
  ! Read Headers Model & Image 
  call gildas_null(high)
  call gildas_null(low)
  call gildas_null(all)
  !
  call sic_parse_file (nameh,' ','.lmv-sky',high%file)
  call gdf_read_header (high,error)
  if (error) return
  call sic_parse_file (namel,' ','.lmv-sky',low%file)
  call gdf_read_header (low,error)
  if (error) return
  !
  ier = 0
  call s_uv_consistency (high,low,error,ier)
  !
  if (error) then
    if (ier.eq.0) return ! Spectro error or Swapping error
    !
    ! Space error : reproject LOW on HIGH if requested
    if (auto_space) then
      call map_message(seve%w,rname,'Images do not match, resampling LOWres one')
      continue
    else
      call gdf_close_image(high,err)
      call gdf_close_image(low,err)
      call map_message(seve%e,rname,'Images do not match')
      return
    endif
  else
    auto_space = .false.
  endif
  !
  if (auto_space) then
    call gdf_close_image(low,err)
    call gildas_null(htmp)
    htmp%file = low%file
    call gdf_read_header (htmp,error)
    if (error) return
    call s_reproject_init (htmp,high,low,sys_code,error)
    if (error) return
    call gdf_allocate(low,error)
    if (error) return
    call gdf_allocate(htmp,error)
    if (error) return
    call gdf_read_data(htmp,htmp%r3d,error)
    if (error) return
    call s_reproject_do(htmp,low,sys_code,error)
    if (error) return
    call gdf_close_image(htmp,err)
  else
    call gdf_allocate (low,error)
    if (error) return
    call gdf_read_data (low,low%r3d,error)
    if (error) return
    call gdf_close_image(low,err)
  endif
  !
  ! Allocate and Read data - Rank 3 at this stage already...
  call gdf_allocate (high,error)
  if (error) return
  call gdf_read_data (high,high%r3d,error)
  if (error) return
  !
  ! Define output image. Comes from the High Resolution one...
  call gdf_copy_header(high,all,error)
  if (error) return
  call gdf_allocate(all,error)
  if (error) return
  !
  call s_uv_hybrid (high,low,all,uvradius,ratio,expo,range,error)
  if (error) return
  !
  call sic_parse_file (name_out,' ','.lmv-sky',all%file)
  call gdf_write_image(all,all%r3d,error)
  !
  call gdf_close_image(high,err)
  !! Not needed with gdf_write_image call gdf_close_image(all,err)
  !
end subroutine t_uv_feather
!
subroutine c_uv_feather(all,high,low,uvradius,scale,expo,range, &
  & auto,error)
  use image_def
  use gkernel_interfaces
  use gbl_message
  !---------------------------------------------------------------------
  ! @ private
  !
  ! IMAGER
  !   Support routine for command
  !
  ! FEATHER [/REPROJECT]
  !   Uses  FEATHER_RADIUS
  !         FEATHER_SCALE
  !         FEATHER_RANGE[2]
  ! "Feather" (in the UV plane) the SKY and SHOW data cubes
  !   and put the result in FEATHERED 
  !---------------------------------------------------------------------
  type(gildas), intent(in) :: high
  type(gildas), intent(in) :: low 
  type(gildas), intent(inout) :: all
  real, intent(in) ::  uvradius            ! Transition radius
  real, intent(in) ::  scale               ! Scale factor LOW / HIGH
  real, intent(in) ::  expo                ! Sharpness of transition
  real, intent(in) ::  range(2)            ! Range of overlap (m) 
  logical, intent(in) :: auto              ! Auto reproject
  logical, intent(out) :: error            ! Error flag
  !
  character(len=*), parameter :: rname='FEATHER'
  type(gildas) :: htmp
  !
  real :: ratio
  integer :: ier
  integer :: sys_code
  logical :: auto_space
  logical :: err
  !
  error = .false.
  ratio = scale
  auto_space = auto
  !
  if (uvradius.eq.0) error = .true.
  if (ratio.eq.0) ratio = 1.0
  if (range(2).le.range(1)) error = .true.
  if (error) then
    call map_message(seve%e,rname,'Invalid FEATHER_* input values')
    return
  endif
  !
  ier = 0
  call s_uv_consistency (high,low,error,ier)
  !
  if (error) then
    if (ier.eq.0) return ! Spectro error or Swapping error
    !
    ! Space error : reproject LOW on HIGH if requested
    if (auto_space) then
      call map_message(seve%w,rname,'Images do not match, resampling LOWres one')
      continue
    else
      call map_message(seve%e,rname,'Images do not match')
      return
    endif
  else
    auto_space = .false.
  endif
  !
  if (auto_space) then
    call gildas_null(htmp)
    call gdf_copy_header(low,htmp,error)
    if (error) return
    call s_reproject_init (low,high,htmp,sys_code,error)
    if (error) return
    call gdf_allocate(htmp,error)
    if (error) return    
    call s_reproject_do (low,htmp,sys_code,error)
    if (error) return
    call s_uv_hybrid (high,htmp,all,uvradius,ratio,expo,range,error)
    if (error) return
  else
    call s_uv_hybrid (high,low,all,uvradius,ratio,expo,range,error)
  endif
  !
end subroutine c_uv_feather
!
subroutine s_uv_consistency(high,low,error,ier)
  use image_def
  use gbl_message
  use gkernel_interfaces
  !---------------------------------------------------------------------
  ! @ public
  ! 
  ! IMAGER
  !   Support routine for command FEATHER
  !
  ! Verify data cube consistencies
  !---------------------------------------------------------------------
  type (gildas), intent(in) :: high        ! High resolution image
  type (gildas), intent(in) :: low         ! Low resolution image
  logical, intent(out) :: error            ! Error flag
  integer, intent(out) :: ier
  !
  character(len=*), parameter :: rname='FEATHER'
  real(kind=8), parameter :: pi=3.14159265358979323846d0
  !
  character(len=160) :: chain
  real :: tole=1e-4
  logical :: equal
  integer :: i
  real(8), dimension(2) :: bhigh, blow, bhinc, blinc
  real(4) :: rhigh, rlow
  !
  real(8) :: lambda
  real :: uinc, vinc, scale
  real :: threshold, expo
  real, allocatable :: flux_ratio(:), flux2(:)
  real :: umin2, umax2, xx, ww
  integer :: jer
  !
  ! Check spectroscopic consistency
  error = .false.
  !
  ier = 0
  call spectrum_consistency(rname,high,low,tole,error)
  if (error) return
  !
  jer = 0
  !
  ! Test that the HIGH image is the largest one and has the smallest beam
  call gdf_compare_shape (high,low,equal)
  if (.not.equal) then
    call map_message(seve%w,rname,'Images do not match')
    bhinc = abs(high%gil%inc(1:2))*180*3600/pi
    blinc = abs(low%gil%inc(1:2))*180*3600/pi
    bhigh = bhinc * high%gil%dim(1:2)
    blow  = blinc * low%gil%dim(1:2)
    if (any(bhigh.ne.blow)) then
      call map_message(seve%i,rname,'Image                Field (")              Pixel (")             Size')
      write(chain,'(A10,2(2X,F10.3,A,F10.3),I8,A,I7)') 'HIGHres ',bhigh(1),' x',bhigh(2) &
        & ,bhinc(1),' x',bhinc(2),high%gil%dim(1),' x',high%gil%dim(2)
      call map_message(seve%i,rname,chain)
      write(chain,'(A10,2(2X,F10.3,A,F10.3),I8,A,I7)') 'LOWres ',blow(1),' x',blow(2) &
        & ,blinc(1),' x',blinc(2),low%gil%dim(1),' x',low%gil%dim(2)
      call map_message(seve%i,rname,chain)
      jer = 1
    endif
  endif
  rhigh = sqrt(high%gil%majo*high%gil%mino)
  rlow  = sqrt(low%gil%majo*low%gil%mino)
  if (rhigh.ge.rlow) jer = jer+2
  if (jer.ne.0) then
    ier = 0
    if (mod(jer,2).ne.0) then
      call map_message(seve%w,rname,'Field of view of "HIGHres" and "LOWres" images differ')
      ier = 1
      jer = jer/2
    endif
    if (jer.ne.0) then
      call map_message(seve%e,rname,'Resolution of "LOWres" image is better than that of "HIGHres" image')
      call map_message(seve%e,rname,'Consider swapping images !...')
    endif
    error = .true.
  endif
end subroutine s_uv_consistency
!
subroutine s_uv_hybrid(high,low,all,uvradius,ratio,exponent,range,error)
  use image_def
  use gbl_message
  use gkernel_interfaces, no_interface=>fourt
  use imager_interfaces, except_this => expand, and_this=> s_uv_hybrid
  !---------------------------------------------------------------------
  ! @ public
  !
  ! IMAGER
  !   Support for command FEATHER
  !
  !   Take HIGH map (high resolution map) 
  !   Take LOW  map (map with shortest spacings)
  !
  !   Make oversampled Fourier Transform of boths
  !   Compute the truncation function f(r)
  !   Make the Truncated compact Fourier Transform, f(r) x T(LOW)
  !   Make the complement long baseline Fourier Transform, (1-f(r)) x T(HIGH)
  !   Sum them T(ALL) = f(r) x T(LOW) + (1-f(r)) x T(HIGH)
  !   Make the inverse Fourier Transform
  !   Truncate the resulting image to original size and Mask
  !---------------------------------------------------------------------
  type (gildas), intent(in) :: high        ! High resolution image
  type (gildas), intent(in) :: low         ! Low resolution image
  type (gildas), intent(inout) :: all      ! Combined image
  real, intent(in) ::  uvradius            ! Transition radius
  real, intent(in) ::  ratio               ! Scale factor LOW / HIGH
  real, intent(in) ::  exponent            ! Sharpness of transition                
  real, intent(in) ::  range(2)            ! Range of overlap (m) 
  logical, intent(out) :: error            ! Error flag
  !
  character(len=*), parameter :: rname='FEATHER'
  real(kind=8), parameter :: pi=3.14159265358979323846d0
  !
  character(len=80) :: chain
  real, pointer :: dhigh(:,:,:),dlow(:,:,:)
  real, pointer :: dout(:,:,:)
  real, allocatable :: f(:,:), x(:), y(:)
  complex, allocatable :: chigh(:,:), wfft(:), cout(:,:)
  complex, allocatable, target :: clow(:,:)
  complex, pointer :: cbig(:,:)
  real, allocatable :: rmask(:,:)
  !
  integer :: nx,ny,nc,ic,nn,dim(2),ier,mx,my,i,j, expand
  real(8) :: lambda
  real :: uinc, vinc, scale
  real :: threshold, expo
  real, allocatable :: flux_ratio(:), flux2(:)
  real :: umin2, umax2, xx, ww
  !
  nx = high%gil%dim(1)
  ny = high%gil%dim(2)
  nc = high%gil%dim(3)
  !
  mx = low%gil%dim(1)
  my = low%gil%dim(2)
  !
  dout => all%r3d
  !
  ! Skip blanking - Set it to zero, although this  will unavoidably
  ! cause some ringing issues if there is such a blanking.
  ! A better procedure would be to interpolate to Zero with a scale
  ! length larger than the (HIGHres) Beam size
  if (high%gil%eval.ge.0) then
    allocate(dhigh(high%gil%dim(1),high%gil%dim(2),high%gil%dim(3)), stat=ier)
    dhigh = high%r3d  ! Copy 
    where (abs(dhigh-high%gil%bval).le.high%gil%eval) dhigh = 0.0
  else
    dhigh => high%r3d ! Pointer
  endif
  if (low%gil%eval.ge.0) then
    allocate(dlow(low%gil%dim(1),low%gil%dim(2),low%gil%dim(3)), stat=ier)
    dlow = low%r3d
    where (abs(dlow-low%gil%bval).le.low%gil%eval) dlow = 0.0
  else
    dlow => low%r3d
  endif
  ! 
  ! Get the user units in "m".
  ! Oops, quite a difficult problem, sure ? This is the wavelength !
  lambda = 299792458.d-6/all%gil%freq
  !! Print *,'Lambda ',lambda
  !
  ! Allocate Fourier space
  allocate(chigh(nx,ny),stat=ier)
  allocate(clow(mx,my),stat=ier)
  allocate(wfft(max(nx,ny)),stat=ier)
  allocate(cout(nx,ny),f(nx,ny),stat=ier)
  allocate(x(nx),y(ny),stat=ier)
  nn = 2
  dim = (/nx,ny/)
  !
  ! Compute the combination factor F
  uinc = lambda / (all%gil%inc(1) * nx) 
  vinc = lambda / (all%gil%inc(2) * ny) 
  write(chain,'(A,F8.2,F8.2,A,F8.2)') 'UV Cell size ',uinc,vinc, &
    & ' transition radius ',uvradius
  call map_message(seve%i,rname,chain)
  !
  uinc = uinc / uvradius
  do i=1,nx/2
    x(i) = (i-1)*uinc
  enddo
  do i=nx/2+1,nx
    x(i) = (i-nx-1)*uinc
  enddo
  x = x**2
  !
  vinc = vinc / uvradius
  do i=1,ny/2
    y(i) = (i-1)*vinc
  enddo
  do i=ny/2+1,ny
    y(i) = (i-ny-1)*vinc
  enddo
  y = y**2
  !	
  ! We apply here a sharp truncation, by a exp(-r^(2*expo)) function
  !
  ! The "natural" combination for Single-Dish plus Interferometer data
  ! would be to use as the combination function the Fourier transform 
  ! of the single-dish Beam. But this does not necessarily apply to
  ! arbitrary images that we handle here...
  !
  ! In particular, the initial images may already include the Short
  ! spacings in some way. 
  !
  expo = exponent
  if (expo.eq.0) expo = 8.0
  threshold = log(huge(1.0))**(1.0/expo)
  write(chain,'(A,F6.2,A,F6.2)') 'Threshold ',threshold,' Exponent ',expo
  call map_message(seve%i,rname,chain)
  !
  do j=1,ny
    do i=1,nx
      f(i,j) = x(i) + y(j)            ! This is r^2
      if (f(i,j).lt.threshold) then   ! It falls to Zero at Threshold   
        f(i,j) = exp(-(f(i,j)**expo)) 
      else
        f(i,j) = 0.0
      endif
    enddo
  enddo
  !
  ! Get space for the Flux scale ratio fit, and compute the mask
  umin2 = (range(1)/uvradius)**2
  umax2 = (range(2)/uvradius)**2
  allocate(flux_ratio(nc),flux2(nc),rmask(nx,ny),stat=ier)
  do j=1,ny
    do i=1,nx
      rmask(i,j) = x(i) + y(j)            ! This is r^2
      if (rmask(i,j).gt.umin2 .and. rmask(i,j).lt.umax2) then     
        rmask(i,j) = 1.0
      else
        rmask(i,j) = 0.0 ! Not f(i,j)
      endif
    enddo
  enddo
  !
  expand = (nx*ny)/(mx*my)
  if (expand.ne.1) then
    allocate(cbig(nx,ny),stat=ier)  
  else
    cbig => clow
  endif
  !
  ! Get the beams and rescale the data accordingly
  !! Print *,'Computed F, Expand ',expand
  !
  scale = (high%gil%majo*high%gil%mino)/(low%gil%majo*low%gil%mino)
  !
  do ic = 1,nc
    !! Print *,'IC ',ic
    chigh = cmplx(dhigh(:,:,ic),0.0)
    clow = cmplx(dlow(:,:,ic),0.0)
    dim = (/nx,ny/)
    call fourt (chigh,dim,2,1,1,wfft)
    !
    ! LOWres
    dim = (/mx,my/)
    call fourt (clow,dim,2,1,1,wfft)
    !
    ! Multiply by the other beam
    call mulgau(clow,mx,my, &
    high%gil%majo,high%gil%mino,high%gil%posa*180.0/real(pi), &
    1.0,real(low%gil%inc(1)),real(low%gil%inc(2)),-1) 
    !
    ! Divide by its own beam - order matters...
    call mulgau(clow,mx,my, &
    low%gil%majo,low%gil%mino,low%gil%posa*180.0/real(pi), &
    1.0,real(low%gil%inc(1)),real(low%gil%inc(2)),1) 
    !
    clow = clow * scale * ratio
    !
    ! Expansion part
    if (expand.ne.1) then
      cbig = 0
      ! Load inner quarter
      do j=1,my/2
        cbig(1:mx,j) = clow(1:mx,j)
        cbig(1+nx-mx/2:nx,j) = clow(mx/2+1:mx,j)
      enddo
      do j=my/2+1,my
        cbig(1:mx,j+ny-my) = clow(1:mx,j)
        cbig(1+nx-mx/2:nx,j+ny-my) = clow(mx/2+1:mx,j)
      enddo
      cbig = cbig*expand ! rescale to appropriate units
    endif
    !
    ! At this stage, we have cbig & chigh in same units.
    ! We can compare their relative intensities in the same
    ! UV Plane region by masking both of them by the overlap
    ! region, and fitting a scale factor.
    !
    call scale_factor(nx,ny,cbig,chigh,rmask,flux_ratio(ic),flux2(ic))
    !
    cout = f * cbig + (1.0-f) * chigh
    !
    ! Back transform
    dim = (/nx,ny/) 
    call fourt (cout,dim,2,-1,1,wfft)
    dout(:,:,ic) = real(cout)
  enddo
  !
  xx = 0
  ww = 0
  do ic=1,nc
    xx = xx + flux_ratio(ic)*flux2(ic)
    ww = ww + flux2(ic)
  enddo
  if (ww.gt.0) then
    xx = xx/ww
  else
    xx = sum(flux_ratio)/nc
  endif
  write(chain,'(A,F10.3)') 'Weighted flux factor (HIGHres / LOWres) ',xx
  call map_message(seve%i,rname,chain)
  !!Print *,'           actors   ',flux_ratio(1:nc)
  !
  dout = dout / (nx*ny)
  !
  ! Reset the Blanking where the HIGHres had one
  if (high%gil%eval.ge.0) then
    where (abs(high%r3d-high%gil%bval).le.high%gil%eval) dout = all%gil%bval
  endif
  !
  ! Reset the Extrema
  all%loca%size = all%gil%dim(1)*all%gil%dim(2)*all%gil%dim(3)
  call map_minmax(all)
  !
  deallocate (chigh,clow,cout,wfft,f)
end subroutine s_uv_hybrid
!
subroutine extract_real (c,mx,my,r,nx,ny)  ! checked with cmtore ...
  !-----------------------------------------------------------------
  !     Extract a Real array from a larger Complex array
  !-----------------------------------------------------------------
  integer nx,ny                      ! size of input array
  real r(nx,ny)                      ! input real array
  integer mx,my                      ! size of output array
  complex c(mx,my)                   ! output complex array
  !
  integer kx,ky,lx,ly
  integer i,j
  !
  kx = nx/2+1
  lx = mx/2+1
  ky = ny/2+1
  ly = my/2+1
  !
  do j=1,ny
    do i=1,nx
      r(i,j) = real(c(i-kx+lx,j-ky+ly)) 
    enddo
  enddo
end subroutine extract_real
!
subroutine spectrum_consistency(rname,ima,imb,tole,error)
  use image_def
  use gbl_message
  !---------------------------------------------------------------------
  ! @ private
  !
  ! IMAGER
  !   Support routine for command UV_SHORT
  !
  !   Verify spectral axis consistency
  !---------------------------------------------------------------------
  !
  character(len=*), intent(in) :: rname
  type(gildas), intent(in) :: ima
  type(gildas), intent(in) :: imb
  real, intent(in) :: tole
  logical, intent(out) :: error
  !
  integer :: na, nb, fa, fb
  real :: va, vb
  character(len=message_length) :: mess
  !
  error = .false.
  fa = ima%gil%faxi
  fb = imb%gil%faxi
  !
  ! Number of channels
  na = ima%gil%dim(fa)
  nb = ima%gil%dim(fb)
  !
  if (na.ne.nb) then
    write(mess,*) 'Mismatch in number of channels ',na,nb
    call map_message(seve%w,rname,mess)
    error = .true.
  endif
  !
  ! Do not check spectral axis if only 1 channel
  if (na.eq.1 .and. nb.eq.1) return
  !
  ! Check here the spectral axis mismatch
  if (abs(ima%gil%vres-imb%gil%vres).gt.abs(imb%gil%vres*tole)) then
    write(mess,*) 'Mismatch in spectral resolution ',ima%gil%vres,imb%gil%vres
    call map_message(seve%w,rname,mess)
    error = .true.
  endif
  if (abs(ima%gil%freq-imb%gil%freq).gt.abs(imb%gil%fres*tole)) then
    write(mess,*) 'Mismatch in frequency axis ',ima%gil%freq,imb%gil%freq
    call map_message(seve%w,rname,mess)
    error = .true.
  endif
  !
  ! Velocity should be checked too
  va = (1.d0-ima%gil%ref(fa))*ima%gil%vres + ima%gil%voff
  vb = (1.d0-imb%gil%ref(fb))*imb%gil%vres + imb%gil%voff
  if (abs(va-vb).gt.abs(imb%gil%vres*tole)) then
    write(mess,*) 'Mismatch in velocity axis ',va, vb 
    call map_message(seve%w,rname,mess)
    error = .true.
  endif
end subroutine spectrum_consistency
!
subroutine scale_factor(nx,ny,cbig,chigh,rmask,fscale,flux) 
  integer, intent(in) :: nx,ny
  complex, intent(in) :: cbig(nx,ny)
  complex, intent(in) :: chigh(nx,ny)
  real, intent(in) :: rmask(nx,ny)
  real, intent(out) :: fscale, flux
  !
  integer :: ix,iy
  real(kind=8) :: xx, xy, dx, dy
  !
  !  Linear fit of flux scale factor - Unweighted
  !
  ! A = Sum (X Y / sigma^2) / Sum (X^2/Sigma^2)
  ! A = Sum (X Y W) / Sum (X X W)
  !
  xx = 0
  xy = 0
  !
  do iy=1,ny
    do ix=1,nx
      if (rmask(ix,iy).ne.0.) then
        dx = real(cbig(ix,iy))**2 + imag(cbig(ix,iy))**2
        dy = real(chigh(ix,iy))*real(cbig(ix,iy)) + imag(chigh(ix,iy))*imag(cbig(ix,iy)) 
        xx = xx + dx
        xy = xy + dy
      endif
    enddo
  enddo
  if (xx.ne.0.) then
    fscale = xy/xx
    flux   = xx
  else
    fscale = 1.
    flux = 0.
  endif
end subroutine scale_factor
!
!
subroutine s_reproject_init(in,tem,out,sys_code,error)
  use image_def
  use gkernel_interfaces
  use gbl_message
  !
  ! Automatic resampling of LOWres data set
  !
  type(gildas), intent(in) :: in
  type(gildas), intent(in) :: tem
  type(gildas), intent(inout) :: out
  integer, intent(out) :: sys_code
  logical :: error
  !
  character(len=*), parameter :: pname='REPROJECT'
  !
  sys_code = 0
  call sanity_check(tem,error)
  if (error) return
  !
  ! Copy part of header of Template, but NOT the Resolution section !...
  out%char%syst = tem%char%syst
  out%gil%ra   = tem%gil%ra
  out%gil%dec  = tem%gil%dec
  out%gil%lii  = tem%gil%lii
  out%gil%bii  = tem%gil%bii
  out%gil%epoc = tem%gil%epoc
  out%gil%ptyp = tem%gil%ptyp
  out%gil%a0   = tem%gil%a0
  out%gil%d0   = tem%gil%d0
  out%gil%pang = tem%gil%pang
  out%char%code(out%gil%xaxi) = tem%char%code(tem%gil%xaxi)
  out%char%code(out%gil%yaxi) = tem%char%code(tem%gil%yaxi)
  out%gil%dim(out%gil%xaxi) = tem%gil%dim(tem%gil%xaxi)
  out%gil%dim(out%gil%yaxi) = tem%gil%dim(tem%gil%yaxi)
  out%loca%size = out%gil%dim(1)*out%gil%dim(2)*out%gil%dim(3)*out%gil%dim(4)
  out%gil%proj_words = tem%gil%proj_words
  out%gil%convert(:,out%gil%xaxi) = tem%gil%convert(:,tem%gil%xaxi)
  out%gil%convert(:,out%gil%yaxi) = tem%gil%convert(:,tem%gil%yaxi)
  !
  call get_sys_code(in,out,sys_code, error)
  if (error) return
  !
end subroutine s_reproject_init
!
subroutine s_reproject_do(in,out,sys_code,error)
  use image_def
  use gkernel_interfaces
  use gbl_message
  !
  ! Automatic resampling of LOWres data set
  !
  type(gildas), intent(in) :: in
  type(gildas), intent(inout) :: out
  integer, intent(in) :: sys_code
  logical :: error
  !
  character(len=*), parameter :: rname='REPROJECT'
  real :: in_blank(2)  ! Input blanking
  real(kind=8), allocatable :: work1(:,:),work2(:,:)
  integer :: ier
  !
  ! Set blanking
  in_blank = [in%gil%bval,in%gil%eval] 
  out%gil%blan_words = 2
  if (in%gil%blan_words.eq.0) then
    in_blank = [0.,-1.]
    !!Print *,'Using blanking ',out%gil%bval,out%gil%eval
  endif
  !
  ! Method of solution
  if (in_blank(2).lt.0.0) then
    call map_message(seve%i,rname, &
    & 'Using bilinear gridding method with no input blanking')
  else
    call map_message(seve%i,rname, &
    & 'Using bilinear gridding method with input blanking')
  endif
  !
  error = .false.
  if (.not.associated(in%r3d)) then
    call map_message(seve%e,rname,'Input cube is not associated')
    error = .true.
  endif
  if (.not.associated(out%r3d)) then
    call map_message(seve%e,rname,'Output cube is not associated')
    error = .true.
  endif
  if (error) return
  !  
  call gridlin(                                                        &
    in%r3d,in%gil%dim(1),in%gil%dim(2),in%gil%dim(3),                  &
    in%gil%convert,in%gil%ptyp,in%gil%a0,in%gil%epoc,in_blank,         &
    out%r3d,out%gil%dim(1),out%gil%dim(2),                             &
    out%gil%convert,out%gil%ptyp,out%gil%a0,out%gil%epoc,out%gil%bval, &
    sys_code,error)
  !
end subroutine s_reproject_do
!
subroutine sanity_check(x,error)
  use image_def
  use gbl_message
  !-------------------------------------------------------------------
  ! Sanity check for input or template file
  !-------------------------------------------------------------------
  type(gildas), intent(in)    :: x
  logical,      intent(inout) :: error
  !
  character(len=*), parameter :: rname='FEATHER'
  !
  ! Check if the 2 first dimensions are spatial. We could support any
  ! order but this is not implemented (and would be unefficient)
  if (x%gil%xaxi*x%gil%yaxi.ne.2) then
    call map_message(seve%e,rname,'Spatial dimensions must be first and second dimensions')
    error = .true.
    return
  endif
  !
end subroutine sanity_check
!
subroutine get_sys_code(in,out,sys_code,error)
  use gbl_format
  use image_def
  use gbl_message
  !---------------------------------------------------------------------
  ! Get old-system to new-system conversion code
  ! Trap illegal/impossible conversions.
  !---------------------------------------------------------------------
  type(gildas),    intent(in)  :: in        ! Input header
  type(gildas),    intent(in)  :: out       ! Desired output header
  integer(kind=4), intent(out) :: sys_code  ! Conversion code
  logical,         intent(out) :: error
  ! Local
  character(len=*), parameter :: pname='REPROJECT'
  integer(kind=4), parameter :: conv_none   = 0
  integer(kind=4), parameter :: conv_equ2gal= 1
  integer(kind=4), parameter :: conv_gal2equ=-1
  integer(kind=4), parameter :: conv_equ2equ=-2
  !
  error = .false.
  select case (out%char%syst)
  case ('GALACTIC')
    if (in%char%syst.eq.'EQUATORIAL') then
      sys_code = conv_equ2gal
    elseif (in%char%syst.eq.'GALACTIC') then
      sys_code = conv_none
    else
      error = .true.
    endif
    !
  case ('EQUATORIAL')
    if (in%char%syst.eq.'EQUATORIAL') then
      if (in%gil%epoc.ne.out%gil%epoc) then
        ! Both Equatorial, not same equinox
        call map_message(seve%i,pname, &
          & 'Converting from equinox '//trim(equinox_name(in%gil%epoc))//  &
          &                     ' to '//trim(equinox_name(out%gil%epoc)))
        sys_code = conv_equ2equ
      else
        ! Both Equatorial, same equinox
        sys_code = conv_none
      endif
    elseif (in%char%syst.eq.'GALACTIC') then
      sys_code = conv_gal2equ
    else
      error = .true.
    endif
    !
  case ('UNKNOWN')
    if (in%char%syst.eq.'GALACTIC') then
      error = .true.
    elseif (in%char%syst.eq.'EQUATORIAL') then
      error = .true.
    else
      sys_code = conv_none
    endif
    !
  case default
    call map_message(seve%w,pname,'Unknown system '//out%char%syst//', no conversion applied')
    sys_code = conv_none
    !
  end select
  !
  if (error) then
    call map_message(seve%e,pname,'Cannot convert from '//in%char%syst//' to '//out%char%syst)
  endif
  return
  !
contains
  function equinox_name(equinox)
    real(kind=4), intent(in) :: equinox
    character(len=10) :: equinox_name
    if (equinox.eq.equinox_null) then
      equinox_name = 'Unknown'
    else
      write(equinox_name,'(F0.2)') equinox
    endif
  end function equinox_name
  !
end subroutine get_sys_code
!
subroutine gridlin(a,mx,my,mz,aconv,atype,aproj,aepoc,ablank,  &
                   b,nx,ny,   bconv,btype,bproj,bepoc,bblank,  &
                   code,error)
  use gildas_def
  use gkernel_interfaces, no_interface1=>abs_to_rel_1dn4,  &
                          no_interface2=>rel_to_abs_1dn4
  use gkernel_types
  !$ use omp_lib
  !---------------------------------------------------------------------
  ! GREG Stand alone subroutine
  !  Regrid an input map in a given projection to an output map in
  ! another projection, bilinear interpolation
  !---------------------------------------------------------------------
  integer(kind=index_length), intent(in) :: mx  ! Size of A
  integer(kind=index_length), intent(in) :: my  ! Size of A
  integer(kind=index_length), intent(in) :: mz  ! Size of A
  real(kind=4)    :: a(mx,my,mz)  ! Input map of dimensions MX MY MZ
  real(kind=8)    :: aconv(6)     ! Pixel conversion formulae: CONV(1) = Xref, CONV(2)=Xval, CONV(3)=Xinc
  integer(kind=4) :: atype        ! Type of projection
  real(kind=8)    :: aproj(3)     ! Projection constants PROJ(1)=A0, PROJ(2)=D0, PROJ(3)=Angle
  real(kind=4)    :: aepoc        ! Epoch if Needed
  real(kind=4)    :: ablank(2)    !
  integer(kind=4) :: nx           !
  integer(kind=4) :: ny           !
  real(kind=4)    :: b(nx,ny,mz)  ! Output map of dimensions NX,NY,MZ
  real(kind=8)    :: bconv(6)     !
  integer(kind=4) :: btype        !
  real(kind=8)    :: bproj(3)     !
  real(kind=4)    :: bepoc        ! Epoch if Needed
  real(kind=4)    :: bblank       ! Blanking value
  integer(kind=4) :: code         ! Galactic to Equatorial (-1) or Equatorial to Galactic (1)
  logical, intent(out) :: error   !
  !
  integer(kind=4), parameter :: conv_none   = 0
  integer(kind=4), parameter :: conv_equ2gal= 1
  integer(kind=4), parameter :: conv_gal2equ=-1
  integer(kind=4), parameter :: conv_equ2equ=-2
  ! Local
  real(kind=4) :: bval,eval
  integer(kind=index_length) :: ia,ja,k
  integer(kind=4) :: ib,jb,ier
  real(kind=8) :: axref,axval,axinc, ayref,ayval,ayinc
  real(kind=8) :: bxref,bxval,bxinc, byref,byval,byinc
  real(kind=8) :: xa,ya
  real(kind=4) :: xr,yr
  ! Work arrays to handle the coordinates conversion
  real(kind=8), allocatable :: axb(:),ayb(:),bxb(:),byb(:)
  real(kind=8), allocatable :: xb(:,:), yb(:,:)
  type(projection_t) :: proj
  integer :: ithread,nthread
  !
  error = .false.
  !
  ! Preliminary processing
  bval=ablank(1)
  eval=ablank(2)
  !
  ! I force the conversion formula for input to be adapted to the
  ! NINT function that will be used below. Now Xref=1.0
  axref = 1.0d0                ! was ACONV(1)
  axval = (1.0d0-aconv(1))*aconv(3)+aconv(2)
  axinc = aconv(3)
  ayref = 1.0d0                ! was ACONV(4)
  ayval = (1.0d0-aconv(4))*aconv(6)+aconv(5)
  ayinc = aconv(6)
  ! the output is unchanged.
  bxref = bconv(1)
  bxval = bconv(2)
  bxinc = bconv(3)
  byref = bconv(4)
  byval = bconv(5)
  byinc = bconv(6)
  !
  ! Setup Output projection
  error = .false.
  call gwcs_projec(bproj(1),bproj(2),bproj(3),btype,proj,error)
  if (error) return
  !
  if (code.ne.conv_none) then
    allocate(xb(nx,ny),yb(nx,ny),axb(nx*ny),ayb(nx*ny),bxb(nx*ny),byb(nx*ny),stat=ier)
  else
    allocate(xb(nx,ny),yb(nx,ny),axb(nx*ny),ayb(nx*ny),stat=ier)
  endif
  if (ier.ne.0) then
    error = .true.
    return
  endif    
  ! Convert Output projection to Absolute coordinates
  do jb=1,ny
    do ib=1,nx
      xb(ib,jb) = (ib-bxref)*bxinc + bxval
      yb(ib,jb) = (jb-byref)*byinc + byval
    enddo
  enddo
  !
  if (code.ne.conv_none) then
    call rel_to_abs_1dn4(proj,xb,yb,bxb,byb,nx*ny)
    !
    ! Change coordinate system. use new epoch also. Warning: codes are
    ! reversed since we start form output map!
    if (code.eq.1) then
      call gal_equ(bxb,byb,axb,ayb,aepoc,nx*ny,error)
    elseif (code.eq.-1) then
      call equ_gal(bxb,byb,bepoc,axb,ayb,nx*ny,error)
    elseif (code.eq.-2) then
      call equ_equ(bxb,byb,bepoc,axb,ayb,aepoc,nx*ny,error)
    endif
    if (error) return
  else
    call rel_to_abs_1dn4(proj,xb,yb,axb,ayb,nx*ny)
  endif
  !
  ! Setup Input projection
  call gwcs_projec(aproj(1),aproj(2),aproj(3),atype,proj,error)
  if (error) return
  !
  ! Convert Absolute coordinates to input projection
  call abs_to_rel_1dn4(proj,axb,ayb,xb,yb,nx*ny)
  !
  ! Interpolate
  !
  ! Take Blanking into account, version #1
  nthread = 0
  !$ nthread = omp_get_max_threads()
  if (eval.lt.0.0) then
    ! Loop over Output data points
    !$OMP PARALLEL DEFAULT(NONE) &
    !$OMP &  SHARED(xb,yb,b,a, mz,mx,my,nx,ny) &
    !$OMP &  SHARED(axinc, axval, axref, ayinc, ayval, ayref,bblank) &
    !$OMP &  PRIVATE(k,ib,jb,xa,ya,xr,yr,ia,ja, nthread, ithread)
    !$OMP DO COLLAPSE (2)
    do k = 1,mz
      do jb = 1,ny
        do ib = 1,nx
          !
          ! Find pixel coordinate in input map
          xa = xb(ib,jb)
          ya = yb(ib,jb)
          xr = (xa-axval)/axinc+axref
          yr = (ya-ayval)/ayinc+ayref
          ia = int(xr)
          ja = int(yr)
          !
          ! Avoid edges
          if (ia.lt.1 .or. ia.ge.mx .or. ja.lt.1 .or. ja.ge.my) then
            b(ib,jb,k) = bblank
          else
            ! Interpolate
            xr = xr-float(ia)
            yr = yr-float(ja)
            b(ib,jb,k) = (1-xr)*(1-yr)*a(ia,ja,k)+   &
                         xr*(1-yr)*a(ia+1,ja,k)+   &
                         xr*yr*a(ia+1,ja+1,k)+   &
                         (1-xr)*yr*a(ia,ja+1,k)
          endif
        enddo
      enddo
    enddo
    !$OMP END DO
    !$OMP END PARALLEL
  else
    ! Loop over Output data points
    !$OMP PARALLEL DEFAULT(NONE) &
    !$OMP &  SHARED(xb,yb,b,a, mz,mx,my,nx,ny) &
    !$OMP &  SHARED(axinc, axval, axref, ayinc, ayval, ayref, bblank, bval, eval) &
    !$OMP &  PRIVATE(k,ib,jb,xa,ya,xr,yr,ia,ja, nthread, ithread)
    !$OMP DO COLLAPSE (2)
    do k = 1,mz
      do jb = 1,ny
        do ib = 1,nx
          xa = xb(ib,jb)
          ya = yb(ib,jb)
          xr = (xa-axval)/axinc+axref
          yr = (ya-ayval)/ayinc+ayref
          ia = int(xr)
          ja = int(yr)
          if (ia.lt.1 .or. ia.ge.mx .or. ja.lt.1 .or. ja.ge.my) then
            b(ib,jb,k) = bblank
          elseif (abs(a(ia,ja,k)-bval).le.eval) then
            b(ib,jb,k) = bblank
          elseif (abs(a(ia+1,ja,k)-bval).le.eval) then
            b(ib,jb,k) = bblank
          elseif (abs(a(ia,ja+1,k)-bval).le.eval) then
            b(ib,jb,k) = bblank
          elseif (abs(a(ia+1,ja+1,k)-bval).le.eval) then
            b(ib,jb,k) = bblank
          else
            xr = xr-float(ia)
            yr = yr-float(ja)
            b(ib,jb,k) = (1-xr)*(1-yr)*a(ia,ja,k)+  &
                         xr*(1-yr)*a(ia+1,ja,k)+   &
                         xr*yr*a(ia+1,ja+1,k)+   &
                         (1-xr)*yr*a(ia,ja+1,k)
          endif
        enddo
      enddo
    enddo
    !$OMP END DO
    !$OMP END PARALLEL
  endif
  !
end subroutine gridlin

