!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
!
module cubetools_unit_magnitude
  public :: unit_magnitude_t
  public :: cubetools_unit_magnitude_parse
  public :: exa,peta,tera,giga,mega,kilo,unity,centi,milli,micro,nano,pico,femto
  private
  !
  type :: unit_magnitude_t
    character(len=1), public  :: prefix     ! Standard prefix
    character(len=1), private :: alternate  ! Case insensitive alternate prefix
    real(kind=8),     public  :: factor     ! Multiplicative factor
  end type unit_magnitude_t
  !
  ! NB: in case of case-sensitive prefix, the alternate prefix is set identical
  !     to the standard prefix for simplicity of declaration and parsing.
  type(unit_magnitude_t), parameter :: exa   = unit_magnitude_t( 'E','e',1d18  )  ! Case-insensitive
  type(unit_magnitude_t), parameter :: peta  = unit_magnitude_t( 'P','P',1d15  )  ! Case-sensitive (conflict)
  type(unit_magnitude_t), parameter :: tera  = unit_magnitude_t( 'T','t',1d12  )  ! Case-insensitive
  type(unit_magnitude_t), parameter :: giga  = unit_magnitude_t( 'G','g',1d9   )  ! Case-insensitive
  type(unit_magnitude_t), parameter :: mega  = unit_magnitude_t( 'M','M',1d6   )  ! Case-sensitive (conflict)
  type(unit_magnitude_t), parameter :: kilo  = unit_magnitude_t( 'k','K',1d3   )  ! Case-insensitive
  type(unit_magnitude_t), parameter :: unity = unit_magnitude_t( ' ',' ',1d0   )  !
  type(unit_magnitude_t), parameter :: centi = unit_magnitude_t( 'c','C',1d-2  )  ! Case-insensitive
  type(unit_magnitude_t), parameter :: milli = unit_magnitude_t( 'm','m',1d-3  )  ! Case-sensitive (conflict)
  type(unit_magnitude_t), parameter :: micro = unit_magnitude_t( 'u','U',1d-6  )  ! Case-insensitive
  type(unit_magnitude_t), parameter :: nano  = unit_magnitude_t( 'n','N',1d-9  )  ! Case-insensitive
  type(unit_magnitude_t), parameter :: pico  = unit_magnitude_t( 'p','p',1d-12 )  ! Case-sensitive (conflict)
  type(unit_magnitude_t), parameter :: femto = unit_magnitude_t( 'f','F',1d-15 )  ! Case-insensitive
  !
  integer(kind=4), parameter :: nmagnitude=13
  type(unit_magnitude_t), parameter :: magnitudes(nmagnitude) =  &
    [ exa,peta,tera,giga,mega,kilo,unity,centi,milli,micro,nano,pico,femto ]
  !
contains
  !
  function cubetools_unit_magnitude_parse(prefix,magnitude)
    !-------------------------------------------------------------------
    ! Check if prefix is a known magnitude letter, and return this
    ! magnitude. Function evaluates as .true. or .false. if a matching
    ! magnitude is found.
    ! Case-sensitive search. Note that blank (or null-size) string
    ! matches unity.
    !-------------------------------------------------------------------
    logical :: cubetools_unit_magnitude_parse
    character(len=*),       intent(in)  :: prefix
    type(unit_magnitude_t), intent(out) :: magnitude
    !
    integer(kind=4) :: imagnitude
    !
    cubetools_unit_magnitude_parse = .false.
    do imagnitude=1,nmagnitude
      if (prefix.eq.magnitudes(imagnitude)%prefix .or. &
          prefix.eq.magnitudes(imagnitude)%alternate) then
        magnitude = magnitudes(imagnitude)
        cubetools_unit_magnitude_parse = .true.
        return
      endif
    enddo
  end function cubetools_unit_magnitude_parse
end module cubetools_unit_magnitude
!
module cubetools_unit
  use phys_const
  use cubetools_parameters
  use cubetools_messaging
  use cubetools_unit_magnitude
  !
  public :: unit_user_t,unitkinds,nunitkinds
  public :: code_unit_auto
  public :: unit_uv,code_unit_uv
  public :: unit_fov,code_unit_fov
  public :: unit_pang,code_unit_pang
  public :: unit_beam,code_unit_beam
  public :: unit_freq,code_unit_freq
  public :: unit_velo,code_unit_velo
  public :: unit_wave,code_unit_wave
  public :: unit_chan,code_unit_chan
  public :: unit_pixe,code_unit_pixe
  public :: unit_dist,code_unit_dist
  public :: unit_unk,code_unit_unk
  public :: code_unit_brig,code_unit_flux
  public :: cubetools_unit_print,cubetools_unit_init,cubetools_unit_defaults
  public :: cubetools_unit_set
  public :: cubetools_unit_get_kind,cubetools_unit_get_list
  public :: operator(.eq.)
  !
  private
  !
  ! *** JP: Should unit_l also be defined here?
  integer(kind=4), parameter :: unit_k = 4
  integer(kind=4), parameter :: keys_l = 10*unit_l  ! Store several variants of the same unit
  !
  ! Description of a unit default
  type :: unit_prog_t
    integer(kind=code_k),   private :: dimension  ! Backpointer to the list of associated units (e.g. to Hz)
    type(unit_magnitude_t), private :: magnitude  ! Possible extra factor (e.g. mega)
  end type unit_prog_t
  ! Description of variants for a given unit
  type :: unit_variant_t
    integer(kind=unit_k),  private              :: n        ! Number of keys
    character(len=unit_l), private, allocatable :: keys(:)  ! List of keys
  end type unit_variant_t
  ! Description of a unit kind
  type :: unit_kind_t
    integer(kind=code_k),  public               :: id       ! Unit kind identifier
    integer(kind=unit_k),  private              :: n        ! Number of different units supported
    character(len=unit_l), private, allocatable :: name(:)  ! Unit name for output
    character(len=keys_l), private, allocatable :: keys(:)  ! Blank-separated list of allowed input name for each unit
    type(unit_variant_t),  private, allocatable :: vars(:)  ! Keys processed to list of variants
    real(kind=coor_k),     private, allocatable :: conv(:)  ! Relative conversion factor between supported units
    type(unit_prog_t),     private              :: prog     ! Program internal unit description
    type(unit_prog_t),     private              :: user     ! User default unit
  contains
    procedure,             private :: alloc     => cubetools_unit_kind_alloc
    procedure,             private :: init      => cubetools_unit_kind_init
    procedure,             private :: debug     => cubetools_unit_kind_debug
    procedure,             public  :: prog_name => cubetools_unit_kind_prog_name
  end type unit_kind_t
  !
  interface operator(.eq.)  ! Offers syntax e.g. "myunit.eq.unit_unk" to programmers
    module procedure cubetools_unit_kind_eq
  end interface
  !
  integer(kind=unit_k),  parameter :: nunitkinds = 11
  !
  ! Brightness and Flux units codes are included here as negative
  ! indexes because they are treated in a separate module
  ! (cubetools_brightness). Hence All loops from 1 to nunitkinds will
  ! treat only statical units, i.e. units whose convertions do not
  ! depend on cube information.
  !
  character(len=unit_l), parameter :: unitkinds(-1:nunitkinds) = [&
       'FLUX      ','BRIGHTNESS','UV        ','FOV       ','POSANGLE  ',&
       'BEAM      ','FREQUENCY ','VELOCITY  ','WAVELENGTH','CHANNEL   ',&
       'PIXEL     ','DISTANCE  ','UNKNOWN   ']
  !
  integer(kind=code_k), parameter :: code_unit_auto = -1000  ! Code for automatic kind recognition
  integer(kind=code_k), parameter :: code_unit_flux = -1
  integer(kind=code_k), parameter :: code_unit_brig = 0
  integer(kind=code_k), parameter :: code_unit_uv=1
  integer(kind=code_k), parameter :: code_unit_fov=2
  integer(kind=code_k), parameter :: code_unit_pang=3
  integer(kind=code_k), parameter :: code_unit_beam=4
  integer(kind=code_k), parameter :: code_unit_freq=5
  integer(kind=code_k), parameter :: code_unit_velo=6
  integer(kind=code_k), parameter :: code_unit_wave=7
  integer(kind=code_k), parameter :: code_unit_chan=8
  integer(kind=code_k), parameter :: code_unit_pixe=9
  integer(kind=code_k), parameter :: code_unit_dist=10
  integer(kind=code_k), parameter :: code_unit_unk=11
  !
  type(unit_kind_t), target  :: allunits(nunitkinds)
  type(unit_kind_t), pointer :: unit_uv
  type(unit_kind_t), pointer :: unit_fov
  type(unit_kind_t), pointer :: unit_pang
  type(unit_kind_t), pointer :: unit_beam
  type(unit_kind_t), pointer :: unit_freq
  type(unit_kind_t), pointer :: unit_velo
  type(unit_kind_t), pointer :: unit_wave
  type(unit_kind_t), pointer :: unit_chan
  type(unit_kind_t), pointer :: unit_pixe
  type(unit_kind_t), pointer :: unit_dist
  type(unit_kind_t), pointer :: unit_unk
  !
  type unit_user_t
    type(unit_prog_t), private :: user                 ! Unit and magnitude description
    character(len=unit_l)      :: name = strg_unk      ! Unit name
    real(kind=coor_k)          :: user_per_prog = 1d0  ! Conversion factor
    real(kind=coor_k)          :: prog_per_user = 1d0  ! Reverse conversion factor
    type(unit_kind_t), pointer :: kind => null()       ! Associated program unit
  contains
    procedure, public  :: list                  => cubetools_unit_list
    procedure, public  :: get                   => cubetools_unit_get
    procedure, public  :: get_from_name         => cubetools_unit_get_from_name
    procedure, public  :: get_from_kind         => cubetools_unit_get_from_kind
    procedure, private :: get_from_name_or_kind => cubetools_unit_get_from_name_or_kind
    procedure, public  :: is_consistent_with    => cubetools_unit_is_consistent_with
  end type unit_user_t
  !
  type unit_setup_t
    type(unit_prog_t) :: program(nunitkinds)  ! Internal program unit
    type(unit_prog_t) :: default(nunitkinds)  ! User default unit
    type(unit_prog_t) :: current(nunitkinds)  ! User current unit
    character(len=unit_l) :: prognames(nunitkinds)  ! Internal program unit names
    character(len=unit_l) :: usernames(nunitkinds)  ! User current unit names
    real(kind=coor_k) :: prog_per_user(nunitkinds)  ! User current conversion (ppu)
    real(kind=coor_k) :: user_per_prog(nunitkinds)  ! User current conversion (upp)
  end type unit_setup_t
  !
  type(unit_setup_t) :: prog
  !
contains
  !
  subroutine cubetools_unit_init(error)
    !---------------------------------------------------------------------
    ! Re(define) all individual units and user current units
    !---------------------------------------------------------------------
    logical, intent(inout) :: error
    !
    unit_uv => allunits(code_unit_uv)
    unit_uv%id   = code_unit_uv
    call unit_uv%alloc(1,error)
    if (error)  return
    unit_uv%name(:) = ['meter']
    unit_uv%keys(:) = ['METER']
    unit_uv%conv(:) = [ 1.d0  ]
    unit_uv%prog%dimension = 1
    unit_uv%prog%magnitude = unity
    unit_uv%user%dimension = 1
    unit_uv%user%magnitude = unity
    call unit_uv%init(error)
    !
    unit_fov => allunits(code_unit_fov)
    unit_fov%id   = code_unit_fov
    call unit_fov%alloc(4,error)
    if (error)  return
    unit_fov%name(:) = ['radian',    'degree',    'arcmin',    'arcsec'    ]
    unit_fov%keys(:) = ['RADIAN;RAD','DEGREE;DEG','ARCMIN    ','ARCSEC    ']
    unit_fov%conv(:) = [ 1d0,        rad_per_deg, rad_per_min, rad_per_sec ]
    unit_fov%prog%dimension = 1
    unit_fov%prog%magnitude = unity
    unit_fov%user%dimension = 4
    unit_fov%user%magnitude = unity
    call unit_fov%init(error)
    !
    unit_pang => allunits(code_unit_pang)
    unit_pang%id   = code_unit_pang
    call unit_pang%alloc(2,error)
    if (error)  return
    unit_pang%name(:) = ['radian',   'degree']
    unit_pang%keys(:) = ['RADIAN',   'DEGREE']
    unit_pang%conv(:) = [     1d0,rad_per_deg]
    unit_pang%prog%dimension = 1
    unit_pang%prog%magnitude = unity
    unit_pang%user%dimension = 2
    unit_pang%user%magnitude = unity
    call unit_pang%init(error)
    !
    unit_beam => allunits(code_unit_beam)
    unit_beam%id   = code_unit_beam
    call unit_beam%alloc(4,error)
    if (error)  return
    unit_beam%name(:) = ['radian',    'degree',    'arcmin',    'arcsec'    ]
    unit_beam%keys(:) = ['RADIAN;RAD','DEGREE;DEG','ARCMIN    ','ARCSEC    ']
    unit_beam%conv(:) = [ 1d0,        rad_per_deg, rad_per_min, rad_per_sec ]
    unit_beam%prog%dimension = 1
    unit_beam%prog%magnitude = unity
    unit_beam%user%dimension = 4
    unit_beam%user%magnitude = unity
    call unit_beam%init(error)
    !
    unit_freq => allunits(code_unit_freq)
    unit_freq%id   = code_unit_freq
    call unit_freq%alloc(1,error)
    if (error)  return
    unit_freq%name(:) = ['Hz']
    unit_freq%keys(:) = ['HZ']
    unit_freq%conv(:) = [ 1d0]
    unit_freq%prog%dimension = 1
    unit_freq%prog%magnitude = mega
    unit_freq%user%dimension = 1
    unit_freq%user%magnitude = mega
    call unit_freq%init(error)
    !
    unit_velo => allunits(code_unit_velo)
    unit_velo%id   = code_unit_velo
    call unit_velo%alloc(1,error)
    if (error)  return
    unit_velo%name(:) = ['m/s']
    unit_velo%keys(:) = ['M/S']
    unit_velo%conv(:) = [  1d0]
    unit_velo%prog%dimension = 1
    unit_velo%prog%magnitude = kilo
    unit_velo%user%dimension = 1
    unit_velo%user%magnitude = kilo
    call unit_velo%init(error)
    !
    unit_wave => allunits(code_unit_wave)
    unit_wave%id   = code_unit_wave
    call unit_wave%alloc(2,error)
    if (error)  return
    unit_wave%name(:) = ['m  ','Ang']
    unit_wave%keys(:) = ['M  ','ANG']
    unit_wave%conv(:) = [  1d0,1d-10]
    unit_wave%prog%dimension = 1
    unit_wave%prog%magnitude = micro
    unit_wave%user%dimension = 1
    unit_wave%user%magnitude = milli
    call unit_wave%init(error)
    !
    unit_chan => allunits(code_unit_chan)
    unit_chan%id   = code_unit_chan
    call unit_chan%alloc(1,error)
    if (error)  return
    unit_chan%name(:) = ['channel']
    unit_chan%keys(:) = ['CHANNEL']
    unit_chan%conv(:) = [      1d0]
    unit_chan%prog%dimension = 1
    unit_chan%prog%magnitude = unity
    unit_chan%user%dimension = 1
    unit_chan%user%magnitude = unity
    call unit_chan%init(error)
    !
    unit_pixe => allunits(code_unit_pixe)
    unit_pixe%id   = code_unit_pixe
    call unit_pixe%alloc(1,error)
    if (error)  return
    unit_pixe%name(:) = ['pixel']
    unit_pixe%keys(:) = ['PIXEL']
    unit_pixe%conv(:) = [    1d0]
    unit_pixe%prog%dimension = 1
    unit_pixe%prog%magnitude = unity
    unit_pixe%user%dimension = 1
    unit_pixe%user%magnitude = unity
    call unit_pixe%init(error)
    !
    unit_dist => allunits(code_unit_dist)
    unit_dist%id   = code_unit_dist
    call unit_dist%alloc(1,error)
    if (error)  return
    unit_dist%name(:) = ['pc']
    unit_dist%keys(:) = ['PC']
    unit_dist%conv(:) = [ 1d0]
    unit_dist%prog%dimension = 1
    unit_dist%prog%magnitude = unity
    unit_dist%user%dimension = 1
    unit_dist%user%magnitude = unity
    call unit_dist%init(error)
    !
    unit_unk => allunits(code_unit_unk)
    unit_unk%id   = code_unit_unk
    call unit_unk%alloc(1,error)
    if (error)  return
    unit_unk%name(:) = [ strg_unk]
    unit_unk%keys(:) = ['UNKNOWN']
    unit_unk%conv(:) = [      1d0]
    unit_unk%prog%dimension = 1
    unit_unk%prog%magnitude = unity
    unit_unk%user%dimension = 1
    unit_unk%user%magnitude = unity
    call unit_unk%init(error)
    !
    call cubetools_unit_defaults(error)
    if (error)  return
    !
    ! call cubetools_unit_kind_print()  ! Debug
  end subroutine cubetools_unit_init
  !
  subroutine cubetools_unit_defaults(error)
    !---------------------------------------------------------------------
    ! Reset prog and user units to defaults
    !---------------------------------------------------------------------
    logical, intent(inout) :: error
    !
    character(len=*), parameter :: rname='UNIT>DEFAULTS'
    !
    call cubetools_message(toolseve%trace,rname,'Welcome')
    !
    prog%program(unit_uv%id)   = unit_uv%prog
    prog%program(unit_fov%id)  = unit_fov%prog
    prog%program(unit_pang%id) = unit_pang%prog
    prog%program(unit_beam%id) = unit_beam%prog
    prog%program(unit_freq%id) = unit_freq%prog
    prog%program(unit_velo%id) = unit_velo%prog
    prog%program(unit_wave%id) = unit_wave%prog
    prog%program(unit_chan%id) = unit_chan%prog
    prog%program(unit_pixe%id) = unit_pixe%prog
    prog%program(unit_dist%id) = unit_dist%prog
    prog%program(unit_unk%id)  = unit_unk%prog
    !
    prog%default(unit_uv%id)   = unit_uv%user
    prog%default(unit_fov%id)  = unit_fov%user
    prog%default(unit_pang%id) = unit_pang%user
    prog%default(unit_beam%id) = unit_beam%user
    prog%default(unit_freq%id) = unit_freq%user
    prog%default(unit_velo%id) = unit_velo%user
    prog%default(unit_wave%id) = unit_wave%user
    prog%default(unit_chan%id) = unit_chan%user
    prog%default(unit_pixe%id) = unit_pixe%user
    prog%default(unit_dist%id) = unit_dist%user
    prog%default(unit_unk%id)  = unit_unk%user
    !
    prog%current(unit_uv%id)   = unit_uv%user
    prog%current(unit_fov%id)  = unit_fov%user
    prog%current(unit_pang%id) = unit_pang%user
    prog%current(unit_beam%id) = unit_beam%user
    prog%current(unit_freq%id) = unit_freq%user
    prog%current(unit_velo%id) = unit_velo%user
    prog%current(unit_wave%id) = unit_wave%user
    prog%current(unit_chan%id) = unit_chan%user
    prog%current(unit_pixe%id) = unit_pixe%user
    prog%current(unit_dist%id) = unit_dist%user
    prog%current(unit_unk%id)  = unit_unk%user
    !
    prog%prognames(unit_uv%id)   = cubetools_unit_name(unit_uv,unit_uv%prog)
    prog%prognames(unit_fov%id)  = cubetools_unit_name(unit_fov,unit_fov%prog)
    prog%prognames(unit_pang%id) = cubetools_unit_name(unit_pang,unit_pang%prog)
    prog%prognames(unit_beam%id) = cubetools_unit_name(unit_beam,unit_beam%prog)
    prog%prognames(unit_freq%id) = cubetools_unit_name(unit_freq,unit_freq%prog)
    prog%prognames(unit_velo%id) = cubetools_unit_name(unit_velo,unit_velo%prog)
    prog%prognames(unit_wave%id) = cubetools_unit_name(unit_wave,unit_wave%prog)
    prog%prognames(unit_chan%id) = cubetools_unit_name(unit_chan,unit_chan%prog)
    prog%prognames(unit_pixe%id) = cubetools_unit_name(unit_pixe,unit_pixe%prog)
    prog%prognames(unit_dist%id) = cubetools_unit_name(unit_dist,unit_dist%prog)
    prog%prognames(unit_unk%id)  = cubetools_unit_name(unit_unk,unit_unk%prog)
    !
    prog%usernames(unit_uv%id)   = cubetools_unit_name(unit_uv,unit_uv%user)
    prog%usernames(unit_fov%id)  = cubetools_unit_name(unit_fov,unit_fov%user)
    prog%usernames(unit_pang%id) = cubetools_unit_name(unit_pang,unit_pang%user)
    prog%usernames(unit_beam%id) = cubetools_unit_name(unit_beam,unit_beam%user)
    prog%usernames(unit_freq%id) = cubetools_unit_name(unit_freq,unit_freq%user)
    prog%usernames(unit_velo%id) = cubetools_unit_name(unit_velo,unit_velo%user)
    prog%usernames(unit_wave%id) = cubetools_unit_name(unit_wave,unit_wave%user)
    prog%usernames(unit_chan%id) = cubetools_unit_name(unit_chan,unit_chan%user)
    prog%usernames(unit_pixe%id) = cubetools_unit_name(unit_pixe,unit_pixe%user)
    prog%usernames(unit_dist%id) = cubetools_unit_name(unit_dist,unit_dist%user)
    prog%usernames(unit_unk%id)  = cubetools_unit_name(unit_unk,unit_unk%user)
    !
    prog%prog_per_user(unit_uv%id)   = cubetools_unit_pperu(unit_uv,unit_uv%user)
    prog%prog_per_user(unit_fov%id)  = cubetools_unit_pperu(unit_fov,unit_fov%user)
    prog%prog_per_user(unit_pang%id) = cubetools_unit_pperu(unit_pang,unit_pang%user)
    prog%prog_per_user(unit_beam%id) = cubetools_unit_pperu(unit_beam,unit_beam%user)
    prog%prog_per_user(unit_freq%id) = cubetools_unit_pperu(unit_freq,unit_freq%user)
    prog%prog_per_user(unit_velo%id) = cubetools_unit_pperu(unit_velo,unit_velo%user)
    prog%prog_per_user(unit_wave%id) = cubetools_unit_pperu(unit_wave,unit_wave%user)
    prog%prog_per_user(unit_chan%id) = cubetools_unit_pperu(unit_chan,unit_chan%user)
    prog%prog_per_user(unit_pixe%id) = cubetools_unit_pperu(unit_pixe,unit_pixe%user)
    prog%prog_per_user(unit_dist%id) = cubetools_unit_pperu(unit_dist,unit_dist%user)
    prog%prog_per_user(unit_unk%id)  = cubetools_unit_pperu(unit_unk,unit_unk%user)
    !
    prog%user_per_prog(unit_uv%id)   = 1d0/prog%prog_per_user(unit_uv%id)
    prog%user_per_prog(unit_fov%id)  = 1d0/prog%prog_per_user(unit_fov%id)
    prog%user_per_prog(unit_pang%id) = 1d0/prog%prog_per_user(unit_pang%id)
    prog%user_per_prog(unit_beam%id) = 1d0/prog%prog_per_user(unit_beam%id)
    prog%user_per_prog(unit_freq%id) = 1d0/prog%prog_per_user(unit_freq%id)
    prog%user_per_prog(unit_velo%id) = 1d0/prog%prog_per_user(unit_velo%id)
    prog%user_per_prog(unit_wave%id) = 1d0/prog%prog_per_user(unit_wave%id)
    prog%user_per_prog(unit_chan%id) = 1d0/prog%prog_per_user(unit_chan%id)
    prog%user_per_prog(unit_pixe%id) = 1d0/prog%prog_per_user(unit_pixe%id)
    prog%user_per_prog(unit_dist%id) = 1d0/prog%prog_per_user(unit_dist%id)
    prog%user_per_prog(unit_unk%id)  = 1d0/prog%prog_per_user(unit_unk%id)
  end subroutine cubetools_unit_defaults
  !
  subroutine cubetools_unit_kind_alloc(unit,n,error)
    !---------------------------------------------------------------------
    ! Allocate to requested size
    !---------------------------------------------------------------------
    class(unit_kind_t),   intent(inout) :: unit
    integer(kind=unit_k), intent(in)    :: n
    logical,              intent(inout) :: error
    !
    ! No test for reallocation as this type is expected to be allocated
    ! only once
    allocate(unit%name(n),unit%keys(n),unit%vars(n),unit%conv(n))
    unit%n = n
  end subroutine cubetools_unit_kind_alloc
  !
  subroutine cubetools_unit_kind_init(unit,error)
    use cubetools_string
    !---------------------------------------------------------------------
    ! Called once at startup
    !---------------------------------------------------------------------
    class(unit_kind_t), intent(inout) :: unit
    logical,            intent(inout) :: error
    !
    integer(kind=unit_k) :: ikey
    !
    do ikey=1,unit%n
      call cubetools_string_split(unit%keys(ikey),';',unit%vars(ikey)%keys,error)
      if (error)  return
      unit%vars(ikey)%n = size(unit%vars(ikey)%keys)
    enddo
  end subroutine cubetools_unit_kind_init
  !
  subroutine cubetools_unit_kind_debug(unit)
    !---------------------------------------------------------------------
    ! For debugging purpose
    !---------------------------------------------------------------------
    class(unit_kind_t), intent(in) :: unit
    !
    integer(kind=unit_k) :: ikey,ivar
    !
    write(*,'(t3,a,t10,i0)')  'Id:',unit%id
    write(*,'(t3,a,t10,i0)')  'N:', unit%n
    do ikey=1,unit%n
      write(*,'(t3,i0,a1,t10,a)')         ikey,':',unit%name(ikey)
      write(*,'(         t10,1pg23.16)')           unit%conv(ikey)
      write(*,'(         t10,a)')                  unit%keys(ikey)
      do ivar=1,unit%vars(ikey)%n
        write(*,'(t13,i0,a1,a)')          ivar,':',unit%vars(ikey)%keys(ivar)
      enddo
    enddo
    write(*,*)
  end subroutine cubetools_unit_kind_debug
  !
  function cubetools_unit_kind_eq(unit1,unit2)
    logical :: cubetools_unit_kind_eq
    type(unit_kind_t), intent(in) :: unit1,unit2
    cubetools_unit_kind_eq = unit1%id.eq.unit2%id
  end function cubetools_unit_kind_eq
  !
  subroutine cubetools_unit_kind_print()
    !---------------------------------------------------------------------
    ! For debugging purpose
    !---------------------------------------------------------------------
    call unit_uv%debug()
    call unit_fov%debug()
    call unit_pang%debug()
    call unit_beam%debug()
    call unit_freq%debug()
    call unit_velo%debug()
    call unit_wave%debug()
    call unit_chan%debug()
    call unit_pixe%debug()
    call unit_dist%debug()
    call unit_unk%debug()
  end subroutine cubetools_unit_kind_print
  !
  subroutine cubetools_unit_print(error)
    !---------------------------------------------------------------------
    ! For user feedback
    !---------------------------------------------------------------------
    logical, intent(inout) :: error
    !
    character(len=message_length) :: mess
    integer(kind=unit_k) :: iunit
    character(len=*), parameter :: rname='UNIT>PRINT'
    !
    call cubetools_message(toolseve%trace,rname,'Welcome')
    !
    write(mess,'(3x,a,10x,a,9x,a,5x,a)') 'KIND','USER','INTERNAL','CONVERSION'
    call cubetools_message(seve%r,rname,mess)
    call cubetools_message(seve%r,rname,'')
    do iunit=1,nunitkinds
       write(mess,'(3x,a,2x,2(a,1x),1pg23.16)') unitkinds(iunit),prog%usernames(iunit),&
            prog%prognames(iunit),prog%prog_per_user(iunit)
       call cubetools_message(seve%r,rname,mess)
    enddo ! iunit
  end subroutine cubetools_unit_print
  !
  function cubetools_unit_name(unit,user)
    !-------------------------------------------------------------------
    ! Build the unit name given its 'user' description (which base unit
    ! and which magnitude)
    !-------------------------------------------------------------------
    character(len=unit_l) :: cubetools_unit_name
    type(unit_kind_t), intent(in) :: unit
    type(unit_prog_t), intent(in) :: user
    cubetools_unit_name = trim(user%magnitude%prefix)//unit%name(user%dimension)
  end function cubetools_unit_name
  !
  function cubetools_unit_pperu(unit,user)
    !-------------------------------------------------------------------
    ! Build the conversion factor from prog to user, given the 'user'
    ! descriptions (which base unit and which magnitude in
    ! both) cases.
    !-------------------------------------------------------------------
    real(kind=coor_k) :: cubetools_unit_pperu
    type(unit_kind_t), intent(in) :: unit
    type(unit_prog_t), intent(in) :: user
    !
    cubetools_unit_pperu = (user%magnitude%factor/unit%prog%magnitude%factor) *  &
                           (unit%conv(user%dimension)/unit%conv(unit%prog%dimension))
  end function cubetools_unit_pperu
  !
  subroutine cubetools_unit_set(name,kind,error)
    use gkernel_interfaces
    !---------------------------------------------------------------------
    !
    !---------------------------------------------------------------------
    character(len=*),     intent(in)    :: name
    integer(kind=code_k), intent(in)    :: kind
    logical,              intent(inout) :: error
    !
    type(unit_user_t) :: user
    character(len=*), parameter :: rname='UNIT>SET'
    !
    call cubetools_message(toolseve%trace,rname,'Welcome')
    !
    call user%get_from_name_or_kind(name,kind,error)
    if (error) return
    !
    prog%current(kind)       = user%user
    prog%usernames(kind)     = user%name
    prog%prog_per_user(kind) = user%prog_per_user
    prog%user_per_prog(kind) = user%user_per_prog
  end subroutine cubetools_unit_set
  !
  !-----------------------------------------------------------------------
  !
  subroutine cubetools_unit_list(user,error)
    !---------------------------------------------------------------------
    ! 
    !---------------------------------------------------------------------
    class(unit_user_t), intent(in)    :: user
    logical,            intent(inout) :: error
    !
    character(len=message_length) :: mess
    character(len=*), parameter :: rname='UNIT>LIST'
    !
    call cubetools_message(toolseve%trace,rname,'Welcome')
    write(mess,'(3x,a,2x,a,2x,2(1pg23.16,1x))') unitkinds(user%kind%id),user%name,user%user_per_prog,user%prog_per_user
    call cubetools_message(seve%r,rname,mess)
  end subroutine cubetools_unit_list
  !
  subroutine cubetools_unit_get(user,name,kind,error)
    !---------------------------------------------------------------------
    ! Fill the unit_user_t from input name & kind. 3 possibilities:
    ! 1) if kind is auto code, try automatic kind recognition through name
    !    parsing,
    ! 2) if name is * or =, return with user provided defaut,
    ! 3) if name is an explicit unit, parse this unit and fill the user
    !    structure with appropriate conversions to program unit.
    !---------------------------------------------------------------------
    class(unit_user_t),   intent(out)   :: user
    character(len=*),     intent(in)    :: name
    integer(kind=code_k), intent(in)    :: kind
    logical,              intent(inout) :: error
    !
    character(len=*), parameter :: rname='UNIT>GET'
    !
    call cubetools_message(toolseve%trace,rname,'Welcome')
    !
    if (kind.eq.code_unit_auto) then
       ! Kind is unset: parse name and automatic kind recognition
       call user%get_from_name(name,error)
       if (error) return
    else if (name.eq.strg_equal.or.name.eq.strg_star) then
       ! Kind is set but name is unset: return user from given kind
       call user%get_from_kind(kind,error)
       if (error) return
    else
       ! Kind or name are set: return unit from parsed name or from given kind
       call user%get_from_name_or_kind(name,kind,error)
       if (error) return
    endif
  end subroutine cubetools_unit_get
  !
  subroutine cubetools_unit_get_from_name(user,name,error)
    !------------------------------------------------------------------------
    ! Fill the output unit_user_t structure by parsing the given name
    !------------------------------------------------------------------------
    class(unit_user_t), intent(out)   :: user
    character(len=*),   intent(in)    :: name
    logical,            intent(inout) :: error
    !
    character(len=*), parameter :: rname='UNIT>GET>NAME'
    !
    call cubetools_message(toolseve%trace,rname,'Welcome')
    !
    user%kind => unit_unk  ! Default if not found
    !
    if (unit_found(unit_uv))    return
    ! Angular units except for msecond are going to be returned as a
    ! FOV unit. This is a choice maybe because all angular other units
    ! are fully ambiguous between FOV, BEAM and POSANGLE. The choice of
    ! FOV as the main angular unit in this case is dictated by the
    ! main use case. It is left to the calling routine to disambiguate
    ! between FOV, BEAM, and POSANGLE based on their specific use case.
    if (unit_found(unit_fov))   return
    if (unit_found(unit_beam))  return
    if (unit_found(unit_freq))  return
    if (unit_found(unit_velo))  return
    if (unit_found(unit_wave))  return
    if (unit_found(unit_chan))  return
    if (unit_found(unit_pixe))  return
    if (unit_found(unit_dist))  return
    !
    ! BRIGHTNESS and FLUX not yet supported
    !
  contains
    function unit_found(unit) result(found)
      logical :: found  ! Function value on return
      type(unit_kind_t), intent(in), target :: unit
      !
      call cubetools_unit_parse(name,unit%vars,user%user)
      found = .false.
      if (user%user%dimension.ne.code_unresolved) then
        found = .true.
        user%name          = cubetools_unit_name(unit,user%user)
        user%prog_per_user = cubetools_unit_pperu(unit,user%user)
        user%user_per_prog = 1d0/user%prog_per_user
        user%kind          => unit
      endif
    end function unit_found
  end subroutine cubetools_unit_get_from_name
  !
  subroutine cubetools_unit_get_from_kind(user,code_kind,error)
    !---------------------------------------------------------------------
    ! Fill the output unit_user_t structure according to current user
    ! unit.
    !---------------------------------------------------------------------
    class(unit_user_t),   intent(out)   :: user
    integer(kind=code_k), intent(in)    :: code_kind
    logical,              intent(inout) :: error
    !
    character(len=mess_l) :: mess
    character(len=*), parameter :: rname='UNIT>GET>FROM>KIND'
    !
    call cubetools_message(toolseve%trace,rname,'Welcome')
    !
    if (code_kind.lt.0 .or. code_kind.gt.nunitkinds) then
       write(mess,'(a,i0)') 'Unknown unit code ',code_kind
       call cubetools_message(seve%e,rname,mess)
       error = .true.
       return
    endif
    !
    user%user = prog%current(code_kind)
    user%name = prog%usernames(code_kind)  ! From pre-computed
    user%user_per_prog = prog%user_per_prog(code_kind)  ! From pre-computed
    user%prog_per_user = prog%prog_per_user(code_kind)  ! From pre-computed
    user%kind => allunits(code_kind)
  end subroutine cubetools_unit_get_from_kind
  !
  subroutine cubetools_unit_get_from_name_or_kind(user,name,kind,error)
    !---------------------------------------------------------------------
    ! Fill the output unit_user_t structure from either user unit name or
    ! a given kind of unit.
    !---------------------------------------------------------------------
    class(unit_user_t),   intent(out)   :: user
    character(len=*),     intent(in)    :: name
    integer(kind=code_k), intent(in)    :: kind
    logical,              intent(inout) :: error
    !
    character(len=mess_l) :: mess
    character(len=*), parameter :: rname='UNIT>GET>FROM>NAME>OR>KIND'
    !
    call cubetools_message(toolseve%trace,rname,'Welcome')
    !
    select case(kind)
    case (code_unit_uv:code_unit_dist)
      call cubetools_unit_parse(name,allunits(kind)%vars,user%user)
      if (user%user%dimension.ne.code_unresolved) then
        user%name          = cubetools_unit_name(allunits(kind),user%user)
        user%prog_per_user = cubetools_unit_pperu(allunits(kind),user%user)
        user%user_per_prog = 1d0/user%prog_per_user
        user%kind          => allunits(kind)
      endif
   case (code_unit_unk)
      ! *** JP: user may know better the unit than the program
      ! call cubetools_message(seve%w,rname,'UNKNOWN unit kind')
      user%user%dimension = 1
      user%user%magnitude = unity
      ! user%name           = cubetools_unit_name(unit_unk,user%user)
      user%name           = name
      user%prog_per_user  = cubetools_unit_pperu(unit_unk,user%user)
      user%user_per_prog  = 1d0/user%prog_per_user
      user%kind           => unit_unk
    case default
      write(mess,'(a,i0)') 'Unknown unit kind code ',kind
      call cubetools_message(seve%e,rname,mess)
      error = .true.
      return
    end select
    !
    if (user%user%dimension.eq.code_unresolved) then
      ! Nothing found
      call cubetools_message(seve%e,rname,trim(name)//' is not a valid unit')
      error = .true.
      return
    endif
  end subroutine cubetools_unit_get_from_name_or_kind
  !
  subroutine cubetools_unit_is_consistent_with(unit1,unit2,error)
    !---------------------------------------------------------------------
    ! 
    !---------------------------------------------------------------------
    class(unit_user_t), intent(in)    :: unit1
    class(unit_user_t), intent(in)    :: unit2
    logical,            intent(inout) :: error
    !
    character(len=message_length) :: mess
    character(len=*), parameter :: rname='UNIT>IS>CONSISTENT>WITH'
    !
    if (unit1%kind%id.ne.unit2%kind%id) then
       write(mess,'(a,x,a,2x,a)') 'Inconsistent unit kinds:',unit1%name,unit2%name
       call cubetools_message(seve%e,rname,mess)
       error = .true.
       return
    endif
  end subroutine cubetools_unit_is_consistent_with
  !
  !----------------------------------------------------------------------
  !
  subroutine cubetools_unit_get_kind(name,kind,error)
    use cubetools_disambiguate
    use cubetools_brightness
    !------------------------------------------------------------------------
    ! Returns unit kind based on unit name, returns unknown for unrecognized
    ! units.
    ! ---
    ! This subroutine returns only the unit code. Most (all) the uses
    ! requires to parse again the name to get a user_unit_t. Consider using
    ! cubetools_unit_get with kind = code_unit_auto.
    !------------------------------------------------------------------------
    character(len=*),     intent(in)    :: name
    integer(kind=code_k), intent(out)   :: kind
    logical,              intent(inout) :: error
    !
    integer(kind=4) :: ikey
    character(len=*), parameter :: rname='UNIT>GET>KIND'
    !
    call cubetools_message(toolseve%trace,rname,'Welcome')
    !
    kind = unit_unk%id  ! Default if not found
    !
    if (unit_found(unit_uv))    return
    ! Angular units except for msecond are going to be returned as a
    ! FOV unit. This is a choice maybe because all angular other units
    ! are fully ambiguous between POSANG, FOV and BEAM. The choice of
    ! FOV as the main angular unit in this case is dictated by the
    ! main use case. It is left to the calling routine to disambiguate
    ! between FOV,POSANGLE and BEAM based on their specific use case.
    if (unit_found(unit_fov))   return
    if (unit_found(unit_beam))  return
    if (unit_found(unit_freq))  return
    if (unit_found(unit_velo))  return
    if (unit_found(unit_wave))  return
    if (unit_found(unit_chan))  return
    if (unit_found(unit_pixe))  return
    if (unit_found(unit_dist))  return
    !
    ! BRIGHTNESS and FLUX here
    ikey = flux_get(name)
    if (unit_found2(code_unit_flux,ikey)) return
    ikey = brightness_get(name)
    if (unit_found2(code_unit_brig,ikey)) return
    !
  contains
    function unit_found(unit) result(found)
      logical :: found  ! Function value on return
      type(unit_kind_t), intent(in) :: unit
      !
      type(unit_prog_t) :: user
      !
      call cubetools_unit_parse(name,unit%vars,user)
      found = user%dimension.ne.code_unresolved
      if (found) kind = unit%id
    end function unit_found
    !
    function unit_found2(thiskind,ikey) result(found)
      logical :: found  ! Function value on return
      integer(kind=code_k), intent(in) :: thiskind
      integer(kind=4),      intent(in) :: ikey
      !
      found = ikey.ne.code_unresolved
      if (found) kind = thiskind
    end function unit_found2
  end subroutine cubetools_unit_get_kind
  !
  subroutine cubetools_unit_parse(name,variants,user)
    !-------------------------------------------------------------------
    ! From a unit name and a list of authorized units, return which
    ! unit is matched and its magnitude.
    ! Match on the base unit name is case-insensitive.
    ! Return user%dimension = code_unresolved if no match. It is the
    ! choice of the caller to raise an error/warning or not if relevant.
    !-------------------------------------------------------------------
    character(len=*),     intent(in)  :: name
    type(unit_variant_t), intent(in)  :: variants(:)
    type(unit_prog_t),    intent(out) :: user
    !
    integer(kind=4) :: lname,lunit
    integer(kind=unit_k) :: iunit,ivar
    character(len=unit_l) :: uname
    character(len=*), parameter :: rname='UNIT>PARSE'
    !
    user%magnitude = unity
    user%dimension = code_unresolved
    !
    uname = name
    call sic_upper(uname)
    lname = len_trim(name)
    !
    do iunit=1,size(variants)  ! For all supported units
      do ivar=1,variants(iunit)%n
        lunit = len_trim(variants(iunit)%keys(ivar))
        !
        ! If input name is too short, it can not match this unit
        if (lunit.gt.lname)  cycle
        !
        ! Check if base unit matches (upper => case-insensitive!!!)
        if (uname(lname-lunit+1:lname).ne.variants(iunit)%keys(ivar)(1:lunit))  cycle
        !
        ! Check if a valid prefix matches (case-sensitive!!!)
        if (.not.cubetools_unit_magnitude_parse(name(1:lname-lunit),user%magnitude))  cycle
        !
        ! Found it!
        user%dimension = iunit
        return
      enddo
    enddo
  end subroutine cubetools_unit_parse
  !
  subroutine cubetools_unit_get_list(kind,list,error)
    use cubetools_brightness
    !------------------------------------------------------------------------
    ! Returns the list of supported units given the kind code
    !------------------------------------------------------------------------
    integer(kind=code_k), intent(in)    :: kind
    character(len=*),     allocatable   :: list(:)
    logical,              intent(inout) :: error
    !
    character(len=mess_l) :: mess
    character(len=*), parameter :: rname='UNIT>GET>LIST'
    !
    select case (kind)
    case (code_unit_uv)
      list = unit_uv%name
    case (code_unit_fov)
      list = unit_fov%name
    case (code_unit_pang)
      list = unit_pang%name
    case (code_unit_beam)
      list = unit_beam%name
    case (code_unit_freq)
      list = unit_freq%name
    case (code_unit_velo)
      list = unit_velo%name
    case (code_unit_wave)
      list = unit_wave%name
    case (code_unit_chan)
      list = unit_chan%name
    case (code_unit_pixe)
      list = unit_pixe%name
    case (code_unit_dist)
      list = unit_dist%name
    case (code_unit_unk)
      list = unit_unk%name
    case default
      write(mess,'(a,i0)') 'Unknown unit kind code ',kind
      call cubetools_message(seve%e,rname,mess)
      error = .true.
      return
    end select
  end subroutine cubetools_unit_get_list
  !
  function cubetools_unit_kind_prog_name(unit)
    !-------------------------------------------------------------------
    ! Return the name of the program internal unit name
    !-------------------------------------------------------------------
    character(len=unit_l) :: cubetools_unit_kind_prog_name
    class(unit_kind_t), intent(in) :: unit
    cubetools_unit_kind_prog_name = prog%prognames(unit%id)  ! Pre-computed
  end function cubetools_unit_kind_prog_name
end module cubetools_unit
!
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
