!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
!
module cubemain_histo1d
  use cube_types
  use cubetools_structure
  use cubetools_keyword_arg
  use cubeadm_cubeid_types
  use cubemain_messaging
  use cubemain_range
  !
  public :: histo1d
  public :: cubemain_histo1d_command
  private
  !
  character(len=*), parameter :: comm_abstract = &
       'Produce a histogram from a cube'  
  character(len=*), parameter :: comm_help = &
       'Compute the histogram of a cube. The bins are sampled following a &
       &linear or logarithmic scale. The number of bins are automatically &
       &computed using the ??? heuristic or defined by the user. The &
       &histogram can be normalized.'
  character(len=*), parameter :: opt_lin_help = &
       'With this option bins are equally spaced on a linear scale.&
       & Min and Max are by default the minimum and maximum of the&
       & input cube'
  character(len=*), parameter :: opt_log_help = &
       'With this option bins are equally spaced on a log10 scale.&
       & The Maximum value is by default the maximum of cubname and&
       & the default value for range is 1000. The minimum value to be&
       & taken into account for the histogram is: min=max/range.'
  character(len=*), parameter :: opt_nbin_help = &      
       'This option defines the number of bins to be used. Its&
       & argument can be a positive integer or one of 3 methods: RICE&
       &, STURGES or SQRT. If this option is not given explicitly by&
       & the user the number of bins defaults to the value defined by&
       & the STURGES method.'
  ! VVV This was part of the help in the Nroff file, what do we do in
  ! this case:
  !
  !   The number of bins is defined by each&
  !        & method as demonstrated in the table below, where n is the&
  !        & number of valid values in cubname (nl*nm*nc-nnans)
  ! .nf
  !  | Method  | Nbins              |
  !  |---------+--------------------| 
  !  | RICE    | ceiling(2n**(1/3)) |
  !  | STURGES | ceiling(log2(n))+1 |
  !  | SQRT    | ceiling(sqrt(n))   |
  ! .nf
  character(len=*), parameter :: opt_norm_help = &
       'Normalise the histogram to 1 by dividing the number of counts&
       & in each bin by the total number of counts in the histogram.'  
  !
  type :: histo1d_comm_t
     type(option_t),      pointer :: comm
     ! type(option_t),      pointer :: cumulative
     type(option_t),      pointer :: lin
     type(option_t),      pointer :: log
     type(option_t),      pointer :: nbin
     type(keyword_arg_t), pointer :: nbin_arg
     type(option_t),      pointer :: norm
   contains
     procedure, public  :: register         => cubemain_histo1d_register
     procedure, private :: parse            => cubemain_histo1d_parse
     ! procedure, private :: parse_cumulative => cubemain_histo1d_parse_cumulative
     procedure, private :: parse_nbin       => cubemain_histo1d_parse_nbin
     procedure, private :: main             => cubemain_histo1d_main
  end type histo1d_comm_t
  type(histo1d_comm_t) :: histo1d
  !
  integer(kind=4), parameter :: icube = 1
  type histo1d_user_t
     type(cubeid_user_t)   :: cubeids
     character(len=argu_l) :: sort   ! Sorting scheme when doing cumulative histograms
     real(kind=sign_k)     :: lin(2) ! [***] Min and Max for a linear histogram in the cube unit
     real(kind=sign_k)     :: log(2) ! [***] Min and Max for a logarithm histogram in the cube unit
     character(len=argu_l) :: nbin   ! [---] Number of bins to be used or an automatic method
     logical :: documu = .false.     ! Cumulative or standard histogram?
     logical :: dolin  = .false.     ! Linear scale?
     logical :: dolog  = .false.     ! Logarithmique scale?
     logical :: donbin = .false.     ! User defined number of bins?
     logical :: donorm = .false.     ! Normalize?
   contains
     procedure, private :: toprog      => cubemain_histo1d_user_toprog
     procedure, private :: define_nbin => cubemain_histo1d_user_define_nbin
  end type histo1d_user_t
  !
  type histo1d_prog_t
     type(cube_t), pointer :: cube               ! Input cube
     type(cube_t), pointer :: histo              ! Output histogram
     real(kind=sign_k)     :: min                ! Minimum value for the histogram
     real(kind=sign_k)     :: max                ! Maximum value for the histogram
     logical               :: dolog              ! Logarithmic or linear scale?
     logical               :: donorm             ! Normalize?
     integer(kind=4)       :: nbin               ! Number of bins
     real(kind=sign_k)     :: step               ! Bin size
     real(kind=sign_k), allocatable :: bins(:)   ! Bin values at start of the bin
                        ! *** JP it should be in the middle of the bin to conform with images
     real(kind=4),      allocatable :: counts(:) ! Counts of each bin
     procedure(cubemain_histo1d_prog_loop_image), pointer :: loop => null()
   contains
     procedure, private :: compute_bins => cubemain_histo1d_prog_compute_bins
     procedure, private :: header       => cubemain_histo1d_prog_header
     procedure, private :: data         => cubemain_histo1d_prog_data
     procedure, private :: output       => cubemain_histo1d_prog_output
     procedure, private :: image        => cubemain_histo1d_prog_image
     procedure, private :: spectrum     => cubemain_histo1d_prog_spectrum
  end type histo1d_prog_t
  !
contains
  !
  subroutine cubemain_histo1d_command(line,error)
    !----------------------------------------------------------------------
    !
    !----------------------------------------------------------------------
    character(len=*), intent(in)    :: line
    logical,          intent(inout) :: error
    !
    type(histo1d_user_t) :: user
    character(len=*), parameter :: rname='HISTO1D>COMMAND'
    !
    call cubemain_message(mainseve%trace,rname,'Welcome')
    !
    call histo1d%parse(line,user,error)
    if (error) return
    call histo1d%main(user,error)
    if (error) continue
  end subroutine cubemain_histo1d_command
  !
  !----------------------------------------------------------------------
  !
  subroutine cubemain_histo1d_register(histo1d,error)
    use cubedag_allflags
    !----------------------------------------------------------------------
    !
    !----------------------------------------------------------------------
    class(histo1d_comm_t), intent(inout) :: histo1d
    logical,               intent(inout) :: error
    !
    type(cubeid_arg_t) :: cubearg
    type(standard_arg_t) :: stdarg
    type(keyword_arg_t) :: keyarg
    character(len=argu_l), parameter :: methods(3) = ['SQRT   ','STURGES','RICE   ']
    character(len=*), parameter :: rname='HISTO1D>REGISTER'
    !
    call cubemain_message(mainseve%trace,rname,'Welcome')
    !
    ! Command
    call cubetools_register_command(&
         'HISTO1D','[cube]',&
         comm_abstract,&
         comm_help,&
         cubemain_histo1d_command,&
         histo1d%comm,error)
    if (error) return
    call cubearg%register( &
         'CUBE', &
         'Input cube',  &
         strg_id,&
         code_arg_optional,  &
         [flag_any], &
         error)
    if (error) return
    !
    ! call cubetools_register_option(&
    !      'CUMULATIVE','[ASCENDING|DESCENDING|+|-]',&
    !      'Compute a cumulative histogram',&
    !      strg_id,&
    !      histo1d%cumulative,error)
    ! if (error) return
    ! call stdarg%register( &
    !      'sorting',  &
    !      'Sorting of the cumulative histogram', &
    !      strg_id,&
    !      code_arg_optional, &
    !      error)
    ! if (error) return
    ! Option #1
    call cubetools_register_option(&
         'LINEAR','[min [max]]',&
         'Compute a linear scale histogram',&
         opt_lin_help,&
         histo1d%lin,error)
    if (error) return
    call stdarg%register( &
         'min',  &
         'Minimum of the histogram', &
         strg_id,&
         code_arg_optional, &
         error)
    if (error) return
    call stdarg%register( &
         'max',  &
         'Maximum of the histogram', &
         strg_id,&
         code_arg_optional, &
         error)
    if (error) return
    ! Option #2
    call cubetools_register_option(&
         'LOGARITHMIC','[max [range]]',&
         'Compute a logarithm scale histogram',&
         opt_log_help,&
         histo1d%log,error)
    if (error) return
    call stdarg%register( &
         'max',  &
         'Maximum of the histogram', &
         strg_id,&
         code_arg_optional, &
         error)
    if (error) return
    call stdarg%register( &
         'range',  &
         'range of the histogram', &
         strg_id,&
         code_arg_optional, &
         error)
    if (error) return
    ! Option #3
    call cubetools_register_option(&
         'NBIN','nbin',&
         'Define the number of bins',&
         opt_nbin_help,&
         histo1d%nbin,error)
    if (error) return
    call keyarg%register( &
         'nbin',  &
         'Number of bins or method to determine it', &
         strg_id,&
         code_arg_mandatory, &
         methods, &
         flexible, &
         histo1d%nbin_arg, &
         error)
    if (error) return
    ! Option #4
    call cubetools_register_option(&
         'NORMALISE','',&
         'Compute a normalised histogram',&
         opt_norm_help,&
         histo1d%norm,error)
    if (error) return
  end subroutine cubemain_histo1d_register
  !
  subroutine cubemain_histo1d_parse(histo1d,line,user,error)
    !----------------------------------------------------------------------
    ! HISTO1D cubname
    !   /CUMULATIVE [(ascending)|descending|+|-]
    !   /LINEAR min max
    !   /LOGARITHM [max range]
    !   /NBINS n
    !   /NORMALISE
    !----------------------------------------------------------------------
    class(histo1d_comm_t), intent(in)    :: histo1d
    character(len=*),      intent(in)    :: line
    type(histo1d_user_t),  intent(out)   :: user
    logical,               intent(inout) :: error
    !
    character(len=*), parameter :: rname='HISTO1D>PARSE'
    !
    call cubemain_message(mainseve%trace,rname,'Welcome')
    !
    call cubeadm_cubeid_parse(line,histo1d%comm,user%cubeids,error)
    if (error) return
    ! call histo1d%parse_cumulative(line,user,error)
    ! if (error) return
    call cubemain_histo1d_parse_range(line,histo1d%lin,user%lin,user%dolin,error)
    if (error) return
    call cubemain_histo1d_parse_range(line,histo1d%log,user%log,user%dolog,error)
    if (error) return
    call histo1d%parse_nbin(line,user,error)
    if (error) return
    call histo1d%norm%present(line,user%donorm,error)
    if (error) return
    !
    if (user%dolin.and.user%dolog) then
       call cubemain_message(seve%e,rname,'Options /LINEAR and /LOGARITHM are incompatible')
       error = .true.
    endif
    if (user%documu.and.(user%donbin.or.user%dolog)) then
       if (user%donbin) call cubemain_message(seve%e,rname,'Options /CUMULATIVE and /NBINS are incompatible')
       if (user%dolog) call cubemain_message(seve%e,rname,&
            'Options /CUMULATIVE and /LOGARITHM are incompatible')
       error = .true.
    endif
  end subroutine cubemain_histo1d_parse
  !
  ! subroutine cubemain_histo1d_parse_cumulative(histo1d,line,user,error)
  !   use cubetools_disambiguate
  !   !----------------------------------------------------------------------
  !   ! /CUMULATIVE [(ascending)|descending|+|-]
  !   !----------------------------------------------------------------------
  !   class(histo1d_comm_t), intent(in)    :: histo1d
  !   character(len=*),      intent(in)    :: line
  !   type(histo1d_user_t),  intent(inout) :: user
  !   logical,               intent(inout) :: error
  !   !
  !   integer(kind=4), parameter ::  nsort = 4
  !   integer(kind=4) :: ie
  !   character(len=argu_l) :: arg,sortings(nsort)
  !   character(len=*), parameter :: rname='HISTO1D>PARSE>RANGE'
  !   data sortings/'ascending','descending','+','-'/
  !   !
  !   call cubemain_message(mainseve%trace,rname,'Welcome')
  !   !
  !   call histo1d%cumulative%present(line,user%documu,error)
  !   if (error) return
  !   if (user%documu) then
  !      arg = sortings(1)
  !      call cubetools_getarg(line,histo1d%cumulative,1,arg,.not.mandatory,error)
  !      if (error)  return
  !      call cubetools_disambiguate_strict(arg,sortings,ie,user%sort,error)
  !      if (error)  return
  !   endif
  ! end subroutine cubemain_histo1d_parse_cumulative
  !
  subroutine cubemain_histo1d_parse_range(line,opt,range,doit,error)
    use cubetools_nan
    !----------------------------------------------------------------------
    ! /OPT [min max]
    !----------------------------------------------------------------------
    character(len=*),  intent(in)    :: line
    type(option_t),    intent(in)    :: opt
    real(kind=sign_k), intent(out)   :: range(2)
    logical,           intent(out)   :: doit
    logical,           intent(inout) :: error
    !
    character(len=*), parameter :: rname='HISTO1D>PARSE>RANGE'
    !
    call cubemain_message(mainseve%trace,rname,'Welcome')
    !
    range(:) = gr4nan
    call opt%present(line,doit,error)
    if (error) return
    if (doit) then
       call cubetools_getarg(line,opt,1,range(1),.not.mandatory,error)
       if(error) return
       call cubetools_getarg(line,opt,2,range(2),.not.mandatory,error)
       if(error) return       
    endif
  end subroutine cubemain_histo1d_parse_range
  !
  subroutine cubemain_histo1d_parse_nbin(histo1d,line,user,error)
    use gkernel_interfaces
    !----------------------------------------------------------------------
    ! /NBIN STURGES|RICE|SQRT|n
    !----------------------------------------------------------------------
    class(histo1d_comm_t), intent(in)    :: histo1d
    character(len=*),      intent(in)    :: line
    type(histo1d_user_t),  intent(inout) :: user
    logical,               intent(inout) :: error
    !
    character(len=*), parameter :: rname='HISTO1D>PARSE>NBIN'
    !
    call cubemain_message(mainseve%trace,rname,'Welcome')
    !
    call histo1d%nbin%present(line,user%donbin,error)
    if (error) return
    if (user%donbin) then
       call cubetools_getarg(line,histo1d%nbin,1,user%nbin,mandatory,error)
       if(error) return
    endif
  end subroutine cubemain_histo1d_parse_nbin
  !
  subroutine cubemain_histo1d_main(histo1d,user,error)
    use cubeadm_timing
    !----------------------------------------------------------------------
    !
    !----------------------------------------------------------------------
    class(histo1d_comm_t), intent(in)    :: histo1d
    type(histo1d_user_t),  intent(in)    :: user
    logical,               intent(inout) :: error
    !
    type(histo1d_prog_t) :: prog
    character(len=*), parameter :: rname='HISTO1D>MAIN'
    !
    call cubemain_message(mainseve%trace,rname,'Welcome')
    !
    call user%toprog(prog,error)
    if (error) return
    call prog%header(error)
    if (error) return
    call cubeadm_timing_prepro2process()
    call prog%data(error)
    if (error) return
    call cubeadm_timing_process2postpro()
  end subroutine cubemain_histo1d_main
  !
  !------------------------------------------------------------------------
  !
  subroutine cubemain_histo1d_user_toprog(user,prog,error)
    use cubetools_nan
    use cubeadm_get
    !----------------------------------------------------------------------
    !
    !----------------------------------------------------------------------
    class(histo1d_user_t), intent(in)    :: user
    type(histo1d_prog_t),  intent(out)   :: prog
    logical,               intent(inout) :: error
    !
    character(len=*), parameter :: rname='HISTO1D>USER>TOPROG'
    !
    call cubemain_message(mainseve%trace,rname,'Welcome')
    !
    call cubeadm_cubeid_get_header(histo1d%comm,icube,user%cubeids,code_access_imaset_or_speset,  &
         code_read,prog%cube,error)
    if (error) return    
    if (user%dolin) then
       prog%dolog = .false.
       if (ieee_is_nan(user%lin(1))) then
          prog%min = prog%cube%head%arr%min%val
       else
          prog%min = user%lin(1)
       endif
       if (ieee_is_nan(user%lin(2))) then
          prog%max = prog%cube%head%arr%max%val
       else
          prog%max = user%lin(2)
       endif
    else if(user%dolog) then
       prog%dolog = .true.
       if (ieee_is_nan(user%log(1))) then
          prog%max = prog%cube%head%arr%max%val
       else
          if (user%log(1).gt.0) then
             prog%max = user%log(1)
          else
             call cubemain_message(seve%e,rname,'Maximum of a logarithmic histogram must be positive')
             error = .true.
             return
          endif
       endif
       if (ieee_is_nan(user%log(2))) then
          prog%min = prog%max/1e3
       else
          if (user%log(2).gt.0) then
             prog%min = prog%max/user%log(2)
          else
             call cubemain_message(seve%e,rname,'Dynamic range of histogram must be positive')
             error = .true.
             return
          endif
       endif
    else
       prog%dolog = .false.
       prog%min = prog%cube%head%arr%min%val
       prog%max = prog%cube%head%arr%max%val
    endif
    !
    if (user%documu) then
       call cubemain_message(seve%e,rname,'Cumulative histograms not yet implemented')
       error = .true.
       return
    endif
    !
    call user%define_nbin(prog,error)
    if (error) return
    !
    prog%donorm = user%donorm
    if (prog%cube%order().eq.code_cube_imaset) then
       prog%loop => cubemain_histo1d_prog_loop_image
    else
       prog%loop => cubemain_histo1d_prog_loop_spectrum
    endif
  end subroutine cubemain_histo1d_user_toprog
  !
  subroutine cubemain_histo1d_user_define_nbin(user,prog,error)
    use cubetools_user2prog
    !----------------------------------------------------------------------
    !
    !----------------------------------------------------------------------
    class(histo1d_user_t), intent(in)    :: user
    type(histo1d_prog_t),  intent(inout) :: prog
    logical,               intent(inout) :: error
    !
    character(len=argu_l) :: method
    character(len=mess_l) :: mess
    integer(kind=4) :: ikey
    integer(kind=index_length) :: ndata
    integer(kind=4), parameter :: defnbin = -1000
    character(len=*), parameter :: rname='HISTO1D>USER>DEFINE>NBIN'
    !
    call cubemain_message(mainseve%trace,rname,'Welcome')
    !
    if (user%donbin) then
       if (user%nbin.eq.strg_star) then
          method = 'STURGES'
       else
          call cubetools_keyword_user2prog(histo1d%nbin_arg,user%nbin,ikey,method,error)
          if (error)  return
          if (method.eq.strg_unresolved) then
             call cubetools_user2prog_resolve_star(user%nbin,defnbin,prog%nbin,error)
             if (error) return
             if (prog%nbin.lt.1) then
                call cubemain_message(seve%e,rname,'Number of bins must be greater than 0')
                error = .true.
                return
             endif
             return
          endif
       endif
    else
       ! Sturges is the default since it is the choice in R
       method = 'STURGES'
    endif
    !
    ! VVV ndata is only an approximation to the real size of the data
    ! to be put into the histo1d as we can't know a priori how many
    ! data points will be excluded for being above or below the
    ! prog%min,prog%max
    ndata = prog%cube%head%arr%n%l*prog%cube%head%arr%n%m*prog%cube%head%arr%n%c
    select case(method)
    case('SQRT')
       prog%nbin = ceiling(sqrt(1.0*ndata))
    case('STURGES')
       prog%nbin = ceiling(log(1.0*ndata))+1
    case('RICE')
       prog%nbin = ceiling(2*(1.0*ndata)**(1/3.))
    case default
       call cubemain_message(seve%e,rname,'Unknown method for defining nbin: '//trim(method))
       error = .true.
       return
    end select
    !
    write(mess,'(a,i0,a)') 'Using ',prog%nbin,' bins, method = '//trim(method)
    call cubemain_message(seve%i,rname,mess)
  end subroutine cubemain_histo1d_user_define_nbin
  !
  !----------------------------------------------------------------------
  !
  subroutine cubemain_histo1d_prog_header(prog,error)
    use cubedag_allflags
    use cubeadm_clone
    use cubetools_unit
    use cubetools_axis_types
    use cubetools_header_methods
    !----------------------------------------------------------------------
    ! Here we are stretching at the maximum what we can currently do
    ! because we try to store the x and y axis of an histo1d as two 1D
    ! array of identical size n in a lc array of size [n,2].
    ! *** JP 1. This should be a table, not an image. This will force us
    ! *** JP    to write many complex code to go around the fact it indeed is
    ! *** JP    a table, not an image...
    ! *** JP 2. Or it would make more sense to keep the x axis as the axis
    ! *** JP    coordinate and the histo1d as the intensity.
    ! *** JP 3. We should keep the information whether it is in log or lin.
    !----------------------------------------------------------------------
    class(histo1d_prog_t), intent(inout) :: prog
    logical,               intent(inout) :: error
    !
    type(axis_t) :: axis
    character(len=*), parameter :: rname='HISTO1D>PROG>HEADER'
    !
    call cubemain_message(mainseve%trace,rname,'Welcome')
    !
    call prog%compute_bins(error)
    if (error) return
    !
    call cubeadm_clone_header(prog%cube,flag_histo1d,prog%histo,error,access=code_cube_speset)
    if (error) return
    !
    call cubetools_header_get_axis_head_c(prog%histo%head,axis,error)
    if (error) return
    axis%name = '1D histogram'
    if (prog%dolog) then ! *** JP we could keep here if it is lin or log
       axis%unit = 'Log10 scale'
    else
       axis%unit = 'Linear scale'
    endif
    axis%kind = code_unit_unk
    axis%genuine = .true.
    axis%regular = .true. ! Should it be false?
    axis%ref = 5d-1
    axis%val = prog%min
    axis%inc = prog%step
    axis%n = prog%nbin
    call cubetools_header_update_axset_c(axis,prog%histo%head,error)
    if (error) return
    !
    call cubetools_header_nullify_axset_l(prog%histo%head,error)
    if (error) return
    call cubetools_header_nullify_axset_m(prog%histo%head,error)
    if (error) return
  end subroutine cubemain_histo1d_prog_header
  !
  subroutine cubemain_histo1d_prog_compute_bins(prog,error)
    use gkernel_interfaces
    !----------------------------------------------------------------------
    !
    !----------------------------------------------------------------------
    class(histo1d_prog_t), intent(inout) :: prog
    logical,               intent(inout) :: error
    !
    integer(kind=4) :: ier,ibin
    character(len=*), parameter :: rname='HISTO1D>PROG>COMPUTE_BINS'
    !
    call cubemain_message(mainseve%trace,rname,'Welcome')
    !
    allocate(prog%bins(prog%nbin),prog%counts(prog%nbin),stat=ier)
    if (failed_allocate(rname,'Bin list',ier,error)) return
    prog%counts(:) = 0.0
    if (prog%dolog) then
       prog%step = log10(prog%max/prog%min)/prog%nbin
    else
       prog%step = (prog%max-prog%min)/prog%nbin
    endif
    do ibin=1,prog%nbin
       if (prog%dolog) then
          prog%bins(ibin) = 10**(log10(prog%min)+prog%step*ibin)
       else
          prog%bins(ibin) = prog%min+prog%step*ibin
       endif
    enddo
  end subroutine cubemain_histo1d_prog_compute_bins
  !
  subroutine cubemain_histo1d_prog_data(prog,error)
    use cubeadm_opened
    !----------------------------------------------------------------------
    !
    !----------------------------------------------------------------------
    class(histo1d_prog_t), intent(inout) :: prog
    logical,               intent(inout) :: error
    !
    type(cubeadm_iterator_t) :: iter
    character(len=*), parameter :: rname='HISTO1D>PROG>DATA'
    !
    call cubemain_message(mainseve%trace,rname,'Welcome')
    !
    call cubeadm_datainit_all(iter,error)
    if (error) return
    !$OMP PARALLEL DEFAULT(none) SHARED(prog,error) FIRSTPRIVATE(iter)
    !$OMP SINGLE
    do while (cubeadm_dataiterate_all(iter,error))
       if (error)  exit
       !$OMP TASK SHARED(prog) FIRSTPRIVATE(iter,error)
       if (.not.error) then
          call prog%loop(iter%first,iter%last,error)
       endif
       !$OMP END TASK
    enddo ! ie
    !$OMP END SINGLE
    !$OMP END PARALLEL
    !
    call prog%output(error)
    if (error) return
  end subroutine cubemain_histo1d_prog_data
  !
  !----------------------------------------------------------------------
  !
  subroutine cubemain_histo1d_prog_loop_image(prog,first,last,error)
    use gkernel_interfaces
    use cubeadm_entryloop
    use cubemain_image_real
    !----------------------------------------------------------------------
    !
    !----------------------------------------------------------------------
    class(histo1d_prog_t), intent(inout) :: prog
    integer(kind=entr_k),  intent(in)    :: first
    integer(kind=entr_k),  intent(in)    :: last
    logical,               intent(inout) :: error
    !
    integer(kind=entr_k) :: ie
    integer(kind=4) :: ier
    type(image_t) :: inima
    real(kind=4), allocatable :: counts(:)
    character(len=*), parameter :: rname='HISTO1D>PROG>LOOP>IMAGE'
    !
    call cubemain_message(mainseve%trace,rname,'Welcome')
    !
    call inima%init(prog%cube,error)
    if (error) return
    allocate(counts(prog%nbin),stat=ier)
    if (failed_allocate(rname,'Task counts',ier,error)) return
    counts(:) = 0.0
    !
    do ie=first,last
       call cubeadm_entryloop_iterate(ie,error)
       if (error)  return
       call prog%image(ie,inima,counts,error)
       if (error)  return
    enddo
    !
    !$OMP CRITICAL
    prog%counts(:) = prog%counts(:)+counts(:)
    !$OMP END CRITICAL
  end subroutine cubemain_histo1d_prog_loop_image
  !
  subroutine cubemain_histo1d_prog_image(prog,ie,inima,counts,error)
    use cubetools_nan
    use cubemain_image_real
    !----------------------------------------------------------------------
    !
    !----------------------------------------------------------------------
    class(histo1d_prog_t), intent(inout) :: prog
    integer(kind=entr_k),  intent(in)    :: ie
    type(image_t),         intent(inout) :: inima
    real(kind=4),          intent(inout) :: counts(:)
    logical,               intent(inout) :: error
    !
    integer(kind=4) :: ibin 
    integer(kind=pixe_k) :: ix,iy
    character(len=*), parameter :: rname='HISTO1D>PROG>IMAGE'
    !
    call cubemain_message(mainseve%trace,rname,'Welcome')
    !
    call inima%get(prog%cube,ie,error)
    if (error)  return
    if (prog%dolog) then
       do iy=1,prog%cube%head%arr%n%m
          do ix=1,prog%cube%head%arr%n%l
             if (.not.ieee_is_nan(inima%z(ix,iy))&
                  .and.inima%z(ix,iy).le.prog%max&
                  .and.inima%z(ix,iy).ge.prog%min) then
                if (inima%z(ix,iy).eq.prog%min) then
                   ibin = 1
                else
                   ibin = ceiling(log10(inima%z(ix,iy)/prog%min)/prog%step)
                endif
                counts(ibin) = counts(ibin)+1
             endif
          enddo ! iy
       enddo ! ix
    else
       do iy=1,prog%cube%head%arr%n%m
          do ix=1,prog%cube%head%arr%n%l
             if (.not.ieee_is_nan(inima%z(ix,iy)) &
                  .and.inima%z(ix,iy).le.prog%max &
                  .and.inima%z(ix,iy).ge.prog%min) then
                if (inima%z(ix,iy).eq.prog%min) then
                   ibin = 1
                else
                   ibin = ceiling((inima%z(ix,iy)-prog%min)/prog%step)
                endif
                counts(ibin) = counts(ibin)+1
             endif
          enddo ! iy
       enddo ! ix
    endif
  end subroutine cubemain_histo1d_prog_image
  !
  !----------------------------------------------------------------------
  !
  subroutine cubemain_histo1d_prog_loop_spectrum(prog,first,last,error)
    use gkernel_interfaces
    use cubeadm_entryloop
    use cubemain_spectrum_real
    !----------------------------------------------------------------------
    !
    !----------------------------------------------------------------------
    class(histo1d_prog_t), intent(inout) :: prog
    integer(kind=entr_k),  intent(in)    :: first
    integer(kind=entr_k),  intent(in)    :: last
    logical,               intent(inout) :: error
    !
    integer(kind=entr_k) :: ie
    integer(kind=4) :: ier
    type(spectrum_t) :: inspec
    real(kind=4), allocatable :: counts(:)
    character(len=*), parameter :: rname='HISTO1D>PROG>LOOP>SPECTRUM'
    !
    call cubemain_message(mainseve%trace,rname,'Welcome')
    !
    call inspec%reassociate_and_init(prog%cube,error)
    if (error) return
    allocate(counts(prog%nbin),stat=ier)
    if (failed_allocate(rname,'Task counts',ier,error)) return
    counts(:) = 0.0
    !
    do ie=first,last
       call cubeadm_entryloop_iterate(ie,error)
       if (error)  return
       call prog%spectrum(ie,inspec,counts,error)
       if (error)  return
    enddo
    !
    !$OMP CRITICAL
    prog%counts(:) = prog%counts(:)+counts(:)
    !$OMP END CRITICAL
  end subroutine cubemain_histo1d_prog_loop_spectrum
  !
  subroutine cubemain_histo1d_prog_spectrum(prog,ie,inspec,counts,error)
    use cubetools_nan
    use cubemain_spectrum_real
    !----------------------------------------------------------------------
    !
    !----------------------------------------------------------------------
    class(histo1d_prog_t), intent(inout) :: prog
    integer(kind=entr_k),  intent(in)    :: ie
    type(spectrum_t),      intent(inout) :: inspec
    real(kind=4),          intent(inout) :: counts(:)
    logical,               intent(inout) :: error
    !
    integer(kind=4) :: ibin
    integer(kind=chan_k) :: ic
    character(len=*), parameter :: rname='HISTO1D>SPECTRUM'
    !
    call cubemain_message(mainseve%trace,rname,'Welcome')
    !
    call inspec%get(prog%cube,ie,error)
    if (error)  return
    if (prog%dolog) then
       do ic=1,prog%cube%head%arr%n%c
          if (.not.ieee_is_nan(inspec%t(ic))&
               .and.inspec%t(ic).le.prog%max&
               .and.inspec%t(ic).ge.prog%min) then
             if (inspec%t(ic).eq.prog%min) then
                ibin = 1
             else
                ibin = ceiling(log10(inspec%t(ic)/prog%min)/prog%step)
             endif
             counts(ibin) = counts(ibin)+1
          endif
       enddo ! ic
    else
       do ic=1,prog%cube%head%arr%n%c
          if (.not.ieee_is_nan(inspec%t(ic))&
               .and.inspec%t(ic).le.prog%max&
               .and.inspec%t(ic).ge.prog%min) then
             if (inspec%t(ic).eq.prog%min) then
                ibin = 1
             else
                ibin = ceiling((inspec%t(ic)-prog%min)/prog%step)
             endif
             counts(ibin) = counts(ibin)+1
          endif
       enddo ! ic
    endif
  end subroutine cubemain_histo1d_prog_spectrum
  !
  !------------------------------------------------------------------------
  !
  subroutine cubemain_histo1d_prog_output(prog,error)
    use cubeadm_ioloop
    use cubemain_spectrum_real
    !----------------------------------------------------------------------
    ! Here we put the counts computed previously on the output cube
    !----------------------------------------------------------------------
    class(histo1d_prog_t), intent(inout) :: prog
    logical,               intent(inout) :: error
    !
    type(spectrum_t) :: output
    integer(kind=chan_k), parameter :: one = 1
    character(len=*), parameter :: rname='HISTO1D>OUTPUT'
    !
    call cubemain_message(mainseve%trace,rname,'Welcome')
    !
    call output%reallocate('histo1d',prog%histo%head%arr%n%c,error)
    if(error) return
    call cubeadm_io_iterate(one,one,prog%histo,error)
    if (error) return
    if (prog%donorm) prog%counts(:) = prog%counts(:)/sum(prog%counts)
    output%t(:) = prog%counts(:)
    call output%put(prog%histo,one,error)
    if (error) return
  end subroutine cubemain_histo1d_prog_output
end module cubemain_histo1d
!
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
